
/*********************************************************************

Copyright (C) 1993, 1994,1995  Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

  contained herein is code that implements packaging of data for the
  software renderer for each primitive type.  calls to the polygon
  scan converter (software) and line dda (software) are made from this
  file.

*********************************************************************/



/*
   
Portions of this code are derivative works of code contained in the
distribution for Graphics Gems I.

*/

/*
Generic Convex Polygon Scan Conversion and Clipping
by Paul Heckbert
from "Graphics Gems", Academic Press, 1990
*/


/*
 * poly.c: simple utilities for polygon data structures
 */
#include <design.h>
#include "rmonster_objs.h"
#include "poly.h"
#include "clip.h"
#include "camera.h"
#include "rmatrix.h"
#include "volrender.h"

/* proto from shader.c */
int get_def_shader PROTO((void));

/* proto from line_draw.c */
int line_draw PROTO((Poly *line,
	  PWindow *win,
	  void (*pixelproc)(),
	  surface_colors_rgba *color1,
	  surface_colors_rgba *color2));


Poly_vert *poly_dummy;	/* used superficially by POLY_MASK macro */

/*
 * poly_print: print Poly p to kstdout, prefixed by the label str */

void poly_print(str, p)
char *str;
Poly *p;
{
    int i;

    printf("%s: %d sides\n", str, p->n);
    poly_vert_label("        ", p->mask);
    for (i=0; i<p->n; i++) {
		printf("   v[%d] ", i);
		poly_vert_print("", &p->vert[i], p->mask);
    }
}


void poly_vert_label(str, mask)
char *str;
int mask;
{
    printf("%s", str);
    if (mask&POLY_MASK(sx))   printf("   sx  ");
    if (mask&POLY_MASK(sy))   printf("   sy  ");
    if (mask&POLY_MASK(sz))   printf("   sz  ");
    if (mask&POLY_MASK(sw))   printf("   sw  ");
    if (mask&POLY_MASK(x))    printf("   x   ");
    if (mask&POLY_MASK(y))    printf("   y   ");
    if (mask&POLY_MASK(z))    printf("   z   ");
    if (mask&POLY_MASK(u))    printf("   u   ");
    if (mask&POLY_MASK(v))    printf("   v   ");
    if (mask&POLY_MASK(w))    printf("   q   ");
    if (mask&POLY_MASK(r))    printf("   r   ");
    if (mask&POLY_MASK(g))    printf("   g   ");
    if (mask&POLY_MASK(b))    printf("   b   ");
    if (mask&POLY_MASK(a))    printf("   a   ");
    if (mask&POLY_MASK(nx))   printf("   nx  ");
    if (mask&POLY_MASK(ny))   printf("   ny  ");
    if (mask&POLY_MASK(nz))   printf("   nz  ");
    printf("\n");
}

void poly_vert_print(str, v, mask)
char *str;
Poly_vert *v;
int mask;
{
    printf("%s", str);
    if (mask&POLY_MASK(sx)) printf(" %6.1f", v->sx);
    if (mask&POLY_MASK(sy)) printf(" %6.1f", v->sy);
    if (mask&POLY_MASK(sz)) printf(" %6.2f", v->sz);
    if (mask&POLY_MASK(sw)) printf(" %6.2f", v->sw);
    if (mask&POLY_MASK(x))  printf(" %6.2f", v->x);
    if (mask&POLY_MASK(y))  printf(" %6.2f", v->y);
    if (mask&POLY_MASK(z))  printf(" %6.2f", v->z);
    if (mask&POLY_MASK(u))  printf(" %6.2f", v->u);
    if (mask&POLY_MASK(v))  printf(" %6.2f", v->v);
    if (mask&POLY_MASK(w))  printf(" %6.2f", v->w);
    if (mask&POLY_MASK(r))  printf(" %6.4f", v->r);
    if (mask&POLY_MASK(g))  printf(" %6.4f", v->g);
    if (mask&POLY_MASK(b))  printf(" %6.4f", v->b);
    if (mask&POLY_MASK(a))  printf(" %6.4f", v->a);
    if (mask&POLY_MASK(nx)) printf(" %6.3f", v->nx);
    if (mask&POLY_MASK(ny)) printf(" %6.3f", v->ny);
    if (mask&POLY_MASK(nz)) printf(" %6.3f", v->nz);
    printf("\n");
}

/*******  FIREWALL  ********/

/**
  * Code added by Wes Bethel, Lawrence Berkeley Laboratory, January 1993
**/


extern void flat_shaderfunc();

void noprim PROTO ((rmonster_prim *,int,int,
	    vertex_3d *, vertex_3d *, vertex_3d *,
	    float *, int, int, rmonsterobj *, int, int));
void polytri_dis PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
extern void polytri_con PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void polyline_dis PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void polyline_con PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void qmesh PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void omesh PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void polyhedron PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void sphere PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void dirpoint PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));
void text PROTO((rmonster_prim *,int,int,vertex_3d *,vertex_3d *,vertex_3d *,float *,int,int,rmonsterobj *,int,int));

static Poly work_poly;
static PWindow work_window;

void (*software_primfuncs[])() =
{
    noprim,
    polytri_dis,
    polytri_con,
    polyline_dis,
    polyline_con,
    qmesh,
    omesh,
    polyhedron,
    sphere,
    dirpoint,
    text
};

extern void (*opaque_shaderfuncs[])();
extern void (*transparent_shaderfuncs[])();
extern void (*init_shaderfuncs[])();
extern int num_init_shaderfuncs;

static int map_to_work_area PROTO((Poly *,Poly *,vertex_3d *,vertex_3d *, \
				   float,float,surface_attribs *, \
				   surface_colors_rgba *,float *,int,int));

void noprim(rmonster_prim *p,
	    int xs,int ys,
	    vertex_3d *obj_space_verts,
	    vertex_3d *view_space_verts,
	    vertex_3d *norms,
	    float *unused,
	    int projection,
	    int opaque_status,
	    rmonsterobj *r,
	    int hither,
	    int yon)
{
}


/**
  * routines for accessing polytri color information.  there are different
  * routines provided for accessing color information in the rmonster_prim
  * struct for all the various permutations of has alpha, no alpha, per-vertex,
  * or per-segment color.
**/

static void polytri_dis_no_color PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_dis_alpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_dis_alpha_per_face PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_dis_noalpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polytri_dis_noalpha_per_face PROTO((rmonster_prim *,rmonsterobj *,int,float *));

/* routines for dealing with the normals */
static void polytri_dis_per_vertex_normals PROTO((vertex_3d *,vertex_3d *,int));
static void polytri_dis_per_face_normals PROTO((vertex_3d *,vertex_3d *,int));

void polytri_dis(rmonster_prim *p,
		 int xs,int ys,
		 vertex_3d *obj_space_verts,
		 vertex_3d *view_space_verts,
		 vertex_3d *norms,
		 float *unused,
		 int projection,
		 int opaque_status,
		 rmonsterobj *r,
		 int hither,
		 int yon)
{
    float xscale,xd,yscale,yd,zscale;
    float xview,yview,zview;
    float epsilon = 0.001;
    float xscreen,yscreen,zscreen;
    int use_global_obj_color;
    int ncolors,status;
    int i,j,k;
    Poly lwork_poly;
    void (**shaderfuncs)();
    int in_use_shader;
    int has_alpha,layout_type;
    int index;
    void (*colorfunc)();
    void (*normalfunc)();
    surface_colors_rgba *work_colors,*work_colors2;
    int has_normals;
    float zmin;
    vertex_3d *work_norms;

    get_zmin(&zmin);
    has_alpha = r->has_alpha;

    /* temp kludge till steve k fixes the normals thing */
    if (norms == NULL)
	has_normals = 0;
    else
	has_normals = 1;
    
    in_use_shader = r->shader;  /* range check?? */
    
    if (opaque_status == TRUE)
	shaderfuncs = opaque_shaderfuncs;
    else
	shaderfuncs = transparent_shaderfuncs;

    work_window.x0 = work_window.y0 = 0;
    work_window.x1 = xs-1;
    work_window.y1 = ys-1;
    
    /** see Jim Blinn's Corner, IEEE CG&A, mmm/yyy **/
    xscale = (xs - epsilon)/2.0;
    xd = xscale;
    yscale = (ys - epsilon)/2.0;  /* the aspect ratio term goes here. */
    yd = yscale;
    zscale = 1.;

    lwork_poly.n = 3;
    lwork_poly.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz);

    if (in_use_shader == KSHADER_GOUROUD)
    {
	if (opaque_status == TRUE)
	    lwork_poly.mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b);
	else
	    lwork_poly.mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b) | POLY_MASK(a);
}

    if ((p->ncolors == 0) || (p->ncolors == p->nverts))
    {
	layout_type = KPER_VERTEX;
	ncolors = p->nverts;
    }
    else
    {
	layout_type = KPER_FACE;
	ncolors = p->nverts / 3;
    }
    
    if (p->ncolors == 0)
	colorfunc = polytri_dis_no_color;
    else if (has_alpha)
    {
	if (layout_type == KPER_FACE)
	    colorfunc = polytri_dis_alpha_per_face;
	else /* assume KPER_vertex */
	    colorfunc = polytri_dis_alpha_per_vertex;
    }
    else
    {
	if (layout_type == KPER_VERTEX)
	    colorfunc = polytri_dis_noalpha_per_vertex;
	else /* assume KPER_FACE */
	    colorfunc = polytri_dis_noalpha_per_face;
    }
	    
    /**
      * deal with shading the vertices.
    **/
    
    work_colors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*
						 ncolors);
    work_colors2 = NULL;

    /* set up the work colors array.  the purpose is to have per-vertex
     colors, and the various color routines we call rectify the difference
     between per-face and per-vertex colors.  */
    
    for (i=0;i<ncolors;i++)
	(*colorfunc)(p,r,i,work_colors+i);

    /*  deal with normals.  they may be per-face or per-vertex. */
    if (p->nthings == 0)	/* bail - there are no normals */
    {
	kerror("GEOMETRY","rmonster","polytri_dis(): normals must not be computed at render time.");
	return;
    }
    else
    {
	work_norms = (vertex_3d *)kmalloc(sizeof(vertex_3d)*p->nverts);
	if (p->nverts == p->nthings) /* per vertex normals already */
	    normalfunc = polytri_dis_per_vertex_normals;
	else
	    normalfunc = polytri_dis_per_face_normals;

	for (i=0;i < p->nverts;i++)
	    (*normalfunc)(norms,work_norms,i);
    }

    /* compute the per-vertex shade */
    vertex_shader(obj_space_verts,norms,ncolors,
		  work_colors,&work_colors2,&(r->attribs),
		  &(r->colors.a),
		  r->shader);
    
    for (i=0;i<p->nverts;) /* i is incremented below */
    {
	for (j=0;j<3;j++)
	{
	    xview = view_space_verts[i].v[0];
	    yview = view_space_verts[i].v[1];
	    zview = view_space_verts[i].v[2];

	    if (projection == KPROJECTION_PERSPECTIVE)
	    {
		xview = xview/zview;
		yview = yview/zview;
	    }
	    
	    xscreen = xview*xscale + xd;
	    yscreen = yview*yscale + yd;
	    zscreen = zview;

	    lwork_poly.vert[j].sx = xscreen;
	    lwork_poly.vert[j].sy = yscreen;
	    lwork_poly.vert[j].sz = zscreen;
#if 0
	    lwork_poly.vert[j].x = obj_space_verts[i].v[0];
	    lwork_poly.vert[j].y = obj_space_verts[i].v[1];
	    lwork_poly.vert[j].z = obj_space_verts[i].v[2];
#endif
	    lwork_poly.vert[j].x = xview;
	    lwork_poly.vert[j].y = yview;
	    lwork_poly.vert[j].z = zview;
	    
	    if (layout_type == KPER_VERTEX)
		index = i;
	    else
		index = i/3;
	    
	    lwork_poly.vert[j].r = work_colors2[index].r;
	    lwork_poly.vert[j].g = work_colors2[index].g;
	    lwork_poly.vert[j].b = work_colors2[index].b;
	    lwork_poly.vert[j].a = work_colors2[index].a;
	    
	    if (has_normals) /* deal with normals -assume per-vertex layout*/
	    {
		lwork_poly.vert[j].nx = norms[i].v[0];
		lwork_poly.vert[j].ny = norms[i].v[1];
		lwork_poly.vert[j].nz = norms[i].v[2];
	    }
	    i++;
	}

	(*init_shaderfuncs[in_use_shader])(&(r->attribs),&(r->colors),NULL,
					      NULL,&lwork_poly);
	status = (*p->clipfunc)(&lwork_poly,projection,&zmin,hither,yon);
	if (status == CHILL)
	    poly_scan(&lwork_poly,&work_window,shaderfuncs[in_use_shader]);
    }
    kfree(work_colors2);
    kfree(work_colors);
}

/**
  * routines for accessing line color information.  there are different
  * routines provided for accessing color information in the rmonster_prim
  * struct for all the various permutations of has alpha, no alpha, per-vertex,
  * or per-segment color.
**/

static void polyline_con_no_color PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_con_alpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_con_alpha_per_segment PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_con_noalpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_con_noalpha_per_segment PROTO((rmonster_prim *,rmonsterobj *,int,float *));
    
static void polyline_dis_no_color PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_dis_alpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_dis_alpha_per_segment PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_dis_noalpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void polyline_dis_noalpha_per_segment PROTO((rmonster_prim *,rmonsterobj *,int,float *));
    
void polyline_dis(rmonster_prim *p,
		  int xs,int ys,
		  vertex_3d *obj_space_verts,
		  vertex_3d *view_space_verts,
		  vertex_3d *norms,
		  float *unused,
		  int projection,
		  int opaque_status,
		  rmonsterobj *r,
		  int hither,
		  int yon)
{
    /**
      * NOTES: at present (8 July 93), per vertex line color is supported
      * at all levels except at the lowest; in other words, the renderer.
    **/
    
    float xscale,xd,yscale,yd,zscale;
    float xview,yview,zview;
    float epsilon = 0.001;
    float xscreen,yscreen,zscreen;
    int use_global_obj_color;
    float obj_rgba[4],color_v1[4],color_v2[4];
    int i,j;
    Poly line;
    int px1,py1,px2,py2;
    surface_colors_rgba *tc;
    void (**shaderfuncs)();
    surface_colors_rgba color1,color2;
    int layout_type;
    float *f;
    int has_alpha;
    int status;
    int (*cfunc)();
    float zmin;
    void (*colorfunc)();
    int vertsize;

    vertsize = sizeof(Poly_vert);

    get_zmin(&zmin);

    has_alpha = r->has_alpha;

    if (p->nverts == p->ncolors)
	layout_type = KPER_VERTEX;
    else
	layout_type = KPER_LINE;
    
    if (opaque_status == TRUE)
	shaderfuncs = opaque_shaderfuncs;
    else
	shaderfuncs = transparent_shaderfuncs;

    work_window.x0 = work_window.y0 = 0;
    work_window.x1 = xs-1;
    work_window.y1 = ys-1;
    
    /** see Jim Blinn's Corner, IEEE CG&A, mmm/yyy **/
    xscale = (xs - epsilon)/2.0;
    xd = xscale;
    yscale = (ys - epsilon)/2.0;  /* the aspect ratio term goes here. */
    yd = yscale;
    zscale = 1.;

    line.n = 3;
    line.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz);

    if (p->ncolors == 0)
	colorfunc = polyline_dis_no_color;
    else if (has_alpha)
    {
	if (layout_type == KPER_VERTEX)
	    colorfunc = polyline_dis_alpha_per_vertex;
	else /* assume KPER_LINE */
	    colorfunc = polyline_dis_alpha_per_segment;
    }
    else
    {
	if (layout_type == KPER_VERTEX)
	    colorfunc = polyline_dis_noalpha_per_vertex;
	else /* assume KPER_LINE */
	    colorfunc = polyline_dis_noalpha_per_segment;
    }

    
    for (i=0;i<p->nverts;) /* i is incremented below */
    {
	for (j=0;j<2;j++)
	{
	    xview = view_space_verts[i].v[0];
	    yview = view_space_verts[i].v[1];
	    zview = view_space_verts[i].v[2];
	    
	    line.vert[j].x = xview;
	    line.vert[j].y = yview;
	    line.vert[j].z = zview;
	    
	    i++;
	}

	status = (*p->clipfunc)(p,&line,&line,projection,&zmin,hither,yon);
	if (status != CHILL)
	    continue;

	for (j=0;j<2;j++)
	{
	    xview = line.vert[j].x;
	    yview = line.vert[j].y;
	    zview = line.vert[j].z;
	    
	    if (projection == KPROJECTION_PERSPECTIVE)
	    {
		xview = xview/zview;
		yview = yview/zview;
	    }

	    xscreen = xview*xscale + xd;
	    yscreen = yview*yscale + yd;
	    zscreen = zview;

	    line.vert[j].sx = xscreen;
	    line.vert[j].sy = yscreen;
	    line.vert[j].sz = zscreen;
	}


	/** NOTE: per-vertex line color isn't yet supported at the
	 renderer level only */

	(*colorfunc)(p,r,i-2,&color1);
	(*colorfunc)(p,r,i-1,&color2);

	line.vert[0].r = color1.r;
	line.vert[0].g = color1.g;
	line.vert[0].b = color1.b;
	line.vert[0].a = color1.a;

	line.vert[1].r = color2.r;
	line.vert[1].g = color2.g;
	line.vert[1].b = color2.b;
	line.vert[1].a = color2.a;

	kmemcpy(&(line.vert[2]),&(line.vert[0]),vertsize);

	(*init_shaderfuncs[KSHADER_NO_LIGHT])(&(r->attribs),&color1,NULL,NULL,&line);
	
	/** need to do line clipping here. **/
	line_draw(&line,&work_window,shaderfuncs[KSHADER_NO_LIGHT],&color1,&color2); 

    }
}

void polyline_con(rmonster_prim *p,
		  int xs, int ys,
		  vertex_3d *obj_space_verts,
		  vertex_3d *view_space_verts,
		  vertex_3d *norms,
		  float *unused,
		  int projection,
		  int opaque_status,
		  rmonsterobj *r,
		  int hither,
		  int yon)
{
    /**
      * NOTES: at present (8 July 93), per vertex line color is supported
      * at all levels except at the lowest; in other words, the renderer.
      *
      * 8/17/94 this routine hasn't been as highly vectorized, as, say
      * qmesh or the triangle primitives.
    **/
    
    float xscale,xd,yscale,yd,zscale;
    float xview,yview,zview;
    float epsilon = 0.001;
    float xscreen,yscreen,zscreen;
    int use_global_obj_color;
    float obj_rgba[4],color_v1[4],color_v2[4];
    int i,j;
    Poly line_in,line_clipped;
    int px1,py1,px2,py2;
    surface_colors_rgba *tc;
    void (**shaderfuncs)();
    surface_colors_rgba color1,color2;
    int layout_type;
    float *f;
    int has_alpha;
    float zmin;
    int status;
    void (*colorfunc)();
    int vertsize;

    vertsize = sizeof(Poly_vert);

    has_alpha = r->has_alpha;
    
    get_zmin(&zmin);

    if (p->nverts == p->ncolors)
	layout_type = KPER_VERTEX;
    else
	layout_type = KPER_LINE;
    
    if (opaque_status == TRUE)
	shaderfuncs = opaque_shaderfuncs;
    else
	shaderfuncs = transparent_shaderfuncs;

    work_window.x0 = work_window.y0 = 0;
    work_window.x1 = xs-1;
    work_window.y1 = ys-1;
    
    /** see Jim Blinn's Corner, IEEE CG&A, mmm/yyy **/
    xscale = (xs - epsilon)/2.0;
    xd = xscale;
    yscale = (ys - epsilon)/2.0;  /* the aspect ratio term goes here. */
    yd = yscale;
    zscale = 1.;

    line_clipped.n = 2;
    line_in.n = 2;
    line_clipped.mask = line_in.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz);

    has_alpha = r->has_alpha;

    if (p->ncolors == 0)
	colorfunc = polyline_con_no_color;
    else if (has_alpha)
    {
	if (layout_type == KPER_LINE)
	    colorfunc = polyline_con_alpha_per_segment;
	else /* assume KPER_VERTEX */
	    colorfunc = polyline_con_alpha_per_vertex;
    }
    else
    {
	if (layout_type == KPER_LINE)
	    colorfunc = polyline_con_noalpha_per_segment;
	else /* assume KPER_VERTEX */
	    colorfunc = polyline_con_noalpha_per_segment;
    }

    for (i=0;i<p->nverts;i++)
    {
	if (i == 0)
	{
	    line_in.vert[0].x = view_space_verts[i].v[0];
	    line_in.vert[0].y = view_space_verts[i].v[1];
	    line_in.vert[0].z = view_space_verts[i].v[2];
	    
	    (*colorfunc)(p,r,i,color_v1);
	    continue;
	}
	else
	{
	    line_in.vert[1].x = view_space_verts[i].v[0];
	    line_in.vert[1].y = view_space_verts[i].v[1];
	    line_in.vert[1].z = view_space_verts[i].v[2];
	    
	    (*colorfunc)(p,r,i,color_v2);
	}

	status = (*p->clipfunc)(p,&line_in,&line_clipped,projection,&zmin,hither,yon);
	if (status != CHILL)
	{
	    memcpy((char *)&(line_in.vert[0]),(char *)&(line_in.vert[1]),sizeof(Poly_vert));
	    kmemcpy(color_v1,color_v2,sizeof(float)*4);
	    continue;
	}

	for (j=0;j<2;j++)
	{
	    zview = line_clipped.vert[j].z;
	    if (projection == KPROJECTION_PERSPECTIVE)
	    {
		xview = (line_clipped.vert[j].x)/zview;
		yview = (line_clipped.vert[j].y)/zview;
	    }
	    else
	    {
		xview = line_clipped.vert[j].x;
		yview = line_clipped.vert[j].y;
	    }

	    line_clipped.vert[j].sx = xview*xscale + xd;
	    line_clipped.vert[j].sy = yview*yscale + yd;
	    line_clipped.vert[j].sz = zview;
	}

	if (i != 0)
	{
	    line_clipped.vert[0].r = color_v1[0];
	    line_clipped.vert[0].g = color_v1[1];
	    line_clipped.vert[0].b = color_v1[2];
	    line_clipped.vert[0].a = color_v1[3];
	    
	    line_clipped.vert[1].r = color_v2[0];
	    line_clipped.vert[1].g = color_v2[1];
	    line_clipped.vert[1].b = color_v2[2];
	    line_clipped.vert[1].a = color_v2[3];

	    (*init_shaderfuncs[KSHADER_NO_LIGHT])(&(r->attribs),&color1,NULL,NULL,&line_clipped);

	    line_draw(&line_clipped,&work_window,shaderfuncs[KSHADER_NO_LIGHT],&color1,&color2); 
	    memcpy((char *)&(line_in.vert[0]),(char *)&(line_in.vert[1]),sizeof(Poly_vert));
	    kmemcpy(color_v1,color_v2,sizeof(float)*4);
	}
    }
}

static void qmesh_no_color PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void qmesh_alpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,surface_colors_rgba *));
static void qmesh_alpha_per_face PROTO((rmonster_prim *,rmonsterobj *,int,surface_colors_rgba *));
static void qmesh_noalpha_per_vertex PROTO((rmonster_prim *,rmonsterobj *,int,float *));
static void qmesh_noalpha_per_face PROTO((rmonster_prim *,rmonsterobj *,int,float *));
    
void qmesh(rmonster_prim *p,
	   int xs,int ys,
	   vertex_3d *obj_space_verts,
	   vertex_3d *view_space_verts,
	   vertex_3d *norms,
	   float *unused,
	   int projection,
	   int opaque_status,
	   rmonsterobj *r,
	   int hither,
	   int yon)
{

    /* for each triangle, set up and call the scanline renderer. */
    int i,j,k;
    int iu,iv;
    int layout_type,has_alpha;
    int *c;  /* connectivity index pointer */
    int *cind;  /* connectivity incrementable index pointer */
    int csize,usize,vsize;
    int use_global_obj_color = 0;
    surface_colors_rgba *c1,*c2,*c3,blend;
    int has_normals;
    float zscale;
    float xview,yview,zview;
    float xscreen,yscreen,zscreen;
    float xscale,yscale,xd,yd;
    float epsilon = 0.001;
    float zmin;
    int in_use_shader;
    void (**shaderfuncs)();
    void (*colorfunc)();
    surface_colors_rgba *work_colors,*work_colors2;
    int status;
    
    has_alpha = r->has_alpha;

    usize = p->mesh_dims[0];
    vsize = p->mesh_dims[1];

    if (opaque_status == TRUE)
	shaderfuncs = opaque_shaderfuncs;
    else
	shaderfuncs = transparent_shaderfuncs;

    get_zmin(&zmin);
    
    xscale = (xs - epsilon)/2.0;
    xd = xscale;
    yscale = (ys - epsilon)/2.0;  /* the aspect ratio term goes here. */
    yd = yscale;
    zscale = 1.;

    work_window.x0 = work_window.y0 = 0;
    work_window.x1 = xs-1;
    work_window.y1 = ys-1;
    
    work_poly.n = 3;

    /* Z's will be converted from the range 0..1 to the range
       0..imageres/2.  this makes lighting calculations work, all
       of which are performed in screen space. */
    
    if (p->ncolors == 0)
	use_global_obj_color = 1;
    if (norms == NULL)
	has_normals = 0;
    else
	has_normals = 1;

    in_use_shader = r->shader;
    
    /* the mask will be set to reflect various things such as
       flat/gouroud shading, presence of texture, etc. */

    work_poly.mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz);
	
    if (in_use_shader == KSHADER_GOUROUD)
    {
	if (opaque_status == TRUE)
	    work_poly.mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b);
	else
	    work_poly.mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b) |
		POLY_MASK(a);;
    }

    if ((p->nverts == p->ncolors) && (p->ncolors != 0))
	layout_type = KPER_VERTEX;
    else
	layout_type = KPER_FACE;

    if (p->ncolors == 0)
	colorfunc = qmesh_no_color;
    else if (has_alpha)
    {
	if (layout_type == KPER_VERTEX)
	    colorfunc = qmesh_alpha_per_vertex;
	else /* assume KPER_LINE */
	    colorfunc = qmesh_alpha_per_face;
    }
    else
    {
	if (layout_type == KPER_VERTEX)
	    colorfunc = qmesh_noalpha_per_vertex;
	else /* assume KPER_FACE */
	    colorfunc = qmesh_noalpha_per_face;
    }

    
    /* build a "connectivity" list.  this list consists of a bunch
       of integers used as indices into the 2D array of vertices and
       colors of the qmesh, effectively linearizing the 2D object
       into a single list of triangles. */

    csize = ((usize-1)*2);  /* # of triangles per row */
    csize *= (vsize-1);     /* times number of rows */
    csize *= 3;             /* times 3 vertices per triangle; */

    /* should build this at object read-in time?? */
    c = (int *)kmalloc(sizeof(int)*csize);

    csize = 0; /* use as temp index */

    for (iv=0;iv<vsize-1;iv++)  
    {
	for (iu=0;iu<usize-1;iu++)
	{
	    c[csize++] = iu + iv*usize;
	    c[csize++] = iu+1 + iv*usize;
	    c[csize++] = iu+1 + (iv+1)*usize;

	    c[csize++] = iu+1 + (iv+1)*usize;
	    c[csize++] = iu + (iv+1)*usize;
	    c[csize++] = iu + iv*usize;
	}
    }

    /**
      * what we want to do now is to compute the colors for each
      * face or vertex.
      *
      * if per-face shading is specified, need to compute the center
      * point on the face, along with an appropriate normal.
      *
      * it is assumed that the object is passed in with normals, as
      * they are computed at read-in time if not already present.
    **/

    if (layout_type == KPER_FACE)
	;
    else
    {
	work_colors = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*usize*vsize);
	work_colors2 = NULL;

	/* set up the work colors array */

	for (i=0;i<usize*vsize;i++)
	    (*colorfunc)(p,r,i,work_colors+i);

	/* compute the per-vertex shade */
	vertex_shader(obj_space_verts,norms,usize*vsize,
		      work_colors,&work_colors2,&(r->attribs),
		      &(r->colors.a),
		      r->shader);
    }

    work_poly.n = 3;

    cind = c;
    for (i=0;i<((usize-1)*2*(vsize-1));i++)
    {
	for (j=0;j<3;j++)  /* for each triangle vertex, load info */
	{
	    /* load xyz's */
	    xview = view_space_verts[*cind].v[0];
	    yview = view_space_verts[*cind].v[1];
	    zview = view_space_verts[*cind].v[2];

	    if (projection == KPROJECTION_PERSPECTIVE)
	    {
		xview = xview/zview;
		yview = yview/zview;
	    }
	    
	    xscreen = xview*xscale + xd;
	    yscreen = yview*yscale + yd;
	    zscreen = zview;
		
	    work_poly.vert[j].sx = xscreen;
	    work_poly.vert[j].sy = yscreen;
	    work_poly.vert[j].sz = zscreen;
#if 0
	    work_poly.vert[j].x = obj_space_verts[*cind].v[0];
	    work_poly.vert[j].y = obj_space_verts[*cind].v[1];
	    work_poly.vert[j].z = obj_space_verts[*cind].v[2];
#endif
	    work_poly.vert[j].x = xview;
	    work_poly.vert[j].y = yview;
	    work_poly.vert[j].z = zview;
	    
	    work_poly.vert[j].r = work_colors2[*cind].r;
	    work_poly.vert[j].g = work_colors2[*cind].g;
	    work_poly.vert[j].b = work_colors2[*cind].b;
	    work_poly.vert[j].a = work_colors2[*cind].a;

	    work_poly.vert[j].nx = norms[*cind].v[0];
	    work_poly.vert[j].ny = norms[*cind].v[1];
	    work_poly.vert[j].nz = norms[*cind].v[2];

	    cind++;
	}
	(*init_shaderfuncs[in_use_shader])(&(r->attribs),&(r->colors),NULL,NULL,&work_poly);
	status = (*p->clipfunc)(&work_poly,projection,&zmin,hither,yon);
	if (status == CHILL)
	    poly_scan(&work_poly,&work_window,shaderfuncs[in_use_shader]);
    }
    
    kfree(work_colors);
    kfree(work_colors2);
    kfree(c);
}

void omesh(rmonster_prim *p,
	   int xs, int ys,
	   vertex_3d *obj_space_verts,
	   vertex_3d *view_space_verts,
	   vertex_3d *norms,
	   float *unused2,
	   int projection,
	   int opaque_status,
	   rmonsterobj *r,
	   int hither,
	   int yon)
{
    float xscale,yscale,zscale,xd,yd;
    float epsilon = 0.0001;
    float xview,yview,zview;
    int i;
    int loc_type;
    int has_alpha;
    void (**shaderfuncs)();
    void (*colorfunc)();
    vertex_3d *rv;
    surface_colors_rgba *color;
    surface_colors_rgba *work_colors,*work_colors2=NULL;
    int usize,vsize,wsize;
    
    has_alpha = r->has_alpha;

    if (opaque_status == TRUE)
	shaderfuncs = opaque_shaderfuncs;
    else
	shaderfuncs = transparent_shaderfuncs;

    /* assume for now curvilinear location data. */
    
    xscale = (xs - epsilon)/2.0;
    xd = xscale;
    yscale = (ys - epsilon)/2.0;  /* the aspect ratio term goes here. */
    yd = yscale;
    zscale = 1.;

    /**
      * load up colors.
      * 8/18/94 assume per vertex RGBA. need to build general purpose
      * color grabbing function.
    **/

    work_colors = p->rgba_color_list;
    usize = p->mesh_dims[0];
    vsize = p->mesh_dims[1];
    wsize = p->mesh_dims[2];

    /* compute the per-vertex shade */
    volume_vertex_shader(obj_space_verts,norms,usize*vsize*wsize,
			 work_colors,&work_colors2,&(r->attribs),
			 &(r->colors.a),
			 r->shader);
    
    rv = view_space_verts;

    /**
      * do in-situ view-space xformation of all view space points.
    **/
    
    for (i=0;i<p->nverts;i++,rv++)
    {
	xview = rv->v[0];
	yview = rv->v[1];
	zview = rv->v[2];
	
	if (projection == KPROJECTION_PERSPECTIVE)
	{
	    xview = xview/zview;
	    yview = yview/zview;
	}
	
	rv->v[0] = xview*xscale + xd;
	rv->v[1] = yview*yscale + yd;
	rv->v[2] = zview;
    }

    /** HACK HACK HACK **/
    color = p->rgba_color_list;

    
    volrender(r,p,view_space_verts,work_colors2,xs,ys);
    kfree(work_colors2);
}

void polyhedron(rmonster_prim *p,
		int xs, int ys,
		vertex_3d *obj_space_verts,
		vertex_3d *view_space_verts,
		vertex_3d *normals,
		float *unused,
		int projection,
		int opaque_status,
		rmonsterobj *r,
		int hither,
		int yon)
{
}

static int ntriangles[4] = {8,32,128,512};

void sphere(rmonster_prim *p,
	    int xs, int ys,
	    vertex_3d *obj_space_verts,
	    vertex_3d *view_space_verts,
	    vertex_3d *unused,
	    float *radii,
	    int projection,
	    int opaque_status,
	    rmonsterobj *r,
	    int hither,
	    int yon)
{
    int i,j,k;
    float xscreen,yscreen,zscreen;
    int ir,ig,ib;
    int use_global_obj_color=0;
    float xscale,yscale,zscale,xd,yd;
    float epsilon=0.001;
    float xview,yview,zview;
    vertex_3d wnormal,pt;
    surface_colors_rgba blend,*tc;
    float color_ptr[4];
    float obj_rgba[4];
    int radius_in_screen_pixels, subdivisions;
    Poly *sphere_tess;
    vertex_3d center_screen,center_view;
    float screen_radius,view_radius;
    double mag;
    Poly **spheres,**work_area,*pp;
    int list_index;
    int in_use_shader;
    void (**shaderfuncs)();
    int has_alpha;
    float zmin,zproj;
    float t;
    surface_colors_rgba sphere_color;
    int this_sphere;
    int status;

    get_zproj(&zproj);
    get_zmin(&zmin);

    has_alpha = r->has_alpha;

    if (opaque_status == TRUE)
	shaderfuncs = opaque_shaderfuncs;
    else
	shaderfuncs = transparent_shaderfuncs;
    
    in_use_shader = r->shader;
    
    work_window.x0 = work_window.y0 = 0;
    work_window.x1 = xs-1;
    work_window.y1 = ys-1;

    xscale = (xs - epsilon)/2.0;
    xd = xscale;
    yscale = (ys - epsilon)/2.0;  /* the aspect ratio term goes here. */
    yd = yscale;
    zscale = 1.;

    if (p->ncolors == 0)
    {
	use_global_obj_color = 1;
	obj_rgba[0] = r->colors.r;
	obj_rgba[1] = r->colors.g;
	obj_rgba[2] = r->colors.b;
	obj_rgba[3] = r->colors.a;
    }
    else
	use_global_obj_color = 0;

    /**
      * initialize the sphere tesselator.  what we do here is construct
      * polygon lists which contain tesselated unit spheres.  each
      * list contains more and more triangles which better approximate
      * the surface of the sphere.  later on in this routine, we choose
      * the resolution which is most appropriate as a function of
      * the number of pixels covered by each sphere.  we do it this
      * way so that the sphere tesselator is run once for a range
      * of resolutions, rather than run for each sphere that is input.
    **/

    spheres = (Poly **)malloc(sizeof(Poly *)*4);
    work_area = (Poly **)malloc(sizeof(Poly *)*4);
    
    for (i=0;i<4;i++)
    {
	pp = (Poly *)malloc(sizeof(Poly)*ntriangles[i]);
	spheres[i] = pp;
	work_area[i] = (Poly *)malloc(sizeof(Poly)*ntriangles[i]);
	
	sphere_tesselator(i+1,obj_rgba,spheres[i]);
    }

    if (p->ncolors == 0)
	use_global_obj_color = 1;
    else
	use_global_obj_color = 0;
    
    this_sphere = 0;
    for (i=0;i<p->nverts;i++,this_sphere++)
    {
	/* load xyz's */

	for (j=0;j<3;j++)
	    center_view.v[j] = view_space_verts[i].v[j];
	
	if (projection == KPROJECTION_PERSPECTIVE)
	{
	    t = 1./center_view.v[2];
	    center_view.v[0] = center_view.v[0] * t;
	    center_view.v[1] = center_view.v[1] * t;
	    /**
	      * the radius of the sphere is scaled according to it's
	      * depth - for a given radius value, it appears "smaller"
	      * when it's further away; it gets larger as it gets
	      * closer.
	    **/
	    view_radius = radii[i] * (zproj * t);
	}
	else
	    view_radius = radii[i];

	center_screen.v[0] = center_view.v[0]*xscale+xd;
	center_screen.v[1] = center_view.v[1]*yscale+yd;
	center_screen.v[2] = center_view.v[2];

	screen_radius = view_radius * xscale * 2.;

	/**
	  * the following values were chosen emperically.  we are setting
	  * the relationship between the screen-size of the sphere and the
	  * number of triangles used to approximate it.
	**/

	if (screen_radius < 3.)
	    list_index = 0;
	else if (screen_radius < 6.)
	    list_index = 1;
	else if (screen_radius < 12.)
	    list_index = 2;
	else /* hack */
	    list_index = 3;

	/**
	 * subdivisions: 1 == 8 triangles,
	 *               2 == 32, 3 == 128, 4 == 512, etc.
	 **/

	/**
	  * now, copy over from tesselated sphere area to work
	  * area, incorporating screen/view centers and scale values.
	**/

	if (use_global_obj_color)
	{
	    map_to_work_area(spheres[list_index],work_area[list_index],
			     &center_screen,&center_view,screen_radius,
			     view_radius,&(r->attribs),&(r->colors),
			     &(r->colors.a),ntriangles[list_index],r->shader);
	}
	else
	{
	    if (p->rgba_color_list)
	        kmemcpy(&sphere_color,(p->rgba_color_list+i),sizeof(surface_colors_rgba));
	    else
	    {
		memcpy(&sphere_color,(p->rgb_color_list+i),sizeof(surface_colors_rgb));
		sphere_color.a = 1.;
		
	    }
	    
	    map_to_work_area(spheres[list_index],work_area[list_index],
			     &center_screen,&center_view,screen_radius,
			     view_radius,&(r->attribs),&sphere_color,
			     &(r->colors.a),ntriangles[list_index],r->shader);
	}

	sphere_tess = work_area[list_index];

	
	for (j=0;j<ntriangles[list_index];j++)
	{
	    if (cheap_sphere_clip(view_space_verts+this_sphere,&zmin,
				  projection))
	    {
	        (*init_shaderfuncs[in_use_shader])(&(r->attribs),&(r->colors),NULL,
						   NULL,sphere_tess+j);
		
		status = (*p->clipfunc)(sphere_tess+j,projection,&zmin,hither,yon);
		if (status == CHILL)
		    poly_scan(sphere_tess+j,&work_window,shaderfuncs[in_use_shader]);
	    }
	}
    }

}

void dirpoint(rmonster_prim *p,
	      int xs, int ys,
	      vertex_3d *obj_space_verts,
	      vertex_3d *view_space_verts,
	      vertex_3d *norms,
	      float *unused,
	      int projection,
	      int transparent,
	      rmonsterobj *r,
	      int hither,
	      int yon)
{
}

void text(rmonster_prim *p,
	  int xs, int ys,
	  vertex_3d *obj_space_verts,
	  vertex_3d *view_space_verts,
	  vertex_3d *norms,
	  float *unused,
	  int projection,
	  int transparent,
	  rmonsterobj *r,
	  int hither,
	  int yon)
{

}

/** misc support routines. **/

static void
init_sphere_triangles(Poly *p,
		      int n,
		      int shader,
		      int opaque_status)
{
    register Poly *pp;
    int i;

    pp = p;
    
    for (i=0;i<n;i++,pp++)
    {
	pp->mask  = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz);
	
	if (shader == KSHADER_GOUROUD)
	{
	    if (opaque_status == FALSE)
		pp->mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b) | POLY_MASK(a);
	    else
		pp->mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b);

	}
    }
}

static int
map_to_work_area(Poly *s,
		 Poly *d,
		 vertex_3d *center_screen,
		 vertex_3d *center_view,
		 float screen_radius,
		 float view_radius,
		 surface_attribs *attribs,
		 surface_colors_rgba *color,
		 float *global_alpha,
		 int n,
		 int shader)

{
    register Poly *ps,*pd;
    int i,j,offset;
    vertex_3d *work_v,*work_n;
    int work_nverts;
    surface_colors_rgba *work_in,*work_out=NULL;

    work_v = (vertex_3d *)kmalloc(sizeof(vertex_3d)*n*3);
    work_n = (vertex_3d *)kmalloc(sizeof(vertex_3d)*n*3);
    work_in = (surface_colors_rgba *)kmalloc(sizeof(surface_colors_rgba)*n*3);

    ps = s;
    offset = 0;
    for (i=0;i<n;i++,ps++)
    {
        /* copy over verts and normals from poly list to work buffer */
        for (j=0;j<3;j++)
	{
	    work_v[offset].v[0] = ps->vert[j].x * view_radius + center_view->v[0];
	    work_v[offset].v[1] = ps->vert[j].y * view_radius + center_view->v[1];
	    work_v[offset].v[2] = ps->vert[j].z * view_radius + center_view->v[2];
	    work_n[offset].v[0] = ps->vert[j].nx;
	    work_n[offset].v[1] = ps->vert[j].ny;
	    work_n[offset].v[2] = ps->vert[j].nz;

	    offset++;
	}
    }

    for (i=0;i<n*3;i++)
      kmemcpy(work_in+i,color,sizeof(surface_colors_rgba));

    /** call the routine which shades the sucka's **/
    vertex_shader(work_v,work_n,n*3,work_in,&work_out,attribs,
		  global_alpha,shader);

#if 0
    kfree(work_in);
    work_in = work_out;
#endif

    ps = s;
    pd = d;
    offset = 0;
    for (i=0;i<n;i++,ps++,pd++)
    {
	pd->n = ps->n;
	
	pd->mask = POLY_MASK(sx) | POLY_MASK(sy) | POLY_MASK(sz);

	if (shader == KSHADER_GOUROUD)
	{
	    pd->mask |= POLY_MASK(r) | POLY_MASK(g) | POLY_MASK(b) | POLY_MASK(a);
	}
	
/*	for (j=0;j<ps->n;j++) */
	for (j=0;j<3;j++)
	{
	    pd->vert[j].x = ps->vert[j].x * view_radius + center_view->v[0];
	    pd->vert[j].y = ps->vert[j].y * view_radius + center_view->v[1];
	    pd->vert[j].z = ps->vert[j].z * view_radius + center_view->v[2];
	    
	    pd->vert[j].sx = ps->vert[j].sx * screen_radius + center_screen->v[0];
	    pd->vert[j].sy = ps->vert[j].sy * screen_radius + center_screen->v[1];
	    pd->vert[j].sz = ps->vert[j].sz * screen_radius + center_screen->v[2];
	    pd->vert[j].nx = ps->vert[j].nx;
	    pd->vert[j].ny = ps->vert[j].ny;
	    pd->vert[j].nz = ps->vert[j].nz;
	    
	    pd->vert[j].r = work_out[offset].r;
	    pd->vert[j].g = work_out[offset].g;
	    pd->vert[j].b = work_out[offset].b;
	    pd->vert[j].a = work_out[offset].a;
	    offset++;
	}
    }

    kfree(work_out);
    kfree(work_in);
    kfree(work_v);
    kfree(work_n);
    return(CHILL);
}

static void
polyline_con_no_color(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    *rgba++ = r->colors.r;
    *rgba++ = r->colors.g;
    *rgba++ = r->colors.b;
    *rgba = r->colors.a;
}

static void
polyline_con_alpha_per_vertex (rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    float *f;
    f = (float *)(p->rgba_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
}

static void
polyline_con_alpha_per_segment(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    float *f;
    if (n != p->nverts-1)
    {
	f = (float *)(p->rgba_color_list+n);
	*rgba++ = *f++;
	*rgba++ = *f++;
	*rgba++ = *f++;
	*rgba = *f;
    }
}

static void
polyline_con_noalpha_per_vertex (rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    float *f;
    f = (float *)(p->rgb_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.0;
}

static void
polyline_con_noalpha_per_segment(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    float *f;

    if (n != p->nverts-1)  /* there's nothing defined at location
			      colors[p->nverts-1] to grab */
    {
	f = (float *)(p->rgb_color_list+n);
	*rgba++ = *f++;
	*rgba++ = *f++;
	*rgba++ = *f;
	*rgba = 1.0;
    }
}

static void
polyline_dis_no_color (rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    *rgba++ = r->colors.r;
    *rgba++ = r->colors.g;
    *rgba++ = r->colors.b;
    *rgba = r->colors.a;
}

static void
polyline_dis_alpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgba_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
}

static void
polyline_dis_alpha_per_segment(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgba_color_list+(n/2));
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
}

static void
polyline_dis_noalpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgb_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}

static void
polyline_dis_noalpha_per_segment(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgb_color_list+(n/2));
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}

/**
  * routines for handling color in disjoint polytri prims.
**/
static void
polytri_dis_no_color (rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    *rgba++ = r->colors.r;
    *rgba++ = r->colors.g;
    *rgba++ = r->colors.b;
    *rgba = r->colors.a;
}

static void
polytri_dis_alpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgba_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
}

static void
polytri_dis_alpha_per_face(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgba_color_list+(n/3));
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
}

static void
polytri_dis_noalpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgb_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}

static void
polytri_dis_noalpha_per_face(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgb_color_list+(n/3));
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}


/**
  * routines for handling color in qmesh prims
**/
static void
qmesh_no_color (rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    *rgba++ = r->colors.r;
    *rgba++ = r->colors.g;
    *rgba++ = r->colors.b;
    *rgba = r->colors.a;
}

static void
qmesh_alpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,surface_colors_rgba *rgba)
{
#if 0
    register float *f;
    f = (float *)(p->rgba_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
#endif
    *rgba = p->rgba_color_list[n];
}

static void
qmesh_alpha_per_face(rmonster_prim *p,rmonsterobj *r,int n,
		     surface_colors_rgba *rgba)
{
#if 0
    register float *f;
    f = (float *)(p->rgba_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba = *f;
#endif
    *rgba = p->rgba_color_list[n];
}

static void
qmesh_noalpha_per_vertex(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgb_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}

static void
qmesh_noalpha_per_face(rmonster_prim *p,rmonsterobj *r,int n,float *rgba)
{
    register float *f;
    f = (float *)(p->rgb_color_list+n);
    *rgba++ = *f++;
    *rgba++ = *f++;
    *rgba++ = *f;
    *rgba = 1.;
}


static void
polytri_dis_per_vertex_normals(vertex_3d *s,
			       vertex_3d *d,
			       int index)
{
    kmemcpy(d+index,s+index,sizeof(vertex_3d));
}

static void
polytri_dis_per_face_normals(vertex_3d *s,
			     vertex_3d *d,
			     int index)
{
    int index2;

    index2 = index/3;
    kmemcpy(d+index,s+index2,sizeof(vertex_3d));
}
