
/*********************************************************************

Copyright (C) 1993, 1994, 1995, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory
		Berkeley, California

  "this software is 100% hand-crafted by a human being in the USA"

jan 1995, wes.
    add code to handle auto-sizing of camera aperture as well as
    reset to startup defaults for the view parameters.

*********************************************************************/



#include <design.h>
#include <geometry.h>

#include "rmonster.h"
#include "camera.h"
#include "rmatrix.h"
#include "rmonster_objs.h"
#include "imaging.h"

static vertex_3d s_eye,s_at,s_up;
static float s_fov,s_user_hither,s_user_yon;
static float s_rmonster_hither,s_rmonster_yon;
static int s_projection;
static matrix4x4 s_view,s_view_rotation_matrix;
static surface_colors_rgb background_rgb = {DEFAULT_BACKGROUND_RED,
					    DEFAULT_BACKGROUND_GREEN,
					    DEFAULT_BACKGROUND_BLUE};
static float s_zmin,s_zproj;
static float s_xyscale,s_depthscale;
static int c_yon_clip_flag=TRUE,c_hither_clip_flag=TRUE;

static vertex_3d startup_eye,startup_at,startup_up;
static float startup_fov,startup_user_hither,startup_user_yon;
static int startup_user_projection;

int compute_parallel_projection PROTO((matrix4x4 *, vertex_3d *, vertex_3d *, vertex_3d *, float *, float *, float *, float *, float *));

int
init_camera(clui_info_struct *rmc)
{
    /**
      * set default parms for all camera parameters.  these
      * default values are in geom.h.
      *
      * note that at present, this routine ignores all command line args.
    **/
    int temp_int;

    /* 3 april 1995  - the default values (startup) come from the clui,
     not a header file. */
    startup_eye.v[0] = rmc->eye_x_float;
    startup_eye.v[1] = rmc->eye_y_float;
    startup_eye.v[2] = rmc->eye_z_float;
    set_eye(&startup_eye);

    startup_at.v[0] = rmc->at_x_float;
    startup_at.v[1] = rmc->at_y_float;
    startup_at.v[2] = rmc->at_z_float;
    set_at(&startup_at);

    startup_up.v[0] = rmc->up_x_float;
    startup_up.v[1] = rmc->up_y_float;
    startup_up.v[2] = rmc->up_z_float;
    set_up(&startup_up);

    startup_user_hither = rmc->hither_float;
    set_user_hither(startup_user_hither);

    startup_user_yon = rmc->yon_float;
    set_user_yon(startup_user_yon);

    temp_int = rmc->bc_logic;
    set_yon_clip_flag(temp_int);

    startup_fov = DEFAULT_FOV;
    set_fov(startup_fov);

    startup_user_projection = (rmc->proj_list == 0) ? KPROJECTION_PERSPECTIVE:KPROJECTION_PARALLEL;
    set_projection(startup_user_projection);

    compute_view_xform();
    
    temp_int = rmc->norm_logic;

    /* -- dont set normalized view if we have no input -- */
    if (temp_int == 1 && 
	(rmc->i1_flag || rmc->i2_flag || rmc->i3_flag || rmc->i4_flag))
	set_normalized_view();
    
    return(CHILL);
}

int
software_set_camera()
{
    compute_view_xform();
    return(CHILL);
}

int
compute_view_xform()
{
    vertex_3d l_eye,l_at,l_up;
    float l_fov,l_hither,l_yon,l_zmin,l_zproj;
    int l_proj;
    matrix4x4 l_view;

    get_eye(&l_eye);
    get_at(&l_at);
    get_up(&l_up);
    get_fov(&l_fov);
    get_rmonster_hither(&l_hither);
    get_rmonster_yon(&l_yon);
    /* note: interpretation of hither/yon changed on 3 april 95. */
    get_projection(&l_proj);
    identity_4x4(&l_view);

    if (l_proj == KPROJECTION_PARALLEL)
	compute_parallel_projection(&l_view,&l_eye,&l_at,&l_up,&l_fov,
				    &l_hither,&l_yon,&l_zmin,&l_zproj);
    else
	compute_perspective_projection(&l_view,&l_eye,&l_at,&l_up,&l_fov,
				       &l_hither,&l_yon,&l_zmin,&l_zproj);

    set_view_xform(&l_view);
    set_zmin(l_zmin);
    set_zproj(l_zproj);
    return(CHILL);
}

int
compute_parallel_projection(matrix4x4 *view,
			    vertex_3d *eye,
			    vertex_3d *at,
			    vertex_3d *up,
			    float *fov,
			    float *hither,
			    float *yon,
			    float *zmin,
			    float *zproj)
{
    vector4 vrp,vrp_not;
    vector4 v_at,v_eye,v_up;
    double d,d2,a,b,mag;
    vertex_3d vert;
    matrix4x4 trans,rot_all,rtol,scale,cumulative,trans_par;
    float front,back;
    vector4 p1p2,p1p3,pcross,rx,ry,rz;
    int i;

    for (i=0;i<3;i++)
    {
	v_eye.v[i] = eye->v[i];
	v_at.v[i] = at->v[i];
	v_up.v[i] = up->v[i];
    }
    v_eye.v[3] = v_at.v[3] = v_up.v[3] = 1.0;

    /* taken from Foley & van Dam, version 1 (w/ corrections) 1983. */

    vrp.v[0] = at->v[0];
    vrp.v[1] = at->v[1];
    vrp.v[2] = at->v[2];
    vrp.v[3] = 1.0;

    /* 1. translate view reference point to the origin. */
    identity_4x4(&trans);

    for (i=0;i<3;i++)
	trans.m[3][i] = -1. * vrp.v[i];
    
    matrix_4x4copy(&cumulative,&trans);
    point_xfrm(&vrp,&trans,&vrp_not);

    /* 2. rotate so that VPN becomes parallel to negative Z axis in
       right handed world coord sys.  this is a mess.  */

    vector_4sub(&v_at,&v_eye,&p1p2); 
    vector_4mag(&p1p2,&d);
    d = 1./d;
    rz.v[0] = -1. * p1p2.v[0] * d;
    rz.v[1] = -1. * p1p2.v[1] * d;
    rz.v[2] = -1. * p1p2.v[2] * d;
    rz.v[3] = 1.;

    vector_4copy(&p1p3,&v_up);
    vector_4cross(&p1p2,&p1p3,&pcross);
    vector_4mag(&pcross,&d);
    d = 1./d;
    rx.v[0] = pcross.v[0] * d;
    rx.v[1] = pcross.v[1] * d;
    rx.v[2] = pcross.v[2] * d;
    rx.v[3] = 1.;

    vector_4cross(&rz,&rx,&ry);
    vector_4mag(&ry,&d);
    d = 1./d;
    for (i=0;i<3;i++)
	ry.v[i] *= d;
    
    identity_4x4(&rot_all);
    
    for (i=0;i<3;i++)
	rot_all.m[i][0] = rx.v[i];
    
    for (i=0;i<3;i++)
	rot_all.m[i][1] = ry.v[i];
    
    for (i=0;i<3;i++)
	rot_all.m[i][2] = rz.v[i];

    mmul_4x4(&cumulative,&rot_all,&cumulative);
    
    /* 3. change from right to left handed coord system. */
    identity_4x4(&rtol);
    rtol.m[2][2] = -1.;
    
    mmul_4x4(&cumulative,&rtol,&cumulative);
    matrix_4x4copy(&s_view_rotation_matrix,&cumulative);

    /* 4.  we induce no shear as vpn == eye-vrp */

    /* 5. scale so the view volume becomes to the unit cube,
       then scaled into [-1..1] in x and y, 0..1 in Z */

    /* 5a. (bethel method) compute window half-width and height
       using a little trig - a function of the FOV parameter and
       the distance from the eye to "at". */

    point_xfrm(&vrp,&cumulative,&vrp_not);

    d2 = *fov/2.;
    d = 90. - d2;
    d = DEGREES_TO_RADIANS(d);
    d = ksin(d);

#if 0
    b = vrp_not.v[2];  /* this should be somewhere on the +z axis */
#endif
    npoint_diff((float *)eye,(float *)at,(float *)(&vert),1);
    vertex_unit(&vert,&mag);
    b = mag; 

    d2 = DEGREES_TO_RADIANS(d2);
    d2 = ksin(d2);

    a = b * d2 / d;  /* this is window half-width.  at some point,
			the aspect ratio may be added to produce differing
			widths and heights */

    front =  -1. * *hither;
    back =  *yon;

    identity_4x4(&scale);

    /**
      * the scale factor is twice what it should be.  f&v.d. calls
      * for 1./(umax-umin).  what we want is 2./(umax-umin) so that
      * the only difference between the perspective and parallel
      * transformations is a perspective divide.  this scale
      * factor maps the view volume into -1..1 in [x,y] and 0..1 in z.
    **/
    scale.m[0][0] = ((2.)/(2.*a));  /* this formulation produces */
    scale.m[1][1] = ((2.)/(2.*a));  /* a square window */

    /** just do the scale in x & y for now...this puts things in
      the range -1..1 **/
    
/*     scale.m[2][2] = 1./(2.*(back-front)); */
/*    scale.m[2][2] = 1./(back-front); */
    
    mmul_4x4(&cumulative,&scale,&cumulative);

    /* now, valid primitives are defined in the range -front .. back.
       we want to translate in Z so that -front becomes Z==0 */
    
    identity_4x4(&trans_par);
    trans_par.m[3][2] = -1. * front;
    mmul_4x4(&cumulative,&trans_par,&cumulative);

    /* now, scale so that stuff in Z, now in the range 0..(front+back)
       lies in the range 0..1 */
    identity_4x4(&scale);
    scale.m[2][2] = 1./(back-front);
    mmul_4x4(&cumulative,&scale,&cumulative);
    
    set_world_to_view_scale((float)(scale.m[0][0]),(float)(scale.m[2][2]));
    
    matrix_4x4copy(view,&cumulative);
    
    *zmin = front/back;
    *zproj = b/back;
    return(CHILL);
}


int
compute_perspective_projection(
  matrix4x4 *view,
  vertex_3d *eye,
  vertex_3d *at,
  vertex_3d *up,
  float *fov,
  float *hither,
  float *yon,
  float *zmin,
  float *zproj)
{
    vector4 vrp,vrp_not;
    vector4 v_at,v_eye,v_up;
    double d,d2,a,b;
    matrix4x4 trans,rot_all,rtol,scale,cumulative;
    float front,back;
    vector4 p1p2,p1p3,pcross,rx,ry,rz;
    int i;

    for (i=0;i<3;i++)
    {
	v_eye.v[i] = eye->v[i];
	v_at.v[i] = at->v[i];
	v_up.v[i] = up->v[i];
    }
    v_eye.v[3] = v_at.v[3] = v_up.v[3] = 1.0;

    /* taken from Foley & van Dam, version 1 (w/ corrections) 1983. */

    vrp.v[0] = at->v[0];
    vrp.v[1] = at->v[1];
    vrp.v[2] = at->v[2];
    vrp.v[3] = 1.0;

    /* 1. translate "center of projection", or the eye point, to the origin. */
    identity_4x4(&trans);

    for (i=0;i<3;i++)
	trans.m[3][i] = -1. * eye->v[i];
    
    matrix_4x4copy(&cumulative,&trans);
    point_xfrm(&vrp,&trans,&vrp_not);

    /* 2. rotate so that VPN becomes parallel to negative Z axis in
       right handed world coord sys.  this is a mess.  */

    vector_4sub(&v_at,&v_eye,&p1p2); 
    vector_4mag(&p1p2,&d);
    d = 1./d;
    rz.v[0] = -1. * p1p2.v[0] * d;
    rz.v[1] = -1. * p1p2.v[1] * d;
    rz.v[2] = -1. * p1p2.v[2] * d;
    rz.v[3] = 1.;

    vector_4copy(&p1p3,&v_up);
    vector_4cross(&p1p2,&p1p3,&pcross);
    vector_4mag(&pcross,&d);
    d = 1./d;
    rx.v[0] = pcross.v[0] * d;
    rx.v[1] = pcross.v[1] * d;
    rx.v[2] = pcross.v[2] * d;
    rx.v[3] = 1.;

    vector_4cross(&rz,&rx,&ry);
    vector_4mag(&ry,&d);
    d = 1./d;
    for (i=0;i<3;i++)
	ry.v[i] *= d;
    
    identity_4x4(&rot_all);
    
    for (i=0;i<3;i++)
	rot_all.m[i][0] = rx.v[i];
    
    for (i=0;i<3;i++)
	rot_all.m[i][1] = ry.v[i];
    
    for (i=0;i<3;i++)
	rot_all.m[i][2] = rz.v[i];

    mmul_4x4(&cumulative,&rot_all,&cumulative);
    matrix_4x4copy(&s_view_rotation_matrix,&rot_all);
    
    /* 3. change from right to left handed coord system. */
    identity_4x4(&rtol);
    rtol.m[2][2] = -1.;
    
    mmul_4x4(&cumulative,&rtol,&cumulative);

    /* 4.  we induce no shear as vpn == eye-vrp */

    /* 5. scale so the view volume becomes the truncated right
       pyramid defined by x=z,-x=z,y=z,-y=z,z=zmin,z=1 */

    /* 5a. (bethel method) compute window half-width and height
       using a little trig - a function of the FOV parameter and
       the distance from the eye to "at". */

    point_xfrm(&vrp,&cumulative,&vrp_not);

    d2 = *fov/2.;
    d = 90. - d2;
    d = DEGREES_TO_RADIANS(d);
    d = ksin(d);

    b = vrp_not.v[2];  /* this should be somewhere on the +z axis */

    d2 = DEGREES_TO_RADIANS(d2);
    d2 = ksin(d2);

    a = b * d2 / d;  /* this is window half-width.  at some point,
			the aspect ratio may be added to produce differing
			widths and heights */

    front = b - *hither;
    back = b + *yon;

    identity_4x4(&scale);
#if 0    
    scale.m[0][0] = ((2.*b)/(2.*a*back));  /* this formulation produces */
    scale.m[1][1] = ((2.*b)/(2.*a*back));  /* a square window */
    scale.m[2][2] = 1./back;
#endif
    scale.m[0][0] = ((2.*b)/(2.*a*back));  /* this formulation produces */
    scale.m[1][1] = ((2.*b)/(2.*a*back));  /* a square window */
    scale.m[2][2] = 1./back;
    
    set_world_to_view_scale((float)(scale.m[0][0]),(float)(scale.m[2][2]));
    
    mmul_4x4(&cumulative,&scale,&cumulative);
    matrix_4x4copy(view,&cumulative);
    
    *zmin = front/back;
    *zproj = b/back;
    return(CHILL);
}

int
get_view_rotation_matrix(matrix4x4 *m)
{
    memcpy((char *)m,(char *)&s_view_rotation_matrix,sizeof(matrix4x4));
    return(CHILL);
}


int
get_view_xform(matrix4x4 *m)
{
    memcpy((char *)m,(char *)&s_view,sizeof(matrix4x4));
    return(CHILL);
}

int
set_view_xform(matrix4x4 *m)
{
    memcpy((char *)&s_view,(char *)m,sizeof(matrix4x4));
    return(CHILL);
}

int
set_background_color (float r, float g, float b)
{
    /* range check on input ?? */
    background_rgb.r = r;
    background_rgb.g = g;
    background_rgb.b = b;
    return(CHILL);
}

int
get_background_color (float *r, float *g, float *b)
{
    *r = background_rgb.r;
    *g = background_rgb.g;
    *b = background_rgb.b;
    return(CHILL);
}

int
set_projection (int p)
{
    /* range check ??*/
    s_projection = p;
    return(CHILL);
}

int
get_projection (int *p)
{
    *p = s_projection;
    return(CHILL);
}

int
set_eye (vertex_3d *e)
{
    memcpy((char *)&s_eye,(char *)e,sizeof(vertex_3d));
    return(CHILL);
}

int
get_eye (vertex_3d *e)
{
    memcpy((char *)e,(char *)&s_eye,sizeof(vertex_3d));
    return(CHILL);
}

int
set_at (vertex_3d *a)
{
    memcpy((char *)&s_at,(char *)a,sizeof(vertex_3d));
    return(CHILL);
}

int
get_at (vertex_3d *a)
{
    memcpy((char *)a,(char *)&s_at,sizeof(vertex_3d));
    return(CHILL);
}


int
set_up (vertex_3d *u)
{
    memcpy((char *)&s_up,(char *)u,sizeof(vertex_3d));
    return(CHILL);
}

int
get_up (vertex_3d *u)
{
    memcpy((char *)u,(char *)&s_up,sizeof(vertex_3d));
    return(CHILL);
}

int
set_fov (float f)
{
    /** range check ?? **/
    s_fov = f;
    return(CHILL);
}

int
get_fov (float *f)
{
    *f = s_fov;
    return(CHILL);
}

int
set_user_hither(float h)
{
    vertex_3d t;
    double mag;
    int i;

    for (i=0;i<3;i++)
	t.v[i] = s_eye.v[i] - s_at.v[i];

    mag = t.v[0]*t.v[0] + t.v[1]*t.v[1] + t.v[2]*t.v[2];
    if (mag == 0.)		/* the user interface code failed */
    {
	/* insert intelligent error msg here */
	fprintf(stderr,"set_user_hither : invalid data space\n");
	exit(1);
    }
    mag = ksqrt(mag);		/* distance from eye to at point */

    s_rmonster_hither = (1. - h) * mag;	/* convert from user distance (eye
					   to clip plane) to code distance
					   (at point to clip plane) */

    s_user_hither = h;
    return(CHILL);
}

int
get_user_hither (float *h)
{
    *h = s_user_hither;
    return(CHILL);
}

int
set_rmonster_hither(float h)
{
    s_rmonster_hither = h;
    return(CHILL);
}

int
get_rmonster_hither (float *h)
{
    *h = s_rmonster_hither;
    return(CHILL);
}


int
set_user_yon (float y)
{
    /* range check ?? */
    vertex_3d t;
    double mag;
    int i;

    for (i=0;i<3;i++)
	t.v[i] = s_eye.v[i] - s_at.v[i];

    mag = t.v[0]*t.v[0] + t.v[1]*t.v[1] + t.v[2]*t.v[2];
    if (mag == 0.)		/* the user interface code failed */
    {
	/* insert intelligent error msg here */
	fprintf(stderr,"set_user_yon : invalid data space\n");
	exit(1);
    }
    mag = ksqrt(mag);		/* distance from eye to at point */

    s_rmonster_yon = y * mag;	/* convert from user distance (eye
					   to clip plane) to code distance
					   (at point to clip plane) */
    s_user_yon = y;
    return(CHILL);
}

int
get_user_yon (float *y)
{
    *y = s_user_yon;
    return(CHILL);
}
int
set_rmonster_yon (float y)
{
    /* range check ?? */
    s_rmonster_yon = y;
    return(CHILL);
}

int
get_rmonster_yon (float *y)
{
    *y = s_rmonster_yon;
    return(CHILL);
}

int
set_zmin (float zm)
{
    /* range check?? */
    s_zmin = zm;
    return(CHILL);
}

int
get_zmin (float *zm)
{
    *zm = s_zmin;
    return(CHILL);
}

int
set_zproj (float zp)
{
    /* range check?? */
    s_zproj = zp;
    return(CHILL);
}

int
get_zproj (float *zp)
{
    *zp = s_zproj;
    return(CHILL);
}

int
set_world_to_view_scale (float xyscale, float depthscale)
{
    s_xyscale = xyscale;
    s_depthscale = depthscale;
    return(CHILL);
}

int
get_world_to_view_scale (float *xyscale, float *depthscale)
{
    *xyscale = s_xyscale;
    *depthscale = s_depthscale;
    return(CHILL);
}

int
set_hither_clip_flag(int i)
{
    c_hither_clip_flag = i;
    return(CHILL);
}

int
get_hither_clip_flag(int *i)
{
    *i = c_hither_clip_flag;
    return(CHILL);
}

int
set_yon_clip_flag(int i)
{
    c_yon_clip_flag = i;
    return(CHILL);
}

int
get_yon_clip_flag(int *i)
{
    *i = c_yon_clip_flag;
    return(CHILL);
}

int 
set_normalized_view ()
{
    rmonsterobj *r;
    vertex_3d bmin,bmax,b1,b2;
    vertex_3d eye,at,vert;
    float fov,tmp;
    double d,d2,b,mag,a,target;
    int i;

    /* get bbox of root, center point. */
    r = get_root_object();
    get_object_bbox(r,&b1,&b2);

    /* need to apply the object's transformation matrix to the box. */
    npoint_xfrm((KGEOM_VERTEX_TYPE *)&(b1.v[0]),&(r->xfrm),
		(KGEOM_VERTEX_TYPE *)&(bmin.v[0]),1);
    npoint_xfrm((KGEOM_VERTEX_TYPE *)&(b2.v[0]),&(r->xfrm),
		(KGEOM_VERTEX_TYPE *)&(bmax.v[0]),1);

    /* compute eye-at so that the box lies within eye-av X FOV */
    get_eye(&eye);
    get_at(&at);
    get_fov(&fov);

    /**
      * for now (10 jan 95), compute "target" to be the max of the width 
      * or height of the object's bounding box.  this needs to be expanded
      * in the following ways:
      * 1. include depth,
      * 2. consider object transformations, ie, possibly apply object's
      *    transformation matrices to viewpoint.  ie, the bounding box
      *    yes/no (???) reflects the object transformation?
      *
      * at the time of writing this code, we're assuming an eye/at vector
      * which is parallel to the object's Z axis, which is the case for
      * this code's default values, but is not general at all.
    **/

    /* we want a single scalar value which will be used in computing the
       new eye point.  this value is a function (now) of the object's
       bounding box only. */

    target = bmax.v[0] - bmin.v[0];
    d = bmax.v[1] - bmin.v[1];
    target = (target > d) ? target : d;

    d2 = fov/2.;
    d = 90. - d2;
    d = DEGREES_TO_RADIANS(d);
    d = ksin(d);

    npoint_diff((float *)&(eye),(float *)&(at),(float *)(&vert),1);
    vertex_unit(&vert,&mag);
    b = mag; 

    d2 = DEGREES_TO_RADIANS(d2);
    d2 = ksin(d2);

#if 0
    a = b * d2 / d;  /* this is window half-width.  at some point,
			the aspect ratio may be added to produce differing
			widths and heights */
#endif
    mag = target * d / d2;

    /**
      * now, mag contains the length of the eye/at vector that we _want_.
    **/

    for (i=0;i<3;i++)
        vert.v[i] *= mag;

    /* now, set the at point to point at the object's center point.  for
     now, use the average of the bounding box min/max.  this is not such
     a good idea, because the user may have set a center point on the object
     and we should use that instead. */

    
    for (i=0;i<3;i++)
        at.v[i] = bmin.v[i] + 0.5 * (bmax.v[i] - bmin.v[i]);

    set_at(&at);

    for (i=0;i<3;i++)
        eye.v[i] = at.v[i] + vert.v[i];

    set_eye(&eye);

    /* reset hither and yon */
    get_user_hither(&tmp);
    set_user_hither(tmp);

    get_user_yon(&tmp);
    set_user_yon(tmp);

    /* call routine to recompute view xform */
    compute_view_xform();

    return(CHILL);
}

int 
set_startup_default_camera_parms()
{
    set_eye(&startup_eye);
    set_at(&startup_at);
    set_up(&startup_up);
    
    set_fov(startup_fov);
    set_user_hither(startup_user_hither);
    set_user_yon(startup_user_yon);
    set_projection(startup_user_projection);

    compute_view_xform();
    return(CHILL);
}
