
/*####################################################################

Copyright (C) 1993, 1994 Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is nearly 100% hand-crafted by a human being in the USA"

####################################################################*/

 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  *
  * the strategy for stuffing a "vis colormap" into a kobject is
  * as follows:
  *
  * the RGBA components go into the MAP segment.
  * the VALUE segment is simply a ramp of integers [0..nvalues-1]
  *     where nvalues is the number of entries in the colormap.
  * the bracket values in the vismap go into the LOCATION segment.
  *
  * thus, the kobject looks like a 1d signal with location data
  *     and map data.  the map data MUST BE in RGB format, and the
  *     HAS_ALPHA attribute must be set.
  *
  * at present, the colormaps have a maximum number of entries of
  * 256 due to the fact that the value data in the kobject is a byte.
  * longer cmaps can be supported by changing this storage format to an int.
  *
  * we use the image services api for getting and putting info to/from
  * the kobject.  some of the ads api is used for global attributes.
  *
  * 14 may 1993
  * wes
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



#include "internals.h"

/****************************************************************
* 
*  Routine Name: lvismap_to_kobj
* 
*       Purpose: Library Routine for test
*         Input:
*                  vis_cmap * - a vis colormap structure
*        Output:
*                  kobject - with the vis cmap stuffed into it.
*       Returns: CHILL (1) on success, WHACKED (0) on failure
*  Restrictions: 
*    Written By: wes
*          Date: 13 may 1993
*      Verified: 
*  Side Effects: 
* Modifications:
*
*      Comments:
****************************************************************/

int
lvismap_to_kobj(vis_cmap *map, kobject k)
{
    int i,n;
    unsigned char indeces[MAX_VIS_CMAP_ENTRIES];
    /* temporary */
    n = map->nentries;

    /* stick a "magic cookie" into the comment field. */
    kpds_set_attribute(k,KPDS_COMMENT,vis_map_cookie);

    /* set attributes to create value, map and loc segments */
    kpds_create_value(k);
    kpds_create_map(k);
    kpds_create_location(k);

    /* set the value segment attributes to reflect the size
       of the value data. */
    kpds_set_attribute(k,KPDS_VALUE_DATA_TYPE,KUBYTE);

    /* set the map data type. */
    kpds_set_attribute(k,KPDS_MAP_DATA_TYPE,KFLOAT);

    /* set the colorspace of the "image" */
    kcolor_set_attribute(k,KCOLOR_COLORSPACE,KRGB);

    /* and set the flag to indicate that we have alpha */
    kcolor_set_attribute(k,KCOLOR_HAS_ALPHA,TRUE); 

    /* set the size of the map: width is 1, height is
      the size of the colormap, and 4 is RGBA */
    kpds_set_attribute(k,KPDS_MAP_SIZE,4,n,1,1,1);

    /* set the size of the "image" */
    kpds_set_attribute(k,KPDS_VALUE_SIZE,n,1,1,1,1);

    /* now we have to do something with the location data. */

    kpds_set_attribute(k,KPDS_LOCATION_SIZE,n+1,1,1,1);
    kpds_set_attribute(k,KPDS_LOCATION_DATA_TYPE,KFLOAT);
    kpds_put_data(k,KPDS_LOCATION_ALL,map->bracket);

    /* create some value data.  the purpose of the value data is to
     act as indices into the colormap so it can be displayed with something
     like putdata. */
    for (i=0;i<n;i++)
	indeces[i] = i;

    /* put the indeces into the value segment */
    kpds_put_data(k,KPDS_VALUE_PLANE,indeces);

    /* write the data into the map segment */
    for (i=0;i<n;i++)
    {
	kpds_put_data(k,KPDS_MAP_POINT,&(map->r[i]));
	kpds_put_data(k,KPDS_MAP_POINT,&(map->g[i]));
	kpds_put_data(k,KPDS_MAP_POINT,&(map->b[i]));
	kpds_put_data(k,KPDS_MAP_POINT,&(map->a[i]));
    }

    return(CHILL);
}

/****************************************************************
* 
*  Routine Name: lkobj_to_vismap
* 
*       Purpose: Library Routine for test
*         Input: kobject
*        Output: vis_cmap structure
*       Returns: CHILL (1) on success, WHACKED (0) on failure
*  Restrictions: Only kobjects creattted by lvismap_to_viff
*                will successfully be processed by this routine.
*    Written By: wes
*          Date: 13 may 1993
*      Verified: 
*  Side Effects: 
* Modifications: april 1995
*                remove check for magic cookie in vis colormaps.
*                this will permit more flexibility in how colormaps
*                are generated.  now, we just require a certain
*                consistency between the dimensions in the input object.
*      Comments:
****************************************************************/

int
lkobj_to_vismap(kobject k, vis_cmap *map)
{
    int map_width,map_height,map_el;
    int loc_width,loc_height,loc_depth,loc_dimension;
    int map_data_type,loc_data_type;
    int img_colorspace;
    int i,n;
    char buffer[256],*buffer_addr=NULL;

    buffer_addr = buffer; 

    /**
      * the things that we want to check for (ie, is this a
      * valid "vis map" file):
      *
      * a valid visualization colormap should have the following attributes:
      *
      * 1. dimesnions of map segment: must be 4wide by Nhigh by 1deep,
      *      which says that there are N map entries, consisting of RGBA
      * 2. dimensions of location segment: must be N+1 wide by 1high, deep,etc.
      *
    **/
    
    kpds_get_attribute(k,KPDS_MAP_SIZE,
		       &map_width,&map_height,&map_el,NULL,NULL);
    if ((map_width != 4) || (map_el != 1) /* do we really care about map_el?*/
	|| (map_height > MAX_VIS_CMAP_ENTRIES))
    {
	sprintf(buffer,"\tThe dimensions of the map segment are out of range: w/h/e are %d/%d/%d.\n\t\tValid ranges are width=4,1<=h<=N,e=1\n",map_width,map_height,map_el);
	errno = KINVALID_INPUT;
	kerror("geometry","lkobj_to_vismap",buffer);
	return(WHACKED);
    }

    n = map_height;
    map->nentries = n;
    
    kpds_get_attribute(k,KPDS_LOCATION_SIZE,&loc_width,
		       &loc_height,&loc_depth,&loc_dimension);
    /* loc_width is one more than n because of the extra bracket value. KC */
    if (loc_width != n + 1)
    {
	sprintf(buffer,"\tThe width of the location segment, %d, doesn't equal the height of the map segment, %d.\n\t\tThese dimensions must be equal.\n",loc_width,n);
	errno = KINVALID_INPUT;
	kerror("geometry","lkobj_to_vismap",buffer);
	return(WHACKED);
    }

    /* set the colorspace of the "image" */
    kcolor_get_attribute(k,KCOLOR_COLORSPACE,&img_colorspace);
    if (img_colorspace != KRGB)
    {
	sprintf(buffer,"\tThe image colorspace is set to %d, but must be set to KRGB=%d.\n",img_colorspace,KRGB);
	errno = KINVALID_INPUT;
	kerror("geometry","lkobj_to_vismap",buffer);
/*	return(WHACKED); */
    }

    /* do a check on the data type in the map. */
    kpds_get_attribute(k,KPDS_MAP_DATA_TYPE,&map_data_type);
    if (map_data_type != KFLOAT)
    {
	sprintf(buffer,"\tThe datatype in the map is %d, but must be set to KFLOAT=%d.\n",map_data_type,KFLOAT);
	errno = KINVALID_INPUT;
	kerror("geometry","lkobj_to_vismap",buffer);
    }

    /* do a check on the data type in the location segment. */
    kpds_get_attribute(k,KPDS_LOCATION_DATA_TYPE,&loc_data_type);
    if (loc_data_type != KFLOAT)
    {
	sprintf(buffer,"\tThe datatype in the location is %d, but must be set to KFLOAT=%d.\n",loc_data_type,KFLOAT);
	errno = KINVALID_INPUT;
	kerror("geometry","lkobj_to_vismap",buffer);
    }

    /* do a check on the vis-map-magic cookie */
#if 0
    /* removed this restriction april 4 1995 to allow for more flexibility
     in colormap-generating sources. */
    buffer_addr=NULL;
    kpds_get_attribute(k,KPDS_COMMENT,&buffer_addr);

    if (kstrcmp(buffer_addr,vis_map_cookie) != 0)
    {
	sprintf(buffer,"\tBad Magic Cookie. Expected <%s> but got <%s> instead.\n",vis_map_cookie,buffer);
	errno = KINVALID_INPUT;
	kerror("geometry","lkobj_to_vismap",buffer);
    }
#endif


    /* get the location data */
    kpds_get_data(k,KPDS_LOCATION_ALL,map->bracket);

    /* get the map data */
    for (i=0;i<n;i++)
    {
	kpds_get_data(k,KPDS_MAP_POINT,&(map->r[i]));
	kpds_get_data(k,KPDS_MAP_POINT,&(map->g[i]));
	kpds_get_data(k,KPDS_MAP_POINT,&(map->b[i]));
	kpds_get_data(k,KPDS_MAP_POINT,&(map->a[i]));
    }

    return(CHILL);
}

int
lget_map_index(vis_cmap *map,
	       float *val)
{
    /* look through the bracket values in the map, return the index
       into the map. right now, we use a brain dead (slow) leainar
       search.  we should use a binary search, particularly since the
       bracket values are (by definition) in increasing order.  */

    int i;
    for (i=0;i<map->nentries+1;i++)
    {
	if (*val <= map->bracket[i])
	    return(i);
    }
    return(map->nentries - 1);
    
}

void
lget_map_rgba(vis_cmap *map,
	      int n,
	      float *r,
	      float *g,
	      float *b,
	      float *a)
{
    if (map == NULL)  /* error */
	return;
    
    if (n > map->nentries-1)
	n = map->nentries-1;
    else if (n < 0)
	n = 0;
    
    *r = map->r[n];
    *g = map->g[n];
    *b = map->b[n];
    *a = map->a[n];
}

