
/*####################################################################

Copyright (C) 1993, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus is is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory

  "this software is 100% hand-crafted by a human being in the USA"

####################################################################*/



#include "internals.h"

/**
  * identity_4x4(matrix4x4 *m)
  * clear_4x1(vector4 *v)
  * print_4x4matrix(matrix4x4 *m)
  * print_4x1vector(vector4 *v)
  * vector_4copy(vector4 *d,*s)
  * matrix_4x4copy(matrix4x4 *d,*s)
  * point_xfrm(vector4 *s,matrix4x4 *m,vector4 *d)
  * clear_4x4(matrix4x4 *m)
  * mmul_4x4(matrix4x4 *s1,*s2,*d)
  * vector_4add(vector4 *s1,*s2,*d)
  * vector_4sub(vector4 *s1,*s2,*d)
  * vector_4scale(vector4 *s, float *c, vector4 *d)
  * vector_4unit(vector4 *v, double *mag)
  * vector_4mag(vector4 *v, double *mag)
  * vector_4cross(vector4 *p,*r,*cross)
  * npoint_diff(float *s1,*s2,*d, int n) // d[i] = s1[i] - s2[i], i=0..n-1 //
  * npoint_add(float *s1,*s2,*d, int n) // d[i] = s1[i] + s2[i], i=0..n-1 //
  * npoint_xfrm(vector4 *s,matrix4x4 *m,vector4 *d,int n)
  * add_const_then_scale(float *s,const,scale,*dst,int n,stride)
  * scale_then_add_const(float *s,const,scale,*dst,int n,stride)
  * scale_vector(float *src,scale,*dst, int n,stride)
  * vect_min(float *src, int n,stride)
  * vect_max(float *src, int n,stride)
  * vmemset(char *dst,int value,int offset,int stride);
  * 
  * normal_xfrm(vertex_3d *s,matrix4x4 *m,vertex_3d *d)
  * nnormal_xfrm(vertex_3d *s,matrix4x4 *m,vertex_3d *d,int n)
  * nnormal_unit(vertex_3d *s, vertex_3d *d, int n)
  *
  * vertex_copy(vertex_3d *s,*d,int n);
  * vertex_unit(vertex_3d *v,double *mag)
  * vertex_cross(vertex_3d *p1,*p2,*cross)
**/


int
identity_4x4(matrix4x4 *m)
{
    int i,j;
    for (j=0;j<4;j++)
	for (i=0;i<4;i++)
	    m->m[i][j] = (i == j) ? 1. : 0.;
    return(CHILL);
}

int
clear_4x1(vector4 *v)
{
    v->v[0] = v->v[1] = v->v[2] = 0.;
    v->v[3] = 1.;
    return(CHILL);
}

int
print_4x4matrix(matrix4x4 *m)
{
    int i,j;

    for (j=0;j<4;j++)
    {
	for (i=0;i<4;i++)
	    kprintf("\t%g",m->m[j][i]);
	kprintf("\n");
    }
    return(CHILL);
}

int
print_4x1vector(vector4 *v)
{
    int i;
    for (i=0;i<4;i++)
	kprintf("\t%g",v->v[i]);
    kprintf("\n");
    return(CHILL);
}

int
vector_4copy(vector4 *d, vector4 *s)  
{
    memcpy((char *)d,(char *)s,sizeof(vector4));
    return(CHILL);
}

int
matrix_4x4copy(matrix4x4 *d,matrix4x4 *s)
{
    memcpy((char *)d,(char *)s,sizeof(matrix4x4));
    return(CHILL);
}


int
point_xfrm(vector4 *v,matrix4x4 *m,vector4 *r)
{
    int i,j;
    
    for (j=0;j<4;j++)
    {
	r->v[j] = 0.;
	for (i=0;i<4;i++)
	    r->v[j] += v->v[i] * m->m[i][j];
    }
    return(CHILL);
}

int
vert_xfrm(vertex_3d *v,matrix4x4 *m,vertex_3d *r)
{
    int i,j;
    
    for (j=0;j<3;j++)
    {
	r->v[j] = 0.;
	for (i=0;i<3;i++)
	    r->v[j] += v->v[i] * m->m[i][j];
	r->v[j] += m->m[3][j];
    }
    return(CHILL);
}

int
normal_xfrm(vertex_3d *v,matrix4x4 *m,vertex_3d *r)
{
    /* this is like the usual xformation, but we don't add in
       the translational component. */
    
    int i,j;
    float dest[3];
    
    for (j=0;j<3;j++)
    {
	dest[j] = 0.;
	for (i=0;i<3;i++)
	    dest[j] += v->v[i] * m->m[i][j];
    }
    for (j=0;j<3;j++)
	r->v[j] = dest[j];
    return(CHILL);
}

int
nnormal_xfrm(vertex_3d *s, matrix4x4 *m, vertex_3d *d, int n)
{
    int i;
    for (i=0;i<n;i++)
	normal_xfrm(s+i,m,d+i);
    return(CHILL);
}

int
clear_4x4(matrix4x4 *m)
{
    int i,j;
    for (i=0;i<4;i++)
        for (j=0;j<4;j++)
            m->m[i][j] = 0.;
    return(CHILL);
}

int
mmul_4x4(matrix4x4 *a, matrix4x4 *b, matrix4x4 *c)
 /* take product of a and b, put result in c */
{
    matrix4x4 tmp;
    int row,col;

    clear_4x4(&tmp);

    for (row=0;row<4;row++)
    {
	for (col=0;col<4;col++)
	    tmp.m[row][col] = a->m[row][0]*b->m[0][col] +
		a->m[row][1]*b->m[1][col] + a->m[row][2]*b->m[2][col] +
		    a->m[row][3]*b->m[3][col];
    }
    for (row=0;row<4;row++)
	for (col=0;col<4;col++)
	    c->m[row][col] = tmp.m[row][col];
    return(CHILL);
}

int
vector_4add(vector4 *a, vector4 *b,vector4 *c)
/* c = a + b */
{
    int i;
    for (i=0;i<3;i++)
	c->v[i] = a->v[i] + b->v[i];
    c->v[3] = 1.0;
    return(CHILL);
}

int
vector_4sub(vector4 *a, vector4 *b, vector4 *c)
 /* c = a - b */
{
    int i;
    for (i=0;i<3;i++)
	c->v[i] = a->v[i] - b->v[i];
    c->v[3] = 1.0;
    return(CHILL);
}

int
vector_4scale(vector4 *a, float *s, vector4 *b)
 /* b = a * s */
{
    int i;
    for (i=0;i<3;i++)
	b->v[i] = a->v[i] * *s;
    return(CHILL);
}

int
vector_4unit(vector4 *v,double *d)
{
    double t;
    int i;

    t = v->v[0]*v->v[0] + v->v[1]*v->v[1] + v->v[2]*v->v[2];
    t = ksqrt(t);

    for (i=0;i<3;i++)
	v->v[i] = v->v[i]/t;
    
    *d = t;
    
    return(CHILL);
}

int
vector_4mag(vector4 *v, double *d)
{
    double t;

    t = v->v[0]*v->v[0] + v->v[1]*v->v[1] + v->v[2]*v->v[2];
    t = ksqrt(t);
    *d = t;
    
    return(CHILL);
}

int
vector_4cross(vector4 *p,vector4 *r,vector4 *c)
/* c = p x r */
{
    c->v[0] = p->v[1]*r->v[2] - p->v[2]*r->v[1];
    c->v[1] = p->v[2]*r->v[0] - p->v[0]*r->v[2];
    c->v[2] = p->v[0]*r->v[1] - p->v[1]*r->v[0];
    return(CHILL);
}

int
npoint_diff(KGEOM_VERTEX_TYPE *a, KGEOM_VERTEX_TYPE *b, KGEOM_VERTEX_TYPE *c, int n)
{
   /**
     * this routine computes:
     * 1. c[0..2] = a[0..2] - b[0..2]
     *
     * for the N points listed in a.  b is a single point.
   **/
   register KGEOM_VERTEX_TYPE *s1,*s2,*d;
   register int i,j;

   /* do all the x coords first. */

   s1 = a;
   s2 = b;
   d = c;
   for (i=0;i<n;i++)
   {
       for (j=0;j<3;j++)
	   *d++ = *s1++ - *(s2+j);
   }
    return(CHILL);
}

int
npoint_add(KGEOM_VERTEX_TYPE *a, KGEOM_VERTEX_TYPE *b, KGEOM_VERTEX_TYPE *c,int n)
{
   /**
     * this routine computes:
     * 1. c[0..2] = a[0..2] + b[0..2]
     *
     * for the N points listed in a.  b is a single point.
   **/
   register KGEOM_VERTEX_TYPE *s1,*s2,*d;
   register int i,j;

   /* do all the x coords first. */

   s1 = a;
   s2 = b;
   d = c;
   for (i=0;i<n;i++)
   {
       for (j=0;j<3;j++)
	   *d++ = *s1++ + *(s2+j);
   }
    return(CHILL);
}

int
npoint_xfrm(KGEOM_VERTEX_TYPE *v_in,
	    matrix4x4 *xfrm,
	    KGEOM_VERTEX_TYPE *v_out,
	    int n)
{
    /**
      * this code has been written to be fast, not readable. sorry.
      *
      * basically what we do is multiply the x portion of each input
      * coordinate with the top row of the matrix, the y portion with
      * the second row, the z with the third part.  then, we add in
      * the translational component directly to the z-product term.
      * next, we then add each of the three components contributing
      * to the output x coord, then the three parts contributing to the
      * y output coord, and then the z part.
      *
      * i was trying to minimize pointer arithmetic and memory accesses.
    **/
    register KGEOM_VERTEX_TYPE *s1,*s2,*d;
    register int i,j,k;
    KGEOM_VERTEX_TYPE dest[9];

    for (k=0;k<n;k++,v_in+=3,v_out+=3)
    {
	s1 = v_in;
	s2 = &(xfrm->m[0][0]);
	d = dest;
	for (i=0;i<3;i++,s1++)
	{
	    for (j=0;j<3;j++)
		*d++ = *s1 * *s2++;
	    s2++;
	}
	d -= 3;
	for (j=0;j<3;j++)
	    *d++ += *s2++;

	d = v_out;
	for (i=0;i<3;i++,d++)
	{
	    s1 = dest+i;
	    *d = *s1;
	    s1 += 3;
	    for (j=1;j<3;j++,s1+=3)
		*d += *s1;
	}
    }
    return(CHILL);
}

int
add_const_then_scale(KGEOM_VERTEX_TYPE *src,
		     KGEOM_VERTEX_TYPE constant,
		     KGEOM_VERTEX_TYPE scale,
		     KGEOM_VERTEX_TYPE *dst,
		     int n,
		     int stride)
{
    /**
      * dst[i] = (src[i] + constant) * scale; i=0..n-1,i+=stride
    **/
    register KGEOM_VERTEX_TYPE *s,*d;
    register KGEOM_VERTEX_TYPE c,sc;
    register int i;

    d = dst;
    s = src;
    c = constant;
    sc = scale;
    
    for (i=0;i<n;i++,d+=stride,s+=stride)
	*d = (*s + c) * sc;
    return(CHILL);

}

int
scale_then_add_const(KGEOM_VERTEX_TYPE *src,
		     KGEOM_VERTEX_TYPE constant,
		     KGEOM_VERTEX_TYPE scale,
		     KGEOM_VERTEX_TYPE *dst,
		     int n,
		     int stride)
{
    /**
      * dst[i] = src[i] * scale + const; i=0..n-1,i+=stride
    **/
    register KGEOM_VERTEX_TYPE *s,*d;
    register KGEOM_VERTEX_TYPE c,sc;
    register int i;

    d = dst;
    s = src;
    c = constant;
    sc = scale;
    
    for (i=0;i<n;i++,d+=stride,s+=stride)
	*d = *s * sc + c;

    return(CHILL);
}

int
scale_vector(KGEOM_VERTEX_TYPE *src,
	     KGEOM_VERTEX_TYPE scale,
	     KGEOM_VERTEX_TYPE *dst,
	     int n,
	     int stride)
{
    /**
      * dst[i] = src[i] * scale; i=0..n-1,i+=stride
    **/
    register KGEOM_VERTEX_TYPE *s,*d;
    register KGEOM_VERTEX_TYPE sc;
    register int i;

    d = dst;
    s = src;
    sc = scale;
    
    for (i=0;i<n;i++,d+=stride,s+=stride)
	*d = *s * sc;

    return(CHILL);
}

int
vect_min(KGEOM_VERTEX_TYPE *src,
	 int n,
	 int offset,
	 int stride,
	 KGEOM_VERTEX_TYPE *m)
{
    register KGEOM_VERTEX_TYPE minval=KGEOM_BIGNUM;
    register int i;
    register KGEOM_VERTEX_TYPE *s;

    s = src+offset;
    
    for (i=0;i<n;i+=stride,s+=stride)
	if (*s < minval)
	    minval = *s;
    *m = minval;
    return(CHILL);
}

int
vect_max(KGEOM_VERTEX_TYPE *src,
	 int n,
	 int offset,
	 int stride,
	 KGEOM_VERTEX_TYPE *m)
{
    register KGEOM_VERTEX_TYPE maxval= -1. * KGEOM_BIGNUM;
    register int i;
    register KGEOM_VERTEX_TYPE *s;

    s = src+offset;

    for (i=0;i<n;i+=stride,s+=stride)
	if (*s > maxval)
	    maxval = *s;
    *m = maxval;
    return(CHILL);
}

int
vmemset(char *dst,
	int value,
	int offset,
	int stride,
	int n)
#if 0
char *dst;      /* user memory to fill */
int value;      /* value to fill with */
int offset;     /* offset in char *'s from dst to begin */
int stride;     /* stride in char *'s through mem */
int n;          /* # of steps/strides to take */
#endif
{
    register char *d;
    register int val,s,i;

    s = stride;
    d = dst+offset;
    val = value;
    for (i=0;i<n;i++,d+=s)
	*d = val;
    return(CHILL);
}

int
nnormal_unit(vertex_3d *s,
	     vertex_3d *d,
	     int n)
{
    double mag;
    int i,j;

    for (i=0;i<n;i++)
    {
	mag = 0.;
	for (j=0;j<3;j++)
	    mag += s[i].v[j]*s[i].v[j];
	mag = ksqrt(mag);
	if (mag == 0.)
	    mag = 1e10;
	else
	    mag = 1./mag;
	for (j=0;j<3;j++)
	    d[i].v[j] = s[i].v[j]*mag;
    }
    return(CHILL);
}

int
vertex_copy(vertex_3d *s,vertex_3d *d,int n)
{
    memcpy((char *)d,(char *)s,sizeof(vertex_3d)*n);
    return(CHILL);
}

int
vertex_unit(vertex_3d *v,double *d)
{
    register int i;
    double t;

    t = 0;
    for (i=0;i<3;i++)
	t += (v->v[i] * v->v[i]);
    t = ksqrt(t);
    *d =t ;
    if (t != 0.)
    {
	t = 1./t;
	for (i=0;i<3;i++)
	    v->v[i] *= t;
    }
    return(CHILL);
}

int
vertex_cross(vertex_3d *p,vertex_3d *r,vertex_3d *c) /* c = p x r */
{
    c->v[0] = p->v[1]*r->v[2] - p->v[2]*r->v[1];
    c->v[1] = p->v[2]*r->v[0] - p->v[0]*r->v[2];
    c->v[2] = p->v[0]*r->v[1] - p->v[1]*r->v[0];

    return(CHILL);
}

int
vertex_mag(vertex_3d *v,double *d)
{
    register int i;
    double t;

    t = 0;
    for (i=0;i<3;i++)
	t += (v->v[i] * v->v[i]);
    t = ksqrt(t);
    *d =t ;
    return(CHILL);
}

int
matrix_inverse(matrix4x4 *in,
	       matrix4x4 *out)
{
    /**
      * the following is a software interface to some routines from
      * LINPACK which are called to compute a matrix inverse.
    **/
#if 0
    float m[4][4];
    int lda,n,ipvt[4],info,job;
    float work[4*2],det[4][2];
    float newv[3];
 
    memcpy((char *)m,(char *)in,sizeof(matrix4x4));
    job = 1;
    lda = 4;
    n = 4;
 
    SGEFA(m,&lda,&n,ipvt,&info);
    SGEDI(m,&lda,&n,ipvt,det,work,&job);
 
    memcpy((char *)out,(char *)m,sizeof(matrix4x4));
#endif

    kfmatrix_inverse((float *)in,4,(float *)out);

    return(CHILL);

}
