 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for gquadmesh
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lgquadmesh
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
#include "internals.h"


static int ok_mesh_input PROTO(( kobject ));
static int make_quad_mesh PROTO(( kobject, kobject, int *, vis_cmap *, int,int,float *));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lgquadmesh - *
* 
*       Purpose: to create a quad mesh geom object from 2D data.
*                the geom may be optionally colorized with
*                a colormap, whose presence is optional.
*
*         Input: inobj - an opened kobject containing the source data
*                cmapobj - an opened kobject containing a visualization cmap
*                name - a char string containing the name for the geom object
*
*                have_mapfile - boolean indicating if we have a colormap
*                have_tcoords - boolean indicating if we have are to
*                    generate texture coords for the quadmesh.
*
*                NOTE: the use of a colormap and generation of texture coords
*                    are mutually exclusive.
*
*                tc - array of 4 floats specifing umin,vmin,umax,vmax for
*                    texture coords.  this routine will compute texture coords
*                    such that (umin,vmin) will be at the index (0,0) in
*                    the quadmesh, and (umax,vmax) will be at index
*                    (usize-1,vsize-1) of the quadmesh.
*
*        Output: outobj - an opened kobject containing the geom object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Yo Momma
*          Date: Apr 13, 1995
*      Verified: 
*  Side Effects: do not drive heavy machinery while operating this software
* Modifications: 
****************************************************************/
/* -library_def */
int lgquadmesh(kobject inobj,
	      int have_mapfile,
	      kobject cmapobj,
	      kobject outobj,
	      char *name,
	      int have_tcoords,
	      float *tc)
/* -library_def_end */

/* -library_code */
{
    vis_cmap cmap;
    int i,num_dims,dims[5];

    /** first, verify that the input object is processable **/

    /** at present, we want 2 dimensional data and 3d coords and
      will barf if we get anything else **/

    if (ok_mesh_input(inobj) == WHACKED)
	return(WHACKED);

    /* next, convert the colormap if it's present to a vis_cmap */
    if (have_mapfile)
    {
	if (lkobj_to_vismap(cmapobj,&cmap) == WHACKED)
        {
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","lquadmesh","The object supplied for a vis colormap is invalid. \n");
	    return(WHACKED);
	}
    }

    /* set up the output object */
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_NAME,name);
    kgeom_create_primitive_list(outobj);
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_NUMBER_PRIMITIVES,1);
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_LAYOUT,KPER_VERTEX);
    kgeom_set_attribute(outobj,KGEOM_OBJECT,KGEOM_HAS_ALPHA,TRUE);

    /* -- xvisual needs these --*/
    kdms_copy_attributes(cmapobj, outobj, NULL,
                         KCOLOR_ALPHA_MIN, KCOLOR_ALPHA_MAX, NULL);
    
    lset_common_geom_attributes(outobj);
    
    /* build the output object */

    /* figure out if we are workign with 2 or 3d data. */
    kpds_get_attribute(inobj,KPDS_VALUE_SIZE,dims,dims+1,dims+2,
		       dims+3,dims+4);
	
    /* in the following loop, we`ll scan through the sizes associated
       with each value segment dimension (EXCEPT for elements) and
       count if the value is greater than 1 */

    num_dims=0;
    for (i=0;i<4;i++)
	if (dims[i] > 1)
	    num_dims++;

    /* if num_dims==2, then we'll (try to) make a quadmesh, else if
       num_dims==3, then we'll try to make an oct mesh */

    if (num_dims == 2)
	make_quad_mesh(inobj,outobj,dims,&cmap,have_mapfile,have_tcoords,tc); 
    else
    {
	errno = KINVALID_DATATYPE;
	kerror("GEOMETRY","lquadmesh","There must be 2 dimmensions of value data.  You have given me %d\n",num_dims);
	return(WHACKED);
    }
    return(CHILL);
}

static int
ok_mesh_input(kobject inobj)
{
    int lw,lh,ld,ldims;
    int vw,vh,vd,ve,vt;

    if (kpds_query_location(inobj))
    {
       kpds_get_attribute(inobj,KPDS_LOCATION_SIZE,&lw,&lh,&ld,&ldims);
       if (ldims != 3)
       {
	  errno = KINVALID_DATATYPE;
	  kerror("GEOMETRY","lquadmesh","The location data must have dimensions=3.  The input file has dimensions=%d.\n",ldims);
	  return(WHACKED);
       }
    }
    else
    {
       errno = KINVALID_DATATYPE;
       kerror("GEOMETRY","lquadmesh","The input file must have explicit location data\n");
       return(WHACKED);
    }

    /* set up a happy presentation for the location data */
    kpds_set_attribute(inobj, KPDS_LOCATION_DATA_TYPE, KFLOAT);

    kpds_get_attribute(inobj,KPDS_VALUE_SIZE,&vw,&vh,&vd,&vt,&ve);
    if (ve != 1)  /* it's not scalar data .. */
    {
	if (ve != 4) /* can't possibly be anything but RGBA data */
	{
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","lquadmesh","This datafile has elements = %d.  Only elements=1 or elements=4 (interpreted as RGBA) are supported.");
	    return(WHACKED);
	}
    }

    /* set up a happy presentation for the value data */
    kpds_set_attribute(inobj, KPDS_VALUE_DATA_TYPE, KFLOAT);

    return(CHILL);
}

static int
make_quad_mesh(kobject kin,
	       kobject kout,
	       int *dims,
	       vis_cmap *cmap,
	       int use_map,
	       int use_tc,
	       float *tc)
{
    int i;
    int usize,vsize;
    float *iloc=NULL,*ival=NULL,*oloc=NULL,*temp=NULL,*ocolors=NULL;
    float *tcoords=NULL;
    vertex_3d *normals=NULL;
    int vw,vh,vd,vt,ve;
    int index;
    vertex_3d bmin,bmax,center;

    /* assign values to usize and vsize.  from the calling routine,
       dims[] holds the size of the value data.  i'm assuming that
       there are at two and only two values in this array that are > 1.
       such an assumption eliminates a lot of boring error checking
       code, but the below code will seg fault if this is not the case */
    
    for (i=0;i<4;i++)
	if (dims[i] != 1)
	    break;
    
    usize = dims[i];
    for (i=i+1;i<4;i++)
	if (dims[i] != 1)
	    break;
    vsize = dims[i];

    /* grab the location data from the input. */

    /* at some point, we'll want to do this in a more intelligent way
       than to just malloc space for the entire shootin' match.
       maybe loop over columns or something */

    iloc = (float *)kmalloc(sizeof(float)*usize*vsize*3); 
    oloc = (float *)kmalloc(sizeof(float)*usize*vsize*3); 

    kpds_set_attribute(kin,KPDS_LOCATION_POSITION,0,0,0,0);
    kpds_get_data(kin,KPDS_LOCATION_ALL,iloc);

    kgeom_set_attribute(kout,KGEOM_QUADMESH,KGEOM_QUADMESH_SIZE,usize,vsize);

    kgeom_set_attribute(kout,KGEOM_QUADMESH,KGEOM_QUADMESH_LOCATION_POSITION,
			0,0);

    /* unscramble location data from XXXYYYZZZ to XYZXYZXYZ */

    {
	float *x,*y,*z,*d;
	int j;
	
	x = iloc;
	y = x + usize*vsize;
	z = y + usize*vsize;

	d = oloc;
	
	for (j=0;j<usize*vsize;j++)
	{
	    *d++ = *x++;
	    *d++ = *y++;
	    *d++ = *z++;
	}
    }

    kgeom_put_data(kout,KGEOM_QUADMESH_LOCATION_ALL,oloc);

    /**
      * compute the normals for the quadmesh and put them, too.
    **/

    lcompute_quadmesh_vertex_normals((vertex_3d *)oloc,
				     &normals,
				     usize,vsize);
    kgeom_put_data(kout,KGEOM_QUADMESH_NORMAL_ALL,(float *)(normals));

    /**
      * compute the bounding box and center point.
    **/

    lcompute_bbox((vertex_3d *)oloc,usize*vsize,&bmin,&bmax);
    for (i=0;i<3;i++)
	center.v[i] = bmin.v[i] + (bmax.v[i]-bmin.v[i])*0.5;
    kgeom_set_attribute(kout,KGEOM_OBJECT,KGEOM_BOUNDING_BOX,&(bmin.v[0]),&(bmax.v[0]));
    kgeom_set_attribute(kout,KGEOM_OBJECT,KGEOM_CENTER,&(center.v[0]));
			
     

    /**
     * the possibilities for colorizing the output quadmesh are as
     * follows:
     * 1. use_map == TRUE, && scalar data,
     * 2. use_map == FALSE, && e==4 value data (interpreted as
     *    RGBA in [0..1] )
     * 3. use_map == FALSE, scalar data, (or no data), AND presence
     *     of texture coords.
    **/
    
    kpds_get_attribute(kin,KPDS_VALUE_SIZE,&vw,&vh,&vd,&vt,&ve);
    if (use_map)
    {
	if (ve != 1)
	{
	    errno = KINVALID_DATATYPE;
	    kerror("GEOMETRY","lquadmesh","When using a colormap with 2D data, the elements size of the value segment MUST be equal to 1.  The file you supplied has something different.");
	    return(WHACKED);
	}
	ival = (float *)kmalloc(sizeof(float)*usize*vsize);
	temp = ocolors = (float *)kmalloc(sizeof(float)*usize*vsize*4);
	kpds_set_attribute(kin,KPDS_VALUE_POSITION,0,0,0,0,0);
	kpds_get_data(kin,KPDS_VALUE_ALL,ival);

	for (i=0;i<usize*vsize;i++,temp+=4)
	{
	    index = lget_map_index(cmap,ival+i);
	    lget_map_rgba(cmap,index,temp,temp+1,temp+2,temp+3);
	}

	kgeom_put_data(kout,KGEOM_QUADMESH_COLOR_ALL,ocolors);
	kfree(ocolors);

	
    }
    else
    {
	/* don't care what kind of value data we have */
	if (use_tc == 2) /* make 2d texture coords */
	{
	    double U,dU,V,dV;
	    int iu,iv,offset=0;
	    
	    tcoords = (float *)kmalloc(usize*vsize*2*sizeof(float));

	    U = tc[0];
	    dU = (tc[1]-tc[0])/(usize-1);
	    V = tc[2];
	    dV = (tc[3]-tc[2])/(vsize-1);
	    
	    for (iv = 0;iv<vsize;iv++)
	    {
		U = tc[0];
		for (iu=0;iu<usize;iu++,U+=dU)
		{
		    tcoords[offset++] = U;
		    tcoords[offset++] = V;
		}
		V += dV;
	    }
	    kgeom_set_attribute(kout,KGEOM_OBJECT,KGEOM_TEXTURE_COORD_SIZE,2);
	    kgeom_put_data(kout,KGEOM_QUADMESH_TEXTURE_COORD_ALL,tcoords);
	    kgeom_get_attribute(kout,KGEOM_OBJECT,KGEOM_TEXTURE_COORD_SIZE,&iu);
	    if (iu != 2)
		fprintf(stderr," TEXTURE_COORD_SIZE attribute hosed.  wanted 2, but read back %d. \n",iu);
	}
	else /* assume use_tc == 3 */
	{
	    double U,dU,V,dV,W,dW;
	    int iu,iv,offset=0;
	    
	    tcoords = (float *)kmalloc(usize*vsize*3*sizeof(float));
	    
	    U = tc[0];
	    dU = (tc[1]-tc[0])/(usize-1);
	    V = tc[2];
	    dV = (tc[3]-tc[2])/(vsize-1);
	    W = tc[4];
	    dW = (tc[5]-tc[4])/(vsize-1);

	    /**
	      * in the absence of anything better, W will be interpolated
	      * across the V dimension.
	    **/
	    for (iv = 0;iv<vsize;iv++)
	    {
		U = tc[0];
		for (iu=0;iu<usize;iu++,U+=dU)
		{
		    tcoords[offset++] = U;
		    tcoords[offset++] = V;
		    tcoords[offset++] = W;
		}
		V += dV;
		W += dW;
	    }
	    kgeom_set_attribute(kout,KGEOM_OBJECT,KGEOM_TEXTURE_COORD_SIZE,3);
	    kgeom_put_data(kout,KGEOM_QUADMESH_TEXTURE_COORD_ALL,tcoords);
	    
	    kgeom_get_attribute(kout,KGEOM_OBJECT,KGEOM_TEXTURE_COORD_SIZE,&iu);
	    if (iu != 3)
		fprintf(stderr," TEXTURE_COORD_SIZE attribute hosed.  wanted 3, but read back %d. \n",iu);
	}
	kfree(tcoords);
    }
    
    kfree(iloc);
    return(CHILL);
}

int
lcompute_quadmesh_vertex_normals(vertex_3d *oloc,
				 vertex_3d **normals,
				 int usize,
				 int vsize)
{
    int uv,vv;
    int i,j,k;
    vertex_3d *n;
    vertex_3d s1,s2;
    vertex_3d *v;
    double mag;

    /**
      * here, we compute normals on a per-vertex basis.  if per-facet
      * normals are desired, they could be derived from those
      * computed here, or from the raw vertices directly
    **/

    uv = usize;
    vv = vsize;

    /* alloc space for one normal per vertex */
    n = (vertex_3d *)kmalloc(sizeof(vertex_3d)*uv*vv);
    
    /* alloc work space for vertices. */
/*    v = (vertex_3d *)kmalloc(sizeof(vertex_3d)*uv*vv); */
    v = oloc;


    /* do general case first. */
    for (j=0;j<vv;j++)
    {
	for (i=0;i<uv;i++)
	{
	    if ((i != 0) && (i != uv-1) && (j != 0) && (j != vv-1))
	    {
		for (k=0;k<3;k++)
		{
		    s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
		    s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
		}
		vertex_unit(&s1,&mag);
		vertex_unit(&s2,&mag);
		vertex_cross(&s1,&s2,(n+i+j*uv));
	    }
	}
    }
    
    /* do the j=0 row. */
    j=0;
    for (i=0;i<uv;i++)
    {
	if ((i != 0) && (i != uv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+1)->v[k] - (v+i-1)->v[k];
		s2.v[k] = (v+i+uv)->v[k] - (v+i)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i));
	}
    }
    
    /* do the j=vv-1 row. */
    j=vv-1;
    for (i=0;i<uv;i++)
    {
	if ((i != 0) && (i != uv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
		s2.v[k] = (v+i+j*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i+j*uv));
	}
    }
    
    /* do the i=0 column */
    i=0;
    for (j=0;j<vv;j++)
    {
	if ((j != 0) && (j != vv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i+j*uv)->v[k];
		s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i+j*uv));
	}
    }
    
    /* do the i=uv column */
    i=uv-1;
    for (j=0;j<vv;j++)
    {
	if ((j != 0) && (j != vv-1))
	{
	    for (k=0;k<3;k++)
	    {
		s1.v[k] = (v+i+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
		s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
	    }
	    vertex_unit(&s1,&mag);
	    vertex_unit(&s2,&mag);
	    vertex_cross(&s1,&s2,(n+i+j*uv));
	}
    }
    /* now, do the four corners. these normals are computed as the
      cross product of the segments adjoining the 2 nearest points. */
    
    /* i==0,j==0 */
    i=0;
    j=0;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i+j*uv)->v[k];
	s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+j*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));
    

    /* i=uv-1,j=0 */
    i=uv-1;
    j=0;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
	s2.v[k] = (v+i+(j+1)*uv)->v[k] - (v+i+j*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));
    
    /* i=0,j=vv-1 */
    i=0;
    j=vv-1;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+1+j*uv)->v[k] - (v+i+j*uv)->v[k];
	s2.v[k] = (v+i+j*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));
    
    /* i=uv-1,j=vv-1 */
    i=uv-1;
    j=vv-1;
    for (k=0;k<3;k++)
    {
	s1.v[k] = (v+i+j*uv)->v[k] - (v+i-1+j*uv)->v[k];
	s2.v[k] = (v+i+j*uv)->v[k] - (v+i+(j-1)*uv)->v[k];
    }
    vertex_unit(&s1,&mag);
    vertex_unit(&s2,&mag);
    vertex_cross(&s1,&s2,(n+i+j*uv));
    
/*    kgeom_add_normals(kobj,n,uv*vv); */

#if 0
    free((char *)v);
    free((char *)n);
#endif
    *normals = n;
    return(CHILL);
}
/* -library_code_end */
