 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>      Adding/Deleting Clusters from Classes
   >>>>
   >>>>  Private:
   >>>>           spc_add_or_delete_cluster
   >>>>           spc_add_cluster_to_class
   >>>>           spc_delete_cluster_from_class
   >>>>           spc_update_image_position
   >>>>           spc_set_pseudo_color
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "spectrum.h"

/*-----------------------------------------------------------
|
|  Routine Name: spc_add_or_delete_cluster
|
|       Purpose: Adds the specified cluster to the current class or
|                deletes the specified cluster from the current class.
|
|         Input: cluster     - the cluster to be added or deleted
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Feb 20, 1995
| Modifications: 
|
------------------------------------------------------------*/
void spc_add_or_delete_cluster(
    int cluster)
{
        /*
         * add cluster to class only if it is not already assigned
         */
        if (gui_info->cluster_operation_val == SPC_ADD)
        {
            if ((spc_legend_lookup == NULL) ||
                (spc_legend_lookup[cluster] == NULL))
            {
                if (spc_add_cluster_to_class(spc_current_class, cluster))
                {
                    kfprintf(kstderr,
                            "Adding cluster %d to current class '%s' \n",
                            cluster, spc_legend_lookup[cluster]->classname);
                }

                /* if this is the first cluster we add to a class, need to
                   set the pseudocolor list */
                if ((spc_current_class->clusters != NULL) &&
                    (spc_current_class->clusters[0] == cluster))
                     spc_set_pseudo_color(spc_current_class);
            }
            spc_update_classcontents_label(spc_current_class);
        }

        /*
         *  only want to delete clusters that are currently assigned
         */
        else
        {
            if ((spc_legend_lookup != NULL) &&
                (spc_legend_lookup[cluster] == spc_current_class))
            {
                if (spc_delete_cluster_from_class(spc_current_class,cluster))
                {
                    kfprintf(kstderr,
                            "Deleting cluster %d from current class '%s' \n",
                             cluster, spc_current_class->classname);
                }
            }
            spc_update_classcontents_label(spc_current_class);
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_add_cluster_to_class
|
|       Purpose: Adds the specified cluster to the specified class
|
|         Input: legend_ptr  - points to legend node w/ class to be added to
|                cluster     - the cluster to add to the class
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/

int spc_add_cluster_to_class(
    LegendEntry *legend_ptr,
    int          cluster)
{
	int  i, num_bytes;
	char temp[KLENGTH];

	/*
	 *  allocate (or re-allocate) clusters array
	 */
        num_bytes = (legend_ptr->clusternum + 1) * sizeof(int);

        if (legend_ptr->clusters == NULL)
             legend_ptr->clusters = (int *) kmalloc(sizeof(int));
        else legend_ptr->clusters = (int *) krealloc(legend_ptr->clusters,
                                                     num_bytes);
	/*
	 * return failure if we cannot allocate
	 */
	if (legend_ptr->clusters == NULL)
	{
	    kerror(NULL, "spc_add_cluster_to_class",
		   "Unable to allocate memory for internal legend pointer");
	    kinfo(KHOSTILE, "Buy a new computer, this one is worthless.");
	    return(FALSE);
	}

	/*
	 *  protect against a cluster being added twice
	 */
	for (i = 0; i < legend_ptr->clusternum; i++)
	{
	    if (legend_ptr->clusters[i] == cluster)
		return(TRUE);
	}

	/*
	 *  set the cluster, increment cluster num, set the spc_legend_lookup
	 */
        legend_ptr->clusters[legend_ptr->clusternum] = cluster;
        legend_ptr->clusternum++;
	if (spc_legend_lookup == NULL)
	    spc_legend_lookup = (LegendEntry **)
				 kcalloc(1, sizeof(LegendEntry *));
        spc_legend_lookup[cluster] = legend_ptr;
        legend_ptr->hidden = FALSE;


	/*
	 * 1ST CLUSTER OF CLASS CREATED DIRECTLY BY THE USER:
	 * if this is the first cluster to be added to the class, and color
	 * has not already been defined, set the class color to be the color
	 * of the pixel.
	 */
	if ((legend_ptr->clusternum == 1) &&
	    (legend_ptr->redval     == 0) &&
	    (legend_ptr->greenval   == 0) &&
	    (legend_ptr->blueval    == 0) &&
	    (legend_ptr->orig_type  == SPC_USER_CREATED))
	{
	    /*
             *  add cluster to the colorcell list as 1st cluster for class
	     */
	    if (legend_ptr->colorbox != NULL)
	        xvw_set_attribute(legend_ptr->colorbox, 
				  XVW_COLORCELL_ADD, cluster);
	    
	    /*
	     *  change pseudocolor RGB bars to reflect color of 1st cluster
	     */
	    if (legend_pseudo != NULL)
	        xvw_set_attribute(legend_pseudo, XVW_PSEUDO_ADD, cluster);

	    /*
	     *  find out what the color of the 1st cluster is,
             *  and reflect RGB values in the label that prints RGB values
	     */
	    xvw_get_attributes(legend_ptr->colorbox,
                               XVW_COLORCELL_REDVAL,   &(legend_ptr->redval),
                               XVW_COLORCELL_GREENVAL, &(legend_ptr->greenval),
                               XVW_COLORCELL_BLUEVAL,  &(legend_ptr->blueval),
                               NULL);
	    ksprintf(temp, " RGB values:  %d %d %d", legend_ptr->redval, 
		     legend_ptr->greenval, legend_ptr->blueval);
	    xvw_set_attribute(legend_ptr->RGBobj, XVW_LABEL,  temp);
	}

	/*
         * 1ST CLUSTER OF CLASS CREATED BECAUSE IT WAS LISTED IN LGD FILE:
         * if this is the first cluster to be added to the class, and color
         * has not already been defined, set the class color to be the color
         * of the pixel.
         */
        else if ((legend_ptr->clusternum == 1) &&
                 (legend_ptr->orig_type  == SPC_LGD_CREATED))
        {
	    /*
             *  add first cluster to the colorcell list for the class,
             *  so that it will appear in the color specified by legend.
             */
            if (legend_ptr->colorbox != NULL)
                xvw_set_attribute(legend_ptr->colorbox,
                                  XVW_COLORCELL_ADD, cluster);

	    /*
	     *  change colorbox so that it appears in colors specified by lgd.
	     */
            xvw_set_attributes(legend_ptr->colorbox,
                               XVW_COLORCELL_REDVAL,   legend_ptr->redval,
                               XVW_COLORCELL_GREENVAL, legend_ptr->greenval,
                               XVW_COLORCELL_BLUEVAL,  legend_ptr->blueval,
                               NULL);
	    /*
	     *  add first cluster to pseudocolor list, so that RGB scrollbars
	     *  will reflect color specified by the legend.
             */
	    if (legend_pseudo != NULL)
	        xvw_set_attribute(legend_pseudo, XVW_PSEUDO_ADD, cluster);
        }

	/*
 	 * LATER CLUSTERS ADDED TO CLASSES THAT ALREADY HAVE CLUSTERS
	 */
	else
	{
	    /*
             *  add latest cluster to the colorcell list for the class,
             *  so that it will also appear in the color for the class
	     */
	    if (legend_ptr->colorbox != NULL)
	        xvw_set_attribute(legend_ptr->colorbox, 
				  XVW_COLORCELL_ADD, cluster);

	    /*
	     *  add latest cluster to pseudocolor list,
	     *  so that it will change color when RGB scrollbars are used. 
             */
	    if (legend_pseudo != NULL)
	        xvw_set_attribute(legend_pseudo, XVW_PSEUDO_ADD, cluster);
	}

	/*
	 * mark class id in spc_classes array
	 */
	spc_classes[cluster] = legend_ptr->class;

	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_delete_cluster_from_class
|
|       Purpose: Deletes the specified cluster from the specified class
|
|         Input: legend_ptr  - points to legend node w/ class to be deleted from
|                cluster     - the cluster to delete from the class
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: June 13, 1993
| Modifications: Updated from Khoros 1.0 (DA)
|
------------------------------------------------------------*/
int spc_delete_cluster_from_class(
    LegendEntry *legend_ptr,
    int         cluster)
{
        int i = 0;

        /* 
	 * no clusters to be deleted 
         */
        if (legend_ptr->clusternum == 0) 
	    return(FALSE);

	/* 
	 * find the cluster to be deleted 
         */
        while ((legend_ptr->clusters[i] != cluster) &&
               (i < legend_ptr->clusternum)) i++;

        /* 
	 * failure if cluster to be deleted is not in this class 
         */
        if (i == legend_ptr->clusternum) 
	    return(FALSE);

        /* 
         * delete the cluster from the class: scoot back all other clusters
         * to over-write this cluster's element, decrement cluster num,
         * unset the spc_legend_lookup
         */
        legend_ptr->clusternum--;
        while (i < legend_ptr->clusternum)
        {
            legend_ptr->clusters[i] = legend_ptr->clusters[i+1]; i++;
        }
        spc_legend_lookup[cluster] = NULL;

	/*
	 *  delete cluster from the group of clusters in the color of the class
	 */
	xvw_set_attribute(legend_ptr->colorbox, XVW_COLORCELL_DELETE, cluster);

	/*
	 * unset class id in spc_classes array
	 */
	spc_classes[cluster] = -1;

	/*
	 *  this is the last cluster.
	 *  We *really* want to restore the color of the image to normal,
	 *  and the color of the colorbox to the color defined for the
	 *  class.  However, right now the calls to set XVW_COLORCELL_REDVAL,
	 *  XVW_COLORCELL_GREENVAL, and XVW_COLORCELL_BLUEVAL won't work
	 *  if there is no index assigned to the colorcell. So turn the 
	 *  colorcell white "by hand".
	 */
	if (legend_ptr->clusternum == 0)
	{
	    xvw_set_attributes(legend_ptr->colorbox,
        /*               XVW_COLORCELL_REDVAL,   legend_ptr->redval,
                         XVW_COLORCELL_GREENVAL, legend_ptr->blueval,
                         XVW_COLORCELL_BLUEVAL,  legend_ptr->greenval,  */
                         XVW_COLORCELL_UPDATE,   TRUE,
                         NULL);
	    xvw_set_attribute(legend_ptr->colorbox, 
		   	      XVW_BACKGROUND, white);
	}

        /*
         *  update the "Class Contents" display
         */
	spc_update_classcontents_label(legend_ptr);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: spc_update_image_position
|
|       Purpose: updates the image position display, on which the
|                color of the class of the cluster is displayed,
|                along with the cluster number and the 
|                position of the pointer in the image workspace
|
|         Input: object      - main image workspace, 
|			       zoom workspace, or scatterplot
|		 client_data - not used
|		 event       - the PointerMotion event
|		 dispatch    - whether or not to propogate event (not used)
|        Output: None
|       Returns: None
|    Written By: Danielle Argiro
|          Date: May 12, 1993
| Modifications:
|
------------------------------------------------------------*/

/* ARGSUSED */
void spc_update_image_position(
    xvobject  object,
    kaddr     client_data,
    XEvent    *event,
    int       *dispatch)
{
	Coord *coords;
	int   i, cluster, point_index, xposition, yposition;
	char  temp[KLENGTH], position[KLENGTH];
	unsigned long fg;
	double tmp_double;

	/* 
	 * here, the event came from the scatter plot.  if we can't identify
	 * the device coordinate as a point in the plot, just return;  if
	 * so, then get the pixel (cluster) value represented by the point.
	 */
	if (object == spc_scatter->area)
	{
            if (!(spc_identify_plot_point(event->xbutton.x,
                                          event->xbutton.y,
                                          &cluster, &point_index)))
                return;

	     ksprintf(position, "(Not Applicable)");
	}

	/*
	 * event came from the main spectrum workspace or the 
	 * zoom window. get pixel value at that point.
	 */
	else 
	{
	     xvw_get_attributes(object, 
				XVW_IMAGE_VALUE,     &tmp_double,
			        XVW_IMAGE_XPOSITION, &xposition,
				XVW_IMAGE_YPOSITION, &yposition,
				NULL);
             cluster = (int) tmp_double;
	     ksprintf(position, "(%d x %d)", xposition, yposition);
	}
	     
	/*
	 *  update label to indicate cluster number, associated class (if any)
	 */
	if ((spc_legend_lookup != NULL) && 
            (spc_legend_lookup[cluster] != NULL))
	    ksprintf(temp,  "    Cluster Number %d, member of class '%s'", 
		     cluster, (spc_legend_lookup[cluster])->classname);
	else ksprintf(temp, "    Cluster Number %d, not member of class", 
                     cluster);
	xvw_set_attribute(spc_display->cluster_label, XVW_LABEL, temp);

	/*
	 * update label to show image position
	 */
	ksprintf(temp, "Image Position = %s", position);
	xvw_set_attribute(spc_display->position_label, XVW_LABEL, temp);

	xvw_set_attribute(spc_display->colorbox, XVW_COLORCELL_INDEX, cluster);
}


/*-----------------------------------------------------------
|
|  Routine Name: spc_set_pseudo_color
|
|       Purpose: Updates the psuedocolor bars to reflect the color 
|                associated with the class, and makes sure that all 
|                the clusters in the class appear in that color.
|                Also updates the associated point in the scatter plot
|                to appear in that color.
|
|         Input: legend_ptr  - legend pointer to current class
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Nov 3, 1993
| Modifications: 
|
------------------------------------------------------------*/

void spc_set_pseudo_color(
    LegendEntry *legend_ptr)
{
	int i, point_index;

	if (legend_ptr == NULL) return;

	/* clear the list of indices in the pseudocolor object */
	if (legend_pseudo != NULL)
	    xvw_set_attribute(legend_pseudo, XVW_PSEUDO_CLEAR, 0);

	/* 
	 * add in the indices associated with this class.  this
         *  will have the effect both of updating the pseudocolor
	 *  bars to the color associated with the current class,
         *  as well as changing the clusters in the class to be
         *  that color (if they aren't already)
         */
	if ((legend_pseudo != NULL) && (legend_ptr != NULL))
	{
	    for (i = 0; i < legend_ptr->clusternum; i++)
	        xvw_set_attribute(legend_pseudo, XVW_PSEUDO_ADD, 
			          legend_ptr->clusters[i]);
	}

	if ((legend_ptr != NULL) && (legend_ptr->clusternum > 0))
	{
	   point_index = spc_scatter->plot_lookup[legend_ptr->clusters[0]];
           spc_scatter->plotdata[point_index].d = legend_ptr->clusters[0]; 
	}
}

