#include <xvinclude.h>

/*
 *  This program displays 3D data as 3 orthogonal slices.  It puts up
 *  a display that looks like this:
 *
 *   -------------------------------------------
 *   |                                         |
 *   |  ------------------------  -----------  |
 *   |  |                      |  |         |  |
 *   |  |    Z image:          |  | X image:|  |
 *   |  |                      |  |         |  |
 *   |  |   displays volume    |  | displays|  |
 *   |  |         as           A  |  volume |  |
 *   |  |   (width X height)   |  |   as    |  |
 *   |  |                      |  |(depth X |  |
 *   |  |                      |  | height) |  |
 *   |  |                      |  |         |  |
 *   |  |                      |  |         |  |
 *   |  ----------B------------|  ----------|  |
 *   |  -----------------------                |
 *   |  |    Y image:         |                |
 *   |  | displays volume as  |   (position)   |
 *   |  |  (width x depth)    |                |
 *   |  -----------------------                |
 *   |_________________________________________|
 *
 *
 *   To help in picturing the volume, imagine the X image to the right as 
 *   folded into the screen by 90 degrees, from the right edge of the Z image
 *   marked A;  imagine the Y image as folded into the screen by 90 degrees,
 *   from the bottom edge of the Z image marked B. Thus, the display represents
 *   half of a cube that has been "flattened" onto the screen.
 *
 *   When you click in any of the images, the position object will indicate 
 *   the value at that position, where the dimensions are specified by the 
 *   image you clicked in -- it is (width x height) in the Z image, 
 *   (depth x height) in the X image, and (width x depth) in the Y image.
 *
 *   Event handlers are installed on each of the images so that if you click
 *   anywhere in an image, the other two images will be updated so as to
 *   display that same location from the other two views.
 */

static int xslice, yslice, zslice;
static xvobject ximage, yimage, zimage;

/*
 *  event handler to get mouse coordinates in zimage
 */
void  
zimage_handler( xvobject object, kaddr clientData, 
                XEvent *event, Boolean  *dispatch)
{
	int x,y;

	xvw_get_attribute(object, XVW_IMAGE_XPOSITION, &x);
	xvw_get_attribute(object, XVW_IMAGE_YPOSITION, &y);
	xslice = x;
	yslice = y;

	xvw_set_attribute( ximage, XVW_IMAGE_BANDNUM,  xslice);
	xvw_set_attribute( yimage, XVW_IMAGE_BANDNUM,  yslice);
	printf("%d %d\n",x,y);
}

/*
 *  event handler to get mouse coordinates in ximage
 */
void  
ximage_handler( xvobject object, kaddr clientData, 
                XEvent *event, Boolean  *dispatch)
{
	int z,y;

	xvw_get_attribute(object, XVW_IMAGE_XPOSITION, &z);
	xvw_get_attribute(object, XVW_IMAGE_YPOSITION, &y);
	zslice = z;
	yslice = y;

	xvw_set_attribute( zimage, XVW_IMAGE_BANDNUM,  zslice);
	xvw_set_attribute( yimage, XVW_IMAGE_BANDNUM,  yslice);
	printf("%d %d\n",z,y);
}

/*
 *  event handler to get mouse coordinates in yimage
 */
void  
yimage_handler( xvobject object, kaddr clientData, 
                XEvent *event, Boolean  *dispatch)
{
	int x,z;

	xvw_get_attribute(object, XVW_IMAGE_XPOSITION, &x);
	xvw_get_attribute(object, XVW_IMAGE_YPOSITION, &z);
	xslice = x;
	zslice = z;

	xvw_set_attribute( ximage, XVW_IMAGE_BANDNUM,  xslice);
	xvw_set_attribute( zimage, XVW_IMAGE_BANDNUM,  zslice);
	printf("%d %d\n",x,z);
}

void main(
   int  argc,
   char *argv[],
   char *envp[])
{
	kobject  object;
	int      i, mask;
	char     *filename = "volume:sagittal";
	xvobject parent, last = NULL;
	xvobject position;

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	if (argc > 1)
	   filename = argv[1];
	
	xslice = yslice = zslice = 0;
	object = kpds_open_object(filename, KOBJ_READ);
	kcolor_set_attribute(object, KCOLOR_MAP_AUTOCOLOR, KRGB_CUBE); 

	/* initialize the xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror(NULL, "main", "unable to open display");
	   kexit(KEXIT_FAILURE);
	}

	/*
	 * create a manager to be the backplane 
	 */
	parent = xvw_create_manager(NULL, "parent");
	xvw_set_attributes(parent, 	
		XVW_SELECTABLE, TRUE,
		XVW_RESIZABLE,  TRUE,
		NULL);

	/*
	 *  This is the larger image in the center of the display;
	 *  it will display the original axis assignment of the volume:
	 *  (width, height, depth).  The image object is wired to interpret
	 *  the first two dimensions as image width and image height, so
	 *  this will present the "default" view of the volume.
	 */
	zimage = xvw_create_image(parent, "zimage");
	xvw_set_attributes( zimage,
		XVW_IMAGE_IMAGEOBJ, object,
		XVW_IMAGE_BANDNUM,  zslice, 
		XVW_RIGHT_OF,       NULL,
		XVW_BELOW,          NULL,
		NULL);

	/*
	 *  This is the smaller image that appears to the right;
	 *  it will display the (depth x height) side of the image.
         *  The key to the way this works is the setting of the
	 *  KPDS_AXIS_ASSIGNMENT attribute;  by setting it to
	 *  (depth, height, width) here, we will "fool" the image object
	 *  into showing the side of the image  (remember that the image
	 *  object will interpret the 1st two dimensions, whatever they are,
	 *  as image width and height).
	 */
	ximage = xvw_create_image(parent, "ximage");
	kpds_set_attribute(object, KPDS_AXIS_ASSIGNMENT,
			  KDEPTH, KHEIGHT, KWIDTH);

	xvw_set_attributes( ximage,
		XVW_IMAGE_IMAGEOBJ, object,
		XVW_IMAGE_BANDNUM,  xslice, 
		XVW_RIGHT_OF,       zimage,
		NULL);

	/*
         *  This is the smaller image that appears to the bottom;
         *  it will display the (width x depth) side of the image.
         *  Again, we use the KPDS_AXIS_ASSIGNMENT attribute to specify
	 *  (width x depth) as the dimensions that the image object will
	 *  interpret as (width x height).
         */
	yimage = xvw_create_image(parent, "yimage");
	kpds_set_attribute(object,KPDS_AXIS_ASSIGNMENT,
			  KWIDTH, KDEPTH, KHEIGHT);

	xvw_set_attributes( yimage,
		XVW_IMAGE_IMAGEOBJ, object,
		XVW_IMAGE_BANDNUM,  yslice, 
		XVW_BELOW,          zimage,
		XVW_RIGHT_OF,       last,
		NULL);

	/*
	 *  throw in a position object, so we can see where we are
	 *  in the image.  it will work on any of the three images.
	 */
	position = xvw_create_position(parent, "position");
	xvw_set_attributes( position,
		XVW_POSITION_OBJECT, object,
		XVW_BELOW,           yimage,
		XVW_LEFT_OF,         last,
		NULL);

	mask = ButtonPressMask|ButtonReleaseMask|ButtonMotionMask;
	xvw_add_event(zimage, mask, zimage_handler, NULL);
	xvw_add_event(yimage, mask, yimage_handler, NULL);
	xvw_add_event(ximage, mask, ximage_handler, NULL);

	/* display and run */
	xvf_run_form();
}
