#include <xvinclude.h>

#define IMAGE_WIDTH  512
#define IMAGE_HEIGHT 256
#define IMAGE_SIZE   131072 /* IMAGE_HEIGHT x IMAGE_WIDTH */

void initialize_data PROTO((void));
void alter_or_quit PROTO((xvobject, kaddr, XEvent *, Boolean *));

/*
 * This example program creates a window containing an image object 
 * that we will update in different ways.  The initial image is empty,
 * containing no data -- it will appear to be a black image.
 *
 * An event handler is placed on the image visual object.  This event
 * handler will check for button press events.  Whenever a button is
 * pressed, it will initiate an update of the image.The same event
 * handler allows the user to quit the program by pressing 'q' or 'Q'.
 *
 * The mechanism by which the image is updated is via data services 
 * data callbacks -- the installation of these callbacks is all handled 
 * by the image visual object.  You need not do anything to set this up
 * other than set the XVW_IMAGE_IMAGEOBJ attribute on the visual object.
 *
 * Through the magic of data callbacks, whenever data is written to your
 * data object, the visual object will update only the region that has
 * changed.  This example program illustrates writing different units
 * of data and shows how fast they are updated on the image.
 *
 * CLICK THE MOUSE on the image to start the first update.  This
 * creates a sine image, putting data one line at a time.  This is
 * being done with backing store turned on.
 * 
 * CLICK THE MOUSE on the image to go to the next update.  This
 * creates a cosine image, putting data one line at a time.  This time
 * backing store is turned off.  Notice that the update time is much
 * faster.
 *
 * CLICK THE MOUSE on the image to go to the next update.  This
 * creates a sine image, putting the entire image at once. 
 *
 * CLICK THE MOUSE on the image to go to the next update.  This
 * creates a grey region on the image.  This is a simple region put.
 *
 * Note that whether the data being put is a single line, or the whole
 * image, or even just a region, the update time is roughly the same.
 *
 * Clicking again will cycle back to the first case. 
 *
 *
 * SOME COMMENTS ON BACKING STORE :
 *
 * The drawback of this feature is this : annotations can not be interactively
 * moved around on the image as cleanly.  If you don't need interactive 
 * annotations, you certainly don't need to consider setting this attribute.
 *
 * Under consideration is a method for the image visual object to automagically
 * detect if any of it's children are annotations (or more precisely, if
 * they are gadgets) and set the backing store accordingly.
 * 
 */
void main(
   int  argc,
   char *argv[],
   char *envp[])
{
        xvobject object;  /* image visual object */
	kobject  data_object;  /* data services data object */
	
        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

	/* initialize xvwidgets library */
        if (!xvw_initialize(XVW_MENUS_XVFORMS))
        {
           kerror(NULL, "main", "unable to open display");
           kexit(KEXIT_FAILURE);
        }

	/* create image visual object w/ default parent, default toplevel */
        object = xvw_create_image(NULL, "image");

	/* create an empty data object in which to place the image data */
	data_object = kpds_create_object();

	/* initialize the data object to have value data */
	kpds_create_value(data_object);	

	/* 
	 * set the size of the value segment, along with its data type.
	 *
	 *  The size arguments are (in order) :
	 *    width, height, depth, time, elements
	 *
	 *  We want our image to be 512 pixels wide by 512 pixels high.
	 *
	 *  Since we have an image, and not a volume, our depth is 1.
	 *
	 *  Since we have only one image, and not a time series of
	 *  images, out time size is 1.
	 *
	 *  Our element size will contain the per-pixel data.  Since we
	 *  are going to be mapping singular values into a separate RGB
	 *  colormap, we will have an element size of 1.  If we were
	 *  going to fill out the RGB values directly, we could make 
	 *  this element size 3.   The image visual object can cope with 
	 *  either.
	 *
	 */
	kpds_set_attributes(data_object, 
    		    KPDS_VALUE_SIZE, IMAGE_WIDTH, IMAGE_HEIGHT, 1, 1, 1,
    		    KPDS_VALUE_DATA_TYPE, KUBYTE,
		    NULL);
	
	/*
	 *  Tell the visual object it needs to display the image we
	 *  just created.  It will be black (0) initially, but we will
	 *  be changing the data later in the image callback.
	 */	
	xvw_set_attribute(object, XVW_IMAGE_IMAGEOBJ, data_object);

	/* add event handler to alter image & quit */
	xvw_add_event(object, ButtonPressMask | KeyPressMask,
		      alter_or_quit, data_object);

	/* Initialize all the data to be put -- this is done separately
	 * so that the event handler illustrates only the update time 
	 */
	initialize_data();
	
	/* print a prompt explaining what to do */
	kfprintf(kstdout, "\nPLACE the window and then CLICK on the "
		 "image to see an update...");

	/* display & run */
        xvf_run_form();
}

/* global data to be used for putting */
unsigned char sin_data[IMAGE_WIDTH];
unsigned char cos_data[IMAGE_WIDTH];
unsigned char all_data[IMAGE_SIZE];
unsigned char region_data[2500];

/* 
 * initialize all the relevant data in this routine so the event handler
 * shows only the update time
 */
void initialize_data()
{
   int i, j;
   

   /* Generate a row of sinusoidal image data */
   for (i = 0; i < IMAGE_WIDTH; i++)
      sin_data[i] = (unsigned char) (128.0 * ksin(i*15.0/IMAGE_WIDTH)) + 127;

   /* Generate a row of cosinusoidal image data */
   for (i = 0; i < IMAGE_WIDTH; i++)
      cos_data[i] = (unsigned char) (128.0 * kcos(i*15.0/IMAGE_WIDTH)) + 127;

   /* Generate an entire image of sinusoid data */
   for (i = 0; i < IMAGE_WIDTH; i++)
      for (j = 0; j < IMAGE_HEIGHT; j++)
	 all_data[i+j*IMAGE_WIDTH] = (unsigned char)
	                 (128.0 * ksin(j*15.0/IMAGE_HEIGHT)) + 127;

   /* Generate a 50x50 region */
   for (i = 0; i < 2500; i++)
      region_data[i] = 100;
}


/*
 * the event handler which will update the image on Button Press,
 * or quit the program on Key press 'Q' or 'q'
 */
void alter_or_quit(
   xvobject object,    
   kaddr    clientData,
   XEvent   *event,
   Boolean  *dispatch)
{

	char   ch;
	static int count = 1;
	kobject data_object = (kobject) clientData;
        int i;
	
	

	/* 
	 * since there is no formalized GUI, need a way to exit
         * the program. quit if they type 'Q' or 'q'
	 */
	if (event->type == KeyPress)
	{
	   if (XLookupString(&(event->xkey), &ch, 1, NULL, NULL)  == 0)
	      return;
	   if (ch == 'q' || ch == 'Q')
	   {
	        xvw_destroy(object);
		kexit(KEXIT_FAILURE);
	   }
	}

	/*
	 *  If they button pressed in the image, switch to next image.
	 */
        else if (event->type == ButtonPress)
        {
	   switch (count)
	   {
	      /* Put data one line at a time without backing store */
	      case 1 :
		 {
		    /* Print an explanation of what is happening */
		    kfprintf(kstdout, "\nEXAMPLE of updating one line at "
			     "a time with XVW_IMAGE_BACKING set to FALSE");

		    /* Ensure that the backing store is on */
		    xvw_set_attribute(object, XVW_IMAGE_BACKING, TRUE);

    		    /* Put the sine row of data, one line at a time for each 
		     * row in the image
		     */
		    for (i = 0; i < IMAGE_HEIGHT; i++)
		       kpds_put_data(data_object, KPDS_VALUE_LINE, sin_data);

		    /* Go to the next case on the next button press */
		    count++;
		    kfprintf(kstdout, "\n\nCLICK on the image to see an "
			     "update...");
		    
		 } break;
		 
	      /* Put data one line at a time *with* backing store */
	      case 2 :
		 {
		    /* Print an explanation of what is happening */
		    kfprintf(kstdout, "\nEXAMPLE of updating one line at "
			     "a time with XVW_IMAGE_BACKING set to TRUE");

		    /* Turn off the backing store */
		    xvw_set_attribute(object, XVW_IMAGE_BACKING, FALSE);


    		    /* Put the cos row of data, one line at a time for each 
		     * row in the image
		     */
		    for (i = 0; i < IMAGE_HEIGHT; i++)
		       kpds_put_data(data_object, KPDS_VALUE_LINE, cos_data);

		    /* Go to the next case on the next button press */
		    count++;

		    kfprintf(kstdout, "\n\nCLICK on the image to see an "
			     "update...");
		    
		 } break;

	      /* Put all the image data at once */
	      case 3 :
		 {
		    /* Print an explanation of what is happening */
		    kfprintf(kstdout, "\nEXAMPLE of updating entire image "
			     "at once");

		    /* Put the image data */
		    kpds_put_data(data_object, KPDS_VALUE_ALL, all_data);

		    /* update differently on the next button press */
		    count++;

		    kfprintf(kstdout, "\n\nCLICK on the image to see an "
			     "update...");
		    
		 } break;

	      /* Put only a single region of data */
	      case 4 :
		 {
		    /* Print an explanation of what is happening */
		    kfprintf(kstdout, "\nEXAMPLE of updating only a region");

		    /* Set the position and region size*/
		    kpds_set_attributes(data_object, 
				KPDS_VALUE_POSITION, 30, 30, 0, 0, 0,
				KPDS_VALUE_REGION_SIZE, 50, 50, 1, 1, 1,
				NULL);
		    
		    /* Put the region data */
		    kpds_put_data(data_object, KPDS_VALUE_REGION, region_data);

		    /* Cycle back to the first case */
		    count = 1;		    
		    kfprintf(kstdout, "\n\nCLICK on the image to "
			     "start over  -or-  press 'q' to quit...\n\n");


		    /* Reset the position and region size*/
		    kpds_set_attribute(data_object, 
				       KPDS_VALUE_POSITION, 0, 0, 0, 0, 0);

		 } break;
	   }
	}
}

