#include <xvinclude.h>

/*
 * This example demonstrates how the XVW_COLORCELL_DELETE attribute
 * can be used to change restore pixels in an image to their original
 * color after they've been changed to a new color specified
 * by its Red, Green, and Blue components.
 *
 * Like the first example, this program creates an image object which displays 
 * the kitten image, and then creates a colorcell object.  
 *
 * When the "set red" button is clicked on, a number of the pixels in the 
 * image are colored red.  The event handler
 * which is installed for button motion in the image will delete the pixels
 * from the colorcell list, restoring them to their original values as the
 * user moves the pointer across the image, keeping the button down.
 *
 */

void  set_red_pixels	PROTO((xvobject, kaddr, kaddr));
void  quit_cb       	PROTO((xvobject, kaddr, kaddr));
void  restore_colorcell	PROTO((xvobject, kaddr, XEvent *, Boolean *));

void main(
   int  argc,
   char **argv,  
   char **envp)
{
	char *filename = "image:kitten";
	xvobject manager, image, colorcell, set, quit;

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

        /* initialize the xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror(NULL, "main", "unable to open display");
	   kexit(KEXIT_FAILURE);
	}

	if (argc > 1)
	   filename = argv[1];

        /* Create a manager to parent the image and colorcell objects  */
	manager = xvw_create_manager(NULL, "manager");

	/*
	 *  create the colorcell object, and associate it with the
         *  kitten image.  setting XVW_COLORCELL_RESTORE_ONDELETE to TRUE
         *  will cause pixels to have their original colors restored by the
         *  colorcell when they are deleted from the colorcell pixel list.
	 */
	colorcell = xvw_create_colorcell(manager, "colorcell");
	xvw_set_attributes(colorcell,
		XVW_COLORCELL_RESTORE_ONDELETE, TRUE,
		XVW_COLOR_COLORFILE,            filename,
		XVW_BELOW,	                NULL,
		XVW_LEFT_OF,	                NULL,
		XVW_WIDTH,                      35,
		XVW_HEIGHT,                     35,
		NULL);

	image = xvw_create_image(manager, "image");
	xvw_set_attributes(image,
		XVW_IMAGE_IMAGEFILE, filename,
		XVW_IMAGE_BACKING,   FALSE,
		XVW_BELOW,	     colorcell,
		NULL);

	/* 
	 * add the event handler to restore pixels to their original colors
         * when they are deleted from the colorcell list
         */
	xvw_add_event(image, ButtonMotionMask,
                      restore_colorcell, (kaddr)colorcell);

	/* add a "Set Red" button to color various pixels red */
	set = xvw_create_button(manager, "set");
	xvw_set_attributes(set, 
			   XVW_LABEL,   "Set Red",
			   XVW_LEFT_OF,  colorcell,
			   XVW_RIGHT_OF, NULL,
			   NULL);
	xvw_add_callback(set, XVW_BUTTON_SELECT, set_red_pixels, colorcell);

	/* add a quit button */
        quit = xvw_create_button(manager, "quit");
        xvw_set_attributes(quit,
                           XVW_LABEL,   "quit",
                           XVW_LEFT_OF,  colorcell,
                           XVW_RIGHT_OF, set,
                           NULL);

        xvw_add_callback(quit, XVW_BUTTON_SELECT,
                         quit_cb, manager);

	/* display & run */
	xvf_run_form();
}


/*
 *  event handler to restore the pixel colors by deleting 
 *  them from colorcell list
 */
void  restore_colorcell(
    xvobject object, 
    kaddr    clientData,  
    XEvent   *event,
    Boolean  *dispatch)
{
	xvobject colorcell = (xvobject) clientData;
	double   value;

	/* get the value of the pixel where the mouse was clicked */
	xvw_get_attribute(object, XVW_IMAGE_VALUE, &value);

	/* 
	 * delete the pixel to the colorcell pixel list.  since
         * XVW_COLORCELL_RESTORE_ONDELETE has been set to TRUE, the
         * pixel will be immediately recolored to it's original color.
         */ 
	if ((value == 0.0)  || (value == 10.0)  || (value == 20.0) ||
	    (value == 30.0) || (value == 40.0)  || (value == 50.0) ||
	    (value == 60.0) || (value == 70.0)  || (value == 80.0) ||
	    (value == 90.0) || (value == 100.0) || (value == 110.0))
	    xvw_set_attribute(colorcell, XVW_COLORCELL_DELETE, (int) value);
}



/*
 *  callback to set red pixels in image
 */
void  set_red_pixels(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject colorcell = (xvobject) client_data;
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 0);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 10);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 20);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 30);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 40);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 50);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 60);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 70);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 80);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 90);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 100);
	xvw_set_attribute(colorcell, XVW_COLORCELL_ADD, 110);
	xvw_set_attributes(colorcell, 
			   XVW_COLORCELL_REDVAL,   255,
			   XVW_COLORCELL_GREENVAL, 5,
			   XVW_COLORCELL_BLUEVAL,  5,
		           NULL);
}

/*
 *  callback to quit program
 */
void  quit_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject manager = (xvobject) client_data;
	xvw_destroy(manager);
}

