#include <xvinclude.h>

/*
 * This example demonstrates how the XVW_COLORCELL_REDVAL, 
 * XVW_COLORCELL_GREENVAL, and XVW_COLORCELL_BLUEVAL attributes 
 * can be used to change pixels in an image to be a particular color specified
 * by its Red, Green, and Blue components.
 *
 * Like the first & second examples, this program creates an image object 
 * which displays the kitten image, and then creates a colorcell object.  
 * The colorcell object is hardwired to reflect the color of pixel value 50.
 *
 * The "set" button will pop up a query object which allows you to enter,
 * values for red, green, and blue.  All pixels with value 50 will then
 * change to be the color specified.
 *
 * The "restore" button will restore pixel 50 to its original color.
 */

void  set_colorcell	PROTO((xvobject, kaddr, kaddr));
void  restore_colorcell	PROTO((xvobject, kaddr, kaddr));
void  quit_cb          	PROTO((xvobject, kaddr, kaddr));

void main(
     int  argc,
     char **argv,
     char **envp)
{
	char *filename = "image:kitten";
	xvobject manager, image, colorcell, set, restore, quit;

        /* initialize Khoros program */
        khoros_initialize(argc, argv, envp, "ENVISION");

        /* initialize the xvwidgets lib */
	if (!xvw_initialize(XVW_MENUS_XVFORMS))
	{
	   kerror(NULL, "main", "unable to open display");
	   kexit(KEXIT_FAILURE);
	}

        /* Create a manager to parent the image and colorcell objects  */
	manager = xvw_create_manager(NULL, "manager");

	/*
	 *  create the colorcell object, and associate it with the
         *  kitten image.  setting XVW_COLORCELL_INDEX to 50
         *  will cause the colorcell object to reflect the color of 
         *  pixel value 50.
	 */
	colorcell = xvw_create_colorcell(manager, "colorcell");
	xvw_set_attributes(colorcell,
		XVW_COLOR_COLORFILE, filename,
		XVW_COLORCELL_INDEX, 50,
		XVW_BELOW,	     NULL,
		XVW_LEFT_OF,	     NULL,
		XVW_WIDTH,           35,
		XVW_HEIGHT,          35,
		NULL);

	image = xvw_create_image(manager, "image");
	xvw_set_attributes(image,
		XVW_IMAGE_IMAGEFILE, filename,
		XVW_IMAGE_BACKING,   FALSE,
		XVW_BELOW,	     colorcell,
		NULL);

	/* add a set button to pop up query object for setting RGB values */
	set = xvw_create_button(manager, "set");
	xvw_set_attributes(set, 
			   XVW_LABEL,   "set",
			   XVW_RIGHT_OF, NULL,
			   NULL);
	xvw_add_callback(set, XVW_BUTTON_SELECT,
		         set_colorcell, colorcell);

	/* add a restore button to restore colors of pixels */
	restore = xvw_create_button(manager, "restore");
	xvw_set_attributes(restore, 
			   XVW_LABEL,   "restore",
			   XVW_RIGHT_OF, set,
			   NULL);
	xvw_add_callback(restore, XVW_BUTTON_SELECT,
		         restore_colorcell, colorcell);

	/* add a quit button */
        quit = xvw_create_button(manager, "quit");
        xvw_set_attributes(quit,
                           XVW_LABEL,   "quit",
                           XVW_RIGHT_OF, restore,
                           NULL);

        xvw_add_callback(quit, XVW_BUTTON_SELECT,
                         quit_cb, manager);

	/* display & run */
	xvf_run_form();
}


/*
 *  callback to set the colorcell RGB values 
 */
void  set_colorcell(
    xvobject object, 
    kaddr    client_data,  
    kaddr    call_data)
{
	int  i, redval, greenval, blueval;
	char **prompts, **answers;
	xvobject colorcell = (xvobject) client_data;

	/*
	 *  get current RGB values of pixel 50
	 */
	xvw_get_attributes(colorcell,
			   XVW_COLORCELL_REDVAL,   &redval,
			   XVW_COLORCELL_GREENVAL, &greenval,
			   XVW_COLORCELL_BLUEVAL,  &blueval,
			   NULL);

	/*
	 *  initialize prompts & answers arrays for query object
	 */
	prompts = (char **) kcalloc(3, sizeof(char *));
	answers = (char **) kcalloc(3, sizeof(char *));
	for (i = 0; i < 3; i++)
	{
	    answers[i] = (char *) kcalloc(1, KLENGTH*sizeof(char));
	    prompts[i] = (char *) kcalloc(1, KLENGTH*sizeof(char));
	}
	ksprintf(prompts[0], "Red  :");
	ksprintf(prompts[1], "Green:");
	ksprintf(prompts[2], "Blue :");


	ksprintf(answers[0], "%d", redval);
	ksprintf(answers[1], "%d", greenval);
	ksprintf(answers[2], "%d", blueval);

	/*
	 *  use xvu_query_wait to pop up query object and prompt the user
         *  for new Red, Green, and Blue values
	 */
	(void) xvu_query_wait("Provide new RGB values:", prompts, 
			      "ok", answers, 3, 25);

	redval   = atoi(answers[0]);
	greenval = atoi(answers[1]);
	blueval  = atoi(answers[2]);

	/*
	 *  set new RGB values for pixel 50
	 */
	xvw_set_attributes(colorcell,
                           XVW_COLORCELL_REDVAL,   redval,
                           XVW_COLORCELL_GREENVAL, greenval,
                           XVW_COLORCELL_BLUEVAL,  blueval,
                           NULL);

	karray_free(prompts, 3, NULL);
	karray_free(answers, 3, NULL);

}

/*
 *  callback to restore colorcell
 */
void  restore_colorcell(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
	xvobject colorcell = (xvobject) client_data;

	/* 
	 * restore pixel 50 to its original colors
         */
	xvw_set_attribute(colorcell, XVW_COLORCELL_RESTORE, TRUE);

}


/*
 *  callback to quit program
 */
void  quit_cb(
   xvobject object,
   kaddr    client_data,
   kaddr    call_data)
{
        xvobject manager = (xvobject) client_data;
	xvw_destroy(manager);
}

