 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Functionality routines for pane program
   >>>> 
   >>>>  Private: 
   >>>> 	initialize_create_program
   >>>> 	program_create
   >>>> 	program_toolboxes
   >>>> 	program_objects
   >>>> 	program_category
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "cantata.h"
#include "kcms/kcms.h"

/*-----------------------------------------------------------
| 
|  Routine Name: initialize_create_program
| 
|       Purpose: routine which is called when initializing the toolbox
|		 list
| 
|         Input: program_info - ptr to PaneInfo struct for program pane
| 
|        Output: None
|    Written By: Mark Young
|          Date: Feb 03, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void initialize_create_program(
     create_program *program_info)
{
	int     num;
	klist   *list = NULL;
	char    **tmp, *name;
	kform_struct *formstruct;

	/*
	 *  get array of toolbox names
	 */
	if ((tmp = kcms_query_toolboxes(&num)) == NULL)
	{
	   kinfo(KSTANDARD, "Warning: No toolboxes accessable via cantata");
	   return;
	}

	/*
	 *  Now update the toolbox display list object.  This will allow
	 *  the user to browse existing toolboxes in which to create their
	 *  cantata workspace program.
	 */
	tmp = karray_sort(tmp, num, TRUE);
	formstruct = program_info->toolboxes_struct;
	xvf_set_attribute(formstruct, XVF_LIST_SIZE, num);
	xvf_set_attribute(formstruct, XVF_LIST_CONTENTS, tmp);
	karray_free(tmp, num, NULL);

	num = 0; tmp = NULL;
	xvw_get_contents(&list, NULL);
	while (list != NULL)
	{
	   name = ktoken_to_string((int) list->identifier);
	   tmp  = karray_add(tmp, name, num++);
	   list = klist_next(list);
	}
	formstruct = program_info->category_struct;
	xvf_set_attribute(formstruct, XVF_LIST_SIZE, num);
	xvf_set_attribute(formstruct, XVF_LIST_CONTENTS, tmp);
	kfree(tmp);

	xvf_remove_extra_call(gui_info->create_struct,
                initialize_create_program, (kaddr) program_info);
}

/*-----------------------------------------------------------
| 
|  Routine Name: program_create
| 
|       Purpose: Do routine which is called when
|                  pane action button create is used
| 
|         Input: program_info - ptr to PaneInfo struct for program pane
| 
|        Output: None
|    Written By: Mark Young
|          Date: Feb 03, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void program_create(
     create_program *program_info)
{
	kobject toolbox, object, fileobj = NULL;
	int     existed = FALSE, otype = KCMS_PANE, generate_code = FALSE;
	char    *tbname, *oname, *iconname, *opath, path[KLENGTH],
		*category, *subcategory;


	/*
	 *  Get all the pertinent information from the pane...
	 */
	tbname   = program_info->tbname;
	oname    = program_info->oname;
	iconname = program_info->icon_name;
	category = program_info->category;
	subcategory = program_info->subcategory;

	if ((tbname == NULL) || (oname == NULL) || (iconname == NULL) ||
            (category == NULL) || (subcategory == NULL))
	{
	    kerror(NULL, "program_create", "You must provide strings for toolbox name, object name, icon name, category, and subcategory. Please make sure that all these strings are provided and try again.");
	    return;
	}

	/*
	 *  Open the desired toolbox...
	 */
	kannounce(NULL, "program_create",
		"Creating program '%s' in toolbox '%s'", oname, tbname);
	if ((toolbox = kcms_open_toolbox(tbname)) == NULL)
	{
	   kerror(NULL, "program_create", "Cannot open toolbox '%s'.", tbname);
	   return;
	}

	/*
	 *  Check to see if the object already exists...
	 */
	if ((object = kcms_open_cmobj(toolbox, oname)) == NULL)
	{
	   generate_code = TRUE;
	   object = kcms_generate_cmobj(toolbox, oname, otype, NULL, category,
			subcategory, "workspace object", NULL, KCMS_NOLANG,
			FALSE, TRUE, TRUE, NULL);
	   if (!object)
	   {
	      kerror(NULL, "program_create", "Cannot create object '%s' within \
toolbox '%s'.  Please make sure that you have permission to create an object \
this toolbox.", oname, tbname);
	      kcms_close(toolbox);
	      return;
	   }
	} else existed = TRUE;

	if (!kcms_get_attribute(object, KCMS_CMOBJ_WORKSPACE,
			&fileobj) || !fileobj)
	{
	   if (kcms_get_attribute(object, KCMS_PATH, &opath))
	   {
	      ksprintf(path, "%s/misc/%s.wksp", opath, oname);
	      fileobj = kcms_create_fileobj(object, path, NULL,
			KCMS_FOBJ_TYPE_DBM, KCMS_FOBJ_SUBTYPE_WORKSPACE,
			KCMS_FOBJ_GEN_NONE, KCMS_FOBJ_ACCESS_RDWR);
	   }
	   else
	   {
	      kerror(NULL, "program_create", "Cannot create workspace for the \
object '%s' within toolbox '%s'.  Please make sure that you have permission to \
create an object this toolbox.", oname, tbname);
	      kcms_close(object);
	      kcms_close(toolbox);
	      return;
	   }
	}

	/*
	 *  Now that the object has been created, we need to save the
	 *  additional information.  The category, subcategory has already
	 *  been saved, but if this was an existing object then re-save them.
	 */
	kcms_set_attribute(object, KCMS_CMOBJ_CATEGORY,     category);
	kcms_set_attribute(object, KCMS_CMOBJ_SUBCATEGORY,  subcategory);
	kcms_set_attribute(object, KCMS_CMOBJ_ICON_NAME,    iconname);
	kcms_set_attribute(object, KCMS_CMOBJ_CI_INSTALLED, TRUE);
	kcms_set_attribute(object, KCMS_CMOBJ_GENERATE_EXECUTABLE, TRUE);

	/*
	 *  If we have a valid workspace object, then go ahead and save the
	 *  workspace.
	 */
	if (fileobj && kcms_get_attribute(fileobj, KCMS_PATH, &opath))
	   save_workspace(opath, FALSE);

	/*
	 *  Better save the front-panel as the object's pane or clui interface.
	 */
	if (kcms_get_attribute(object, KCMS_CMOBJ_UIS_PANE, &fileobj) &&
	    kcms_get_attribute(fileobj, KCMS_PATH, &opath))
	{
	   kannounce(NULL, "program_create", "Saving front panel for program \
'%s' in toolbox '%s'", oname, tbname);
	   xvw_set_attribute(workspace, XVW_WORKSPACE_SAVE_WKSPGUI, opath);
	}

	/*
	 *  If generate code is true.  Then better generate the code for
	 *  the program object.  This will generate a script front end for
	 *  us.
	 */
	if (generate_code == TRUE)
	{
	   kannounce(NULL, "program_create", "Generating code for program \
'%s' in toolbox '%s'", oname, tbname);
	   if (kgen_generate_clui(object) == FALSE)
	   {
	      kerror(NULL, "program_create", "Failed to generate code for \
program '%s' in toolbox '%s'.  Please make sure that you have permission \
to write into this toolbox.", oname, tbname);
	      kcms_close(object);
	      kcms_close(toolbox);
	      return;
	   }
	}

	/*
	 *  Close the object and toolbox...
	 */
	kannounce(NULL, "program_create",
		"Syncing program '%s' in toolbox '%s' to disk", oname, tbname);
	kcms_close(object);
	kcms_close(toolbox);
	kannounce(NULL, "program_create", "Creation of program '%s' in \
toolbox '%s' Successful (done)", oname, tbname);

	if (!existed)
	   xvf_set_attribute(program_info->objects_struct, XVF_LIST_ADD, oname);
}

/*-----------------------------------------------------------
| 
|  Routine Name: program_toolboxes
| 
|       Purpose: Do routine which is called when
|                  displaylist selection toolboxes is used
| 
|         Input: program_info - ptr to PaneInfo struct for program pane
| 
|        Output: None
|    Written By: Mark Young
|          Date: Feb 05, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void program_toolboxes(
     create_program *program_info)
{
        kobject toolbox;
	int     i, size = 0, num, *incantata;
        char    *tbname, **programs, **tmp = NULL;


	/*
	 *  Open the desired toolbox...
	 */
        tbname = program_info->toolboxes_label;
        if ((toolbox = kcms_open_toolbox(tbname)) == NULL)
        {
           (void) kerror(NULL, "program_toolboxes", "Cannot open toolbox '%s'.",
                        tbname);
           kcms_close(toolbox);
           return;
        }

	/*
	 *  Get the different programs from the toolbox...
	 */
	kcms_get_attribute(toolbox, KCMS_TB_CATEGORYINFO, &programs, NULL,
		NULL, NULL, NULL, NULL, NULL, &incantata, NULL, &num);

	for (i = 0; i < num; i++)
	{
	   if (incantata[i])
	      tmp = karray_add(tmp, programs[i], size++);
	}
	xvf_set_attribute(program_info->objects_struct, XVF_LIST_SIZE, size);
	xvf_set_attribute(program_info->objects_struct, XVF_LIST_CONTENTS, tmp);

	xvf_set_attribute(program_info->tbname_struct, XVF_STRING_VAL, tbname);
	kfree(program_info->tbname);
	program_info->tbname = kstrdup(tbname);

        kcms_close(toolbox);
}

/*-----------------------------------------------------------
| 
|  Routine Name: program_objects
| 
|       Purpose: Do routine which is called when
|                  displaylist selection objects is used
| 
|         Input: program_info - ptr to PaneInfo struct for program pane
| 
|        Output: None
|    Written By: Mark Young
|          Date: Feb 05, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void program_objects(
     create_program *program_info)
{
	kobject toolbox = NULL, object = NULL;
	char    *tbname, *oname, *iconname = NULL, *category = NULL,
		*subcategory = NULL;


	/*
	 *  Open the desired toolbox and object...
	 */
	tbname = program_info->toolboxes_label;
	oname  = program_info->objects_label;
        if (tbname && oname)
	{
	   if ((toolbox = kcms_open_toolbox(tbname)) == NULL ||
               (object  = kcms_open_cmobj(toolbox, oname)) == NULL)
           {
	      (void) kerror(NULL, "program_objects", "Cannot open %s '%s'.",
			(!toolbox) ? "toolbox" : "object",
			(!toolbox) ? tbname : oname);
	      if (object)  kcms_close(object);
	      if (toolbox) kcms_close(toolbox);
              return;
	   }

	   /*
	    *  Get the category, subcategory, and icon name of the object and
	    *  update the appropriate part of the user interface.
	    */
	   kcms_get_attribute(object, KCMS_CMOBJ_CATEGORY,     &category);
	   kcms_get_attribute(object, KCMS_CMOBJ_SUBCATEGORY,  &subcategory);
	   kcms_get_attribute(object, KCMS_CMOBJ_ICON_NAME,    &iconname);

	   xvf_set_attribute(program_info->oname_struct,
			  XVF_STRING_VAL, oname);
	   kfree(program_info->oname);
   	   program_info->oname = kstrdup(oname);
	
	   xvf_set_attribute(program_info->icon_name_struct,
			  XVF_STRING_VAL, iconname);
	   kfree(program_info->icon_name);
	   program_info->icon_name = kstrdup(iconname);

	   xvf_set_attribute(program_info->category_struct,
			     XVF_STRING_VAL, category);
	   kfree(program_info->category);
	   program_info->category = kstrdup(category);
   
	   xvf_set_attribute(program_info->subcategory_struct,
			     XVF_STRING_VAL, subcategory);
	   kfree(program_info->subcategory);
	   program_info->subcategory = kstrdup(subcategory);

	   if (object)  kcms_close(object);
	   if (toolbox) kcms_close(toolbox);
	}
}

/*-----------------------------------------------------------
| 
|  Routine Name: program_category
| 
|       Purpose: Do routine which is called when
|                  string list selection category is used
| 
|         Input: program_info - ptr to PaneInfo struct for program pane
| 
|        Output: None
|    Written By: 
|          Date: Feb 07, 1994
| Modifications: 
| 
------------------------------------------------------------*/
/* ARGSUSED */
void program_category(
     create_program *program_info)
{
	int     num = 0;
	char    *name, **tmp = NULL;
	klist   *list = NULL;
	kform_struct *formstruct = program_info->subcategory_struct;


	xvw_get_contents(&list, NULL);
        while (list != NULL)
        {
	   name = ktoken_to_string((int) list->identifier);
	   if (kstrcmp(name, program_info->category) == 0)
	   {
	      list = (klist *) list->client_data;
	      break;
	   }
	   list = klist_next(list);
        }

	while (list != NULL)
	{
	   name = ktoken_to_string((int) list->identifier);
	   tmp = karray_add(tmp, name, num++);
	   list = klist_next(list);
	}
	xvf_set_attribute(formstruct, XVF_LIST_SIZE, num);
	xvf_set_attribute(formstruct, XVF_LIST_CONTENTS, tmp);
	kfree(tmp);
}

