 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>             Window Manager Protocol Routines
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>             protocol_handler()
   >>>>             init_protocol_handler()
   >>>>             add_protocol_entry()
   >>>>             add_protocol_handler()
   >>>>             remove_protocol_entry()
   >>>>   Public:
   >>>>             xvw_add_protocol()
   >>>>             xvw_remove_protocol()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


static void protocol_handler PROTO((Widget, XEvent *, kstring *, Cardinal *));

/*
 *  structure used by the xvw_protocol routines
 */
typedef struct _protocol
{
	Atom	   atom;
	xvobject   object;
	kfunc_void routine;
	kaddr      client_data;

	struct  _protocol *next, *prev;
} xvw_protocol;

static XtTranslations wm_translations;
static xvw_protocol *protocol_list = NULL;

static char translations[] =
   "<Message>WM_PROTOCOLS: protocol()";
/*
 \n\
    <Message>WM_CHANGE_STATE: protocol() \n\
    <Message>WM_STATE: protocol()";
 */

static XtActionsRec actions[] =
{
    { "protocol", protocol_handler }
};


/*-----------------------------------------------------------
|
|  Routine Name: protocol_handler - protocol dispatch handler
|
|       Purpose: This routine is used to call the appropriate callback
|		 routine for the desired ... WHAT?	-SK 
|
|         Input: object - the object which the action was detected for
|                event  - the event detected
|                params - the parameters to be passed back
|                num_params - the number of parameters
|
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 16:22
| Modifications: Converted from protocol_handler() in Khoros 1.0 (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void protocol_handler(
   Widget   widget,
   XEvent   *event,
   kstring   *params,
   Cardinal *num_params)
{
	Atom	     atom;
	kaddr	     data;
	kfunc_void   routine;
	xvw_protocol *entry;
	xvobject     object = xvw_object(widget);


	if (event->type == ClientMessage)
	   atom = event->xclient.data.l[0];
	else
	   atom = (Atom) -1;

	entry = protocol_list;
	while (entry != NULL)
	{
	   if (entry->object == object &&
	       (entry->atom == atom || atom == (Atom) -1))
	   {
	      break;
	   }
	   entry = entry->next;
	}

	if (entry == NULL)
	   return;

	if (entry->routine != NULL)
	{
	   routine = entry->routine;
	   data	   = entry->client_data;
	   (void) routine(object, data);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: init_protocol_handler
|
|       Purpose: This routine is used to initialize the protocol
|		 handler if it hasn't been already initialized.
|
|         Input: none
|
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 16:22
| Modifications: Converted from protocol_handler() in Khoros 1.0 (MY)
|
------------------------------------------------------------*/

static void init_protocol_handler(void)
{
	static int init = FALSE;

	if (init == FALSE)
	{
	   XtAppAddActions(xvw_appcontext(NULL), actions, knumber(actions));
	   wm_translations = XtParseTranslationTable(translations);
	   XInternAtom(xvw_display(NULL), "WM_PROTOCOLS", FALSE);
	   XInternAtom(xvw_display(NULL), "WM_DELETE_WINDOW", FALSE);
/*
	   XInternAtom(xvw_display(NULL), "WM_TAKE_FOCUS", FALSE);
	   XInternAtom(xvw_display(NULL), "WM_CHANGE_STATE", FALSE);
	   XInternAtom(xvw_display(NULL), "WM_STATE", FALSE);
 */
	   init = TRUE;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: add_protocol_entry - add a protocol entry to the list
|
|       Purpose: This routine adds the protocol entry to the
|		 protocol_list.
|
|         Input: atom    - the desired atom
|                object  - the object associated with the atom
|                routine - the routine to callback
|                client_data - the client data to be associated with the
|			       callback routine
|
|        Output: none 
|	Returns: the xvw_protocol structure on success,
|		 NULL otherwise
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 16:28
| Modifications: Converted from add_protocol_entry() in Khoros 1.0 (MY)
|
------------------------------------------------------------*/

static xvw_protocol *add_protocol_entry(
   Atom       atom,
   xvobject   object,
   kfunc_void routine,
   kaddr      client_data)
{
	xvw_protocol *entry;

	/*
	 *  create a new entry in which to record the protocol information.
	 *  This is where a object records it's atom and callback information.
	 */
	if ((entry = (xvw_protocol *) kcalloc(1, sizeof(xvw_protocol))) == NULL)
	{
	   return(NULL);
	}
	entry->next   = protocol_list;
	entry->prev   = NULL;

	if (protocol_list != NULL)
	   protocol_list->prev = entry;

	protocol_list	  = entry;
	entry->atom	  = atom;
	entry->routine	  = routine;
	entry->client_data = client_data;
	entry->object	  = object;
	return(entry);
}

/*-----------------------------------------------------------
|
|  Routine Name: add_protocol_handler - add a protocol entry to a object
|
|       Purpose: This routine adds the protocol to a object after
|		 it's been realized.  Since you can't add the protocol 
|		 handler until the object is realized, we add this 
|		 temporary event handler so that once the object has 
|		 been realized we can install it.
|
|         Input: object - the object that was realized
|                client_data - the xvw_protocol entry
|                event  - the structure notify event
|
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 16:35
| Modifications: Converted from add_protocol_handler() in Khoros 1.0 (MY)
|
------------------------------------------------------------*/
/* ARGSUSED */
static void add_protocol_handler(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int      *dispatch)
{
	xvw_protocol *entry = (xvw_protocol *) client_data;

	if (entry != NULL)
	{
	   (void) XSetWMProtocols(xvw_display(object), xvw_window(object),
			&entry->atom, 1);
	}
	xvw_remove_event(object, StructureNotifyMask, add_protocol_handler,
			(kaddr) entry);
	*dispatch = FALSE;
}


/*-----------------------------------------------------------
|
|  Routine Name: remove_protocol_entry - remove a protocol entry from the list
|
|       Purpose: This routine removes the protocol entry from the
|		 protocol_list.
|
|         Input: entry - the xvw_protocol entry to removed
|
|        Output: none
|	Returns: none
|
|    Written By: Mark Young
|          Date: Jul 15, 1992 16:30
| Modifications: Converted from remove_protocol_entry() in Khoros 1.0 (MY)
|
------------------------------------------------------------*/

static void remove_protocol_entry(
   xvw_protocol *entry)
{
	if (entry->next == NULL && entry->prev == NULL)
	{
	   protocol_list =  NULL;
	}
	else if (entry->prev == NULL)
	{
	   entry->next->prev = NULL;
	   protocol_list = entry->next;
	}
	else if (entry->next == NULL)
	{
	   entry->prev->next = NULL;
	}
	else
	{
	   entry->prev->next = entry->next;
	   entry->next->prev = entry->prev;
	}
	kfree(entry);
}



/*-----------------------------------------------------------
|
|  Routine Name: xvw_add_protocol - add a protocol handler to a object
|
|       Purpose: Allows you to call a particular routine when the user 
|                uses their window manager to do something to a toplevel
|                window of an application.
|
|                Most especially, if the user tries to use their window
|                manager to close a subform, (popup message, browser, etc) 
|                of an application, the window manager will kill the whole
|                application immediately unless we install a protocol handler 
|                to do something else.  
|
|	         For this reason, protocol handlers are used by 
|                xvw_create_application_shell() and xvw_create_transient_shell()|                to prevent the program from exiting (sometimes violently)
|                in these types of situations.
|
|         Input: object      - object to which to add protocol handler
|                protocol    - name of the protocol to add
|                routine     - routine to handle the protocol request
|                client_data - client data to be associated with
|                              the protocol handler
|        Output: none
|	Returns: none
|  Restrictions:
|    Written By: Mark Young
|          Date: Jul 20, 1992 09:48
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
void xvw_add_protocol(
   xvobject     object,
   char         *protocol,
   kfunc_void   routine,
   kaddr        client_data)
{
	Atom	     atom;
	xvw_protocol *entry;

	(void) init_protocol_handler();
	XtOverrideTranslations(xvw_widget(object), wm_translations);
	atom = XInternAtom(xvw_display(object), protocol, FALSE);
	entry = add_protocol_entry(atom, object, routine, client_data);
	if (entry == NULL)
	    return;

	/*
	 *  yuck poo.  kludge city 
	 *
	 *  If the object isn't realized then we add an event handler that
	 *  will set the XSetWMProtocols() at some later date.
	 */
	if (xvw_check_realized(object) == FALSE)
	{
	   xvw_add_event(object, StructureNotifyMask, add_protocol_handler,
			 (kaddr) entry);
	}
	else
	   (void) XSetWMProtocols(xvw_display(object), xvw_window(object),
				  &atom, 1);
}




/*------------------------------------------------------------
|
|  Routine Name: xvw_remove_protocol - remove a protocol handler from a object
|
|       Purpose: This routine removes a protocol handler from a object.  
|                See xvw_add_protocol() for more info on protocol handlers.
|
|         Input: object      - object from which to remove protocol handler
|		 protocol    - name of the protocol we are removing
|                routine     - routine that was handling the protocol request 
|                client_data - client data which was associated with 
|		  	       the protocol handler 
|        Output:
|	Returns: 
|  Restrictions: 
|    Written By: Mark Young
|          Date: Jul 20, 1992 09:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void xvw_remove_protocol(
   xvobject     object,
   char         *protocol,
   kfunc_void   routine,
   kaddr        client_data)
{
	Atom	     atom;
	xvw_protocol *entry;


	if (protocol != NULL)
	   atom = XInternAtom(xvw_display(object), protocol, FALSE);
	else
	   atom = AnyPropertyType;

	entry = protocol_list;
	while (entry != NULL)
	{
	   if (entry->client_data == client_data && entry->routine == routine &&
	       entry->object == object && entry->atom == atom)
	   {
	      break;
	   }
	   entry = entry->next;
	}

	if (entry != NULL)
	   remove_protocol_entry(entry);
}
