 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>		    Gadget/xvobject Timeout Handler Routines
   >>>>
   >>>>   These routines are used to add an timeout handler.
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>
   >>>>   Public:  
   >>>>		  xvw_add_timeout()
   >>>>		  xvw_remove_timeout()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


/*
 *  We need to keep track of all the object timeout handlers.
 */
typedef struct {
	xvobject    object;
	double	    timeout;
	kaddr       client_data;
	kfunc_void  routine;

	XtIntervalId  id;
	Boolean timer;
	Boolean destroy;
} HandlerTimeout;

static klist *timeout_list = NULL;


/*-----------------------------------------------------------
|
|  Routine Name: timeout_remove - remove a handler from the timeout list
|
|       Purpose: Race thru the timeout handler list removing all the
|		 timeout handlers that are marked for deletion, but are
|		 not in the middle of being dispatched.
|
|         Input: list - the gadget's timeout handler list
|		 override - destroy the timeout list no matter what the
|			    setting of the destroy and dispatch list.
|	Returns: returns the new timeout list
|
|    Written By: Mark Young
|          Date: May 31, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static klist *timeout_remove(
   klist   *head,
   int     override)
{
	klist *list = head;


	while (list != NULL)
	{
	   HandlerTimeout *handler = (HandlerTimeout *) klist_clientdata(list);

	   if (override || handler->destroy)
	   {
	      if (klist_checkhead(list))
		 head = klist_next(list);

	      kfree(list->client_data);
	      list = klist_delete(list, list->identifier);
	   }
	   else
	      list = klist_next(list);
	}
	return(head);
}

/*-----------------------------------------------------------
|
|  Routine Name: timeout_handler - the timeout_handler objects
|
|       Purpose: This routine is used to dispatch timeouts.
|
|         Input: client_data - the object to check the timeout for
|		 id	     - the Xt timeout id
|
| dispatch   - whether we should continuing with the
|			      dispatching of this timer  
|
|        Output:
|       Returns: none
|
|    Written By: Mark Young
|          Date: May 31, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void timeout_handler(
   XtPointer     client_data,
   XtIntervalId  *id)
{
	HandlerTimeout *handler = (HandlerTimeout *) client_data;

	xvobject object;
	unsigned long speed;
	int dispatch = TRUE;
	XtAppContext app_context;


	/*
	 *  Sanity check to see if the handler is NULL
	 */
	if (!handler)
	   return;

	/*
	 *  Call the users timeout and pass in the "dispatch" set to TRUE.
	 *  If "dispatch" is changed to FALSE then we effectively remove
	 *  the timeout.
	 */
	object = handler->object;
	(*(handler->routine))(object, handler->client_data, &dispatch);
	if (dispatch == TRUE)
	{
	   speed = handler->timeout * 1000;
	   app_context = xvw_appcontext(object);
	   handler->id = XtAppAddTimeOut(app_context, speed,
                        timeout_handler, (XtPointer) handler);
	}
	else
	{
	   handler->destroy = TRUE;
	   if (object != NULL)
	      object->timeouts = timeout_remove(object->timeouts, FALSE);
	   else
	      timeout_list = timeout_remove(timeout_list, FALSE);
	}
}


/************************************************************
*
*  Routine Name: xvw_add_timeout - add a timeout to an object
*
*       Purpose: Adds a timeout to a GUI or visual object. After the
*                specified interval of time has elapsed, the timeout
*                will be called.  Since the X Toolkit cannot support
*                timeouts on gadgets, we support them by placing the timeout 
*                on the parent and then directing the dispatch of the 
*                timeout directly.
*
*                The timeout can be associated with an object, so that
*                it will be automatically removed when the object is
*                destroyed.  If NULL is passed for the object, then the
*                timeout is added to the global file detection list.
*
*                An timeout must be declared in the following form:\f(CW
*                !void timeout(
*                !   xvobject object,
*                !   kaddr    client_data,
*                !   int      *stop_timer)\fP
*
*		 !\fIobject -\fP
*                !   If \fIxvw_add_timeout()\fP is called with a particular
*                !    xvobject, that object will be passed into the timeout.
*
*                !\fIclient_data -\fP
*                !    The pointer to the client data, used to pass parameters
*                !    from the application to the timeout.
*
*                !\fIstop_timer -\fP
*                !    By default, the timeout will be invoked again after 
*                !    the specified time interval passes once more. To stop 
*                !    the timeout from being called after the next time 
*                !    interval is up, set the stop_timer to TRUE, as in:
*                !    \f(CW*stop_timer = TRUE;\fP
*
*         Input: object      - object on which to install timeout
*                interval    - the time interval (in seconds) after
*			       which the timeout will be called.
*                routine     - the timeout routine to be installed
*                client_data - private data to be used by the timeout
*
*        Output: 
*       Returns: 
*  Restrictions:
*    Written By: Mark Young
*          Date:
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
void xvw_add_timeout(
   xvobject   object,
   double     interval,
   kfunc_void routine,
   kaddr      client_data)
{
	HandlerTimeout *handler;
	unsigned long speed = interval * 1000;
	XtAppContext app_context = xvw_appcontext(object);

	/*
	 *  Add it to our object's timer list...
	 */
	if ((handler = (HandlerTimeout *) kcalloc(1,
				sizeof(HandlerTimeout))) == NULL)
	{
	   kerror("xvwidgets", "xvw_add_timeout",
		  "Unable to allocate internal timeout handler structure.");
	   return;
	}
	handler->object      = object;
	handler->routine     = routine;
	handler->client_data = client_data;
	handler->timeout     = interval;
	if (object != NULL)
	{
	   object->timeouts = (kaddr) klist_insert(object->timeouts,
				(kaddr) speed, handler, KLIST_TAIL, TRUE);
	}
	else
	{
	   timeout_list = (kaddr) klist_insert(timeout_list,
				(kaddr) speed, handler, KLIST_TAIL, TRUE);
	}
	handler->id = XtAppAddTimeOut(app_context, speed, timeout_handler,
				      (XtPointer) handler);
}

/************************************************************
*
*  Routine Name: xvw_remove_timeout - removes a timeout from an object
*
*       Purpose: Remove a timeout from a object or gadget.
*
*         Input: object      - object from which to remove installed timeout
*                routine     - the timeout to remove
*                client_data - private data being be used by the timeout
*
*	 Output: 
*       Returns:
*  Restrictions:
*    Written By: Mark Young
*          Date: Dec 09, 1992 10:58
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
/* ARGSUSED */
void xvw_remove_timeout(
   xvobject   object,
   kfunc_void routine,
   kaddr      client_data)
{
        klist  *list;
 
 
        /*
         *  Now that we deleted the real event handler we need to cleanup
         *  our handler event list.
         */
	if (object != NULL)
           list = object->timeouts;
	else
	   list = timeout_list;

        while (list != NULL)
        {
           HandlerTimeout *handler = (HandlerTimeout *) klist_clientdata(list);
 
           if (routine == handler->routine &&
	       client_data == handler->client_data)
           {
              handler->destroy = TRUE;
	      XtRemoveTimeOut(handler->id);
           }
           list = klist_next(list);
        }

	if (object != NULL)
           object->timeouts = timeout_remove(object->timeouts, FALSE);
	else
           timeout_list = timeout_remove(timeout_list, FALSE);
}

/************************************************************
*
*  Routine Name: xvw_destroy_timeouts - destroy all timeouts currently
*				        installed on an object
*
*       Purpose: Destroys all timeouts currently installed on a 
*		 GUI or visual object. 
*
*		 This function is automatically called when an object
*		 is destroyed, but is publicly provided since it is
*		 sometimes necessary to explicitly remove all timeouts 
*		 from an object.
*
*         Input: object - object for which to destroy all timeouts
*
*    Written By: Mark Young
*          Date: May 19, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
void xvw_destroy_timeouts(
   xvobject   object)
{
	klist  *list;
	HandlerTimeout *handler;


	/*
	 *  Now that we deleted the real event handler we need to cleanup
	 *  our handler event list.
	 */
	if (object != NULL)
	   list = object->timeouts;
	else
	   list = timeout_list;

	while (list != NULL)
	{
	   handler = (HandlerTimeout *) klist_clientdata(list);
	   XtRemoveTimeOut(handler->id);
	   list = klist_next(list);
	}

	if (object != NULL)
	   object->timeouts = timeout_remove(object->timeouts, TRUE);
	else
	   timeout_list = timeout_remove(timeout_list, TRUE);
}
