 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>	              Khoros TextDisplay Widget
   >>>>  Private:
   >>>>   Static:
   >>>> 		ClassInitialize()
   >>>> 		Initialize()
   >>>> 		Destroy()
   >>>> 		SetValues()
   >>>> 		SetFilename()
   >>>> 		GetFilename()
   >>>> 		ChangeTextDisplay()
   >>>> 		TextDisplayHandler()
   >>>>
   >>>> 		FindFont()
   >>>> 		AddText()
   >>>> 		ClearText()
   >>>> 		ReadText()
   >>>>   Public:
   >>>>			xvw_create_textdisplay()
   >>>>	
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include <xvobjects/TextDisplayP.h>


static void     ClassInitialize PROTO((void));
static void     Initialize	PROTO((Widget, Widget, ArgList, Cardinal *));
static void     Destroy		PROTO((Widget));
static Boolean	SetValues       PROTO((Widget, Widget, Widget, ArgList,
				       Cardinal *));

static int  SetFilename		PROTO((xvobject, char *, kaddr));
static int  GetFilename		PROTO((xvobject, char *, kaddr));
static int  ChangeTextDisplay	PROTO((xvobject, char *, kaddr));
static void TextDisplayHandler	PROTO((xvobject, kaddr, XEvent *, int *));

static void FindFont  PROTO((XvwTextDisplayWidget, GC *,XFontStruct **,char *));
static void AddText   PROTO((XvwTextDisplayWidget, char *));
static void ClearText PROTO((XvwTextDisplayWidget));
static void ReadText  PROTO((XvwTextDisplayWidget));

#define ASCII     1
#define ROMAN     2
#define BOLD      3
#define ITALIC    4
#define SYMBOL    5
#define HELVETICA 6

#define MANPAGE_ASCII  "6x13"
#define MANPAGE_ROMAN  "-adobe-new century schoolbook-medium-r-normal--12-120-*"
#define MANPAGE_BOLD   "-adobe-new century schoolbook-bold-r-normal--12-120-*"
#define MANPAGE_ITALIC "-adobe-new century schoolbook-bold-i-normal--12-120-*"
#define MANPAGE_SYMBOL "-adobe-symbol-medium-r-normal--*-120-*"
#define MANPAGE_HELVETICA "-adobe-helvetica-medium-r-normal--12-120-*"

/*-------------------------------------------------------------------*
|
|   Full class attributes
|
--------------------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_TEXTDISPLAY_ROFF,    NULL,  XtRInt,	 XtRBoolean},
{XVW_TEXTDISPLAY_INDENT,  NULL,  XtRInt,	 XtRDimension},
{XVW_TEXTDISPLAY_FILE,	  NULL,  XtRPointer,     NULL},
{XVW_TEXTDISPLAY_ASCII,   NULL,  XtRFontStruct,  NULL},
{XVW_TEXTDISPLAY_ROMAN,   NULL,  XtRFontStruct,  NULL},
{XVW_TEXTDISPLAY_BOLD,    NULL,  XtRFontStruct,  NULL},
{XVW_TEXTDISPLAY_ITALIC,  NULL,  XtRFontStruct,  NULL},
{XVW_TEXTDISPLAY_SYMBOL,  NULL,  XtRFontStruct,  NULL},
{XVW_TEXTDISPLAY_CALLBACK,NULL,  XtRCallback,    NULL},
{XVW_TEXTDISPLAY_WORDPIXEL,NULL, XtRPixel,       NULL},
{XVW_TEXTDISPLAY_WORDCOLOR, XVW_TEXTDISPLAY_WORDPIXEL, XtRString, XtRPixel},
{XVW_TEXTDISPLAY_HIGHLIGHTPIXEL,NULL, XtRPixel,       NULL},
{XVW_TEXTDISPLAY_HIGHLIGHTCOLOR, XVW_TEXTDISPLAY_HIGHLIGHTPIXEL,
				XtRString, XtRPixel},

{XVW_FONT,      XVW_TEXTDISPLAY_ROMAN,  XtRFontStruct,  NULL},
{XVW_FONTNAME,  XVW_TEXTDISPLAY_ROMAN,  XtRString,      XtRFontStruct},
{XVW_TEXTDISPLAY_ASCII_FONTNAME, XVW_TEXTDISPLAY_ASCII,
	XtRString, XtRFontStruct},
{XVW_TEXTDISPLAY_ROMAN_FONTNAME, XVW_TEXTDISPLAY_ROMAN,
	XtRString, XtRFontStruct},
{XVW_TEXTDISPLAY_BOLD_FONTNAME,   XVW_TEXTDISPLAY_BOLD,
	XtRString, XtRFontStruct},
{XVW_TEXTDISPLAY_ITALIC_FONTNAME, XVW_TEXTDISPLAY_ITALIC,
	XtRString, XtRFontStruct},
{XVW_TEXTDISPLAY_HELVETICA_FONTNAME, XVW_TEXTDISPLAY_HELVETICA,
	XtRString, XtRFontStruct},
{XVW_TEXTDISPLAY_SYMBOL_FONTNAME, XVW_TEXTDISPLAY_SYMBOL,
	XtRString, XtRFontStruct},
};


/*-------------------------------------------------------------------*
|
|   Full class record constant
|
--------------------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwTextDisplayWidgetRec, textdisplay.field)

static XtResource resources[] = { 
{XVW_TEXTDISPLAY_BOLD, XVW_TEXTDISPLAY_BOLD_FONTNAME, XtRFontStruct, 
        sizeof(XFontStruct *), offset(bold), XtRString, MANPAGE_BOLD},
{XVW_TEXTDISPLAY_ASCII, XVW_TEXTDISPLAY_ASCII_FONTNAME, XtRFontStruct, 
        sizeof(XFontStruct *), offset(ascii), XtRString, MANPAGE_ASCII},
{XVW_TEXTDISPLAY_ROMAN, XVW_TEXTDISPLAY_ROMAN_FONTNAME, XtRFontStruct, 
        sizeof(XFontStruct *), offset(roman), XtRString, MANPAGE_ROMAN},
{XVW_TEXTDISPLAY_ITALIC, XVW_TEXTDISPLAY_ITALIC_FONTNAME, XtRFontStruct, 
        sizeof(XFontStruct *), offset(italic), XtRString, MANPAGE_ITALIC},
{XVW_TEXTDISPLAY_HELVETICA, XVW_TEXTDISPLAY_HELVETICA_FONTNAME, XtRFontStruct, 
        sizeof(XFontStruct *), offset(helvetica), XtRString, MANPAGE_HELVETICA},
{XVW_TEXTDISPLAY_SYMBOL, XVW_TEXTDISPLAY_SYMBOL_FONTNAME, XtRFontStruct,
        sizeof(XFontStruct *), offset(symbol), XtRString, MANPAGE_SYMBOL},
{XVW_TEXTDISPLAY_ROFF, NULL, XtRBoolean, 
        sizeof(Boolean), offset(roff), XtRImmediate, (XtPointer) FALSE},
{XVW_TEXTDISPLAY_INDENT, NULL, XtRDimension, 
        sizeof(Dimension), offset(indent), XtRImmediate, (XtPointer) 10},
{XVW_TEXTDISPLAY_FILE, NULL, XtRPointer,   
        sizeof(XtPointer), offset(file), XtRImmediate, (XtPointer) NULL},
{XVW_TEXTDISPLAY_WORDPIXEL, NULL, XtRPixel, 
        sizeof(Pixel), offset(word_pixel), XtRString, (XtPointer) "blue"},
{XVW_TEXTDISPLAY_HIGHLIGHTPIXEL, NULL, XtRPixel, sizeof(Pixel),
	offset(highlight_pixel), XtRString, (XtPointer) "red"},
{XVW_TEXTDISPLAY_CALLBACK, NULL, XtRCallback, sizeof(XtPointer),
      offset(textdisplay_callbacks), XtRCallback, (XtPointer) NULL},

{XVW_VP_FORCE_HORIZ, NULL, XtRBoolean, sizeof(Boolean),
	XtOffsetOf(XvwViewportWidgetRec, viewport.force_horizontal),
        XtRImmediate, (XtPointer) FALSE},
{XVW_VP_FORCE_VERT, NULL, XtRBoolean, sizeof(Boolean),
	XtOffsetOf(XvwViewportWidgetRec, viewport.force_vertical),
        XtRImmediate, (XtPointer) FALSE},
{XVW_VP_ALLOW_VERT, NULL, XtRBoolean, sizeof(Boolean),
	XtOffsetOf(XvwViewportWidgetRec, viewport.allow_vertical),
        XtRImmediate, (XtPointer) TRUE},
{XVW_VP_ALLOW_HORIZ, NULL, XtRBoolean, sizeof(Boolean),
	XtOffsetOf(XvwViewportWidgetRec, viewport.allow_horizontal),
        XtRImmediate, (XtPointer) TRUE},
};
#undef offset


/*-------------------------------------------------------------------*
|
|   Class Declaration for TextDisplay Widget
|
--------------------------------------------------------------------*/

#define superclass (&xvwViewportWidgetClassRec)

XvwTextDisplayWidgetClassRec xvwTextDisplayWidgetClassRec =
{
  {
    (WidgetClass) superclass,		/* superclass		  */	
    "TextDisplay",			/* class_name		  */
    sizeof(XvwTextDisplayWidgetRec),	/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    XtInheritRealize,			/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    TRUE,				/* compress_motion	  */
    XtExposeCompressMaximal,		/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    NULL,				/* resize		  */
    XtInheritExpose,			/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    XtInheritQueryGeometry,		/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* CoreClass fields initialization */
  {
    NULL,                    		/* geometry_manager	  */
    XtInheritChangeManaged,		/* change_managed	  */
    XtInheritInsertChild,		/* insert_child	  	  */
    XtInheritDeleteChild,		/* delete_child	  	  */
    NULL,				/* extension	 	  */
  },  /* CompositeClass fields initialization */
  {
    NULL,				    /* subresources	  */
    0,					    /* subresources_count */
    sizeof(XvwTextDisplayWidgetConstraintsRec),  /* constraint_size	  */
    NULL,				    /* initialize	  */
    NULL,				    /* destroy		  */
    NULL,				    /* set_values	  */
    NULL,				    /* extension	  */
  },  /* ConstraintClass fields initialization */
  {
    XtInheritLayout,                        /* child layout routine  */
    XtInheritChangeSel,			    /* change selected proc   */
    XtInheritEraseSel,			    /* erase selected proc    */
    XtInheritRefreshSel,		    /* refresh selection proc */
    XtInheritResize,			    /* resize		      */
    XtInheritGeometryManager,		    /* geometry_manager	      */
  },  /* XvwManagerWidgetClass fields initialization */
  {
    NULL,  		                    /* extension          */
  },  /* XvwTextDisplayWidgetClass fields initialization */
  {
    NULL,  		                    /* extension          */
  },  /* XvwTextDisplayWidgetClass fields initialization */
};
#undef superclass

  /* for public consumption */
WidgetClass xvwTextDisplayWidgetClass = (WidgetClass) &xvwTextDisplayWidgetClassRec;


/*-------------------------------------------------------------------*
|
|   Miscelleanous defines for Class and Constraint Declarations
|
--------------------------------------------------------------------*/

#undef kwidget
#undef kwidgetclass
#undef kconstraint

#define kwidget(widget)	     (XvwTextDisplayWidget) (widget)
#define kwidgetclass(widget) (XvwTextDisplayWidgetClass) (widget)->core.widget_class
#define kconstraint(widget)  (XvwTextDisplayWidgetConstraints) (widget)->core.constraints


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an
|                instance of a XvwTextDisplayWidget class has been created.
|                This is where the attributes for the class are
|		 initialized. 
|
|         Input:
|
|        Output:
|       Returns: 
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ClassInitialize(void)
{
	xvw_init_attributes(xvwTextDisplayWidgetClass, attributes,
		XtNumber(attributes), NULL, 0, NULL);
	xvw_load_resources("$DESIGN/objects/library/xvobjects/app-defaults/TextDisplay");

	xvw_define_attributes(xvwTextDisplayWidgetClass,
		XVW_TEXTDISPLAY_ADDWORD,    XtRString, ChangeTextDisplay, NULL,
		XVW_TEXTDISPLAY_DELETEWORD, XtRString, ChangeTextDisplay, NULL,
		XVW_TEXTDISPLAY_CLEARWORD,  XtRInt,    ChangeTextDisplay, NULL,
		XVW_TEXTDISPLAY_ADDTEXT,    XtRString, ChangeTextDisplay, NULL,
		XVW_TEXTDISPLAY_CLEARTEXT,  XtRInt,    ChangeTextDisplay, NULL,
		XVW_TEXTDISPLAY_FILENAME, XtRString, SetFilename, GetFilename,
		NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This method will set up the initial textdisplay
|                for the textdisplay widget instance.
|
|         Input: request - not used
|                new     - widget instance after initialization, with
|                          initial image
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwTextDisplayWidget xwid = kwidget(new);
	xvobject object = xvw_object(new);

	XtGCMask mask;
	XGCValues values;


	mask = GCForeground | GCBackground | GCFont;
	values.background = xwid->textdisplay.word_pixel;
	values.foreground = xwid->manager.foreground_pixel;
  
	values.font = xwid->textdisplay.ascii->fid;
	xwid->textdisplay.ascii_gc = XtGetGC(new, mask, &values);

	values.font = xwid->textdisplay.roman->fid;
	xwid->textdisplay.roman_gc = XtGetGC(new, mask, &values);

	values.font = xwid->textdisplay.italic->fid;
	xwid->textdisplay.italic_gc = XtGetGC(new, mask, &values);

	values.font = xwid->textdisplay.bold->fid;
	xwid->textdisplay.bold_gc = XtGetGC(new, mask, &values);

	values.font = xwid->textdisplay.helvetica->fid;
	xwid->textdisplay.helvetica_gc = XtGetGC(new, mask, &values);

	values.font = xwid->textdisplay.symbol->fid;
	xwid->textdisplay.symbol_gc = XtGetGC(new, mask, &values);

	xwid->textdisplay.text  = NULL;
	xwid->textdisplay.words = NULL;
	xwid->textdisplay.numwords = 0;
	xwid->textdisplay.numlines = 0;
	xwid->textdisplay.maxwidth = 0;

	/*
	 *  Currently highlighted word...
	 */
	xwid->textdisplay.hword  = NULL;
	xwid->textdisplay.hword_index = -1;

	xwid->textdisplay.width  = XTextWidth(xwid->textdisplay.roman, " ", 1);
	xwid->textdisplay.height =
		xwid->textdisplay.roman->max_bounds.ascent +
		xwid->textdisplay.roman->max_bounds.descent;

	/*
	 *  Initialize the notifywindow to NULL
	 */
	xwid->textdisplay.notifywindow = NULL;

	/*
	 *  Add event to handle repaint the text to the window.
	 */
	xvw_add_event(xwid->viewport.plane, ExposureMask |
		ButtonPressMask | ButtonReleaseMask,
		TextDisplayHandler, (kaddr) new);

	/*
	 *  Change the cursor on the text window to be the xterm
	 */
	xvw_set_attribute(xwid->viewport.plane, XVW_CURSORNAME, "xterm");
}

/*-----------------------------------------------------------
|
|  Routine Name: Destroy
|
|       Purpose: Close the bold, italic, symbol, helvetica, and roman font.
|
|         Input: widget - the textdisplay widget being destroyed
|
|        Output:
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Destroy(
   Widget widget)
{
        XvwTextDisplayWidget xwid = kwidget(widget);
 
 
	/*
	 *  If the textdisplay file is open, then close it...
	 */
	if (xwid->textdisplay.file != NULL)
	   kfclose(xwid->textdisplay.file);

        /*
         *  Release GCs
         */
	XtReleaseGC(widget, xwid->textdisplay.ascii_gc);
	XtReleaseGC(widget, xwid->textdisplay.italic_gc);
	XtReleaseGC(widget, xwid->textdisplay.bold_gc);
	XtReleaseGC(widget, xwid->textdisplay.symbol_gc);
	XtReleaseGC(widget, xwid->textdisplay.roman_gc);
	XtReleaseGC(widget, xwid->textdisplay.helvetica_gc);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: This method is used to set the public values
|                of a XvwTextDisplayWidget instance.  The public values
|                which can be changed are all related to the display
|		 of the textdisplay.
|
|         Input: current - the widget containing current settings
|                request - the widget containing requested settings
|                new     - the widget processed through all set values methods
|
|        Output:
|       Returns:
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues (
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num)
{
	XvwTextDisplayWidget cxwid = kwidget(current);
	XvwTextDisplayWidget nxwid = kwidget(new);

	XtGCMask  mask;
	XGCValues values;


        mask = GCForeground | GCBackground | GCFont;
        values.background = nxwid->core.background_pixel;
        values.foreground = nxwid->manager.foreground_pixel;

	if (cxwid->textdisplay.roman != nxwid->textdisplay.roman)
	{ 
           values.font = nxwid->textdisplay.roman->fid;
	   XtReleaseGC(current, nxwid->textdisplay.roman_gc);
	   nxwid->textdisplay.roman_gc = XtGetGC(new, mask, &values);
	   nxwid->textdisplay.width = XTextWidth(nxwid->textdisplay.roman,
		" ", 1);
	   nxwid->textdisplay.height =
		nxwid->textdisplay.roman->max_bounds.ascent +
		nxwid->textdisplay.roman->max_bounds.descent;
	}

	if (cxwid->textdisplay.ascii != nxwid->textdisplay.ascii)
	{ 
           values.font = nxwid->textdisplay.ascii->fid;
	   XtReleaseGC(current, cxwid->textdisplay.ascii_gc);
           nxwid->textdisplay.ascii_gc = XtGetGC(new, mask, &values);
	}

	if (cxwid->textdisplay.bold != nxwid->textdisplay.bold)
	{ 
           values.font = nxwid->textdisplay.bold->fid;
	   XtReleaseGC(current, cxwid->textdisplay.bold_gc);
           nxwid->textdisplay.bold_gc = XtGetGC(new, mask, &values);
	}

	if (cxwid->textdisplay.italic != nxwid->textdisplay.italic)
	{ 
           values.font = nxwid->textdisplay.italic->fid;
	   XtReleaseGC(current, cxwid->textdisplay.italic_gc);
           nxwid->textdisplay.italic_gc = XtGetGC(new, mask, &values);
	}

	if (cxwid->textdisplay.helvetica != nxwid->textdisplay.helvetica)
	{ 
           values.font = nxwid->textdisplay.helvetica->fid;
	   XtReleaseGC(current, cxwid->textdisplay.helvetica_gc);
           nxwid->textdisplay.helvetica_gc = XtGetGC(new, mask, &values);
	}

	if (cxwid->textdisplay.symbol != nxwid->textdisplay.symbol)
	{ 
           values.font = nxwid->textdisplay.symbol->fid;
	   XtReleaseGC(current, cxwid->textdisplay.symbol_gc);
           nxwid->textdisplay.symbol_gc = XtGetGC(new, mask, &values);
	}

	if (cxwid->textdisplay.file != nxwid->textdisplay.file)
	{
	   if (cxwid->textdisplay.file != NULL)
	      kfclose(cxwid->textdisplay.file);

	   ClearText(nxwid);
	   nxwid->textdisplay.file = kfdup(nxwid->textdisplay.file);
	   ReadText(nxwid);
	   kfclose(nxwid->textdisplay.file);
	   nxwid->textdisplay.file = NULL;
	}

	if (cxwid->textdisplay.indent != nxwid->textdisplay.indent)
	{
	   xvw_set_attribute(nxwid->viewport.plane, XVW_PREFERRED_WIDTH,
		nxwid->textdisplay.maxwidth + nxwid->textdisplay.indent);
	}
	return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetFilename - sets the filename to be displayed
|
|       Purpose: Open and display the textdisplay file.
|
|         Input: xvobject - the textdisplay object
|                attribute - the attribute XVW_TEXTDISPLAY_FILENAME
|                calldata  - the name of the filename to be displayed
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int SetFilename(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	char **string = (char **) calldata;
	kfile *file;

	if ((file = kfopen(*string, "r")) == NULL)
	{
	   kerror(XVOBJECTS, "SetFilename", "can't open input file '%s'",
		  *string);
	   return(FALSE);
	}
	return(xvw_set_attribute(object, XVW_TEXTDISPLAY_FILE, file));
}

/*-----------------------------------------------------------
|
|  Routine Name: GetFilename - gets the filename currently being displayed
|
|       Purpose: Get the name of the file being displayed.
|
|         Input: xvobject - the textdisplay object
|                attribute - the attribute XVW_TEXTDISPLAY_FILENAME
|        Output: calldata  - the name of the filename being displayed
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int GetFilename(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	char **string = (char **) calldata;
	XvwTextDisplayWidget xwid = (XvwTextDisplayWidget) xvw_widget(object);


	if ((*string = kfile_filename(kfileno(xwid->textdisplay.file))) == NULL)
	   return(FALSE);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ChangeTextDisplay - changes the current list of words
|
|       Purpose: This routine is used to manipulate the list of words
|		 that we perform the textdisplay callback
|
|         Input: xvobject - the textdisplay object
|		 attribute - the attribute which is one of:
|				XVW_TEXTDISPLAY_ADDWORD
|				XVW_TEXTDISPLAY_DELETEWORD
|				XVW_TEXTDISPLAY_CLEARWORD
|				XVW_TEXTDISPLAY_ADDTEXT
|				XVW_TEXTDISPLAY_CLEARTEXT
|		 calldata - the word to be added, deleted, or to clear the list
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int ChangeTextDisplay(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
        XvwTextDisplayWidget xwid = (XvwTextDisplayWidget) xvw_widget(object);
 
	char *text = *((char **) calldata);
	int  *words = xwid->textdisplay.words;
	int  token, num = xwid->textdisplay.numwords;


	if (kstrcmp(attribute, XVW_TEXTDISPLAY_ADDWORD) == 0)
	{
	   token = kstring_to_token(text);
	   if (karray_locate((char **) words, (kaddr) token, num) == -1)
	   {
	      words = (int *) karray_add((char **) words, (kaddr) token, num);
	      num++;
	   }
	}
        else if (kstrcmp(attribute, XVW_TEXTDISPLAY_DELETEWORD) == 0)
	{
	   token = kstring_to_token(text);
	   if (karray_locate((char **) words, (kaddr) token, num) != -1)
	   {
	      words = (int *) karray_delete((char **) words,(kaddr) token, num);
	      num--;
	   }
	}
        else if (kstrcmp(attribute, XVW_TEXTDISPLAY_CLEARWORD) == 0)
	{
	   kfree(words);
	   words = NULL; num = 0;
	}
	else if (kstrcmp(attribute, XVW_TEXTDISPLAY_ADDTEXT) == 0)
	{
	   unsigned int height;
	   int font_height, position;

	   AddText(xwid, text);

	   /*
	    *  Reset the to the bottom of the scrolled text
	    */
	   font_height = xwid->textdisplay.height;
	   xvw_geometry(xwid->viewport.clip, NULL, NULL, NULL, &height, NULL);
	   position = kmax(0, xwid->textdisplay.numlines * font_height -
				((int) height - font_height));
	   xvw_set_attribute(object, XVW_VP_YOFFSET, position);
	}
	else if (kstrcmp(attribute, XVW_TEXTDISPLAY_CLEARTEXT) == 0)
	{
	   ClearText(xwid);
	}
	else
	{
	   kerror(XVOBJECTS, "ChangeTextDisplay", "invalid attribute %s",
			attribute);
	   return(FALSE);
	}
	xwid->textdisplay.numwords = num;
	xwid->textdisplay.words    = words;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: TextDisplayHandler - refresh the text displayed
|
|       Purpose: updates or refresh textdisplay area.
|
|         Input: object      - viewport plane object
|                client_data - client data (the textdisplay object)
|                event       - type of event that triggered us
|
|        Output: dispatch    - whether to continue dispatching this event
|       Returns: none
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void TextDisplayHandler(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int      *dispatch)
{
        XvwTextDisplayWidget xwid = (XvwTextDisplayWidget) client_data;
	xvobject textdisplay = xvw_object((Widget) client_data);

	Widget  widget   = (Widget) xwid;
	Window  window   = xvw_window(object);
	Display *display = xvw_display(object);

	GC	gc;
	Pixel	pixel;
	char    *data;
	XFontStruct *font;
	klist   *list, **text;
	int     *words, token, num, numlines;
	int     i, x, y, h, len, xpos, ypos, indent, width, height;


	indent   = xwid->textdisplay.indent;
	height   = xwid->textdisplay.height;
	text     = xwid->textdisplay.text;
	numlines = xwid->textdisplay.numlines;
	words    = xwid->textdisplay.words;
	num	 = xwid->textdisplay.numwords;
	if (event->type == ButtonPress)
	{
	   x = event->xbutton.x;
	   y = event->xbutton.y;

	   if ((y % height) > 0)
	      y += height;

	   if ((i = (y/height - 1)) < 0 || i >= numlines)
	      return;

	   list = text[i];
	   ypos = (i+1)*height;
	   while (list != NULL)
	   {
	      xpos = indent + (int) klist_identifier(list);
	      data = (char *) klist_clientdata(list);
	      FindFont(xwid, &gc, &font, data++);

	      width = XTextWidth(font, data, kstrlen(data));
	      if (x > xpos && x < (xpos + width))
	      {
		 if ((token = ktoken_check(data)) != 0 &&
		     karray_locate((char **) words, (kaddr) token, num) != -1)
	         {
		    len = kstrlen(data);
		    pixel = xwid->textdisplay.highlight_pixel;
		    xwid->textdisplay.hword = list;
		    xwid->textdisplay.hword_index = i;

		    XSetForeground(display, gc, pixel);
	            XDrawString(display, window, gc, xpos, ypos, data, len);
		    XSetForeground(display, gc, xwid->manager.foreground_pixel);
		 }
		 break;
	      }
	      list = klist_next(list);
	   }
	}
	else if (event->type == ButtonRelease)
	{
	   x = event->xbutton.x;
	   y = event->xbutton.y;

	   if ((y % height) > 0)
	      y += height;

	   if (xwid->textdisplay.hword == NULL)
	      return;

	   list = xwid->textdisplay.hword;
	   data = (char *) klist_clientdata(list);
	   xpos = indent + (int) klist_identifier(list);
	   FindFont(xwid, &gc, &font, data++);
	   width = XTextWidth(font, data, kstrlen(data));

	   /*
	    *  Check to see if we selected the same word...
	    */
	   i = y/height - 1;
	   if (x > xpos && x < (xpos + width) &&
	       i == xwid->textdisplay.hword_index)
	   {
	      XtCallCallbacks(widget, XVW_TEXTDISPLAY_CALLBACK, &data);
	      if (xwid->textdisplay.hword == NULL)
		 return;
	   }
	   ypos = (xwid->textdisplay.hword_index+1)*height;
	   pixel = xwid->textdisplay.word_pixel;

	   XSetForeground(display, gc, pixel);
	   XDrawString(display, window, gc, xpos, ypos, data, kstrlen(data));
	   XSetForeground(display, gc, xwid->manager.foreground_pixel);

	   /*
	    *  Set the highlighted word to NULL
	    */
	   xwid->textdisplay.hword = NULL;
	   xwid->textdisplay.hword_index = -1;
	}
	else if (event->type == Expose)
	{
	   y = event->xexpose.y;
	   h = y + event->xexpose.height;

	   if ((h % height) > 0)
	      h += height;

	   for (y = event->xexpose.y; y < h; y += height)
	   {
	      if ((i = y/height) < 0)
		 continue;
	      else if (i >= numlines)
		 break;

	      x = 0; list = text[i];
	      while (list != NULL)
	      {
		 data = (char *) klist_clientdata(list);
		 FindFont(xwid, &gc, NULL, data++);
		 len  = kstrlen(data);
		 xpos = indent + (int) klist_identifier(list);
		 ypos = (i+1)*height;

		 if ((token = ktoken_check(data)) != 0 &&
		     karray_locate((char **) words, (kaddr) token, num) != -1)
		 {
		    XSetForeground(display, gc, xwid->textdisplay.word_pixel);
	            XDrawString(display, window, gc, xpos, ypos, data, len);
		    XSetForeground(display, gc, xwid->manager.foreground_pixel);
		 }
		 else
	            XDrawString(display, window, gc, xpos, ypos, data, len);

		 list = klist_next(list);
	      }
	   }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: FindFont - 
|
|       Purpose: This routine finds the font to be used...
|
|         Input: xwid - the textdisplay widget
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 01, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void FindFont(
   XvwTextDisplayWidget xwid,
   GC	  	*gc,
   XFontStruct	**font,
   char	        *data)
{
	switch (data[0])
	{
	   case ASCII:
		if (font) *font = xwid->textdisplay.ascii;
		if (gc)   *gc   = xwid->textdisplay.ascii_gc;
		break;
	   case ROMAN:
		if (font) *font = xwid->textdisplay.roman;
		if (gc)   *gc   = xwid->textdisplay.roman_gc;
		break;
	   case BOLD:
		if (font) *font = xwid->textdisplay.bold;
		if (gc)   *gc   = xwid->textdisplay.bold_gc;
		break;
	   case ITALIC:
		if (font) *font = xwid->textdisplay.italic;
		if (gc)   *gc   = xwid->textdisplay.italic_gc;
		break;
	   case SYMBOL:
		if (font) *font = xwid->textdisplay.symbol;
		if (gc)   *gc   = xwid->textdisplay.symbol_gc;
		break;
	   case HELVETICA:
		if (font) *font = xwid->textdisplay.helvetica;
		if (gc)   *gc   = xwid->textdisplay.helvetica_gc;
		break;
	   default:
		if (font) *font = xwid->textdisplay.roman;
		if (gc)   *gc   = xwid->textdisplay.roman_gc;
		break;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: AddText - 
|
|       Purpose: This routine adds to the contents given the supplied
|		 text.
|
|         Input: xwid - the text display widget
|    Written By: Mark Young
|          Date: Nov 01, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void AddText(
   XvwTextDisplayWidget xwid,
   char  *text)
{
	XFontStruct *font;
	klist **textlist, *list;
	char  *tmp, word[10*KLENGTH];
	int   y, h, size, numlines, pos, maxwidth, fontnum = ROMAN;

	Window  window   = xvw_window(xwid->viewport.plane);
	Display *display = xvw_display(xwid->viewport.plane);


	/*
	 *  Sanity check
	 */
	if (text == NULL)
	   return;

	pos  = 0;
	list = NULL;
	textlist = xwid->textdisplay.text;
	maxwidth = xwid->textdisplay.maxwidth;
        numlines = xwid->textdisplay.numlines;


        /*
         * Go through the file setting a line pointer to the character after
         * each new line.  This is done by breaking the string into seperate
         * lines.
         */
	if (xwid->textdisplay.roff == TRUE)
	{
        do
        {
	   tmp = text; size = 1; word[0] = fontnum;
	   while (isalnum(*tmp) || *tmp == '_' || *tmp == '' ||
		  isspace(*tmp) && *tmp != '\n' || ispunct(*tmp))
	   {
	      if (*tmp == '')
	      {
		 if (tmp[-1] == tmp[1])
		 {
		    tmp++;
 		    if (fontnum != BOLD)
		    {
		       fontnum = BOLD; size--;
		       break;
		    }
		    tmp++;
		 }
		 else if (tmp[-1] == '_')
		 {
		    tmp++;
 		    if (fontnum != ITALIC)
		    {
		       fontnum = ITALIC; size--;
		       break;
		    }
	            word[size-1] = *tmp++;
		 }
		 else
	            word[size++] = *tmp++;
	      }
	      else if (size > 1 && fontnum != ROMAN && tmp[1] != '')
	      {
		 fontnum = ROMAN;
		 break;
	      }
	      else
	         word[size++] = *tmp++;
	   }

	   if (size > 1)
	   {
	      word[size] = '\0';
	      list = klist_add(list, (kaddr) pos, (kaddr) kstrdup(word));
	      FindFont(xwid, NULL, &font, word);
	      pos += XTextWidth(font, &word[1], size);
	   }

	   text = tmp; size = 1; word[0] = fontnum;
	   while (*tmp && !isalnum(*tmp) && *tmp != '_' && *tmp != '\n')
	   {
	      if (*tmp == '')
	      {
		 if (tmp[-1] == tmp[1])
		 {
		    tmp++;
 		    if (fontnum != BOLD)
		    {
		       fontnum = BOLD; size--;
		       break;
		    }
		    tmp++;
		 }
		 else if (tmp[-1] == '_')
		 {
		    tmp++;
 		    if (fontnum != ITALIC)
		    {
		       fontnum = ITALIC; size--;
		       break;
		    }
		    tmp++;
		 }
		 else
	            word[size++] = *tmp++;
	      }
	      else if (size > 1 && fontnum != ROMAN && tmp[1] != '')
	      {
		 fontnum = ROMAN;
		 break;
	      }
	      else
	         word[size++] = *tmp++;
	   }

	   if (size > 1)
	   {
	      word[size] = '\0';
	      list = klist_add(list, (kaddr) pos, (kaddr) kstrdup(word));
	      FindFont(xwid, NULL, &font, word);
	      pos += XTextWidth(font, &word[1], size);
	   }

	   if (*tmp == '\0' || *tmp == '\n')
	   {
	      fontnum = ROMAN;
	      textlist = (klist **) karray_insert((char **) textlist,
				(kaddr) list, numlines++, KLIST_TAIL, TRUE);

	      if (*tmp == '\n')
		 tmp++;
	      if (pos > maxwidth)
		 maxwidth = pos;

	      pos = 0; list = NULL;
	   }
	   else if (*tmp == '\0')
	      fontnum = ROMAN;

	   text = tmp;
        } while (*text != '\0');
	}
	else
	{
	   word[0] = ASCII;
	   do
	   {
	       if ((tmp = kstrchr(text, '\n')) != NULL)
	       {
		  size = tmp - text;
		  kstrncpy(&word[1], text, size);
	          word[size+1] = '\0'; text = tmp+1;
	       }
	       else
	       {
		  size = kstrlen(text);
		  kstrcpy(&word[1], text);
	       }
	       kstring_replace(word, "\t", "        ", word);
	       list = klist_add(NULL, (kaddr) 0, (kaddr) kstrdup(word));
	       textlist = (klist **) karray_add((char **) textlist,
				(kaddr) list, numlines++);

               FindFont(xwid, NULL, &font, word);
               if ((pos = XTextWidth(font, &word[1], size)) > maxwidth)
		  maxwidth = pos;

	   } while (tmp != NULL);
	}
 
	if (window != 0)
	{
	   y = xwid->textdisplay.numlines*xwid->textdisplay.height;
	   h = (numlines - xwid->textdisplay.numlines)*xwid->textdisplay.height;
	   h = kmin(h, (int) xwid->core.height);
	   XClearArea(display, window, 0, y, 0, h, TRUE);
	}

	/*
	 * Save the text and number of lines...
	 */
	xwid->textdisplay.text     = textlist;
	xwid->textdisplay.numlines = numlines;
	xwid->textdisplay.maxwidth = maxwidth;
	xvw_set_attributes(xwid->viewport.plane,
		XVW_PREFERRED_WIDTH,  maxwidth + xwid->textdisplay.indent,
		XVW_PREFERRED_HEIGHT, (numlines+1) * xwid->textdisplay.height,
		NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: ClearText - 

|       Purpose: This routine clears the contents from the text display
|		 object.
|
|         Input: xwid - the text display widget
|    Written By: Mark Young
|          Date: Nov 13, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ClearText(
   XvwTextDisplayWidget xwid)
{
	int i;


	for (i = 0; i < xwid->textdisplay.numlines; i++)
	   klist_free(xwid->textdisplay.text[i], NULL);
 
	kfree(xwid->textdisplay.text);
        xwid->textdisplay.text = NULL;
        xwid->textdisplay.numlines = 0;
        xwid->textdisplay.maxwidth = 0;
	xwid->textdisplay.hword = NULL;
	xwid->textdisplay.hword_index = -1;
	xvw_set_attributes(xvw_object((Widget) xwid),
		XVW_VP_XOFFSET, 0,
		XVW_VP_YOFFSET, 0,
		NULL);
	xvw_set_attributes(xwid->viewport.plane,
		XVW_PREFERRED_WIDTH,  xwid->textdisplay.indent*2,
		XVW_PREFERRED_HEIGHT, xwid->textdisplay.indent*2,
		NULL);
	xvw_refresh(xwid->viewport.plane);
}

/*-----------------------------------------------------------
|
|  Routine Name: ReadText - 
|
|       Purpose: This routine reads the contents from the khoros transport
|		 file into the textdisplay.
|
|         Input: xwid - the text display widget
|    Written By: Mark Young
|          Date: Nov 01, 1993
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ReadText(
   XvwTextDisplayWidget xwid)
{
	int   fid;
	char  *text, *data, *tmp;


	fid = kfileno(xwid->textdisplay.file);
	xvw_set_attribute(xwid->viewport.vertical, XVW_SCROLL_VALUE, 0.0);

	if (xwid->textdisplay.notifywindow == NULL)
	{
	   xvobject object = xvw_object((Widget) xwid);

	   xwid->viewport.override = TRUE;
	   xwid->textdisplay.notifywindow = xvw_create_notifywindow(object,
		"notifywindow");
	   xvw_set_attributes(xwid->textdisplay.notifywindow,
		XVW_NOTIFYWINDOW_VISIBLE, FALSE,
		XVW_ABOVE,    NULL,
		XVW_BELOW,    NULL,
		XVW_LEFT_OF,  NULL,
		XVW_RIGHT_OF, NULL,
		NULL);
	   xwid->viewport.override = FALSE;
	}
	xvw_set_attributes(xwid->textdisplay.notifywindow,
		XVW_NOTIFYWINDOW_VISIBLE, TRUE,
		XVW_NOTIFYWINDOW_MESSAGE, "Formatting Page....",
		NULL);
	/*
	 * Copy the file into memory. 
	 */
	if ((text = kfile_readdata(fid, NULL, NULL)) == NULL)
	{
	   kerror(XVOBJECTS, "TextDisplay",
		  "Failure to read data from the file '%s'.",
		  kfile_filename(fid));
	}
	xvw_set_attribute(xwid->textdisplay.notifywindow,
		XVW_NOTIFYWINDOW_VISIBLE, FALSE);

	data = text;
	while (*data == '\n' && *data != '\0')
	   data++;

	if (data != text)
	   data--;

	if (*data != '\0')
	{
	   tmp = data + kstrlen(data) - 1;
	   while (*tmp == '\n' && *tmp != '\0')
	      tmp--;
	   *(tmp+1) = '\0';
	}
	AddText(xwid, data);
	kfree(text);
}


/************************************************************
*
*  Routine Name: xvw_create_textdisplay - create a textdisplay object
*
*       Purpose: The textdisplay object supports display of text files.
*
*		 For files that were output from \fIroff\fP interpreters,
*		 the textdisplay object will format the text as specified
*		 by the control characters produced by the \fIroff\fP 
*		 interpreter.  In other words, the textdisplay object cannot 
*		 format files with \fIroff\fP commands still in them;  rather, 
*		 it supports display of files that were output by a \fIroff\fP
*		 interpreter such as \fInroff\fP or \fIgroff\fP, after such
*		 a program was already run on the file with the \fIroff\fP
*		 commands in it.  
*
*		 Of course, the textdisplay object also supports display
*		 of plain ascii text files.
*
*		 The textdisplay object supports specification of five 
*		 different fonts, the roman font that is used with "normal" 
*		 text, as well as bold, italic, and helvetica fonts for emphasis
*		 and a symbol font for use with equations.
*
*	         The textdisplay object can also be used to support
*		 hypertext-type functionality.  The textdisplay object
*		 is capable of keeping a special list of words that 
*		 might appear in the text;  such "special" words will
*		 appear in a different color from the rest of the text.
*		 A callback can be installed on the textdisplay object 
*		 which will be fired when the user clicks on a word that
*		 is part of the textdisplay list of "special" words.
*		 Thus, the application can use this capability to display
*		 a new text file determined by the word selected by the user.
*
*         Input: parent - parent of the textdisplay object; NULL will cause
*                         a default toplevel to be created automatically
*                name   - name with which to reference textdisplay object
*
*        Output: 
*       Returns: The textdisplay object on success, NULL on failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 01, 1993
*      Verified: 
*  Side Effects:
* Modifications:
*
*******************************************************************/

xvobject xvw_create_textdisplay(
   xvobject parent,
   char     *name)
{
	xvobject object;


	object = xvw_create(parent, TRUE, TRUE, name, TextDisplayWidgetClass);
	return(object);
}
