 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>               Label Gadget Routines
   >>>>
   >>>>	 Private:
   >>>>	  Static:
   >>>>			ClassInitialize()
   >>>>			Initialize()
   >>>>			Realize()
   >>>>			Redisplay()
   >>>>			Destroy()
   >>>>			SetValues()
   >>>>			CompileString()
   >>>>			RecomputeSize()
   >>>>   Public:
   >>>>			xvw_create_labelstr()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"
#include <xvobjects/ManagerP.h>
#include <xvobjects/LabelP.h>

static void ClassInitialize	 PROTO((void));
static void Initialize		 PROTO((Widget, Widget, ArgList, Cardinal *));
static void Realize              PROTO((Widget));
static void Redisplay		 PROTO((Widget, XEvent *, Region));
static void Destroy		 PROTO((Widget));
static Boolean SetValues	 PROTO((Widget, Widget, Widget, ArgList,
					Cardinal *));
static void CompileString	 PROTO((Widget));
static void RecomputeSize	 PROTO((Widget));


/*--------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_LABEL,		NULL,    XtRString,    NULL}, 
{XVW_LABEL_EMPHASIZE,	NULL,    XtRInt,       XtRBoolean}, 
{XVW_LABEL_FILLED,	NULL,    XtRInt,       XtRBoolean}, 
{XVW_LABEL_JUSTIFY,	NULL,    XtRInt,       NULL}, 
};

/*--------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwLabelGadgetRec, label.field)

static XtResource resources[] = { 
{XVW_LABEL, NULL, XtRString, sizeof(String),
      offset(string), XtRImmediate, (XtPointer) NULL},
{XVW_LABEL_EMPHASIZE, NULL, XtRBoolean, sizeof(Boolean),
      offset(emphasize), XtRImmediate, (XtPointer) FALSE},
{XVW_LABEL_FILLED, NULL, XtRBoolean, sizeof(Boolean),
      offset(filled), XtRImmediate, (XtPointer) FALSE},
{XVW_LABEL_JUSTIFY, NULL, XtRInt, sizeof(int),
      offset(justify), XtRImmediate, (XtPointer) KLABEL_JUSTIFY_LEFT},
};
#undef offset


#define SUPERCLASS (&xvwManagerGadgetClassRec)

XvwLabelGadgetClassRec xvwLabelGadgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "LabelStr",				/* class_name		  */
    sizeof(XvwLabelGadgetRec),		/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    (XtProc) Realize,			/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    FALSE,				/* compress_exposure	  */
    FALSE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    XtInheritResize,			/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    NULL,				/* query_geometry	  */
    NULL,				/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* RectObjClass fields initialization */
  {
    XtInheritPicking,			/* pick object proc       */
    XtInheritEraseSel,			/* erase selected proc    */
    XtInheritRefreshSel,		/* refresh selection proc */
    XtInheritChangeGeometry,		/* change geometry proc   */
  },  /* XvwManagerGadgetClass fields initialization */
  {
    NULL,				/* extension - not used   */
  },  /* XvwLabelGadgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwLabelGadgetClass for public consumption
 */
WidgetClass xvwLabelGadgetClass = (WidgetClass) &xvwLabelGadgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of label object class has been created.
|                It will initialize all the class attributes. 
|         Input:
|        Output:
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications:
|
------------------------------------------------------------*/

static void ClassInitialize(void)
{
        xvw_init_attributes(xvwLabelGadgetClass, attributes,
		XtNumber(attributes), NULL, 0, "xvobject/Label.pane");
	xvw_load_resources("$DESIGN/objects/library/xvobject/app-defaults/Label");
}


/*-----------------------------------------------------------
|
|  Routine Name: Initialize 
|
|       Purpose: This method will set up the initial values 
|                for the fields of a label object instance.
|                The width, height, ascent, and descent are
|		 all initialized to sane values. 
|
|         Input: request - not used 
|                new     - object instance after initialization, with
|                          date initialized 
|        Output:
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwLabelGadget nobj = (XvwLabelGadget) new;

	/*
	 *  Compute the position of the object in device coordinates
	 */
	nobj->label.lines = NULL;
	nobj->label.numlines = 0;
	nobj->label.string_width   =
	nobj->label.string_height  = 0;
	nobj->label.string_ascent  = 0;
	nobj->label.string_descent = 0;
}

/*-----------------------------------------------------------
|
|  Routine Name: Realize
|
|       Purpose: This method will redisplay the label if force redisplay
|		 is set.
|
|         Input: widget - the label object to be realized
|        Output:
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Realize(
   Widget   widget)
{
	XvwLabelGadget xobj = (XvwLabelGadget) widget;


	/*
	 *  Refresh the label if force_redisplay is set.
	 */
	if (xobj->manager.force_redisplay)
	{
	   Redisplay(widget, NULL, NULL);
	   XFlush(XtDisplay(widget));
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: Redisplay
|
|       Purpose: This routine will redraw the label object in response
|                to an expose event. 
|
|         Input: widget - the label object that was exposed 
|                event  - the event that caused the redraw
|                region - the region that was exposed 
|        Output:
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	XvwLabelGadget xobj = (XvwLabelGadget) widget;

	unsigned int w, h;
	int  i, x, y, length, numlines;

	String  *lines;
	GC      gc = xobj->manager.gc;
	Display *display = XtDisplay(widget);
	Window  window   = XtWindowOfObject(widget);


	/*
	 *  Check to see if this if there is a label to draw
	 */
	if (kstrlen(xobj->label.string) == 0)
	   return;

	/*
	 *  Compute position using the label's current position
	 */
	x = xobj->rectobj.x;
	y = xobj->rectobj.y;
	w = xobj->rectobj.width;
	h = xobj->rectobj.height;

	lines = xobj->label.lines;
	numlines = xobj->label.numlines;
	if (xobj->label.filled)
	{
	   XSetForeground(display, gc, xobj->manager.background);
	   XFillRectangle(display, window, gc, x, y, w, h);
	   XSetForeground(display, gc, xobj->manager.foreground);
	}

	y += xobj->label.string_ascent;
	for (i = 0; i < numlines; i++)
	{
	   length = kstrlen(lines[i]);
	   if (xobj->label.emphasize)
	   {
	      XSetForeground(display, gc, xobj->manager.background);
	      XDrawString(display, window, gc, x+1, y+1, lines[i], length);
	      XSetForeground(display, gc, xobj->manager.foreground);
	   }
	   XDrawString(display, window, gc, x, y, lines[i], length);

	   y += xobj->label.string_ascent+xobj->label.string_descent;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: Destroy
|
|       Purpose: This method will free the label string.
|
|         Input: widget - the widget being destroyed
|
|        Output: none
|
|    Written By: Mark Young & John Salas
|          Date: Apr 17, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Destroy(
   Widget widget)
{
	XvwLabelGadget xobj = (XvwLabelGadget) widget;

	karray_free(xobj->label.lines, xobj->label.numlines, NULL);
	kfree(xobj->label.string);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: If the the justify or postion of the 
|		 label as changed, recalculate the new position. 
|
|         Input: current - the object containing current settings 
|                request - the object containing requested settings
|                new     - the object processed through all set values methods 
|        Output:
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwLabelGadget cobj = (XvwLabelGadget) current;
	XvwLabelGadget nobj = (XvwLabelGadget) new;

	Boolean clear_area, redisplay = FALSE;


	if (cobj->label.justify   != nobj->label.justify   ||
	    cobj->label.emphasize != nobj->label.emphasize ||
	    cobj->manager.font	  != nobj->manager.font)
	{
	   RecomputeSize(new);
	   redisplay = TRUE;
	}

	if (cobj->label.string != nobj->label.string)
	{
	   /*
	    *  Free the previously allocated private string
	    */
	   kfree(cobj->label.string);
	   nobj->label.string = kstring_copy(nobj->label.string, NULL);
	   CompileString(new);
	   RecomputeSize(new);
	   redisplay = TRUE;
	}

	if (redisplay && nobj->manager.force_redisplay)
	{
	   clear_area = !nobj->label.filled;
	   ManagerObjForceRedisplay(current, new, clear_area, &redisplay);
	}
	return(redisplay);
}

/*-----------------------------------------------------------
|
|  Routine Name: RecomputeSize
|
|       Purpose: Given the current position recompute the size.
|
|         Input: widget - the label object 
|        Output: None directly 
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications: 
|
------------------------------------------------------------*/
/* ARGSUSED */
static void RecomputeSize(
   Widget widget)
{
	XvwLabelGadget xobj = (XvwLabelGadget) widget;

	String *lines;
	XtWidgetGeometry temp;
	int i, numlines, len, ascent, descent, width;


	/*
	 *  Get the extent of the string
	 */
	len = kstrlen(xobj->label.string);
	if (len > 0 && xobj->manager.font != NULL)
	{
	   lines = xobj->label.lines;
	   numlines = xobj->label.numlines;
	   ascent  = xobj->manager.font->max_bounds.ascent;
	   descent = xobj->manager.font->max_bounds.descent;
	   xobj->label.string_ascent  = ascent;
	   xobj->label.string_descent = descent;
	   xobj->label.string_height  = (ascent + descent) * numlines;

	   for (i = 0, width = 0; i < numlines; i++)
	   {
	      len = XTextWidth(xobj->manager.font, lines[i], kstrlen(lines[i]));
	      if (len > width) width = len;
	   }
	   xobj->label.string_width = width;
	}
	else
	{
	   xobj->label.string_width   =
	   xobj->label.string_height  =
	   xobj->label.string_ascent  =
	   xobj->label.string_descent = 0;
	}

	/*
	 *  Request the new size....
	 */
	temp.request_mode = CWWidth | CWHeight;
	temp.width  = (Dimension) xobj->label.string_width;
	temp.height = (Dimension) xobj->label.string_height;
	(void) XtMakeGeometryRequest(widget, &temp, &temp);
}

/*-----------------------------------------------------------
|
|  Routine Name: CompileString - this routine compiles the string
|
|       Purpose: This routine actually compiles the string into a
|		 multiline detab'ed string.
|
|         Input: widget - the label object
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Apr 10, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void CompileString(
   Widget widget)
{
	XvwLabelGadget xobj = (XvwLabelGadget) widget;

	int    i, size, len = 0, numlines = 0;
	char   temp[KLENGTH], *string, *tmp, **lines = NULL;

	string = xobj->label.string;
	do
	{
	    tmp = kstrpbrk(string, "\t\r\n");
	    if (!tmp)
	       size = kstrlen(string);
	    else
	       size = tmp-string;

	    kstring_ncopy(string, size, &temp[len]); len += size;
	    if (tmp && *tmp == '\t')
	    {
	       size = 8 - (len % 8);
	       for (i = 0; i < size; i++)
		   temp[len++] = ' ';
	       temp[len] = '\0'; tmp++;
	    }
	    else if (!tmp || *tmp == '\n' || *tmp == '\r')
	    {
	       len = 0;
	       lines = karray_add(lines, kstrdup(temp), numlines++);
	       if (tmp) tmp++;
	    }
	    string = tmp;
	} while (string != NULL);

	/*
	 * Save the text and number of lines...
	 */
	karray_free(xobj->label.lines, xobj->label.numlines, NULL);
	xobj->label.lines    = lines;
	xobj->label.numlines = numlines;
}

/************************************************************
*
*  Routine Name: xvw_create_labelstr - create a labelstring object
*
*       Purpose: A labelstring object is for the display of
*		 non-editable text.  The difference between a 
*		 \fIlabelstring\fP object and a \fIlabel\fP object
*		 is that a \fIlabelstring\fP object is a \fIgadget\fP;
*		 ie, it has no window of its own, but uses the window
*		 of its parent.  A \fIlabel\fP object, on the other
*		 hand, is implemented as a \fIwidget\fP; it has a
*		 dedicated window of its own.  For most uses, a
*		 labelstring object is sufficient, and also faster
*		 to refresh than a label object.  
*
*         Input: parent - parent of the labelstring object; NULL will 
*			  cause a default toplevel to be created automatically
*                name   - name with which to reference labelstring object
*        Output:
*	Returns: The labelstring object on success, NULL on failure
*
*  Restrictions: 
*    Written By: Mark Young
*  	   Date: Apr 10, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_labelstr(
   xvobject parent,
   char     *name)
{
	xvobject  object;


	/*
	 *  Call the xvw_create() routine to do the actual creation.
	 */
	object = xvw_create(parent, FALSE, TRUE, name, LabelGadgetClass);
	if (object != NULL)
	{
	   xvw_set_attributes(object,
		XVW_MENUABLE,     TRUE,     /* menuable       */
		XVW_RESIZABLE,    FALSE,    /* resizable      */
		XVW_SELECTABLE,   TRUE,     /* selectable     */
		NULL);
	}
	return(object);
}
