 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */



/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                3D Plot Gadget Routines
   >>>>
   >>>>	 Private:
   >>>>	  Static:
   >>>>			ClassInitialize()
   >>>>			Initialize()
   >>>>			Destroy()
   >>>>			ColorCallback()
   >>>>			Redisplay()
   >>>>			SetValues()
   >>>>			FileDetect()
   >>>>			GetObjectName()
   >>>>			SetObjectName()
   >>>>			FindMinMax()
   >>>>			ReloadPlot()
   >>>>   Public:
   >>>>			xvw_create_plot3d()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/Plot3DP.h>


static void ClassInitialize PROTO((void));
static void Initialize	    PROTO((Widget, Widget, ArgList, Cardinal *));
static void Redisplay	    PROTO((Widget, XEvent *, Region));
static int  FileDetect	    PROTO((xvobject, char *, kaddr));
static int  GetObjectName   PROTO((xvobject, char *, kaddr));
static int  SetObjectName   PROTO((xvobject, char *, kaddr));
static Boolean SetValues    PROTO((Widget, Widget, Widget, ArgList,
				   Cardinal *));
static void ColorCallback   PROTO((xvobject, kaddr));
static void Destroy	    PROTO((Widget));
static void FindMinMax	    PROTO((XvwPlot3DGadget));
static void ReloadPlot	    PROTO((XvwPlot3DGadget));


/*--------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_PLOT3D_PLOTOBJ,       NULL,    XtRInt,        NULL},
{XVW_PLOT3D_DATA_MAX_X,    NULL,    XtRDouble,     NULL},
{XVW_PLOT3D_DATA_MAX_Y,    NULL,    XtRDouble,     NULL},
{XVW_PLOT3D_DATA_MAX_Z,    NULL,    XtRDouble,     NULL},
{XVW_PLOT3D_DATA_MIN_X,    NULL,    XtRDouble,     NULL},
{XVW_PLOT3D_DATA_MIN_Y,    NULL,    XtRDouble,     NULL},
{XVW_PLOT3D_DATA_MIN_Z,    NULL,    XtRDouble,     NULL},
{XVW_PLOT3D_PLOTTYPE,      NULL,    XtRInt,        NULL},
{XVW_PLOT3D_SHADETYPE,     NULL,    XtRInt,        NULL},
{XVW_PLOT3D_POINTS,        NULL,    XtRPointer,    NULL},
{XVW_PLOT3D_PLOTSIZE,      NULL,    XtRInt,        NULL},
{XVW_PLOT3D_PLOTWIDTH,     NULL,    XtRInt,        NULL},
};

/*--------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwPlot3DGadgetRec, plot3d.field)

static double data_min     = 0.0,
              data_max     = 1.0;

static XtResource resources[] = {
{XVW_PLOT3D_DATA_MAX_X,  NULL, XtRDouble, sizeof(double),
      offset(data_max_x), XtRDouble, (XtPointer) &data_max},
{XVW_PLOT3D_DATA_MAX_Y,  NULL, XtRDouble, sizeof(double),
      offset(data_max_y), XtRDouble, (XtPointer) &data_max},
{XVW_PLOT3D_DATA_MAX_Z,  NULL, XtRDouble, sizeof(double),
      offset(data_max_z), XtRDouble, (XtPointer) &data_max},
{XVW_PLOT3D_DATA_MIN_X,  NULL, XtRDouble, sizeof(double),
      offset(data_min_x), XtRDouble, (XtPointer) &data_min},
{XVW_PLOT3D_DATA_MIN_Y,  NULL, XtRDouble, sizeof(double),
      offset(data_min_y), XtRDouble, (XtPointer) &data_min},
{XVW_PLOT3D_DATA_MIN_Z,  NULL, XtRDouble, sizeof(double),
      offset(data_min_z), XtRDouble, (XtPointer) &data_min},
{XVW_PLOT3D_PLOTTYPE,  NULL, XtRInt, sizeof(int),
      offset(plot_type), XtRImmediate, (XtPointer) KPLOT3D_LINEPLOT},
{XVW_PLOT3D_SHADETYPE,  NULL, XtRInt, sizeof(int),
      offset(shade_type), XtRImmediate, (XtPointer) KPLOT3D_SHADE_IMAGERY},
{XVW_PLOT3D_POINTS, NULL, XtRPointer, sizeof(Coord *),
      offset(points), XtRPointer, (XtPointer) NULL},
{XVW_PLOT3D_PLOTSIZE, NULL, XtRInt, sizeof(int),
      offset(plot_size), XtRImmediate, (XtPointer) 0},
{XVW_PLOT3D_PLOTWIDTH, NULL, XtRInt, sizeof(int),
      offset(plot_width), XtRImmediate, (XtPointer) 0},
{XVW_PLOT3D_PLOTOBJ, NULL, XtRPointer, sizeof(kobject),
      offset(pobject), XtRPointer, (XtPointer) NULL},

{XVW_GRAPHICS_CLIPPING, NULL, XtRBoolean, sizeof(Boolean), 
      XtOffsetOf(XvwPlot3DGadgetRec, graphics.clipping), XtRImmediate,
      (XtPointer) FALSE},
{NULL, NULL, XtRInt, sizeof(int), 
      XtOffsetOf(XvwPlot3DGadgetRec, graphics.graphics_type), XtRImmediate,
      (XtPointer) KGRAPHICS_3D},
};
#undef offset


#define SUPERCLASS (&xvwColorGadgetClassRec)

XvwPlot3DGadgetClassRec xvwPlot3DGadgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Plot3D",				/* class_name		  */
    sizeof(XvwPlot3DGadgetRec),		/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    NULL,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    knumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    FALSE,				/* compress_exposure	  */
    FALSE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    NULL,				/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    NULL,				/* tm_table		  */
    NULL,				/* query_geometry	  */
    NULL,				/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* RectObjClass fields initialization */
  {
    XtInheritPicking,			/* pick object proc       */
    XtInheritEraseSel,			/* erase selected proc    */
    XtInheritRefreshSel,		/* refresh selection proc */
    XtInheritChangeGeometry,		/* change geometry proc   */
  },  /* XvwManagerGadgetClass fields initialization */
  {
    XtInheritRecomputePosition,		/* recompute position proc    */
  },  /* XvwGraphicsGadgetClass fields initialization */
  {
    ColorCallback,                      /* reload color proc    */
  },  /* XvwColorGadgetClass fields initialization */
  {
    NULL,				/* extension - not used   */
  },  /* XvwPlot3DGadgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwPlot3DGadgetClass for public consumption
 */
WidgetClass xvwPlot3DGadgetClass = (WidgetClass) &xvwPlot3DGadgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of plot3d object class has been created.
|                It will initialize all the class attributes.
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/

static void ClassInitialize(void)
{
        xvw_init_attributes(xvwPlot3DGadgetClass, attributes,
		knumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvisual/uis/Plot3D.pane");
	xvw_load_resources("$DESIGN/objects/library/xvisual/app-defaults/Plot3D");

        xvw_define_attribute(xvwPlot3DGadgetClass, XVW_PLOT3D_PLOTFILE,
		XtRString, SetObjectName, GetObjectName);
}

/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This method will set up the initial settings 
|                for a 3D plot object instance. 
|
|         Input: request - not used
|                new     - object instance after initialization, 
|                          with attributes set
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwPlot3DGadget xobj = (XvwPlot3DGadget) new;


	/*
	 *  Find the world coordinate minimum and maximum values (some of
	 *  the 3D plotting routines need them.
	 */
	ReloadPlot(xobj);
	FindMinMax(xobj);
}

/*-----------------------------------------------------------
|
|  Routine Name: Redisplay 
|
|       Purpose: This method is used to initially draw the 
|                the 3D plot object and for redrawing the object 
|                whenever part of it becomes exposed.
|
|         Input: widget - the object being exposed
|                event  - the expose event with information
|                         about what is being exposed
|                region - the region being exposed
|
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
-----------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	XvwPlot3DGadget xobj = (XvwPlot3DGadget) widget;

	register Coord *points;
	XColor   *bg, *plotcolor, *xcolors = NULL;
	register int i, id, width, size, ncolors = 0;


	/*
	 *  Check to see if we are currently visible and we have points
	 *  to be displayed
	 */
	if (!xobj->plot3d.points || !xvw_check_visible(xvw_object(widget)))
	   return;

        if (!xobj->plot3d.plot_size)
        {
           kerror(XVISUAL, "Redisplay",
                  "The plot size is 0 can not display plot");
           return;
           
        }

        if (!xobj->plot3d.plot_width)
        {
           kerror(XVISUAL, "Redisplay",
                  "The plot width is 0 can not display plot");
           return;
        }

	id	  = xobj->graphics.id;
	size	  = xobj->plot3d.plot_size;
	width	  = xobj->plot3d.plot_width;
	points	  = xobj->plot3d.points;
	bg	  = &xobj->graphics.bg;
	plotcolor = &xobj->graphics.fg;

	if (xobj->plot3d.plot_type == KPLOT3D_CONTOUR_2D	||
	    xobj->plot3d.plot_type == KPLOT3D_CONTOUR_3D	||
	    xobj->plot3d.plot_type == KPLOT3D_COLORMESH		||
	    xobj->plot3d.plot_type == KPLOT3D_CONSTANT_SHADING	||
	    xobj->plot3d.plot_type == KPLOT3D_GHOURAUD_SHADING	||
	    xobj->plot3d.plot_type == KPLOT3D_PHONG_SHADING)
	{
	   unsigned long *histogram = ColorGetHistogram(widget);

	   if ((xcolors = ColorGetXColors(widget)) == NULL)
	      return;

	   ncolors = xobj->color.info->ncolors;
	   for (i = 0; i < ncolors; i++)
	   {
	      if (histogram[i] > 0)
	         (void) ColorAllocate(widget, i);
	   }
	}

	switch (xobj->plot3d.plot_type)
	{
	   case KPLOT3D_LINEPLOT:
		X3D_draw_lineplot(id, points, width, size, plotcolor);
		break;

	   case KPLOT3D_MESH:
		X3D_draw_mesh(id, points, width, size, plotcolor, bg);
		break;

	   case KPLOT3D_IMPULSE:
		X3D_draw_impulse(id, points, size, plotcolor);
		break;

	   case KPLOT3D_SCATTER:
		X3D_draw_polymarker(id, points, size,
			xobj->graphics.marker_type, plotcolor);
		break;

	   case KPLOT3D_HORIZON:
		X3D_draw_horizon(id, points, width, size, plotcolor, bg,
			xobj->graphics.wcmin, xobj->graphics.wcmax);
		break;

	   case KPLOT3D_WIREFRAME:
		X3D_draw_wireframe(id, points, width, size, plotcolor);
		break;

	   case KPLOT3D_CONTOUR_2D:
		X3D_draw_contour(id, points, width, size, NULL,
			xcolors, xobj->color.begin, ncolors, TRUE);
		break;

	   case KPLOT3D_CONTOUR_3D:
		X3D_draw_contour(id, points, width, size, NULL,
			xcolors, xobj->color.begin, ncolors, FALSE);
		break;  

	   case KPLOT3D_CONSTANT_SHADING:
		X3D_shade_constant(id, points, width, size, NULL,
			xcolors, xobj->color.begin, ncolors,
		        xobj->graphics.wcmin, xobj->graphics.wcmax,
			xobj->plot3d.shade_type);
		break;

	   case KPLOT3D_GHOURAUD_SHADING:
		X3D_shade_ghouraud(id, points, width, size, NULL,
			xcolors, xobj->color.begin, ncolors,
			xobj->graphics.wcmin, xobj->graphics.wcmax,
			xobj->plot3d.shade_type);
		break;

	   case KPLOT3D_PHONG_SHADING:
		X3D_shade_phong(id, points, width, size, NULL,
			xcolors, xobj->color.begin, ncolors,
			xobj->graphics.wcmin, xobj->graphics.wcmax,
			xobj->plot3d.shade_type);
		break;

	   case KPLOT3D_COLORMESH:
		X3D_draw_colormesh(id, points, width, size, NULL,
			xcolors, xobj->color.begin, ncolors, bg,
			xobj->graphics.wcmin, xobj->graphics.wcmax);
		break;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: GetObjectName
|
|       Purpose: GetObjectName is used to get the filename associated
|		 with the attribute.
|
|         Input: widget    - the widget in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|
|        Output: calldata  - pointer to retrieve the object name
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int GetObjectName(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	kobject plot;
	XvwPlot3DGadget xobj = (XvwPlot3DGadget) xvw_widget(object);
	String *filename = (String *) calldata;


	if (kstrcmp(attribute, XVW_PLOT3D_PLOTFILE) == 0)
	   plot = xobj->plot3d.pobject;
	else
	{
	   kerror(XVISUAL, "GetObjectName", "invalid attribute %s",
			attribute);
	   return(FALSE);
	}
	*filename = NULL;
	return(kpds_get_attribute(plot, KPDS_NAME, filename));
}

/*-----------------------------------------------------------
|
|  Routine Name: SetObjectName
|
|       Purpose: SetObjectName is used to set the filename associated
|		 with the attribute.
|
|         Input: widget    - the widget in which we will be getting the
|			     filename
|		 attribute - the old object to be closed
|		 calldata  - the filename to be set
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int SetObjectName(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	kobject plot = NULL;
	int	locate = TRUE;
	XvwPlot3DGadget xobj =  (XvwPlot3DGadget) xvw_widget(object);
	String *filename = (String *) calldata;


	if (*filename != NULL && (plot = kdms_locate(*filename)) == NULL)
	{
	   locate = FALSE;
	   if ((plot = kpds_open_object(*filename, KOBJ_READ)) == NULL)
	   {
	      kerror(XVISUAL, "SetObjectName", "Failed to load 3D Plot '%s'",
			*filename);
	      return(TRUE);
	   }
	}

	if (kstrcmp(attribute, XVW_PLOT3D_PLOTFILE) == 0)
	   xvw_set_attribute(object, XVW_PLOT3D_PLOTOBJ, plot);
	else
	{
	   kerror(XVISUAL, "SetObjectName", "invalid attribute %s",
			attribute);
	   if (locate) kpds_close_object(plot);
	   return(FALSE);
	}
	if (locate) kpds_close_object(plot);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: This method is used to set the public values
|                of a 3D plot object instance.  These values
|                include the name, width, and height of the plot as
|                well as the plot image or object. 
|
|         Input: current - the object containing current settings
|                request - the object containing requested settings
|                new     - the object processed through all set values methods
|
|       Returns: FALSE (0) - XPlotRedraw will handle the redraw in any case  
|    Written By: Mark Young
|          Date: Aug 03, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwPlot3DGadget cobj = (XvwPlot3DGadget) current;
	XvwPlot3DGadget robj = (XvwPlot3DGadget) request;
	XvwPlot3DGadget nobj = (XvwPlot3DGadget) new;

	xvobject object;
	Boolean  redisplay = FALSE;


	if (cobj->plot3d.pobject       != nobj->plot3d.pobject ||
            cobj->plot3d.points        != nobj->plot3d.points  ||
            robj->graphics.reset_world != nobj->graphics.reset_world)
	{
	   if (cobj->plot3d.points != nobj->plot3d.points)
	   {
	      Coord *tmp;
	      int   size = (nobj->plot3d.plot_size * sizeof(Coord));

              if ((tmp = (Coord *) kmalloc(size)) == NULL)
              {
                 kerror(XVISUAL, "SetValues",
                        "Could not allocate memory for plot data");
              }
              else
              {
                 if (cobj->plot3d.plot_width == nobj->plot3d.plot_width &&
                     nobj->plot3d.plot_size != 0)
                 {
                    nobj->plot3d.plot_width = ksqrt(nobj->plot3d.plot_size);
                 }
                 kfree(cobj->plot3d.points);
                 kmemcpy(tmp, nobj->plot3d.points, size);
                 nobj->plot3d.points = tmp;
              }
	   }

	   if (cobj->plot3d.pobject != nobj->plot3d.pobject)
	   {
	      object = xvw_object(new);
	      nobj->plot3d.pobject =kpds_reference_object(nobj->plot3d.pobject);
	      kpds_close_object(cobj->plot3d.pobject);
	      xvw_set_attribute(xvw_object(new), XVW_COLOR_COLOROBJ,
                        nobj->plot3d.pobject);
	   }
	   ReloadPlot(nobj);
	   FindMinMax(nobj);
	   redisplay = TRUE;
	}

	if (cobj->plot3d.plot_type  != nobj->plot3d.plot_type ||
	    cobj->plot3d.shade_type != nobj->plot3d.shade_type)
	{
	   redisplay = TRUE;
	}
	return(redisplay);
}

/*-----------------------------------------------------------
|
|  Routine Name: Destroy
|
|       Purpose: Close the image, shape, colormap, and overlay
|                objects before the image widget is destroyed.
|
|         Input: widget - the image widget being destroyed
|
|    Written By: Mark Young
|          Date: Nov 21, 1992 12:52
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Destroy(
   Widget widget)
{
        XvwPlot3DGadget xobj = (XvwPlot3DGadget) widget;

        /*
         *  Close plot object
         */
        kpds_close_object(xobj->plot3d.pobject);
        kfree(xobj->plot3d.points);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorCallback
|
|       Purpose: Refresh the palette display since our ColorClass
|		 has had a change to its map data, and we are required
|		 to change the colors in the palette.
|
|         Input: object - the object to be refreshed (the Plot3D)
|		 call_data   - (not used)
|        Output:
|    Written By: Mark Young & Danielle Argiro
|          Date: June 28, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ColorCallback(
   xvobject object,
   kaddr    call_data)
{
	Widget widget = xvw_widget(object);
	Region region = XCreateRegion();
	XEvent event;

	event.type = Expose;
	event.xexpose.x = event.xexpose.y = 0;
	event.xexpose.width  = widget->core.width;
	event.xexpose.height = widget->core.height;
	XtAddExposureToRegion(&event, region);
	Redisplay(widget, &event, region);
	XDestroyRegion(region);
}

/*-----------------------------------------------------------
|
|  Routine Name: FindMinMax
|
|       Purpose: This routine will find the min and max values
|                from the data in a given plot 3D object
|
|         Input: xobj - the object to find the min max for 
|
|        Output: None
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Oct 24, 1992 23:59
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void FindMinMax(
   XvwPlot3DGadget xobj)
{
	register int i;
	register Coord wcmin, wcmax, *points;


	if ((points = xobj->plot3d.points) == NULL)
	   return;

	wcmin.x = wcmax.x = points[0].x;
	wcmin.y = wcmax.y = points[0].y;
	wcmin.z = wcmax.z = points[0].z;
	wcmin.d = wcmax.d = points[0].d;

	for (i = xobj->plot3d.plot_size -1; i >= 0; i--)
	{
	   if (points[i].x > wcmax.x) wcmax.x = points[i].x;
	   else if (points[i].x < wcmin.x) wcmin.x = points[i].x;

	   if (points[i].y > wcmax.y) wcmax.y = points[i].y;
	   else if (points[i].y < wcmin.y) wcmin.y = points[i].y;

	   if (points[i].z > wcmax.z) wcmax.z = points[i].z;
	   else if (points[i].z < wcmin.z) wcmin.z = points[i].z;

	   if (points[i].d > wcmax.d) wcmax.d = points[i].d;
	   else if (points[i].d < wcmin.d) wcmin.d = points[i].d;
	}
	xobj->plot3d.data_max_x = wcmax.x;
	xobj->plot3d.data_max_y = wcmax.y;
	xobj->plot3d.data_max_z = wcmax.z;
	xobj->plot3d.data_min_x = wcmin.x;
	xobj->plot3d.data_min_y = wcmin.y;
	xobj->plot3d.data_min_z = wcmin.z;

	if (xobj->graphics.reset_world)
	{
	   xvw_set_attributes(xvw_object((Widget) xobj),
			XVW_GRAPHICS_WCMAX_X, wcmax.x,
			XVW_GRAPHICS_WCMAX_Y, wcmax.y,
			XVW_GRAPHICS_WCMAX_Z, wcmax.z,
			XVW_GRAPHICS_WCMIN_X, wcmin.x,
			XVW_GRAPHICS_WCMIN_Y, wcmin.y,
			XVW_GRAPHICS_WCMIN_Z, wcmin.z,
			NULL);
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ReloadPlot
|
|       Purpose: This routine will subsample the data from
|		 a given 3D plot object.  I DON'T KNOW WHY
|		 THIS WOULD BE DONE		-SK 
|
|         Input: xobj - the object to subsample 
|
|        Output: None
|
|       Returns: None 
|
|    Written By: Mark Young
|          Date: Oct 24, 1992 23:59
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ReloadPlot(
   XvwPlot3DGadget xobj)
{
	double factor;
	kobject object;
	int i, w, h, d, t, e, type, dimension;

	register Coord  *points;
	register double *data = NULL; 
	register double val;
	register int plot_size, init;
	register unsigned char *pixels;


	/*
	 *  Sanity check to make sure that the image object exists
	 */ if ((object = xobj->plot3d.pobject) == NULL)
	   return;

	/*
	 *  Get the value data demsionality
	 */
	kpds_get_attributes(xobj->plot3d.pobject,
		KPDS_VALUE_SIZE, &w, &h, &d, &t, &e,
		KPDS_VALUE_DATA_TYPE, &type,
		NULL);


	/*
         * Special case: triples of ascii data points
         */
        if (w == 3)
        {
            /*
             * Go ahead and malloc room for the new data...
             */
            if (xobj->plot3d.plot_size != h)
            {
                xobj->plot3d.plot_size = h;
                xobj->plot3d.points = (Coord *) krealloc(xobj->plot3d.points,
                                      xobj->plot3d.plot_size*sizeof(Coord));
                if (xobj->plot3d.points == NULL)
                {
                    kerror(XVISUAL, "ReloadPlot",
                           "Unable to allocate plot coordinate data");
                    return;
                }
            }

            /*
             * get data points in triples
             */

            points = xobj->plot3d.points;
	    xobj->plot3d.plot_width = 1;
            kpds_set_attribute(xobj->plot3d.pobject, KPDS_VALUE_REGION_SIZE,
                               w, 1, 1, 1, 1);
            for (i = 0; i < h; i++)
            {
                data = (double *)kpds_get_data(object, KPDS_VALUE_REGION, data);
                points[i].x = data[0];
                points[i].y = data[1];
                points[i].z = data[2];
                points[i].d = (unsigned long) data[2];
            }
            kfree(data);
        }

	/*
         *  General case; Get the image plane
         */
	else
	{
	    /*
	     *  Make sure that the width and height are equally shrunk
	     */
	    factor = kmin(64.0/w, 64.0/h);
	    if (w > 64)
	       w *= factor;
	    if (h > 64)
	       h *= factor;

	    /*
	     *  Set the position to be the beginning of the value and location
	     *  data segments.
	     */
	    kpds_set_attributes(xobj->plot3d.pobject,
		    KPDS_VALUE_POSITION,    0, 0, 0, 0, 0,
		    KPDS_VALUE_SIZE,        w, h, d, t, e,
		    KPDS_VALUE_SCALING,     KNONE,
		    KPDS_VALUE_DATA_TYPE,   KDOUBLE,
		    KPDS_VALUE_INTERPOLATE, KZERO_ORDER,
		    NULL);
	    /*
	     *  Get the image plane
	     */
	    if ((data = (double *) kpds_get_data(object,
			    KPDS_VALUE_PLANE, NULL)) == NULL)
	    {
	       kerror(XVISUAL, "ReloadPlot", "Unable to get image plane");
	       return;
	    }

	    /*
	     *  Go ahead and malloc the new data...
	     */
	    if (xobj->plot3d.plot_size != w*h)
	    {
	       xobj->plot3d.plot_size = w*h;
	       xobj->plot3d.points = (Coord *)krealloc(xobj->plot3d.points,
				        xobj->plot3d.plot_size*sizeof(Coord));
	       if (xobj->plot3d.points == NULL)
	       {
	          kerror(XVISUAL, "ReloadPlot", 
			 "Unable to allocate plot coordinate data");
	          return;
	       }
	    }
    
	    points     = xobj->plot3d.points;
	    plot_size  = xobj->plot3d.plot_size;
	    xobj->plot3d.plot_width = w;
	    for (i = 0; i < plot_size; i++)
	    {
	       points[i].x = i % w;
	       points[i].y = i/w;
	       points[i].z = data[i];
	    }
    
	    if (kpds_query_location(object))
	    {
	        kpds_get_attribute(xobj->plot3d.pobject, KPDS_LOCATION_SIZE,
			            NULL, NULL, &d, &dimension);
	        kpds_set_attributes(xobj->plot3d.pobject,
				KPDS_LOCATION_POSITION, 0, 0, 0, 0,
				KPDS_LOCATION_SIZE, w, h, d, dimension,
				KPDS_LOCATION_SCALING,     KNONE,
				KPDS_LOCATION_DATA_TYPE,   KDOUBLE,
				KPDS_LOCATION_INTERPOLATE, KZERO_ORDER,
				NULL);
	       /*
	        *  Get the location data plane
	        */
	       if (dimension > 1 && 
		   kpds_get_data(object, KPDS_LOCATION_PLANE,data))
	       {
	          for (i = 0; i < plot_size; i++)
	             points[i].x = data[i];
	       }
    
	       /*
	        *  Get the location data plane
	        */
	       if (dimension > 1 &&
		   kpds_get_data(object, KPDS_LOCATION_PLANE,data))
	       {
	          for (i = 0; i < plot_size; i++)
	             points[i].y = data[i];
	       }
    
	       /*
	        *  Get the location data plane
	        */
	       if (kpds_get_data(object, KPDS_LOCATION_PLANE, data))
	       {
	          for (i = 0; i < plot_size; i++)
	             points[i].z = data[i];
	       }
	    }
    
	    /*
	     *  Get the Image Pixel data
	     */
            kpds_set_attributes(xobj->plot3d.pobject,
		KPDS_VALUE_DATA_TYPE, KUBYTE,
		KPDS_VALUE_POSITION, 0, 0, 0, 0, 0,
		NULL);

	    if (type == KFLOAT || type == KDOUBLE || type == KCOMPLEX ||
	        type == KDCOMPLEX)
	    {
               kpds_set_attributes(xobj->plot3d.pobject,
		    KPDS_VALUE_SCALING, KNORMALIZE,
		    KPDS_VALUE_NORM_MIN, 0.0,
		    KPDS_VALUE_NORM_MAX, 255.0,
	            NULL);
	    }
    
	    if ((pixels = (unsigned char *) kpds_get_data(object,
			    KPDS_VALUE_PLANE, data)))
	    {
	       for (i = 0; i < plot_size; i++)
	          points[i].d = pixels[i];
	    }
    
	    /*
	     *  Get the Mask data
	     */
	    if (kpds_query_mask(xobj->plot3d.pobject))
	    {
               kpds_set_attributes(xobj->plot3d.pobject,
		    KPDS_MASK_DATA_TYPE, KUBYTE,
		    KPDS_MASK_POSITION, 0, 0, 0, 0, 0,
		    KPDS_VALUE_INTERPOLATE, KZERO_ORDER,
		    NULL);
	       if ((pixels = (unsigned char *) kpds_get_data(object,
			    KPDS_MASK_PLANE, data)))
	       {
	          for (init = FALSE, i = 0; i < plot_size; i++)
	          {
	             if (!pixels[i]) continue;
    
		     if (!init || val > points[i].z)
		        val = points[i].z, init = TRUE;
	          }
    
	          for (i = 0; i < plot_size; i++)
	             points[i].z = !pixels[i] ? val : points[i].z;
	       }
	    }
	    kfree(data);
	}
}


/************************************************************
*
*  Routine Name: xvw_create_plot3d - create a 3D plot object
*
*       Purpose: A 3D plot object is used to display a 3D plot.
*                A variety of plot types are supported, including
*                line plots, mesh plots, scatter plots, 2D and 3D
*                contour plots, impulse plots, horizon plots, wireframe
*                plots, and shaded plots.
*
*                The 3D plot object is generally used in conjunction 
*                with a 3D axis object, where both are created as children
*                of an area object so that a coordinated world coordinate
*                view is automatically supported.
*
*         Input: parent - the parent object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference the object 
*
*        Output: none
*       Returns: Returns the created 3D plot xvobject, or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young and John M. Salas
*          Date: Aug 03, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_plot3d(
   xvobject parent,
   char   *name)
{
	xvobject  object;


	/*
	 *  Create the main plot workspace
	 */
	object = xvw_create(parent, FALSE, TRUE, name, Plot3DGadgetClass);
	if (object != NULL)
	{
	   xvw_set_attributes(object,
                XVW_MENUABLE,     TRUE,     /* menuable       */
                XVW_RESIZABLE,    FALSE,    /* resizable      */
                XVW_SELECTABLE,   TRUE,     /* selectable     */
		NULL);
	}
	return(object);
}
