 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>                Palette Widget Routines
   >>>>
   >>>>	Private:
   >>>>	 Static:
   >>>>			ClassInitialize()
   >>>>			Initialize()
   >>>>			Realize()
   >>>>			Redisplay()
   >>>>			SetValues()
   >>>>			UpdatePalette()
   >>>>			ColorCallback()
   >>>>  Public:
   >>>>			xvw_create_palette()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/PaletteP.h>


static void ClassInitialize PROTO((void));
static void Initialize	    PROTO((Widget, Widget, ArgList, Cardinal *));
static void Realize	    PROTO((Widget, XtValueMask *,
				   XSetWindowAttributes *));
static void Redisplay	    PROTO((Widget, XEvent *, Region));
static Boolean SetValues    PROTO((Widget, Widget, Widget, ArgList,
				   Cardinal *));

static void UpdatePalette   PROTO((xvobject, kaddr, XEvent *, int *));
static void ColorCallback   PROTO((xvobject, kaddr));

#undef kwidget
#undef kwidgetclass
#undef kconstraint

#define kwidget(widget)      (XvwPaletteWidget) (widget)
#define kwidgetclass(widget) (XvwPaletteWidgetClass) (widget)->core.widget_class
#define kconstraint(widget)  (XvwPaletteWidgetConstraints) (widget)->core.constraints

/*-------------------------------------------------------
|
|   Full class attributes
|
--------------------------------------------------------*/

static xvattribute attributes[] = {
{XVW_PALETTE_TYPE,      NULL,    XtRInt,    NULL},
{XVW_PALETTE_CALLBACK,	NULL,	 XtRCallback, NULL},
};

/*-------------------------------------------------------
|
|   Full class record constant
|
--------------------------------------------------------*/

#define offset(field) XtOffsetOf(XvwPaletteWidgetRec, palette.field)

static XtResource resources[] = { 

{XVW_PALETTE_TYPE, NULL, XtRInt, sizeof(int),
      offset(type), XtRImmediate, (XtPointer) KPALETTE_TYPE_COLORBAR},
{XVW_PALETTE_CALLBACK, NULL, XtRCallback, sizeof(XtPointer),
      offset(update_callback), XtRCallback, (XtPointer) NULL},

{XVW_WIDTH, NULL, XtRDimension, sizeof(Dimension),
        XtOffsetOf(CoreRec, core.width),
        XtRImmediate, (XtPointer) 300},
{XVW_HEIGHT, NULL, XtRDimension, sizeof(Dimension),
        XtOffsetOf(CoreRec, core.height),
        XtRImmediate, (XtPointer) 150},
};
#undef offset


#define SUPERCLASS (&xvwColorWidgetClassRec)

XvwPaletteWidgetClassRec xvwPaletteWidgetClassRec =
{
  {
    (WidgetClass) SUPERCLASS,		/* superclass		  */	
    "Palette",				/* class_name		  */
    sizeof(XvwPaletteWidgetRec),	/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    Realize,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    knumber(resources),			/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    TRUE,				/* compress_motion	  */
    XtExposeCompressMaximal,		/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    NULL,				/* destroy		  */
    NULL,				/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    XtInheritTranslations,		/* tm_table		  */
    NULL,				/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* CoreClass fields initialization */
  {
    NULL,                    		/* geometry_manager	  */
    XtInheritChangeManaged,		/* change_managed	  */
    XtInheritInsertChild,		/* insert_child	  	  */
    XtInheritDeleteChild,		/* delete_child	  	  */
    NULL,				/* extension	 	  */
  },  /* CompositeClass fields initialization */
  {
    NULL,	            		/* subresources       */
    0,					/* subresources_count */
    sizeof(XvwManagerWidgetConstraintsRec),/* constraint_size    */
    NULL,				/* initialize         */
    NULL,                               /* destroy            */
    NULL,				/* set_values         */
    NULL,                               /* extension          */
  },  /* ConstraintClass fields initialization */
  {
    XtInheritLayout,			/* child layout routine  */
    XtInheritChangeSel,		        /* change selected proc   */
    XtInheritEraseSel,			/* erase selected proc    */
    XtInheritRefreshSel,		/* refresh selection proc */
    XtInheritResize,			/* resize		  */
    XtInheritGeometryManager,  		/* geometry_manager	  */
  }, /* XvwManagerWidgetClass fields initialization */
  {
    NULL,                               /* field not used    */
  },  /* XvwGraphicsWidgetClass fields initialization */
  {
    ColorCallback,                      /* reload color proc    */
  },  /* XvwColorWidgetClass fields initialization */
  {
    NULL,                               /* field not used    */
  },  /* XvwPaletteWidgetClass fields initialization */
};
#undef SUPERCLASS

/*
 * xvwPaletteWidgetClass for public consumption
 */
WidgetClass xvwPaletteWidgetClass = (WidgetClass) &xvwPaletteWidgetClassRec;


/*-----------------------------------------------------------
|
|  Routine Name: ClassInitialize
|
|       Purpose: This method is called the first time an  
|                instance of palette widget class has been created.
|                It will initialize all the class attributes.
|
|         Input: none 
|
|        Output: none
|
|    Written By: Mark Young 
|          Date: Nov 17, 1992 19:24
| Modifications:
|
------------------------------------------------------------*/

static void ClassInitialize(void)
{
        xvw_init_attributes(xvwPaletteWidgetClass, attributes,
		knumber(attributes), NULL, 0, 
		"$DESIGN/objects/library/xvisual/uis/Palette.pane");
	xvw_load_resources("$DESIGN/objects/library/xvisual/app-defaults/Palette");
}


/*-----------------------------------------------------------
|
|  Routine Name: Initialize
|
|       Purpose: This method will set up the initial image
|		 for a palette widget instance. 
|
|         Input: request - not used
|		 new	 - widget instance after initialization, with 
|		           initial palette
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Oct 23, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Initialize(
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	xvw_add_event(xvw_object(new), ButtonPressMask, UpdatePalette, NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: Realize
|       Purpose: This method will set up the window attributes 
|                and create a window for the palette widget.
|
|         Input: widget	    - the widget to realize 
|		 valuemask  - determines the window attributes being passed in
|		 attributes - the list of window attributes to be set 
|    Written By: Mark Young
|          Date: Nov 28, 1994
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Realize(
   Widget               widget,
   XtValueMask          *valuemask,
   XSetWindowAttributes *winattrib)
{
	XvwPaletteWidget xwid = kwidget(widget);

	XtGCMask     mask;
	XGCValues    values;


	/*
	 *  Create the gc to be used with the refresh mechanism
	 */
        values.line_width = 1;
        values.cap_style  = CapButt;
        values.join_style = JoinMiter;
	mask = GCForeground | /*GCJoinStyle | GCCapStyle |*/ GCLineWidth;

        values.foreground = xwid->manager.bottom_shadow;
        xwid->palette.gc1 = XtGetGC(widget, mask, &values);
        values.foreground = xwid->manager.top_shadow;
        xwid->palette.gc2 = XtGetGC(widget, mask, &values);
	(*xvwPaletteWidgetClass->core_class.superclass->core_class.realize)
                (widget, valuemask, winattrib);
}

/*-----------------------------------------------------------
|
|  Routine Name: ColorCallback
|
|       Purpose: Refresh the palette display since our ColorClass
|		 has had a change to its map data, and we are required
|		 to change the colors in the palette.
|
|         Input: object - the object to be refreshed (the Palette)
|		 call_data   - (not used)
|        Output:
|    Written By: Mark Young & Danielle Argiro
|          Date: May 24, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void ColorCallback(
   xvobject object,
   kaddr    call_data)
{
	XvwPaletteWidget xwid = (XvwPaletteWidget) xvw_widget(object);
	XEvent event;
	Region  region   = XCreateRegion();

	event.type = Expose;
	event.xexpose.x = event.xexpose.y = 0;
	event.xexpose.width  = xwid->core.width;
	event.xexpose.height = xwid->core.height;
	XtAddExposureToRegion(&event, region);
	Redisplay((Widget) xwid, &event, region);
	XDestroyRegion(region);
}

/*-----------------------------------------------------------
|
|  Routine Name: UpdatePalette
|
|       Purpose: updates the position of the cursor in the palette object
|		 so that if there is other devices that have update callbacks
|		 we will know it.
|
|         Input: widget      - image widget
|                client_data - client data (unused)
|                event       - type of event that triggered us
|        Output: dispatch    - whether to continue dispatching this event
|       Returns: none
|
|    Written By: Mark Young
|          Date: Oct 26, 1993
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void UpdatePalette(
   xvobject object,
   kaddr    client_data,
   XEvent   *event,
   int      *dispatch)
{
	double   factor;
	unsigned long *histogram;
	XvwPaletteWidget xwid = (XvwPaletteWidget) xvw_widget(object);
	int      i, x, y, w, h, xpos, ypos, width, height, indx, num, position;


	if (event->type != ButtonPress || !xwid->palette.update_callback ||
	    !xwid->color.info || xwid->color.info->ncolors <= 0)
	{
	   return;
	}
	width  = xwid->core.width;
	height = xwid->core.height;
	xpos = event->xbutton.x; ypos = event->xbutton.y;

	/*
	 *  Just do it...  According to the different palette types, figure
	 *  out which index we are abusing...
	 */
	histogram = ColorGetHistogram((Widget) xwid);
	for (i = 0, num = 0; i < xwid->color.info->ncolors; i++)
	   if (histogram[i] > 0) num++;

	if (xwid->palette.type == KPALETTE_TYPE_COLORBAR)
	{
	   factor = (width-4)/((double) num);
	   position = kmax((xpos-2)/factor, 0);
	}
	else if (xwid->palette.type == KPALETTE_TYPE_COLORWHEEL)
	{
           double angle, degrees;

	   x = xpos - (width-4)/2; y = ypos - (height-4)/2;
           if (x == 0 && y == 0)
              degrees = 0.0;
	   else
	   {
              angle = katan2((double) y, (double) x);
              degrees = kradians_degrees(angle);
              if (degrees < 0.0)
	         degrees = kabs(degrees);
	      else
	         degrees = 360.0 - degrees;
	   }
	   factor = 360.0/num;
	   position = degrees/factor; 
	}
	else if (xwid->palette.type == KPALETTE_TYPE_COLORCELL)
	{
           /*
            *  Compute the palette width & height to hold the color boxes.
            */
	   factor = ksqrt((double) (((width-4)*(height-4))/num));
	   w = h = factor;
	   if (num > (((width-4)/w) * ((height-4)/h)))
	   {
	      if (w > h)
		 h--;
	      else
		 w--;
	   }
	   x = kmax((xpos-2)/w, 0);
	   y = kmax((ypos-2)/h, 0);
	   num = (width-4)/w;
	   position = num*y + x;
	}
	else
	   return;

	/*
	 *  Just do it...  Finde the position in which the user has clicked
	 *  on the palette.  And then call the callbacks with that indx.
	 */
	for (indx = i = 0; indx < xwid->color.info->ncolors; indx++)
	{
	   if (histogram[indx] > 0)
	   {
	      if (i == position)
	      {
		 XtCallCallbacks((Widget) xwid, XVW_PALETTE_CALLBACK, &indx);
		 return;
	      }
	      i++;
	   }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: Redisplay
|
|       Purpose: This routine will redraw the area widget in response
|                to an expose event 
|
|         Input: widget - widget data structure for palette
|		 event	- the event that caused the redraw
|		 region	- the region that was exposed
|
|        Output: None
|
|    Written By: Mark Young
|          Date: Nov 17, 1992 19:24
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static void Redisplay(
   Widget widget,
   XEvent *event,
   Region region)
{
	XvwPaletteWidget xwid = kwidget(widget);

	Pixel  pixel;
	double factor;
	unsigned long *histogram;
	int    i, l, width, height, w, h, x, y, num;

	XPoint  pts[3];
	GC      gc, gc1, gc2;
	Display *display = XtDisplay(widget);
	Window  window   = XtWindowOfObject(widget);


	/*
	 *  Refresh the color palette display
	 */
	width  = xwid->core.width;
	height = xwid->core.height;
	gc = xwid->graphics.gc;

	/*
	 *  Sanity check to make sure that we have legal values to render
	 *  the color palette.
	 */
	if (!xwid->color.info || xwid->color.info->ncolors <= 0 ||
	    !xvw_check_visible(xvw_object(widget)))
	   return;

	histogram = ColorGetHistogram(widget);
	for (i = 0, num = 0; i < xwid->color.info->ncolors; i++)
	   if (histogram[i] > 0) num++;

	if (xwid->palette.type == KPALETTE_TYPE_COLORBAR)
	{
	   factor = (width-4)/((double) num);
	}
	else if (xwid->palette.type == KPALETTE_TYPE_COLORWHEEL)
	{
	   factor = 360.0/num;
	}
	else if (xwid->palette.type == KPALETTE_TYPE_COLORCELL)
	{
           /*
            *  Compute the palette width & height to hold the color boxes.
            */
	   factor = ksqrt((double) (((width-4)*(height-4))/num));
	   w = h = factor;
	   if (num > (((width-4)/w) * ((height-4)/h)))
	   {
	      if (w > h)
		 h--;
	      else
		 w--;
	   }
	   w -= 4; h -= 4;
	}
	else
	   return;

	/*
	 *  Just do it...  Render the range by gapping the slots between
	 *  valid and invalid (non-allocated) pixels.  Eventually we want
	 *  this to be some kind of logrithmic or radial pattern, but for
	 *  now just do it simple and clean.
	 */
	x = y = 4;
	for (i = l = 0; i < xwid->color.info->ncolors; i++)
	{
	   if (histogram[i] > 0)
	   {
	      l++;

	      pixel = ColorAllocate(widget, i);
	      XSetForeground(display, gc, pixel);
	      if (xwid->palette.type == KPALETTE_TYPE_COLORBAR)
	      {
		 w = (factor * l) - x;
	         XFillRectangle(display, window, gc, x, 2, kmax(w,1), height-4);
	         x += w;
	      }
	      else if (xwid->palette.type == KPALETTE_TYPE_COLORCELL)
	      {
	         if (x + w >= width)
		 {
	            x = 4;
		    y += h + 4;
		 }

		 gc1 = xwid->palette.gc1;
		 gc2 = xwid->palette.gc2;
		 pts[0].x = x - 1;     pts[0].y = y + h;
		 pts[1].x = x - 1;     pts[1].y = y - 1;
		 pts[2].x = x + w;     pts[2].y = y - 1;
		 XDrawLines(display, window, gc1, pts, 3, CoordModeOrigin);
		 pts[1].x = x + w;     pts[1].y = y + h;
		 XDrawLines(display, window, gc2, pts, 3, CoordModeOrigin);

	         XFillRectangle(display, window, gc, x, y, w, h);
	         x += w + 4;
	      }
	      else if (xwid->palette.type == KPALETTE_TYPE_COLORWHEEL)
	      {
		 w = (64 * factor * l) - x;
	         XFillArc(display, window, gc, 2, 2, width-4, height-4, x, w);
	         x += w;
	      }
	   }
	}
	(*xvwPaletteWidgetClass->core_class.superclass->core_class.expose)
                (widget, event, region);
}


/*-----------------------------------------------------------
|
|  Routine Name: SetValues
|
|       Purpose: Determine what the widget should do by
|		 comparing the current values verse the
|		 new settings.
|
|         Input: current - the widget containing current settings 
|		 request - the widget containing requested settings
|		 new	 - the widget processed through all set values methods 
|
|        Output: None directly
|
|       Returns: TRUE (1) if redisplay is required, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Nov 17, 1992 19:24
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static Boolean SetValues(
   Widget   current,
   Widget   request,
   Widget   new,
   ArgList  args,
   Cardinal *num_args)
{
	XvwPaletteWidget cxwid = kwidget(current);
	XvwPaletteWidget nxwid = kwidget(new);

	Boolean redisplay = FALSE;


	if (cxwid->palette.type   != nxwid->palette.type)
	   redisplay = TRUE;

	return(redisplay);
}


/************************************************************
*
*  Routine Name: xvw_create_palette - create a palette object
*
*       Purpose: The palette visual object provides a visual display 
*                of the colors defined by a colormap. The colors may be 
*                displayed as a grid of color cells, a color palette, or 
*                a color wheel.
*
*         Input: parent - the parent object; NULL will cause a
*                         default toplevel to be created automatically
*                name   - the name with which to reference the object 
*
*        Output: None
*
*	Returns: The palette xvobject on success, NULL otherwise
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Nov 17, 1992 19:24
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvobject xvw_create_palette(
   xvobject parent,
   char     *name)
{
	xvobject object;


	/*
	 *  Call the xvw_create() routine to do the actual creation.
	 */
	object = xvw_create(parent, FALSE, TRUE, name, PaletteWidgetClass);
	return(object);
}
