 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Color Normalization and Mapping Utilities
   >>>>
   >>>>   Static:
   >>>>			CreateVariableList()
   >>>>			CreateFunctionDefinition()
   >>>>			CreateFunctionCall()
   >>>>			ComputeMeanStandardDeviation()
   >>>>  Private:
   >>>>			ApplyMapcolFunction()
   >>>>			ComputeNormalization()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <xvisual/ColorP.h>


static void ComputeMeanStandardDeviation PROTO((Widget, double *, double *,
						double *, unsigned long *));

/*-----------------------------------------------------------
|
|  Routine Name: CreateVariableList
|
|       Purpose: Scans a function string (to be passed to kexpr parser)
|                for variables, where all variables MUST be of the
|                form "MX" where X is from 0 to M, M being the number
|                of rows in the map data.  For example, if you passed
|                in "M1 + M2 / M5", variable_scan() would return a list
|                of strings consisting of "M1", "M2", and "M5".
|                of the list of strings passed in.
|
|         Input: string       - the function string, eg, "M1 + M2 / M5"
|		 map_width    - the number of columns in the map
|        Output: variable_num - number of variables found
|                               in the function string.
|       Returns: Array of strings containing variables
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications: 
|
------------------------------------------------------------*/
char **CreateVariableList(
    char *string,
    int  map_width,
    int  *variable_num)
{
        char **list, temp[KLENGTH];
        int  i, j, k, l, length, colcheck, repeat;
 
        *variable_num = 0;
        list = (char **) kcalloc(1, map_width * sizeof(char *));
 
        length = kstrlen(string);
        if (length < 2) return(NULL);
        k = 0;
        i = 0;
        while (i < length)
        {
             while ((string[i] != 'M') && (i < length)) i++;
 
             if (i == length) break;
             if (i < length)
             {
                 /* get a variable specified by MX */
                 j = 0;
                 temp[j++] = string[i++];
                 while (isdigit(string[i]))
                 temp[j++] = string[i++];
                 temp[j] = '\0';
             }
 
            /* make sure MX does not specify an invalid map column */
            colcheck = atoi(&temp[1]);
            if (colcheck > map_width-1)
            {
                kerror(XVISUAL, "CreateVariableList", "Legal variables include \
M0 to M%d only.", map_width-1);
                return(NULL);
            }
 
            /* look for repeat of a certain MX */
            repeat = FALSE;
            for (l = 0; l < k; l++)
	    {
               if (kstrcmp(list[l], temp) == 0) repeat = TRUE;
	    }
            /* add variable to list if it is not a repeat */
            if (!repeat) list[k++] = kstrdup(temp);
        }
        *variable_num = k;
        return(list);
}

/*-----------------------------------------------------------
|
|  Routine Name: CreateFunctionDefinition
|
|       Purpose: Creates a function definition to be passed to the kexpr
|                parser, based on the string that the user types in as
|                the function defining the red band, green band, or blue band.
|                For example,  "Red(M1, M2, M5) = M1 + M2 / M5".
|
|         Input: string      - the function string, eg, "M1 + M2 / M5".
|                init_string - initial part of function def, eg, "Red("
|                var_list    - list of variables, obtained with
|                              spc_create_strings_variable_list().
|                var_num     - size of var_list
|
|        Output: None
|       Returns: The string containing function definition
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications: 
|
------------------------------------------------------------*/

char *CreateFunctionDefinition(
    char *string,
    char *init_string,
    char **var_list,
    int  var_num)
{
        int  i;
        char temp[10*KLENGTH];
        char temp2[10*KLENGTH];

        ksprintf(temp, init_string);
        for (i = 0; i < var_num; i++)
        {
            kstrcat(temp, var_list[i]);
            if (i < var_num-1)  kstrcat(temp, ",");
        }
        kstrcat(temp, ")");
        ksprintf(temp2, "= %s", string);
        kstrcat(temp, temp2);

        return(kstrdup(temp));
}

/*-----------------------------------------------------------
|
|  Routine Name: CreateFunctionCall
|
|       Purpose: Creates a function call to be passed to the kexpr
|                parser, based on the number of variables (MX's) that
|                are used in the current function, the pointers to the
|                appropriate columns, and the current row.
|                For example,  "Red(25,0,200)"
|
|         Input: indx      - the current cluster (row) across all columns
|                columns   - array of columns that will have values in function
|                var_num   - number of variables (columns) in the function
|                map_width   - map width
|                init_string - the initial part of the function
|                maps	   - the original map data
|
|        Output: None
|       Returns: The string containing function call in correct format
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications: 
|
------------------------------------------------------------*/

char *CreateFunctionCall(
    int  indx,
    int  *columns,
    int  var_num,
    int  map_width,
    char *init_string,
    double *maps)
{
        int  i;
        char func[10*KLENGTH];
        char temp[10*KLENGTH];

        ksprintf(func, init_string);
        for (i = 0; i < var_num; i++)
        {
	     ksprintf(temp,"%g", maps[indx*map_width + columns[i]]);
             kstrcat(func,  temp);
             if (i < var_num-1)  kstrcat(func, ",");
        }
        kstrcat(func, ")");
        return(kstrdup(func));
}

/*-----------------------------------------------------------
|
|  Routine Name: ApplyMapcolFunction
|
|       Purpose: apply mapping function for red,green,blue maps.
|         Input: widget  - the color widget
|		 column  - which map column that we are working on
|        Output: 
|       Returns: 
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Jun 22, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ApplyMapcolFunction(
   Widget widget,
   int    column)
{
	XvwColorPart *color = GetColorPart(widget);
	double *maps = color->info->maps;
	int    map_width  = color->info->map_width;
	int    map_height = color->info->map_height;

	float  fvalue;
	double value, *data;
	int i, parsed, var_num, indx = -1, *cols;
	char **var_list, *function, *function_def;
        char init_string[KLENGTH], temp[KLENGTH], funcstr[KLENGTH];


	if (column == RED_COLOR)
	{
	   data = color->info->red;
	   function = color->red_function;
	   ksprintf(init_string, "Red(");
	}
	else if (column == GREEN_COLOR)
	{
	   data = color->info->green;
	   function = color->green_function;
	   ksprintf(init_string, "Green(");
	}
	else if (column == BLUE_COLOR)
	{
	   data = color->info->blue;
	   function = color->blue_function;
	   ksprintf(init_string, "Blue(");
	}

	kstring_cleanup(function, funcstr);
	temp[0] = '\0';
	if (ksscanf(funcstr, "M%d%s", &indx, temp) <= 0 &&
	    ksscanf(funcstr, "%lg%s", &value, temp) <= 0)
	{
	   parsed = FALSE;
	}
	else if (funcstr == NULL || kstrlen(temp) == 0)
	   parsed = TRUE;
	else
	   parsed = FALSE;

	/* 
	 * the function string is not really a function -- 
         * just a single map column 
	 */
	if ((parsed == TRUE) || (maps == NULL))
	{
	   if (maps == NULL)
	   {
	      for (i = 0; i < map_height; i++)
	            data[i] = i;
	      return;
	   }
	   else if (function == NULL || indx != -1 && indx < map_width)
	   {
	      if (function != NULL) column = indx;
	      for (i = 0; i < map_height; i++)
	            data[i] = maps[i*map_width + column];

	      return;
	   }
	   else if (indx == -1)
	   {
	      for (i = 0; i < map_height; i++)
	            data[i] = value;
	      return;
	   }
	}

	/* parse variables MX out of function string */
        if ((var_list = CreateVariableList(function, map_width,
					   &var_num)) == NULL)
	{
	   return;
	}
	function_def = CreateFunctionDefinition(function, init_string,
						var_list, var_num);

	/* send function definition to kexpr parser */
        if (!(kexpr_evaluate_float((int) widget, function_def, NULL, temp)))
        {
            kerror(XVISUAL, "ApplyMapcolFunction", temp);
            kfree(function_def);
            karray_free(var_list, var_num, NULL);
            return;
        }

        cols = (int *) kmalloc(var_num * sizeof(int));
        for(i = 0; i < var_num; i++)
        {
            ksprintf(temp, var_list[i]);
            cols[i] = atoi(&temp[1]);
        }

        if (column == RED_COLOR)
            ksprintf(init_string, "Red(");
        else if (column == GREEN_COLOR)
            ksprintf(init_string, "Green(");
        else if (column == BLUE_COLOR)
            ksprintf(init_string, "Blue(");

        for (i = 0; i < map_height; i++)
        {
             function = CreateFunctionCall(i, cols, var_num, map_width,
				init_string, maps);
             kexpr_evaluate_float((int) widget, function, &fvalue, temp);
	     data[i] = fvalue;
	     kfree(function);
        }
        kfree(cols);
        kfree(function_def);
}

/*-----------------------------------------------------------
|
|  Routine Name: ComputeMeanStandardDeviation
|
|       Purpose: Computes the Mean and Standard Deviation of a map column.
|		 Given the histogram we compute the mean and standard deviation
|		 of the map column, which we use to compute the min & max
|		 values used for the normalization of the map column.
|         Input: widget - color widget
|		 column - the map column of data
|        Output: vmin    - returns the minimum normalization value (-stddev)
|        	 vmax    - returns the maximum normalization value (+stddev)
|       Returns: 
|
|    Written By: Danielle Argiro & Mark Young
|          Date: Jun 23, 1994
| Modifications:
|
------------------------------------------------------------*/

static void ComputeMeanStandardDeviation(
   Widget        widget,
   double        *column,
   double        *vmin,
   double        *vmax,
   unsigned long *histogram)
{
	XvwColorPart *color = GetColorPart(widget);

        unsigned long count;
        double   sum, sum2, mean, stddev;
        int      i, map_height = color->info->map_height;

 
        for (i = 0, count = 0, sum = sum2 = 0.0; i < map_height; i++)
        {
           count += histogram[i];
           sum   += column[i] * histogram[i];
           sum2  += ksqr(column[i]) * histogram[i];
        }

	mean   = sum/count;
	stddev = sqrt((sum2 - 2.0*mean*sum + ksqr(mean)*count)/(count-1.0));
	if (color->norm_method == KCOLOR_NORM_1STDDEV)
	{
	   *vmin = mean - stddev;
	   *vmax = mean + stddev;
	}
	else if (color->norm_method == KCOLOR_NORM_2STDDEV)
	{
	   *vmin = mean - 2*stddev;
	   *vmax = mean + 2*stddev;
	}
	else if (color->norm_method == KCOLOR_NORM_3STDDEV)
	{
	   *vmin = mean - 3*stddev;
	   *vmax = mean + 3*stddev;
	}
	else
	{
	   *vmin = 0.0;
	   *vmax = 255.0;
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: ComputeNormalization
|
|       Purpose: Computes the scale and offset used in normalizing colormaps.
|		 Given the color widget's normalization policy, type, and
|		 method a scale and offset is used in which to normalize the
|		 map data to be between 0 and 65535 (max intensity for an
|		 X Display).
|         Input: widget - the color widget
|        Output: rscale  - returns red normalization scale value
|		 roffset - returns red normalization offset value
|        	 gscale  - returns green normalization scale value
|		 goffset - returns green normalization offset value
|        	 bscale  - returns blue normalization scale value
|		 boffset - returns blue normalization offset value
|       Returns: 
|
|    Written By: Mark Young & Danielle Argiro
|          Date: Jun 22, 1994
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
void ComputeNormalization(
   Widget widget,
   double *rscale,
   double *roffset,
   double *gscale,
   double *goffset,
   double *bscale,
   double *boffset)
{
	XvwColorPart  *color = GetColorPart(widget);
	unsigned long *histogram = ColorGetHistogram(widget);
	int    map_height = color->info->map_height;
	double *red   = color->info->red;
	double *green = color->info->green;
	double *blue  = color->info->blue;

	int    i;
	double rmin, rmax, gmin, gmax, bmin, bmax, tmin, tmax;


	if ((color->norm_method == KCOLOR_NORM_MAXCOLORS) ||
	   (histogram == NULL))
	{
	   if ((histogram == NULL) && 
	       (color->norm_method != KCOLOR_NORM_MAXCOLORS))
	   {
		errno = 0;
                if (color->info->ncolors > MaxMapLength)
		{
	            kerror("design", "ComputeMeanStandardDeviation",
	  	           "Range of pixel values exceed the maximum number of %d; unable to perform standard deviation normalization.  Doing a 0 - maxcolors normalization instead.", MaxMapLength);
		}
	   	else 
		{
		    kerror("design", "ComputeMeanStandardDeviation",
                            "Internal error: failed to get histogram.  Unable to do standard deviation normalization.  Doing a 0 - maxcolors normalization instead.");
		    kinfo(KHOSTILE, "CONGRADULATIONS!  YOU HAVE FOUND A BUG. GO BUY YOURSELF A BEER.");

		}
	   }
	   rmin = rmax = red[0]; gmin = gmax = green[0]; bmin = bmax = blue[0];
	   for (i = 1; i < map_height; i++)
	   {
	      if (red[i] < rmin)
	         rmin = red[i];
	      else if (red[i] > rmax)
	         rmax = red[i];

	      if (green[i] < gmin)
	         gmin = green[i];
	      else if (green[i] > gmax)
	         gmax = green[i];

	      if (blue[i] < bmin)
	         bmin = blue[i];
	      else if (blue[i] > bmax)
	         bmax = blue[i];
	   }
	}
	else
	{
	   ComputeMeanStandardDeviation(widget, red,   &rmin, &rmax, histogram);
	   ComputeMeanStandardDeviation(widget, green, &gmin, &gmax, histogram);
	   ComputeMeanStandardDeviation(widget, blue,  &bmin, &bmax, histogram);
	}
	tmin = kmin3(rmin, gmin, bmin);
	tmax = kmax3(rmax, gmax, bmax);

#if 0
	kfprintf(kstderr,"rmin,rmax: %g %g\n", rmin, rmax);
	kfprintf(kstderr,"gmin,gmax: %g %g\n", gmin, gmax);
	kfprintf(kstderr,"bmin,bmax: %g %g\n", bmin, bmax);
	kfprintf(kstderr,"tmin,tmax: %g %g\n", tmin, tmax);
#endif
	if (color->norm_type == KCOLOR_NORM_GLOBAL)
	{
           *roffset = *goffset = *boffset = -tmin;
           *rscale = *gscale = *bscale = (tmin==tmax) ? 1.0 : 65535/(tmax-tmin);
	}
	else if (color->norm_type == KCOLOR_NORM_LOCAL)
	{
           *roffset = -rmin; *goffset = -gmin; *boffset = -bmin;
           *rscale  = (rmax == rmin) ? 1.0 : 65535/(rmax-rmin);
           *gscale  = (gmax == gmin) ? 1.0 : 65535/(gmax-gmin);
           *bscale  = (bmax == bmin) ? 1.0 : 65535/(bmax-bmin);
	}
}
