/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>   		2D Drawing  Utilities
   >>>>
   >>>>  Private:
   >>>>			X2D_draw_polyline()
   >>>>			X2D_draw_discrete()
   >>>>			X2D_draw_bargraph()
   >>>>			X2D_draw_histogram()
   >>>>			X2D_draw_polymarker()
   >>>>			X2D_draw_linemarker()
   >>>>			X2D_draw_polygon()
   >>>>			X2D_draw_segments()
   >>>>			X2D_draw_line()
   >>>>			X2D_draw_marker()
   >>>>			X2D_draw_rectangle()
   >>>>			X2D_draw_circle()
   >>>>			X2D_draw_colorline()
   >>>>			X2D_fill_rectangle()
   >>>>			X2D_fill_circle()
   >>>>   Static:
   >>>>			draw_colorline()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "graphics.h"


/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_polyline
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coords	- array of world coordinate points
|		 size	- number of points
|		 fg	- color of polyline
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_polyline (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	Coord		*ncoords;
	XSegment	*segments;
	int		nsize, seg_num;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_polyline:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((ncoords = _X2D_clip_lines(graphics, coords, size, &nsize)) == NULL)
	   return;

	seg_num = nsize/2;
	if ((segments = (XSegment *) kmalloc ((unsigned) seg_num * 
			sizeof (XSegment))) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_polyline:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (!(_X2D_convert_wc_to_dc_seg(graphics, ncoords, segments, nsize)))
	{
	   kfree(segments);
	   return;
	}

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].segments(graphics, segments, seg_num);

	if (coords != ncoords)
	   kfree(ncoords);
	kfree(segments);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_discrete
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coords	- array of world coordinate points
|		 size	- number of points
|		 fg	- color of discrete plot
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_discrete (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	XPoint		*points;

	int		baseline;
	Real		xmin, ymin;
	int		i, nsize, clip_code;
	Coord		zero_coord, *ncoords;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_discrete:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if (graphics->wc_min.y < 0  && graphics->wc_max.y > 0)
	{
	   zero_coord.x = 0.0;
	   zero_coord.y = 0.0;
	   if (!(X2D_convert_point_wc_to_dc(id, zero_coord, &xmin, &ymin)))
	   {
	      return;
	   }
	}
	else
	{
	   if (!(X2D_convert_point_wc_to_dc(id, graphics->wc_min, &xmin,
		&ymin)))
	   {
	      return;
	   }
	}
	baseline = (ymin + 0.5);

	clip_code = X2D_TOP | X2D_BOTTOM;
	if ((ncoords = _X2D_clip_points(graphics, coords, size, clip_code,
	   &nsize)) == NULL)
	{
	   return;
	}

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof(XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X2D_draw_discrete:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	{
	   kfree(points);
	   return;
	}

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	for (i = 0; i < nsize; i++)
	{
	   X3D_draw[graphics->device].line(graphics, points[i].x,
			baseline, points[i].x, points[i].y);
	}
	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_bargraph - Draws a two dimensional bargraph
|
|       Purpose: 
|
|         Input: id	- X3D graphics structure ID
|		 coords	- array of world coordinate points
|		 size	- number of points
|		 fg	- color of bargraph
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_bargraph (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		*points;
	Coord		*ncoords;

	int		i, x, y;
	int		bottom, left, right;
	Real		xmin, ymin, xmax, ymax;
	unsigned int	width, height;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_bargraph:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if(!(X2D_convert_point_wc_to_dc (id, graphics->wc_min, &xmin, &ymin)))
	   return;

	if(!(X2D_convert_point_wc_to_dc (id, graphics->wc_max, &xmax, &ymax)))
	   return;

	left   = xmin + 0.5;
	right  = xmax + 0.5;
	bottom = ymin + 0.5;

	if ((ncoords = _X2D_clip_rectangles(graphics, coords, size, &nsize))
		== NULL)
	   return;

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof(XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X2D_draw_bargraph:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if(!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	{
	   kfree(points);
	   return;
	}

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);

	/*
	 *  Initialize the width and the first x position.
	 */
	if (nsize == 1)
	   width = kabs(right-left)/2;
	else width = kabs(points[0].x - points[1].x);

	x = points[0].x - (width/2.0 + 0.5);
	for (i = 0; i < nsize; i++)
	{
	   if (i+1 != nsize)
	      width = kabs(points[i].x - points[i+1].x);

	   y = points[i].y;
	   height = kabs(bottom - y);
	   X3D_draw[graphics->device].rectangle(graphics, x, y, width, height);
	   x += width;
	}

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_histogram
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coords	- array of world coordinate points
|		 size	- number of points
|		 wcmin	- world coordinate minimum (x,y,z)
|		 wcmax	- world coordinate maximum (x,y,z)
|		 num	- NOT USED
|		 fg	- color of histogram
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_histogram (
   int    id,
   Coord  *coords,
   int    size,
   Coord  wcmin,
   Coord  wcmax,
   int    num,
   XColor *fg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		*points;
	Coord		*ncoords;

	int		i, x, y;
	int		baseline;
	Real		xmin, ymin;
	unsigned int	width, height;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_bargraph:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if(!(X2D_convert_point_wc_to_dc (id, graphics->wc_min, &xmin, &ymin)))
	   return;

	baseline = ymin;

	if ((ncoords = _X2D_clip_rectangles(graphics, coords, size, &nsize))
		== NULL)
	   return;

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X2D_draw_histogram:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if(!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	{
	   kfree(points);
	   return;
	}

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);

	for (i = 1; i <= nsize; i++)
	{
	    width = kabs((points[i].x - points[i+1].x));
	    x = points[i].x;
	    y = points[i].y;
	    height = kabs(baseline - y);
	    X3D_draw[graphics->device].rectangle(graphics, x, y, width, height);
	}

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_polymarker
|
|       Purpose:
|
|         Input: id	     - X3D graphics structure ID
|		 coords	     - array of world coordinate points
|		 size	     - number of points
|		 marker_type - type of marker to display
|		 fg	     - color of polymarker
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_polymarker (
   int    id,
   Coord  *coords,
   int    size,
   int    marker_type,
   XColor *fg)
{
	X3DGraphics	*graphics;
	XPoint		*points;
	Coord		*ncoords;
	int		nsize, clip_code;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_polymarker:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	clip_code = X2D_CENTER;
	if ((ncoords = _X2D_clip_points(graphics, coords, size, clip_code,
		&nsize)) == NULL)
	{
	   return;
	}

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X2D_draw_polymarker:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	{
	   kfree(points);
	   return;
	}
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	_X2D_draw_markers(id, points, nsize, marker_type);

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}


/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_colormarker
|
|       Purpose:
|
|         Input: id          - X3D graphics structure ID
|                coords      - array of world coordinate points
|                size        - number of points
|                marker_type - type of marker to display
|		 colors	     - color array
|		 num   	     - size of color array
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date:
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_colormarker (
   int    id,
   Coord  *coords,
   int    size,
   int    marker_type,
   XColor *colors,
   int    num,
   XColor *plotcolor)
{
	XColor		*fg;
	X3DGraphics	*graphics;
	XPoint		*points;
	Coord		*ncoords;
	int		i, nsize, clip_code;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_colormark:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

        clip_code = X2D_CENTER;
        if ((ncoords = _X2D_clip_points(graphics, coords, size, clip_code,
                &nsize)) == NULL)
        {
           return;
        }

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X2D_draw_marker:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

        if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
        {
           kfree(points);
           return;
        }

	for (i = 0; i < size; i++)
	{
	   if ( ncoords[i].d < num )
	   {
	     fg = &colors[ncoords[i].d];
	   }
	   else
	   {
	     fg = &colors[i];
	   }
	   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	   _X2D_draw_markers(id, &points[i], 1, marker_type);
	}

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}


/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_linemarker
|
|       Purpose:
|
|         Input: id	     - X3D graphics structure ID
|		 coords	     - array of world coordinate points
|		 size	     - number of points
|		 marker_type - type of marker to display
|		 fg	     - color of linemarker
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_linemarker (
   int    id,
   Coord  *coords,
   int    size,
   int    marker_type,
   XColor *fg)
{
	if (_X3D_get_graphics(id) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_linemarker:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
        }
	X2D_draw_polymarker (id, coords, size, marker_type, fg);
	X2D_draw_polyline (id, coords, size, fg);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_polygon
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coords	- array of world coordinate points
|		 size	- number of points
|		 fg	- color of polygon
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_polygon (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		*points;
	Coord		*ncoords;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_polygon:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((ncoords = _X2D_clip_polygon(graphics, coords, size, &nsize))
		== NULL)
	   return;

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X2D_draw_marker:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	{
	   kfree(points);
	   return;
	}

	points[nsize] = points[0];

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].fill_polygon(graphics, points, nsize,
			Convex, CoordModeOrigin);
	X3D_draw[graphics->device].lines(graphics,points, ++nsize,
			CoordModeOrigin);

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_segments
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coords	- array of world coordinate points
|		 size	- number of points
|		 fg	- color of segment
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_segments (
   int    id,
   Coord  *coords,
   int    size,
   XColor *fg)
{
	X3DGraphics	*graphics;
	Coord		*ncoords;
	XSegment	*segments;
	int		seg_num, nsize;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_segments:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((ncoords = _X2D_clip_segments(graphics, coords, size, &nsize))
		== NULL)
	   return;

	seg_num = nsize/2;
	if ((segments = (XSegment *) kmalloc ((unsigned) seg_num * 
			sizeof (XSegment))) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_segments:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (!(_X2D_convert_wc_to_dc_seg(graphics, ncoords, segments, nsize)))
	{
	   kfree(segments);
	   return;
	}

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].segments(graphics, segments, seg_num);

	if (coords != ncoords)
	   kfree(ncoords);
        kfree(segments);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_line
|
|       Purpose: 
|
|         Input: id     - X3D graphics structure ID
|		 coord1 - the beginning world coordinate point
|		 coord2 - the end world coordinate point
|		 fg	- color of line
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_line (
   int    id,
   Coord  coord1,
   Coord  coord2,
   XColor *fg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		points[2];
	Coord		coords[2], *ncoords;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_line:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	coords[0] = coord1;
	coords[1] = coord2;
	if ((ncoords = _X2D_clip_lines(graphics, coords, 2, &nsize)) == NULL)
	   return;

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	   return;

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].line(graphics, points[0].x,
		points[0].y, points[1].x, points[1].y);

	if (coords != ncoords)
	   kfree(ncoords);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_marker
|
|       Purpose:
|
|         Input: id	     - X3D graphics structure ID
|		 coord	     - single coordinate marker point
|		 marker_type - type of marker to display
|		 fg	     - color of marker
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_marker (
   int    id,
   Coord  coord,
   int    marker_type,
   XColor *fg)
{
	int    tmp;
	XPoint point;
	X3DGraphics	*graphics;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_rectangle:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if (!_X2D_clip_points(graphics, &coord, 1, X2D_CENTER, &tmp))
	{
	   return;
	}

	if (!(_X2D_convert_wc_to_dc(graphics, &coord, &point, 1)))
	{
	   return;
	}

	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	_X2D_draw_markers(id, &point, 1, marker_type);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_rectangle
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coord	- the world coordinate upper left edge of rectangle 
|		 width	- the world coordinate width of rectangle
|		 height	- the world coordinate height of rectangle
|		 fg	- color of rectangle
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_rectangle (
   int    id,
   Coord  coord,
   Real   width,
   Real   height,
   XColor *fg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		points[2];
	Coord		coords[2], *ncoords;

	int		x, y;
	unsigned int	w, h;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_rectangle:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}
	coords[0] = coord;
	coords[1].x = coord.x + width;
	coords[1].y = coord.y + height;

	if ((ncoords = _X2D_clip_rectangles(graphics, coords, 2, &nsize))
		== NULL)
	   return;

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	   return;

	x = kmin(points[0].x, points[1].x);
	y = kmin(points[0].y, points[1].y);
	w = kabs(points[1].x - points[0].x);
	h = kabs(points[1].y - points[0].y);
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].rectangle(graphics, x, y, w, h);

	if (coords != ncoords)
	   kfree(ncoords);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_circle
|
|       Purpose: 
|
|         Input: id	- X3D graphics structure ID
|		 coord	- the world coordinate center of the circle
|		 radius - the world coordinate radius of the circle
|		 fg	- color of circle
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_circle (
   int    id,
   Coord  coord,
   Real   radius,
   XColor *fg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		points[2];
	Coord		coords[2], *ncoords;

	int		x, y;
	unsigned int	w, h;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_rectangle:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	coords[0].x = coord.x - radius;
	coords[0].y = coord.y - radius;
	coords[1].x = coord.x + radius;
	coords[1].y = coord.y + radius;
	if ((ncoords = _X2D_clip_rectangles(graphics, coords, 2, &nsize))
		== NULL)
	   return;

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	   return;

	if ( points[0].x < points[1].x )
	{
	   x = (points[1].x - points[0].x)/2.0 + points[0].x;
	}
	else
	{
	   x = (points[0].x - points[1].x)/2.0 + points[1].x;
	}

	if ( points[0].y < points[1].y )
	{
	   y = (points[1].y - points[0].y)/2.0 + points[0].y;
	}
	else
	{
	   y = (points[0].y - points[1].y)/2.0 + points[1].y;
	}
	w = kabs(points[1].x - points[0].x);
	h = kabs(points[1].y - points[0].y);
	X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	X3D_draw[graphics->device].arc(graphics, x, y, w, h, 0, 360);

	if (coords != ncoords)
	   kfree(ncoords);
}

/*-----------------------------------------------------------
|
|  Routine Name: draw_colorline
|
|       Purpose: Draws a multi-colored line according to the
|		 different color levels.
|
|		 draw_colorline() is the real routine that computes
|		 the different colored line segments and draws them.
|		 This is done in a seperate routine in order to speed
|		 up the graphics processing by avoiding the costly
|		 setup time required for X2D_draw_colorline().
|		 draw_colorline() runs thru the coordinates finding
|		 the contour level indxes for a given line segment.
|		 If the levels are the same then the line is drawn
|		 with that color, otherwise the line is divided into
|		 segements for each contour level for the given line.
|
|         Input: graphics - X3D graphics structure ID
|		 coords	  - coordinates that make up the line
|		 points	  -
|		 size	  - number of coordinates
|		 levels	  - level divisions at which each new color will begin
|		 pixels	  - pixel values (color) for each level
|		 num	  - number of levels
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

static void draw_colorline(
   X3DGraphics *graphics,
   Coord       *coords,
   XPoint      *points,
   int         size,
   Real        *levels,
   XColor      *colors,
   int         num)
{
	int	i, indx1, indx2;
	XPoint	start, end;
	Coord   point;

	if (levels == NULL) return;
	indx1 = _X3D_find_level(coords[0].y, levels, num);
	X3D_draw[graphics->device].draw_color(graphics, &colors[indx1], TRUE);

	for (i = 1; i < size; i++)
	{
	    indx2 = _X3D_find_level(coords[i].y, levels, num);

	    if (indx1 == indx2)
	    {
	       X3D_draw[graphics->device].line(graphics, points[i-1].x,
		   points[i-1].y, points[i].x, points[i].y);
	    }
	    else
	    {
	       start = points[i -1];

	       if (indx2 < indx1)
	       {
	          while (indx1 != indx2)
	          {
		     /*
		      *  Compute the end point for 
		      */
		     point.x = kintercept(coords[i-1].x, coords[i].x, coords[i-1].y,
				      coords[i].y, levels[indx1]);
		     point.y = levels[indx1];

		     if (_X2D_convert_wc_to_dc(graphics, &point, &end, 1))
		     {
			X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx1], TRUE);
	                X3D_draw[graphics->device].line(graphics, start.x,
			        start.y, end.x, end.y);
		     }

		     /*
		      *  Save the end of the segment as the start of the new
		      *  segment.
		      */
		     start = end;
		     indx1--;
	          }
		  X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx2], TRUE);
	          X3D_draw[graphics->device].line(graphics, start.x,start.y,
			   points[i].x, points[i].y);
	       }
	       else
	       {
	          while (indx1 != indx2)
	          {
		     /*
		      *  Compute the end point for 
		      */
		     point.x = kintercept(coords[i-1].x, coords[i].x, coords[i-1].y,
				      coords[i].y, levels[indx1 +1]);
		     point.y = levels[indx1 +1];

		     if (_X2D_convert_wc_to_dc(graphics, &point, &end, 1))
		     {
			X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx2], TRUE);
	                X3D_draw[graphics->device].line(graphics, start.x,
			         start.y, end.x, end.y);
		     }

		     /*
		      *  Save the end of the segment as the start of the new
		      *  segment.
		      */
		     start = end;
		     indx1++;
	          }
		  X3D_draw[graphics->device].draw_color(graphics,
				&colors[indx2], TRUE);
	          X3D_draw[graphics->device].line(graphics, start.x,start.y,
			   points[i].x, points[i].y);
	       }
	    }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_draw_colorline
|
|       Purpose: Draws a multi-colored line according to the
|		 different color levels.  X2D_draw_colorline()
|		 is a frontend for the real routine static
|		 draw_colorline().
|
|		 X2D_draw_colorline() gets the graphics structure,
|		 perpares the contour levels by making sure they
|		 are sorted in ascending order, converts the world
|		 coordinates to device coordinates, then calls
|		 draw_colorline() to actually draw the multi-colored
|		 line.
|
|		 draw_colorline() is the real routine that computes
|		 the different colored line segments and draws them.
|
|         Input: id	    - X3D graphics structure ID
|		 coords	    - coordinates that make up the line
|		 size	    - number of coordinates
|		 levels	    - level divisions at which each new color will begin
|		 colors	    - pixel values (color) for each level
|		 num_levels - number of levels
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_draw_colorline(
   int    id,
   Coord  *coords,
   int    size,
   Real   *levels,
   XColor *colors,
   int    num_levels)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		*points;
	Coord		*ncoords;

	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_colorline:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	if ((ncoords = _X2D_clip_polygon(graphics, coords, size, &nsize))
		== NULL)
	   return;

	if ((points = (XPoint *) kmalloc ((unsigned) nsize * sizeof (XPoint)))
		== NULL)
	{
	   kfprintf (kstderr,"X2D_draw_colorline:\n");
	   kfprintf (kstderr,"\t not enough memory\n");
	   return;
	}

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	{
	   kfree(points);
	   return;
	}

	/*
	 *  Sort contours in ascending order.
	 *
	 sort_contour(levels, &contours, colors, &colors, num_levels);
	 */

	/*
	 *  Call draw_colorline() to compute the different colored line
	 *  segments and draw them.
	 */
	draw_colorline(graphics, coords, points, nsize, levels, colors,
		       num_levels);

	if (coords != ncoords)
	   kfree(ncoords);
	kfree(points);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_fill_rectangle
|
|       Purpose:
|
|         Input: id	- X3D graphics structure ID
|		 coord	- the beginning world coordinate point
|		 width	- the world coordinate width of rectangle
|		 height	- the world coordinate height of rectangle
|		 fg	- color of rectangle
|		 bg	- fill color of rectangle
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_fill_rectangle(
   int    id,
   Coord  coord,
   Real   width,
   Real   height,
   XColor *fg,
   XColor *bg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		points[2];
	Coord		coords[2], *ncoords;

	int		x, y;
	unsigned int	w, h;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_rectangle:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	coords[0] = coord;
	coords[1].x = coord.x + width;
	coords[1].y = coord.y + height;
	if ((ncoords = _X2D_clip_rectangles(graphics, coords, 2, &nsize))
		== NULL)
	   return;

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	   return;

	x = kmin(points[0].x, points[1].x);
	y = kmin(points[0].y, points[1].y);
	w = kabs(points[1].x - points[0].x);
	h = kabs(points[1].y - points[0].y);
	X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
	X3D_draw[graphics->device].fill_rectangle(graphics, x, y, w, h);

	if (graphics->line_width != KLINE_NONE)
	{
	   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	   X3D_draw[graphics->device].rectangle(graphics, x, y, w, h);
	}

	if (coords != ncoords)
	   kfree(ncoords);
}

/*-----------------------------------------------------------
|
|  Routine Name: X2D_fill_circle
|
|       Purpose: 
|
|	  Input: id	- X3D graphics structure ID
|		 coord	- the world coordinate center of the circle
|		 radius	- the world coordinate radius of the circle
|		 fg	- color of circle
|		 bg	- fill color of circle
|
|        Output: none
|
|       Returns: nothing
|
|    Written By: Mark Young
|          Date: 
| Modifications:
|
------------------------------------------------------------*/

void X2D_fill_circle(
   int    id,
   Coord  coord,
   Real   radius,
   XColor *fg,
   XColor *bg)
{
	X3DGraphics	*graphics;
	int		nsize;
	XPoint		points[2];
	Coord		coords[2], *ncoords;

	int		x, y;
	unsigned int	w, h;


	if ((graphics = _X3D_get_graphics(id)) == NULL)
	{
	   kfprintf (kstderr,"X2D_draw_rectangle:\n");
	   kfprintf (kstderr,"\t unknown graphics id\n");
	   return;
	}

	coords[0].x = coord.x - radius;
	coords[0].y = coord.y - radius;
	coords[1].x = coord.x + radius;
	coords[1].y = coord.y + radius;
	if ((ncoords = _X2D_clip_rectangles(graphics, coords, 2, &nsize))
		== NULL)
	   return;

	if (!(_X2D_convert_wc_to_dc(graphics, ncoords, points, nsize)))
	   return;

	if ( points[0].x < points[1].x )
	{
	   x = (points[1].x - points[0].x)/2.0 + points[0].x;
	}
	else
	{
	   x = (points[0].x - points[1].x)/2.0 + points[1].x;
	}

	if ( points[0].y < points[1].y )
	{
	   y = (points[1].y - points[0].y)/2.0 + points[0].y;
	}
	else
	{
	   y = (points[0].y - points[1].y)/2.0 + points[1].y;
	}
	w = kabs(points[1].x - points[0].x);
	h = kabs(points[1].y - points[0].y);

	X3D_draw[graphics->device].fill_color(graphics, bg, FALSE);
	X3D_draw[graphics->device].fill_arc(graphics, x, y, w, h, 0, 360);
	if (graphics->line_width != KLINE_NONE)
	{
	   X3D_draw[graphics->device].draw_color(graphics, fg, FALSE);
	   X3D_draw[graphics->device].arc(graphics, x, y, w, h, 0, 360);
	}

	if (coords != ncoords)
	   kfree(ncoords);
}
