/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>             Attribute Actions For Toggles              <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>	     		xvf_get_toggle_size()	              <<<<
   >>>>	     		xvf_set_toggle_size()	              <<<<
   >>>>	     		xvf_get_toggle_num()		      <<<<
   >>>>	     		xvf_set_toggle_num()		      <<<<
   >>>>	     		xvf_get_toggle_val()		      <<<<
   >>>>	     		xvf_set_toggle_val()		      <<<<
   >>>>	     		xvf_get_toggle_type()		      <<<<
   >>>>	     		xvf_set_toggle_members()	      <<<<
   >>>>	     		xvf_switch_toggle_value()	      <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


/*-----------------------------------------------------------
|
|       Routine: xvf_get_toggle_size
|       Purpose: Gets the number of toggle values by counting
|                the selections that occur directly after 
|                the (-T) UIS line of a toggle selection
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_SIZE ("xvf_toggle_size")
|        Output: calldata    - passes back number of toggle members
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_toggle_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_toggle_size(kformstruct, KVF_TOGGLE_SIZE, calldata));
}


/*-----------------------------------------------------------
|
|       Routine: xvf_set_toggle_size
|       Purpose: Sets the number of toggle values by adding
|                or subtracting selections directly after
|                the (-T) UIS line of a toggle selection
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_SIZE ("xvf_toggle_size")
|                calldata    - provides number of toggle members
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_toggle_size(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int	     old_size;
	int          *new_size;
	kselection   *toggle, *toggle_start;
	
	new_size = (int *) calldata;

	if (!(kvf_get_attribute(kformstruct, KVF_TOGGLE_SIZE, 
			        &old_size)))
            return(FALSE);

	if (!(kvf_set_toggle_size(kformstruct, KVF_TOGGLE_SIZE, calldata)))
	    return(FALSE);


	/*
	 * more toggle members than before;  need to
         * add in the requisite number of additional toggle members
         * and map them to the displayed GUI
	 */
	if (*new_size > old_size)
	{
	    toggle_start = kformstruct->Selptr;
	    toggle = kformstruct->Selptr->toggle_next;
	    while (toggle != NULL)
	    {
		if (toggle->back == NULL)
		    xvf_create_selection(toggle, toggle_start->back, KPANE);
		toggle = toggle->next;
	    }
	}

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_toggle_num
|       Purpose: Gets the number (the index) of the toggle member 
|                that is currently selected.
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_NUM ("xvf_toggle_num")
|        Output: calldata    - passes back number of selected toggle member
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_toggle_num(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_toggle_num(kformstruct, KVF_TOGGLE_NUM, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_toggle_num
|       Purpose: Sets the value of a toggle selection to the member
|                with the number (the index) specified.
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_NUM ("xvf_toggle_num")
|                calldata    - provides number of member to be selected
|        Output: none
|
|          Date: March 22, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_toggle_num(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          old_toggle_num;
        int          *toggle_num;

        /* the value to which XVF_TOGGLE_NUM is to be set */
        toggle_num  = (int *) calldata;

	if (!(kvf_get_toggle_num(kformstruct, KVF_TOGGLE_NUM, &old_toggle_num)))
	    return(FALSE);

	if (!(kvf_set_toggle_num(kformstruct, KVF_TOGGLE_NUM, calldata)))
	    return(FALSE);

	if (!(xvf_switch_toggle_value(kformstruct->Selptr, 
				      old_toggle_num, *toggle_num)))
	    return(FALSE);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_toggle_val
|       Purpose: Gets the value of the toggle member that is
|                currently selected.  We don't know what data type
|                that value will be, so we put the value into a
|                string.
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_VAL ("xvf_toggle_val")
|        Output: calldata    - passes back string representation
|                            of the toggle value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_toggle_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_toggle_val(kformstruct, KVF_TOGGLE_VAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_toggle_val
|       Purpose: Sets the value of a toggle to the value  
|                specified.  We don't know what data type that 
|                value will be, so the value comes in as a string.
|                menuform field => current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_VAL ("xvf_toggle_val")
|                calldata    - provides string representation of 
|                            of the toggle value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_toggle_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int  old_toggle_num, new_toggle_num;

        if (!(kvf_get_toggle_num(kformstruct, KVF_TOGGLE_NUM, &old_toggle_num)))
            return(FALSE);

	if (!(kvf_set_toggle_val(kformstruct, KVF_TOGGLE_VAL, calldata)))
	    return(FALSE);

        if (!(kvf_get_toggle_num(kformstruct, KVF_TOGGLE_NUM, &new_toggle_num)))
            return(FALSE);

	if (!(xvf_switch_toggle_value(kformstruct->Selptr, old_toggle_num,
                                      new_toggle_num)))
            return(FALSE);

	return(TRUE);
}



/*-----------------------------------------------------------
|
|       Routine: xvf_get_toggle_type
|       Purpose: Gets the type of the toggle (InputFile, OutputFile,
|                Logical, Flag, Integer, Float, or String)
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_TYPE ("xvf_toggle_type")
|        Output: calldata    - sets number of toggle member currently selected
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_toggle_type(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_toggle_type(kformstruct, KVF_TOGGLE_TYPE, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_toggle_members
|       Purpose: Sets the labels for the members of the toggle 
|                menuform field <= current selection
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_TOGGLE_MEMBERS ("xvf_toggle_members")
|        Output: calldata    - ignored
|
|          Date: June 23, 1993
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_toggle_members(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
        static char  *top_label = "Strings Representing Choices";
        char         **prompts, **answers, *title;
        char         temp[KLENGTH];
        int          ok, i, toggle_size;
        kselection   *selection, *member;

        selection = kformstruct->Selptr;

        kvf_get_attribute(kformstruct, KVF_TOGGLE_SIZE, 
			  &toggle_size);

        /*
         *  set up for use of query object
         */
	prompts = (char **) kcalloc(1, toggle_size*sizeof(char *));
        if (prompts == NULL)
        {
            kerror("xvforms", "xvf_set_toggle_members",
                   "Cannot allocate room for prompt strings");
            return(FALSE);
        }

	answers = (char **) kcalloc(1, toggle_size*sizeof(char *));
        if (answers == NULL)
        {
            kerror("xvforms", "xvf_set_toggle_members",
                   "Cannot allocate room for answers strings");
            return(FALSE);
        }

	ok = FALSE;
	while (!ok)
	{
	member = selection->toggle_next;
	for (i = 0; i < toggle_size; i++)
        {
            ksprintf(temp, "String assoc. w/ %d:", i+1);
	    if (prompts[i] == NULL)
                prompts[i] = kstrdup(temp);
	    if (answers[i] == NULL)
                answers[i] = kcalloc(1, KLENGTH*sizeof(char));
	    if (answers[i] == NULL)
	    {
		kerror("xvforms", "xvf_set_toggle_members",
		       "Unable to allocate string for toggle member label");
		return(FALSE);
	    }
	    kvf_get_attribute(member->back_kformstruct, KVF_TITLE, &title);
	    if (title != NULL)
                ksprintf(answers[i], "%s", title);
	    else answers[i][0] = '\0';
	    member = member->next;
        }

        /*
         *  bring up query object w/ sets of prompts
         */
        if (!(xvu_query_wait(top_label, prompts, "Use", answers,
                             toggle_size, 40)))
            return(FALSE);
	ok = TRUE;
	for (i = 0; i < toggle_size; i++)
	{
	    if (kstrlen(answers[i]) == 0)
	    {
		ok = FALSE; 
		kerror("xvforms", "xvf_set_toggle_choices",
		       "You must provide a valid string for each toggle member's title.");
		break;
	    }
	}
	}

	
        /*
         *  move answers into titles of toggle members 
         */
	member = selection->toggle_next;
        for (i = 0; i < toggle_size; i++)
	{
	    xvf_set_attributes(member->back_kformstruct, 
			       XVF_TITLE,       answers[i],
	                       XVF_DESCRIPTION, answers[i],
			       NULL);
	    member = member->next;
	}

        for (i = 0; i < toggle_size; i++)
            kfree(prompts[i]);
        for (i = 0; i < toggle_size; i++)
            kfree(answers[i]);

        kfree(answers);
        kfree(prompts);

	if (kformstruct->callback_list != NULL)
            xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xvf_switch_toggle_value
|
|       Purpose: Changes the (appearance of) current value 
|                of a toggle to the member specified
|
|         Input: selection      - header of toggle group (-T)
|                old_toggle_num - index of previously selected toggle member 
|                new_toggle_num - index of newly selected toggle member 
|
|        Output: none
|       Returns: TRUE on success, FALSE on failure
|          Date: March 23, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int xvf_switch_toggle_value(
    kselection *selection,
    int        old_toggle_num,
    int        new_toggle_num)
{
	kselection *toggle;
	int        n, old_found, new_found, toggle_num;

	/*
	 * trying to be robust... error checking..
	 */
	if (selection->type != KUIS_TOGGLE)
	{
	    kerror("xvforms", "xvf_switch_toggle_value",
		   "Selection passed in not of type KUIS_TOGGLE");
	    return(FALSE);
	}
	if ((old_toggle_num < 0)  || (new_toggle_num < 0))
	{
	    kerror("xvforms", "xvf_switch_toggle_value",
		   "old and/or new toggle values may not be < 0");
	    return(FALSE);
	}
	kvf_get_attribute(selection->back_kformstruct, 
		          KVF_TOGGLE_SIZE, &toggle_num);
	if ((old_toggle_num > toggle_num) || (new_toggle_num > toggle_num))
	{
	    kerror("xvforms", "xvf_switch_toggle_value",
                   "Neither old nor new toggle values may be > total toggle member num");
            return(FALSE);
	}
	
        toggle = selection->toggle_next;
	old_found = FALSE;
	new_found = FALSE;
        n = 1;
        while (toggle != NULL)
        {
            /*
             * if its number is the same as the one requested,
             * reverse its colors, turning it on, and set it
             * to be selected
             */
            if (n == old_toggle_num)
            {
                xvw_reverse_colors(toggle->opt_object, FALSE);
                toggle->opt_selected = FALSE;
	        old_found = TRUE;
            }
            if (n == new_toggle_num)
            {
                xvw_reverse_colors(toggle->opt_object, TRUE);
                toggle->opt_selected = TRUE;
	        new_found = TRUE;
            }
            n++;
            toggle = toggle->next;
        }
	if (old_found && new_found)
	    return(TRUE);
	else return(FALSE);
}
