/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>		Attribute Actions For Integer Selections      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                 xvf_get_nobounds_sel()                <<<<
   >>>>                 xvf_set_nobounds_sel()                <<<<
   >>>>                 xvf_get_bounds_sel()                  <<<<
   >>>>                 xvf_set_bounds_sel()                  <<<<
   >>>>	     		xvf_get_int_gt0_sel()	              <<<<
   >>>>	     		xvf_set_int_gt0_sel()	              <<<<
   >>>>	     		xvf_get_int_ge0_sel()	              <<<<
   >>>>	     		xvf_set_int_ge0_sel()	              <<<<
   >>>>	     		xvf_get_int_lt0_sel()	              <<<<
   >>>>	     		xvf_set_int_lt0_sel()	              <<<<
   >>>>	     		xvf_get_int_le0_sel()	              <<<<
   >>>>	     		xvf_set_int_le0_sel()	              <<<<
   >>>>	     		xvf_get_int_upper()		      <<<<
   >>>>	     		xvf_set_int_upper()		      <<<<
   >>>>	     		xvf_get_int_lower()		      <<<<
   >>>>	     		xvf_set_int_lower()		      <<<<
   >>>>	     		xvf_get_int_def()		      <<<<
   >>>>	     		xvf_set_int_def()		      <<<<
   >>>>	     		xvf_get_int_val()		      <<<<
   >>>>	     		xvf_set_int_val()		      <<<<
   >>>>	     		xvf_get_mechanism()		      <<<<
   >>>>	     		xvf_set_mechanism()		      <<<<
   >>>>                                                       <<<<
   >>>>	     		xvf_change_intbounds()	              <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<  */

#include "internals.h"
#include "attributes.h"


/*-----------------------------------------------------------
|
|       Routine: xvf_get_nobounds_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not bounds are set on an integer selection 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute - XVF_INT_NOBOUNDS_SEL   ("xvf_int_nobounds_sel")
|                            XVF_FLOAT_NOBOUNDS_SEL ("xvf_float_nobounds_sel")
|                            XVF_DOUBLE_NOBOUNDS_SEL ("xvf_double_nobounds_sel")
|        Output: calldata    - passes back TRUE if no bounds set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_nobounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kformstruct->flag == KUIS_INTEGER)
	    return (kvf_get_nobounds_sel(kformstruct, 
				         KVF_INT_NOBOUNDS_SEL, calldata));
	else if (kformstruct->flag == KUIS_FLOAT)
            return (kvf_get_nobounds_sel(kformstruct, 
                                         KVF_FLOAT_NOBOUNDS_SEL, calldata));
        else if (kformstruct->flag == KUIS_DOUBLE)
            return (kvf_get_nobounds_sel(kformstruct, 
                                         KVF_DOUBLE_NOBOUNDS_SEL, calldata));
	else
	{
	    kerror("xvforms", "xvf_get_bounds_sel",
	           "Unexpected selection type!");
	    return(FALSE);
	}
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_nobounds_sel
|       Purpose: Sets no bounds on an integer selection 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute - XVF_INT_NOBOUNDS_SEL   ("xvf_int_nobounds_sel")
|                            XVF_FLOAT_NOBOUNDS_SEL ("xvf_float_nobounds_sel")
|                            XVF_DOUBLE_NOBOUNDS_SEL ("xvf_double_nobounds_sel")
|                calldata    - provides boolean value of TRUE if bounds not set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_nobounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kformstruct->flag == KUIS_INTEGER)
	{
	    if (!(kvf_set_nobounds_sel(kformstruct, KVF_INT_NOBOUNDS_SEL, 
				       calldata)))
	        return(FALSE);

	    xvf_change_intbounds(kformstruct, "nobounds");
	}
	else if (kformstruct->flag == KUIS_FLOAT)
        {
	    if (!(kvf_set_nobounds_sel(kformstruct, KVF_FLOAT_NOBOUNDS_SEL, 
				       calldata)))
	        return(FALSE);
            xvf_change_floatbounds(kformstruct, "nobounds");
	}
	else if (kformstruct->flag == KUIS_DOUBLE)
        {
            if (!(kvf_set_nobounds_sel(kformstruct, KVF_DOUBLE_NOBOUNDS_SEL,
                                       calldata)))
                return(FALSE);
            xvf_change_doublebounds(kformstruct, "nobounds");
        }
	else
	{
	    kerror("xvforms", "xvf_get_bounds_sel",
	           "Unexpected selection type!");
	    return(FALSE);
	}

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_bounds_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not bounds are set on an integer selection 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute - XVF_INT_NOBOUNDS_SEL   ("xvf_int_nobounds_sel")
|                            XVF_FLOAT_NOBOUNDS_SEL ("xvf_float_nobounds_sel")
|                            XVF_DOUBLE_NOBOUNDS_SEL ("xvf_double_nobounds_sel")
|        Output: calldata    - passes back TRUE if bounds set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_bounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kformstruct->flag == KUIS_INTEGER)
            return (kvf_get_bounds_sel(kformstruct, 
				       KVF_INT_BOUNDS_SEL, calldata));
	else if (kformstruct->flag == KUIS_FLOAT)
            return (kvf_get_bounds_sel(kformstruct, 
                                       KVF_FLOAT_BOUNDS_SEL, calldata));
        else if (kformstruct->flag == KUIS_DOUBLE)
            return (kvf_get_bounds_sel(kformstruct, 
                                       KVF_DOUBLE_BOUNDS_SEL, calldata));
	else
	{
	    kerror("xvforms", "xvf_get_bounds_sel",
	           "Unexpected selection type!");
	    return(FALSE);
	}
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_bounds_sel
|       Purpose: Sets the bounds on an integer selection 
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute - XVF_INT_NOBOUNDS_SEL   ("xvf_int_nobounds_sel")
|                            XVF_FLOAT_NOBOUNDS_SEL ("xvf_float_nobounds_sel")
|                            XVF_DOUBLE_NOBOUNDS_SEL ("xvf_double_nobounds_sel")
|                calldata    - provides boolean value of TRUE if bounds not set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_bounds_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (kformstruct->flag == KUIS_INTEGER)
	{
            if (!(kvf_set_bounds_sel(kformstruct, KVF_INT_BOUNDS_SEL, 
				     calldata)))
                return(FALSE);
	    xvf_change_intbounds(kformstruct, "bounds");
	}
	else if (kformstruct->flag == KUIS_FLOAT)
	{
            if (!(kvf_set_bounds_sel(kformstruct, KVF_FLOAT_BOUNDS_SEL, 
				     calldata)))
                return(FALSE);
	    xvf_change_floatbounds(kformstruct, "bounds");
	}
	else if (kformstruct->flag == KUIS_DOUBLE)
	{
            if (!(kvf_set_bounds_sel(kformstruct, KVF_DOUBLE_BOUNDS_SEL, 
				     calldata)))
                return(FALSE);
	    xvf_change_doublebounds(kformstruct, "bounds");
	}
        else
        {
            kerror("xvforms", "xvf_set_bounds_sel",
                   "Unexpected selection type!");
            return(FALSE);
        }


	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_gt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_GT0_SEL ("xvf_int_gt0_sel")
|        Output: calldata    - passes back TRUE if bounds > 0 set
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_gt0_sel(kformstruct, KVF_INT_GT0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_gt0_sel
|       Purpose: Sets the bounds of an int selection to > 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_GT0_SEL ("xvf_int_gt0_sel")
|                calldata    - provides value of TRUE if bounds > 0 are set
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_gt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_int_gt0_sel(kformstruct, KVF_INT_GT0_SEL, calldata)))
	    return(FALSE);

	xvf_change_intbounds(kformstruct, "gt0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_ge0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_GE0_SEL ("xvf_int_ge0_sel")
|        Output: calldata    - passes back TRUE if bounds are set >= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_ge0_sel(kformstruct, KVF_INT_GE0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_ge0_sel
|       Purpose: Sets the bounds of an int selection to >= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_GE0_SEL ("xvf_int_ge0_sel")
|                calldata    - provides boolean value of TRUE if bounds are >= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_ge0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_int_ge0_sel(kformstruct, KVF_INT_GE0_SEL, calldata)))
	    return(FALSE);

	xvf_change_intbounds(kformstruct, "ge0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_lt0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_LT0_SEL ("xvf_int_lt0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to < 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_lt0_sel(kformstruct, KVF_INT_LT0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_lt0_sel
|       Purpose: Sets the bounds of an int selection to < 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_LT0_SEL ("xvf_int_lt0_sel")
|                calldata    - provides boolean value of TRUE if bounds are < 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_lt0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_int_lt0_sel(kformstruct, KVF_INT_LT0_SEL, calldata)))
	    return(FALSE);

	xvf_change_intbounds(kformstruct, "lt0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);
	
	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_le0_sel
|       Purpose: Gets the boolean TRUE or FALSE depending on whether
|                or not the bounds of an int selection are <= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_LE0_SEL ("xvf_int_le0_sel")
|        Output: calldata    - passes back TRUE if bounds are set to <= 0
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_le0_sel(kformstruct, KVF_INT_LE0_SEL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_le0_sel
|       Purpose: Sets the bounds of an int selection to <= 0
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_LE0_SEL ("xvf_int_le0_sel")
|                calldata    - provides boolean value of TRUE if bounds are <= 0
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_le0_sel(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_int_le0_sel(kformstruct, KVF_INT_LE0_SEL, calldata)))
	    return(FALSE);

	xvf_change_intbounds(kformstruct, "le0");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_upper
|       Purpose: Gets the value of the upper bound for an int,
|                assuming that "xvf_int_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_UPPER ("xvf_int_upper")
|        Output: calldata    - passes back int upper bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_upper(kformstruct, KVF_INT_UPPER, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_upper
|       Purpose: Sets the value of the upper bound for an int,
|                assuming that "xvf_int_upper_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_UPPER ("xvf_int_upper")
|                calldata    - provides int upper bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_upper(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_int_upper(kformstruct, KVF_INT_UPPER, calldata)))
	    return(FALSE);

	xvf_change_intbounds(kformstruct, "bounds");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_lower
|       Purpose: Gets the value of the lower bound for an int,
|                assuming that "xvf_int_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_LOWER ("xvf_int_lower")
|        Output: calldata    - passes back int lower bound value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_lower(kformstruct, KVF_INT_LOWER, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_lower
|       Purpose: Sets the value of the lower bound for an int,
|                assuming that "xvf_int_lower_sel" is TRUE
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_LOWER ("xvf_int_lower")
|                calldata    - provides int lower bound value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_lower(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	if (!(kvf_set_int_lower(kformstruct, KVF_INT_LOWER, calldata)))
	    return(FALSE);

	xvf_change_intbounds(kformstruct, "bounds");

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_def
|       Purpose: Gets the value of the default for an int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_DEF ("xvf_int_def")
|        Output: calldata    - passes back int default value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_def(kformstruct, KVF_INT_DEF, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_def
|       Purpose: Sets the value of the default for an int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_DEF ("xvf_int_def")
|                calldata    - provides int default value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_def(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int *int_def;

	if (!(kvf_set_int_def(kformstruct, KVF_INT_DEF, calldata)))
	    return(FALSE);

	int_def = (int *) calldata;
	if (!(xvf_set_int_val(kformstruct, XVF_INT_VAL, int_def)))
	    return(FALSE);

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}


/*-----------------------------------------------------------
|
|       Routine: xvf_get_int_val
|       Purpose: Gets the value of the int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_VAL ("xvf_int_val")
|        Output: calldata    - passes back int value
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_int_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_int_val(kformstruct, KVF_INT_VAL, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_int_val
|       Purpose: Sets the value of an int
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_INT_VAL ("xvf_int_val")
|                calldata    - provides int value
|        Output: none
|
|          Date: March 20, 1994
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_int_val(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	kselection   *selection;
	Line_Info    line_info;
	char         temp[KLENGTH];

	if (!(kvf_set_int_val(kformstruct, KVF_INT_VAL, calldata)))
	    return(FALSE);

	selection = kformstruct->Selptr;

	if ((selection->value_object != NULL) ||
            (selection->scroll != NULL))
	{
	    kvf_clear_line_info(&line_info);
	    kvf_parse_int_line(selection->line, &line_info);
	    ksprintf(temp, "%d", line_info.int_val); 

 	    if (selection->value_object != NULL)
                xvw_set_attribute(selection->value_object, 
			          XVW_TEXT_STRING, temp);

	    if (selection->scroll != NULL)
            {
                xvw_set_attributes(selection->scroll,
                           XVW_SCROLL_MIN,   (double) line_info.lower_int,
                           XVW_SCROLL_MAX,   (double) line_info.upper_int,
                           XVW_SCROLL_VALUE, (double) line_info.int_val,
                           NULL);
            }
	    kvf_free_line_info_strings(&line_info);
	}

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|       Routine: xvf_get_mechanism
|       Purpose: Gets the mechanism for an integer,
|                float, or double selection 
|                (0 = NONE, 1 = scrollbar, 2 = dial)
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_MECHANISM ("xvf_mechanism")
|        Output: calldata    - passes back mechanism value
|
|          Date: May 7, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_get_mechanism(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	return (kvf_get_mechanism(kformstruct, KVF_MECHANISM, calldata));
}

/*-----------------------------------------------------------
|
|       Routine: xvf_set_mechanism
|       Purpose: Sets the mechanism for an integer,
|                float, or double selection 
|                (0 = NONE, 1 = scrollbar, 2 = dial)
|
|         Input: kformstruct - generic kformstruct holding ptr to selection
|                attribute   - XVF_MECHANISM ("xvf_mechanism")
|                calldata    - passes in mechanism value
|        Output: none
|
|          Date: May 7, 1993
|    Written By: Danielle Argiro 
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
int xvf_set_mechanism(
   kform_struct *kformstruct,
   char         *attribute,
   kaddr        calldata)
{
	int          *mechanism, bounds; 
	Line_Info    line_info;
	double       upper, lower;
	kselection   *selection;

        /* the value to which XVF_MECHANISM is to be set */
        mechanism  = (int *) calldata;

	if (!(kvf_set_mechanism(kformstruct, KVF_MECHANISM, calldata)))
	    return(FALSE);

	kvf_gen_parse_kformstruct(kformstruct, &line_info);

	selection = kformstruct->Selptr;
        if (selection->type == KUIS_INTEGER)
        {
            upper = (double) line_info.upper_int;
            lower = (double) line_info.lower_int;
            if (upper != lower)
                bounds = TRUE;
            else bounds = FALSE;
        }
        else if (selection->type == KUIS_FLOAT)
        {
            upper = (double) line_info.upper_float;
            lower = (double) line_info.lower_float;
            if (upper != lower)
                bounds = TRUE;
            else bounds = FALSE;
        }
        else if (selection->type == KUIS_DOUBLE)
        {
            upper = line_info.upper_double;
            lower = line_info.lower_double;
            if (upper != lower)
                bounds = TRUE;
            else bounds = FALSE;
        }
	else 
	{
	    kerror("xvforms", "xvf_set_mechanism", 
		   "Selection of unexpected type");
	    return(FALSE);
	}

	if (bounds && (*mechanism == 1))
        {
            xvf_append_selection_scrollbar(selection, &line_info);
        }
        else if (!bounds || (*mechanism == 0))
        {
            xvf_delete_scrollbar(selection);
        }

	if (kformstruct->callback_list != NULL)
	    xvf_call_gui_callbacks(kformstruct, attribute, calldata);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvf_change_intbounds
|
|       Purpose: Changes the bounds for a int selection
|
|         Input: kformstruct - generic kform_struct pointer
|		 string    - string rep of int bounds, may be one of:
|			     "gt0"        : bounds >  0
|			     "ge0"        : bounds >= 0
|			     "lt0"        : bounds <  0
|			     "le0"        : bounds <= 0
|                            "lower = %f" : lower bound 
|                            "upper = %f" : upper bound 
|			     "nobounds"   : no bounds 
|
|        Output: Returns TRUE on success, FALSE on Failure
|          Date: March 20, 1994 
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

int xvf_change_intbounds(
   kform_struct *kformstruct,
   char         *string)
{
	kselection *selection;
	Line_Info  line_info;
	int        scrollbar = FALSE;
	float      sb_width;
	char       temp[KLENGTH];

	kvf_clear_line_info(&line_info);
	selection = kformstruct->Selptr;
        kvf_gen_parse(selection->line, &line_info);

	if (kstrcmp(string, "bounds") == 0)
	   scrollbar = TRUE;

	/*
         *  take off scrollbar if there is one
         */
        if ((!scrollbar) && (selection->scroll != NULL))
	{
	     xvw_set_attribute(selection->back, XVF_MECHANISM, 0);
	}

	else if ((scrollbar) && (selection->scroll == NULL))
        {
            sb_width = xvf_calculate_sb_width(line_info.width,
                                line_info.optional, line_info.opt_sel,
                                line_info.title, (double) line_info.upper_int,
                                (double) line_info.lower_int,
                                line_info.precision, line_info.typeflag);
            if (sb_width >= 1.0)
                xvw_set_attribute(selection->back, XVF_MECHANISM, 1);
        }

	if (kformstruct->Selptr->value_object != NULL)
	{
 	    ksprintf(temp, "%d", line_info.int_val);
	    xvw_set_attribute(kformstruct->Selptr->value_object,
			      XVW_TEXT_STRING, temp);
	}

	kvf_free_line_info_strings(&line_info);

	return(TRUE);
}

