 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */ 

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>       	   Athena xvobject Utilities		      <<<<
   >>>>                                                       <<<<
   >>>>  Private:                                             <<<<
   >>>>                                                       <<<<
   >>>>         Text Utilities:                               <<<<
   >>>>		   xvw_get_textstring()                       <<<<
   >>>>		   xvw_set_textstring()                       <<<<
   >>>>		   xvw_get_textmulti()                        <<<<
   >>>>		   xvw_set_textmulti()                        <<<<
   >>>>                                                       <<<<
   >>>>         Scrollbar Utilities:                          <<<<
   >>>>		   xvw_get_scrollbar()                        <<<<
   >>>>		   xvw_set_scrollbar()                        <<<<
   >>>>		   xvw_update_scroll_value()                  <<<<
   >>>>                                                       <<<<
   >>>>         List Utilities:                               <<<<
   >>>>		   xvw_get_list_num()                         <<<<
   >>>>		   xvw_set_list_num()                         <<<<
   >>>>		   xvw_get_list_highlight()                   <<<<
   >>>>		   xvw_set_list_highlight()                   <<<<
   >>>>		   xvw_set_list_unhighlight()                 <<<<
   >>>>		   xvw_update_list_struct()                   <<<<
   >>>>                                                       <<<<
   >>>>         Misc Utilities:                               <<<<
   >>>>		   xvw_get_misc()                             <<<<
   >>>>		   xvw_set_misc()                             <<<<
   >>>>                                                       <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>		   xvw_change_list()                          <<<<
   >>>>		   xvw_reverse_colors()                       <<<<
   >>>>		   xvw_retrieve_menu()                        <<<<
   >>>>		   xvw_retrieve_list()                        <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

#ifdef KATHENA_DEF

#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>
#include "Xfwf/ScrList.h"

extern klist   *xvw_menuwidget_list;
extern klist   *xvw_listwidget_list;


/*################################################################
#
#   Text utilities
#
################################################################*/

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_textstring()
|
|       Purpose: Allocates & returns the contents of a text object
|                text field => string
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_TEXT_STRING
|                calldata  - returns the contents of text object
|
|        Output: none
|    Written By: Danielle Aogiro
|          Date: Nov 16, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_textstring(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int i;
        Arg arg[MAX_ARGS];
	char **string_return;

	string_return = (char **) calldata;

        i = 0;
        XtSetArg(arg[i], XtNstring, string_return);    i++;
        XtGetValues(xvw_widget(object), arg, i);
	return(TRUE);

}

/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_textstring()
|
|       Purpose: Replaces the contents of a text object
|                text field <= string
|
|         Input: object    - the object to have contents set
|                attribute - XVW_TEXT_STRING
|                calldata  - passes desired contents of string
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: Nov 16, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_textstring(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
        char         **string;
	char         *old_string, *fullpath;
        int          status, type, old_length; 
	int          edit_type, read_only = FALSE;
        XawTextBlock text_block;

	string = (char **) calldata;

	xvw_get_attribute(object, XVW_TEXT_TYPE, &type);

	if (type == KTEXT_TYPE_FILE)
	{
	    fullpath = kfullpath(*string, NULL, NULL);
	    /* file not there, or cannot be opened */
   	    if (kaccess(fullpath, R_OK))
   	    {  
      	        kerror("xvwidgets", "xvw_set_textstring",
		        "Invalid file");
                return(FALSE);
            }
	    XtVaSetValues(xvw_widget(object), XtNstring, fullpath, NULL);
	}
	else
	{
	    xvw_get_attribute(object, XVW_TEXT_EDIT_TYPE, &edit_type);
	    if (edit_type == KTEXT_READ)
	    {
		xvw_set_attribute(object, XVW_TEXT_EDIT_TYPE, KTEXT_EDIT);
		read_only = TRUE;
	    }
	    xvw_get_attribute(object, XVW_TEXT_STRING, &old_string); 
	    old_length = kstrlen(old_string);

            text_block.firstPos = 0;
            text_block.length   = kstrlen(*string);
            text_block.ptr      = *string;
            text_block.format   = FMT8BIT;
            status = XawTextReplace(xvw_widget(object), 
                                    (XawTextPosition) 0,
                                    (XawTextPosition) old_length, 
    				    &text_block);
            if (status == XawPositionError)
            {
           	kerror("xvobject", "xvw_set_textstring",
                       "XawPositionError for %s", xvw_name(object));
           	return(FALSE);
            }
            else if (status == XawEditError)
            {
           	kerror("xvobject", "xvw_set_textstring",
                       "XawEditError for %s", xvw_name(object));
           	return(FALSE);
            }

	    if (read_only)
	        xvw_set_attribute(object, XVW_TEXT_EDIT_TYPE, KTEXT_READ);

            XawTextDisplay(xvw_widget(object));
	}
	return(TRUE);
}


/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_textmulti()
|
|       Purpose: Gets the current text multiline type of the label object
|
|         Input: object    - the object to have contents obtained
|                attribute - XVW_TEXT_MULTILINE
|                calldata  - returns TRUE for multiline text, FALSE otherwise
|
|        Output: none
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993 11:53
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_textmulti(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	float num;
	int *multiline = (int *) calldata;


	xvw_get_attribute(object, XVW_CHAR_HEIGHT, &num);
	*multiline = (((int) num) > 1) ? TRUE : FALSE;
	return(TRUE);
}


/*-------------------------------------------------------------
|
|  Routine Name: xvw_set_textmulti()
|
|       Purpose: Sets the current text multiline of the text object.
|
|         Input: object    - the object to have contents set
|                attribute - XVW_TEXT_MULTILINE
|                calldata  - passes TRUE for multiline text, FALSE otherwise
|
|        Output: none
|    Written By: Mark Young & John Salas
|          Date: Feb 15, 1993 11:53
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_textmulti(
   xvobject object,
   char   *attribute,
   kaddr  calldata)
{
	int *multiline = (int *) calldata;


	if (*multiline)
	{
	   XtVaSetValues(xvw_widget(object),
		 	 XtNleftMargin,	     2,
		 	 XtNrightMargin,     2,
		 	 XtNtopMargin,       2,
		 	 XtNbottomMargin,    2,
		 	 XtNinsertPosition,  0,
		 	 XtNdisplayPosition, 0,
                         XtNscrollVertical,  XawtextScrollWhenNeeded,
			 NULL);
	   xvw_set_attribute(object, XVW_CHAR_MAX_HEIGHT, 100.0);
	}
	else
	{
	   XtVaSetValues(xvw_widget(object),
		 	 XtNleftMargin,	     1,
		 	 XtNrightMargin,     1,
		 	 XtNtopMargin,       1,
		 	 XtNbottomMargin,    1,
		 	 XtNinsertPosition,  0,
		 	 XtNdisplayPosition, 0,
                         XtNscrollVertical,  XawtextScrollNever,
			 NULL);
	   xvw_set_attribute(object, XVW_CHAR_MAX_HEIGHT, 1.0);
	}
	return(TRUE);
}

/*################################################################
#
#   Scrollbar utilities
#
################################################################*/

typedef struct _xvw_scrollbar
{
        double scroll_min;
        double scroll_max;
        double scroll_incr;
        double scroll_value;
	double athena_maxval;
	int    increment_set;
} xvw_scrollbar;

klist *scrollbar_list = NULL;

/*--------------------------------------------------------------
|
|  Routine Name: xvw_get_scrollbar()
|
|       Purpose: Allocates & returns the contents of a scrollbar
|                scrollbar field => {min,value,incr,max} value
|                for one of the attributes:
|
|                       XVW_SCROLL_MIN
|                       XVW_SCROLL_MAX
|                       XVW_SCROLL_VALUE
|                       XVW_SCROLL_INCR
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above value
|                calldata  - Returns the WC value of scrollbar   
|
|        Output: none
|    Written By: Mark Young
|          Date: Nov 16, 1992
| Modifications:
|
------------------------------------------------------------------*/

/* ARGSUSED */
int xvw_get_scrollbar(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
        klist         *entry;
        xvw_scrollbar *temp;
	double *retval = (double *) calldata;
        int    *orientation = (int *) calldata;
        float  maxval, incr = 0.0, tmpval = 0.0;


        if ((entry = klist_locate(scrollbar_list, (kaddr) object)) == NULL)
        {
           kerror("xvobject", "xvw_get_scrollbar",
                  "Can't retrieve %s for %s", attribute, 
		  xvw_name(object));
           return(FALSE);
        }
        temp = (xvw_scrollbar *) entry->client_data;

        if (kstrcmp(attribute, XVW_SCROLL_MIN) == 0)
           *retval = temp->scroll_min;
        else if (kstrcmp(attribute, XVW_SCROLL_MAX) == 0)
           *retval = temp->scroll_max;
        else if (kstrcmp(attribute, XVW_SCROLL_INCR) == 0)
           *retval = temp->scroll_incr;
        else if (kstrcmp(attribute, XVW_ORIENTATION) == 0)
	{
	   XtVaGetValues(xvw_widget(object), XtNorientation, orientation, NULL);
	   if (*orientation == XtorientHorizontal)
	       *orientation = KSCROLLBAR_ORIENT_HORIZ;
	   else if (*orientation == XtorientVertical)
	       *orientation = KSCROLLBAR_ORIENT_VERT;
	}
        else if (kstrcmp(attribute, XVW_SCROLL_VALUE) == 0)
	{
	   maxval = temp->athena_maxval;
	   XtVaGetValues(xvw_widget(object), XtNtopOfThumb, &tmpval, NULL);
	   if (maxval == 0.0)
	      temp->scroll_value = temp->scroll_min;
	   else
              temp->scroll_value = (((temp->scroll_max - temp->scroll_min) * 
			kmin(tmpval, maxval))/maxval) + temp->scroll_min;

           *retval = temp->scroll_value;
	}
        else
        {
           kerror("xvobject", "xvw_get_scrollbar",
                   "Unknown attribute '%s' for scrollbar object %s",
                   attribute, xvw_name(object));
           return(FALSE);
        }
        return(TRUE);
}


/*------------------------------------------------------------
|
|  Routine Name: xvw_set_scrollbar()
|
|       Purpose: Sets a scrollbar according to the min, max
|                for one of the attributes:
|
|                       XVW_SCROLL_MIN
|                       XVW_SCROLL_MAX
|                       XVW_SCROLL_VALUE
|                       XVW_SCROLL_INCR
|
|         Input: object    - the object to have contents obtained
|                attribute - one of the above values
|                calldata  - passes the desired WC value for scrollbar
|
|        Output: none
|    Written By: Danielle Argiro and Mark Young
|          Date: Nov 25, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_scrollbar(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	klist         *entry;
        xvw_scrollbar *temp;
	float         value, incr;
	int	      orientation, val;
	Dimension     length, thickness;
	Widget	      widget = xvw_widget(object);
	double        diff, *setval = (double *) calldata;


	if ((entry = klist_locate(scrollbar_list, (kaddr) object)) == NULL)
        {
           temp = (xvw_scrollbar *) kcalloc(1, sizeof(xvw_scrollbar));
           temp->scroll_max    = 1.0;
           temp->scroll_incr   = 0.1;
           temp->scroll_min    =
           temp->scroll_value  = 0.0;
           temp->athena_maxval = 0.9;
           temp->increment_set = FALSE;
           scrollbar_list = klist_add(scrollbar_list, (kaddr) object,
                                      (kaddr) temp);
        }
        else temp = (xvw_scrollbar *) entry->client_data;

	
        if (kstrcmp(attribute, XVW_SCROLL_MIN) == 0)
	{
           temp->scroll_min = *setval;
	   if (!temp->increment_set)
	      temp->scroll_incr = (temp->scroll_max - temp->scroll_min)*0.1;
	}
        else if (kstrcmp(attribute, XVW_SCROLL_MAX) == 0)
	{
           temp->scroll_max = *setval;
	   if (!temp->increment_set)
	      temp->scroll_incr = (temp->scroll_max - temp->scroll_min)*0.1;
	}
        else if (kstrcmp(attribute, XVW_SCROLL_INCR) == 0)
           temp->scroll_incr = *setval, temp->increment_set = TRUE;
        else if (kstrcmp(attribute, XVW_SCROLL_VALUE) == 0)
           temp->scroll_value = *setval;
        else if (kstrcmp(attribute, XVW_ORIENTATION) == 0)
	{
	   orientation = *((int *) calldata);

	   if (orientation == KSCROLLBAR_ORIENT_VERT) 
	       orientation = XtorientVertical; 
	   else if (orientation == KSCROLLBAR_ORIENT_HORIZ)
	       orientation = XtorientHorizontal;
	   else
	   {
		errno = KCALL;
	        kerror("xvwidgets", "xvw_set_scrollbar",
	 	       "Invalid value for XVW_ORIENTATION"); 
		return(FALSE);
	   }
	   XtVaGetValues(widget, XtNorientation, &val, NULL);

	   if (val != orientation)
	   {
	      if (orientation == XtorientHorizontal)
	      {
		 length    = widget->core.width;
		 thickness = widget->core.height;
	      }
	      else
	      {
		 length    = widget->core.height;
		 thickness = widget->core.width;
	      }
	      XtVaSetValues(widget, XtNorientation, orientation,
			XtNthickness, thickness, XtNlength, length, NULL);
	   }
	}
        else
        {
           kerror("xvobject", "xvw_set_scrollbar",
                   "Unknown attribute '%s' for scrollbar object %s",
                   attribute, xvw_name(object));
           return(FALSE);
        }
	diff = kabs(temp->scroll_max - temp->scroll_min);

	incr  = krange(0.0, temp->scroll_incr/diff, 1.0);
	value = ((temp->scroll_value - temp->scroll_min)/diff) * (1.0-incr);
	value = krange(0.0, value, 1.0-incr); temp->athena_maxval = 1.0-incr;

        XvwScrollbarSetThumb(xvw_widget(object), value, incr);
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_update_scroll_value()
|
|       Purpose: Sets the fields in the xvw_list_struct structure
|                according to the Athena widget set.
|
|         Input: movement - the scrollbar value to be set
|                type     - the type of scrollbar callback (CONT or INCR)
|                call_data - the callback call data
|        Output: none
|       Returns: none
|    Written By: Mark Young, Danielle Argiro, John Salas
|          Date: April 21, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_update_scroll_value(
    xvobject object,
    char     *type,
    kaddr    call_data,
    double   *value)
{
	int    movement = (int) call_data;

	float  tmpvalue, oldvalue, incr;
	Widget widget = xvw_widget(object);

	/*
	 *  Get current scrollbar value
	 */
	XtVaGetValues(widget, XtNshown, &incr, XtNtopOfThumb, &tmpvalue, NULL);

	oldvalue = tmpvalue;
	if (kstrcmp(type, XVW_SCROLL_INCR_MOTION) == 0)
	   tmpvalue += (movement < 0) ? incr : -incr;

	if (tmpvalue > 1.0-incr)
	   XvwScrollbarSetThumb(widget, 1.0-incr, incr);
	else if (tmpvalue < 0.0)
	   XvwScrollbarSetThumb(widget, 0.0, incr);
	else if (tmpvalue != oldvalue)
	   XvwScrollbarSetThumb(widget, tmpvalue, incr);

	xvw_get_attribute(object, XVW_SCROLL_VALUE, value);
}

/*################################################################
#
#   List utilities
#
################################################################*/

/*------------------------------------------------------------
|
|  Routine Name: xvw_get_list_num()
|
|       Purpose: Gets the size of the list.
|
|         Input: object    - the object to have contents obtained
|                attribute - should be XVW_LIST_SIZE 
|                calldata  - returns size of list                
|
|        Output: none
|    Written By: Danielle Argiro and Mark Young
|          Date: Nov 25, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_list_num(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int    *num    = (int *) calldata;

        if (kstrcmp(attribute, XVW_LIST_SIZE) == 0)
	{
	   *num = 0;
	   XtVaGetValues(xvw_widget(object), XtNnumberStrings, num, NULL);
	}
        else
        {
           kerror("xvobject", "xvw_get_list_num",
                   "Unknown attribute '%s' for list object %s",
                   attribute, xvw_name(object));
           return(FALSE);
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_num()
|
|       Purpose: Sets the size of the list.
|
|         Input: object    - the object to have contents obtained
|                attribute - should be XVW_LIST_SIZE 
|                calldata  - passes desired size of list          
|
|        Output: none
|    Written By: Danielle Argiro and Mark Young
|          Date: Nov 25, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_num(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	xvobject list;
	int       *num = (int *) calldata;

        if (kstrcmp(attribute, XVW_LIST_SIZE) == 0)
	{
	   list = xvw_retrieve_list(object);
	   XtVaSetValues(xvw_widget(list), XtNnumberStrings, *num, NULL);
	   xvw_set_attribute(list, XVW_CHAR_HEIGHT, *num);
	}
        else
        {
           kerror("xvobject", "xvw_set_list_num",
                  "Unknown attribute '%s' for list object %s",
                  attribute, xvw_name(object));
           return(FALSE);
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_get_list_highlight()
|
|       Purpose: Gets the highlighted element of the list.
|
|         Input: object    - the object to have highlighted element obtained
|                attribute - should be XVW_LIST_HIGHLT_ELEM 
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_list_highlight(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int *indx = (int *) calldata;
	XfwfScrolledListReturnStruct *list_return;
	XfwfScrolledListWidget widget =
		(XfwfScrolledListWidget) xvw_widget(object);

        if (kstrcmp(attribute, XVW_LIST_HIGHLT_ELEM) == 0)
	{
	    list_return = XfwfScrolledListGetHighlighted(widget);
	    if (list_return->index == XAW_LIST_NONE)
		*indx = -1;
	    else *indx = list_return->index;
	}
        else
        {
           kerror("xvobject", "xvw_get_list_highlight",
                   "Unknown attribute '%s' for list object %s",
                   attribute, xvw_name(object));
           return(FALSE);
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_highlight()
|
|       Purpose: Sets the highlighted element of the list.
|
|         Input: object    - the object to have list element highlighted
|                attribute - should be XVW_LIST_HIGHLT_ELEM
|                calldata  - passes to index of element to be highlighted
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 15, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_highlight(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	int      *indx = (int *) calldata;
	Widget widget = xvw_widget(object);


        if (kstrcmp(attribute, XVW_LIST_HIGHLT_ELEM) == 0)
	{
	   XfwfScrolledListHighlightItem(widget, *indx);
	}
        else
        {
           kerror("xvobject", "xvw_set_list_highlight",
                  "Unknown attribute '%s' for list object %s",
                  attribute, xvw_name(object));
           return(FALSE);
        }

	return(TRUE);
}


/*------------------------------------------------------------
|
|  Routine Name: xvw_set_list_unhighlight()
|
|       Purpose: Unhighlights an element of the list.
|
|         Input: object    - the object to have list element highlighted
|                attribute - should be XVW_LIST_UNHIGHLT_ELEM
|                calldata  - passes to index of element to be unhighlighted
|
|        Output: none
|    Written By: Danielle Argiro
|          Date: May 15, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_list_unhighlight(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	Widget widget = xvw_widget(object);

        if (kstrcmp(attribute, XVW_LIST_UNHIGHLT_ELEM) == 0)
	{
	   XfwfScrolledListUnhighlightAll(widget);
	}
        else
        {
           kerror("xvobject", "xvw_set_list_unhighlight",
                  "Unknown attribute '%s' for list object %s",
                  attribute, xvw_name(object));
           return(FALSE);
        }

	return(TRUE);
}
/*-----------------------------------------------------------
|
|  Routine Name: xvw_update_list_struct()
|
|       Purpose: Sets the fields in the xvw_list_struct structure
|                according to the Athena widget set.
|
|         Input: object      - the list object
|                type        - the type of callback
|                call_data   - the structure passed in as the call_data
|                              for a list widget by OLIT
|                list_struct - xvwidgets list structure to be set
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: June 1, 1992
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_update_list_struct(
    xvobject        object,
    char            *type,
    kaddr           call_data,
    int		    *double_click,
    xvw_list_struct *list_struct)
{
        /* in Athena, list item is returned in a XfwfScrolledListReturnStruct */
        XfwfScrolledListReturnStruct *list_return;

        /*
         *  translate information passed back in Athena's
         *  "XawListReturnStruct" form into the xvw_list_struct
         */
        list_return = (XfwfScrolledListReturnStruct *) call_data;
        list_struct->string = list_return->string;
        list_struct->list_index = list_return->index;
}

/*################################################################
#
#   Miscellaneous utilities
#
################################################################*/


/*------------------------------------------------------------
|
|  Routine Name: xvw_get_misc()
|
|       Purpose: Gets miscellaneous attributes of various xvobjects.
|
|         Input: object    - the object to have attribute obtained
|                attribute - one of the miscellaneous attributes, eg,
|                            XVW_TEXT_EDIT_TYPE
|        Output: calldata  - passes back the miscellaneous value to be set
|    Written By: Mark Young Danielle Argiro
|          Date: June 7, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_get_misc(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	Widget widget = xvw_widget(object);
	int    athena_value;
	int    *value = (int *) calldata;

	if (kstrcmp(attribute, XVW_TEXT_TYPE) == 0)
	{
            XtVaGetValues(widget, XtNtype, &athena_value, NULL);
	    if (athena_value == XawAsciiString)
	        *value = KTEXT_TYPE_STRING;
	    else if (athena_value == XawAsciiFile)
	        *value = KTEXT_TYPE_FILE;
        }
	else if (kstrcmp(attribute, XVW_TEXT_EDIT_TYPE) == 0)
        {
            XtVaGetValues(widget, XtNeditType, &athena_value, NULL);
	    if (athena_value == XawtextRead)
	        *value = KTEXT_READ;
	    else if (athena_value == XawtextEdit)
	        *value = KTEXT_EDIT;
	    else if (athena_value == XawtextAppend)
	        *value = KTEXT_APPEND;
        }
	else if (kstrcmp(attribute, XVW_TEXT_WRAP) == 0)
        {
            XtVaGetValues(widget, XtNwrap, &athena_value, NULL);
	    if (athena_value == XawtextWrapWord)
	        *value = KTEXT_WRAP_WORD;
	    else if (athena_value ==XawtextWrapLine)
	        *value = KTEXT_WRAP_LINE;
	    else if (athena_value ==XawtextWrapNever)
	        *value = KTEXT_WRAP_NEVER;
	}

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_VERT) == 0)
        {
            XtVaGetValues(widget, XtNscrollVertical, &athena_value, NULL);
	    if (athena_value == XawtextScrollAlways)
                *value = KSCROLLBAR_ALWAYS;
	    else if (athena_value == XawtextScrollWhenNeeded)
                *value = KSCROLLBAR_IFNEEDED;
	    else if (athena_value == XawtextScrollNever)
                *value = KSCROLLBAR_NEVER;
        }

        else if (kstrcmp(attribute, XVW_TEXT_SCROLL_HORIZ) == 0)
        {
            XtVaGetValues(widget, XtNscrollHorizontal, &athena_value, NULL);
	    if (athena_value == XawtextScrollAlways)
                *value = KSCROLLBAR_ALWAYS;
	    else if (athena_value == XawtextScrollWhenNeeded)
                *value = KSCROLLBAR_IFNEEDED;
	    else if (athena_value == XawtextScrollNever)
                *value = KSCROLLBAR_NEVER;
        }

        else if (kstrcmp(attribute, XVW_BUTTON_SHAPE) == 0)
        {
            XtVaGetValues(widget, XtNshapeStyle, &athena_value, NULL);
	    if (athena_value == XmuShapeOval)
                *value = KBUTTON_SHAPE_OVAL;
	    else if (athena_value == XmuShapeRectangle)
                *value = KBUTTON_SHAPE_RECT;
        }

        else if ((kstrcmp(attribute, XVW_LABEL_JUSTIFY) == 0) ||
                 (kstrcmp(attribute, XVW_PIXMAP_JUSTIFY) == 0))
        {
            XtVaGetValues(widget, XtNjustify, &athena_value, NULL);
	    if (athena_value == XtJustifyLeft)
                *value = KLABEL_JUSTIFY_LEFT;
	    else if (athena_value == XtJustifyRight)
                *value = KLABEL_JUSTIFY_RIGHT;
	    else if (athena_value == XtJustifyCenter)
                *value = KLABEL_JUSTIFY_CENTER;
        }
	return(TRUE);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_set_misc()
|
|       Purpose: Sets miscellaneous attributes of various xvobjects.
|
|         Input: object    - the object to have attribute set
|                attribute - one of the miscellaneous attributes, eg,
|                            XVW_TEXT_EDIT_TYPE
|                calldata  - passes in the miscellaneous value to be set
|
|        Output: none
|    Written By: Mark Young Danielle Argiro
|          Date: June 7, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
int xvw_set_misc(
   xvobject object,
   char     *attribute,
   kaddr    calldata)
{
	Widget widget = xvw_widget(object);
	int    *value = (int *) calldata;
	
	if (kstrcmp(attribute, XVW_TEXT_TYPE) == 0)
	{
	    if (*value == KTEXT_TYPE_STRING)
		XtVaSetValues(widget, XtNtype, XawAsciiString, NULL);
	    else if (*value == KTEXT_TYPE_FILE)
		XtVaSetValues(widget, XtNtype, XawAsciiFile, NULL);
	    else
	    {
		errno = KCALL;
		kerror("xvwidgets", "xvw_set_misc",
		       "Invalid value passed in for XVW_TEXT_TYPE");
		return(FALSE);
	    }
	}

	else if (kstrcmp(attribute, XVW_TEXT_EDIT_TYPE) == 0)
        {  

	    if (*value == KTEXT_READ)
		XtVaSetValues(widget, XtNeditType, XawtextRead, NULL);
	    else if (*value == KTEXT_EDIT)
		XtVaSetValues(widget, XtNeditType, XawtextEdit, NULL);
	    else if (*value == KTEXT_APPEND)
		XtVaSetValues(widget, XtNeditType, XawtextAppend, NULL);
	    else
	    {
		errno = KCALL;
		kerror("xvwidgets", "xvw_set_misc",
		       "Invalid value passed in for XVW_TEXT_EDIT_TYPE");
		return(FALSE);
	    }
	}

	else if (kstrcmp(attribute, XVW_TEXT_WRAP) == 0)
	{
	    if (*value == KTEXT_WRAP_WORD)
                XtVaSetValues(widget, XtNwrap, XawtextWrapWord, NULL);
            else if (*value == KTEXT_WRAP_LINE)
                XtVaSetValues(widget, XtNwrap, XawtextWrapLine, NULL);
            else if (*value == KTEXT_WRAP_NEVER)
                XtVaSetValues(widget, XtNwrap, XawtextWrapNever, NULL);
	    else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_WRAP");
                return(FALSE);
            }

	}

	else if (kstrcmp(attribute, XVW_TEXT_SCROLL_VERT) == 0)
        {
	    if (*value == KSCROLLBAR_ALWAYS)
                XtVaSetValues(widget, XtNscrollVertical, 
			      XawtextScrollAlways, NULL);
	    else if (*value == KSCROLLBAR_IFNEEDED)
                XtVaSetValues(widget, XtNscrollVertical, 
			      XawtextScrollWhenNeeded, NULL);
	    else if (*value == KSCROLLBAR_NEVER)
                XtVaSetValues(widget, XtNscrollVertical, 
			      XawtextScrollNever, NULL);
	    else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_SCROLL_VERT");
                return(FALSE);
            }
	}

	else if (kstrcmp(attribute, XVW_TEXT_SCROLL_HORIZ) == 0)
        {
	    if (*value == KSCROLLBAR_ALWAYS)
                XtVaSetValues(widget, XtNscrollHorizontal, 
			      XawtextScrollAlways, NULL);
	    else if (*value == KSCROLLBAR_IFNEEDED)
                XtVaSetValues(widget, XtNscrollHorizontal, 
			      XawtextScrollWhenNeeded, NULL);
	    else if (*value == KSCROLLBAR_NEVER)
                XtVaSetValues(widget, XtNscrollHorizontal, 
			      XawtextScrollNever, NULL);
	    else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_TEXT_SCROLL_HORIZ");
                return(FALSE);
            }
	}
	
        else if (kstrcmp(attribute, XVW_BUTTON_SHAPE) == 0)
        {
            if (*value == KBUTTON_SHAPE_OVAL)
                XtVaSetValues(widget, XtNshapeStyle, XmuShapeOval, NULL);
            else if (*value == KBUTTON_SHAPE_RECT)
                XtVaSetValues(widget, XtNshapeStyle, XmuShapeRectangle, NULL);
            else
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for XVW_BUTTON_SHAPE");
                return(FALSE);
            }
        }

	else if ((kstrcmp(attribute, XVW_LABEL_JUSTIFY) == 0) ||
		 (kstrcmp(attribute, XVW_PIXMAP_JUSTIFY) == 0))
	{
	    if (*value == KLABEL_JUSTIFY_LEFT)
                XtVaSetValues(widget, XtNjustify, XtJustifyLeft, NULL);
	    else if (*value == KLABEL_JUSTIFY_RIGHT)
                XtVaSetValues(widget, XtNjustify, XtJustifyRight, NULL);
	    else if (*value == KLABEL_JUSTIFY_CENTER)
                XtVaSetValues(widget, XtNjustify, XtJustifyCenter, NULL);
            else 
            {
		errno = KCALL;
                kerror("xvwidgets", "xvw_set_misc",
                       "Invalid value passed in for %s", attribute);
                return(FALSE);
	    }
	}
	return(TRUE);
}


/*################################################################
#
#   Public utilities
#
################################################################*/


/*-----------------------------------------------------------
|
|  Routine Name: xvw_change_list()
|
|       Purpose: Changes the contents of a list object
|                while it is displayed
|
|         Input: object  - the list object to have contents changed
|                list    - new contents of list object
|                size    - size of the new list (# entries)
|                resize  - TRUE if list object should try to resize
|                          itself after the change, FALSE otherwise.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: July 17, 1992
| Modifications:
|
-------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_change_list(
   xvobject object,
   char     **list,
   int      size,
   int      resize)
{
	int      i, width;
	static char *empty_list[] = { "" };
	Widget widget = xvw_widget(object);


	for (i = 0, width = 0.0; i < size; i++)
	   width = kmax(width, kstrlen(list[i]));

	if (size > 0)
	{
	   list = karray_copy(list, size, TRUE);
	   XfwfScrolledListSetList(widget, (String *) list, size, TRUE, NULL);
	   XfwfScrolledListUnhighlightAll(widget);
	   XtVaSetValues((Widget) widget, XtNsensitive, TRUE, NULL);
	   karray_free(list, size, NULL);
	}
	else
	{
	   XfwfScrolledListSetList(widget,(String *) empty_list, 1, TRUE, NULL);
	   XtVaSetValues((Widget) widget, XtNsensitive, FALSE, NULL);
	   XfwfScrolledListUnhighlightAll(widget);
	}
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_reverse_colors
|
|       Purpose: Reverses the foreground and background colors
|                on a button object to indicate whether it is selected.
|
|         Input: object - the button object on which to reverse colors
|                value  - TRUE if button should be changed to look selected,
|                         FALSE if button should be changed to look un-selected.
|        Output: none
|       Returns: none
|    Written By: Danielle Argiro
|          Date: Jan 14, 1991
| Modifications: converted from Khoros 1.0 (DA)
|
------------------------------------------------------------------*/
/* ARGSUSED */
void xvw_reverse_colors(
   xvobject object,
   int      value)
{
	int  i;
	Arg arg[MAX_ARGS];
	Pixel fg,bg;
 
	/*
	 * get current fg & bg
	 */
	i = 0;
	XtSetArg(arg[i], XtNforeground, &fg);  i++;
	XtSetArg(arg[i], XtNbackground, &bg);  i++;
	XtGetValues(xvw_widget(object), arg, i);
 
	/*
	 * toggle current fg & bg
	 */
	i = 0;
	XtSetArg(arg[i], XtNforeground, bg);	i++;
	XtSetArg(arg[i], XtNbackground, fg);	i++;
	XtSetValues(xvw_widget(object), arg, i);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvw_retrieve_list
|
|       Purpose: Retrieve the actual list widget associated
|                with the compound list object.
|
|         Input: list - the compound object for which the
|                       actual list object is desired
|        Output: none
|       Returns: The list object on success, NULL on failure.
|    Written By: Danielle Argiro
|          Date: April 21, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
xvobject xvw_retrieve_list(
    xvobject list)
{
        return(list);
}

/*------------------------------------------------------------
|
|  Routine Name: xvw_retrieve_menu
|
|       Purpose: Given a compound menubutton object, returns the
|                menu associated with the menubutton
|
|         Input: menubutton - the menubutton for which the menu is desired
|        Output:
|       Returns: The menu object on success, NULL on failure
|    Written By: Danielle Argiro
|          Date: May 18, 1993
| Modifications:
|
------------------------------------------------------------------*/
/* ARGSUSED */
xvobject xvw_retrieve_menu(
    xvobject menubutton)
{
	xvobject menu;
	klist    *menuwidget_list;

	menuwidget_list = klist_locate(xvw_menuwidget_list, menubutton);
	if (menuwidget_list != NULL)
	{
	   menu = (xvobject) klist_clientdata(menuwidget_list);
	   return(menu);
	}
	else return(NULL);
}

#else
/*
 * These are dummy function calls so that if the widget set is not supported
 * on the machine a message is printed out.
 */

/* ARGSUSED */
void xvw_change_list(
   xvobject object,
   char   **list,
   int    size,
   int    resize)
{
   kfprintf(kstderr,"\nThe Athena widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return;
}

/* ARGSUSED */
void xvw_reverse_colors(
   xvobject object,
   int    value)
{
   kfprintf(kstderr,"\nThe Athena widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return;
}

/* ARGSUSED */
xvobject xvw_retrieve_list(
    xvobject list)
{
   kfprintf(kstderr,"\nThe Athena widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return(NULL);
}

/* ARGSUSED */
xvobject xvw_retrieve_menu(
    xvobject menubutton)
{
   kfprintf(kstderr,"\nThe Athena widget set is either not supported for this \
machine or was not\ncompiled in by the installer of Khoros.  Please contact \
the installer.\n\n");
   return(NULL);
}
#endif  /* KATHENA_DEF */
/* Don't add after the endif */
