 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Routines relating to transport IO.
   >>>>
   >>>>         All code that directly manipulates an
   >>>>         open transport should be in this file.
   >>>>
   >>>>   Static:
   >>>>                 __kdms_transport_segment_data
   >>>>                 _kdms_transport_segment_data
   >>>>                 _kfile2kdms_flags
   >>>>  Private:
   >>>>                 _kdms_flags
   >>>>                 _kdms_write_data
   >>>>                 _kdms_read_data
   >>>>                 _kdms_read_buffer
   >>>>                 _kdms_make_temporary
   >>>>                 _kdms_make_temporaries
   >>>>                 _kdms_merge_temps
   >>>>
   >>>>   Public:
   >>>>
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/*-----------------------------------------------------------
|
|  Routine Name: (static) __kdms_transport_segment_data
|
|       Purpose: This is a recursive helper routine to store or
|		 retrieve an n-space regularly gridded chunk of 
|		 data in or out of a transport.
|
|         Input: transport  - a data transport
|		 function   - the function that either reads or writes
|			      arrays of type datatype into the transport
|			      indicated in the first argument.
|		 seeker	    - the seek functionthat advances in the
|			      transport indicated by the first argument
|		 permanence - indicates that seeking is possible.
|		 rank       - the highest dimension of the dataset
|			      (base 0).
|		 size       - an array of size (rank + 1) that 
|		      	      designates the size of the data being
|			      retrieved.
|		 skip       - an array of size (rank +1) of skip 
|			      factors that indicate seek advancements
|			      from one dimension to the next.
|		 datatype   - the data type of the transport
|		 linesize   - the size of the lowest axis in bytes
|		 position   - the current offset in data.
|		 
|        Output: data       - a pointer to sufficient memory to load
|			      the requested data from the transport if
|			      func is a read operation, or a pointer
|			      to a chunk of data that is to be written
|			      to the transport if the func argument is
|			      a write operation.
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jun 16, 1994 08:50
| Modifications:
|
------------------------------------------------------------*/

static void
__kdms_transport_segment_data(
   kaddr       transport,
   kaddr       (*function) PROTO((kaddr, kaddr, int, int)),
   int         (*seeker)   PROTO((kaddr, int,   int)),
   int         permanence,
   int         rank,
   int        *size, 
   int        *skip,
   int         datatype, 
   int         linesize,
   int        *position,
   char       *data)
{
   int i;

   if (rank > 0) 
      for (i = 0; i < size[rank]; i++)
	 __kdms_transport_segment_data(transport, function, seeker, 
				       permanence, rank - 1, size, skip, 
				       datatype, linesize, position, data);
   else 
   {
      if (datatype == KBIT)
	 function(transport, data + *position, (size[0] + 7) >> 3, 
			KUBYTE);
      else
	 function(transport, data + *position, size[0], datatype);

      *position += linesize;
   }

   if (skip[rank] != 0 && permanence != FALSE)
      seeker (transport, skip[rank], SEEK_CUR);

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_transport_segment_data
|
|       Purpose: Performs an ambiguous operation specified
|                by func on a portion of data for a given data
|                segment involving an open transport (which
|                this mysterious func function operates on)
|                containing an object.
|
|         Input: transport -
|		 machtype  -
|                function  -
|		 seeker    -
|		 offset    -
|		 datatype  -
|		 dimension -
|		 size      -
|		 data      -
|                x0        - the "upper-left" corner of the
|                            data region to be retrieved in
|                            n-space, where n is the value of
|			     dimension.
|                x1        - the "lower-right" corner of the
|                            data region to be retrieved in
|                            n-space, where n is the value of
|			     dimension.
|
|        Output:  
|
|       Returns: Something that is as of yet undetermined.
|
|    Written By: Jeremy Worley
|          Date: Oct 06, 1992 15:06
| Modifications:
|
------------------------------------------------------------*/

kaddr 
_kdms_transport_segment_data(
   kaddr transport,
   int machtype,
   kfunc_int function,
   kfunc_int seeker,
   int offset,
   int datatype,
   int dimension,
   int *size,
   int *offsets,
   kaddr * data,
   int *x0,
   int *x1)
{
   int local_size[KDMS_MAX_DIM];
   int i;
   int j;
   int skip[KDMS_MAX_DIM];
   int position = 0;
   int el_size;
   int rem_size;
   int boff;
   char *segdata;
   int permanence = seeker != NULL;
   int mult;
   int linesize;
   int tmp_begin[KDMS_MAX_DIM];
   int tmp_end[KDMS_MAX_DIM];
   
   kaddr (*func) PROTO((kaddr, kaddr, int, int)) = (kfunc_kaddr)function;
   int (*seek) PROTO((kaddr, int, int)) = seeker;
   
   /*
    * The first thing that it is natural to check is if the corners are
    * legal.  The two checks that must be performed are (1) all index values
    * of x0 less than or equal to all index values of x1; and (2)  both x0
    * and x1 positions lie within the data space.  This is true if all index
    * values are non-negative, and all index values are less that their
    * corresponding sizes in dim.
    */
   for (i = 0; i < dimension; i++)
      if (x0[i] > x1[i] || x0[i] >= size[i] || x1[i] >= size[i])
      {
         _kdms_set_error(KDMS_EINTERNAL);
         return (NULL);
      }

   /*
    * Obtain the size of a single element of the specified datatype on the
    * current machine.
    */
   el_size = kdata_size(datatype);

   /*
    * Obtain the size of a single element on the transport...this may be of a
    * different size than the current architecture's corresponding datatype.
    * This conditional is because kmach_sizeof returns 0 instead of 1
    * for KBIT.  kdata_size, on the otherhand, returns 1.  Go figure.
    */
   rem_size = (datatype == KBIT) ? 1 : kmach_sizeof(machtype, datatype);

   /*
    * tweak the data corners to be on byte boundaries if this is 
    * bit data.  Use temporary pointers to dodge painful, disasterous
    * and embarrassing side effects.
    */
   if (datatype == KBIT)
   {
      tmp_begin[0] = x0[0] - (x0[0] % 8);
      tmp_end[0] = x1[0] + ((((x1[0] + 1) % 8) != 0) ? 7 - (x1[0] % 8) : 0);
      for (i = 1; i < dimension; i++)
      {
	 tmp_begin[i] = x0[i];
	 tmp_end[i] = x1[i];
      }
      x0 = (int *)tmp_begin;
      x1 = (int *)tmp_end;
   }
   
   /*
    * The next thing to do is to determine if the calling routing wants this
    * routine to allocate space for this operation. If so, then allocate the
    * space...otherwise, set the internal pointer and forget about it.
    */
   if (*data == NULL)
   {
      if (datatype == KBIT)
	 *data = (char *)kcalloc(
	    (unsigned)(_kdms_num_elements(x0, x1, dimension) >> 3), 1);
      else
	 *data = (char *)kcalloc(
	    (unsigned)_kdms_num_elements(x0, x1, dimension),(unsigned)el_size);

      if (*data == NULL)
         return (NULL);
   }

   /*
    * ansi blows a fuse if we dont do this
    */
   segdata = (char *)(*data);

   /*
    * compute the offset
    */
   boff = _kdms_buffer_offset(x0, size, dimension, datatype);

   if (permanence)
   {
      if (seek(transport, (boff + offset), SEEK_SET) < 0)
         return (NULL);         /* error already trapped */
   }
   else
   {
      kinfo(KSYSLIB, 
	    "Currently unable to determine offset in a stream based "
	    "transport.  Thus, it is possible that the current offset is "
	    "not correct.  If you are experiencing difficulties when using "
	    "a stream based transport, this could be the cause.");
   }

   /*
    * at this point, we know that the x0 and x1 positions are
    * rational, and that the kfid position is set to x0.  the next
    * thing to do is to determine how many elements of data we can
    * access at a time. this is dictated by the difference between x0
    * and x1 of the lowest order size.  this next code segment will
    * calculate how much data we can grab at a time, and how to jump
    * through the data space in order to access the data rationally. 
    *
    * Note that in the case of bit data, the local_size will represent
    * the number of bits including the padding at the end of the low
    * index out to a factor of 8.  This code is a little tricky, but
    * the objective is skip factors in bytes...what a mess.
    */
   for (i = 0, mult = 1; i < dimension; i++) {
      local_size[i] = x1[i] - x0[i] + 1;
      if (datatype == KBIT && i == 0)
      {
	 local_size[0] = ((local_size[0] + 7) >> 3);
	 skip[i] = ((size[i] + 7) >> 3) - local_size[0];
	 local_size[0] <<= 3;
      }
      else
	 skip[i] = (size[i] - local_size[i]) * mult * rem_size;
      if (offsets != NULL)
	 skip[i] += offsets[i];
      mult *= size[i];
   }

   /*
    * I'm not entirely satisfied with this bit of code.  The idea is
    * to combine lower dimensions into a single dimension if the 
    * lower dimension is fully spanned.  The effect is that a single
    * transport access can be used to get multiple lines of data.
    * This code probably deserves some scrutiny.
    */
   for (i = 0; i < dimension - 1; i++)
   {
      for (j = 0; j < dimension - 1; j++)
      {
         if (skip[j] == 0)
         {
            skip[j] = skip[j + 1];
            skip[j + 1] = 0;
            local_size[j] *= local_size[j + 1];
            local_size[j + 1] = 1;
         }
      }
   }

   /*
    * call the recursive transport clip routine.
    */
   linesize = (datatype == KBIT) ? (local_size[0] >> 3) :
      local_size[0] * el_size;
   
   __kdms_transport_segment_data(transport, func, seeker, permanence,
				 dimension - 1, local_size, skip,
				 datatype, linesize, &position, segdata);
   
   /*
    * cool...now we're done.
    */
   return (*data);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _beyond_bound
|       Purpose: Returns true if the first argument is
|		 beyond the position of the second bound
|		 along any axis.  FALSE otherwise.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Mar 24, 1995 15:33
| Modifications:
|
------------------------------------------------------------*/

static int
_beyond_bound(int *pos, int *bound, int dim)
{
   int i;

   for (i = 0; i < dim; i++)
      if (pos[i] > bound[i])
	 return TRUE;

   return FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_copy_temp_to_perm
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 02, 1993 10:58
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_copy_temp_to_perm(kobject object, kpresentation * pres)
{
   kaddr data = NULL;
   int begin[KDMS_MAX_DIM] = {0, 0, 0, 0, 0}; 
   int end[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int chunk[KDMS_MAX_DIM] = {1, 1, 1, 1, 1}; 
   int chunk_size;
   int i; 
   int already_preset = FALSE;
   
   _kdms_compute_chunk_size(pres->segment->size,
			    pres->segment->dimension, chunk,
			    pres->segment->datatype, -1);
   
   for (i = 0, chunk_size = 1; i < KDMS_MAX_DIM; i++)
   {
      end[i] = chunk[i] - 1;
      chunk_size *= chunk[i];
   }

   /*
    * This loop copies the data from the temporary transport to the
    * permanent transport.  The beyond bound nonsense is to protect
    * against problems with bogus data getting replicated in the
    * destination.  We had this problem with kinset -i1 image:ball -i2
    * sequence:baby where every fifth frame had a copy of the ball
    * data set.  This protects against the problem.  The already
    * preset variable is used to prevent repeated presetting when it
    * is not necessary.  This can only be done if the begin marker of
    * the data set is already beyond the real bounds.
    */
   if (pres->segment->file != NULL)
   {
      do 
      {
	 if (!already_preset && _beyond_bound(end, pres->segment->outerbound,
					     pres->segment->dimension) &&
	    data != NULL)
	    _kdms_preset_data(data, pres->segment->datatype,
			      chunk_size, pres->pad_real, pres->pad_imag);

	 if (!_beyond_bound(begin, pres->segment->outerbound,
			   pres->segment->dimension))
	    data = _kdms_transport_segment_data(
	       pres->segment->file,
	       kfile_getmachtype(kfileno(pres->segment->file)),
	       kfread_generic,
	       kfseek,
	       0,
	       pres->segment->datatype,
	       pres->segment->dimension,
	       pres->segment->size,
	       NULL,
	       &data,
	       begin, end);
	 else
	    already_preset = TRUE;
	 
	 _kdms_glue_write_data(object, pres->id, data, begin, end);
	 
      }
      while (_kdms_advance_position(begin, end, pres->segment->size, chunk,
				    pres->segment->dimension));

      kfree(data);
   }

   if (pres->segment->data != NULL)
      _kdms_glue_write_data(object, pres->id, pres->segment->data,
                            pres->segment->begin, pres->segment->end);


   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_copy_perm_to_temp
|
|       Purpose: 
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Nov 02, 1993 10:58
| Modifications:
|
------------------------------------------------------------*/

static int
_kdms_copy_perm_to_temp(kobject object, kpresentation * pres)
{
   int begin[KDMS_MAX_DIM],
      end[KDMS_MAX_DIM],
      i,
      done;
   kaddr tmp = NULL;

   for (i = 0; i < KDMS_MAX_DIM; i++)
      begin[i] = end[i] = 0;

   end[0] = pres->segment->size[0] - 1;

   for (done = FALSE; done != FALSE;)
   {
      _kdms_glue_read_data(object, pres->id, &tmp, begin, end);
      _kdms_transport_segment_data(
                                pres->segment->file,
                   kfile_getmachtype(kfileno(pres->segment->file)),
				kfwrite_generic,
				kfseek,
				0,
                                pres->segment->datatype,
                                pres->segment->dimension,
                                pres->segment->size,
				NULL,
                                &tmp,
                                begin, end);

      /*
       * increment through space
       */
      for (i = 1; i < pres->segment->dimension; i++)
      {
         begin[i] = end[i] += 1;
         if (end[i] < pres->segment->size[i])
            break;
         else
            end[i] = begin[i] = 0;
      }
      if (!(end[pres->segment->dimension - 1] <
            pres->segment->size[pres->segment->dimension - 1]))
         done = TRUE;
   }
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms2kfile_flags
|
|       Purpose: converts kdms style flags into kfile language.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley & Steve Kubica
|          Date: Apr 14, 1994 16:19
| Modifications:
|
------------------------------------------------------------*/

int
_kdms2kfile_flags(int flags)
{
   int kf = 0;

   /* read-write case */
   if ((KOBJ_WRITE & flags) && (KOBJ_READ & flags))
      kf = O_RDWR;

   /* read-only case */
   if (!(KOBJ_WRITE & flags) && (KOBJ_READ & flags))
      kf = O_RDONLY;

   /* write-only case */
   if ((KOBJ_WRITE & flags) && !(KOBJ_READ & flags))
      kf = O_WRONLY | O_CREAT;

   return (kf);
}

/*-----------------------------------------------------------
|
|  routine name: _kdms_write_data
|
|       purpose: writes a portion of data for a given data
|                segment into an open transport containing
|                an object.
|
|         input: object  - a transport descriptor that contains
|                          the viff.
|                segment - the ksegment into which you want
|                          read data.
|                x0      - the "upper-left" corner of the
|                          data region to be retrieved in
|                          5-space.
|                x1      - the "lower-right" corner of the
|                          data region to be retrieved in
|                          5-space.
|
|        output:
|
|       returns: something that is as of yet undetermined.
|
|    written by: jeremy worley
|          date: oct 16, 1992 16:50
| modifications:
|
------------------------------------------------------------*/

int
_kdms_write_data(
                   kobject object,
                   kdsegment * dsegment,
                   int *x0,
                   int *x1)
{
   int status;
   int i;
   
   /*
    * The opening of the file is delayed as long as possible
    */
   if (dsegment->file == NULL && (dsegment->temporary || !dsegment->locked))
   {
      dsegment->file = ktmpfile();
      dsegment->temporary = TRUE;
   }

   if (!dsegment->locked)
   {
      status = (_kdms_transport_segment_data(
                    dsegment->file, kfile_getmachtype(kfileno(dsegment->file)),
		    kfwrite_generic, kfseek, 0, dsegment->datatype, 
		    dsegment->dimension, dsegment->size, 
		    NULL, &dsegment->data, x0, x1) != NULL);

      /*
       * this kludgey bit is to maintain the outer bound of valid
       * data.  This is only used by the temporary transports right
       * now so that _kdms_copy_temp_to_perm wont do stupid things
       * if the whole data set hasn't been written.
       */
      for (i = 0; i < dsegment->dimension; i++)
	    dsegment->outerbound[i] = kmax(x1[i], dsegment->outerbound[i]);
      
      kfflush(dsegment->file);
   }
   else
   {
      status = _kdms_glue_write_data(object, dsegment->id, dsegment->data, x0, x1);
   }

   return (status);
}

/*-----------------------------------------------------------
|
|  routine name: _kdms_read_data
|
|       purpose: reda a portion of data for a given data
|                segment out of an open transport containing
|                an object.
|
|         input: object  - a transport descriptor that contains
|                          the viff.
|                segment - the ksegment into which you want
|                          read data
|                x0      - the "upper-left" corner of the
|                          data region to be retrieved in
|                          5-space.
|                x1      - the "lower-right" corner of the
|                          data region to be retrieved in
|                          5-space.
|
|        output:
|
|       returns: something that is as of yet undetermined.
|
|    written by: jeremy worley
|          date: oct 16, 1992 16:48
| modifications:
|
------------------------------------------------------------*/

char *
_kdms_read_data(
                  kobject object,
                  kdsegment * dsegment,
                  int *x0,
                  int *x1)
{
   int   i;
   char *status;

   /*
    * The opening of the file is delayed as int as possible
    */
   if (dsegment->file == NULL)
   {
      status = _kdms_glue_read_data(object, dsegment->id, &dsegment->data, x0, x1);
      /*
       * One of the assumptions made is that the offset is with
       * respect to the top of the value data segment. Since we need
       * the information anyway, set the segment's begin and end
       * pointers.  Armed with this information, we can calculate the
       * net offset to x0.  Then, we'll jump to that position.
       */
      for (i = 0; i < dsegment->dimension; i++)
      {
         dsegment->begin[i] = x0[i];
         dsegment->end[i] = x1[i];
      }
   }
   else
   {
      status = _kdms_transport_segment_data(
                                         dsegment->file,
                             kfile_getmachtype(kfileno(dsegment->file)), 
					 kfread_generic, kfseek, 0,
                                         dsegment->datatype,
                                         dsegment->dimension,
                                         dsegment->size,
					 NULL,
                                         &dsegment->data,
                                         x0, x1);
      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         dsegment->begin[i] = x0[i];
         dsegment->end[i] = x1[i];
      }
   }
   return (status);
}

/*-----------------------------------------------------------
|
|  routine name: _kdms_read_buffer
|
|       purpose: this routine is used to "intelligently"
|                read into the segment buffer a "reasonable"
|                amount of data.  the begin and end arguments
|                are passed in to indicate a specific region
|                of the data that must be loaded.
|
|                this routine will always read from the
|                device, so it is assumed that the routine
|                calling this one has already determined that
|                data must be read in.
|
|         input: object   - the data object that contains the
|                           segment->  this is really only
|                           necessary to get the transport
|                           descriptor.
|                dsegment - the data segment that is going
|                           to be operated on.
|                begin    - the beginning of the "required"
|                           region of data.
|                end      - the end corner of the "required"
|                           region of data.
|
|        output: dsegment - the same dsegment, but with
|                           new buffered data.
|
|       returns: TRUE (1) on success, FALSE (0) otherwise
|
|    written by: jeremy worley & mark young
|          date: nov 12, 1992 15:56
| modifications:
|
------------------------------------------------------------*/

int
_kdms_read_buffer(
                    kobject object,
                    kdsegment * dsegment,
                    int *begin,
                    int *end)
{
   int x0[KDMS_MAX_DIM],
      x1[KDMS_MAX_DIM];     /* similar to begin,end except that these mark the
                             * begin and end of the buffered data */
   int i,
      status;

   /*
    * check to see if the file exists.  if not, then there is really
    * no data to get, so just return FALSE.  What this does is prevent
    * unused memory from being allocated and instructs the calling
    * routine to generate data if necessary since none was available.
    * Without this, the calling routine will think that dsegment->data
    * is valid data, when in fact it is just a bunch of zeros.
    */
   if (dsegment->file == NULL && !object->phys->headerless && (dsegment->temporary || !dsegment->locked))
      return FALSE;
   
   /*
    * determine the optimal region to read into the buffer. you might notice
    * that this segment of code is still incredibly stupid--it will only read
    * in as much data as is necessary to satisfy the begin-end region.
    */
   for (i = 0; i < KDMS_MAX_DIM; i++)
   {
      x0[i] = begin[i];
      x1[i] = end[i];
   }

   /*
    * if the size of the 5d rectangle is now larger than it once was,
    * then we must free the segment data so that the readdata will allocate a
    * new buffer of sufficient size.
    */
   for (i = 0, status = 0x0; i < dsegment->dimension; i++)
      status |= (dsegment->end[i] - dsegment->begin[i] < end[i] - begin[i]);

   if (status || dsegment->data == NULL)
   {
      kfree(dsegment->data);

      dsegment->data = (char *)kcalloc((unsigned)_kdms_num_elements(x0, x1,
                      dsegment->dimension), 
				     (unsigned)kdata_size(dsegment->datatype));


      if (!dsegment->data)
         return (TRUE);
   }

   /*
    * grab the data
    */
   if (!_kdms_read_data(object, dsegment, x0, x1))
   {
      _kdms_set_error(KDMS_ESEG_READFAIL);
      return (FALSE);
   }
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: _kdms_make_temporary
|
|       Purpose: Moves the contents of a single segment of the
|                input object to a temporary file.
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Apr 23, 1993 15:22
| Modifications:
|
------------------------------------------------------------*/


int
_kdms_make_temporary(kobject object, kpresentation * pres)
{
   kfile *tf;

   tf = ktmpfile();

   pres->segment->file = tf;
   pres->segment->temporary = TRUE;
   object->phys->temporary = TRUE;      /* this just indicates that we are using
                                         * temps */

   if (object->phys->flags & KOBJ_READ)
   {
      _kdms_copy_perm_to_temp(object, pres);
   }
   krewind(tf);
   kinfo(KSYSLIB,
         "Data for the segment '%s' in the object '%s' is now in '%s'",
         ktoken_to_string(pres->id), object->phys->filename,
         kfile_filename(kfileno(tf)));

   _kdms_flush_segment(object, pres);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_merge_temps
|
|       Purpose: This thing takes all of the temporary
|                files for each segment and writes them
|                out.  Also writes the header if necessary.
|
|                This thing must work in a stream environment.
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 25, 1993 12:29
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_merge_temps(kobject object)
{
   /*
    * first write the header out.  This will allow us to construct a list of
    * segments sorted by file offset.
    */
   if (object->phys->resources == NULL)
      _kdms_glue_output(object, object->phys->filename, object->phys->flags);

   _kdms_copy_temps(object);

   /*
    * the end
    */
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdms_copy_temps
|
|       Purpose: This thing copies the temporary files from
|                each segment out to the output.
|
|                NOTE: No need to flush because the buffer
|                in the segment is still ok.  This just
|                copies the previously flushed data out.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jul 13, 1993 13:26
| Modifications:
|
------------------------------------------------------------*/

int
_kdms_copy_temps(kobject object)
{
   kpresentation *pres = NULL;
   klist *list;

   /*
    * dump each segment
    */
   for (list = klist_head(object->presentations); list != NULL;
        list = klist_next(list))
   {
      if (!_kdms_get_segment_info(object, klist_token(list), &pres) || !pres)
         return (FALSE);

      _kdms_copy_temp_to_perm(object, pres);
      kfclose(pres->segment->file);
      pres->segment->file = NULL;
      pres->segment->temporary = FALSE;
   }

   return (TRUE);
}
