 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros XWD Data Service Routines
   >>>>
   >>>>   Static:
   >>>>  Private:
   >>>>             xwd_check()
   >>>>             xwd_input()
   >>>>             xwd_output()
   >>>>             _init()
   >>>>             xwd_destroy()
   >>>>   Public:
   >>>>             None - no public should ever exist as these are
   >>>>                    internal routines only accessible via the
   >>>>                    DataServiceInformation *services[] structure.
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


#if !defined(KXWD_DEF)
DataServiceInformation xwd_format[] =
{NULL};

#else

#include "kdatafmt/xwd.h"


static int xwd_check PROTO((int));
static int xwd_input PROTO((kobject, int, int));
static int xwd_output PROTO((kobject, int, int));
static int xwd_destroy PROTO((kobject));
static int xwd_order PROTO((char *, int *));
static int xwd_architecture PROTO((kobject));

DataServiceInformation xwd_format[] =
{
   {
      "X11 Window Dump File Format (xwd)",
      "xwd",
      xwd_check,
      xwd_input,
      xwd_output,
      xwd_destroy,
      NULL,
      NULL,
      xwd_order,
      xwd_architecture,
      NULL,
      NULL,
   }
};

/*
 *  Internal Resource Structure for the following data services
 *
 *              xwd  -  X11 Window Dump File Format Utilities (xwd)
 */
typedef struct
{
   kfile *file;
   xwd *image;
}
ResourceStruct;


/*-----------------------------------------------------------
|
|  Routine Name: _init - Creates an xwd image
|
|       Purpose: This function is used to create an xwd image.  Which
|                means that we use the current dimension values in order
|                to create a proper xwd.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
_init(kobject object)
{
   ResourceStruct *resources;


   /*
    * Create the image to the object file
    */
   resources = (ResourceStruct *) kcalloc(1, sizeof(ResourceStruct));

   if (resources == NULL)
      return (FALSE);

   _kdms_glue_set_resources(object, resources);
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xwd_check - Checks to see if the file is
|                                an xwd
|
|       Purpose: This function is used to check the first few bytes of
|                the stream to see if it is an xwd.
|
|         Input: object - the xwd object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
xwd_check(int fid)
{
   xwd *tmp;
   
   /*
    * Seems like the easiest way to do this is to check the header
    */
   if ((tmp = xwd_readheader(fid)) == NULL)
      return (FALSE);

   kfree(tmp);
   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: xwd_input - Reda a xwd image
|
|       Purpose: This function is used to read in the image if the
|                supplied data is xwd image.  The data was checked
|                by the xwd_check() routine in which the xwd
|                identifier indicated that the data is a valid xwd.
|                Since the data was valid we can assume simply read
|                the header in and initialize that data segments.
|
|         Input: object - the xwd object to be initialized
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

/*ARGSUSED */
static int
xwd_input(kobject object, int fid, int flags)
{
   ResourceStruct *resources;
   xwd *image;

   int type = KUBYTE,
      i,
      j;
   int size[KDMS_MAX_DIM],
      order[KDMS_MAX_DIM],
      begin[KDMS_MAX_DIM],
      end[KDMS_MAX_DIM];

   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * Read the image from the object file and go ahead and set the xwd image
    * within the resource structure.
    */
   image = resources->image = xwd_fdread(fid);
   if (image == NULL)
   {
      _kdms_set_error(KDMS_EFMT_FAILURE);
      return (FALSE);
   }

   /*
    * Initialize the data segments according to the data found in the xwd
    * structure.
    * 
    * The xwd structure only contains value (image->data)
    */

   /* check the value data */
   if (image->width != 0 && image->height != 0)
   {
      switch (image->bits_per_pixel)
      {
         case 1:
            type = KBIT;
            break;
         case 8:
            type = KUBYTE;
            break;
         case 16:
            type = KUSHORT;
            break;
         case 32:
            type = KUINT;
            break;
         default:
            _kdms_set_error(KDMS_EFMT_FAILURE);
            return (FALSE);
      }

      size[0] = image->bytes_per_line / (image->bits_per_pixel >> 3);
      size[1] = image->pixmap_height;
      size[2] = size[3] = size[4] = 1;

      order[0] = KWIDTH;
      order[1] = KHEIGHT;
      order[2] = KELEMENTS;
      order[3] = KDEPTH;
      order[4] = KTIME;

      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      _kdms_set_segment(object, KDMS_SEGMENT_VALUE, image->data, type,
                        order, size, begin, end, 5);

      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_VALUE, KDMS_COUPLING,KUNCOUPLED);
   }

   /*
    * set the image data to null.  This is because above, we "gave" it
    * to data services.  This prevents a resource freeing from doing
    * a second free of the data.
    */
   image->data = NULL;
   
   /* do the maps */
   if (image->ncolors != 0)
   {
      unsigned short *data;

      size[0] = 3;
      size[1] = image->ncolors;
      size[2] = size[3] = size[4] = 1;

      data = (unsigned short *)kmalloc(size[0] * size[1] *
                                       sizeof(unsigned short));

      order[0] = KMAP_WIDTH;
      order[1] = KMAP_HEIGHT;
      order[2] = KMAP_ELEMENTS;
      order[3] = KMAP_DEPTH;
      order[4] = KMAP_TIME;

      for (i = 0; i < KDMS_MAX_DIM; i++)
      {
         end[i] = size[i] - 1;
         begin[i] = 0;
      }

      for (i = 0, j = 0; i < size[1]; i++)
      {
         data[j++] = image->maps[i].red;
         data[j++] = image->maps[i].green;
         data[j++] = image->maps[i].blue;
      }

      _kdms_set_segment(object, KDMS_SEGMENT_MAP, data, KUSHORT,
                        order, size, begin, end, 5);

      /*
       * it is vitally important that you include this step because if the
       * data set larger than the buffer threshold, then spurous flushes
       * will ensue that should not happen...it turns out that xvimages and
       * viffs will cope with that situation (though not very efficiently),
       * but file formats that buffer the whole thing in memory won't.
       */
      kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_COUPLING,KUNCOUPLED);
   }

   kdms_create_attribute(object, NULL, _INTERNAL_COLORSPACE, 
			 1, 1, KINT, TRUE, TRUE);
   kdms_set_attribute(object, NULL, _INTERNAL_COLORSPACE, KRGB);

   return (TRUE);
}



/*-----------------------------------------------------------
|
|  Routine Name: xwd_output - Closes the xwd image
|
|       Purpose: This function is used to close the xwd image.  Which
|                means if the image was modified we then re-write image
|                back out.
|
|         Input: object - the xwd object to be closed
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/
#define XWD_MAX_VAL 65535.0

/*ARGSUSED */
static int
xwd_output(kobject object, int fid, int flags)
{
   ResourceStruct *resources;

   unsigned short *data;
   xwd *image;

   int vorder[KDMS_MAX_DIM] = {KELEMENTS, KWIDTH, KHEIGHT, KDEPTH, KTIME};
   int morder[KDMS_MAX_DIM] ={KMAP_WIDTH, KMAP_HEIGHT, KMAP_DEPTH, KMAP_TIME, 
			      KMAP_ELEMENTS};
   int begin[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int end[KDMS_MAX_DIM];
   int *size;
   int *msize;
   
   int has_maps;
   int has_vals;
   int type;
   int status;
   int i;
   int j;
   
   if (!_kdms_initialized(object))
      _init(object);

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources->image == NULL)
   {
      resources->image = xwd_create();
      if (!resources->image)
      {
         _kdms_traceback("xwd_output");
         _kdms_set_error(KDMS_EFMT_FAILURE);
         return (FALSE);
      }
   }

   image = resources->image;

   /*
    * check to see if the value and map segments exist.
    */
   has_maps = kdms_query_segment(object, KDMS_SEGMENT_MAP);
   has_vals = kdms_query_segment(object, KDMS_SEGMENT_VALUE);

   if (!has_vals)
   {
      kerror("kdms", "xwd_output", "The '%s' object has no value data.  It "
	     "cannot be written out as xwd.", kfile_filename(fid));
      return(FALSE);
   }

   /*
    * set a few handy attributes on the value segment and get the size
    */
   kdms_get_attribute(object, KDMS_SEGMENT_VALUE, KDMS_DATA_TYPE, &type);
   
   switch (type)
   {
      case KBIT:
      case KBYTE:
      case KUBYTE:
         type = KUBYTE;
	 image->pixmap_depth = image->bits_per_pixel = 8;
         break;
      case KSHORT:
      case KUSHORT:
	 image->pixmap_depth = image->bits_per_pixel = 16;
         type = KUSHORT;
         break;
      case KINT:
      case KUINT:
	 image->pixmap_depth = image->bits_per_pixel = 32;
         type = KUINT;
         break;
      default:
	 image->pixmap_depth = image->bits_per_pixel = 32;
         type = KULONG;
         break;
   }

   if (!kdms_set_attributes(object, KDMS_SEGMENT_VALUE, KDMS_DATA_TYPE, type,
                            KDMS_INDEX_ORDER, vorder, NULL))
      return (FALSE);

   kdms_get_attribute(object, KDMS_SEGMENT_VALUE, KDMS_SIZE, &size);

   /*
    * set a few attributes on the map segment.
    */
   if (has_maps)
   {
      kdms_set_attributes(object, KDMS_SEGMENT_MAP, KDMS_DATA_TYPE, KUSHORT, 
			  KDMS_INDEX_ORDER, morder, KDMS_SCALING, KNORMALIZE,
			  KDMS_NORM_MIN, 0.0, KDMS_NORM_MAX, XWD_MAX_VAL,
			  NULL);

      kdms_get_attribute(object, KDMS_SEGMENT_MAP, KDMS_SIZE, &msize);
   }
   
   end[0] = size[0] - 1;
   end[1] = size[1] - 1;
   end[2] = size[2] - 1;
   end[3] = end[4] = 0;
   
   image->pixmap_depth = size[0] * image->bits_per_pixel;
   image->pixmap_width =  image->width = size[1];
   image->pixmap_height = image->height = size[2];
   image->bytes_per_line = (image->pixmap_width * image->pixmap_depth) >> 3;

   image->data = kdms_get_data(object, KDMS_SEGMENT_VALUE, begin, end, NULL);

   /*
    * if the map exists, then get it and set the appopriate xwd fields
    */
   if (has_maps)
   {
      if (msize[0] == 1)
      {
	 image->visual_class = GrayScale;
      }
      else
      {
	 if (size[0] == 1)
	 {
	    image->bits_per_rgb = 8;
	    image->visual_class = PseudoColor;
	 }
	 else
	 {
	    image->bits_per_rgb = 24;
	    image->visual_class = DirectColor;
	 }
	 
	 msize[0] = 3;
	 kdms_set_attribute(object, KDMS_SEGMENT_MAP, KDMS_SIZE, msize);
      }

      end[0] = msize[0] - 1;
      end[1] = msize[1] - 1;
      end[2] = msize[2] - 1;
      end[3] = msize[3] - 1;
      end[4] = msize[4] - 1;
      
      data = kdms_get_data(object, KDMS_SEGMENT_MAP, begin, end, NULL);
      image->maps = (xwdcolor *) kmalloc(msize[1] * sizeof(xwdcolor));

      for (i = 0, j = 0; i < msize[1]; i++)
      {
         image->maps[i].red = data[j++];
         image->maps[i].green = data[j++];
         image->maps[i].blue = data[j++];
         image->maps[i].flags = 0xff;
	 image->maps[i].pixel = i;
      }
      image->colormap_entries = image->ncolors = msize[1];
      kfree(data);
   }
   else
   {
      image->visual_class = GrayScale;
      image->bits_per_rgb  = 8;
   }

   /*
    * Write the xwd structure out to the specified file
    */
   image->pixmap_format = ZPixmap;
   image->byte_order = MSBFirst;
   image->bitmap_pad = 8;
   image->bitmap_unit = 8;
   
   status = xwd_fdwrite(fid, image);
   
   return(status);
}

   
/*-----------------------------------------------------------
|
|  Routine Name: xwd_destroy - Frees an xwd image
|
|       Purpose: This function is used to create an xwd image.  Which
|                means that we use the current dimension values in order
|                to create a proper xwd.
|
|         Input: data - initial sample of data
|                num  - number of bytes in the sample
|
|        Output: Returns: returns TRUE or FALSE if an error occurs
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
xwd_destroy(kobject object)
{
   ResourceStruct *resources;

   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   /*
    * free the xwder structure
    */
   xwd_free(resources->image);

   /*
    * free the resources
    */
   kfree(resources);

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xwd_order
|
|       Purpose:
|
|         Input:
|
|        Output:
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:53
| Modifications:
|
------------------------------------------------------------*/

static int
xwd_order(char *seg, int *ord)
{
   if (kstring_to_token(seg) == kstring_to_token(KDMS_SEGMENT_MAP))
   {
      ord[0] = KMAP_WIDTH;
      ord[1] = KMAP_HEIGHT;
      ord[2] = KMAP_DEPTH;
      ord[3] = KMAP_TIME;
      ord[4] = KMAP_ELEMENTS;
   }
   return (TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: xwd_architecture
|
|       Purpose: Return the architecture that the data was
|                generated on in khoros-speak.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Mar 14, 1994 16:54
| Modifications:
|
------------------------------------------------------------*/

static int
xwd_architecture(kobject object)
{
   ResourceStruct *resources;

   /*
    * get the resource structure
    */
   resources = (ResourceStruct *) _kdms_glue_get_resources(object);

   if (resources == NULL)
      return (KMACH_LOCAL);
   else
      return (kfile_getmachtype(kfileno(resources->file)));

}

#endif /* KXWD_DEF */

/* don`t add after the endif */
