 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */
/* See $DATASERV/repos/copyright/Copyright for terms and conditions. */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         XVIMAGE File Format Utility Routines
   >>>>
   >>>>  Private:
   >>>>		    x vimage2khoros_mach()
   >>>>		    khoros2xvimage_mach()
   >>>>		    xvimage2khoros_datatype()
   >>>>		    khoros2xvimage_datatype()
   >>>>		    xvimage2khoros_maptype()
   >>>>		    khoros2xvimage_maptype()
   >>>>   Static:
   >>>>             close_image()
   >>>>             read_mapdata()
   >>>>             read_imagedata()
   >>>>             write_mapdata()
   >>>>             write_imagedata()
   >>>>   Public:
   >>>>             xvimage_size()
   >>>>             xvimage_write()
   >>>>             xvimage_readheader()
   >>>>             xvimage_read()
   >>>>             xvimage_free()
   >>>>             xvimage_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/xvimage.h>
#include <machine/kmachine.h>

#define BYTE_FIELD_SIZE 520

/*-----------------------------------------------------------
|
|  Routine Name: xvimage2khoros_mach 
|
|       Purpose: returns the machtype according to the machine 
|		 ordering for a given machine.  Note that 
|		 since this is a one-to-many mapping, the
|		 output may not correspond to the actual
|		 machine, but rather to a compatible
|		 machine.
|
|         Input: xv_mach - xvimage machorder
|
|	 Output: k_mach  - Khoros machine definition
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Jeremy Worley
|          Date: Jul 21, 1992 13:53
| Modifications:
|
------------------------------------------------------------*/

int xvimage2khoros_mach(
   int  xv_mach,
   int  *k_mach)
{

 switch(xv_mach){
    case VFF_DEP_IEEEORDER:   
         *k_mach = KMACH_SPARC;
	 break;
    case VFF_DEP_DECORDER:    
         *k_mach = KMACH_VAX;
	 break;
    case VFF_DEP_NSORDER:     
         *k_mach = KMACH_NS32000;
         break;
    case VFF_DEP_CRAYORDER:   
         *k_mach = KMACH_CRAY;
         break;
    case VFF_DEP_ALPHAORDER:   
         *k_mach = KMACH_ALPHA;
         break;
    default:
         *k_mach = KMACH_UNKNOWN;
         return(FALSE);
 }
 return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: khoros2xvimage_mach
|
|       Purpose: Returns the xvimage architecture corresponding
|		 to the given Khoros architecture description.
|
|         Input: k_mach - khoros architecture definition
|
|        Output: xv_mach - xvimage architecture definition
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Sep 28, 1992 12:13
| Modifications:
|
------------------------------------------------------------*/

int khoros2xvimage_mach(
   int  k_mach,
   char *xv_mach)
{
  int f,o;

  f = kmach_format(k_mach);
  o = kmach_order(k_mach);

  if(f == KFORMAT_IEEE && o == KORDER_BIG_ENDIAN){
     *xv_mach = VFF_DEP_IEEEORDER;
     return(TRUE); 
  }

  if(f == KFORMAT_IEEE && o == KORDER_LITTLE_ENDIAN){
     *xv_mach = VFF_DEP_NSORDER;
     return(TRUE);
  }

  if(f == KFORMAT_CRAY){
     *xv_mach = VFF_DEP_CRAYORDER;
     return(TRUE);
  }

  if(f == KFORMAT_IEEE_ALPHA){
     *xv_mach = VFF_DEP_ALPHAORDER;
     return(TRUE);
  }

  if(f == KFORMAT_VAX){
     *xv_mach = VFF_DEP_DECORDER;
     return(TRUE);
  }

  return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage2khoros_datatype
|
|       Purpose: The routine converts the xvimage datatype
|		 passed in as input into a khoros 2.0
|		 data format specification.
|
|         Input: xv_type - xvimage data type.
|
|        Output: 
|
|       Returns: The khoros 2.0 data type corresponding to
|		 the input argument. 
|
|    Written By: Jeremy Worley
|          Date: Sep 22, 1992 16:03
| Modifications:
|
------------------------------------------------------------*/

int xvimage2khoros_datatype(
   unsigned long xv_type,
   int           *k_type)
{
   switch (xv_type) {
    case VFF_TYP_BIT:
       *k_type = (int)KBIT;
       break;
    case VFF_TYP_1_BYTE:
       *k_type = (int)KUBYTE;
       break;
    case VFF_TYP_2_BYTE:
       *k_type = (int)KSHORT;
       break;
    case VFF_TYP_4_BYTE:
       *k_type = (int)KLONG;
       break;
    case VFF_TYP_FLOAT:
       *k_type = (int)KFLOAT;
       break;
    case VFF_TYP_DOUBLE:
       *k_type = (int)KDOUBLE;
       break;
    case VFF_TYP_COMPLEX:
       *k_type = (int)KCOMPLEX;
       break;
    case VFF_TYP_DCOMPLEX:
       *k_type = (int)KDCOMPLEX;
       break;
    default:
       return(FALSE);
   }
   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: khoros2xvimage_datatype
|
|       Purpose: The routine converts the Khoros 2.0 datatype
|		 passed in as input into a xvimage (Khoros 1.0)
|		 data format specification.
|
|         Input: k_type - Khoros data type.
|
|        Output: xv_type - Xvimage data type corresponding to 
|		 the input argument
|
|       Returns: TRUE (1) on success, FALSE (0) on failure.
|
|    Written By: Jeremy Worley
|          Date: Sep 28, 1992 16:33
| Modifications:
|
------------------------------------------------------------*/

int khoros2xvimage_datatype(
   int           k_type,
   unsigned long *xv_type)
{
   switch (k_type) {
    case KBIT:
       *xv_type = (int)VFF_TYP_BIT;
       break;
    case KBYTE:
    case KUBYTE:
       *xv_type = (int)VFF_TYP_1_BYTE;
       break;
    case KSHORT:
    case KUSHORT:
       *xv_type = (int)VFF_TYP_2_BYTE;
       break;
    case KLONG:
    case KULONG:
    case KINT:
    case KUINT:
       *xv_type = (int)VFF_TYP_4_BYTE;
       break;
    case KFLOAT:
       *xv_type = (int)VFF_TYP_FLOAT;
       break;
    case KDOUBLE:
       *xv_type = (int)VFF_TYP_DOUBLE;
       break;
    case KCOMPLEX:
       *xv_type = (int)VFF_TYP_COMPLEX;
       break;
    case KDCOMPLEX:
       *xv_type = (int)VFF_TYP_DCOMPLEX;
       break;
    default:
       return(FALSE);
   }
   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage2khoros_maptype
|
|       Purpose: The routine converts the xvimage map data
|		 type passed in as input into a khoros 2.0
|		 data format specification.
|
|         Input: xv_type - xvimage map data type.
|
|        Output: k_type  - Khoros data type.
|
|       Returns: TRUE (1) on success, FALSE (0) on failure.
|
|    Written By: Jeremy Worley
|          Date: Sep 22, 1992 16:22
| Modifications:
|
------------------------------------------------------------*/

int xvimage2khoros_maptype(
   unsigned long xv_type,
   int           *k_type)
{
   switch (xv_type) {
    case VFF_MAPTYP_NONE:
       *k_type = -1;
       break;
    case VFF_MAPTYP_1_BYTE:
       *k_type = (int)KUBYTE;
       break;
    case VFF_MAPTYP_2_BYTE:
       *k_type = (int)KSHORT;
       break;
    case VFF_MAPTYP_4_BYTE:
       *k_type = (int)KLONG;
       break;
    case VFF_MAPTYP_FLOAT:
       *k_type = (int)KFLOAT;
       break;
    case VFF_MAPTYP_DOUBLE:
       *k_type = (int)KDOUBLE;
       break;
    case VFF_MAPTYP_COMPLEX:
       *k_type = (int)KCOMPLEX;
       break;
    default:
       return (FALSE);
   }
   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: khoros2xvimage_maptype
|
|       Purpose: The routine converts the Khoros 2.0 datatype
|		 passed in as input into a xvimage (Khoros 1.0)
|		 map data format specification.
|
|         Input: k_type - Khoros data type.
|
|        Output: xv_type - XVimage data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Sep 28, 1992 16:35
| Modifications:
|
------------------------------------------------------------*/

char khoros2xvimage_maptype(
   int           k_type,
   unsigned long *xv_type)
{
   switch (k_type) {
    case KBYTE:
    case KUBYTE:
       *xv_type = (unsigned long)VFF_MAPTYP_1_BYTE;
       break;
    case KSHORT:
    case KUSHORT:
       *xv_type = (unsigned long)VFF_MAPTYP_2_BYTE;
       break;
    case KLONG:
    case KULONG:
    case KINT:
    case KUINT:
       *xv_type = (unsigned long)VFF_MAPTYP_4_BYTE;
       break;
    case KFLOAT:
       *xv_type = (unsigned long)VFF_MAPTYP_FLOAT;
       break;
    case KDOUBLE:
       *xv_type = (unsigned long)VFF_MAPTYP_DOUBLE;
       break;
    case KCOMPLEX:
       *xv_type = (unsigned long)VFF_MAPTYP_COMPLEX;
       break;
    case KBIT:
    case KDCOMPLEX:
       return(FALSE);
    default:
       return(FALSE);
   }
   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: close_image - close the Khoros 1.0 xvimage file
|
|       Purpose: This routine closes the previously opened xvimage.
|		 The read_xvimage() or write_xvimage() routines
|		 opens the image for reading or writing, and then
|		 calls this routine to close the file and unlock
|		 it.
|
|         Input: file - the file to be closed and unlocked.
|
|        Output: None
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Scott Wilson
|          Date: Jul 24, 1992 13:19
| Modifications:
|
------------------------------------------------------------*/

static int close_image(
   int file)
{
   kclose(file);
   kflock(file,KLOCK_UN); 

   return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage_size - close the Khoros 1.0 xvimage file
|
|       Purpose: This routine closes the previously opened xvimage.
|		 The read_xvimage() or write_xvimage() routines
|		 opens the image for reading or writing, and then
|		 calls this routine to close the file and unlock
|		 it.
|
|         Input: file - the file to be closed and unlocked.
|
|        Output: None
|
|    Written By: Scott Wilson
|          Date: Jul 24, 1992 13:19
| Modifications:
|
------------------------------------------------------------*/

int xvimage_size(
   xvimage *image,
   int     *dsize,
   int     *dcount,
   int     *msize,
   int     *mcount,
   int     *lsize,
   int     *lcount)
{
    long rows,cols;
    int  mach;
    int  datasize,datacount;
    int  mapsize,mapcount;
    int  locsize,loccount;
    int  tmp;

    cols = image->row_size;
    rows = image->col_size;

    if(!xvimage2khoros_mach(image->machine_dep,&mach))
       return(FALSE);

    /* 
    ** Compute total size of DATA in bytes 
    */
    if(image->data_storage_type==VFF_TYP_BIT){
       datasize = ((cols+7)/8)*rows;
       datacount = datasize;
    }else{
       (void)xvimage2khoros_datatype(image->data_storage_type,&tmp);
       datasize = cols*rows*kmach_sizeof(mach, tmp);

       datacount = cols*rows;
    }

    datasize *= image->num_of_images*image->num_data_bands;
    datacount *= image->num_of_images*image->num_data_bands;
 
    /* 
    ** Compute number of MAP data objects 
    */
    switch(image->map_scheme)
      {
        case VFF_MS_NONE:
          mapcount = 0;
          break;
        case VFF_MS_ONEPERBAND:
        case VFF_MS_CYCLE:
          mapcount = 
              image->num_data_bands*image->map_row_size*image->map_col_size;
          break;
        case VFF_MS_SHARED:
        case VFF_MS_GROUP:
          mapcount = image->map_row_size*image->map_col_size;
          break;
        default:
	  errno = KINVALID_FILE;
          return(FALSE);
          /* break; */
      }

    /* 
    ** mapcount now contains the number of CELLS, so convert to bytes 
    */ 
    if(image->map_storage_type==VFF_MAPTYP_NONE){
       mapsize = 0;
    }else{
       xvimage2khoros_maptype(image->map_storage_type, &tmp);
       mapsize = mapcount * kmach_sizeof(mach, tmp);
    }

    /* 
    ** Compute size of LOCATION data in bytes and floats
    */
    (void)xvimage2khoros_datatype(VFF_TYP_FLOAT,&tmp);
    loccount = rows*cols*image->location_dim;
    locsize  = loccount*kmach_sizeof(mach, tmp);

    *dsize = datasize;
    *dcount = datacount;
    *msize = mapsize;
    *mcount = mapcount;
    *lsize = locsize;
    *lcount = loccount;
    return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: write_mapdata - write the Khoros 1.0 xvimage maps
|
|       Purpose: This routine writes the xvimage map data using
|		 the ktransport machine independent architecture
|		 write routines.
|
|         Input: file - the file to write the maps to
|		 maps - the map data to be written
|		 dim  - the number of data elements
|		 type - the data map type
|
|        Output: None
|
|    Written By: Jeremy Worley
|          Date: Jul 24, 1992 13:19
| Modifications:
|
------------------------------------------------------------*/
static int write_mapdata(
   int  file,
   kaddr maps,
   int  dim,
   int  type)
{
   switch(type){
     case VFF_MAPTYP_NONE:   
		return(TRUE);
     case VFF_MAPTYP_1_BYTE: 
		return(kwrite_ubyte(file,(unsigned char *)maps,dim) == dim);
     case VFF_MAPTYP_2_BYTE: 
		return(kwrite_ushort(file,(unsigned short *)maps,dim) == dim);
     case VFF_MAPTYP_4_BYTE: 
		return(kwrite_ulong(file,(unsigned long *)maps,dim) == dim);
     case VFF_MAPTYP_FLOAT:
		return(kwrite_float(file,(float *)maps,dim) == dim);
     case VFF_MAPTYP_DOUBLE: 
		return(kwrite_double(file,(double *)maps,dim) == dim);
     case VFF_MAPTYP_COMPLEX:
		return(kwrite_complex(file,(kcomplex *)maps,dim) == dim);
   }
   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: write_imagedata - write the Khoros 1.0 xvimage imagedata
|
|       Purpose: This routine writes the xvimage map data using
|		 the ktransport machine independent architecture
|		 write routines.
|
|         Input: file - the file to write the image data to
|		 image- the image data to be written
|		 dim  - the number of data elements
|		 type - the data map type
|
|        Output: None
|
|    Written By: Jeremy Worley
|          Date: Jul 24, 1992 13:19
| Modifications:
|
------------------------------------------------------------*/

static int write_imagedata(
   int  file,
   kaddr image,
   int  dim,
   int  type)
{
   switch(type){
      case VFF_TYP_BIT:  /*added by lotufo@dca.fee.unicamp.br 14dec94*/
	 return(kwrite_bit(file,(unsigned char *)image,dim) == dim);
      case VFF_TYP_1_BYTE: 
	 return(kwrite_ubyte(file,(unsigned char *)image,dim) == dim);
      case VFF_TYP_2_BYTE: 
	 return(kwrite_ushort(file,(unsigned short *)image,dim) == dim);
      case VFF_TYP_4_BYTE: 
	 return(kwrite_ulong(file,(unsigned long *)image,dim) == dim);
      case VFF_TYP_FLOAT:  
	 return(kwrite_float(file,(float *)image,dim) == dim);
      case VFF_TYP_DOUBLE: 
	 return(kwrite_double(file,(double *)image,dim) == dim);
      case VFF_TYP_COMPLEX:
	 return(kwrite_complex(file,(kcomplex *)image,dim) == dim);
      case VFF_TYP_DCOMPLEX:
	 return(kwrite_dcomplex(file,(kdcomplex *)image,dim) == dim);
   }
   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: read_mapdata - read the Khoros 1.0 xvimage maps
|
|       Purpose: This routine reads the xvimage map data using
|		 the ktransport machine independent architecture
|		 read routines.
|
|         Input: file - the file to read the maps from
|		 maps - the map data to be written
|		 dim  - the number of data elements
|		 type - the data map type
|
|        Output: None
|
|    Written By: Jeremy Worley
|          Date: Jul 24, 1992 13:19
| Modifications:
|
------------------------------------------------------------*/

static int read_mapdata(
   int  file,
   kaddr maps,
   int  dim,
   int  type)
{
   switch(type){
     case VFF_MAPTYP_NONE:   
		return(TRUE);
     case VFF_MAPTYP_1_BYTE:
		return(kread_ubyte(file,(unsigned char *)maps,dim) == dim);
     case VFF_MAPTYP_2_BYTE:
		return(kread_ushort(file,(unsigned short *)maps,dim) == dim);
     case VFF_MAPTYP_4_BYTE:
		return(kread_ulong(file,(unsigned long *)maps,dim) == dim);
     case VFF_MAPTYP_FLOAT:
		return(kread_float(file,(float *)maps,dim) == dim);
     case VFF_MAPTYP_DOUBLE:
		return(kread_double(file,(double *)maps,dim) == dim);
     case VFF_MAPTYP_COMPLEX:
		return(kread_complex(file,(kcomplex *)maps,dim) == dim);
   }
   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: read_imagedata - write the Khoros 1.0 xvimage imagedata
|
|       Purpose: This routine reads the xvimage map data using
|		 the ktransport machine independent architecture
|		 read routines.
|
|         Input: file - the file to read the image data from
|		 image- the image data to be read
|		 dim  - the number of data elements
|		 type - the data map type
|
|        Output: None
|
|    Written By: Jeremy Worley
|          Date: Jul 24, 1992 13:19
| Modifications:
|
------------------------------------------------------------*/

static int 
read_imagedata(
   int  file,
   kaddr image,
   int  dim,
   int  type)
{
   switch(type){
     case VFF_TYP_1_BYTE: 
		return(kread_ubyte(file,(unsigned char *)image,dim) == dim);
     case VFF_TYP_2_BYTE:
		return(kread_ushort(file,(unsigned short *)image,dim) == dim);
     case VFF_TYP_4_BYTE:
		return(kread_ulong(file,(unsigned long *)image,dim) == dim);
     case VFF_TYP_FLOAT:
		return(kread_float(file,(float *)image,dim) == dim);
     case VFF_TYP_DOUBLE: 
		return(kread_double(file,(double *)image,dim) == dim);
     case VFF_TYP_COMPLEX:
		return(kread_complex(file,(kcomplex *)image,dim) == dim);
     case VFF_TYP_DCOMPLEX:
		return(kread_dcomplex(file,(kdcomplex *)image,dim) == dim);
   }
   return(FALSE);
}

int xvimage_writeheader(
   int     file,
   xvimage *image)
{
   /*
    * write the first 520 bytes...they are just char data.
    */
    if(kwrite_byte(file,(char *)image,BYTE_FIELD_SIZE) < BYTE_FIELD_SIZE){
       close_image(file);
       return(FALSE);
    }

    /*
     * write the row and column sizes, and subrow_size;
     */
     if(kwrite_ulong(file,&(image->row_size),3) < 3){
        close_image(file);
        return(FALSE);
     }

    /*
     * write startx and starty (signed longs)
     */
     if(kwrite_long(file,&(image->startx),2) < 2){
        close_image(file);
        return(FALSE);
     }

    /*
     * write pixsizx and pixsizy (floats)
     */
     if(kwrite_float(file,&(image->pixsizx),2) < 2){
        close_image(file);
        return(FALSE);
     }

    /*
     * write a bunch of stuff
     */
     if(kwrite_ulong(file,&(image->location_type),16) < 16){
        close_image(file);
        return(FALSE);
     }

    /*
     * write the two float spares
     */
     if(kwrite_float(file,&(image->fspare1),2) < 2){
        close_image(file);
        return(FALSE);
     }
    
     return(TRUE);
}

/************************************************************
*
*  Routine Name: xvimage_write - write an xvimage structure to the
*				 specified filename
*
*       Purpose: This routines writes an xvimage structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: filename - the filename in which we will be writing
*			    the xvimage image and associated data
*                image    - the xvimage structure to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson and Jeremy Worley
*          Date: Sep 15, 1992 19:26
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int xvimage_write(
   char    *filename,
   xvimage *image)
{
    int  file;
    int  imgsize,locsize,mapsize,imgcnt,loccnt,mapcnt,n;
    int tmp;

    /* 
     * Check the image pointer 
     */
    if(image == NULL){
       errno = KNULL_PARAMETER;
       return(FALSE);
    }

    /* 
     * Open the output file 
     */
    if((file = kopen(filename,O_WRONLY|O_TRUNC|O_CREAT,0664)) < 0){
       return(FALSE);
    }

    if (kfile_getpermanence(filename) == TRUE) kflock(file, KLOCK_EX);

    if(!xvimage_size(image,&imgsize,&imgcnt,&mapsize,&mapcnt,&locsize, 
       &loccnt)) {
       close_image(file);
       return(FALSE);
    }

    /* 
     * Sanity check 
     */
    if(imgsize != 0 && image->imagedata == NULL){
       errno = KINVALID_INPUT;
       close_image(file);
       return(FALSE);
    }

    if(mapsize != 0 && image->maps == NULL){
       errno = KINVALID_INPUT;
       close_image(file);
       return(FALSE);
    }

    if(locsize != 0 && image->location == NULL){
       errno = KINVALID_INPUT;
       close_image(file);
       return(FALSE);
    }

   /*
    * If it is a cray image, then we want to write the image header out
    * in IEEE format so that it is exactly 1024 bytes.
    */
    if((int)image->machine_dep==VFF_DEP_CRAYORDER){
       (void)xvimage2khoros_mach(VFF_DEP_IEEEORDER,&tmp);
       kfile_setmachtype(file,tmp);
    }

    if(!xvimage_writeheader(file,image))
       return(FALSE);

    /*
     * skip over the reserve area
     */
     if (klseek(file, VIFF_HEADERSIZE, SEEK_SET) == -1){
        close_image(file);
        return(FALSE);
     }

    /* 
     * Write to normal file descriptor 
     */
    switch(image->data_encode_scheme){
       case VFF_DES_RAW:
            if(!write_mapdata(file,(kaddr)image->maps,
			      (int)(image->map_row_size * image->map_col_size),
			      (int)image->map_storage_type)){
               close_image(file);
               return(FALSE);
            }
            break;
       case VFF_DES_COMPRESS:
       case VFF_DES_RLE:
       case VFF_DES_TRANSFORM:
       case VFF_DES_CCITT:
       case VFF_DES_ADPCM:
       case VFF_DES_GENERIC:
	        errno = KLIMITATION;
                close_image(file);
                return(FALSE);
       default:
	        errno = KINVALID_INPUT;
                close_image(file);
                return(FALSE);
    } /* end switch */
                 
    /* 
     * Write the location data 
     */

    switch(image->data_encode_scheme){
       case VFF_DES_RAW: 
            n = image->row_size*image->col_size*image->location_dim;
            if(kwrite_float(file,image->location,n) < n){
               close_image(file);
               return(FALSE);
            }
            break;
       case VFF_DES_COMPRESS:
       case VFF_DES_RLE:
       case VFF_DES_TRANSFORM:
       case VFF_DES_CCITT:
       case VFF_DES_ADPCM:
       case VFF_DES_GENERIC:
	       errno = KLIMITATION;
               close_image(file);
               return(FALSE);
       default:
	       errno = KINVALID_INPUT;
            close_image(file);
            return(FALSE);
    } /* end switch */
                 
    /* 
     * write image data 
     */
    switch(image->data_encode_scheme){
       case VFF_DES_RAW:
            if(!write_imagedata(file,(kaddr)image->imagedata,
				(int)(image->row_size * image->col_size *
				image->num_data_bands * 
				image->num_of_images),
				(int)image->data_storage_type)){
               close_image(file);
               return(FALSE);
            }
            break;
       case VFF_DES_COMPRESS:
       case VFF_DES_RLE:
       case VFF_DES_TRANSFORM:
       case VFF_DES_CCITT:
       case VFF_DES_ADPCM:
       case VFF_DES_GENERIC:
	       errno = KLIMITATION;
               close_image(file);
               return(FALSE);
       default:
	       errno = KINVALID_INPUT;
            close_image(file);
            return(FALSE);
    }

    close_image(file);
    return(TRUE);
}

/************************************************************
*
*  Routine Name: xvimage_readheader - reads an xvimage header structure
*				      from the specified kfile id
*
*       Purpose: This routines reads an xvimage header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the xvimage header.
*
*         Input: fid - the kfile id opened previously with kopen()
*
*        Output: none
*
*       Returns: image - explanation
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson and Jeremy Worley
*          Date: Sep 15, 1992 19:26
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvimage *xvimage_readheader(
   int fid)
{
    unsigned char id1,id2;
    xvimage *image;
    int tmp;

    /* 
     * Grab space for the image structure and read it 
     */

    if((image = (xvimage *)kcalloc(1,sizeof(xvimage)))==NULL){
       return(NULL);
    }

    /*
     * read first few magic bytes of image that don't need to be converted.
     */
     if(kread_ubyte(fid,(unsigned char *)&(image->identifier),
		    BYTE_FIELD_SIZE)<0){
        kfree(image);
	return(NULL);
      }

    /*
     * verify that the user did, in fact, pass in a real-live VIFF file
     */
     id1 = (unsigned char) image->identifier;
     id2 = XV_FILE_MAGIC_NUM;
     if ((int) id1 != (int) id2 || image->file_type != XV_FILE_TYPE_XVIFF) {
	errno = KINVALID_FILE;
        kfree(image);
	return(NULL);
     }

    /* 
     * Check the release and version numbers 
     */

    if (image->release != XV_IMAGE_REL_NUM ||
        image->version != XV_IMAGE_VER_NUM)
    {
	errno = KINVALID_FILE;
        kfree(image);
	return(NULL);
    }

    /*
     * This little bit of wierdness is special to me.  The reason behind
     * it is that back in the dark ages, it was decided (by me) that
     * with the old tools, it would be much easier to deal with headers
     * that were always 1024 bytes long...that's not an issue anymore,
     * but all tradition really is is doing stupid things because
     * "that's the way we always did them"
     */
     if(image->machine_dep == VFF_DEP_CRAYORDER){
        (void)xvimage2khoros_mach(VFF_DEP_IEEEORDER,&tmp);
        kfile_setmachtype(fid, tmp);
     }else{
        (void)xvimage2khoros_mach(image->machine_dep,&tmp);
        kfile_setmachtype(fid, tmp);
     }
     
    /*
     * get the row and column sizes, and subrow_size;
     */
     if(kread_ulong(fid,&(image->row_size),3)<0){
        kfree(image);
	return(NULL);
      }

    /*
     * get startx and starty
     */
     if(kread_long(fid,&(image->startx),2)<0){
        kfree(image);
	return(NULL);
     }

    /*
     * get pixsizx and pixsizy
     */
     if(kread_float(fid,&(image->pixsizx),2)<0){
        kfree(image);
	return(NULL);
     }

    /*
     * get bunch of stuff
     */
     if(kread_ulong(fid,&(image->location_type),16)<0){
        kfree(image);
	return(NULL);
     }

    /*
     * read the two float spares
     */
     if(kread_float(fid,&(image->fspare1),2)<0){
        kfree(image);
	return(NULL);
     }

    /*
     * read the reserve area
     */
     if (klseek(fid, VIFF_HEADERSIZE, 0) == -1){
        kfree(image);
	return(NULL);
     }

    /*
     * the reserve array is not read in....never was...never will be...
     *
     * refer to the comment about tradition above for an explanation
     * of this wierdness.  Oh yeah...thanks for your support.
     */     
     if (image->machine_dep == VFF_DEP_CRAYORDER){
        (void)xvimage2khoros_mach(VFF_DEP_CRAYORDER,&tmp);
        kfile_setmachtype(fid, tmp);
     }

     return(image);
}

/************************************************************
*
*  Routine Name: xvimage_read - read an xvimage structure from the
*				specified filename
*
*       Purpose: This routines reads an xvimage structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the xvimage
*
*        Output: none
*
*       Returns: returns the newly read xvimage or NULL upon failure
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson and Jeremy Worley
*          Date: Sep 15, 1992 19:26
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

xvimage *xvimage_read(
   char *filename)
{
    xvimage	*image;
    int         file, imgsize, mapsize, locsize, imgcnt, mapcnt, loccnt;

    /* 
    ** if file cannot be found, return NULL indicating failure.
    */    
    if((file = kopen(filename,O_RDONLY,0)) < 0){
        return(NULL);
    }

    /* 
     * lock the file for reading...
     */
    kflock(file,KLOCK_SH);
    
    /* 
     * Read the image header 
     */
    if((image = xvimage_readheader(file)) == NULL){
       close_image(file);
       return(NULL);
    }

    /* 
     * Get size of image components in bytes 
     */
    if(!xvimage_size(image,&imgsize,&imgcnt,&mapsize,&mapcnt,&locsize,&loccnt)){
       kfree(image);
       close_image(file);
       return(NULL);
    }

    /* 
     * Read  maps.  NOTE:  I am assuming that xvimage_size returns the number
     * of bytes of data for the *current* architecture.  If this is not
     * not the case, kread_<type>() will not work correctly! 
     */
    if(mapsize!=0){
       if((image->maps =(char*)kmalloc((unsigned)mapsize*sizeof(char)))==NULL){
          xvimage_free(image);
          close_image(file);
          return(NULL);
       }

       if(image->data_encode_scheme == VFF_DES_RAW) {
          if(!read_mapdata(file,(kaddr)image->maps,
			   (int)(image->map_row_size * image->map_col_size),
			   (int)image->map_storage_type)){
             kfree(image->maps);
             kfree(image);
             return(NULL);
	  }
       }else{
	     errno = KLIMITATION;
              xvimage_free(image);
              close_image(file);
              return(NULL);
       } /* fi */

    } /* end if maptype!=0 */

    /* 
     * Read  locations 
     */
    if(locsize!=0){
       if((image->location = (float *)kmalloc(locsize*sizeof(char)))==NULL){
           xvimage_free(image);
           close_image(file);
           return(NULL);
       }

       if(image->data_encode_scheme == VFF_DES_RAW) {
         if(!kread_float(file,image->location,
			 (int)(image->row_size * image->col_size * 
			 image->location_dim))){
             kfree(image->location);
             kfree(image->maps);
             kfree(image);
             return(NULL);
	  }
       } else {
	       errno = KINVALID_INPUT;
               xvimage_free(image);
               close_image(file);
               return(NULL);
       } /* fi */
    } /* end if locsize!=0 */

    /* 
     * Read  image data 
     */
    if(imgsize!=0){
       if((image->imagedata = (char *)kmalloc((unsigned)imgsize*sizeof(char)))
          ==NULL){
          xvimage_free(image);
          close_image(file);
          return(NULL);
       }

       if(image->data_encode_scheme == VFF_DES_RAW) {
         if(!read_imagedata(file,(kaddr)image->imagedata,
			    (int)(image->row_size * image->col_size * 
			    image->num_data_bands * image->num_of_images),
			    (int)image->data_storage_type)){
             kfree(image->location);
             kfree(image->maps);
             kfree(image->imagedata);
             kfree(image);
             return(NULL);
	  }
       } else {
	     errno = KLIMITATION;
             xvimage_free(image);
             close_image(file);
             return(NULL);
       } /* fi */
    } /* end if imgsize!=0 */

  close_image(file);
  image->machine_dep = kmach_type(NULL);

  /* 
   * Return the whole mess to the caller 
   */
  return(image);
}

/************************************************************
*
*  Routine Name: xvimage_free - frees an xvimage structure and it's
*				associated data
*
*       Purpose: This routine frees an khoros xvimage structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros xvimage structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson
*          Date: Sep 15, 1992 19:45
*      Verified:
*  Side Effects: Once this routine is called no further reference
*		 the image should be made.
* Modifications:
*
*************************************************************/

int xvimage_free(
   xvimage *image)
{
	unsigned char id1,id2;

	/*
	 *  Free as much of the xvimage structure as we can. But first check to
	 *  make sure the image pointer is not NULL.
	 */
	if (image != NULL)
	{
	   /* 
            * Now see of the image itself is legal. This catches accidental
	    * attempts to free an image already turned loose by freeimage(). 
            */
            id1 = (unsigned char) image->identifier;
	    id2 = XV_FILE_MAGIC_NUM;
	    if(id1 != id2){
	       errno = KINVALID_INPUT;
	       return(FALSE);
	     }

	   /*  free image data */
	   if (image->imagedata != NULL && (image->row_size * 
	       image->col_size) > 0)
	      kfree(image->imagedata);

	   /*  free map data */
	   if (image->maps != NULL && image->map_row_size > 0)
	      kfree(image->maps);

	   /*  free location data */
	   if (image->location != NULL && image->row_size *
               image->col_size > 0 &&
               image->location_type == VFF_LOC_EXPLICIT)
	      kfree(image->location);

	   /*
	    *  Get rid of the image structure itself.  We know it is not
	    *  NULL, since we checked it above. BUT: before we do this,
            *  fill the header with zeros so that should an unspecting
            *  programmer try to use an already free'd image it will not
            *  just appear that all is well!
	    */
           memset((char *)image,0,sizeof(xvimage));
	   kfree(image);
	}
	else
	   return(FALSE);

	return(TRUE);
}

/**************************************************************
*
* MODULE NAME: xvimage_create
*
*     PURPOSE: Create a generic image
*
*       INPUT: 	col_size -- the size of a column
*		row_size -- the size of a row
*		data_storage_type -- the VFF_TYP_* define of image
*		num_of_images -- # of images pointed to by imagedata
*		num_data_bands -- # of bands/pixel, /image or dim vec data
*		comment -- description of image
*		map_row_size -- # of columns in map array
*		map_col_size -- # of rows in map array
*		map_storage_type -- Storage type of cells in the maps
*		location_type -- implied or explicit location data
*		location_dim -- explicit locations can be of any dimension
*
*      OUTPUT: 	1.  returns a pointer to an xvimage with image defined
*
* CALLED FROM: 
*
* ROUTINES CALLED: 
*
**************************************************************/

xvimage *xvimage_create(
   unsigned long col_size,
   unsigned long row_size,
   unsigned long data_storage_type,
   unsigned long num_of_images,
   unsigned long num_data_bands,
   char          *comment,
   unsigned long map_row_size,
   unsigned long map_col_size,
   unsigned long map_scheme,
   unsigned long map_storage_type,
   unsigned long location_type,
   unsigned long location_dim)
{
    float   *location;
    xvimage *image;
    char    *maps, *imagedata, tmp_comment[512];

    int	cstrlen,
	image_data_size_bytes,		/* # data bytes */
	image_data_count_pixels,	/* # data pixels */
	map_size_bytes,			/* # map bytes */
	map_count_cells,		/* # map cells */
	location_size_bytes,		/* # location bytes */
	location_count_objects;		/* # location objs */


    /* malloc room for the xvimage structure */
    if ((image=(xvimage *)kcalloc(1,sizeof(xvimage)))==NULL)
    {
        return(NULL);
    }

    /* setup the comment (can only be 511 chars) */
    cstrlen = kstrlen(comment);
    if (cstrlen > 0)
    {
       if (cstrlen < 512)
          (void) kstrcpy(tmp_comment, comment);
       else
       {
          (void) kstrncpy(tmp_comment, comment, 512 - 1);
          (void) kstrcat(tmp_comment, "");
       }
    }
    else
       (void) kstrcpy(tmp_comment, "");

    /*
     *  Load the image header with the values. These can be over-ridden by
     *  giving them a different value after returning to the calling routine.
     */
    image->identifier = (char)XV_FILE_MAGIC_NUM;
    image->file_type = XV_FILE_TYPE_XVIFF;
    image->release = XV_IMAGE_REL_NUM;
    image->version = XV_IMAGE_VER_NUM;
    khoros2xvimage_mach(kmach_type(NULL),&(image->machine_dep));
    (void) kstrcpy(image->comment, tmp_comment);
    image->row_size = row_size;
    image->col_size = col_size;
    image->startx = VFF_NOTSUB;
    image->starty = VFF_NOTSUB;
    image->pixsizx = 1.0;
    image->pixsizy = 1.0;
    image->location_type = location_type;
    image->location_dim = location_dim;
    image->num_of_images = num_of_images;
    image->num_data_bands = num_data_bands;
    image->data_storage_type = data_storage_type;
    image->data_encode_scheme = VFF_DES_RAW;
    image->map_scheme = map_scheme;
    image->map_storage_type = map_storage_type;
    image->map_row_size = map_row_size;
    image->map_col_size = map_col_size;
    image->map_subrow_size = 0;
    image->map_enable = VFF_MAP_OPTIONAL;
    image->maps_per_cycle = 0;      /* Don't care */
    image->color_space_model = VFF_CM_NONE;
    image->ispare1 = 0;
    image->ispare2 = 0;
    image->fspare1 = 0;
    image->fspare2 = 0;

    /* get the sizes for the image data, map data, and location data */
    if (!xvimage_size(image, 			/* xvimage */
		    &image_data_size_bytes,	/* # data bytes */
		    &image_data_count_pixels,	/* # data pixels */
		    &map_size_bytes,		/* # map bytes */
		    &map_count_cells,		/* # map cells */
		    &location_size_bytes,	/* # location bytes */
		    &location_count_objects	/* # location objs */
		   ))
    {
	return(NULL);
    }

    /* malloc room for the image data */
    if (image_data_size_bytes > 0)
    {
       if ((imagedata = (char *)
	  kmalloc((unsigned int)image_data_size_bytes)) == NULL)
       {
	  return(NULL);
       }
    }
    else imagedata = NULL;

    /* malloc room for the color map data */
    if (map_size_bytes > 0)
    {
       if ((maps = (char *)kmalloc((unsigned int)map_size_bytes)) == NULL)
       {
	   return(NULL);
       }
    }
    else maps = NULL;


    /* malloc room for the location */
    if (location_size_bytes)
    {
       if ((location = (float *) kmalloc((unsigned int)location_size_bytes)) 
	   == NULL)
           return(NULL);
    }
    else 
       location = NULL;

    /* Load the image data, color map data, and location data */
    image->maps = maps;
    image->location = location;
    image->imagedata = imagedata;
    return(image);
}

/*-----------------------------------------------------------
|
|  Routine Name: khoros2xvimage_csm 
|
|       Purpose: convert khoros colorspace model numbers
|                to xvimage color space model numbers.
|
|         Input: 
|
|	 Output: 
|
|       Returns: 
|    Written By: Jeremy Worley
|          Date: May 12, 1994 13:22
| Modifications:
|
------------------------------------------------------------*/

unsigned long
khoros2xvimage_csm(int kcsm)
{
   switch (kcsm) {
   case KGREY: return(VFF_CM_NONE);
   case KRGB: return(VFF_CM_genericRGB);
   case KCMY: return(VFF_CM_ntscCMY);
   case KYIQ: return(VFF_CM_ntscYIQ);
   case KHSV: return(VFF_CM_HSV);
   case KHLS: return(VFF_CM_HLS);
   case KIHS: return(VFF_CM_IHS);
   case KXYZ: return(VFF_CM_cieXYZ);
   case KUVW: return(VFF_CM_cieUVW);
   case KUCSUVW: return(VFF_CM_cieucsUVW);
   case KUCSSOW: return(VFF_CM_cieucsSOW);
   case KUCSLab: return(VFF_CM_cieucsLab);
   case KUCSLuv: return(VFF_CM_cieucsLuv);
   case KUSERDEFINED: return(VFF_CM_GENERIC);
   default: return(VFF_CM_NONE);
   }
}

/*-----------------------------------------------------------
|
|  Routine Name: xvimage2khros_csm 
|
|       Purpose: convert xvimage colorspace model numbers
|                to khoros color space model numbers.
|
|         Input: 
|
|	 Output: 
|
|       Returns: 
|    Written By: Jeremy Worley
|          Date: May 12, 1994 13:22
| Modifications:
|
------------------------------------------------------------*/

int
xvimage2khoros_csm(unsigned long xcsm)
{
   switch (xcsm) {
   case VFF_CM_NONE:  return(KNONE);
   case VFF_CM_cieRGB:
   case VFF_CM_ntscRGB: 
   case VFF_CM_genericRGB: return(KRGB);
   case VFF_CM_ntscCMY: return(KCMY);
   case VFF_CM_ntscYIQ: return(KYIQ);
   case VFF_CM_HSV: return(KHSV);
   case VFF_CM_HLS: return(KHLS);
   case VFF_CM_IHS: return(KIHS);
   case VFF_CM_cieXYZ: return(KXYZ);
   case VFF_CM_cieUVW: return(KUVW);
   case VFF_CM_cieucsUVW: return(KUCSUVW);
   case VFF_CM_cieucsSOW: return(KUCSSOW);
   case VFF_CM_cieucsLab: return(KUCSLab);
   case VFF_CM_cieucsLuv: return(KUCSLuv);
   case VFF_CM_GENERIC: return(KUSERDEFINED);
   default: return(KNONE);
   }
}
