 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */
/* See $DATASERV/repos/copyright/Copyright for terms and conditions. */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Sun Rasterfile File Format Utilities
   >>>>
   >>>>	    The "rast" format describes the conversion used for
   >>>>	    the format:
   >>>>
   >>>>			Sun Rasterfile File Format
   >>>>
   >>>>	    The format was orignially developed by the company/persons:
   >>>>
   >>>>			Sun Microsystems
   >>>>	    
   >>>>	    The following routines are used by the conversion routines
   >>>>	    to read and write the rast files.  These routines are not a
   >>>>	    standard, just an a set of routines used by the Khoros routines
   >>>>	    to manipulate Sun Rasterfile File Format.
   >>>>
   >>>>
   >>>>  Private:
   >>>>   Static:
   >>>>   Public:
   >>>>             rast_readheader()
   >>>>             rast_read()
   >>>>             rast_writeheader()
   >>>>             rast_write()
   >>>>             rast_free()
   >>>>             rast_create()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include <kdatafmt/rast.h>
#include <machine/kmachine.h>
#define KDATAFMT "kdatafmt"


/************************************************************
*
*  Routine Name: rast_readheader - reads a rast header structure
*				      from the specified kfile id
*
*       Purpose: This routines reads a rast header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the rast header.
*
*         Input: fid - the kfile id opened previously with kopen()
*
*        Output: none
*
*       Returns: image - explanation
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 10:33 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

rast *rast_readheader(
   int fid)
{
	rast *image;


	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (rast *) kcalloc(1, sizeof(rast))) == NULL)
	   return(NULL);

	/*
	 *  Data is always stored in Sun format so we need to set the
	 *  descriptor machine type to be of type KMACH_SPARC.
	 */
	kfile_setmachtype(fid, KMACH_SPARC);

	if (kread_int(fid, &image->ras_magic, 8) != 8)
	{
	   kfree(image);
	   return(NULL);
	}
	else if (image->ras_magic != RAS_MAGIC)
	{
           errno = KINVALID_INPUT;
	   kfree(image);
	   return(NULL);
	}
	return(image);
}



/************************************************************
*
*  Routine Name: rast_fdread - read a rast structure from the
*				specified file descriptor
*
*       Purpose: This routines reads a rast structure and it's
*		 data from the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: fid - file descriptor in which we will be reading
*			    the rast
*
*        Output: none
*
*       Returns: returns the newly read rast or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Sep 30, 1992 10:33
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

rast *rast_fdread(int fid)
{
   rast *image;
   int mapsize;
   int datasize;

   /*
    *  Read the Sun Rasterfile File Format header
    */
   if ((image = rast_readheader(fid)) == NULL)
      return(NULL);

   /*
    *  Sanity check to make sure that the Rasterfile data type and
    *  maptype are valid or are of a type we know how to handle.
    */
   if (image->ras_type != RT_OLD		&&
       image->ras_type != RT_STANDARD	&&
       image->ras_type != RT_BYTE_ENCODED	&&
       image->ras_type != RT_FORMAT_RGB)
   {
      errno = KINVALID_INPUT;
      kfree(image);
      return(NULL);
   }

   if (image->ras_maptype != RMT_NONE	&&
       image->ras_maptype != RMT_EQUAL_RGB	&&
       image->ras_maptype != RMT_RAW)
   {
      errno = KINVALID_DATATYPE;
      kfree(image);
      return(NULL);
   }

   /*
    *  Now read the data associated with the
    *  Sun Rasterfile File Format header
    */
   if (image->ras_maplength > 0 && image->ras_maptype != RMT_NONE)
      mapsize = image->ras_maplength;
   else
      mapsize = 0;

   /*
    *  Read in the maps
    */
   if (mapsize > 0 && (image->map = kmalloc((unsigned) mapsize)) != NULL)
      kread(fid, image->map, mapsize);
   
   /*
    *  Compute the size of the data segment
    */
   if ((image->ras_length == 0 && image->ras_type == RT_OLD) ||
       image->ras_type == RT_BYTE_ENCODED)
   {
      int width  = image->ras_width;
      int height = image->ras_height;
      int depth  = image->ras_depth/8;
      
      if (image->ras_depth == 1)
      {
	 datasize = (width+7)/8;
	 datasize = (datasize + datasize % 2) * height;
      }
      else
	 datasize = (width * depth + (width*depth) % 2) * height;
      
      if (image->ras_type == RT_OLD) image->ras_length = datasize;
   }
   else datasize = image->ras_length;
   
   if (datasize > 0 && (image->data = kmalloc((unsigned) datasize)) != NULL)
   {
      if (image->ras_type == RT_BYTE_ENCODED)
      {
	 kfile *file;
	 register int ch, cnt;
	 unsigned char *end, *data = (unsigned char *) image->data;
	 
	 file = kfdopen(fid, "r");
	 end = data+datasize;
	 while (data != end)
	 {
	    if ((ch = kfgetc(file)) == 128)
	    {
	       if ((cnt = kfgetc(file)) == 0)
		  *data++ = 128;
	       else
	       {
		  ch = kfgetc(file);
		  while(cnt >= 0)
		  {
		     *data++ = (unsigned char)ch;
		     cnt--;
		  }
	       }
	    }
	    else *data++ = (unsigned char)ch;
	 }
      }
      else kread(fid, image->data, datasize);
   }

   return(image);
}

/************************************************************
*
*  Routine Name: rast_read - read a rast structure from the
*				specified filename
*
*       Purpose: This routines reads a rast structure and it's
*		 data from the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote reading of the data and it's header.
*
*         Input: filename - filename in which we will be reading
*			    the rast
*
*        Output: none
*
*       Returns: returns the newly read rast or NULL upon failure
*
*  Restrictions:
*    Written By: Mark Young
*          Date: Sep 30, 1992 10:33
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

rast *rast_read(
   char *filename)
{
   rast *image;
   int fid;

   /* 
    * Open the input file 
    */
   if ((fid = kopen(filename, O_RDONLY, 0664)) < 0)
      return(NULL);

   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_SH);

   image = rast_fdread(fid);
   
   kclose(fid);
   
   return(image);
}

/************************************************************
*
*  Routine Name: rast_fdwrite - write a rast structure to the
*				 specified file descriptor
*
*       Purpose: This routines writes a rast structure and it's
*		 data into the specified file descriptor.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: fid - the file descriptor in which we will be writing
*			    the rast image and associated data
*                image    - the rast structure to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Sep 30, 1992 10:33 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int rast_fdwrite(int fid, rast *image)
{
   int mapsize, datasize;

   /*
    *  Sanity check to make sure that the Rasterfile data type and
    *  maptype are valid or are of a type we know how to handle.
    */
   if (image->ras_type != RT_OLD		&&
       image->ras_type != RT_STANDARD	&&
       image->ras_type != RT_BYTE_ENCODED	&&
       image->ras_type != RT_FORMAT_RGB)
   {
      errno = KINVALID_INPUT;
      kfree(image);
      return(FALSE);
   }
   
   if (image->ras_maptype != RMT_NONE	&&
       image->ras_maptype != RMT_EQUAL_RGB	&&
       image->ras_maptype != RMT_RAW)
   {
      errno = KINVALID_DATATYPE;
      kfree(image);
      return(FALSE);
   }

   /*
    *  Write the Sun Rasterfile File Format header
    */
   if (!rast_writeheader(fid, image))
      return(FALSE);

   /*
    *  Now write the data associated with the Sun Rasterfile File Format
    *  header
    */
   if (image->ras_maplength > 0 && image->ras_maptype != RMT_NONE)
      mapsize = image->ras_maplength;
   else
      mapsize = 0;

   /*
    *  Write in the maps
    */
   if (mapsize > 0) kwrite(fid, image->map, mapsize) ;
   
   /*
    *  Compute the size of the data segment
    */
   if ((image->ras_length == 0 && image->ras_type == RT_OLD) ||
       image->ras_type == RT_BYTE_ENCODED)
   {
      int width  = image->ras_width;
      int height = image->ras_height;
      int depth  = image->ras_depth/8;
      
      if (image->ras_depth == 1)
      {
	 datasize = (width+7)/8;
	 datasize = (datasize + datasize % 2) * height;
      }
      else
	 datasize = (width * depth + (width*depth) % 2) * height;
      
      if (image->ras_type == RT_OLD) image->ras_length = datasize;
   }
   else datasize = image->ras_length;
   
   if (datasize > 0)
   {
      if (image->ras_type == RT_BYTE_ENCODED)
      {
	 kfile *file;
	 register int ch, cnt = 0;
	 unsigned char *end, *data = (unsigned char *) image->data;
	 
	 file = kfdopen(fid, "w");
	 end = data+datasize; ch = (int) *data++;
	 while (data < end)
	 {
	    if (ch == (int) *data && cnt < 256)
	    {
	       cnt++;
	       data++;
	    }
	    else
	    {
	       if (cnt > 0 || ch == 128)
	       {
		  kfputc(128, file);
		  kfputc(cnt, file);
		  if (cnt != 0) kfputc(ch, file);
		  cnt = 0;
	       }
	       else
		  kfputc(ch, file);
	       
	       ch = (int) *data++;
	    }
	 }
      }
      else kwrite(fid, image->data, datasize);
   }

   return(TRUE);
}

/************************************************************
*
*  Routine Name: rast_write - write a rast structure to the
*				 specified filen
*
*       Purpose: This routines writes a rast structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the data.
*
*         Input: filename - the filename in which we will be writing
*			    the rast image and associated data
*                image    - the rast structure to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Mark Young 
*          Date: Sep 30, 1992 10:33 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int rast_write(
   char *filename,
   rast *image)
{
   int fid;
   int status;
   
   /* 
    * Sanity check to make sure that the image pointer is not NULL
    */
   if (image == NULL)
   {
      errno =  KNULL_PARAMETER;
      return(FALSE);
   }
   
   /* 
    * Open the output file 
    */
   if ((fid = kopen(filename, O_WRONLY|O_TRUNC|O_CREAT, 0664)) < 0)
      return(FALSE);
   
   /*
    *  Lock the transport file descriptor if the transport has
    *  data permanence.
    */
   if (kfile_getpermanence(filename) == TRUE)
      kflock(fid, KLOCK_EX);

   status = rast_fdwrite(fid, image);
   
   kclose(fid);
   
   return status;
}



/************************************************************
*
*  Routine Name: rast_writeheader - writes a rast header structure
*				      from the specified kfile id
*
*       Purpose: This routines writes a rast header structure and it's
*		 data into the specified filename.  The routine
*		 uses the ktransport library which allows for
*		 remote writing of the rast header.
*
*         Input: fid - the kfile id opened previously with kopen()
*		 image - the image header to be written
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 10:33 
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int rast_writeheader(
   int  fid,
   rast *image)
{


	/*
	 *  Data is always stored in Sun format so we need to set the
	 *  descriptor machine type to be of type KMACH_SPARC.
	 */
	kfile_setmachtype(fid, KMACH_SPARC);

	if (kwrite_int(fid, &image->ras_magic, 8) != 8)
	   return(FALSE);

	return(TRUE);
}



/************************************************************
*
*  Routine Name: rast_free - frees a rast structure and it's
*				associated data
*
*       Purpose: This routine frees an khoros rast structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros rast structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 10:33
*      Verified:
*  Side Effects: Once this routine is called no further reference
*		 the image should be made.
* Modifications:
*
*************************************************************/

void rast_free(
   rast *image)
{
	/*
	 *  Free the data and all associated resources associated with
	 *  the image.
	 */
	/* add code here please to do just that */

	/*
	 *  Free the rast image structure itself, which was previously
	 */
	kfree(image->data);
	kfree(image->map);
	kfree(image);
}



/************************************************************
*
*  Routine Name: rast_create - creates a rast structure and it's
*				associated data
*
*       Purpose: This routine creates an khoros rast structure and
*		 it's associated data.
*
*         Input: image - a pointer to an khoros rast structure that
*			 contains the image structure to be freed.
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Mark Young 
*          Date: Sep 30, 1992 10:33
*      Verified:
*  Side Effects: 
* Modifications:
*
*************************************************************/

rast *rast_create(void)
{
	rast *image;

	/* 
	 * Sanity check to make sure that the image pointer is not NULL
	 */
	if ((image = (rast *) kcalloc(1, sizeof(rast))) == NULL)
	   return(NULL);

	image->ras_magic = RAS_MAGIC;
	image->ras_type  = RT_OLD;
	return(image);
}

