 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Polymophic Model Data Handlers
   >>>>
   >>>>   Static:
   >>>>			_get_type()
   >>>>
   >>>>                 _size_get()
   >>>>                 _size_set()
   >>>>                 _size_match()
   >>>>                 _size_copy()
   >>>>                 _size_query()
   >>>>                 _size_print() 
   >>>>             
   >>>>                 _type_get()
   >>>>                 _type_set()
   >>>>                 _type_match()
   >>>>                 _type_copy()
   >>>>                 _type_query()
   >>>>                 _type_print() 
   >>>>
   >>>>			_create_mask()
   >>>>			_get_mask_data()
   >>>>			_put_mask_data()
   >>>>  Private:
   >>>>			kpds_init_mask_segment()
   >>>>   Public:
   >>>>			kpds_create_mask()
   >>>>			kpds_query_mask()
   >>>>			kpds_destroy_mask()
   >>>>			kpds_create_mask()
   >>>>			kpds_copy_mask_data()
   >>>>			kpds_copy_mask_attr()
   >>>>			kpds_copy_mask()  
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/* -- pds_attrib.c -- */
extern int sync_sizes;

/* 
 *    ================================================================== 
 *    MASK utilities
 *    ==================================================================
 */

/*-----------------------------------------------------------
|  Routine Name: (static) _get_type
|       Purpose: retrieve the mask type
|       Returns: the mask type
|    Written By: Steve Kubica
------------------------------------------------------------*/
static 
int _get_type(kobject obj)
{
   int type;
   
   if (!kdms_get_attribute(obj, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, &type))
      return FALSE;
   else
      return type;
}

/* 
 *    ================================================================== 
 *    MASK attribute handlers
 *    ==================================================================
 */

/*********** ---------------------------------------------------------------
 ***********  KPDS_MASK_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _size_get
|       Purpose: get the mask size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *w = kva_arg(*list, int *);
   int *h = kva_arg(*list, int *);
   int *d = kva_arg(*list, int *);
   int *t = kva_arg(*list, int *);
   int *e = kva_arg(*list, int *);
   int *size;


   if (!kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE; /* error -- must create mask first */

   if (!kdms_get_attribute(obj, KPDS_SEGMENT_MASK, KDMS_SIZE, &size))
      return FALSE; 

   if (w) *w = size[KWID];
   if (h) *h = size[KHGT];
   if (d) *d = size[KDEP];
   if (t) *t = size[KTIM];

   /* -- if there is a map, then map width is element size -- */
   if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP))
   {
      int map_wid;

      if (!kpds_get_attribute(obj, KPDS_MAP_SIZE, 
			      &map_wid, NULL, NULL, NULL, NULL))
	 return FALSE;

      if (e) *e = size[KELM] * map_wid;
   }
   else
      if (e) *e = size[KELM];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_set
|       Purpose: set the mask size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
	  kva_list *list)
{
   int  w = kva_arg(*list, int);
   int  h = kva_arg(*list, int);
   int  d = kva_arg(*list, int);
   int  t = kva_arg(*list, int);
   int  e = kva_arg(*list, int);
   int  size[KPDS_MASK_DIM] = {-1, -1, -1, -1, -1};

   if (!kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE; /* error -- must create mask first */

   size[KWID] = w;
   size[KHGT] = h;
   size[KDEP] = d;
   size[KTIM] = t;
   size[KELM] = e;

   if (sync_sizes)
   {
      sync_sizes = FALSE;  /* -- avoid calling here again -- */

      /* -- set the shared size of the value segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
	 if (!kpds_set_attribute(obj, KPDS_VALUE_SIZE, w, h, d, t, e))
	    return FALSE;
   
      /* -- set the shared size of the location segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_LOCATION))
	 if (!kpds_set_attribute(obj, KPDS_LOCATION_SIZE, w, h, d, -1))
	    return FALSE;
      
      /* -- set the shared size of the time segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_TIME))
	 if (!kpds_set_attribute(obj, KPDS_TIME_SIZE, t))
	    return FALSE;

      /* -- set the shared size of the map segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_MAP))
      {
	 int *osize;

	 /* -- track size of value depth, time, and element only if previous
	  *    map size is not 1 --  */
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_MAP, KDMS_SIZE, &osize))
	    return FALSE;

	 if (osize[KMAP_DEP]!=1 || osize[KMAP_TIM]!=1 || osize[KMAP_ELM]!=1)
	 {
	    int msize[KPDS_MAP_DIM] = {-1, -1, -1, -1, -1};
	 
	    msize[KMAP_DEP] = (osize[KMAP_DEP] == 1) ? -1 : d;
	    msize[KMAP_TIM] = (osize[KMAP_TIM] == 1) ? -1 : t;
	    msize[KMAP_ELM] = (osize[KMAP_ELM] == 1) ? -1 : e;
	 
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_MAP, KDMS_SIZE, msize))
	       return FALSE;
	 }
      }

      /* -- if mapping mode is on, then map width is affected by esize -- */
      if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP) 
	  && e != -1)  /* e of -1 implies that the esize should not change */
      {
	 int old_e;
      
	 if (!kpds_get_attribute(obj, KPDS_MASK_SIZE, 
				 NULL, NULL, NULL, NULL, &old_e))
	    return FALSE;
      
	 /* only worry about changing the map size if the esize is different */
	 old_e = kmax(old_e, 1);   /* -- don't want esize = 0 -- */
	 if (old_e != e)
	 {
	    if (!kpds_get_attribute(obj, KPDS_MAP_SIZE, 
				    &size[KELM], NULL, NULL, NULL, NULL))
	       return FALSE;
	 
	    /*
	     * if the mask size is mapped, then the current KELEMENTS
	     * size of the mask segment will be divided into the new
	     * KELEMENTS size.  This division will result in the new
	     * size of the KMAP_WIDTH.  The existing KELEMENTS size
	     * will be untouched. 
	     */
	    if (!kpds_set_attribute(obj, KPDS_MAP_SIZE, 
				    size[KELM]/old_e, -1, -1, -1, -1))
	       return FALSE;
	    size[KELM] = old_e;
	 }
      }
      sync_sizes = TRUE;
   }
   
   /* -- set the size of the mask segment -- */
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_MASK, KDMS_SIZE, size))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_match
|       Purpose: match the mask size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int w1, h1, d1, t1, e1;
   int w2, h2, d2, t2, e2;


   if (!kpds_get_attribute(obj1, KPDS_MASK_SIZE, &w1, &h1, &d1, &t1, &e1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_MASK_SIZE, &w2, &h2, &d2, &t2, &e2))
      return FALSE;

   return ((w1 == w2) && (h1 == h2) && (d1 == d2) && (t1 == t2) && (e1 == e2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_copy
|       Purpose: copy the mask size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int w, h, d, t, e;


   if (!kpds_get_attribute(obj1, KPDS_MASK_SIZE, &w, &h, &d, &t, &e))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_MASK_SIZE, w, h, d, t, e))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_query
|       Purpose: query the mask size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE;		/* not an error */

   if (num_args)
      *num_args = 5;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_print
|       Purpose: print the mask size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int w, h, d, t, e;


   if (!kpds_get_attribute(obj, KPDS_MASK_SIZE, &w, &h, &d, &t, &e))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%d %d %d %d %d", w, h, d, t, e);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_MASK_DATA_TYPE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _type_get
|       Purpose: get the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *rtyp = kva_arg(*list, int *);
   int  typ;


   if (!kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE;

   if (!kdms_get_attribute(obj, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, &typ))
      return FALSE;
   
   if (rtyp) *rtyp = typ;
   
   return TRUE;
}


/*-----------------------------------------------------------
|  Routine Name: (static) _type_set
|       Purpose: set the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int typ = kva_arg(*list, int);

   if (!kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE;

   if (!kdms_set_attribute(obj, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, typ))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_match
|       Purpose: match the type attribute
|    Written By: Steve Kubica and John Salas
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int typ1;
   int typ2;

   if (!kdms_query_segment(obj1, KPDS_SEGMENT_MASK) ||
       !kdms_query_segment(obj2, KPDS_SEGMENT_MASK))
      return FALSE;

   if (!kdms_get_attribute(obj1, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, &typ1))
      return FALSE;

   if (!kdms_get_attribute(obj2, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, &typ2))
      return FALSE;
   
   return (typ1 == typ2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_copy
|       Purpose: copy the type attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:43
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int typ;

   if (!kdms_query_segment(obj1, KPDS_SEGMENT_MASK) ||
       !kdms_query_segment(obj2, KPDS_SEGMENT_MASK))
      return FALSE;

   if (!kdms_get_attribute(obj1, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, &typ))
      return FALSE;

   if (!kdms_set_attribute(obj2, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, typ))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_query
|       Purpose: query the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_print
|       Purpose: print the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int typ;

   if (!kdms_query_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE;

   if (!kdms_get_attribute(obj, KPDS_SEGMENT_MASK, KDMS_DATA_TYPE, &typ))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%s (%d)", kdefine_to_datatype(typ), typ);

   return TRUE;
}

/* 
 *    ================================================================== 
 *    MASK segment handlers
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _create_mask
|
|       Purpose: This routine will create the mask segment and
|		 initialize its dimensionality and index order.
|
|         Input: obj     - the obj in which to create the mask segment
|		 segment - the name of the mask segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_create_mask(
   kobject  obj,
   char    *segment)
{
   int  order[KPDS_MASK_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};
   int *vorder = NULL;
   int  oldmode;
   int  w = -1;
   int 	h = -1;
   int 	d = -1;
   int 	t = -1;
   int 	e = -1;


   if (!kdms_create_segment(obj, KPDS_SEGMENT_MASK))
      return FALSE;
   
   if (!kdms_set_attributes(obj, KPDS_SEGMENT_MASK, 
			         KDMS_DIMENSION, KPDS_MASK_DIM,
			    	 KDMS_INDEX_ORDER, order, 
			    	 KDMS_PAD_VALUE, 1.0, 0.0,
			    	 NULL))
      return FALSE;

   /* -- don't want esize to be mapped -- */
   kpds_get_attribute(obj, KPDS_MAPPING_MODE, &oldmode);
   kpds_set_attribute(obj, KPDS_MAPPING_MODE, KUNMAPPED);
      
   /* -- get initial size and index order from existing segments -- */
   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      kpds_get_attribute(obj, KPDS_VALUE_SIZE, &w, &h, &d, &t, &e);

      kdms_get_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_INDEX_ORDER, &vorder);
      kdms_set_attribute(obj, KPDS_SEGMENT_MASK,  KDMS_INDEX_ORDER, vorder);
   }
   else if (kdms_query_segment(obj, KPDS_SEGMENT_LOCATION))
   {
      kpds_get_attribute(obj, KPDS_LOCATION_SIZE, &w, &h, &d, NULL);

      kdms_get_attribute(obj, KPDS_SEGMENT_LOCATION, KDMS_INDEX_ORDER,&vorder);
      order[0] = vorder[0];
      order[1] = vorder[1];
      order[2] = vorder[2];
      order[3] = KTIME;
      order[4] = KELEMENTS;
      kdms_set_attribute(obj, KPDS_SEGMENT_MASK, KDMS_INDEX_ORDER, order);
   }

   else if (kdms_query_segment(obj, KPDS_SEGMENT_TIME))
      kpds_get_attribute(obj, KPDS_TIME_SIZE, &t);

   else if (kdms_query_segment(obj, KPDS_SEGMENT_MAP))
      kpds_get_attribute(obj, KPDS_MAP_SIZE, NULL, NULL, &d, &t, &e);
   

   /* -- set initial size of the mask segment -- */
   kpds_set_attribute(obj, KPDS_MASK_SIZE, w, h, d, t, e);

   /* -- restore mapping mode -- */
   kpds_set_attribute(obj, KPDS_MAPPING_MODE, oldmode);

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _get_mask_data
|
|       Purpose: This routine gets data from the mask segment.
|                It is "map-aware" in that it understands the 
|		 mapping mode and will alter the mask data
|		 size appropirately.
|
|         Input: object  - the object to retrieve data from
|	         segment - segment to retrieve data from
|		 begin   - the begin point for region to get :
|			   if the data is to be mapped, this
|		           begin point represents the mapped begin
|		 end     - the end point for the region to get :
|			   if the data is to be mapped, this
|		           end point represents the mapped end
|
|        Output: data   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|
|       Returns: pointer to the data being returned on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static kaddr
_get_mask_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   kaddr rdata;

   /* -- verify that mask segment exists -- */
   if (!kdms_query_segment(obj, segment))
      return NULL;
   
   if (kaps_mapping_mode(obj))
      rdata = kaps_get_mapped_mask_data(obj, begin, end, data);
   else
      rdata = kdms_get_data(obj, segment, begin, end, data);

   if (kaps_elements_first(obj))
   {
      kaddr ldata;
      
      ldata = kaps_transpose_elements_first(rdata, begin, end, _get_type(obj),
					    KPDS_MASK_DIM);
      return ldata;
   }
   
   return rdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _put_mask_data
|
|       Purpose: This routine puts data into the mask segment.
|                It is "map-aware" in that it understands the 
|		 mapping mode. It actually ignores the mapping 
|		 mode, since it is meaningless for a put,
|		 but it know enough to give debug warnings.
|
|         Input: obj - the object to put data into
|	         segment - the segment to put the data into
|		 begin   - the begin point for the put
|		 end     - the end point for the put
|		 data    - the data to be put
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static int
_put_mask_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   kaddr pdata = data;
   
   /* -- verify that the mask segment exists -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   /* -- transpose if elements are first -- */
   if (kaps_elements_first(obj))
      pdata = kaps_transpose_elements_last(data, begin, end, _get_type(obj),
					   KPDS_MASK_DIM);

   return kdms_put_data(obj, segment, begin, end, pdata);
}


/* 
 *    ================================================================== 
 *    PRIVATE MASK functions
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kpds_init_mask_segment()
|
|       Purpose: Define the mask segment.
|
|         Input: none
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 26, 1994
| Modifications:
|
------------------------------------------------------------*/
int
kpds_init_mask_segment(void)
{
   int status = TRUE;

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_MASK_SIZE, NULL,
					 _size_get,   _size_set, 
					 _size_match, _size_copy, 
					 _size_query, _size_print);

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_MASK_DATA_TYPE, NULL,
					 _type_get,   _type_set, 
					 _type_match, _type_copy,
					 _type_query, _type_print);

   status &= kdms_define_segment(KPDS_SEGMENT_MASK, _create_mask, NULL, NULL, 
				 _get_mask_data, _put_mask_data,
				 NULL, NULL, NULL, NULL, NULL, NULL);

   return status;
}

/* 
 *    ================================================================== 
 *    PUBLIC MASK functions
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kpds_create_mask - create a mask segment within a 
*				    data object.
*
*       Purpose: This function is used to create a mask segment
*		 within a specified data object.  The size of the
*		 mask segment will be initialized to match any
*		 the sizes shared with any other polymorphic segments.
*
*		 Note that the pad value for this segment is
*		 initalized to 1.  This is done so that padded data is
*		 marked as valid according to the mask segment.
*
*		 It is considered an error to create a mask segment
*		 if the object already contains one.
*
*         Input: object  - object in which to create the mask segment.
*
*        Output: none
*
*       Returns: TRUE (1) if the mask segment was successfully created, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_create_mask(
*		!   kobject object)
*
*************************************************************/
int
kpds_create_mask(kobject object)
{
   return (kpds_init() && kdms_create_segment(object, KPDS_SEGMENT_MASK));
}

/************************************************************
*
*  Routine Name: kpds_query_mask - determine if the mask segment
*				   exists in a data object.
*
*       Purpose: This function is used to determine if the mask
*		 segment exists in a data object.  If mask segment
*		 exists in the specified object, then this function
*		 will return TRUE.  If the object is invalid, or mask
*		 data does not exist in the object, then this function
*		 will return FALSE.
*
*         Input: object  - data object to be queried.
*
*        Output: none
*
*       Returns: TRUE (1) if the mask segment exists, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_query_mask(
*		!   kobject object)
*
*************************************************************/
int
kpds_query_mask(kobject object)
{
   return (kpds_init() && kdms_query_segment(object, KPDS_SEGMENT_MASK));
}

/************************************************************
*
*  Routine Name: kpds_destroy_mask - destroy the mask segment in a
*				     data object.
*
*       Purpose: This function is used to destroy the mask segment
*		 contained within an object.  Once the mask segment
*		 has been destroyed, any data or attributes associated
*		 with the mask data will be lost forever.  A new
*		 mask segment can be created in its place with the
*		 function kpds_create_mask.
*
*		 If the mask segment does not exist in the specified
*		 object, it is considered to be an error.
*
*         Input: object  - object from which to remove the mask segment.
*
*        Output: mask segment is destroyed
*
*       Returns: TRUE (1) if the mask segment is successfully destroyed, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_destroy_mask(
*		!   kobject object)
*
*************************************************************/
int
kpds_destroy_mask(kobject object)
{
   return (kpds_init() && kdms_destroy_segment(object, KPDS_SEGMENT_MASK));
}

/************************************************************
*
*  Routine Name: kpds_initialize_mask - preset mask segment to constant
*
*       Purpose: This routine initializes the entire mask segment to a
*		 specified mask.  The last argument is a real number
*		 that the mask data will be initialized to.  The data
*		 type of the mask segment at the time this function is
*		 called will determine how the number is used.
*
*         Input: object - object to initialize mask data on
*		 number - constant number to intialize to.
*
*        Output: mask segment is modified in place
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Aug 03, 1994 14:53
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_initialize_mask(
*                !   kobject object,
*                !   double number)
*
*************************************************************/
int
kpds_initialize_mask(
   kobject object, 
   double number)
{
   int  status = TRUE;
   int  oldmode;

   if (!kpds_init())
      return FALSE;

   /* -- don't want esize to be mapped -- */
   status &= kpds_get_attribute(object, KPDS_MAPPING_MODE, &oldmode);
   status &= kpds_set_attribute(object, KPDS_MAPPING_MODE, KUNMAPPED);
   
   status &= kdut_initialize_segment(object, KPDS_SEGMENT_MASK, number, 0.0);

   /* -- restore mapping mode -- */
   status &= kpds_set_attribute(object, KPDS_MAPPING_MODE, oldmode);

   return status;
}

/************************************************************
*
*  Routine Name: kpds_copy_mask_data - copy all mask data from
*                                      one object to another object.
*
*       Purpose: This function copies all of the data contained in the
*		 mask segment of one object into the mask segment
*		 contained in another object.  If the mask segment
*		 exists in the destination object, then its data will
*		 be replaced with the data from the source object.  If
*		 the mask segment does not exist in the destination
*		 object, it will be created.
*
*		 The mask data can be optionally copied through the
*		 presentations of the two data objects.  This implies
*		 that any presentation stages which are normally
*		 invoked during a kpds_get_data on the source object
*		 or a kpds_put_data call on the destination object
*		 will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : mapping, axis assignment,
*		 position, and offset.  The resulting copy will be set
*		 to the size it appears to be when mapping mode is set
*		 to mapped, and will be transposed to be in the
*		 default axis ordering.  The resulting copy will also
*		 appear to be shifted by the source position and
*		 offset.
*
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has a presentation data
*		 type different from the physical data type, and a
*		 presentation size different from the physical size.
*		 The destination object only has a different
*		 presentation and physical data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size and cast to a
*		 new data type before being copied into the
*		 destination object.  When being copied into the
*		 destination object, the data will be cast yet again
*		 before it finally is stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the presentation.
*		 If the data is not copied through the presentation,
*		 the destination data will be a direct copy of the
*		 physical source data.  The physical size and data
*		 type of the mask segment will be reflected in the
*		 destination data object from the source to the
*		 destination since they describe the physical state of
*		 the data.
*
*         Input: object1 - the object that serves as a source for the data.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Jan 2, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_mask_data(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_mask_data(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   kobject src = NULL;
   kobject dest = object2;
   
   int w, h, d, t, e;
   int ow, oh, od, ot, oe;
   int wps, hps, dps, tps, eps;
   int wof, hof, dof, tof, eof;
   int i;
   int typ;
   int num_regions;
   int old_coupling;
   kaddr data = NULL;
   int order[KPDS_MASK_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS};

   if (!kpds_init())
      return FALSE;

   /* -- sanity check -- */
   if (object1 == NULL || object2 == NULL)
   {
      /* -- set invalid object errorno here -- */
      return FALSE;
   }

   if (!copy_through_presentation)
      return kdut_copy_segment_data(object1, object2, KPDS_SEGMENT_MASK);

   /* -- make sure the segment exists in the source -- */
   if (!kdms_query_segment(object1, KPDS_SEGMENT_MASK))
   {
      /* -- set cool errno here -- */
      return FALSE;
   }

   /* -- make sure the segment exists in the destination -- */
   if (!kdms_query_segment(object2, KPDS_SEGMENT_MASK))
      if (!kdms_create_segment(object2, KPDS_SEGMENT_MASK))
	 return FALSE;

   /* -- get the position and offset before we make the reference -- */
   if (!kpds_get_attributes(object1,
			    KPDS_MASK_POSITION, &wps, &hps, &dps, &tps, &eps,
			    KPDS_MASK_OFFSET,   &wof, &hof, &dof, &tof, &eof,
			    NULL))
      return FALSE;

   /* -- create a reference to avoid side effects on the source -- */
   src  = kdms_reference(object1);

   /* -- get the old coupling on the dest so we can restore it later -- */
   if (!kdms_get_attribute(dest, KPDS_SEGMENT_MASK, 
			         KDMS_COUPLING, &old_coupling))
      return FALSE;
   
   /* -- want to affect physical layer on the destination -- */
   if (!kdms_set_attribute(dest, KPDS_SEGMENT_MASK, KDMS_COUPLING, KCOUPLED))
      return FALSE;

   /* -- the destination should not be mapped -- */
   if (!kpds_set_attribute(dest, KPDS_MAPPING_MODE, KUNMAPPED))
      return FALSE;

   /* -- the resulting copy will have the default index order -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_MASK, KDMS_INDEX_ORDER, order);

   /* -- set up region sizes - also copy size and data type -- */
   if (!kpds_get_attributes(src,
			    KPDS_MASK_SIZE, &w, &h, &d, &t, &e,
			    KPDS_MASK_DATA_TYPE, &typ,
			    KPDS_MASK_OPTIMAL_REGION_SIZE, &ow, &oh,
			    &od, &ot, &oe, &num_regions, 
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(src,
			    KPDS_MASK_POSITION, wps, hps, dps, tps, eps,
			    KPDS_MASK_OFFSET,   wof, hof, dof, tof, eof,
			    KPDS_MASK_REGION_SIZE, ow, oh, od, ot, oe,
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(dest,
			    KPDS_MASK_SIZE, w, h, d, t, e,
			    KPDS_MASK_DATA_TYPE, typ,
			    KPDS_MASK_REGION_SIZE, ow, oh, od, ot, oe,
			    KPDS_MASK_POSITION, 0, 0, 0, 0, 0,
			    KPDS_MASK_OFFSET, 0, 0, 0, 0, 0,
			    NULL))
      return FALSE;

   /* -- copy the data using the pds get'n'put data calls -- */
   for (i = 0; i < num_regions; i++)
   {
      data = kpds_get_data(src, KPDS_MASK_REGION, data);
      kpds_put_data(dest, KPDS_MASK_REGION, data);
   }
   kfree(data);
   
   /* -- reset the position and offset to be zero -- */
   if (!kpds_set_attributes(dest, 
			    KPDS_MASK_POSITION, 0, 0, 0, 0, 0,
			    KPDS_MASK_OFFSET, 0, 0, 0, 0, 0, 
			    NULL))
      return FALSE;

   /* -- set the old coupling back on the segment -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_MASK, KDMS_COUPLING, old_coupling);

   /* -- close the source reference -- */
   kdms_close(src);

   return TRUE;
}

/************************************************************
*
*  Routine Name: kpds_copy_mask_attr - copy all mask attributes from
*	                               one object to another object.
*
*       Purpose: This function copies all the mask attributes from 
*		 one object to another object.
*
*		 If the destination object does not contain a mask
*		 segment, then this function will create a mask
*		 segment, and initialize its size and data type
*		 attributes to those in the source object.  If the
*		 mask data already exists in the destination object,
*		 then the presentation attributes will be set to the
*		 source object's settings.
*
*		 The destination's physical attributes will change
*		 depending on the coupling of the data objects.
*		 If the destination is coupled, then the physical
*		 attributes will be changed as well.  Any data
*		 contained in the destination will be cast, rescaled,
*		 or resized to match its new physical attributes.
*
*         Input: source_object - the object that serves as a
*				 source for the attributes.
*
*        Output: destination_object - the object that serves as
*				      a destination for the
*				      operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 19, 1993 15:43
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_mask_attr(
*		!   kobject object1,
*		!   kobject object2)
*
*************************************************************/
int
kpds_copy_mask_attr(
   kobject object1,
   kobject object2)
{
   return (kpds_init() && 
	   kdut_copy_segment_attr(object1, object2, KPDS_SEGMENT_MASK));
}

/************************************************************
*
*  Routine Name: kpds_copy_mask - copy the mask segment from
*		                  one object to another.
*
*       Purpose: This function copies all of the attributes and data
*		 contained in the mask segment of one object into the
*		 mask segment contained in another object.  If the
*		 mask segment exists in the destination object, then
*		 its data will be replaced with the data from the
*		 source object.  If the mask segment does not exist
*		 in the destination object, it will be created.
*
*		 All mask segment attributes will be copied from one
*		 object to the other.  This includes presentation
*		 attributes, such as position and offset.
*
*		 The mask data can be optionally copied through the
*		 presentations of the two data objects.  This implies
*		 that any presentation stages which are normally
*		 invoked during a kpds_get_data on the source object
*		 or a kpds_put_data call on the destination object
*		 will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : mapping, axis assignment,
*		 position, and offset.  The resulting copy will be
*		 mapped through the map and transposed to be in the
*		 default axis ordering.  The resulting copy will also
*		 appear to be shifted by the source position and
*		 offset.
*
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has a presentation data
*		 type different from the physical data type, and a
*		 presentation size different from the physical size.
*		 The destination object only has a different
*		 presentation and physical data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size and cast to a
*		 new data type before being copied into the
*		 destination object.  When being copied into the
*		 destination object, the data will be cast yet again
*		 before it finally is stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the presentation.
*		 If the data is not copied through the presentation,
*		 the destination data will be a direct copy of the
*		 physical source data.  The physical size and data
*		 type of the mask segment will be reflected in the
*		 destination data object from the source to the
*		 destination since they describe the physical state of
*		 the data.
*
*		 This function is equivalent to performing successive
*		 calls to kpds_copy_mask_attr and kpds_copy_mask_data.
*
*         Input: object1 - the object that serves as a source for the mask
*			   segment.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley & Steve Kubica
*          Date: Sep 19, 1993 15:40
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_mask(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_mask(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   return (kpds_init() && 
	   kpds_copy_mask_attr(object1, object2) &&
	   kpds_copy_mask_data(object1, object2, copy_through_presentation));
}
