 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Polymophic Model Data Handlers
   >>>>
   >>>>   Static:
   >>>>			_query_loc()
   >>>>			_get_type()
   >>>>
   >>>>                 _loc_grid_set()
   >>>>                 _loc_grid_copy()
   >>>>                 _loc_grid_print() 
   >>>>
   >>>>                 _begin_get()
   >>>>                 _begin_set()
   >>>>                 _begin_match()
   >>>>                 _begin_copy()
   >>>>                 _begin_query()
   >>>>                 _begin_print() 
   >>>>	
   >>>>                 _end_get()
   >>>>                 _end_set()
   >>>>                 _end_match()
   >>>>                 _end_copy()
   >>>>                 _end_query()
   >>>>                 _end_print() 
   >>>>
   >>>>                 _size_get()
   >>>>                 _size_set()
   >>>>                 _size_match()
   >>>>                 _size_copy()
   >>>>                 _size_query()
   >>>>                 _size_print() 
   >>>>             
   >>>>                 _type_get()
   >>>>                 _type_set()
   >>>>                 _type_match()
   >>>>                 _type_copy()
   >>>>                 _type_query()
   >>>>                 _type_print() 
   >>>>
   >>>>			_create_location()
   >>>>			_get_location_data()
   >>>>			_put_location_data()
   >>>>			_get_location_atr()
   >>>>  Private:
   >>>>			kpds_init_location_segment()
   >>>>   Public:
   >>>>			kpds_create_location()
   >>>>			kpds_query_location()
   >>>>			kpds_destroy_location()
   >>>>			kpds_create_location()
   >>>>			kpds_copy_location_data()
   >>>>			kpds_copy_location_attr()
   >>>>			kpds_copy_location()  
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/* -- pds_attrib.c -- */
extern int sync_sizes;

/* 
 *    ================================================================== 
 *    LOCATION utilities
 *    ==================================================================
 */

/*-----------------------------------------------------------
|  Routine Name: (static) _query_loc
|       Purpose: determine existence of location data in any grid type 
|       Returns: TRUE if location exists
|    Written By: Steve Kubica
------------------------------------------------------------*/
static 
int _query_loc(kobject obj)
{
   int grid = kaps_location_grid(obj);
   
   switch(grid)
   {
      case KUNIFORM :
	 return kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM);

      case KRECTILINEAR :
	 return kdms_query_segment(obj, KPDS_SEGMENT_WIDTH);
	 
      case KCURVILINEAR : 
	 return kdms_query_segment(obj, KPDS_SEGMENT_LOCATION);
   }

   return FALSE; /* -- not an error ... just KNONE -- */
}

/*-----------------------------------------------------------
|  Routine Name: (static) _get_type
|       Purpose: retrieve the location type
|       Returns: the location type
|    Written By: Steve Kubica
------------------------------------------------------------*/
static 
int _get_type(kobject obj)
{
   int type = KNONE;
   int grid = kaps_location_grid(obj);
   
   switch(grid)
   {
      case KUNIFORM :
      {
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM,
				 KDMS_DATA_TYPE, &type))
	    return FALSE;
      } break;

      case KRECTILINEAR :
      {
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
				 KDMS_DATA_TYPE, &type))
	    return FALSE;
      } break;

      case KCURVILINEAR :
      {
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_LOCATION,
				 KDMS_DATA_TYPE, &type))
	    return FALSE;
      } break;
   }

   return type;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _copy_uniform
|       Purpose: copy uniform location data thru presentation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Steve Kubica
------------------------------------------------------------*/
static 
int _copy_uniform(kobject object1, kobject object2)
{
   kobject src = NULL;
   kobject dest = object2;

   int old_coupling;
   int typ;
   int beg[2] = {0, 0};
   int end[2] = {1, 2};
   
   kaddr data = NULL;


   /* -- create a reference to avoid side effects on the source -- */
   src  = kdms_reference(object1);

   /* -- get the old coupling on the dest so we can restore it later -- */
   if (!kdms_get_attribute(dest, KPDS_SEGMENT_UNIFORM, 
			         KDMS_COUPLING, &old_coupling))
      return FALSE;
   
   /* -- want to affect physical layer on the destination -- */
   if (!kdms_set_attribute(dest, KPDS_SEGMENT_UNIFORM, KDMS_COUPLING, 
			   KCOUPLED))
      return FALSE;

   /* -- copy data type -- */
   if (!kpds_get_attribute(src, KPDS_LOCATION_DATA_TYPE, &typ))
      return FALSE;
   if (!kpds_set_attribute(dest, KPDS_LOCATION_DATA_TYPE, typ))
      return FALSE;

   /* -- copy the data using the pds get'n'put data calls -- */
   data = kdms_get_data(src, KPDS_SEGMENT_UNIFORM, beg, end, NULL);
   kdms_put_data(dest, KPDS_SEGMENT_UNIFORM, beg, end, data);
   kfree(data);
   
   /* -- reset the position and offset to be zero -- */
   if (!kpds_set_attributes(dest, 
			    KPDS_LOCATION_POSITION, 0, 0, 0, 0,
			    KPDS_LOCATION_OFFSET, 0, 0, 0, 0,
			    NULL))
      return FALSE;

   /* -- set the old coupling back on the segment -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_LOCATION, 
		      KDMS_COUPLING, old_coupling);

   /* -- close the source reference -- */
   kdms_close(src);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _copy_rectilinear
|       Purpose: copy rectilinear location data thru presentation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Steve Kubica
------------------------------------------------------------*/
static 
int _copy_rectilinear(kobject object1, kobject object2)
{
   kobject src = NULL;
   kobject dest = object2;

   int w, h, d, n;
   int ow, oh, od, on;
   int wps, hps, dps, nps;
   int wof, hof, dof, nof;
   int i;
   int typ;
   int num_regions;
   int old_coupling;
   int order[KPDS_LOCATION_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};

   kaddr wdata = NULL;
   kaddr hdata = NULL;
   kaddr ddata = NULL;


   /* -- get the position and offset before we make the reference -- */
   if (!kpds_get_attributes(object1,
			    KPDS_LOCATION_POSITION, &wps, &hps, &dps, &nps,
			    KPDS_LOCATION_OFFSET,   &wof, &hof, &dof, &nof,
			    NULL))
      return FALSE;

   /* -- create a reference to avoid side effects on the source -- */
   src  = kdms_reference(object1);

   /* -- get the old coupling on the dest so we can restore it later -- */
   if (!kdms_get_attribute(dest, KPDS_SEGMENT_LOCATION, 
			         KDMS_COUPLING, &old_coupling))
      return FALSE;
   
   /* -- want to affect physical layer on the destination -- */
   if (!kdms_set_attribute(dest, KPDS_SEGMENT_LOCATION, KDMS_COUPLING, 
			   KCOUPLED))
      return FALSE;

   /* -- the resulting copy will have the default index order -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_LOCATION, KDMS_INDEX_ORDER, order);

   /* -- set up region sizes - also copy size and data type -- */
   if (!kpds_get_attributes(src,
			    KPDS_LOCATION_SIZE, &w, &h, &d, &n,
			    KPDS_LOCATION_DATA_TYPE, &typ,
			    KPDS_LOCATION_OPTIMAL_REGION_SIZE, &ow, &oh,
			    &od, &on, &num_regions, 
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(src,
			    KPDS_LOCATION_POSITION, wps, hps, dps, nps,
			    KPDS_LOCATION_OFFSET,   wof, hof, dof, nof,
			    KPDS_LOCATION_REGION_SIZE, ow, oh, od, on, 
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(dest,
			    KPDS_LOCATION_SIZE, w, h, d, n,
			    KPDS_LOCATION_DATA_TYPE, typ,
			    KPDS_LOCATION_REGION_SIZE, ow, oh, od, on,
			    KPDS_LOCATION_POSITION, 0, 0, 0, 0,
			    KPDS_LOCATION_OFFSET, 0, 0, 0, 0,
			    NULL))
      return FALSE;

   /* -- copy the data using the pds get'n'put data calls -- */
   for (i = 0; i < num_regions; i++)
   {
      wdata = kpds_get_data(src, KPDS_LOCATION_WIDTH_REGION, wdata);
      kpds_put_data(dest, KPDS_LOCATION_WIDTH_REGION, wdata);

      hdata = kpds_get_data(src, KPDS_LOCATION_HEIGHT_REGION, hdata);
      kpds_put_data(dest, KPDS_LOCATION_HEIGHT_REGION, hdata);

      ddata = kpds_get_data(src, KPDS_LOCATION_DEPTH_REGION, ddata);
      kpds_put_data(dest, KPDS_LOCATION_DEPTH_REGION, ddata);
   }
   kfree(wdata);
   kfree(hdata);
   kfree(ddata);
   
   /* -- reset the position and offset to be zero -- */
   if (!kpds_set_attributes(dest, 
			    KPDS_LOCATION_POSITION, 0, 0, 0, 0,
			    KPDS_LOCATION_OFFSET, 0, 0, 0, 0,
			    NULL))
      return FALSE;

   /* -- set the old coupling back on the segment -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_LOCATION, 
		      KDMS_COUPLING, old_coupling);

   /* -- close the source reference -- */
   kdms_close(src);

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _copy_curvilinear
|       Purpose: copy curvilinear location data thru presentation
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Steve Kubica
------------------------------------------------------------*/
static 
int _copy_curvilinear(kobject object1, kobject object2)
{
   kobject src = NULL;
   kobject dest = object2;

   int w, h, d, n;
   int ow, oh, od, on;
   int wps, hps, dps, nps;
   int wof, hof, dof, nof;
   int i;
   int typ;
   int num_regions;
   int old_coupling;
   int order[KPDS_LOCATION_DIM] = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};

   kaddr data = NULL;


   /* -- get the position and offset before we make the reference -- */
   if (!kpds_get_attributes(object1,
			    KPDS_LOCATION_POSITION, &wps, &hps, &dps, &nps,
			    KPDS_LOCATION_OFFSET,   &wof, &hof, &dof, &nof,
			    NULL))
      return FALSE;

   /* -- create a reference to avoid side effects on the source -- */
   src  = kdms_reference(object1);

   /* -- get the old coupling on the dest so we can restore it later -- */
   if (!kdms_get_attribute(dest, KPDS_SEGMENT_LOCATION, 
			         KDMS_COUPLING, &old_coupling))
      return FALSE;
   
   /* -- want to affect physical layer on the destination -- */
   if (!kdms_set_attribute(dest, KPDS_SEGMENT_LOCATION, KDMS_COUPLING, 
			   KCOUPLED))
      return FALSE;

   /* -- the resulting copy will have the default index order -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_LOCATION, KDMS_INDEX_ORDER, order);

   /* -- set up region sizes - also copy size and data type -- */
   if (!kpds_get_attributes(src,
			    KPDS_LOCATION_SIZE, &w, &h, &d, &n,
			    KPDS_LOCATION_DATA_TYPE, &typ,
			    KPDS_LOCATION_OPTIMAL_REGION_SIZE, &ow, &oh,
			    &od, &on, &num_regions, 
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(src,
			    KPDS_LOCATION_POSITION, wps, hps, dps, nps,
			    KPDS_LOCATION_OFFSET,   wof, hof, dof, nof,
			    KPDS_LOCATION_REGION_SIZE, ow, oh, od, on, 
			    NULL))
      return FALSE;
   if (!kpds_set_attributes(dest,
			    KPDS_LOCATION_SIZE, w, h, d, n,
			    KPDS_LOCATION_DATA_TYPE, typ,
			    KPDS_LOCATION_REGION_SIZE, ow, oh, od, on,
			    KPDS_LOCATION_POSITION, 0, 0, 0, 0,
			    KPDS_LOCATION_OFFSET, 0, 0, 0, 0,
			    NULL))
      return FALSE;

   /* -- copy the data using the pds get'n'put data calls -- */
   for (i = 0; i < num_regions; i++)
   {
      data = kpds_get_data(src, KPDS_LOCATION_REGION, data);
      kpds_put_data(dest, KPDS_LOCATION_REGION, data);
   }
   kfree(data);
   
   /* -- reset the position and offset to be zero -- */
   if (!kpds_set_attributes(dest, 
			    KPDS_LOCATION_POSITION, 0, 0, 0, 0,
			    KPDS_LOCATION_OFFSET, 0, 0, 0, 0,
			    NULL))
      return FALSE;

   /* -- set the old coupling back on the segment -- */
   kdms_set_attribute(dest, KPDS_SEGMENT_LOCATION, 
		      KDMS_COUPLING, old_coupling);

   /* -- close the source reference -- */
   kdms_close(src);

   return TRUE;
}

/* 
 *    ================================================================== 
 *    LOCATION attribute handlers
 *    ==================================================================
 */

/*********** ---------------------------------------------------------------
 ***********  KPDS_LOCATION_GRID
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_grid_get
|       Purpose: get the location grid attribute
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 14:37
------------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_grid_get(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   int *ret_grid;
   int  grid;


   if (!kdms_get_attribute(obj, NULL, _STORED_LOCATION_GRID, &grid))
      return FALSE;

   /* -- backwards compatibility -- */
   if ((grid == KNONE) && kpds_query_location(obj))
      grid = KCURVILINEAR;
   
   ret_grid = kva_arg(*list, int *);
   if (ret_grid != NULL) *ret_grid = grid;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_grid_set
|       Purpose: set the location grid attribute
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 14:37
------------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_grid_set(kobject obj, int assoc, int attr, kaddr clientData,
	      kva_list *list)
{
   int grid = kva_arg(*list, int);

   /* -- see if the grid is something bogus -- */
   if ( (grid != KUNIFORM)     &&
	(grid != KRECTILINEAR) &&
	(grid != KCURVILINEAR))
   {
      /* errno = bogus grid */
      return FALSE;
   }

   /* -- if location already exists, then they can't set this attribute -- */
   if (kpds_query_location(obj))
   {
      /* errno  = location already exists!  
	 destroy the location first, then set attribute, then create it! */  
      return FALSE;
   }

   /* -- location does not yet exist -- set the attribute -- */
   if (!kdms_set_attribute(obj, NULL, _STORED_LOCATION_GRID, grid))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_grid_match
|       Purpose: match the location grid attribute
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 14:37
------------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_grid_match(kobject obj1, kobject obj2, int assoc, int attrib,
	        kaddr clientData1, kaddr clientData2)
{
   int grid1;
   int grid2;
   

   if (!kdms_get_attribute(obj1, NULL, _STORED_LOCATION_GRID, &grid1))
      return FALSE;

   if (!kdms_get_attribute(obj2, NULL, _STORED_LOCATION_GRID, &grid2))
      return FALSE;

   return (grid1 == grid2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_grid_copy
|       Purpose: copy the location grid attribute
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 14:37
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_grid_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	       kaddr clientData1, kaddr clientData2)
{
   int grid;
   
   if (!kdms_get_attribute(obj1, NULL, KPDS_LOCATION_GRID, &grid))
      return FALSE;

   if (!kdms_set_attribute(obj2, NULL, KPDS_LOCATION_GRID, grid))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_grid_query
|       Purpose: query the location grid attribute
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 14:37
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_grid_query(kobject obj, int assoc, int attrib, kaddr clientData,
	        int *num_args, int *arg_size, int *data_type,
	        int *permanent)
{
   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _loc_grid_print
|       Purpose: print the location grid attribute
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 14:37
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_loc_grid_print(kobject obj, int assoc, int attrib, kaddr clientData,
	        kfile * outfile)
{
   int grid   = kaps_location_grid(obj);
   int exists = kpds_query_location(obj);

   if (outfile)
   {
      switch (grid)
      {
	 case KNONE:
	    kfprintf(outfile, "KNONE");
	    break;
	 case KUNIFORM:
	    kfprintf(outfile, "KUNIFORM");
	    break;
	 case KRECTILINEAR:
	    kfprintf(outfile, "KRECTILINEAR");
	    break;
	 case KCURVILINEAR:
	    kfprintf(outfile, "KCURVILINEAR");
	    break;
	 default:
	    kfprintf(outfile, "%d (invalid)", grid);
      }
      if (exists)
	 kfprintf(outfile, "(location exists)");
      else
	 kfprintf(outfile, "(no location exists)");
   }

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_LOCATION_BEGIN
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _begin_get
|       Purpose: get the location begin attribute
|    Written By: Steve Kubica 
|          Date: Jan 03, 1995
------------------------------------------------------------*/
/* ARGSUSED */
static int
_begin_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   double *x = kva_arg(*list, double *);
   double *y = kva_arg(*list, double *);
   double *z = kva_arg(*list, double *);
   int     b[2] = {0, 0};
   int     e[2] = {0, 2};
   int     otype;
   double *begin = NULL;

   if (!kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM))
      return FALSE; /* error -- must create uniform location first */

   if (kaps_location_grid(obj) != KUNIFORM)
      return FALSE; /* error -- can only get this if you have uniform loc */

   /* -- set data type to double -- save old data type -- */
   if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, &otype))
      return FALSE;
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, KDOUBLE))
      return FALSE;

   begin = kdms_get_data(obj, KPDS_SEGMENT_UNIFORM, b, e, NULL);

   /* -- restore old data type -- */
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, otype))
      return FALSE;
   
   if (begin == NULL)
      return FALSE;
   
   if (x) *x = begin[0];
   if (y) *y = begin[1];
   if (z) *z = begin[2];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _begin_set
|       Purpose: set the location begin attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Jan 03, 1995
------------------------------------------------------------*/
/* ARGSUSED */
static int
_begin_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int     b[2] = {0, 0};
   int     e[2] = {0, 2};
   int     otype;
   int     status;
   double  begin[3] = {0.0, 0.0, 0.0};

   begin[0] = kva_arg(*list, double);
   begin[1] = kva_arg(*list, double);
   begin[2] = kva_arg(*list, double);
   
   if (!kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM))
      return FALSE; /* error -- must create uniform location first */

   if (kaps_location_grid(obj) != KUNIFORM)
      return FALSE; /* error -- can only get this if you have uniform loc */

   /* -- set data type to double -- save old data type -- */
   if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, &otype))
      return FALSE;
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, KDOUBLE))
      return FALSE;

   status = kdms_put_data(obj, KPDS_SEGMENT_UNIFORM, b, e, begin);

   /* -- restore old data type -- */
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, otype))
      return FALSE;
   
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _begin_match
|       Purpose: match the location begin attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
------------------------------------------------------------*/
/* ARGSUSED */
static int
_begin_match(kobject obj1, kobject obj2, int assoc, int attrib,
	     kaddr clientData1, kaddr clientData2)
{
   double x_1, y_1, z_1;
   double x_2, y_2, z_2;
   
   if (!kpds_get_attribute(obj1, KPDS_LOCATION_BEGIN, &x_1, &y_1, &z_1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_LOCATION_BEGIN, &x_2, &y_2, &z_2))
      return FALSE;

   return ((x_1 == x_2) && (y_1 == y_2) && (z_1 == z_2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _begin_copy
|       Purpose: copy the location begin attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_begin_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   double x, y, z;

   if (!kpds_get_attribute(obj1, KPDS_LOCATION_BEGIN, &x, &y, &z))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_LOCATION_BEGIN, x, y, z))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _begin_query
|       Purpose: query the location begin attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_begin_query(kobject obj, int assoc, int attrib, kaddr clientData,
	     int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM))
      return FALSE;		/* not an error */

   if (kaps_location_grid(obj) != KUNIFORM)
      return FALSE;             /* not an error */

   if (num_args)
      *num_args = 3;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KDOUBLE;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _begin_print
|       Purpose: print the location begin attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_begin_print(kobject obj, int assoc, int attrib, kaddr clientData,
	     kfile *outfile)
{
   double x, y, z;

   if (!kpds_get_attribute(obj, KPDS_LOCATION_BEGIN, &x, &y, &z))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%g %g %g", x, y, z);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_LOCATION_END
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _end_get
|       Purpose: get the location end attribute
|    Written By: Steve Kubica 
|          Date: Jan 03, 1995
------------------------------------------------------------*/
/* ARGSUSED */
static int
_end_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   double *x = kva_arg(*list, double *);
   double *y = kva_arg(*list, double *);
   double *z = kva_arg(*list, double *);
   int     b[2] = {1, 0};
   int     e[2] = {1, 2};
   int     otype;
   double *end = NULL;

   if (!kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM))
      return FALSE; /* error -- must create uniform location first */

   if (kaps_location_grid(obj) != KUNIFORM)
      return FALSE; /* error -- can only get this if you have uniform loc */

   /* -- set data type to double -- save old data type -- */
   if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, &otype))
      return FALSE;
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, KDOUBLE))
      return FALSE;

   end = kdms_get_data(obj, KPDS_SEGMENT_UNIFORM, b, e, NULL);

   /* -- restore old data type -- */
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, otype))
      return FALSE;
   
   if (end == NULL)
      return FALSE;
   
   if (x) *x = end[0];
   if (y) *y = end[1];
   if (z) *z = end[2];

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _end_set
|       Purpose: set the location end attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Jan 03, 1995
------------------------------------------------------------*/
/* ARGSUSED */
static int
_end_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int     b[2] = {1, 0};
   int     e[2] = {1, 2};
   int     otype;
   int     status;
   double  end[3] = {0.0, 0.0, 0.0};

   end[0] = kva_arg(*list, double);
   end[1] = kva_arg(*list, double);
   end[2] = kva_arg(*list, double);
   
   if (!kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM))
      return FALSE; /* error -- must create uniform location first */

   if (kaps_location_grid(obj) != KUNIFORM)
      return FALSE; /* error -- can only get this if you have uniform loc */

   /* -- set data type to double -- save old data type -- */
   if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, &otype))
      return FALSE;
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, KDOUBLE))
      return FALSE;

   status = kdms_put_data(obj, KPDS_SEGMENT_UNIFORM, b, e, end);

   /* -- restore old data type -- */
   if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_DATA_TYPE, otype))
      return FALSE;
   
   return status;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _end_match
|       Purpose: match the location end attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
------------------------------------------------------------*/
/* ARGSUSED */
static int
_end_match(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   double x_1, y_1, z_1;
   double x_2, y_2, z_2;
   
   if (!kpds_get_attribute(obj1, KPDS_LOCATION_END, &x_1, &y_1, &z_1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_LOCATION_END, &x_2, &y_2, &z_2))
      return FALSE;

   return ((x_1 == x_2) && (y_1 == y_2) && (z_1 == z_2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _end_copy
|       Purpose: copy the location end attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_end_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	  kaddr clientData1, kaddr clientData2)
{
   double x, y, z;

   if (!kpds_get_attribute(obj1, KPDS_LOCATION_END, &x, &y, &z))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_LOCATION_END, x, y, z))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _end_query
|       Purpose: query the location end attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_end_query(kobject obj, int assoc, int attrib, kaddr clientData,
	   int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM))
      return FALSE;		/* not an error */

   if (kaps_location_grid(obj) != KUNIFORM)
      return FALSE;             /* not an error */

   if (num_args)
      *num_args = 3;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KDOUBLE;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_print
|       Purpose: print the location end attribute
|    Written By: Steve Kubica
|          Date: Jan 03, 1995
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_end_print(kobject obj, int assoc, int attrib, kaddr clientData,
	   kfile *outfile)
{
   double x, y, z;

   if (!kpds_get_attribute(obj, KPDS_LOCATION_END, &x, &y, &z))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%g %g %g", x, y, z);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_LOCATION_SIZE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _size_get
|       Purpose: get the location size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *w = kva_arg(*list, int *);
   int *h = kva_arg(*list, int *);
   int *d = kva_arg(*list, int *);
   int *n = kva_arg(*list, int *);
   int *size;
   int  grid;

   grid = kaps_location_grid(obj);

   switch (grid)
   {
      case KUNIFORM :
      {
	 int *psize;
	 
	 if (!kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM))
	    return FALSE;  /* error -- must create location first */

	 /* -- retreive the presentation uniform location size -- */ 
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM,
				 KPDS_UNIFORM_PRES_LOCATION_SIZE, &size))
	       return FALSE;

	 /* -- initialize uninitialized sizes from the physical layer -- */ 
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM,
				 KPDS_UNIFORM_PHYS_LOCATION_SIZE, &psize))
	       return FALSE;
	 
	 if (w) *w = (size[KWID] < 0) ? psize[KWID] : size[KWID];
	 if (h) *h = (size[KHGT] < 0) ? psize[KHGT] : size[KHGT];
	 if (d) *d = (size[KDEP] < 0) ? psize[KDEP] : size[KDEP];
	 if (n) *n = (size[KDIM] < 0) ? psize[KDIM] : size[KDIM];

      } break;
      
      case KRECTILINEAR : 
      {
	 int dim;
	 
	 if (!kdms_query_segment(obj, KPDS_SEGMENT_WIDTH))
	    return FALSE;  /* error -- must create location first */

	 if (!kdms_get_attribute(obj, KDMS_SEGMENT_WIDTH, KDMS_SIZE, &size))
	    return FALSE;
	 if (w) *w = size[0];

	 if (!kdms_get_attribute(obj, KDMS_SEGMENT_HEIGHT, KDMS_SIZE, &size))
	    return FALSE;
	 if (h) *h = size[0];

	 if (!kdms_get_attribute(obj, KDMS_SEGMENT_DEPTH, KDMS_SIZE, &size))
	    return FALSE;
	 if (d) *d = size[0];

	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
				 KPDS_RECTILINEAR_PRES_DIMENSION, &dim))
	    return FALSE;
	 if (n) *n = dim;

      } break;
      
      case KCURVILINEAR :
      {
	 if (!kdms_query_segment(obj, KPDS_SEGMENT_LOCATION))
	    return FALSE;  /* error -- must create location first */

	 if (!kdms_get_attribute(obj, KDMS_SEGMENT_LOCATION, KDMS_SIZE, &size))
	    return FALSE; 
    
	 if (w) *w = size[KWID];
	 if (h) *h = size[KHGT];
	 if (d) *d = size[KDEP];
	 if (n) *n = size[KDIM];

      } break;

      default :
	 kerror("kappserv", "_size_get", 
		"Invalid location grid type (%d).", grid);
	 return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_set
|       Purpose: set the location size attribute
|    Written By: Steve Kubica and Jeremy Worley
|          Date: Apr 27, 1994 21:34
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_set(kobject obj, int assoc, int attr, kaddr clientData, 
	  kva_list *list)
{
   int  w = kva_arg(*list, int);
   int  h = kva_arg(*list, int);
   int  d = kva_arg(*list, int);
   int  n = kva_arg(*list, int);
   int  cp;
   int grid = kaps_location_grid(obj);
   
   if (!_query_loc(obj))
      return FALSE;  /* error -- must create location first */

   if (sync_sizes)
   {
      /* -- set the shared size of the value segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
	 if (!kpds_set_attribute(obj, KPDS_VALUE_SIZE, w, h, d, -1, -1))
	    return FALSE;
      
      /* -- set the shared size of the mask segment -- */
      if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
	 if (!kpds_set_attribute(obj, KPDS_MASK_SIZE, w, h, d, -1, -1))
	    return FALSE;
   
      /* -- set the shared size of the map segment -- */
      if (kdms_query_segment(obj, KDMS_SEGMENT_MAP))
      {
	 int md;

	 /* -- track size of mask depth only if previous map depth != 1 -- */
	 if (!kpds_get_attribute(obj, KPDS_MAP_SIZE, NULL, NULL, 
				 &md, NULL, NULL))
	    return FALSE;
      
	 if (!kpds_set_attribute(obj, KPDS_MAP_SIZE, -1, -1,
				 ((md == 1) ? -1 : d), -1, -1))
	    return FALSE;
      }

      sync_sizes = TRUE;
   }

   /* set the size of the location segment */
   switch (grid)
   {
      case KUNIFORM :
      {
	 int *psize;
	 
	 /* -- want to initialize any physical size that < 0 -- */
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM, 
				 KPDS_UNIFORM_PHYS_LOCATION_SIZE, &psize))
	    return FALSE;
	    
	 psize[KWID] = (psize[KWID] < 0 && w > 0) ? w : psize[KWID];
	 psize[KHGT] = (psize[KHGT] < 0 && h > 0) ? h : psize[KHGT];
	 psize[KDEP] = (psize[KDEP] < 0 && d > 0) ? d : psize[KDEP];
	 psize[KDIM] = (psize[KDIM] < 0 && n > 0) ? n : psize[KDIM];

	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, 
				 KPDS_UNIFORM_PHYS_LOCATION_SIZE, psize))
	    return FALSE;

	 /* -- set the presentation size -- */
	 if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM, 
				 KPDS_UNIFORM_PRES_LOCATION_SIZE, &psize))
	    return FALSE;

	 psize[KWID] = (w > 0) ? w : psize[KWID];
	 psize[KHGT] = (h > 0) ? h : psize[KHGT];
	 psize[KDEP] = (d > 0) ? d : psize[KDEP];
	 psize[KDIM] = (n > 0) ? n : psize[KDIM];

	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, 
				 KPDS_UNIFORM_PRES_LOCATION_SIZE, psize))
	    return FALSE;
	 
	 /* -- if we are coupled, then set the physical size too -- */
	 kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM, KDMS_COUPLING,&cp);
	 if (cp == KCOUPLED)
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM, 
				    KPDS_UNIFORM_PHYS_LOCATION_SIZE, psize))
	       return FALSE;

      } break;

      case KRECTILINEAR :
      {
	 int wsize[2];
	 int hsize[2];
	 int dsize[2];

	 wsize[0] = w;
	 hsize[0] = h;
	 dsize[0] = d;
	 
	 /* -- size arguments are expected to be arrays -- */
	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH, KDMS_SIZE, wsize))
	    return FALSE;
	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_HEIGHT, KDMS_SIZE, hsize))
	    return FALSE;
	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_DEPTH, KDMS_SIZE, dsize))
	    return FALSE;

	 /* -- set the presentation dimension size -- */
	 if (n > 1)
	 {
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH, 
				    KPDS_RECTILINEAR_PRES_DIMENSION, n))
	    return FALSE;
	 
	    /* -- if we are coupled, then set the physical size too -- */
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,KDMS_COUPLING,&cp))
	    {
	       if (cp == KCOUPLED)
		  if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH, 
					  KPDS_RECTILINEAR_PHYS_DIMENSION,n))
		     return FALSE;
	    }
	 }
	 
      } break;

      case KCURVILINEAR :
      {
	 int size[KPDS_LOCATION_DIM] = {-1, -1, -1, -1};

	 size[KWID] = w;
	 size[KHGT] = h;
	 size[KDEP] = d;
	 size[KDIM] = n;
	 
	 if (!kdms_set_attribute(obj, KDMS_SEGMENT_LOCATION, KDMS_SIZE, size))
	    return FALSE; 
      } break;

      case KNONE:
	 /* -- must create location first -- */
	 return FALSE;

      default :
	 kerror("kappserv", "_loc_size_set", 
		"Invalid location grid type (%d).", grid);
	 return FALSE;
   }
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_match
|       Purpose: match the location size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
------------------------------------------------------------*/
/* ARGSUSED */
static int
_size_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int w1, h1, d1, n1;
   int w2, h2, d2, n2;

   if (!kpds_get_attribute(obj1, KPDS_LOCATION_SIZE, &w1, &h1, &d1, &n1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_LOCATION_SIZE, &w2, &h2, &d2, &n2))
      return FALSE;

   return ((w1 == w2) && (h1 == h2) && (d1 == d2) && (n1 == n2));
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_copy
|       Purpose: copy the location size attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:37
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int w, h, d, n;

   if (!kpds_get_attribute(obj1, KPDS_LOCATION_SIZE, &w, &h, &d, &n))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_LOCATION_SIZE, w, h, d, n))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_query
|       Purpose: query the location size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!_query_loc(obj))
      return FALSE;		/* not an error */

   if (num_args)
      *num_args = 4;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _size_print
|       Purpose: print the location size attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_size_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int w, h, d, n;

   if (!kpds_get_attribute(obj, KPDS_LOCATION_SIZE, &w, &h, &d, &n))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%d %d %d %d", w, h, d, n);

   return TRUE;
}

/*********** ---------------------------------------------------------------
 ***********  KPDS_LOCATION_DATA_TYPE
 *********** --------------------------------------------------------------- */

/*-----------------------------------------------------------
|  Routine Name: (static) _type_get
|       Purpose: get the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_get(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int *rtyp = kva_arg(*list, int *);
   int  typ;

   if (!_query_loc(obj))
      return FALSE;

   if ((typ = _get_type(obj)) == FALSE)
      return FALSE;
      
   if (rtyp) *rtyp = typ;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_set
|       Purpose: set the type attribute
|    Written By: Jeremy Worley, John Salas, and Steve Kubica
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_set(kobject obj, int assoc, int attr, kaddr clientData, kva_list *list)
{
   int typ = kva_arg(*list, int);
   int grid;
   

   /* -- location created will depend on the location grid attrbute -- */
   if (!kpds_get_attribute(obj, KPDS_LOCATION_GRID, &grid))
      return(FALSE);

   switch(grid)
   {
      case KUNIFORM :
      {
	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM,
				 KDMS_DATA_TYPE, typ))
	    return FALSE;
      } break;

      case KRECTILINEAR :
      {
	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
				 KDMS_DATA_TYPE, typ) ||
	     !kdms_set_attribute(obj, KPDS_SEGMENT_HEIGHT,
				 KDMS_DATA_TYPE, typ) ||
	     !kdms_set_attribute(obj, KPDS_SEGMENT_DEPTH,
				 KDMS_DATA_TYPE, typ))
	    return FALSE;
      } break;

      case KCURVILINEAR :
      {
	 if (!kdms_set_attribute(obj, KPDS_SEGMENT_LOCATION,
				 KDMS_DATA_TYPE, typ))
	    return FALSE;
      } break;

      case KNONE:
	 /* -- must create location first -- */
	 return FALSE;

      default :
	 kerror("kappserv", "_type_set", 
		"Invalid location grid type (%d).", grid);
	 return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_match
|       Purpose: match the type attribute
|    Written By: Steve Kubica and John Salas
|          Date: Apr 28, 1994 17:39
------------------------------------------------------------*/
/* ARGSUSED */
static int
_type_match(kobject obj1, kobject obj2, int assoc, int attrib,
	    kaddr clientData1, kaddr clientData2)
{
   int typ1;
   int typ2;

   if (!_query_loc(obj1) || !_query_loc(obj2))
      return FALSE;

   if (!kpds_get_attribute(obj1, KPDS_LOCATION_DATA_TYPE, &typ1))
      return FALSE;

   if (!kpds_get_attribute(obj2, KPDS_LOCATION_DATA_TYPE, &typ2))
      return FALSE;
   
   return (typ1 == typ2);
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_copy
|       Purpose: copy the type attribute
|    Written By: Steve Kubica
|          Date: Apr 28, 1994 17:43
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_copy(kobject obj1, kobject obj2, int assoc, int attrib,
	   kaddr clientData1, kaddr clientData2)
{
   int typ;

   if (!_query_loc(obj1) || !_query_loc(obj2))
      return FALSE;

   if (!kpds_get_attribute(obj1, KPDS_LOCATION_DATA_TYPE, &typ))
      return FALSE;

   if (!kpds_set_attribute(obj2, KPDS_LOCATION_DATA_TYPE, typ))
      return FALSE;
   
   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_query
|       Purpose: query the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_query(kobject obj, int assoc, int attrib, kaddr clientData,
	    int *num_args, int *arg_size, int *data_type, int *permanent)
{
   if (!_query_loc(obj))
      return FALSE;

   if (num_args)
      *num_args = 1;
   if (arg_size)
      *arg_size = 1;
   if (data_type)
      *data_type = KINT;
   if (permanent)
      *permanent = TRUE;

   return TRUE;
}

/*-----------------------------------------------------------
|  Routine Name: (static) _type_print
|       Purpose: print the type attribute
|    Written By: Steve Kubica
|          Date: Mar 30, 1994 14:36
-----------------------------------------------------------*/
/* ARGSUSED */
static int
_type_print(kobject obj, int assoc, int attrib, kaddr clientData,
	    kfile * outfile)
{
   int typ;

   if (!_query_loc(obj))
      return FALSE;

   if (!kpds_get_attribute(obj, KPDS_LOCATION_DATA_TYPE, &typ))
      return FALSE;

   if (outfile)
      kfprintf(outfile, "%s (%d)", kdefine_to_datatype(typ), typ);

   return TRUE;
}

/* 
 *    ================================================================== 
 *    LOCATION segment handlers
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _create_location
|
|       Purpose: This routine will create the location segment and
|		 initialize its dimensionality and index order.  The
|		 setting of KPDS_LOCATION_GRID is used to determine
|		 what kind of location data to build.
|
|         Input: obj     - the obj in which to create the location segment
|		 segment - the name of the location segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_create_location(
   kobject  obj,
   char    *segment)
{
   int  order[KPDS_LOCATION_DIM]  = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};
   int *vorder;
   int  grid;
   int  w = -1;
   int  h = -1;
   int  d = -1;
   int  n = -1;

   /* -- location created will depend on the location grid attrbute -- */
   if (!kpds_get_attribute(obj, KPDS_LOCATION_GRID, &grid))
      return(FALSE);

   switch(grid)
   {
      case KUNIFORM :
      {
	 if (!kdms_create_segment(obj, KPDS_SEGMENT_UNIFORM))
	    return FALSE;

      } break;

      case KRECTILINEAR :
      {
	 if (!kdms_create_segment(obj, KPDS_SEGMENT_WIDTH)  ||
	     !kdms_create_segment(obj, KPDS_SEGMENT_HEIGHT) ||
	     !kdms_create_segment(obj, KPDS_SEGMENT_DEPTH))
	    return FALSE;
	 
	 /* -- create phys attribute to carry along dimension size -- */
	 if (!kdms_create_attribute(obj, KPDS_SEGMENT_WIDTH, 
				    KPDS_RECTILINEAR_PHYS_DIMENSION, 1,
				    1, KINT, TRUE, TRUE) ||
	     !kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH, 
				 KPDS_RECTILINEAR_PHYS_DIMENSION, 3))
	    return FALSE;

	 /* -- corresponding rectilinear dimension presentation
	  *    attribute has been defined, since it needs to be
	  *    dynamically created on any presentation of the width segment.
	  */

      } break;

      case KCURVILINEAR :
      case KNONE :
      {
	 if (!kdms_create_segment(obj, KPDS_SEGMENT_LOCATION))
	    return FALSE;
	 
	 if (!kdms_set_attributes(obj, KPDS_SEGMENT_LOCATION, 
				       KDMS_DIMENSION, KPDS_LOCATION_DIM,
				       KDMS_INDEX_ORDER, order, NULL))
	    return FALSE;
	 
	 /*   since we may've created on grid of KNONE, 
	  *   set stored location grid to be KCURVILINEAR 
	  */
	 if (!kdms_set_attribute(obj, NULL, 
				 _STORED_LOCATION_GRID, KCURVILINEAR))
	    return FALSE;

      } break;
	 
      default :
	 kerror("kappserv", "_create_location", 
		"Invalid location grid type (%d).", grid);
	 return FALSE;
   }

   if (kdms_query_segment(obj, KPDS_SEGMENT_VALUE))
   {
      kpds_get_attribute(obj, KPDS_VALUE_SIZE, &w, &h, &d, NULL, NULL);

      kdms_get_attribute(obj, KPDS_SEGMENT_VALUE, KDMS_INDEX_ORDER, &vorder);
      order[0] = vorder[0];
      order[1] = vorder[1];
      order[2] = vorder[2];
      order[3] = KDIMENSION;
      kdms_set_attribute(obj, KPDS_SEGMENT_LOCATION, KDMS_INDEX_ORDER, order);
   }
   else if (kdms_query_segment(obj, KPDS_SEGMENT_MASK))
   {
      kpds_get_attribute(obj, KPDS_MASK_SIZE, &w, &h, &d, NULL, NULL);

      kdms_get_attribute(obj, KPDS_SEGMENT_MASK, KDMS_INDEX_ORDER, &vorder);
      order[0] = vorder[0];
      order[1] = vorder[1];
      order[2] = vorder[2];
      order[3] = KDIMENSION;
      kdms_set_attribute(obj, KPDS_SEGMENT_LOCATION, KDMS_INDEX_ORDER, vorder);
   }
   else if (kdms_query_segment(obj, KPDS_SEGMENT_MAP))
      kpds_get_attribute(obj, KPDS_MAP_SIZE, NULL, NULL, &d, NULL, NULL);
   
   return kpds_set_attribute(obj, KPDS_LOCATION_SIZE, w, h, d, n);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _destroy_location
|
|       Purpose: This routine will destroy the location segment,
|		 regardless of the undelying grid type.
|
|         Input: obj     - the obj from which to destroy the location segment
|		 segment - the name of the location segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_destroy_location(
   kobject  obj,
   char    *segment)
{
   int grid;
   
   /* -- location created will depend on the location grid attrbute -- */
   if (!kpds_get_attribute(obj, KPDS_LOCATION_GRID, &grid))
      return(FALSE);

   switch(grid)
   {
      case KUNIFORM :
	 return kdms_destroy_segment(obj, KPDS_SEGMENT_UNIFORM);

      case KRECTILINEAR :
	 return (kdms_destroy_segment(obj, KPDS_SEGMENT_WIDTH) &&
		 kdms_destroy_segment(obj, KPDS_SEGMENT_HEIGHT) &&
		 kdms_destroy_segment(obj, KPDS_SEGMENT_DEPTH));

      case KCURVILINEAR :
	 return kdms_destroy_segment(obj, KPDS_SEGMENT_LOCATION);
   }

   kerror("kappserv", "_destroy_location", 
	  "Invalid location grid type (%d).", grid);

   return FALSE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _query_location
|
|       Purpose: This routine will query the location segment,
|		 with an understanding of the underlying grid type.
|
|         Input: obj     - the obj from which to query the location segment
|		 segment - the name of the location segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_query_location(
   kobject  obj,
   char    *segment)
{
   return _query_loc(obj);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _get_location_data
|
|       Purpose: This routine gets data from the location segment.
|                It contains an understanding of the underlying
|		 grid type.
|
|         Input: object  - the object to retrieve data from
|	         segment - segment to retrieve data from
|		 begin   - the begin point for region to get :
|			   if the data is to be mapped, this
|		           begin point represents the mapped begin
|		 end     - the end point for the region to get :
|			   if the data is to be mapped, this
|		           end point represents the mapped end
|
|        Output: data   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|
|       Returns: pointer to the data being returned on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static kaddr
_get_location_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   int grid = kaps_location_grid(obj);
   kaddr rdata;

   /* -- verify that location segment exists -- */
   if (!_query_loc(obj))
      return NULL;
   
   if (grid == KUNIFORM)
      rdata = kaps_get_uniform_data(obj, begin, end, data);
   else if (grid == KRECTILINEAR)
      rdata = kaps_get_rectilinear_data(obj, begin, end, data);
   else
      rdata = kdms_get_data(obj, segment, begin, end, data);
   
   if (kaps_elements_first(obj))
   {
      kaddr ldata;
      
      ldata = kaps_transpose_elements_first(rdata, begin, end, _get_type(obj),
					   KPDS_LOCATION_DIM);
      return ldata;
   }
   
   return rdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _put_location_data
|
|       Purpose: This routine puts data into the location segment.
|                It understands the underlying grid type and will
|		 not allow curvilinear primitives to be put into
|		 rectilinear data.
|
|         Input: obj - the object to put data into
|	         segment - the segment to put the data into
|		 begin   - the begin point for the put
|		 end     - the end point for the put
|		 data    - the data to be put
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static int
_put_location_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   int grid = kaps_location_grid(obj);
   kaddr pdata = data;

   /* -- verify that location segment exists -- */
   if (!_query_loc(obj))
      return FALSE;

   if (grid == KRECTILINEAR)
   {
      kerror("kappserv", "_put_location_data", 
	     "Not possible to put a curvilinear primitive if location "
	     "data is rectilinear.");
      return FALSE;
   }

   /* -- transpose if elements are first -- */
   if (kaps_elements_first(obj))
      pdata = kaps_transpose_elements_last(data, begin, end, _get_type(obj),
					   KPDS_LOCATION_DIM);

   /* -- put the data -- */
   return kdms_put_data(obj, segment, begin, end, pdata);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _get_location_atr
|
|       Purpose: This routine will filter out specific attributes
|		 which may not actually "exist" with uniform or
|		 rectilinear location grids.  This allows dms
|		 programs to operate on uniform and rectilinear
|		 location data as if it were curvilinear location
|		 data.
|
|		 The specific attributes this routine will 'emulate' are:
|
|		    KDMS_DIMENSION
|		    KDMS_SIZE
|		    KDMS_DATATYPE
|		    KDMS_INDEX_ORDER
|
|         Input: obj       - the obj from which to destroy the location segment
|		 segment   - the name of the location segment.
|		 attribute - the name of the attribute to get.
|		 list      - already opened variable argument list.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_get_location_atr(
   kobject   obj,
   char     *segment,
   char     *attribute,
   kva_list *list)
{
   int grid = kaps_location_grid(obj);

   /* -- verify that location segment exists -- */
   if (!_query_loc(obj))
      return FALSE;

   /* -- uniform and rectilinear are only special cases -- */
   if (grid == KUNIFORM || grid == KRECTILINEAR)
   {
      if (kstrcmp(attribute, KDMS_DIMENSION) == 0)
      {
	 int *dim = kva_arg(*list, int *);

	 if (dim) *dim = 4;
	 return TRUE;
      }
      
      if (kstrcmp(attribute, KDMS_DATA_TYPE) == 0)
      {
	 int *type = kva_arg(*list, int *);

	 if (type) *type = _get_type(obj);
	 return TRUE;
      }
      
      if (kstrcmp(attribute, KDMS_SIZE) == 0)
      {
	 static int   mysize[KPDS_LOCATION_DIM];
	 int        **size = kva_arg(*list, int **);

	 kpds_get_attribute(obj, KPDS_LOCATION_SIZE, 
			    &mysize[0], &mysize[1], &mysize[2], &mysize[3]);
	 if (size) *size = mysize;
	 return TRUE;
      }

      if (kstrcmp(attribute, KDMS_INDEX_ORDER) == 0)
      {
	 static int   myorder[4] = {KWIDTH, KHEIGHT, KDEPTH, KDIMENSION};
	 int        **order = kva_arg(*list, int **);

	 if (order) *order = myorder;
	 return TRUE;
      }
   }   
   else
      return kdms_vget_attribute(obj, segment, attribute, list);
}


/*-----------------------------------------------------------
|
|  Routine Name: (static) _create_rectilinear
|
|       Purpose: This routine will create a rectilinear location 
|		 segment and initialize its dimensionality and index order.
|
|         Input: obj     - the obj in which to create the location segment
|		 segment - the name of the location segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
static int
_create_rectilinear(
   kobject  obj,
   char    *segment)
{
   int order[1] = {KWIDTH};

   if (!kdms_create_segment(obj, segment))
      return FALSE;
   
   if (!kdms_set_attributes(obj, segment,
			    KDMS_DIMENSION, 1,
			    KDMS_INDEX_ORDER, order, NULL))
      return FALSE;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _query_rectilinear
|
|       Purpose: This routine will query a rectilinear location segment,
|		 with an understanding of the underlying grid type.
|
|         Input: obj     - the obj from which to query the location segment
|		 segment - the name of the location segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
static int
/* ARGSUSED */
_query_rectilinear(
   kobject  obj,
   char    *segment)
{
   int grid = kaps_location_grid(obj);
   
   /* -- uniform can appear as rectilinear -- */
   switch(grid)
   {
      case KUNIFORM :
	 return kdms_query_segment(obj, KPDS_SEGMENT_UNIFORM);

      case KRECTILINEAR :
	 return kdms_query_segment(obj, KPDS_SEGMENT_WIDTH);
	 
      case KCURVILINEAR :  /* --can not present curvilinear as rectilinear-- */
	 return FALSE;
   }

   return FALSE; /* -- not an error ... just KNONE -- */
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _get_rectilinear_data
|
|       Purpose: This routine gets data from a rectilinear location 
|		 segment.  It contains an understanding of the
|		 underlying grid type.
|
|         Input: object  - the object to retrieve data from
|	         segment - segment to retrieve data from
|		 begin   - the begin point for region to get :
|			   if the data is to be mapped, this
|		           begin point represents the mapped begin
|		 end     - the end point for the region to get :
|			   if the data is to be mapped, this
|		           end point represents the mapped end
|
|        Output: data   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|
|       Returns: pointer to the data being returned on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static kaddr
_get_rectilinear_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   int grid = kaps_location_grid(obj);

   if (grid == KUNIFORM)
   {
      kerror("kappserv", "_get_rectilinear_data", 
	     "Location data is uniform : presentation of this rectilinear "
	     "primitive is currently unsupported.  Sorry for the "
	     "inconvenience.");
      return NULL;
   }
   
   else if (grid == KCURVILINEAR)
   {
      kerror("kappserv", "_get_rectilinear_data", 
	     "Location data is curvilinear : presentation of this rectilinear "
	     "primitive is currently unsupported.  Sorry for the "
	     "inconvenience.");
      return NULL;
   }

   /* -- verify that rectilinear segment exists -- */
   if (!_query_rectilinear(obj, segment))
      return NULL;

   return kdms_get_data(obj, segment, begin, end, data);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _put_rectilinear_data
|
|       Purpose: This routine puts data into a rectilinear location 
|		 segment.  It contains an understanding of the
|		 underlying grid type and will allow uniform
|		 location data to be  as rectilinear.
|
|         Input: object  - the object to retrieve data from
|	         segment - segment to retrieve data from
|		 begin   - the begin point for region to get :
|			   if the data is to be mapped, this
|		           begin point represents the mapped begin
|		 end     - the end point for the region to get :
|			   if the data is to be mapped, this
|		           end point represents the mapped end
|
|        Output: data   - pointer to already allocated data,
|			  or NULL if we should perform the allocation
|
|       Returns: pointer to the data being returned on success,
|		 NULL otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
static int
_put_rectilinear_data(
   kobject obj,
   char   *segment,
   int    *begin,
   int    *end,
   kaddr   data)
{
   int grid = kaps_location_grid(obj);

   if (grid == KUNIFORM)
   {
      kerror("kappserv", "_put_rectilinear_data", 
	     "Not possible to put a uniform primitive if location "
	     "data is rectilinear.");
      return FALSE;
   }
   
   else if (grid == KCURVILINEAR)
   {
      kerror("kappserv", "_put_rectilinear_data", 
	     "Not possible to put a curvilinear primitive if location "
	     "data is rectilinear.");
      return FALSE;
   }

   /* -- verify that rectilinear segment exists -- */
   if (!_query_rectilinear(obj, segment))
      return FALSE;

   return kdms_put_data(obj, segment, begin, end, data);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _create_uniform
|
|       Purpose: This routine will create a uniform location 
|		 segment and initialize its dimensionality and index order.
|
|         Input: obj     - the obj in which to create the location segment
|		 segment - the name of the location segment.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica & Jeremy Worley
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_create_uniform(
   kobject  obj,
   char    *segment)
{
   /* -- assume an initial dimension of 3 (xyz) -- */
   int init[KPDS_LOCATION_DIM] = {-1, -1, -1, 3};
   int ord[2] = {KWIDTH, KDIMENSION};
   int siz[2] = {2,3};
   int beg[2] = {0,0};
   int end[2] = {0,2};

   /* -- default uniform location begin and end points -- */
   double b[3] = {0.0, 0.0, 0.0};
   double e[3] = {1.0, 1.0, 1.0};

   if (!kdms_create_segment(obj, KPDS_SEGMENT_UNIFORM))
      return FALSE;
   
   if (!kdms_set_attributes(obj, KPDS_SEGMENT_UNIFORM,
			    KDMS_DIMENSION,   2,
			    KDMS_SIZE,        siz,
			    KDMS_DATA_TYPE,   KDOUBLE,
			    KDMS_INDEX_ORDER, ord, NULL))
      return FALSE;

   /* -- put default uniform begin point -- */
   kdms_put_data(obj, KPDS_SEGMENT_UNIFORM, beg, end, b);

   /* -- put default uniform end points -- */
   beg[0] = 1; end[0] = 1;
   kdms_put_data(obj, KPDS_SEGMENT_UNIFORM, beg, end, e);
   
   /* -- create the physical uniform location size attribute -- */
   if (!kdms_create_attribute(obj, KPDS_SEGMENT_UNIFORM,
			      KPDS_UNIFORM_PHYS_LOCATION_SIZE, 1, 
			      KPDS_LOCATION_DIM, KINT, TRUE, TRUE) ||
       !kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM,
			   KPDS_UNIFORM_PHYS_LOCATION_SIZE, init))
      return FALSE;

   /* -- corresponding uniform size presentation attribute has been
    *    defined, since it needs to be dynamically created on any
    *    presentation of the uniform segment.  
    */

   return TRUE;
}



/* 
 *    ================================================================== 
 *    PRIVATE LOCATION functions
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kpds_init_location_segment()
|
|       Purpose: Define the location segment.
|
|         Input: none
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Dec 26, 1994
| Modifications:
|
------------------------------------------------------------*/
int
kpds_init_location_segment(void)
{
   int init[KPDS_LOCATION_DIM] = {-1, -1, -1, 3};
   int status = TRUE; 

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_LOCATION_GRID, NULL,
					 _loc_grid_get,   _loc_grid_set, 
					 _loc_grid_match, _loc_grid_copy, 
					 _loc_grid_query, _loc_grid_print);

   status &= kdms_define_attribute(KDMS_OBJECT,
				   _STORED_LOCATION_GRID, 1, 1, KINT,
				   TRUE, TRUE, KNONE);

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_LOCATION_BEGIN, NULL,
					 _begin_get,   _begin_set, 
					 _begin_match, _begin_copy, 
					 _begin_query, _begin_print);

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_LOCATION_END, NULL,
					 _end_get,   _end_set, 
					 _end_match, _end_copy, 
					 _end_query, _end_print);

   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_LOCATION_SIZE, NULL,
					 _size_get,   _size_set, 
					 _size_match, _size_copy, 
					 _size_query, _size_print);


   status &= kdms_define_quasi_attribute(KDMS_OBJECT, 
					 KPDS_LOCATION_DATA_TYPE, NULL,
					 _type_get,   _type_set, 
					 _type_match, _type_copy,
					 _type_query, _type_print);


   /* -- special definition for uniform size presentation purposes -- */
   status &= kdms_define_attribute(KPDS_SEGMENT_UNIFORM, 
				   KPDS_UNIFORM_PRES_LOCATION_SIZE, 1,
				   KPDS_LOCATION_DIM, KINT, FALSE, FALSE,
				   init);

   /* -- special definition for rectilinear size presentation purposes -- */
   status &= kdms_define_attribute(KPDS_SEGMENT_WIDTH, 
				   KPDS_RECTILINEAR_PRES_DIMENSION, 1,
				   1, KINT, FALSE, FALSE, 3);


   status &= kdms_define_segment(KPDS_SEGMENT_WIDTH, _create_rectilinear, 
				 NULL,
				 _query_rectilinear, 
				 _get_rectilinear_data,
				 _put_rectilinear_data,
				 NULL, NULL, NULL, NULL, NULL, NULL);

   status &= kdms_define_segment(KPDS_SEGMENT_HEIGHT, _create_rectilinear,
				 NULL,
				 _query_rectilinear, _get_rectilinear_data,
				 NULL, NULL, NULL, NULL, NULL, NULL, NULL);

   status &= kdms_define_segment(KPDS_SEGMENT_DEPTH, _create_rectilinear,
				 NULL,
				 _query_rectilinear, _get_rectilinear_data,
				 NULL, NULL, NULL, NULL, NULL, NULL, NULL);

   status &= kdms_define_segment(KPDS_SEGMENT_UNIFORM, _create_uniform,
				 NULL, NULL, NULL, NULL,
				 NULL, NULL, NULL, NULL, NULL, NULL);

   status &= kdms_define_segment(KPDS_SEGMENT_LOCATION, _create_location,
				 _destroy_location,   _query_location,
				 _get_location_data,  _put_location_data,
				 _get_location_atr, 
				 NULL, NULL, NULL, NULL, NULL);
   return status;
}

/* 
 *    ================================================================== 
 *    PUBLIC LOCATION functions
 *    ==================================================================
 */

/************************************************************
*
*  Routine Name: kpds_create_location - create a location segment within a 
*				        data object.
*
*       Purpose: This function is used to create a location segment
*		 within a specified data object.  The size of the
*		 location segment will be initialized to match any
*		 the sizes shared with any other polymorphic segments.
*
*		 Either uniform, rectilinear, or curvilinear location
*		 grids can be created.  The attribute
*		 KPDS_LOCATION_GRID should be set to the desired grid
*		 type of either KUNIFORM, KRECTILINEAR, or
*		 KCURVILINEAR, before calling kpds_create_location().
*		 By default, if the grid attribute is not set,
*		 curvilinear location will be created.
*
*		 It is considered an error to create a location segment
*		 if the object already contains one.
*
*         Input: object  - object in which to create the location segment.
*
*        Output: none
*
*       Returns: TRUE (1) if the location segment was successfully created, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_create_location(
*		!   kobject object)
*
*************************************************************/
int
kpds_create_location(kobject object)
{
   return (kpds_init() && kdms_create_segment(object, KPDS_SEGMENT_LOCATION));
}

/************************************************************
*
*  Routine Name: kpds_query_location - determine if the location segment
*				       exists in a data object.
*
*       Purpose: This function is used to determine if the location
*		 segment exists in a data object.  If location segment
*		 exists in the specified object, then this function
*		 will return TRUE.  If the object is invalid, or location
*		 data does not exist in the object, then this function
*		 will return FALSE.
*
*         Input: object  - data object to be queried.
*
*        Output: none
*
*       Returns: TRUE (1) if the location segment exists, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_query_location(
*		!   kobject object)
*
*************************************************************/
int
kpds_query_location(kobject object)
{
   return (kpds_init() && kdms_query_segment(object, KPDS_SEGMENT_LOCATION));
}

/************************************************************
*
*  Routine Name: kpds_destroy_location - destroy the location segment in a
*				         data object.
*
*       Purpose: This function is used to destroy the location segment
*		 contained within an object.  Once the location segment
*		 has been destroyed, any data or attributes associated
*		 with the location data will be lost forever.  A new
*		 location segment can be created in its place with the
*		 function kpds_create_location.
*
*		 If the location segment does not exist in the specified
*		 object, it is considered to be an error.
*
*         Input: object  - object from which to remove the location segment.
*
*        Output: location segment is destroyed
*
*       Returns: TRUE (1) if the location segment is successfully destroyed, 
*		 FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 20, 1993 08:33
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_destroy_location(
*		!   kobject object)
*
*************************************************************/
int
kpds_destroy_location(kobject object)
{
   return (kpds_init() && kdms_destroy_segment(object, KPDS_SEGMENT_LOCATION));
}

/************************************************************
*
*  Routine Name: kpds_initialize_location - preset location segment to constant
*
*       Purpose: This routine initializes the entire location segment
*		 to a specified value.  The last two arguments are
*		 real and imaginary components of a complex number
*		 that is the value that the data will be initialized
*		 to.  The data type of the location segment at the
*		 time this function is called will determine how the
*		 complex number is used.  A non-complex datatype means
*		 that the real part only will be used.  A
*		 non-floating-point data type means that the integer
*		 part of the real argument will be used.
*
*         Input: object - object to initialize location data on
*		 real   - real part of constant
*		 imag   - imaginary part of constant
*
*        Output: location segment is modified in place
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley
*          Date: Aug 03, 1994 14:53
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_initialize_location(
*                !   kobject object,
*                !   double real,
*                !   double imag)
*
*************************************************************/
int
kpds_initialize_location(
   kobject object, 
   double  real, 
   double  imag)
{
   return (kpds_init() && 
	   kdut_initialize_segment(object, KPDS_SEGMENT_LOCATION, real, imag));
}

/************************************************************
*
*  Routine Name: kpds_copy_location_data - copy all location data from
*                                       one object to another object.
*
*       Purpose: This function copies all of the data contained in the
*		 location segment of one object into the location
*		 segment contained in another object.  If the location
*		 segment exists in the destination object, then its
*		 data will be replaced with the data from the source
*		 object.  If the location segment does not exist in
*		 the destination object, it will be created.
*
*		 The location data can be optionally copied through
*		 the presentations of the two data objects.  This
*		 implies that any presentation stages which are
*		 normally invoked during a kpds_get_data on the source
*		 object or a kpds_put_data call on the destination
*		 object will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : axis assignment, position,
*		 and offset.  The resulting copy will be transposed to
*		 be in the default axis ordering.  The resulting copy
*		 will also appear to be shifted by the source position
*		 and offset.
*
*		 The resulting copy will contain data in the most
*		 explicit grid type between the source and the
*		 destination. 
*		
*	   	 For example, if the destination grid type is
*	   	 curvilinear, and the source grid type is uniform, the
*	   	 resulting copy will contain curvilinear data.  However,
*		 if the destination grid type is uniform, and the
*		 source grid type is curvilinear, then the destination
*		 must be curvilinear.  In general, the copy can not
*		 contain less data than is contained in the source.
*
*		 If the destination grid type is not set, then the
*		 copy will preserve the source grid type.
*		
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has a presentation data
*		 type different from the physical data type, and a
*		 presentation size different from the physical size.
*		 The destination object only has a different
*		 presentation and physical data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size and cast to a
*		 new data type.  When being copied into the
*		 destination object, the data will be cast yet again
*		 before it finally is stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the
*		 presentation.  If the data is not copied through the
*		 presentation, the destination data will be a direct
*		 copy of the physical source data.  The physical size
*		 and data type of the location segment will be
*		 reflected in the destination data object from the
*		 source to the destination since they describe the
*		 physical state of the data.  The grid type of
*		 the destination will always match the grid type
*	 	 of the source in this case.
*
*         Input: object1 - the object that serves as a source for the data.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley and Steve Kubica
*          Date: Jan 2, 1995
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_location_data(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_location_data(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   int grid1;
   int grid2;
   int copy_grid;

   if (!kpds_init())
      return FALSE;

   /* -- sanity check -- */
   if (object1 == NULL || object2 == NULL)
   {
      /* -- set invalid object errorno here -- */
      return FALSE;
   }

   /* -- make sure the segment exists in the source -- */
   if (!kdms_query_segment(object1, KPDS_SEGMENT_LOCATION))
   {
      /* -- set cool errno here -- */
      return FALSE;
   }
   
   /* -- much depends on the two grid types ... get them first -- */
   if (!kpds_get_attribute(object1, KPDS_LOCATION_GRID, &grid1) ||
       !kpds_get_attribute(object2, KPDS_LOCATION_GRID, &grid2)) 
      return FALSE;

   if (!copy_through_presentation)
   {
      /* -- want to make sure destination grid type is same as source -- */
      if (kdms_query_segment(object2, KPDS_SEGMENT_LOCATION))
      {
	 /* -- only recreate it if we have to -- */
	 if (grid1 != grid2)
	    if (!kdms_destroy_segment(object2, KPDS_SEGMENT_LOCATION)   ||
		!kpds_set_attribute(object2, KPDS_LOCATION_GRID, grid1) ||
		!kdms_create_segment(object2, KPDS_SEGMENT_LOCATION))
	    return FALSE;
      }
      else
      {
	 if (!kpds_set_attribute(object2, KPDS_LOCATION_GRID, grid1) ||
	     !kdms_create_segment(object2, KPDS_SEGMENT_LOCATION))
	    return FALSE;
      }
	     

      /* -- the apropriate segments now live in the destination -- */
      if (grid1 == KUNIFORM)
	 return kdut_copy_segment_data(object1, object2, KPDS_SEGMENT_UNIFORM);

      if (grid1 == KRECTILINEAR)
	 return (kdut_copy_segment_data(object1, object2,
					KPDS_SEGMENT_WIDTH) ||
		 kdut_copy_segment_data(object1, object2,
					KPDS_SEGMENT_HEIGHT) ||
		 kdut_copy_segment_data(object1, object2,
					KPDS_SEGMENT_DEPTH));
      if (grid1 == KCURVILINEAR || grid1 == KNONE)
	 return kdut_copy_segment_data(object1, object2,
				       KPDS_SEGMENT_LOCATION);

      /* -- well, darn .. we should have done something by now! -- */
      kerror("kappserv", "kpds_copy_location_data", 
	     "Invalid location grid type (%d).", grid1);
      return FALSE;
   }

   /* -- we're off to copy through the presentation -- */

   /* -- destination will result in the highest of the two grid types -- */
   copy_grid = kmax(grid1, grid2);

   /* -- make sure location exists in destination in 'copy_grid' type -- */
   if (!kdms_query_segment(object2, KPDS_SEGMENT_LOCATION))
   {
      if (!kpds_set_attribute(object2, KPDS_LOCATION_GRID, copy_grid) ||
	  !kdms_create_segment(object2, KPDS_SEGMENT_LOCATION))
	 return FALSE;
   }
   else
   {
      /* -- recreate if destination is not the correct grid type -- */
      if (copy_grid != grid2)
      {
	 if (!kdms_destroy_segment(object2, KPDS_SEGMENT_LOCATION)       ||
	     !kpds_set_attribute(object2, KPDS_LOCATION_GRID, copy_grid) ||
	     !kdms_create_segment(object2, KPDS_SEGMENT_LOCATION))
	    return FALSE;
      }
   }

   /* 
    * -- we now know the destination grid is of the "copy_grid" type 
    * -- perform the copy in the "copy_grid" type
    */

   if (copy_grid == KUNIFORM)
      return _copy_uniform(object1, object2);

   if (copy_grid == KRECTILINEAR)
      return _copy_rectilinear(object1, object2);

   if (copy_grid == KCURVILINEAR)
      return _copy_curvilinear(object1, object2);

   /* -- well, darn .. we should have done something by now! -- */
   kerror("kappserv", "kpds_copy_location_data", 
	  "Invalid location grid type (%d).", copy_grid);

   return FALSE;
}

/************************************************************
*
*  Routine Name: kpds_copy_location_attr - copy all location attributes from
*	                                   one object to another object.
*
*       Purpose: This function copies all the location attributes from 
*		 one object to another object.
*
*		 If the destination object does not contain a location
*		 segment, then this function will create a location
*		 segment, and initialize its size and data type
*		 attributes to those in the source object.  If the
*		 location data already exists in the destination
*		 object, then the presentation attributes will be set
*		 to the source object's settings.
*
*		 The destination's physical attributes will change
*		 depending on the coupling of the data objects.  If
*		 the destination is coupled, then the physical
*		 attributes will be changed as well.  Any data
*		 contained in the destination will be cast, rescaled,
*		 or resized to match its new physical attributes.
*
*         Input: source_object - the object that serves as a
*				 source for the attributes.
*
*        Output: destination_object - the object that serves as
*				      a destination for the
*				      operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Jeremy Worley & John Salas
*          Date: Sep 19, 1993 15:43
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_location_attr(
*		!   kobject object1,
*		!   kobject object2)
*
*************************************************************/
int
kpds_copy_location_attr(
   kobject object1,
   kobject object2)
{
   int grid1;
   int grid2;
   

   if (!kpds_init())
      return FALSE;

   /* -- sanity check -- */
   if (object1 == NULL || object2 == NULL)
   {
      /* -- set invalid object errorno here -- */
      return FALSE;
   }

   /* -- make sure the segment exists in the source -- */
   if (!kdms_query_segment(object1, KPDS_SEGMENT_LOCATION))
   {
      /* -- set cool errno here -- */
      return FALSE;
   }
   
   /* -- much depends on the two grid types ... get them first -- */
   if (!kpds_get_attribute(object1, KPDS_LOCATION_GRID, &grid1) ||
       !kpds_get_attribute(object2, KPDS_LOCATION_GRID, &grid2)) 
      return FALSE;
   

   /* -- want to make sure destination grid type is same as source -- */
   if (kdms_query_segment(object2, KPDS_SEGMENT_LOCATION))
   {
      /* -- only recreate it if we have to -- */
      if (grid1 != grid2)
	 if (!kdms_destroy_segment(object2, KPDS_SEGMENT_LOCATION)   ||
	     !kpds_set_attribute(object2, KPDS_LOCATION_GRID, grid1) ||
	     !kdms_create_segment(object2, KPDS_SEGMENT_LOCATION))
	    return FALSE;
   }
   else
   {
      if (!kpds_set_attribute(object2, KPDS_LOCATION_GRID, grid1) ||
	  !kdms_create_segment(object2, KPDS_SEGMENT_LOCATION))
	 return FALSE;
   }

   /* -- the apropriate segments now live in the destination -- */
   if (grid1 == KUNIFORM)
   {
      /* -- uniform data _looks_ like an attribute, so copy it too -- */
      return (kdut_copy_segment_attr(object1, object2, KPDS_SEGMENT_UNIFORM) &&
	      kdut_copy_segment_data(object1, object2, KPDS_SEGMENT_UNIFORM));
   }
   

   if (grid1 == KRECTILINEAR)
      return (kdut_copy_segment_attr(object1, object2,
				     KPDS_SEGMENT_WIDTH) ||
	      kdut_copy_segment_attr(object1, object2,
				     KPDS_SEGMENT_HEIGHT) ||
	      kdut_copy_segment_attr(object1, object2,
				     KPDS_SEGMENT_DEPTH));
	      
   if (grid1 == KCURVILINEAR || grid1 == KNONE)
      return kdut_copy_segment_attr(object1, object2,
				    KPDS_SEGMENT_LOCATION);

   /* -- well, darn .. we should have done something by now! -- */
   kerror("kappserv", "kpds_copy_location_data", 
	  "Invalid location grid type (%d).", grid1);
   return FALSE;
}

/************************************************************
*
*  Routine Name: kpds_copy_location - copy the location segment from
*		                   one object to another.
*
*       Purpose: This function copies all of the attributes and data
*		 contained in the location segment of one object into
*		 the location segment contained in another object.  If
*		 the location segment exists in the destination
*		 object, then its data will be replaced with the data
*		 from the source object.  If the location segment does
*		 not exist in the destination object, it will be
*		 created.
*
*		 All location segment attributes will be copied from
*		 one object to the other.  This includes presentation
*		 attributes, such as position and offset.
*
*		 The location data can be optionally copied through
*		 the presentations of the two data objects.  This
*		 implies that any presentation stages which are
*		 normally invoked during a kpds_get_data on the source
*		 object or a kpds_put_data call on the destination
*		 object will be used for the copy.
*
*		 Any of the following presentation stages can be
*		 invoked on either the source object or the
*		 destination object : casting, scaling, normalizing,
*		 padding, and interpolating.
*		 
*		 The following presentation stages can be invoked on
*		 only the source object : axis assignment, position,
*		 and offset.  The resulting copy will be transposed to
*		 be in the default axis ordering.  The resulting copy
*		 will also appear to be shifted by the source position
*		 and offset.
*
*		 The resulting copy will contain data in the most
*		 explicit grid type between the source and the
*		 destination. 
*		
*	   	 For example, if the destination grid type is
*	   	 curvilinear, and the source grid type is uniform, the
*	   	 resulting copy will contain curvilinear data.  However,
*		 if the destination grid type is uniform, and the
*		 source grid type is curvilinear, then the destination
*		 must be curvilinear.  In general, the copy can not
*		 contain less data than is contained in the source.
*
*		 If the destination grid type is not set, then the
*		 copy will preserve the source grid type.
*
*		 These presentation stages are brought into the data
*		 pipeline by setting the appropriate presentation
*		 attributes on the data objects. Please see the
*		 chapter on polymorphic data services in the data
*		 services manual for more information on the various
*		 presentation attributes.
*
*		 The following example may help in visualizing the
*		 process.  The source object has a presentation data
*		 type different from the physical data type, and a
*		 presentation size different from the physical size.
*		 The destination object only has a different
*		 presentation and physical data type.
*
*		 !	        _______________________
*		 !	       | _____________________ |
*		 !	  _____||______  ---->        ||
*		 !	 |interpolating|              ||
*		 !	  _____||______          _____||______
*		 !	 |___casting___|        |___casting___|
*		 !	   ____||_____            ____||_____
*		 !	  |           |          |           |
*		 !	  |  source   |        	 |destination|
*		 !	  |   data    |        	 |   data    |
*		 !	  |___________|        	 |___________|
*	         !
*
*		 In the above example, the data resulting from the
*		 copy will be interpolated to a new size and cast to a
*		 new data type.  When being copied into the
*		 destination object, the data will be cast yet again
*		 before it finally is stored.
*
*		 If the presentation and physical layers of the
*		 destination object are coupled then the destination
*		 attributes will be ignored and the source
*		 presentation attributes will be propogated to the
*		 destination physical layer.  The copy, in essence,
*		 will be performed only through the source
*		 presentation, with the destination physical and
*		 presentation layers taking on the characteristics of
*		 the source presentation.  By default, output objects
*		 are not coupled, so this behavior is typical.
*		 
*		 The copy need not be performed using the
*		 presentation.  If the data is not copied through the
*		 presentation, the destination data will be a direct
*		 copy of the physical source data.  The physical size
*		 and data type of the location segment will be
*		 reflected in the destination data object from the
*		 source to the destination since they describe the
*		 physical state of the data.  The grid type of
*		 the destination will always match the grid type
*	 	 of the source in this case.
*
*		 This function is equivalent to performing successive
*		 calls to kpds_copy_location_attr and
*		 kpds_copy_location_data.
*
*         Input: object1 - the object that serves as a source for the location
*			   segment.
*
*		 copy_through_presentation - if set to TRUE, the copy will
*			 		     be performed through the
*				  	     presentation of the source 
*					     and destination objects.
*					     if set to FALSE, the copy will
*					     be a direct copy of the physical
*					     data.
*
*        Output: object2 - the object that will serve as a destination for
*			   the copy operation.
*
*       Returns: TRUE (1) if copy was successful, FALSE (0) otherwise
*
*  Restrictions:
*    Written By: Jeremy Worley & Steve Kubica
*          Date: Sep 19, 1993 15:40
*      Verified:
*  Side Effects:
* Modifications:
*
*   Declaration: int kpds_copy_location(
*               !   kobject object1,
*               !   kobject object2,
*		!   int     copy_through_presentation)
*
*************************************************************/
int
kpds_copy_location(
   kobject object1,
   kobject object2,
   int     copy_through_presentation)
{
   return (kpds_init() && 
	   kpds_copy_location_attr(object1, object2) &&
	   kpds_copy_location_data(object1, object2, 
				   copy_through_presentation));
}
