 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Polymophic Model Utilities
   >>>>
   >>>>  Private:
   >>>>                 kaps_get_segment_size()
   >>>>                 kaps_init()
   >>>>                 kaps_call_get_handler()
   >>>>                 kaps_call_set_handler()
   >>>>                 kaps_current_date()
   >>>>                 kaps_legal_data_type()
   >>>>   Static:
   >>>>   Public:
   >>>>             
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

extern int value_token;
extern int mask_token;
extern int location_token;
extern int time_token;
extern int map_token;
extern int width_token;
extern int height_token;
extern int depth_token;

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_segment_dimension
|
|       Purpose: Get the dimension of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj     - the object to query the segment from
|		 segment - the name of the segment to get the 
|			   dimension of.
|        Output: none
|
|       Returns: the dimension of the segment
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int 
kaps_get_segment_dimension(
   kobject  obj,
   char    *segment)
{
   int seg_token = kstring_to_token(segment);
   int dim;

 
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, segment))
      return 0;

   /* determine the dimension according to the given segment */
   if (seg_token == value_token)
      dim = 5;
   else if (seg_token == mask_token)
      dim = 5;
   else if (seg_token == location_token)
      dim = 4;
   else if (seg_token == time_token)
      dim = 1;
   else if (seg_token == map_token)
      dim = 5;
   else if (seg_token == width_token)
      dim = 1;
   else if (seg_token == height_token)
      dim = 1;
   else if (seg_token == depth_token)
      dim = 1;
   else /* -- generic segment case -- */
   {
      if (!kdms_get_attribute(obj, segment, KDMS_DIMENSION, &dim))
	 return 0;
   }

   return dim;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_segment_size
|
|       Purpose: Get the size of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj     - the object to get from
|                segment - the name of the segment to get
|
|        Output: none
|
|       Returns: an internal array containing the mapped size.
|                NOTE : this array is static and will change whenever 
|		 this function is called.
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int *
kaps_get_segment_size(
   kobject  obj,
   char    *segment)
{
   static int  isize[KDMS_MAX_DIM];
   int         seg_token = kstring_to_token(segment);
   int        *size;
   int         dim;
   int         i;

 
   /* verify that the segment is present */
   if (!kdms_query_segment(obj, segment))
      return NULL;

   isize[0] = isize[1] = isize[2] = isize[3] = isize[4] = 0;
   
   /* retrieve the size from the appropriate segment */
   if (seg_token == value_token)
   {
      if (!kpds_get_attribute(obj, KPDS_VALUE_SIZE, 
			      &isize[KWID], &isize[KHGT], 
			      &isize[KDEP], &isize[KTIM], &isize[KELM]))
	 return NULL;
   }
   else if (seg_token == mask_token)
   { 
      if (!kpds_get_attribute(obj, KPDS_MASK_SIZE, 
			      &isize[KWID], &isize[KHGT], 
			      &isize[KDEP], &isize[KTIM], &isize[KELM]))
	 return NULL;
   }
   else if (seg_token == location_token)
   {
      if (!kpds_get_attribute(obj, KPDS_LOCATION_SIZE, 
			      &isize[KWID], &isize[KHGT], 
			      &isize[KDEP], &isize[KDIM]))
	 return NULL;
   }
   else if (seg_token == time_token)
   {
      if (!kpds_get_attribute(obj, KPDS_TIME_SIZE, &isize[0]))
	 return NULL;
   }
   else if (seg_token == map_token)
   {
      if (!kpds_get_attribute(obj, KPDS_MAP_SIZE, 
			      &isize[KMAP_WID], &isize[KMAP_HGT], 
			      &isize[KMAP_DEP], &isize[KMAP_TIM], 
			      &isize[KMAP_ELM]))
	 return NULL;
   }
   else if (seg_token == width_token)
   {
      if (!kpds_get_attribute(obj, KPDS_LOCATION_SIZE, 
			      &isize[0], NULL, NULL, NULL))
	 return NULL;
   }
   else if (seg_token == height_token)
   {
      if (!kpds_get_attribute(obj, KPDS_LOCATION_SIZE, 
			      NULL, &isize[0], NULL, NULL))
	 return NULL;
   }
   else if (seg_token == depth_token)
   {
      if (!kpds_get_attribute(obj, KPDS_LOCATION_SIZE, 
			      NULL, NULL, &isize[0], NULL))
	 return NULL;
   }
   else /* -- generic segment case -- */
   {
      if (!kdms_get_attributes(obj, segment,
			       KDMS_DIMENSION, &dim,
			       KDMS_SIZE, &size,
			       NULL))
	 return NULL;
      
      for (i = 0; i < dim; i++)
	 isize[i] = size[i];
   }

   return &isize[0];
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_segment_position
|
|       Purpose: Get the position of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj       - the object to get from
|                segment   - the name of the segment to get
|
|        Output: none
|
|       Returns: an internal array containing the mapped position.
|                NOTE : this array is static and will change whenever 
|		 this function is called.
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int *
kaps_get_segment_position(
   kobject  obj,
   char    *segment)
{
   int *pos;

   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   /* -- get the position -- */
   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      /* -- treat value as a special case cause it gets hit the most -- */
      if (!kdms_get_attribute(obj, segment, _INTERNAL_POSITION, &pos))
	 return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      int grid = kaps_location_grid(obj);

      switch (grid)
      {
	 case KUNIFORM :
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM,
				    _INTERNAL_POSITION, &pos)) 
	       return FALSE;
	    break;

	 case KRECTILINEAR :
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
				    _INTERNAL_POSITION, &pos)) 
	       return FALSE;
	    break;
	    
	 case KCURVILINEAR :
	 case KNONE:          /* -- backwards compatibility -- */
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_LOCATION,
				    _INTERNAL_POSITION, &pos))
	       return FALSE;

	    break;

	 default:
	    kerror("kappserv", "kaps_get_segment_position", 
		   "Invalid location grid type (%d).", grid);
	    return FALSE;
      }
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_WIDTH) == 0)
   {
      int *rpos;

      /* -- get the relevant part of the rectilinear position -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, &rpos)) 
	 return FALSE;

      pos = &rpos[KWID];
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_HEIGHT) == 0)
   {
      int *rpos;

      /* -- get the relevant part of the rectilinear position -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, &rpos)) 
	 return FALSE;

      pos = &rpos[KHGT];
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_DEPTH) == 0)
   {
      int *rpos;

      /* -- get the relevant part of the rectilinear position -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, &rpos)) 
	 return FALSE;

      pos = &rpos[KDEP];
   }
   else
   {
      /* -- get the position directly -- */
      if (!kdms_get_attribute(obj, segment, _INTERNAL_POSITION, &pos))
	 return FALSE;
   }

   return pos;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_set_segment_position
|
|       Purpose: Set the position of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj       - the object to get from
|                segment   - the name of the segment to get
|		 pos       - position array to set
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int 
kaps_set_segment_position(
   kobject  obj,
   char    *segment,
   int     *pos)
{
   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   /* -- set the position -- */
   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      /* -- treat value as a special case cause it gets hit the most -- */
      if (!kdms_set_attribute(obj, segment, _INTERNAL_POSITION, pos))
	 return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      int grid = kaps_location_grid(obj);

      switch (grid)
      {
	 case KUNIFORM :
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM,
				    _INTERNAL_POSITION, pos)) 
	       return FALSE;
	    break;

	 case KRECTILINEAR :
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
				    _INTERNAL_POSITION, pos)) 
	       return FALSE;
	    break;

	 case KCURVILINEAR :
	 case KNONE:          /* -- backwards compatibility -- */
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_LOCATION,
				    _INTERNAL_POSITION, pos))
	       return FALSE;

	    break;

	 default:
	    kerror("kappserv", "kaps_set_segment_position", 
		   "Invalid location grid type (%d).", grid);
	    return FALSE;
      }
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_WIDTH) == 0)
   {
      int *opos;
      
      /* -- set the relevant part of the rectilinear position -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, &opos)) 
	 return FALSE;

      opos[KWID] = pos[0];

      if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, opos)) 
	 return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_HEIGHT) == 0)
   {
      int *opos;
      
      /* -- set the relevant part of the rectilinear position -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, &opos)) 
	 return FALSE;

      opos[KHGT] = pos[0];

      if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, opos)) 
	 return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_DEPTH) == 0)
   {
      int *opos;
      
      /* -- set the relevant part of the rectilinear position -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, &opos)) 
	 return FALSE;

      opos[KDEP] = pos[0];

      if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_POSITION, opos)) 
	 return FALSE;
   }
   else
   {
      /* -- set the position -- */
      if (!kdms_set_attribute(obj, segment, _INTERNAL_POSITION, pos))
	 return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_segment_offset
|
|       Purpose: Get the offset of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj       - the object to get from
|                segment   - the name of the segment to get
|
|        Output: none
|
|       Returns: an internal array containing the mapped offset.
|                NOTE : this array is static and will change whenever 
|		 this function is called.
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int *
kaps_get_segment_offset(
   kobject  obj,
   char    *segment)
{
   int *off;

   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   /* -- set the offset -- */
   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      /* -- treat value as a special case cause it gets hit the most -- */
      if (!kdms_get_attribute(obj, segment, _INTERNAL_OFFSET, &off))
	 return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      int grid = kaps_location_grid(obj);

      switch (grid)
      {
	 case KUNIFORM :
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM,
				    _INTERNAL_OFFSET, &off)) 
	       return FALSE;
	    break;

	 case KRECTILINEAR :
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
				    _INTERNAL_OFFSET, &off)) 
	       return FALSE;
	    break;
	    
	 case KCURVILINEAR :
	 case KNONE:          /* -- backwards compatibility -- */
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_LOCATION,
				    _INTERNAL_OFFSET, &off))
	       return FALSE;

	    break;

	 default:
	    kerror("kappserv", "kaps_get_segment_offset", 
		   "Invalid location grid type (%d).", grid);
	    return FALSE;
      }
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_WIDTH) == 0)
   {
      int *roff;

      /* -- get the relevant part of the rectilinear offset -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_OFFSET, &roff)) 
	 return FALSE;

      off = &roff[KWID];
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_HEIGHT) == 0)
   {
      int *roff;

      /* -- get the relevant part of the rectilinear offset -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_OFFSET, &roff)) 
	 return FALSE;

      off = &roff[KHGT];
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_DEPTH) == 0)
   {
      int *roff;

      /* -- get the relevant part of the rectilinear offset -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_OFFSET, &roff)) 
	 return FALSE;

      off = &roff[KDEP];
   }
   else
   {
      /* -- get the offset directly -- */
      if (!kdms_get_attribute(obj, segment, _INTERNAL_OFFSET, &off))
	 return FALSE;
   }

   return off;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_set_segment_offset
|
|       Purpose: Set the offset of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj       - the object to get from
|                segment   - the name of the segment to get
|		 off       - offset array to set
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int 
kaps_set_segment_offset(
   kobject  obj,
   char    *segment,
   int     *off)
{
   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   /* -- set the offset -- */
   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      /* -- do value as a special case cause it gets hit the most -- */
      if (!kdms_set_attribute(obj, segment, _INTERNAL_OFFSET, off))
	 return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      int grid = kaps_location_grid(obj);

      switch (grid)
      {
	 case KUNIFORM :
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM,
				    _INTERNAL_OFFSET, off)) 
	       return FALSE;
	    break;

	 case KRECTILINEAR :
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
				    _INTERNAL_OFFSET, off)) 
	       return FALSE;
	    break;
	    
	 case KCURVILINEAR :
	 case KNONE:          /* -- backwards compatibility -- */
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_LOCATION,
				    _INTERNAL_OFFSET, off))
	       return FALSE;

	    break;

	 default:
	    kerror("kappserv", "kaps_set_segment_offset", 
		   "Invalid location grid type (%d).", grid);
	    return FALSE;
      }
   }
   /* -- no code to set rectilinear offset, since it's not autoinc'd -- */
   else 
   {
      /* -- set the offset -- */
      if (!kdms_set_attribute(obj, segment, _INTERNAL_OFFSET, off))
	 return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_segment_inc_state
|
|       Purpose: Get the increment of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj       - the object to get from
|                segment   - the name of the segment to get
|
|        Output: a,b,c   - increment state
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int 
kaps_get_segment_inc_state(
   kobject  obj,
   char    *segment,
   int     *a,
   int     *b,
   int     *c)
{
   int ra, rb, rc;

   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      /* -- do value as a special case cause it gets hit the most -- */
      if (!kdms_get_attribute(obj, segment, 
			      _INTERNAL_INC_STATE, &ra, &rb, &rc))
	 return FALSE;
   } 
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      int grid = kaps_location_grid(obj);

      switch (grid)
      {
	 case KUNIFORM :
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_UNIFORM,
				    _INTERNAL_INC_STATE, &ra, &rb, &rc)) 
	       return FALSE;
	    break;

	 case KRECTILINEAR :
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
				    _INTERNAL_INC_STATE, &ra, &rb, &rc)) 
	       return FALSE;
	    break;
	    
	 case KCURVILINEAR :
	 case KNONE:          /* -- backwards compatibility -- */
	    if (!kdms_get_attribute(obj, KPDS_SEGMENT_LOCATION,
				    _INTERNAL_INC_STATE, &ra, &rb, &rc)) 
	       return FALSE;

	    break;

	 default:
	    kerror("kappserv", "kaps_get_segment_inc_state", 
		   "Invalid location grid type (%d).", grid);
	    return FALSE;
      }
   }
   else if ((kstrcmp(segment, KPDS_SEGMENT_WIDTH)  == 0) ||
	    (kstrcmp(segment, KPDS_SEGMENT_HEIGHT) == 0) ||
	    (kstrcmp(segment, KPDS_SEGMENT_DEPTH)  == 0))
   {
      /* -- get all rectilinear autoincrement from width -- */
      if (!kdms_get_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_INC_STATE, &ra, &rb, &rc)) 
	 return FALSE;
   }
   else
   {
      /* -- get the increment state directly -- */
      if (!kdms_get_attribute(obj, segment, 
			      _INTERNAL_INC_STATE, &ra, &rb, &rc))
	 return FALSE;
   }

   if (a) *a = ra;
   if (b) *b = rb;
   if (c) *c = rc;

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_set_segment_inc_state
|
|       Purpose: Set the increment state of a segment while taking the 
|                mapping mode and location grid into consideration.
|
|         Input: obj       - the object to get from
|                segment   - the name of the segment to get
|		 a, b, c   - the increment state
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 27, 1994 21:34
|
------------------------------------------------------------*/
int 
kaps_set_segment_inc_state(
   kobject  obj,
   char    *segment,
   int      a,
   int      b,
   int      c)
{
   /* -- verify that the segment is present -- */
   if (!kdms_query_segment(obj, segment))
      return FALSE;

   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      /* -- do value as a special case cause it gets hit the most -- */
      if (!kdms_set_attribute(obj, segment, 
			      _INTERNAL_INC_STATE, a, b, c))
	 return FALSE;
   } 
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      int grid = kaps_location_grid(obj);

      switch (grid)
      {
	 case KUNIFORM :
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_UNIFORM,
				    _INTERNAL_INC_STATE, a, b, c))
	       return FALSE;
	    break;

	 case KRECTILINEAR :
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
				    _INTERNAL_INC_STATE, a, b, c))
	       return FALSE;
	    break;
	    
	 case KCURVILINEAR :
	 case KNONE:          /* -- backwards compatibility -- */
	    if (!kdms_set_attribute(obj, KPDS_SEGMENT_LOCATION,
				    _INTERNAL_INC_STATE, a, b, c))
	       return FALSE;

	    break;

	 default:
	    kerror("kappserv", "kaps_set_segment_position", 
		   "Invalid location grid type (%d).", grid);
	    return FALSE;
      }
   }
   else if ((kstrcmp(segment, KPDS_SEGMENT_WIDTH)  == 0) ||
	    (kstrcmp(segment, KPDS_SEGMENT_HEIGHT) == 0) ||
	    (kstrcmp(segment, KPDS_SEGMENT_DEPTH)  == 0))
   {
      /* -- get all rectilinear autoincrement from width -- */
      if (!kdms_set_attribute(obj, KPDS_SEGMENT_WIDTH,
			      _INTERNAL_INC_STATE, a, b, c)) 
	 return FALSE;
   }
   else
   {
      /* -- set the internal increment state -- */
      if (!kdms_set_attribute(obj, segment, 
			      _INTERNAL_INC_STATE, a, b, c))
	 return FALSE;
   }

   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_get_attributes
|
|       Purpose: This routine provides a mechanism for easily 
|		 getting attributes of polymorphic data segments.
|
|		 It encapsulates the complexity of dealing with
|		 mapping mode, as well the different types of 
|		 location grids.  
|
|		 The routine is called with a NULL terminated variable
|		 argument list of attribute flags and pointer with which
|		 to return those attributes.
|
|		 This routine should not be used for the size,
|		 position, offset, increment state, data type,
|		 or dimension attributes.   
|		 
|		 This routine assumes that the requested segment exists.
|
|         Input: obj     - the object to get attributes from
|
|		 kvalist - a NULL terminated variable argument list 
|			   containing a list of attribute flags, and
|			   attribute values.	
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
int
kaps_get_attributes(
   kobject obj,
   char   *segment,
   kvalist)
{
   char    *attribute;
   kva_list list;

   kva_start(list, segment);

   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      char *seg = KPDS_SEGMENT_VALUE;
      
      /* -- if mapping mode is on and we have map, then operate on map -- */
      if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP))
	 seg = KPDS_SEGMENT_MAP;
      
      /* -- go through the variable argument list -- */
      while ((attribute = kva_arg(list, char *)) != NULL)
	 if (!kdms_vget_attribute(obj, seg, attribute, &list))
	    return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      switch (kaps_location_grid(obj))
      {
	 case KUNIFORM:
	 {
	    while ((attribute = kva_arg(list, char *)) != NULL)
	       if (!kdms_vget_attribute(obj, KPDS_SEGMENT_UNIFORM, 
					attribute, &list))
		  return FALSE;
	 } break;
	    
	 case KRECTILINEAR:
	 {
	    /* -- just get attributes from width -- */
	    while ((attribute = kva_arg(list, char *)) != NULL)
	       if (!kdms_vget_attribute(obj, KPDS_SEGMENT_WIDTH, 
					attribute, &list))
		  return FALSE;
	 } break;
	    
	 case KCURVILINEAR:
	 case KNONE:          /* -- backwards compatibility -- */
	 {
	    while ((attribute = kva_arg(list, char *)) != NULL)
	       if (!kdms_vget_attribute(obj, KPDS_SEGMENT_LOCATION, 
					attribute, &list))
		  return FALSE;
	 } break;
	       
	 default:
	    kerror("kappserv", "kaps_get_attributes", 
		   "Invalid location grid type (%d).",
		   kaps_location_grid(obj));
	    return FALSE;
      }
   }
   else  /* -- regular old segment -- */
   {
      /* -- go through the variable argument list -- */
      while ((attribute = kva_arg(list, char *)) != NULL)
	 if (!kdms_vget_attribute(obj, segment, attribute, &list))
	    return FALSE;
   }
   kva_end(list);
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_set_attributes
|
|       Purpose: This routine provides a mechanism for easily 
|		 setting attributes of polymorphic data segments.
|
|		 It encapsulates the complexity of dealing with
|		 mapping mode, as well the different types of 
|		 location grids.  
|
|		 The routine is called with a NULL terminated variable
|		 argument list of attribute flags and values to
|		 assign to those attributes.
|
|		 This routine should not be used for the size,
|		 position, offset, increment state, data type,
|		 or dimension attributes.   
|		 
|		 This routine assumes that the requested segment exists.
|
|         Input: obj     - the object to get attributes from
|
|		 kvalist - a NULL terminated variable argument list 
|			   containing a list of attribute flags, and
|			   attribute values.	
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Jun 01, 1994 16:23
| Modifications:
|
------------------------------------------------------------*/
int
kaps_set_attributes(
   kobject obj,
   char   *segment,
   kvalist)
{
   char     *attribute;
   kva_list  list;

   kva_start(list, segment);

   if (kstrcmp(segment, KPDS_SEGMENT_VALUE) == 0)
   {
      char *seg = KPDS_SEGMENT_VALUE;
      
      /* -- if mapping mode is on and we have map, then operate on map -- */
      if (kaps_mapping_mode(obj) && kdms_query_segment(obj, KPDS_SEGMENT_MAP))
	 seg = KPDS_SEGMENT_MAP;
      
      /* -- go through the variable argument list -- */
      while ((attribute = kva_arg(list, char *)) != NULL)
	 if (!kdms_vset_attribute(obj, seg, attribute, &list))
	    return FALSE;
   }
   else if (kstrcmp(segment, KPDS_SEGMENT_LOCATION) == 0)
   {
      switch (kaps_location_grid(obj))
      {
	 case KUNIFORM:
	 {
	    while ((attribute = kva_arg(list, char *)) != NULL)
	       if (!kdms_vset_attribute(obj, KPDS_SEGMENT_UNIFORM, 
					attribute, &list))
		  return FALSE;
	 } break;
	    
	 case KRECTILINEAR:
	 {
	    /*  -- need three total versions of the vararg list -- */
	    kva_list list_height = list;
	    kva_list list_depth  = list;

	    /* -- set the attribute on width, height, and depth segments -- */
	    while ((attribute = kva_arg(list, char *)) != NULL)

	       if (!kdms_vset_attribute(obj, KPDS_SEGMENT_WIDTH, 
					attribute, &list)         ||
		   !kdms_vset_attribute(obj, KPDS_SEGMENT_HEIGHT, 
					attribute, &list_height)  ||
		   !kdms_vset_attribute(obj, KPDS_SEGMENT_DEPTH, 
					attribute, &list_depth))
		  return FALSE;
	 } break;
	    
	 case KCURVILINEAR:
	 case KNONE:          /* -- backwards compatibility -- */
	 {
	    while ((attribute = kva_arg(list, char *)) != NULL)
	       if (!kdms_vset_attribute(obj, KPDS_SEGMENT_LOCATION, 
					attribute, &list))
		  return FALSE;
	 } break;
	       
	 default:
	    kerror("kappserv", "kaps_get_attributes", 
		   "Invalid location grid type (%d).",
		   kaps_location_grid(obj));
	    return FALSE;
      }
   }
   else  /* -- regular old segment -- */
   {
      /* -- go through the variable argument list -- */
      while ((attribute = kva_arg(list, char *)) != NULL)
	 if (!kdms_vset_attribute(obj, segment, attribute, &list))
	    return FALSE;
   }
   kva_end(list);

   return TRUE;
}

/*
 *    ==================================================================
 *    Utility Routines
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kaps_call_get_handler
|
|       Purpose: This routine allows you to call a get handler
|		 directly and bypass the dms attribute code.
|
|         Input: func       - the get handler to call
|		 obj        - the object to call with 
|		 assoc      - the tokenized asssociation
|		 attr       - the tokenized attribute name
|		 clientData - client data to use on the call
|        Output: kvalist    - variable argument list containing
|			      pointers to variable addresses by
|			      which to return variables
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Steve Kubica
|          Date: Aug 14, 1994 11:50
| Modifications:
|
------------------------------------------------------------*/
int
kaps_call_get_handler(
   int     (*func) PROTO((kobject, int, int, kaddr, kva_list *)),
   kobject obj,
   int     assoc,
   int     attr,
   kaddr   clientData,
   kvalist)
{
   kva_list list;
   int status;
   
   kva_start(list, clientData);
   status = func(obj, assoc, attr, (kaddr) clientData, &list);
   kva_end(list);
   
   return status;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_call_set_handler
|
|       Purpose: This routine allows you to call a set handler
|		 directly and bypass the dms attribute code.
|
|         Input: func       - the set handler to call
|		 obj        - the object to call with 
|		 assoc      - the tokenized asssociation
|		 attr       - the tokenized attribute name
|		 clientData - client data to use on the call
|        Output: kvalist    - variable argument list containing
|			      pointers to variables with which
|			      to set the attributes
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|    Written By: Steve Kubica
|          Date: Aug 14, 1994 11:50
| Modifications:
|
------------------------------------------------------------*/
int
kaps_call_set_handler(
   int     (*func) PROTO((kobject, int, int, kaddr, kva_list *)),
   kobject obj,
   int     assoc,
   int     attr,
   kaddr   clientData,
   kvalist)
{
   kva_list list;
   int status;
   
   kva_start(list, clientData);
   status = func(obj, assoc, attr, (kaddr) clientData, &list);
   kva_end(list);
   
   return status;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_current_date
|
|       Purpose: Creates a string with the current date in it. 
|		 This data is retrieved using strftme calls in a 
|		 "POSIX compliant" way (we checked the book).  If 
|		 it doesn't work on your system, call your OS 
|		 vender and complain ;^)
|
|         Input: buf - buffer to return the date to
|
|        Output: none
|
|       Returns: none
|
|    Written By: Jeremy Worley
|          Date: Mar 08, 1993 14:01
| Modifications:
|
------------------------------------------------------------*/

#define KDMS_DATEFORMAT "%a %b %d %H:%M:%S %Z %Y"

void
kaps_current_date(char *buf)
{
   struct tm *ctm = NULL;
   time_t tt;

   (void)time(&tt);
   ctm = localtime(&tt);
   
   if (ctm == NULL)
      return;

   strftime(buf, KLENGTH, KDMS_DATEFORMAT, ctm);

   return;
}

/*-----------------------------------------------------------
|
|  Routine Name: kaps_legal_data_type
|
|       Purpose: This function determines if the specified 
|		 data type is legal.
|
|         Input: type
|
|        Output: none
|
|       Returns: TRUE (1) if data type is legal, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: May 20, 1993 11:23
| Modifications:
|
------------------------------------------------------------*/
int
kaps_legal_data_type(int datatype)
{
   switch (datatype)
   {
      case KBIT:
      case KBYTE:
      case KUBYTE:
      case KSHORT:
      case KUSHORT:
      case KLONG:
      case KULONG:
      case KINT:
      case KUINT:
      case KFLOAT:
      case KDOUBLE:
      case KCOMPLEX:
      case KDCOMPLEX:
         return (TRUE);
      default:
         return (FALSE);        /* not an error */
   }
}

