 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Geometry Application Services
   >>>>              Primitive Handler Routines 
   >>>>
   >>>>   Static:
   >>>>			_layout_modifier() 
   >>>>			_location_presentation() 
   >>>>			_color_presentation() 
   >>>>			_normal_presentation() 
   >>>>			_tcoord_presentation() 
   >>>>			_radius_presentation() 
   >>>>			_background_color_presentation() 
   >>>>			_string_presentation() 
   >>>>
   >>>>			_location_get() 
   >>>>			_location_put()
   >>>>			_string_get() 
   >>>>			_string_put()
   >>>>
   >>>>			_increment_primitive_position()
   >>>>  Private:
   >>>>			kgeom_define_primitives()
   >>>>			kgeom_get_primitive()
   >>>>			kgeom_put_primitive()
   >>>>
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "geom_internals.h"

static int _kgeom_primitives_initialized = FALSE;

/* 
 *    ================================================================== 
 *    Geometry Custom Data Handlers 
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: (static) _layout_modifier
|
|       Purpose: This routine uses a given number of
|		 vertices and a given primitive and layout
|		 to determine how many points are actually
|		 present.
|		 The modified number is returned
|		
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_layout_modifier(
   int number,
   int primitive,
   int layout)
{
    int new_number = number;


    /* check for each primitive which is affected by layout */
    switch (primitive)
    {
       case KGEOM_POLYLINE_DISJOINT : 
	{
           if (layout == KPER_LINE)  new_number = number / 2;

	} break;
       case KGEOM_POLYLINE_CONNECTED : 
	{
           if (layout == KPER_LINE)  new_number = number - 1;

	} break;
       case KGEOM_TRIANGLES_DISJOINT :
	{
           if (layout == KPER_LINE)  new_number = number;
           if (layout == KPER_FACE)  new_number = new_number / 3;

	} break;
       case KGEOM_TRIANGLES_CONNECTED :
	{
           if (layout == KPER_LINE)  new_number = new_number - 1;
           if (layout == KPER_FACE)  new_number = new_number - 2;

	} break;
       case KGEOM_CYLINDERS :
	{
           if (layout == KPER_LINE)  new_number = new_number / 2;

	} break;
    }
    
    return new_number;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _location_presentation
|
|       Purpose: This routine determines the location
|		 presentation for the current primitive.
|		
|		 size[0] = KGEOM_LOCATION_SIZE;
|		 size[1] = KGEOM_NUMBER_VERTICES;
|		 data_type = KGEOM_LOCATION_DATA_TYPE;
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_location_presentation(
   kobject   obj,
   int       primitive,
   char     *segment,
   int     **size,
   int      *data_type)
{
   int number;
   int loc_size;
   int type;
   static int my_size[2] = {1,1};

   /* -- get the number of vertices for the current primitive -- */
   kgeom_get_attribute(obj, primitive, KGEOM_NUMBER_VERTICES, &number);

   /* -- get the location size for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_LOCATION_SIZE, &loc_size);

   /* -- get the location data type for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_LOCATION_DATA_TYPE, &type);

   my_size[0] = loc_size;
   my_size[1] = number;

   if (size != NULL) *size = my_size;
   if (data_type != NULL) *data_type = type;
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _color_presentation
|
|       Purpose: This routine determines the color
|		 presentation for the current primitive.
|		
|		 size[0] = function of colorspace, maps, and has_alpha
|		 size[1] = function of KGEOM_NUMBER_VERTICES, primitive,
|			   and layout
|
|		 data_type = KGEOM_COLOR_DATA_TYPE
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_color_presentation(
   kobject   obj,
   int       primitive,
   char     *segment,
   int     **size,
   int      *data_type)
{
   int number;
   int type;
   int layout = kgeom_layout(obj);
   static int my_size[2] = {1,1};

   /* -- get the number for the current primitive -- */
   kgeom_get_attribute(obj, primitive, KGEOM_NUMBER_VERTICES, &number);

   /* -- get the location data type for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &type);

   my_size[0] = kgeom_color_size(obj, segment);
   my_size[1] = _layout_modifier(number, primitive, layout);

   if (size != NULL) *size = my_size;
   if (data_type != NULL) *data_type = type;
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _normal_presentation
|
|       Purpose: This routine determines the normals
|		 presentation for the current primitive.
|		
|		 size[0] = KGEOM_LOCATION_SIZE
|		 size[1] = function of KGEOM_NUMBER_VERTICES, primitive,
|			   and layout;
|
|		 data_type = KGEOM_NORMAL_DATA_TYPE;
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_normal_presentation(
   kobject   obj,
   int       primitive,
   char     *segment,
   int     **size,
   int      *data_type)
{
   int number;
   int loc_size;
   int type;
   int layout = kgeom_layout(obj);
   static int my_size[2] = {1,1};

   /* -- get the number for the current primitive -- */
   kgeom_get_attribute(obj, primitive, KGEOM_NUMBER_VERTICES, &number);

   /* -- get the location size for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_LOCATION_SIZE, &loc_size);

   /* -- get the location data type for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_NORMAL_DATA_TYPE, &type);

   my_size[0] = loc_size;
   my_size[1] = _layout_modifier(number, primitive, layout);

   if (size != NULL) *size = my_size;
   if (data_type != NULL) *data_type = type;
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _tcoord_presentation
|
|       Purpose: This routine determines the texture coord
|		 presentation for the current primitive.
|		
|		 size[0] = KGEOM_TEXTURE_COORD_SIZE
|		 size[1] = KGEOM_NUMBER_VERTICES
|
|		 data_type = KGEOM_TEXTURE_COORD_DATA_TYPE;
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_tcoord_presentation(
   kobject   obj,
   int       primitive,
   char     *segment,
   int     **size,
   int      *data_type)
{
   int number;
   int tcoord_size;
   int type;
   static my_size[2] = {1,1};

   /* -- get the number for the current primitive -- */
   kgeom_get_attribute(obj, primitive, KGEOM_NUMBER_VERTICES, &number);

   /* -- get the location size for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, 
		       KGEOM_TEXTURE_COORD_SIZE, &tcoord_size);

   /* -- get the location data type for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, 
		       KGEOM_TEXTURE_COORD_DATA_TYPE, &type);

   my_size[0] = tcoord_size;
   my_size[1] = number;

   if (size != NULL) *size = my_size;
   if (data_type != NULL) *data_type = type;
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _radius_presentation
|
|       Purpose: This routine determines the radius
|		 presentation for the current primitive.
|		
|		 size[0] = 1
|		 size[1] = KGEOM_NUMBER_VERTICES
|
|		 data_type = KGEOM_RADIUS_DATA_TYPE;
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_radius_presentation(
   kobject   obj,
   int       primitive,
   char     *segment,
   int     **size,
   int      *data_type)
{
   int number;
   int type;
   static int my_size[2] = {1,1};


   /* -- get the number for the current primitive -- */
   kgeom_get_attribute(obj, primitive, KGEOM_NUMBER_VERTICES, &number);

   /* -- get the location data type for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, 
		       KGEOM_RADIUS_DATA_TYPE, &type);

   my_size[0] = 1;
   my_size[1] = number;

   if (size != NULL) *size = my_size;
   if (data_type != NULL) *data_type = type;
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _background_color_presentation
|
|       Purpose: This routine determines the background color
|		 presentation for the current primitive.
|		 This will be a single color, of the current colorsize.
|		
|		 size[0] = function of colorspace, maps, and has_alpha
|		 size[1] = 1
|
|		 data_type = KGEOM_COLOR_DATA_TYPE;
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_background_color_presentation(
   kobject   obj,
   int       primitive,
   char     *segment,
   int     **size,
   int      *data_type)
{
   int type;
   static int my_size[2] = {1,1};

   /* -- get the location data type for the given object -- */
   kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &type);

   my_size[0] = kgeom_color_size(obj, segment);
   my_size[1] = 1;

   if (size != NULL) *size = my_size;
   if (data_type != NULL) *data_type = type;
   
   return TRUE;
}

#if 0

/*-----------------------------------------------------------
|  Routine Name: (static) _get_color_component
|       Purpose: This routine gets the data from the color component
|		 according to the size and data type of the 
|		 presentation and maps it through the map
|		 if a map is present.
|    Written By: Steve Kubica
|          Date: Mar 09, 1994 11:03
------------------------------------------------------------*/
/* ARGSUSED */
static kaddr 
_get_color_component(
   kobject obj,
   int     primitive,
   char   *name,
   int    *size,
   int     data_type,
   kaddr   data)
{
   int   begin[2] = {0,0};
   int   end[2]   = {0,0};
   kaddr rdata    = NULL;


   /* sanity check */
   if (name == NULL || size == NULL)
      return NULL;

   /* calculate the end points for the data access */
   end[0] = size[0] - 1;
   end[1] = size[1] - 1;

   /* do we have a map? */
   if (kdms_query_segment(obj, KGEOM_SEGMENT_MAP))
   {
      kaddr map;
      int  *vdata;
      int  *msize;
      int   value_size[5] = {1,1,1,1,1};
      int   map_size[5]   = {1,1,1,1,1};

      int   map_begin[5] = {0,0,0,0,0};
      int   map_end[5]   = {0,0,0,0,0};
      int   type;
      int   total_size;
      int   mapped_size[5] = {1, 1, 1, 1, 1};


      /* need the values to be integers to order to map them */
      kdms_set_attribute(obj, name, KDMS_DATA_TYPE, KINT);
      
      /* get the unmapped value data */
      vdata = (int *) kdms_get_data(obj, name, begin, end, data);   

      /* get the map size -- we want only the first w x h plane of map data */
      msize = kaps_get_segment_size(obj, KGEOM_SEGMENT_MAP);
      map_end[0] = msize[0] - 1;
      map_end[1] = msize[1] - 1;

      /* get the color type */
      kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_COLOR_DATA_TYPE, &type);

      /* set the map to be the color type */
      kdms_set_attribute(obj, KGEOM_SEGMENT_MAP, KDMS_DATA_TYPE, type);
      
      /* get the current colormap */
      map = kdms_get_data(obj, KGEOM_SEGMENT_MAP, map_begin, map_end, NULL);

      /* mock up a 5D size for the value size and the map size */
      value_size[0] = 1;  /* number of values */
      value_size[4] = end[0] + 1;  /* vector size */

      map_size[0] = msize[0];
      map_size[1] = msize[1];
      
      mapped_size[0] = end[1] + 1;  /* number of values */
      mapped_size[4] = end[0] + 1;  /* vector size */

      total_size = mapped_size[0] * mapped_size[1] * mapped_size[2] * 
	           mapped_size[3] * mapped_size[4];

      /* 
       * the mapped data will be ths number of mapped color points times
       * the size of the map data type -- allocate this if we need to 
       */
      if (data == NULL)
	 rdata = kmalloc((unsigned)total_size * kdata_size(type));
      else
	 rdata = data;

      /* do the actual mapping */
      kaps_map_data(mapped_size, vdata, value_size, map, map_size, type, 
		    total_size, rdata, 0.0, 0.0);
      kfree(map);
   }
   else /* just go get the data */
      rdata = kdms_get_data(obj, name, begin, end, data);   
   
   return rdata;
}

#endif

/*-----------------------------------------------------------
|
|  Routine Name: (static) _string_presentation
|
|       Purpose: This routine determines the string
|		 presentation for the current primitive.
|		
|		 size[0] = 1
|		 size[1] = KGEOM_NUMBER_VERTICES
|
|		 data_type = KSTRING
|
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_string_presentation(
   kobject   obj,
   int       primitive,
   char     *segment,
   int     **size,
   int      *data_type)
{
   int number;
   static int my_size[2] = {1,1};

   /* -- get the number for the current primitive -- */
   kgeom_get_attribute(obj, primitive, KGEOM_NUMBER_VERTICES, &number);

   my_size[0] = 1;
   my_size[1] = number;

   if (size != NULL) *size = my_size;
   if (data_type != NULL) *data_type = KSTRING;
   
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _location_get
|
|       Purpose: This routine gets data from a location data 
|		 component given the storage information as 
|		 well as the current presentation size and data type.
|
|		 The storage information consists of the stored
|		 size in terms of element size and number of points
|		 along with the begin point which marks where the 
|		 desired data is located along the second dimension 
|		 in the segment.
|
|         Input: obj       - object to be synced to presentation
|		 segment   - segment to be synced to presentation 
|		 position  - the primitive position of the primitive 
|			     which contains component to be synced
|		 component - the component number within the primitive
|
|		 The following info is from the presentation function :
|
|		   pres_size - array of two numbers indicating the 
|			       presentation element size and number
|			       of points
|		   data_type - presentation data type 
|
|		 The following info is from the primitive manifest :
|		 
|		   size        - array of two numbers indicating the 
|			         stored element size and number
|			         of points
|        	   begin_point - the physical begining index 
|				 along the second dimension
|
|		 data  - a pointer to already allocated memory in which 
|			 to return the data being retrieved.  If this
|		 	 is NULL, then new space will be allocated.
|
|       Returns: the requested data component
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1995 15:29
| Modifications:
------------------------------------------------------------*/
/* ARGSUSED */
static kaddr 
_location_get(
   kobject obj,
   char   *segment,
   int     position,
   int     component,
   int    *pres_size, 
   int     data_type,
   int    *size,      
   int     begin_point,
   kaddr   data)        
{
   int  *internal_size;
   int   old_overall_size[KGEOM_MAX_DIM];
   int   new_overall_size[KGEOM_MAX_DIM];
   float factor;
   
   int   beg[KGEOM_MAX_DIM] = {0,0,0,0};
   int   end[KGEOM_MAX_DIM] = {0,0,0,0};
   int   pres_begin_point;
   
   kaddr rdata    = NULL;

   /* -- sanity check -- */
   if (segment == NULL || size == NULL)
      return NULL;

   /* == handle presentation issues : size and data type == */

   /* -- need to first get the old overall size -- */
   if (!kdms_get_attribute(obj, segment, KDMS_SIZE, &internal_size))
      return NULL;

   /* -- spin off a few working copies of the internal size array -- */
   kmemcpy(old_overall_size, internal_size, KGEOM_MAX_DIM * sizeof(int));
   kmemcpy(new_overall_size, internal_size, KGEOM_MAX_DIM * sizeof(int));

   /*   
    *   trickiness in presentation size change will be along the 
    *   number of points since that is where sharing occurs across
    *   different primitives.   
    *
    *   anyway, here's the basic idea : 
    *
    *      factor = presentation size[1] / manifest size[1]
    *
    *      (we are transposing location, hence the '0')
    *      new overall_size[0] = old overall_size[0] * factor
    *      new begin point     = old begin point * factor
    *
    */
   factor = pres_size[1] / size[1];

   /* -- determine the appropriate presentation size -- */
   new_overall_size[0] = old_overall_size[0] * factor;

   /* -- modify the begin point so it indexes into the new size -- */
   pres_begin_point = begin_point * factor;

   /* -- set the presentation size and data type on the segment -- */
   if (!kdms_set_attributes(obj, segment, 
			    KDMS_SIZE,      new_overall_size,
			    KDMS_DATA_TYPE, data_type, NULL))
      return NULL;


   /* ===  retreive the data from the segment  === */

   /* -- calculate the begin and end points for the data access -- */
   end[3] = pres_size[0] - 1;

   /* -- use the begin and end corners dictated by prior allocation -- */
   beg[0] = pres_begin_point;
   end[0] = pres_begin_point + pres_size[1] - 1;

   /* -- retrieve the data in transposed form -- */
   kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, TRUE);
   rdata = kdms_get_data(obj, segment, beg, end, data);   
   kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, FALSE);

   /* -- return the presentation size to normal -- */
   if (!kdms_set_attribute(obj, segment, KDMS_SIZE, old_overall_size))
      return NULL;

   return rdata;
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _location_put
|
|       Purpose: This routine puts data into a data component
|		 according to the size and data type of a given
|		 presentation.   The space in which to put
|		 the data is allocated by this routine.
|	 	 If the segment in which to store the component
|		 does not yet exist, it will be created by this 
|		 routine.
|		
|		 The assumptions are made that the allocated 
|		 segment will be at least two dimensional, 
|		 that the first dimension will be the element size,
|		 and that the second dimension will mark the direction
|		 for allocation.  This routine forces all other 
|		 dimensions to be of size one.
|
|		 The begin point marking where the allocated space 
|		 is located along the second dimension will be returned.
|
|         Input: obj       - object to put component data into
|		 segment   - segment to put component data into
|		 position  - the primitive position of the primitive 
|			     which will be containing this component
|		 component - the component number within the primitive
|		 size      - array of two numbers indicating the element
|			     size and the number of points to be allocated
|		 data_type - data type to the data is in the segment to
|		 data      - generic pointer to the data to be stored into 
|			     the component
|
|        Output: begin_point - the begining index along the second dimension
|			       marking where the data has been stored
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1995 15:29
| Modifications:
------------------------------------------------------------*/
/* ARGSUSED */
static int 
_location_put(
   kobject obj,
   char   *segment,
   int     position,
   int     component,
   int    *size,
   int     data_type,
   int    *begin_point,
   kaddr   data)
{
   int  beg[KGEOM_MAX_DIM] = {0,0,0,0};
   int  end[KGEOM_MAX_DIM] = {0,0,0,0};
   int  new_size[KGEOM_MAX_DIM] = {1, 1, 1, 1};
   int *old_size;
   int  used;
   
   /* -- sanity check -- */
   if (segment == NULL || size == NULL)
      kgeom_fail_error(KGEOM_EINVALID_PRESENTATION);

   /* ===  allocate space in the segment for this data  === */

   /* -- if segment does not yet exist, then create it -- */
   if (!kdms_query_segment(obj, segment))
   {
      if (!kpds_create_location(obj))
	 return FALSE;

      if (!kdms_create_attribute(obj, segment, KGEOM_USED, 1, 1,
				 KINT, TRUE, TRUE))
	 return FALSE;
   }

   /* -- get old size with assumption that dimension is at least two -- */
   if (!kdms_get_attribute(obj, segment, KDMS_SIZE, &old_size))
      return FALSE;

   /* -- get old used space -- */
   if (!kdms_get_attribute(obj, segment, KGEOM_USED, &used))
      return FALSE;

   /* -- see if it is necessary to allocate new space -- */
   if (used + size[1] >= old_size[0])
   {
      /* 
       *  this appears confusing, so read carefully :
       *
       *  what we are wanting to do is incrment the height dimension
       *  by a number larger than the amount of space we actually need
       *  so we can avoid allocating new space all the time.  to be
       *  clever, we will determine a number that is a factor of 2048
       *  which is greater than the amount of space we need.  the
       *  space we need, recall, is stored in size[1];
       *
       *  recall that 2^11 = 2048, and you'll see that the functional
       *  equivalent of this equation is : ((size[1]/2048) + 1) * 2048
       *
       */

      /* -- the 'allocation' will occur along height -- */
/*       new_size[0] = old_size[0] + ((size[1]/10000)+1)*10000;   */
      new_size[0] = old_size[0] + size[1];   
      new_size[3] = size[0];               /* element size */   

      /* -- set the new size -- */
      if (!kdms_set_attribute(obj, segment, KDMS_SIZE, new_size))
	 return FALSE;
   }
   
   /* -- return the begin point from the allocation -- */
   *begin_point = used;

   /* -- set the new data type -- */
   if (!kdms_set_attribute(obj, segment, KDMS_DATA_TYPE,  data_type))
      return FALSE;

   /* ===  store the data in the allocated space  === */

   /* -- calculate the begin and end points for the data access -- */
   end[3] = size[0] - 1;

   /* -- use the begin and end corners dictated by prior allocation -- */
   beg[0] = *begin_point;
   end[0] = *begin_point + size[1] - 1;

   /* -- used space increases by the amount being put -- */
   used += size[1];
   if (!kdms_set_attribute(obj, segment, KGEOM_USED, used))
      return FALSE;

   /* -- put the data in transposed form -- */
   kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, TRUE);
   kdms_put_data(obj, segment, beg, end, data);   
   kpds_set_attribute(obj, KPDS_ELEMENTS_FIRST, FALSE);

   return TRUE;

}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _string_get
|
|       Purpose: This routine retrieves string data.
|		
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static kaddr
_string_get(
   kobject obj,
   char   *segment,
   int     position,
   int     component,
   int    *pres_size, 
   int     data_type,
   int    *size,      
   int     begin_point,
   kaddr   data)        
{
   char **strings = NULL;
   char   name[KLENGTH];
   kaddr  copy;
   int    overall_size = 0;
   int    i;

   /* -- sanity check -- */
   if (segment == NULL || size == NULL)
      return NULL;

   /* 
    *   create name for attribute :
    *
    *    \-- segment --/\-- primitive position --/\-- component number --/
    */
   ksprintf(name, "%s.%d.%d\0", segment, position, component);

   if (!kdms_get_attribute(obj, NULL, name, &strings))
      return NULL;

   if (strings == NULL)
      return NULL;

   /* -- figure out how much data we've really got -- */
   for (i = 0; i < size[1]; i++)
      overall_size += kstrlen(strings[i]);

   /* -- return data via new copy, or in place if they gave us a pointer -- */
   if (data == NULL)
   {
         copy = (char *) kcalloc(overall_size, kdata_size(KBYTE));
         kmemcpy(copy, strings, overall_size);
         return copy;
   }
   else
      kmemcpy(data, strings, overall_size);

   return data; 
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _string_put
|
|       Purpose: This routine stores string data.
|		
|    Written By: Steve Kubica
|          Date: Feb 16, 1994 15:48
| Modifications:
|
------------------------------------------------------------*/
/* ARGSUSED */
static int
_string_put(
   kobject obj,
   char   *segment,
   int     position,
   int     component,
   int    *size,
   int     data_type,
   int    *begin_point,
   kaddr   data)
{
   char name[KLENGTH];
   
   /* -- sanity check -- */
   if (segment == NULL || size == NULL)
      return FALSE;

   /* 
    *   create name for attribute :
    *
    *    \-- segment --/\-- primitive position --/\-- component number --/
    */
   ksprintf(name, "%s.%d.%d\0", segment, position, component);

   /* -- allocate space for this data by creating a string attribute -- */
   if (kdms_query_attribute(obj, NULL, name, NULL, NULL, NULL, NULL))
      kdms_destroy_attribute(obj, NULL, name);

   kdms_create_attribute(obj, NULL, name, 1, size[1], KSTRING, TRUE, TRUE);

   return kdms_set_attribute(obj, NULL, name, (char **) data);
}

/* 
 *    ================================================================== 
 *    Geometry Primitive Definitions
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_define_primitives
|
|       Purpose: This routine makes the specific define primitive
|                calls for definining geometry services primitives.
|                These primitives are defined for the session.
|
|         Input: None
|
|        Output: None
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Mar 09, 1994 11:03
| Modifications:
|
------------------------------------------------------------*/
int kgeom_define_primitives(void)
{
   int status = TRUE;
 
   if (_kgeom_primitives_initialized)
      return TRUE;

   /* Prevent this routine from being called again */
   _kgeom_primitives_initialized = TRUE;


   /* KGEOM_POLYLINE_DISJOINT */
   status &= kgeom_define_primitive(KGEOM_POLYLINE_DISJOINT,
                		    KGEOM_POLYLINE_DISJOINT_STRING, 2,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL);

   /* KGEOM_POLYLINE_CONNECTED */
   status &= kgeom_define_primitive(KGEOM_POLYLINE_CONNECTED,
                		    KGEOM_POLYLINE_CONNECTED_STRING, 2,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation,
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL);

   /* KGEOM_TRIANGLES_DISJOINT */
   status &= kgeom_define_primitive(KGEOM_TRIANGLES_DISJOINT,
                		    KGEOM_TRIANGLES_DISJOINT_STRING, 4,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_NORMAL, FALSE, 
				    _normal_presentation, NULL, NULL,
			KGEOM_SEGMENT_TEXTURE_COORD, FALSE, 
				    _tcoord_presentation, NULL, NULL);

   /* KGEOM_TRIANGLES_CONNECTED */
   status &= kgeom_define_primitive(KGEOM_TRIANGLES_CONNECTED,
                		    KGEOM_TRIANGLES_CONNECTED_STRING, 4,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_NORMAL, FALSE, 
				    _normal_presentation, NULL, NULL,
			KGEOM_SEGMENT_TEXTURE_COORD, FALSE, 
				    _tcoord_presentation, NULL, NULL);


   /* KGEOM_SPHERES */
   status &= kgeom_define_primitive(KGEOM_SPHERES,
                		    KGEOM_SPHERES_STRING, 3,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_RADIUS, FALSE, 
				    _radius_presentation, NULL, NULL);
   

   /* KGEOM_CYLINDERS */
   status &= kgeom_define_primitive(KGEOM_CYLINDERS,
                		    KGEOM_CYLINDERS_STRING, 4,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_RADIUS, FALSE, 
				    _radius_presentation, NULL, NULL,
			KGEOM_SEGMENT_RADIUS, FALSE, 
				    _radius_presentation, NULL, NULL);

   /* KGEOM_DIRECTED_POINTS */
   status &= kgeom_define_primitive(KGEOM_DIRECTED_POINTS,
                		    KGEOM_DIRECTED_POINTS_STRING, 3,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_NORMAL, FALSE, 
				    _normal_presentation, NULL, NULL);

   /* KGEOM_TEXT */
   status &= kgeom_define_primitive(KGEOM_TEXT,
                		    KGEOM_TEXT_STRING, 8,
                        KGEOM_SEGMENT_STRING, TRUE,
				    _string_presentation, 
				    _string_get, _string_put,
			KGEOM_SEGMENT_STRING, FALSE,
				    _string_presentation, 
				    _string_get, _string_put,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_LOCATION, FALSE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_LOCATION, FALSE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_LOCATION, FALSE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL);

   /* KGEOM_MARKERS */
   status &= kgeom_define_primitive(KGEOM_MARKERS,
                		    KGEOM_MARKERS_STRING, 3,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL);


   /* KGEOM_POLYGON */
   status &= kgeom_define_primitive(KGEOM_POLYGON,
                		    KGEOM_POLYGON_STRING, 3,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_COLOR, FALSE, 
				  _background_color_presentation, NULL, NULL);

   /* KGEOM_RECTANGLES */
   status &= kgeom_define_primitive(KGEOM_RECTANGLES,
                		    KGEOM_RECTANGLES_STRING, 3,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL);


   /* KGEOM_ELLIPSES */
   status &= kgeom_define_primitive(KGEOM_ELLIPSES,
                		    KGEOM_ELLIPSES_STRING, 5,
			KGEOM_SEGMENT_LOCATION, TRUE, 
				    _location_presentation, 
				    _location_get, _location_put,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_COLOR, FALSE, 
				    _color_presentation, NULL, NULL,
			KGEOM_SEGMENT_RADIUS, FALSE, 
				    _radius_presentation, NULL, NULL,
			KGEOM_SEGMENT_RADIUS, FALSE, 
				    _radius_presentation, NULL, NULL);


   /* KGEOM_QUADMESH */
   status &= kgeom_define_primitive(KGEOM_QUADMESH,
                		    "quadmesh", 0);
   /* KGEOM_OCTMESH */
   status &= kgeom_define_primitive(KGEOM_OCTMESH,
                		    "octmesh", 0);
   /* KGEOM_TEXTURE2D */
   status &= kgeom_define_primitive(KGEOM_TEXTURE2D,
                		    "texture2d", 0);
   /* KGEOM_TEXTURE3D */
   status &= kgeom_define_primitive(KGEOM_TEXTURE3D,
                		    "texture3d", 0);

   return status;
}

/* 
 *    ================================================================== 
 *    Primitive List AutoIncrement Code
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: _increment_primitive_position
|
|       Purpose: This function is used for autoincrementing
|		 the primitive position.
|
|         Input: object   - object containing primitive list 
|                position - the current primtive position
|
|        Output: none
|       Returns: TRUE on success, FALSE otherwise 
|
|    Written By: Steve Kubica
|          Date: Apr 09, 1993 13:07
| Modifications:
|
-----------------------------------------------------------*/
static int
_increment_primitive_position(
   kobject         obj,
   kgeom_manifest *manifest,
   int             position)
{
   int state;
   int number;

   /* -- get the number of primtives and the increment state -- */
   if (!kdms_get_attribute(obj, NULL, _INTERNAL_GEOM_INC_STATE, &state))
      kgeom_fail_error(KGEOM_EINTERNAL);

   number = kgeom_get_manifest_number_primitives(manifest);

   /*  if it is okay to increment, then try to -- 
    * 		fail if it would take us out of range 
    */
   if (state == KGEOM_INC_STATE_OK)
   { 
      if (position + 1 < number)
         position++;
      else
         kdms_set_attribute(obj, NULL,
                            _INTERNAL_GEOM_INC_STATE, KGEOM_INC_STATE_FAIL);
   }

   /* -- see if the list was resized recently, if it was nudge it out -- */
   else if (state == KGEOM_INC_STATE_FAIL) 
   {
      if (position + 1 < number)
         position++;
   }

   /* it's been nudged during a resize -- 
    * 	don't increment now, but say it's okay to increment next time 
    */ 
   else if (state == KGEOM_INC_STATE_NUDGE)
      kdms_set_attribute(obj, NULL,
                            _INTERNAL_GEOM_INC_STATE, KGEOM_INC_STATE_OK);

   /* -- set the new position -- */
   kdms_set_attribute(obj, NULL, _INTERNAL_PRIMITIVE_POSITION, position);
   
   return TRUE;
}

/* 
 *    ================================================================== 
 *    Primitive Level get and put Calls
 *    ==================================================================
 */

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_get_primitive
|
|       Purpose: This routine gets a primitive.  The autoincrementing
|                functionality is invoked here.  This routine
|                can handle all regularly defined primitives.  
|
|         Input: obj       - the object to retrieve the primitive from
|                primitive - the primitive to retrieve
|                kva_list  - already started variable argument list of 
|			     pointers by which to return the data
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
int
kgeom_get_primitive(
   kobject obj,
   int     primitive,
   kva_list *list)
{
   kgeom_primitive_defin *defin = NULL;
   kgeom_manifest        *manifest;
   
   int    data_type;
   int    present;
   int    begin_point;
   int    pos;
   int    i;
   int   *pres_size = NULL;
   int   *size      = NULL;
   kaddr *data      = NULL;


   /* -- get the primitive list manifest -- */
   if ((manifest = kgeom_get_manifest(obj)) == NULL)
      return FALSE;  /* errno was set during the get */

   /* -- get the primitive definition -- */
   if ((defin = kgeom_locate_primitive_defin(primitive)) == NULL)
      return FALSE; /* errno was set during the locate */

   /* -- verify that primitive exists at current position -- */
   if (!kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_PRIMITIVE_POSITION, &pos))
      kgeom_fail_error(KGEOM_EINTERNAL);

   if (!kgeom_verify_manifest_primitive(manifest, primitive, pos, defin))
      kgeom_fail_error(KGEOM_EINTERNAL);


   /* -- get the primitive data -- */
   for (i = 0; i < defin->num_components; i++)
   {
      /* -- get the component name for the manifest here -- */
      present     = manifest->primitives[pos]->components[i]->present;
      begin_point = manifest->primitives[pos]->components[i]->begin_point;
      size        = manifest->primitives[pos]->components[i]->size;
      
      /* -- determine the component presentation -- */
      defin->components[i]->presentation(obj, primitive,
					 defin->components[i]->segment,
					 &pres_size, &data_type);

      /* -- get the pointer for returning the data for this component -- */
      if ((data = (kaddr *) kva_arg(*list, kaddr *)) != NULL ) 
      {
	 /* -- get the data if it is present --*/
	 if (present)
	    *data = defin->components[i]->get(obj, 
					      defin->components[i]->segment, 
					      pos, i, pres_size, data_type, 
					      size, begin_point, *data);
          else
             *data = NULL;
       }
   }

   /* -- post-increment the primitive position -- */
   _increment_primitive_position(obj, manifest, pos);
  
   return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_put_primitive
|
|       Purpose: This routine puts a primitive.  The autoincrementing
|                functionality is invoked here.  This routine
|                can handle all regularly defined primitives.  
|
|         Input: obj       - the object to store the primitive in
|                primitive - the primitive being put
|                kva_list  - already started variable argument list of 
|			     pointers by which point to the data 
|		 	     being put.
|
|        Output: none
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Steve Kubica
|          Date: Apr 29, 1994 11:17
| Modifications:
|
------------------------------------------------------------*/
int
kgeom_put_primitive(
   kobject   obj,
   int       primitive,
   kva_list *list)
{
   kgeom_primitive_defin *defin = NULL;
   kgeom_manifest        *manifest;
   
   int    data_type;
   int    present;
   int    begin_point;
   int    pos;
   int    i;
   int   *size   = NULL;
   int    status = TRUE;
   kaddr  data   = NULL;


   /* -- get the primitive list manifest -- */
   if ((manifest = kgeom_get_manifest(obj)) == NULL)
      return FALSE;  /* errno was set during the get */

   /* -- get the primitive definition -- */
   if ((defin = kgeom_locate_primitive_defin(primitive)) == NULL)
      return FALSE; /* errno was set during the locate */

   /* -- verify that primitive exists at current position -- */
   if (!kgeom_get_attribute(obj, KGEOM_OBJECT, KGEOM_PRIMITIVE_POSITION, &pos))
      kgeom_fail_error(KGEOM_EINTERNAL);

   if (!kgeom_verify_manifest_primitive(manifest, primitive, pos, defin))
      kgeom_fail_error(KGEOM_EINTERNAL);


   /* -- put the primitive data -- */
   for (i = 0; i < defin->num_components; i++)
   {
       if (defin->components[i]->segment == NULL)
          kgeom_fail_error(KGEOM_EINTERNAL);

       /* -- determine the component presentation -- */
       defin->components[i]->presentation(obj, primitive,
					  defin->components[i]->segment,
					  &size, &data_type);

       /* get the pointer for the data to be put for this component */
       data = (kaddr) kva_arg(*list, kaddr);
       present = (data == NULL) ? FALSE : TRUE;

       /* if this component is required, error if the data is not present */
       if (!present && defin->components[i]->required)
          kgeom_fail_error(KGEOM_EMISSING_REQUIRED_COMPONENT);

       if (present)
          status &= defin->components[i]->put(obj,
					      defin->components[i]->segment, 
					      pos, i, size, data_type,
					      &begin_point, data);

       /* -- store the information in the manifest -- */
       manifest->primitives[pos]->components[i] = 
	  kgeom_construct_component(present, begin_point, size);
   }

   /* -- post-increment the primitive position -- */
   _increment_primitive_position(obj, manifest, pos);
  
   return status;
}

/*-----------------------------------------------------------
|
|  Routine Name: kgeom_close_hook
|
|       Purpose: This routine will write the primitive manifest
|		 into a special attribute so it can be retreived 
|		 later.
|
|         Input: obj 
|			    
|        Output: none
|
|       Returns: 
|
|    Written By: Steve Kubica
|          Date: Feb 04, 1994 15:29
| Modifications:
|
------------------------------------------------------------*/
int kgeom_close_hook(kobject obj)
{
   int status = TRUE;
   int flags;
   
   if (!kdms_get_attribute(obj, NULL, KDMS_FLAGS, &flags))
      return FALSE;
   
   /* -- write the transcript only if this is not an input object -- */
   if (!(flags & KOBJ_READ))
   {
      status = kgeom_write_transcript(obj, kgeom_get_manifest(obj));

      /* -- shrink down any oversized segments -- */

      if (kdms_query_segment(obj, KGEOM_SEGMENT_LOCATION))
      {
	 int sz[4] = {-1, -1, -1, -1};
	 
	 kdms_get_attribute(obj, KGEOM_SEGMENT_LOCATION, KGEOM_USED, &sz[0]);
	 kdms_set_attribute(obj, KGEOM_SEGMENT_LOCATION, KDMS_SIZE, sz);
	 kdms_destroy_attribute(obj, KGEOM_SEGMENT_LOCATION, KGEOM_USED);
      }

      if (kdms_query_segment(obj, KGEOM_SEGMENT_COLOR))
      {
	 int sz[4] = {-1, -1, -1, -1};

	 kdms_get_attribute(obj, KGEOM_SEGMENT_COLOR, KGEOM_USED, &sz[1]);
	 kdms_set_attribute(obj, KGEOM_SEGMENT_COLOR, KDMS_SIZE, sz);
	 kdms_destroy_attribute(obj, KGEOM_SEGMENT_COLOR, KGEOM_USED);
      }

      if (kdms_query_segment(obj, KGEOM_SEGMENT_NORMAL))
      {
	 int sz[4] = {-1, -1, -1, -1};

	 kdms_get_attribute(obj, KGEOM_SEGMENT_NORMAL, KGEOM_USED, &sz[1]);
	 kdms_set_attribute(obj, KGEOM_SEGMENT_NORMAL, KDMS_SIZE, sz);
	 kdms_destroy_attribute(obj, KGEOM_SEGMENT_NORMAL, KGEOM_USED);
      }

      if (kdms_query_segment(obj, KGEOM_SEGMENT_TEXTURE_COORD))
      {
	 int sz[4] = {-1, -1, -1, -1};

	 kdms_get_attribute(obj,KGEOM_SEGMENT_TEXTURE_COORD,KGEOM_USED,&sz[1]);
	 kdms_set_attribute(obj, KGEOM_SEGMENT_TEXTURE_COORD, KDMS_SIZE, sz);
	 kdms_destroy_attribute(obj, KGEOM_SEGMENT_TEXTURE_COORD, KGEOM_USED);
      }
   }
   
   return status;
}
