 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>          Data Services Copying Utilites
   >>>>
   >>>>   Static:
   >>>>             
   >>>>  Private:
   >>>>
   >>>>   Public:
   >>>>			kdut_initialize_segment()
   >>>>			kdut_copy_segment_data()
   >>>>			kdut_copy_segment_attr()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

/************************************************************
*
*  Routine Name: kdut_initialize_segment
*
*       Purpose: This routine will initialize all the data
*		 in a specified segment to the values provided.
*
*         Input: object - the source for the copy operation
*		 segment - name of the segment to initialize
*		 real    - real value to initialize with
*		 imag    - imaginary value to initialize with
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Steve Kubica and Jeremy Worley 
*          Date: Dec 26, 1994
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration:
*
*************************************************************/
int
kdut_initialize_segment(
   kobject object, 
   char   *segment, 
   double  real, 
   double  imag)
{
   kaddr data;
   int  *size;
   int  *unit;
   int   begin[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int   end[KDMS_MAX_DIM];
   int   dim;
   int   numpts;
   int   type;
   int   i;

   if (!kdms_get_attributes(object, segment, 
			    KDMS_OPTIMAL_REGION_SIZE, &unit,
			    KDMS_SIZE, &size,
			    KDMS_DIMENSION, &dim, 
			    KDMS_DATA_TYPE, &type,
			    NULL))
      return FALSE;

   if (type == KBIT && unit[0] % 8 != 0)
      numpts = unit[0] + 8 - (unit[0] % 8);
   else
      numpts = unit[0];
   end[0] = unit[0] - 1;
   
   for (i = 1; i < dim; i++)
   {
      numpts *= unit[i];
      end[i] = unit[i] - 1;
   }
   
   data = kdms_get_data(object, segment, begin, end, NULL);

   kdata_fill(data, numpts, type, real, imag);

   for (i = 0 ; i < unit[dim]; i++)
   {
      kdms_put_data(object, segment, begin, end, data);
      kdut_advance_position(begin, end, size, unit, dim, FALSE);
   }

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdut_copy_segment_data
*
*       Purpose: This routine will copy all the data in a specified
*		 segment from one object to another.  This copy
*		 copies "underneath" the presentation functionality
*		 of data services.  No presentation considerations are 
*		 taken into account with this copy. This copy duplicates 
*		 the segment exactly.  
*
*		 The attribute KDMS_DIMENSION, KDMS_SIZE,
*		 KDMS_DATA_TYPE, and KDMS_INDEX_ORDER, will be copied
*		 from the source to the destination since they
*		 describe the physical state of the data.
*
*		 If the segment does not yet exist in destination
*		 object, it will be created there.
*
*		 Non-transferable segments can not be copied with
*		 this function.
*
*         Input: object1 - the source for the copy operation
*		 object2 - the destination for the copy operation
*		 segment - the name of the segment to copy
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Steve Kubica and Jeremy Worley 
*          Date: Dec 26, 1994
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration:
*
*************************************************************/
int
kdut_copy_segment_data(
   kobject object1,
   kobject object2,
   char   *segment)
{
   kobject src;
   kobject dest;

   int   begin[KDMS_MAX_DIM] = {0, 0, 0, 0, 0};
   int   end[KDMS_MAX_DIM]   = {0, 0, 0, 0, 0};
   int   zero[KDMS_MAX_DIM]  = {0, 0, 0, 0, 0};
   int  *unit;
   int  *size;
   int   dim;
   int   i;
   int   transferable;
   kaddr data = NULL;

   
   /* -- make sure the segment exists in the source -- */
   if (!kdms_query_segment(object1, segment))
   {
      /* -- set cool errno here -- */
      return FALSE;
   }

   /* -- if the segment is not transferable, then bail out -- */
   if (!kdms_get_attribute(object1, segment, KDMS_TRANSFERABLE, &transferable))
      return FALSE;

   if (!transferable)
      return TRUE;
   
   /* -- make sure the segment exists in the destination -- */
   if (!kdms_query_segment(object2, segment))
      if (!kdms_create_segment(object2, segment))
	 return FALSE;


   /* -- create reference we can muck with the presentation at will -- */
   src  = kdms_reference(object1);
   dest = kdms_reference(object2);

   
   /* -- sync source presentation to reflect physical source -- */
   if (!kdms_set_attribute(src, segment, KDMS_COUPLING, KDEMAND) &&
       !kdms_sync(src, segment, KPHYS2PRES))
      return FALSE;

   /* -- want to affect physical layer on the destination -- */
   if (!kdms_set_attribute(dest, segment, KDMS_COUPLING, KCOUPLED))
      return FALSE;

   /* -- migrate over the dimension, size, data type, and index order -- */
   kdms_copy_attributes(src, dest, segment,
			KDMS_DIMENSION, 
			KDMS_SIZE, 
			KDMS_DATA_TYPE,
			KDMS_INDEX_ORDER, NULL);

   /* -- zero out all other presentation attribute before the copy -- */
   if (!kdms_set_attributes(src, segment,
			    KDMS_SCALING, KNONE,
			    KDMS_OFFSET,  zero, NULL))
      return FALSE;
   
   if (!kdms_set_attributes(dest, segment,
			    KDMS_SCALING, KNONE,
			    KDMS_OFFSET,  zero, NULL))
      return FALSE;


   /* -- determine the optimal region size with which to copy -- */
   if (!kdms_get_attributes(src, segment, 
			    KDMS_DIMENSION,           &dim,
			    KDMS_SIZE,                &size,
			    KDMS_OPTIMAL_REGION_SIZE, &unit, NULL))
      return FALSE;

   /* -- initialize the end point for the first put -- */
   for (i = 0; i < dim; i++)
      end[i] = unit[i] - 1;
   
   /* -- perform the copy using iterative gets-n-puts -- */
   for (i = 0; i < unit[dim]; i++)
   {
      data = kdms_get_data(src, segment, begin, end, data);
      kdms_put_data(dest, segment, begin, end, data);

      kdut_advance_position(begin, end, size, unit, dim, FALSE);
   }
   kfree(data);

   /* -- close down the references -- */
   kdms_close(src);
   kdms_close(dest);

   return TRUE;
}

/************************************************************
*
*  Routine Name: kdut_copy_segment_attr
*
*       Purpose: This routine will copy all the attributes in a specified
*		 segment from one object to another.  
*
*		 If the segment does not yet exist in destination
*		 object, it will be created there.
*
*		 Non-transferable segments can not be copied with
*		 this function.
*
*         Input: object1 - the source for the copy operation
*		 object2 - the destination for the copy operation
*		 segment - the name of the segment to copy
*
*        Output: none
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions:
*    Written By: Steve Kubica and Jeremy Worley 
*          Date: Dec 26, 1994
*      Verified:
*  Side Effects:
* Modifications:
*   Declaration:
*
*************************************************************/
int
kdut_copy_segment_attr(
   kobject object1,
   kobject object2,
   char   *segment)
{
   int transferable;

   /* -- make sure the segment exists in the source -- */
   if (!kdms_query_segment(object1, segment))
   {
      /* -- set cool errno here -- */
      return FALSE;
   }

   /* -- if the segment is not transferable, then bail out -- */
   if (!kdms_get_attribute(object1, segment, KDMS_TRANSFERABLE, &transferable))
      return FALSE;

   if (!transferable)
      return TRUE;

#if 0   
   /* -- make sure the segment exists in the destination -- */
   if (!kdms_query_segment(object2, segment))
      if (!kdms_create_segment(object2, segment))
	 return FALSE;
#endif

   return kdms_copy_segment_attributes(object1, segment, object2, segment);
}

