 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";

#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Utilities for the Data Manipulation routines
   >>>>
   >>>>  Private:
   >>>>			return_data_name()
   >>>>			polymorphic_query()
   >>>>			polymorphic_create()
   >>>>			kdmanip_mask_ops()
   >>>>			kdmanip_multi_input_loc()
   >>>>			kdmanip_multi_input_time()
   >>>>   Static:
   >>>>   Public:
   >>>>                 kcombine_to_complex()
   >>>>                 kdata_insert()
   >>>>
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: return_data_name - return the data name
|
|       Purpose: return_data_name looks for the minimum unique 
|                set of characters to identify a data name
|                and returns the abstract data services name
|                for that data.
|
|         Input: str - name passed in 
|
|        Output: 
|
|       Returns: KPDS data name on success, NULL on failure
|
|    Written By: Donna Koechner
|          Date: Mar 02, 1993
| Modifications:
|
------------------------------------------------------------*/

char *return_data_name(
        char *str)
{

   char *lstr = NULL;
   char *return_str = NULL;

   lstr = kstring_lower(str, lstr);

   /* Segment names returned based on the following evaluations:
        value           va
        time            tim
        location        lo
        map             map
        mask            mas  */

   if (kstrstr(lstr, "va") != NULL) 
      return_str = kstring_copy(KDMS_SEGMENT_VALUE,return_str);
   
   else if (kstrstr(lstr, "tim") != NULL) 
      return_str = kstring_copy(KDMS_SEGMENT_TIME,return_str);
   
   else if (kstrstr(lstr, "lo") != NULL) 
      return_str = kstring_copy(KDMS_SEGMENT_LOCATION,return_str);
   
   else if (kstrstr(lstr, "map") != NULL) 
      return_str = kstring_copy(KDMS_SEGMENT_MAP,return_str);
   
   else if (kstrstr(lstr, "mask") != NULL) 
      return_str = kstring_copy(KDMS_SEGMENT_MASK,return_str);

   return(return_str);

}

/*-----------------------------------------------------------
|
|  Routine Name: polymorphic_query - will query to see
|		 if the polymorphic data exists
|
|       Purpose: 
|
|         Input: object - object to query for the data
|		 str    - name passed in 
|
|        Output: 
|
|       Returns: TRUE if it exists, FALSE (0) if it does not exist
|		 or str is not a valid polymorphic data
|
|    Written By: John M. Salas
|          Date: Oct 07, 1993
| Modifications:
|
------------------------------------------------------------*/

int polymorphic_query(
	kobject object,
        char	*str)
{
   char *lstr = NULL;
   lstr = kstring_lower(str, lstr);

   if (kstrstr(lstr, "va") != NULL && kpds_query_value(object) )
   {
      return(TRUE);
   }
   else if (kstrstr(lstr, "map") != NULL && kpds_query_map(object) )
   {
      return(TRUE);
   }
   else if (kstrstr(lstr, "loc") != NULL && kpds_query_location(object) )
   {
      return(TRUE);
   }
   else if (kstrstr(lstr, "time") != NULL && kpds_query_time(object) )
   {
      return(TRUE);
   }
   else if (kstrstr(lstr, "mask") != NULL && kpds_query_mask(object) )
   {
      return(TRUE);
   }

   return(FALSE);
}

/*-----------------------------------------------------------
|
|  Routine Name: polymorphic_create - will create
|		 the polymorphic data
|
|       Purpose: 
|
|         Input: object - object to create the data in
|		 str    - name passed in 
|
|        Output: 
|
|       Returns: TRUE if it exists, FALSE (0) if it can not
|		 create or str is not a valid polymorphic data
|
|    Written By: John M. Salas
|          Date: Oct 07, 1993
| Modifications:
|
------------------------------------------------------------*/

int polymorphic_create(
	kobject object,
        char	*str)
{
   char *lstr = NULL;
   lstr = kstring_lower(str, lstr);

   if (kstrstr(lstr, "va") != NULL && !kpds_query_value(object) )
   {
      if ( !kpds_create_value(object) ) return(FALSE);
      return(TRUE);
   }
   else if (kstrstr(lstr, "map") != NULL && !kpds_query_map(object) )
   {
      if ( !kpds_create_map(object) ) return(FALSE);
      return(TRUE);
   }
   else if (kstrstr(lstr, "loc") != NULL && !kpds_query_location(object) )
   {
      if ( !kpds_create_location(object) ) return(FALSE);
      return(TRUE);
   }
   else if (kstrstr(lstr, "time") != NULL && !kpds_query_time(object) )
   {
      if ( !kpds_create_time(object) ) return(FALSE);
      return(TRUE);
   }
   else if (kstrstr(lstr, "mask") != NULL && !kpds_query_mask(object) )
   {
      if ( !kpds_create_mask(object) ) return(FALSE);
      return(TRUE);
   }

   return(FALSE);
}

/************************************************************
*
*  Routine Name: kcombine_to_complex - combines 2 double arrays into a double complex array
*
*       Purpose: kcombine_to_complex takes a real and imaginary array
*                of type double and returns a kdcomplex array.
*
*         Input: 2 double arrays containing the real and imag parts
*
*        Output: a dcomplex array to be split into real & imag parts
*
*  Restrictions:
*    Written By: Ashish Malhotra
*          Date: Nov 02, 1993 14:03
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/
 
int kcombine_to_complex(kdcomplex **c, int num, double *r, double *i)
{
   int j;
 
   if ((r==NULL) || (i == NULL))
      return (FALSE);
 
   if (c != NULL && *c == NULL)
   {
      if ((*c = (kdcomplex*)kmalloc(num * sizeof (kdcomplex))) == NULL)
         return(FALSE);
   }
 
   for (j = 0; j < num; j++)
   {
       (*c)[j] = kdccomp(r[j],i[j]);
   }
 
   return (TRUE);
}

/*************************************************************
*
*  Routine Name: kdata_insert - insert a value in memory 
*
*       Purpose: This function is used to insert a value into
*                memory. A datatype is used to determine how to
*                interpret the real and imaginary values specified.
*
*         Input: data      - point to the data to be filled
*                num       - location to be inserted into
*                datatype  - data type to fill with
*                real      - real part of data to fill with
*                imaginary - imaginary part of data to fill with
*
*        Output:
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*    Written By: Ashish Malhotra
*          Date: Mar 30, 1994
* Modifications:
*
*************************************************************/

int kdata_insert(
   kaddr  data,
   int    num,
   int    datatype,
   double real,
   double imaginary)
{
        int n = num;

        switch (datatype) {
           case KBIT:
                {
                  unsigned char *d = (unsigned char *)data;
                  unsigned char v;
                  unsigned long nbytes = (num >> 3);

                  v = (real == 0.0) ? 0x00 : 0x01;
                  v = ( v << (num % 8));
                  d[nbytes] |= v ;
                }
                return(TRUE);
           case KBYTE:
                {
                  char *d = (char *)data, v = (char)real;
                  d[n] = v;
                }
                return(TRUE);
           case KUBYTE:
                {
                  unsigned char *d = (unsigned char *)data,
                        v = (unsigned char)real;
                  d[n] = v;
                }
                return(TRUE);
            case KSHORT:
                {
                  short *d = (short *)data, v = (short)real;
                  d[n] = v;
                }
                return(TRUE);
           case KUSHORT:
                {
                  unsigned short *d = (unsigned short *)data,
                        v = (unsigned short)real;
                  d[n] = v;
                }
                return(TRUE);
           case  KINT:
                {
                  int *d = (int *)data, v = (int)real;
                  d[n] = v;
                }
                return(TRUE);
           case KUINT:
                {
                  unsigned int *d = (unsigned int *)data,
                        v = (unsigned int)real;
                  d[n] = v;
                }
                return(TRUE);
            case KLONG:
                {
                  long *d = (long *)data, v = (long)real;
                  d[n] = v;
                }
                return(TRUE);
            case KULONG:
                {
                  unsigned long *d = (unsigned long *)data,
                        v = (unsigned long)real;
                  d[n] = v;
                }
                return(TRUE);
            case KFLOAT:
                {
                  float *d = (float *)data;
                  d[n] = real;
                }
                return(TRUE);
            case KDOUBLE:
                {
                  double *d = (double *)data;
                  d[n] = real;
                }
                return(TRUE);
           case KCOMPLEX:
                {
                  kcomplex *d = (kcomplex *)data,
                        v = kccomp((float)real,(float)imaginary);
                  d[n] = v;
                }
                return(TRUE);
           case KDCOMPLEX:
                {
                  kdcomplex *d = (kdcomplex *)data,
                        v = kdccomp(real,imaginary);
                   d[n] = v;
                }
                return(TRUE);
           default:
                return(FALSE);
        }
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmanip_mask_ops - perform AND, OR, or XOR on data object masks
|
|       Purpose: This function combines the masks of the source
|		 objects according to the operation flag supplied.
|		 Valid operations are:
|		 !  KDMANIP_MASK_AND
|		 !  KDMANIP_MASK_OR
|		 !  KDMANIP_MASK_XOR
|
|		 If no source object has a mask, then kdmanip_mask_ops
|		 returns without modifying dest_obj.
|
|		 If only one source object has a mask, then its mask
|		 is copied to the destination object.
|
|		 If more than one source object has a mask, then the
|		 specified operation is performed on the source mask
|		 data, and stored in dest_obj.  dest_obj mask data,
|		 if it exists, is not included in the operation.
|
|		 Invalid objects (KOBJECT_INVALID) may be passed in.
|
|         Input: operation - defines operation to perform
|                num_src_objs - number of source objects provided in kvalist
|                kvalist - list of source objects (kobject)
|
|        Output: dest_obj - destination object where resulting mask is stored
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions: 
|    Written By: Donna Koechner
|          Date: Jun  8, 1994
|      Verified:
|  Side Effects: Data may be written to mask of dest_object (mask may 
|		 be created if it does not already exist).
| Modifications:
|
------------------------------------------------------------*/

int kdmanip_mask_ops(
   kobject  dest_obj,
   int	    operation,
   int	    num_src_objs,
   kvalist)
{
        kva_list        valist;
        kobject         *objlist;
	int		*mask_exists; 
	int		i, j, k, mask_count=0, error=FALSE;
	int		dw, dh, dd, dt, de;
	int		rgn_w, rgn_h, rgn_d, rgn_t, rgn_e, num_rgns, num_pts;
	unsigned char	*data=NULL, *result=NULL;
	char		*prog="kdmanip_mask_ops";
	char		*lib="kdatamanip";

	/*
	 * Destination object must be valid.
	 */
	if (dest_obj == KOBJECT_INVALID)
	{
	   kerror(lib,prog,"destination object must be a valid object");
	   return(FALSE);
	}

        if ((objlist = (kobject *) kmalloc(num_src_objs * sizeof(kobject))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for objlist");
	   return(FALSE);
	}

        if ((mask_exists = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for mask_exists array");
	   kfree(objlist);
	   return(FALSE);
	}

        kva_start(valist, num_src_objs);

        for (i=0; i<num_src_objs; i++)
	{
	   objlist[i] = kva_arg(valist, kobject);
	   if (objlist[i] == KOBJECT_INVALID)
	      mask_exists[i] = FALSE;
	   else 
	   {
	      if ((mask_exists[i] = kpds_query_mask(objlist[i])) == TRUE)
	         mask_count ++;
	   }
	}

        kva_end(valist);

	/* 
	 *  If none of the objects contain a mask, do nothing and return.
	 */
	if (mask_count == 0)
	{
	   kfree(objlist);
	   kfree(mask_exists);
	   return(TRUE);
	}
		
	/* 
	 *  Get and set attributes of destination object.
	 */
	if (!kpds_query_mask(dest_obj))
	{
	   if (!kpds_create_mask(dest_obj))
	   {
	      kerror(lib,prog,"Unable to create destination object mask.");
	      kfree(objlist);
	      kfree(mask_exists);
	      return(FALSE);
	   }
	}
	if (!kpds_get_attribute(dest_obj, KPDS_MASK_SIZE, &dw,&dh,&dd,&dt,&de))
	{
	   kerror(lib,prog,"Unable to get destination object attributes.");
           kfree(objlist);
           kfree(mask_exists);
           return(FALSE);
	}

	/* 
	 *  If only one object contains a mask, simply copy it to the 
	 *  destination object and return.
	 */
	if (mask_count == 1)
	{
	   for (i=0; i<num_src_objs; i++)
	   {
	      if (mask_exists[i])
	      {
		 if ((objlist[i] = kpds_reference_object(objlist[i])) 
			== KOBJECT_INVALID)
		 {
	   	    kerror(lib,prog,"Unable to reference object %d", i);
	   	    error = TRUE;
		 }
	         if (!kpds_set_attribute(objlist[i], KPDS_MASK_SIZE, 
				dw,dh,dd,dt,de))
	         {
	            kerror(lib,prog,"Unable to set mask size of object %d.",i);
                    kfree(objlist);
                    kfree(mask_exists);
                    return(FALSE);
	         }
	         if (!kpds_copy_mask((objlist[i]), dest_obj, TRUE))
		    error = TRUE;

		 kpds_close_object(objlist[i]);
	         kfree(objlist);
	         kfree(mask_exists);
		 if (error)
		    return(FALSE);
		 else
	            return(TRUE);
	      }
	   }
	}

	if ((dest_obj = kpds_reference_object(dest_obj)) == KOBJECT_INVALID)
	{
	   kerror(lib,prog,"Unable to reference destination object");
	   error = TRUE;
	}

	if (!kpds_set_attribute(dest_obj, KPDS_MASK_DATA_TYPE, KUBYTE))
	{
	   kerror(lib,prog,"Unable to set destination object mask data type.");
	   error = TRUE;
	}
	if (!kpds_get_attributes(dest_obj, 
			KPDS_MASK_OPTIMAL_REGION_SIZE, 
			&rgn_w, &rgn_h, &rgn_d, &rgn_t, &rgn_e, &num_rgns, 
			NULL))
	{
	   kerror(lib,prog,"Unable to get destination object attributes.");
	   error = TRUE;
	}
	if (!kpds_set_attributes(dest_obj, 
		KPDS_MASK_REGION_SIZE, rgn_w, rgn_h, rgn_d, rgn_t, rgn_e, 
		KPDS_MASK_POSITION, 0,0,0,0,0,
		KPDS_MASK_OFFSET, 0,0,0,0,0,
		NULL))
	{
	   kerror(lib,prog,"Unable to set destination object mask attributes.");
	   error = TRUE;
	}

	if (error)
	{
	   kpds_close_object(dest_obj);
	   kfree(objlist);
	   kfree(mask_exists);
	   return(FALSE);
	}

	/* 
	 *  If more than one mask exists, we will need set attributes and
	 *  get and put data.  Therefore, reference all objects that contain 
	 *  masks so that their attributes are not permanently affected.  
	 */
        for (i=0; i<num_src_objs; i++)
	{
	   if (mask_exists[i])
	   {
	      if ((objlist[i] = kpds_reference_object(objlist[i])) 
			== KOBJECT_INVALID)
	      {
	         kerror(lib,prog,"Unable to reference object %d.",i);
		 error = TRUE;
	      }
	      if (!kpds_set_attributes(objlist[i], 
			KPDS_MASK_SIZE, dw, dh, dd, dt, de,
			KPDS_MASK_REGION_SIZE, rgn_w,rgn_h,rgn_d,rgn_t,rgn_e, 
			KPDS_MASK_POSITION, 0,0,0,0,0,
			KPDS_MASK_OFFSET, 0,0,0,0,0,
			KPDS_MASK_DATA_TYPE, KUBYTE,
			NULL))
	      {
	         kerror(lib,prog,"Unable to set attributes for object %d.",i);
		 error = TRUE;
	      }

	      if (error)
	      {
	         kpds_close_object(dest_obj);
	         for (j=0; j<i; j++)
		 {
		    if (mask_exists[j])
		       kpds_close_object(objlist[j]);
	         }
		 kfree(mask_exists);
		 kfree(objlist);
	         return(FALSE);
	      }
	   }
	}
	
	num_pts = rgn_w * rgn_h * rgn_d * rgn_t * rgn_e;

	/*
	 *  Initialize a temporary results array, initial values are
	 *  based on operation being performed (AND - initialize to 1, 
	 *  OR - initialize to 0, XOR - initialize to 0).
	 */

        if ((result=(unsigned char *)kmalloc(num_pts * sizeof(unsigned char))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for result array.");
	   kpds_close_object(dest_obj);
	   for (j=0; j<i; j++)
	   {
	      if (mask_exists[j])
	         kpds_close_object(objlist[j]);
	   }
	   kfree(mask_exists);
	   kfree(objlist);
	   return(FALSE);
	}
	if (operation == KDMANIP_MASK_AND)
	{
	   for (i=0; i<num_pts; i++)
	      result[i] = 1;
	}
	else
	{
	   for (i=0; i<num_pts; i++)
	      result[i] = 0;
	}
	
	for (i=0; i<num_rgns; i++)
	{
	   for (j=0; j<num_src_objs; j++)
	   {
	      if (mask_exists[j])
	      {
	         if ((data = kpds_get_data(objlist[j], KPDS_MASK_REGION, data))
			   == NULL)
		 {
	            kerror(lib,prog,"Unable to get region from object %d.",j);
		    error = TRUE;
		    break;
	 	 }
	         if (error)  break;
		 if (operation == KDMANIP_MASK_AND)
		 {
		    for (k=0; k<num_pts; k++)
		       result[k] = result[k] & data[k]; 
		 }
		 else if (operation == KDMANIP_MASK_OR)
		 {
		    for (k=0; k<num_pts; k++)
		       result[k] = result[k] | data[k]; 
		 }
		 else if (operation == KDMANIP_MASK_XOR)
		 {
		    for (k=0; k<num_pts; k++)
		       result[k] = result[k] ^ data[k]; 
		 }
		 else 
		 {
	            kerror(lib,prog,"Unrecognized operation (%d).",operation);
		    error = TRUE;
		    break;
		 }
	      }
	      if (error)  break;
	   }

	   if (!kpds_put_data(dest_obj, KPDS_MASK_REGION, result))
	   {
	      kerror(lib,prog,"Unable to get region from object %d.",j);
	      error = TRUE;
	      break;
	   }
	   if (error)  break;
	}
	
	/* clean up */
	kpds_close_object(dest_obj);
	for (j=0; j<i; j++)
	{
	   if (mask_exists[j])
	      kpds_close_object(objlist[j]);
	}
	kfree(objlist);
	kfree(mask_exists);
	kfree(result);
	kfree(data);
	if (error)
	   return(FALSE);
	else
	   return(TRUE);
	
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmanip_multi_input_loc - Checks source object list for location data, and returns TRUE or FALSE to continue processing
|	
|       Purpose: This function checks whether location data exists in
|		 the source objects, and returns TRUE if continued 
|		 processing is recommended, FALSE if it is not.  The 
|		 grid_type parameter is also set by this function to 
|		 one of the valid data services grid types.
|
|		 The "recommendation" is based on the following rules:
|
|		 If more than one source object is supplied, and location 
|		 data exists, the first source object is considered the 
|		 dominant object.  In other words, location attributes
|		 for that object will be used.  (This is how the grid_type
|		 is determined.)  If source 1 has location, and none of 
|		 the other source objects have it, the location data from 
|		 source 1 should be propagated.  
|		 
|		 If the first source object does not contain location or 
|		 time data, but another source object does contain it, go
|		 ahead and process the data, but do not propagate the 
|		 location data.
|
|		 If more one source object contains location data, this
|		 function will set grid_type to KNONE, print an error 
|		 message, and return FALSE.
|
|         Input: num_src_objs - number of source objects provided in kvalist
|                kvalist - list of source objects (kobject)
|
|        Output: grid_type - processing grid type: set to KNONE, KUNIFORM, 
|			     KRECTILINEAR, or KCURVILINEAR
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions: 
|    Written By: Donna Koechner
|          Date: Jan  3, 1995
|      Verified:
|  Side Effects: None
| Modifications:
|
------------------------------------------------------------*/

int kdmanip_multi_input_loc(
   int	    *grid_type,
   int	    num_src_objs,
   kvalist)
{
        kva_list        valist;
        kobject         *objlist;
	int		*grid; 
	int		*loc_exists; 
	int		i, loc_count=0;
	char		*prog="kdmanip_multi_input_loc";
	char		*lib="kdatamanip";

        if ((objlist = (kobject *) kmalloc(num_src_objs * sizeof(kobject))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for objlist");
	   return(FALSE);
	}

        if ((grid = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for grid array");
	   kfree(objlist);
	   return(FALSE);
	}

        if ((loc_exists = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for loc_exists array");
	   kfree(objlist);
	   kfree(grid);
	   return(FALSE);
	}

        kva_start(valist, num_src_objs);

        for (i=0; i<num_src_objs; i++)
	{
	   objlist[i] = kva_arg(valist, kobject);
	   grid[i] = KNONE;
	   if (objlist[i] != KOBJECT_INVALID)
	   {
	      if ((loc_exists[i] = kpds_query_location(objlist[i])) == TRUE)
	      {
	         loc_count ++;
		 if (!kpds_get_attribute(objlist[i], 
				KPDS_LOCATION_GRID, &grid[i]))
		 {
                    kerror(lib,prog,
			   "Unable to get location grid type for object %d",i);
                    kfree(objlist); kfree(grid); kfree(loc_exists);
                    return(FALSE);
		 }
	      }
	   }
	}
        kva_end(valist);

        kfree(objlist);

        *grid_type = KNONE;

	if (loc_count > 0)
	{
	   if (loc_count == 1)
	   {
	      if (loc_exists[0])
	         *grid_type = grid[0];
	   }
	   else
	   {
              kfree(loc_exists);
              kfree(grid);
	      kerror(lib,prog, "Only one source object can have location "
		     "data (%d out of %d have location)", i, num_src_objs);
	      return(FALSE);
	   }
	}
	
        kfree(loc_exists);
        kfree(grid);
	return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmanip_multi_input_time - Checks source object list for time data, and returns TRUE or FALSE to continue processing
|	
|       Purpose: This function checks whether time data exists in
|		 the source objects, and returns TRUE if continued 
|		 processing is recommended, FALSE if it is not.  
|
|		 The "recommendation" is based on the following rules:
|
|		 If more than one source object is supplied, and time 
|		 data exists, the first source object is considered the 
|		 dominant object.  In other words, time attributes for 
|		 that object will be used.  If source 1 has time, and 
|		 none of the other source objects have it, the time 
|		 data from source 1 should be propagated.  In this case,
|		 the has_time pararmeter is set to TRUE.
|		 
|		 If the first source object does not contain time data, but 
|		 another source object does contain it, go ahead and process 
|		 the data, but do not propagate the time data.
|
|		 If more one source object contains time data, this
|		 function will print an error message and return FALSE.
|
|         Input: num_src_objs - number of source objects provided in kvalist
|                kvalist - list of source objects (kobject)
|
|        Output: has_time - set to true if first object has time data.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions: 
|    Written By: Donna Koechner
|          Date: Jan  3, 1995
|      Verified:
|  Side Effects: None
| Modifications:
|
------------------------------------------------------------*/

int kdmanip_multi_input_time(
   int	    *has_time,
   int	    num_src_objs,
   kvalist)
{
        kva_list        valist;
        kobject         *objlist;
	int		*time_exists; 
	int		i, count=0;
	char		*prog="kdmanip_multi_input_time";
	char		*lib="kdatamanip";

        if ((objlist = (kobject *) kmalloc(num_src_objs * sizeof(kobject))) 
			== NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for objlist");
	   return(FALSE);
	}

        if ((time_exists = (int *) kmalloc(num_src_objs * sizeof(int))) == NULL)
	{
	   kerror(lib,prog,"Unable to allocate memory for time_exists array");
	   kfree(objlist);
	   return(FALSE);
	}

        kva_start(valist, num_src_objs);

        for (i=0; i<num_src_objs; i++)
	{
	   objlist[i] = kva_arg(valist, kobject);
	   if (objlist[i] != KOBJECT_INVALID)
	   {
	      if ((time_exists[i] = kpds_query_time(objlist[i])) == TRUE)
	         count ++;
	   }
	}
        kva_end(valist);

        kfree(objlist);

        *has_time = FALSE;

	if (count > 0)
	{
	   if (count == 1)
	   {
	      if (time_exists[0])
	         *has_time = TRUE;
	   }
	   else
	   {
              kfree(time_exists);
	      kerror(lib,prog, "Only one source object can have time "
		     "data (%d out of %d have time)", i, num_src_objs);
	      return(FALSE);
	   }
	}
	
        kfree(time_exists);
	return(TRUE);
}
