 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for ksampline
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lksampline
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
void lksampline_interp(double *,double *,int,double,double,double,double,double,int,int,int,int,int,double *,unsigned char *);

/*
#define LKSAMPLINE_DEBUG 1
*/
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lksampline - Sample a data object along an arbitrary line
* 
*       Purpose: lksampline is used to sample the data in an object 
*	along an arbitrary line and at arbitrary intervals. The value at 
*	each sample point is obtained by an inverse distance weighting scheme.
*
*	The first sample point is precisely at the starting coordinate and the 
*	final sample is precisely at the ending coordinate.
*
*	The line may traverse any portion of the 5D data space. If a particular
*	sample on the line lies outside the data space then the values will be
*	interpolated using a padding value of zero.
*
*	The output object will be of the same data type as the input object. If
*	a map is present in the input object, then the value data is sent 
*	through the map before processing. The output data will thus 
*	have no map.
*
*	Mask data, if present, is used to help control the interpolation.
*	Value points with a corresponding mask value of zero will not be 
*	included in the interpolated value. However, a minimum number of 
*	neighboring values with a valid mask are required in order to 
*	produce a valid output value; non-valid output values are marked 
*	by a zero mask. If the mask segment is not present, then all 
*	value points in the source object are assumed to be valid.
*
*	The actual minimum required neighbor count is determined by 
*	the setting of the "nc" argument.  If nc is zero, then the minimum 
*	required neighbor count will be computed from the dimensionality of 
*	the source data set. For nc=0, the minimum required neighbor count is 
*	0.5*(2^dim)+1 where dim is the dimensionality of the source object 
*	(1 for a line, 2 for a plane, 3 for a volume, etc).  If nc is 
*	non-zero, then the minimum required neighbor count is simply nc.  
*	Beware of non-intuitive results when interpolating across mask wise 
*	non-convex parts of the data object. The default minimum required 
*	neighbor count gives "reasonable" answers in these cases. Reducing 
*	the minimum required neighbor count can provide answers that are 
*	less and less meaningful.
*
*	Note that lksampline samples the data as if it were using a delta 
*	function.  This implies that you may get significantly different
*	results depending on the position of each sample location. You may 
*	wish to lowpass the source data before sampling with lksampline to 
*	avoid aliasing.  This can be particularly important when dealing with 
*	"holey data" or when the sampling involves degenerate geometry (such 
*	as sampling a plane object with a sampling line that does not lie in 
*	the plane).
*
*	lksampline is implemented using \fIpoint\fR data access. A point data 
*	access is done for each sample in the line. This implies that a large 
*	number of samples may require quite some time to compute, particularly 
*	if operating on a higher dimension data set.
*
*         Input: in_obj - input data object
*		 wc1 - Coordinate of start of sampling line in width dimension
*		 hc1 - Coordinate of start of sampling line in height dimension
*		 dc1 - Coordinate of start of sampling line in depth dimension
*		 tc1 - Coordinate of start of sampling line in time dimension
*		 ec1 - Coordinate of start of sampling line in element dimension
*		 wc2 - Coordinate of end of sampling line in width dimension
*		 hc2 - Coordinate of end of sampling line in height dimension
*		 dc2 - Coordinate of end of sampling line in depth dimension
*		 tc2 - Coordinate of end of sampling line in time dimension
*		 ec2 - Coordinate of end of sampling line in element dimension
*		 n - number of samples between start and end, inclusive of 
*		     end points
*		 nc - minimum required neighbor count; if zero, then this 
*                     is computed as 0.5*(2^dim)+1 where dim is the
*		      dimensionality of the source object.
*
*        Output: out_obj - output data object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: Restrictions on data or input as applicable
*    Written By: Scott Wilson
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lksampline(kobject in_obj,
	double wc1,double hc1,double dc1,double tc1,double ec1,
        double wc2,double hc2,double dc2,double tc2,double ec2,
        int n, int nc, kobject out_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kdatamanip", *rtn = "lksampline";
        int w,h,d,t,e;			/* Size of whole value segment */
        int wr,hr,dr,tr,er;		/* Size of value segment region */
        int itype,otype;                /* Data type of input and output */
        int i,nrgns;
        klist *objlist=NULL;
        unsigned char *data;
        double w_delta,h_delta,d_delta,t_delta,e_delta;
        double wn,hn,dn,tn,en;
        int wl,wh,hl,hh,dl,dh,tl,th,el,eh;
        int wold,hold,dold,told,eold;
        double value[32],mask[32];
        double result;
        unsigned char result_mask;
        int ndims_obj,min_valid;
        int in_has_mask;
        kobject out_obj_orig;

        /* Make sure we have valid objects */
        if (in_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus input object");
            return(FALSE);
          }
        if (out_obj == KOBJECT_INVALID)
          {
            kerror(lib, rtn, "Bogus output object");
            return(FALSE);
          }

        /* Make sure there is a value segment */
        if (!kpds_query_value(in_obj))
          {
            kerror(lib,rtn,"No value data in source object to operate on.");
            return(FALSE);
          }
        /* Make sure there is no location segment */
        if (kpds_query_location(in_obj))
          {
            kerror(lib,rtn,"Object with location segment not supported.");
            return(FALSE);
          }

        /* Reference the input object to avoid side effects, then add the
           reference to the list of goodies to be autmatically free'd on
           error. */
        KCALL((in_obj = kpds_reference_object(in_obj)));
        objlist = klist_add(objlist,in_obj,"KOBJECT");

        /* If there's a map, pull the value data through it */
        KCALL(kpds_set_attribute(in_obj,KPDS_MAPPING_MODE, KMAPPED));

        /* See how big the data is */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_SIZE,&w,&h,&d,&t,&e));

        /* Fix up a mask segment for the referenced input object
           if one doesn't already exist */
        in_has_mask = kpds_query_mask(in_obj);
        if (in_has_mask == 0)
          {
            KCALL(kpds_create_mask(in_obj));
            /* Set the mask segment size to that of the value data,
               then do a kpds_get_data() that hits every pixel. This
               will instantiate a mask of all 1's */
            KCALL(kpds_set_attribute(in_obj,KPDS_MASK_SIZE,w,h,d,t,e));
            KCALL(kpds_set_attribute(in_obj,KPDS_MASK_DATA_TYPE,KUBYTE));
            KCALL(kpds_get_attribute(in_obj,KPDS_MASK_OPTIMAL_REGION_SIZE,
                                            &wr,&hr,&dr,&tr,&er,&nrgns));
            KCALL(kpds_set_attribute(in_obj,KPDS_MASK_REGION_SIZE,
                                            wr,hr,dr,tr,er));
            KCALL(kpds_get_attribute(in_obj,KPDS_MASK_REGION_INFO,
                                            &wr,&hr,&dr,&tr,&er,&nrgns));
            KCALL(!((data = (unsigned char *)kmalloc(wr*hr*dr*tr*er*
                                              sizeof(unsigned char))) == NULL));
            for (i=0; i<nrgns; i++)
              {
                data = (unsigned char *)kpds_get_data(in_obj,
                                                  KPDS_MASK_REGION,(kaddr)data);
                kpds_put_data(in_obj,KPDS_MASK_REGION,(kaddr)data);
              }
            kfree(data);
          }
                
        /* Set the mask and value to KPAD with a value of zero. */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_INTERPOLATE, KPAD));
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_PAD_VALUE, 0.0));
        KCALL(kpds_set_attribute(in_obj,KPDS_MASK_INTERPOLATE, KPAD));
        KCALL(kpds_set_attribute(in_obj,KPDS_MASK_PAD_VALUE, 0.0));
        KCALL(kpds_set_attribute(in_obj,KPDS_MASK_DATA_TYPE,KDOUBLE));

        /* Set the output data type according to the COMPLEX-ness of the
           input data */
        KCALL(kpds_get_attribute(in_obj,KPDS_VALUE_DATA_TYPE,&itype));
        if (itype == KCOMPLEX || itype == KDCOMPLEX) otype = KDCOMPLEX;
        else otype = KDOUBLE;
 
        /* Constrain the input data to double or double complex for read */
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_DATA_TYPE, otype));

        /* Set up the output object */
        if (!kpds_query_value(out_obj)) KCALL(kpds_create_value(out_obj));
        if (!kpds_query_mask(out_obj)) KCALL(kpds_create_mask(out_obj));
        KCALL(kpds_set_attributes(out_obj,
                             KPDS_VALUE_SIZE,n,1,1,1,1,
                             KPDS_VALUE_DATA_TYPE,otype,NULL));
        KCALL(kpds_set_attributes(out_obj,
                             KPDS_MASK_SIZE,n,1,1,1,1,
                             KPDS_MASK_DATA_TYPE,KUBYTE,NULL));
        out_obj_orig = out_obj;
        KCALL((out_obj = kpds_reference_object(out_obj)));
        objlist = klist_add(objlist,out_obj,"KOBJECT");
        KCALL(kpds_set_attribute(out_obj,KPDS_VALUE_REGION_SIZE,1,1,1,1,1));
        KCALL(kpds_set_attribute(out_obj,KPDS_MASK_REGION_SIZE,1,1,1,1,1));

        /* Compute the region size for the data access, given the
           dimensionality of the space, the endpoints of the sampling line */
        wr=hr=dr=tr=er=2;
        KCALL(kpds_set_attribute(in_obj,KPDS_VALUE_REGION_SIZE,wr,hr,dr,tr,er));
        KCALL(kpds_set_attribute(in_obj,KPDS_MASK_REGION_SIZE,wr,hr,dr,tr,er));

	/* Constrain the DMS buffer size heavily to improve performance. */
	kdms_set_attribute(in_obj,KDMS_SEGMENT_VALUE,KDMS_BUFFER_THRESHOLD,32*sizeof(kdcomplex));
	kdms_set_attribute(in_obj,KDMS_SEGMENT_MASK,KDMS_BUFFER_THRESHOLD,32*sizeof(kdcomplex));
 
        /* Find out the dimensionality of the input object */
	ndims_obj = 0;
        if (w != 1) ndims_obj++;
        if (h != 1) ndims_obj++;
        if (d != 1) ndims_obj++;
        if (t != 1) ndims_obj++;
        if (e != 1) ndims_obj++;
#ifdef LKSAMPLINE_DEBUG
printf("Object dimensionality: %d\n",ndims_obj);
#endif

        /* Set the minimum required number of valid pixels in the
           neighborhood that must be present for a valid interpolated
           value to be computed. */
        if (nc == 0) min_valid = (int)((kpow(2.0,ndims_obj)/2.0)+1);
        else min_valid = nc;
#ifdef LKSAMPLINE_DEBUG
printf("Min valid points req'd: %d\n",min_valid);
#endif

        /* Compute sampling intervals */
        w_delta = (wc2-wc1)/(double)(n-1);
        h_delta = (hc2-hc1)/(double)(n-1);
        d_delta = (dc2-dc1)/(double)(n-1);
        t_delta = (tc2-tc1)/(double)(n-1);
        e_delta = (ec2-ec1)/(double)(n-1);

        /* Load the previous position marker with a very unlikely value */
        wold = hold = dold = told = eold = KMAXLINT;

        for (i=0; i<n; i++)
          {
#ifdef LKSAMPLINE_DEBUG
printf("Sample #:%d\n",i);
#endif
            /* Compute coordinate of sample point */
            wn = wc1 + i*w_delta;
            hn = hc1 + i*h_delta;
            dn = dc1 + i*d_delta;
            tn = tc1 + i*t_delta;
            en = ec1 + i*e_delta;

            /* Compute 5-D coords of the bounding box the sample point is in */
            wl = kfloor(wn); wh = wl+1;
            hl = kfloor(hn); hh = hl+1;
            dl = kfloor(dn); dh = dl+1;
            tl = kfloor(tn); th = tl+1;
            el = kfloor(en); eh = el+1;
#ifdef LKSAMPLINE_DEBUG
printf("  Lower coords: %d %d %d %d %d\n",wl,hl,dl,tl,el);
printf("  Upper coords: %d %d %d %d %d\n",wh,hh,dh,th,eh);
#endif

            /* Go get the value and mask data for this area */
            if (wl != wold || hl != hold || dl != dold || el != eold || tl != told)
              {
                kpds_set_attribute(in_obj,KPDS_VALUE_POSITION,wl,hl,dl,tl,el);
                kpds_set_attribute(in_obj,KPDS_MASK_POSITION,wl,hl,dl,tl,el);
                kpds_get_data(in_obj,KPDS_VALUE_REGION,(kaddr)value); 
                kpds_get_data(in_obj,KPDS_MASK_REGION,(kaddr)mask);
                wold = wl; hold = hl; dold = dl; told = tl; eold = tl;
              }
      
            /* Interpolate the data */
           (void)lksampline_interp(value,mask,min_valid,wn,hn,dn,tn,en,
                                   wl,hl,dl,tl,el,&result,&result_mask);

            /* Write the result to the output object */
            KCALL(kpds_put_data(out_obj,KPDS_VALUE_REGION,(kaddr)(&result)));
            KCALL(kpds_put_data(out_obj,KPDS_MASK_REGION,(kaddr)(&result_mask)));
          }

        /* Remove output mask if the input object didn't have one. */
        if (in_has_mask == 0) KCALL(kpds_destroy_mask(out_obj_orig));

        (void)klist_free(objlist,(kfunc_void)lkcall_free);
	return TRUE;
}

/*
	lksampline_interp - Perform inverse-distance interpolation
                            *with mask* in 5D.
*/
void
lksampline_interp(double *value,double *mask,int min_valid,
	double wn,double hn,double dn,double tn,double en,
        int wl,int hl,int dl,int tl,int el,double *result,
        unsigned char *result_mask)
  {
    int j,count;
    double wts[32],dst[32],val[32],acc,sum,tot;
    int w,h,d,t,e;
    int zero_dist;

    zero_dist = -1;
    tot = 0.0;
    count = 0;
#ifdef LKSAMPLINE_DEBUG
    printf("In lksampline_interp\n");
    printf("  Interpolating at w:%g h:%g d:%g t:%g e:%g\n",wn,hn,dn,tn,en);
#endif

    /* Trot thru the 5D hypercube, measuring the distance from each data point
       to the location to be interpolated. Load the distances and values into
       a smaller, contiguous array for efficient processing later. */
    j = 0;
    for (e=el; e<el+2; e++)
      {
        for (t=tl; t<tl+2; t++)
          {
            for (d=dl; d<dl+2; d++)
              {
                for (h=hl; h<hl+2; h++)
                  {
                    for (w=wl; w<wl+2; w++)
                      {
                        dst[count] = 0;
                        if (mask[j] != 0)
                          {
                            sum=(wn-w)*(wn-w)+(hn-h)*(hn-h)+(dn-d)*(dn-d)
                               +(tn-t)*(tn-t)+(en-e)*(en-e);
                            dst[count] = sqrt(sum);
                            if (dst[count] == 0.0) zero_dist = count;
                            tot += dst[count];
                            val[count] = value[j];
#ifdef LKSAMPLINE_DEBUG
printf("w:%d h:%d d:%d t:%d e:%d count:%d dst:%g val:%g\n",
        w,h,d,t,e,count,dst[count],val[count]);
#endif
                            count++;
                          }
                        j++;
                      }
                  }
              }
          }
      }
#ifdef LKSAMPLINE_DEBUG
    printf("Total distance: %g point count:%d\n",tot,count);
#endif

    if (zero_dist != -1)
      { /* Hit right on one of the data points */
        *result = val[zero_dist];
        *result_mask = 1;
        return;
      }

    if (count >= min_valid)
      {
        /* Go thru and set the weights. Remember to normalize them to
           sum to 1.0! */
        /* Inverse-distance interpolate */
        for (j=0; j<count; j++) wts[j] = 1.0/dst[j];
        for (j=0,acc=0; j<count; j++) acc += wts[j];
#ifdef LKSAMPLINE_DEBUG
        printf("wts: ");
#endif
        for (j=0; j<count; j++) 
          {
            wts[j] /= acc;
#ifdef LKSAMPLINE_DEBUG
            printf("%g ",wts[j]);
#endif
          }
#ifdef LKSAMPLINE_DEBUG
        printf("\n");
#endif
        /* Now go thru and form the weighted sum of the valid data points */
        for (j=0,acc=0; j<count; j++) acc += val[j]*wts[j];
        *result = acc;
        *result_mask = 1;
        return;
      }
    else
      {
        *result = 0.0;
        *result_mask = 0;
        return;
      }
  }
/* -library_code_end */
