 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for knoise
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lknoise
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/*
 * Operation structure declaration
 */
struct _noise_func
{
        char    *name;
        int     routine;
};


/*
 * Function Declaration list
 */
static struct _noise_func noise_func[] =
{
#define KNOISE_GAUSS 0
    { "gauss", KNOISE_GAUSS },
#define KNOISE_RAYLEIGH 1
    { "ray",KNOISE_RAYLEIGH},
#define KNOISE_UNIFORM  2
    { "unif",KNOISE_UNIFORM},
#define KNOISE_POISSON  3
    { "poiss",KNOISE_POISSON},
#define KNOISE_EXP      4
    {"exp",KNOISE_EXP      },
};
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lknoise - introduce noise in source object
* 
*       Purpose: The following is the library routine that performs
*                the addition of noise to a data object or the 
*                replacement of the value segment of a data object 
*                with noise. The routine takes an input data object,
*                an optional gating data object, a noise type, 
*                specs related to the noise, a flag indicating the add
*                \replace operation, and an output data object in which 
*                to store the result. The following types of noises are
*                supported by this routine:
*                
*                .TS 
*                center tab(:) ;
*                l l .
*                Gaussian Noise    :  (Mean, Variance)
*                Rayleigh Noise    :  (Variance)
*                Uniform Noise     :  (Maximum, Minimum)
*                Poisson Noise     :  (Amount of time, Variance)
*                Exponential Noise :  (Variance) 
*		 .TE
*                
*                The right hand side above indicates the paramaters
*                needed to specify each kind of noise. Therefore if
*                lknoise() were called with:
*
*                ! lknoise(i, NULL, "ray", TRUE, 1.0, 2.0, o);
*
*                then the following would be performed:
*
*                !    o = i + Rayleigh Noise of variance 1.0
*
*                It may be noted here that the value passed in the 
*                second spec (2.0) is ignored since it only 1 value is
*                needed for specifying a Rayleigh distribution.     
*
*         Input: src_obj   - the input object to be processed
*                function  - type of noise to be introduced
*                add_flag  - indicates whether to add or replace w/noise
*                spec1     - first paramater for distribution 
*                spec2     - second paramater for distribution
*
*        Output: dst_obj  - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: Destroys the map segment on the destination object 
*                if the source object contains both map and value data.
*                Also the size and datatype of the value segment is
*                set to the size and datatype of the the mapped
*                source object.
* Modifications: 
****************************************************************/
/* -library_def */
int lknoise(
   kobject src_obj,
   char    *function,
   int     add_flag,
   double  spec1,
   double  spec2,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
        char *lib = "kdatamanip";       /* library name (for error reporting) */
        char *rtn = "lknoise" ;         /* routine name (for error reporting) */
        int   line_size, num_lines;     /* object data size information */
        int   type;                     /* processing data type */
        int   j, num;                   /* count variables */
        int   error = FALSE;
        int   w,h,d,t,e;                /* object size */
        int   map_exists,               /* TRUE if object has MAP data */
              val_exists,               /* TRUE if object has VALUE data */
              mask_exists;              /* TRUE if object has MASK data */
        long   *long_data = NULL;       /* array for long data   */
        unsigned long *ulong_data = NULL; 
        unsigned char *ubyte_data = NULL;
        double *real_data = NULL;       /* array for non-complex data */
        double *imag_data = NULL;       /* array for imag portion of cmplx */
        kdcomplex *cmplx_data = NULL;   /* array for cmplx data. */
        double *noise1 = NULL;          /* array for noise data  */
        double *noise2 = NULL;          /* array for noise data  */
        unsigned char *mask = NULL;     /* array for gating data */
        int  routine, func;
        char *prim;
        char *info;
        char *datasize;
        char *datatype;

        /*
         *  Check which data (VALUE, MAP, MASK) is available for src_obj.
         *  If there is a MAP, operate directly on the map (Don't worry
         *  VALUE or MASK data).
         *  If there is no map, operate on the VALUE data.  If operating on
         *  VALUE data and there is a MASK, use the MASK.
         *  If there is no MAP or VALUE data, fail.
         *  If there is both MAP and VALUE data then delete the MAP segment
         *  on the output object and process the mapped data.
         */
        map_exists  = kpds_query_map(src_obj);
        val_exists  = kpds_query_value(src_obj);
        mask_exists = kpds_query_mask(src_obj);

        if (!map_exists && !val_exists)
        {
           kerror(lib, rtn,
                  "Input data object must have either Value or Map data.");
           return(FALSE);
        }

        /*
         *  Create reference objects for all passed in kobjects.
         */
        if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, rtn, "Failed to duplicate input object");
           return(FALSE);
        }

        /* 
         * Determine the type of noise to be introduced by comparing
         * the function string passed in with the list of supported 
         * noise distributions.
         */

        for (num = 0, func = -1; num < knumber(noise_func); num++)
        {
           if (kstrcmp(function, noise_func[num].name) == 0)
           {
              func = num;
              break;
           }
        }

        if (func == -1)
        {
           kerror(lib, rtn, "Sorry, but the distribution '%s' is not in the \
list of supported distributions at this time.", function);
           kpds_close_object(src_obj);
           kpds_close_object(dst_obj);
           return(FALSE);
        }

        routine = (int) noise_func[func].routine;

        /*
         *  Because operations in knoise are pointwise, if there
         *  is a map, we can work on it directly. If there is value
         *  and map then we wrok on the mapped value.
         */

        if (map_exists && val_exists)
        {
           map_exists = FALSE;
           kpds_set_attribute(src_obj,  KPDS_MAPPING_MODE, TRUE);
           if (kpds_query_map(dst_obj))
           {
              kpds_destroy_map(dst_obj);
              kpds_get_attributes(src_obj,KPDS_VALUE_DATA_TYPE, &type,
                                  KPDS_VALUE_SIZE, &w, &h, &d, &t, &e, NULL);
              kpds_set_attributes(dst_obj,KPDS_VALUE_DATA_TYPE, type,
                                  KPDS_VALUE_SIZE, w, h, d, t, e, NULL);
           }
        }

        if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, rtn, "Failed to duplicate output object");
           kpds_close_object(src_obj);
           return(FALSE);
        }

        if (map_exists)
        {
           prim = KPDS_MAP_LINE;
           info = KPDS_MAP_LINE_INFO;
           datasize = KPDS_MAP_SIZE;
           datatype = KPDS_MAP_DATA_TYPE;
        }
        else
        {
           prim = KPDS_VALUE_LINE;
           info = KPDS_VALUE_LINE_INFO;
           datasize = KPDS_VALUE_SIZE;
           datatype = KPDS_VALUE_DATA_TYPE;
        }

        /*
         *  We will process the data by lines, so get the line size
         *  and the number of lines of data in src_obj.
         */
        kpds_get_attribute(src_obj, info, &line_size, &num_lines);

        /*
         *  Check the data type of the input object. 
         */
        kpds_get_attribute(src_obj, datatype, &type);

        /* If inp object is complex then process the data
         * as KDCOMPLEX else process if it is a float or double
         * process as double else as long.
         */
        type = kdatatype_cast_process(KBIT,type, KDCOMPLEX | KDOUBLE |
				      KULONG | KLONG | KUBYTE);
        kpds_set_attribute(src_obj, datatype, type);
        kpds_set_attribute(dst_obj, datatype, type);

        /* Allocate space for arrays containing noise. Allocate
         * space for noise2 only if complex data is being
         * processed (imag_data != NULL).
         */
        noise1 = (double *) kmalloc(line_size*sizeof(double));
        if (imag_data)
           noise2 = (double *) kmalloc(line_size*sizeof(double));

        for (num = 0; num < num_lines; num++)
        {
            if (mask_exists)
               mask = (unsigned char *) kpds_get_data(src_obj, KPDS_MASK_LINE,
                      (kaddr) mask);

            /* Get complex data and split it into it's real and
             * complex parts. If not complex then get data in its
             * appropriate array.
             */
            if ( (type == KDCOMPLEX) )
            {
               cmplx_data= (kdcomplex *) kpds_get_data(src_obj, prim,
                           (kaddr)cmplx_data);
               kdcomplex_to_arrays(cmplx_data,line_size,&real_data,&imag_data);
            } 
            else if ( (type == KDOUBLE) )
               real_data= (double *) kpds_get_data(src_obj,prim,
                          (kaddr)real_data);
            else if ( (type == KLONG))
               long_data= (long  *) kpds_get_data(src_obj,prim,
                          (kaddr)long_data);
            else if (type == KULONG)
               ulong_data= (unsigned long *)kpds_get_data(src_obj,prim,
                           (kaddr)ulong_data);
            else if (type == KUBYTE)
               ubyte_data= (unsigned char *)kpds_get_data(src_obj,prim,
                           (kaddr)ubyte_data);

            /* 
             * Generate approproate noise depending on 
             * distribution requested.
             */
            switch (routine)
            {
               case KNOISE_GAUSS:
                       kgen_gauss(line_size,spec1,spec2,noise1);
                       if (imag_data)
                          kgen_gauss(line_size,spec1,spec2,noise2); 
                       break;

               case KNOISE_RAYLEIGH:
                       kgen_rayleigh(line_size,spec1,noise1);
                       if (imag_data)
                          kgen_rayleigh(line_size,spec1,noise2);
                       break;

               case KNOISE_UNIFORM:
                       kgen_unif(line_size,spec1,spec2,noise1); 
                       if (imag_data)
                          kgen_unif(line_size,spec1,spec2,noise2);
                       break;

               case KNOISE_POISSON:
                       kgen_poisson(line_size,spec1,spec2,noise1); 
                       if (imag_data)
                          kgen_poisson(line_size,spec1,spec2,noise2);
                       break;

               case KNOISE_EXP :
                       kgen_expon(line_size,spec1,noise1);
                       if (imag_data)
                          kgen_expon(line_size,spec1,noise2);
                       break;
            } 

            if (!mask)
            {
               for (j = 0; j < line_size; j++)
               {
                  if (add_flag)
                  {
                     if (ubyte_data) ubyte_data[j] += (unsigned char)noise1[j];
                     if (ulong_data) ulong_data[j] += (unsigned long)noise1[j];
                     if (long_data) long_data[j] += (long)noise1[j];
                     if (real_data) real_data[j] += noise1[j]; 
                     if (imag_data) imag_data[j] += noise2[j];
                  }
                  else
                  {
                     if (ubyte_data) ubyte_data[j] = (unsigned char)noise1[j];
                     if (ulong_data) ulong_data[j] = (unsigned long)noise1[j];
                     if (long_data) long_data[j] = (long)noise1[j];
                     if (real_data) real_data[j] = noise1[j];
                     if (imag_data) imag_data[j] = noise2[j];
                  }
               }
            }
            else
            {
               for (j = 0; j < line_size; j++)
               {
                  if ((!mask || mask[j]))
                  {
                    if (add_flag)
                    {
                       if (ubyte_data) ubyte_data[j] +=
                                                      (unsigned char)noise1[j];
                       if (ulong_data) ulong_data[j] += 
                                                      (unsigned long)noise1[j];
                       if (long_data) long_data[j] += (long)noise1[j];
                       if (real_data) real_data[j] += noise1[j];
                       if (imag_data) imag_data[j] += noise2[j];
                    }
                    else
                    {
                       if (ubyte_data) ubyte_data[j] = (unsigned char)noise1[j];
                       if (ulong_data) ulong_data[j] = (unsigned long)noise1[j];
                       if (long_data) long_data[j] = (long)noise1[j];
                       if (real_data) real_data[j] = noise1[j];
                       if (imag_data) imag_data[j] = noise2[j];
                    }
                  }
               }
            }
           
            if (type == KDCOMPLEX)
            {
               kcombine_to_complex(&cmplx_data,line_size,real_data,imag_data);
               if (!kpds_put_data(dst_obj, prim, (kaddr) cmplx_data))
               {
                  kerror(lib, rtn, "failed to put line in output object");
                  error = TRUE;
                  break;
               }
            }
            else if (type == KDOUBLE)
            {
               if (!kpds_put_data(dst_obj, prim, (kaddr) real_data))
               {
                  kerror(lib, rtn, "failed to put line in output object");
                  error = TRUE;
                  break;
               }
            } 
            else if (type == KLONG)
            {
               if (!kpds_put_data(dst_obj, prim, (kaddr) long_data))
               {
                  kerror(lib, rtn, "failed to put line in output object");
                  error = TRUE;
                  break;
               }
            }
            else if (type == KULONG)
            {
               if (!kpds_put_data(dst_obj, prim, (kaddr) ulong_data))
               {
                  kerror(lib, rtn, "failed to put line in output object");
                  error = TRUE;
                  break;
               }
            }
            else if (type == KUBYTE)
            {
               if (!kpds_put_data(dst_obj, prim, (kaddr) ubyte_data))
               {
                  kerror(lib, rtn, "failed to put line in output object");
                  error = TRUE;
                  break;
               }
            }
        }

        kpds_close_object(src_obj);
        kpds_close_object(dst_obj);

        if (ulong_data) kfree(ulong_data);
        if (ubyte_data) kfree(ubyte_data);
        if (long_data)  kfree(long_data);
        if (real_data)  kfree(real_data);
        if (imag_data)  kfree(imag_data);
        if (cmplx_data) kfree(cmplx_data);
        if (noise2)     kfree(noise2);
        if (noise1)     kfree(noise1);
        kfree(mask);

        if (error)
           return(FALSE);
        else
	   return TRUE;
}
/* -library_code_end */
