 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kflip
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkflip
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkflip - reflect data in object along specified dimensions
* 
*       Purpose: Library Routine for kflip.  This routine performs
*                data reflection in one, all, or any combination of
*                directions.  The directions supported are:
*                KWIDTH, KHEIGHT, KDEPTH, KTIME, KELEMENTS
*
*         Input: i (src)        - the source data object 
*                num_dim	- No. of dimensions to operate on 
*                dim_list       - list of dimensions on which to operate
*
*        Output: o (dest)       - the destination data object 
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*
*  Restrictions: 
*    Written By: Ramiro Jordan
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkflip( kobject src,
            int num_dim, 
            long *dim_list, 
            kobject dest )
/* -library_def_end */

/* -library_code */
{
   kobject src_ref, dest_ref;      /* ref. objs for source and destination */
   char *lib = "kdatamanip";       /* library name (for error reporting) */
   char *rtn = "lkflip";           /* routine name (for error reporting) */

   int map_exists,               /* TRUE if object has MAP data */
       val_exists,               /* TRUE if object has VALUE data */
       mask_exists,              /* TRUE if object has MASK data */
       time_exists,              /* TRUE if object has TIME data */
       location_exists;          /* TRUE if object has LOCATION data */

   int data_type;                /* source object data type */
   int pdata_type;               /* data type for processing */
   int w, h, d, t, e;            /* number of data elements in each direction */
   int data_length;              /* numer of data points to process */
   int num_regions;              /* number of regions to process */
   int i, loop_count, count;     /* loop counters */
   int region_size[5];
   int grid;
   
   unsigned char *bdata = NULL, *bval = NULL;
   long *idata = NULL, *ival = NULL;
   double *ddata = NULL, *dval = NULL;
   kdcomplex *cmplx_data = NULL, *cmplx_val = NULL;

  /*
   *  Check which data is available for lkflip.
   */
   map_exists  = kpds_query_map(src);
   val_exists  = kpds_query_value(src);
   mask_exists = kpds_query_mask(src);
   time_exists = kpds_query_time(src);
   location_exists = kpds_query_location(src);

  /*
   * Check to see if the source and destination objects passed into
   * the lroutine (lkflip) are valid data objects.  With these checks
   * pushed into the lroutine, the driver routine need not necessarily check
   * the status on the kpds_input and kpds_output calls, though it is a good
   * idea to do it.
   * These checks are provided here to make the lroutine complete in its
   * implementation so that it can be called by other programs.
   * (Reusable code..).
   */
   if ( !src ) {
      kerror(lib, rtn, "Source object is not a valid data object.");
      return(FALSE);
   }
   if ( !dest ) {
      kerror(lib, rtn, "Destination object is not a valid data object.");
      return (FALSE);
   }

   /*
    * Unable to interpret explicit curvilinear or rectilinear location data
    */
   if (location_exists) {
      if (!kpds_get_attribute(src, KPDS_LOCATION_GRID, &grid)) {
	 kerror(lib, rtn, 
		"Unable to retrieve location grid attribute.");
	 return(FALSE);
      }
      if (grid != KUNIFORM && (grid == KRECTILINEAR || 
			       grid == KCURVILINEAR)) {
	 kerror(lib, rtn,
               "Source object contains explicit rectilinear or curvilinear "
               "location data. Unable to process");
	 return(FALSE);
      }
   }


  /*
   *  Set up a reference to the source object so that the library does
   *  not have any side effects on the source object's attributes.
   */
   if ((src_ref = kpds_reference_object(src)) == KOBJECT_INVALID) {
       kerror(lib, rtn, "Failed to reference source object.");
       return(FALSE);
   }

  /*
   * Set up a reference to the destination object to facilitate processing.
   * Do not want to introduce side effects on the destination atributes.
   */
   if ((dest_ref = kpds_reference_object(dest)) == KOBJECT_INVALID) {
       kerror(lib, rtn, "Failed to reference destination object.");
       return(FALSE);
   }

  /*
   * If source object has location, map and/or time data the program
   * does nothing.  Reflection is performed in objects that have VALUE
   * and/or MASK data.
   */ 

  /*
   * Process VALUE data.
   */
   if ( val_exists ) {
     /*
      * Now get the presentation attributes on the source reference object.
      * These will not be applied to the destination object.
      *
      * The source object may contain a collection of data, thus we get
      * the KPDS_VALUE_SIZE attribute which provides the following information:
      *          - w = number of parameters in the width direction
      *          - h = number of parameters in the height direction 
      *          - d = number of parameters in the depth direction
      *          - t = number of parameters in the time direction 
      *          - e = number of parameters in the elements direction 
      *
      */
      if (! kpds_get_attribute(src_ref, KPDS_VALUE_SIZE, &w, &h, &d, &t, &e)) {
         kerror(lib, rtn, "kpds_get_attribute failed for KPDS_VALUE_SIZE.");
         return(FALSE);
      }

     /*
      * Obtain data type of source object to be processed.
      *     - data_type = source object data type
      */
      if (! kpds_get_attribute(src_ref, KPDS_VALUE_DATA_TYPE, &data_type)) {
         kerror(lib, rtn, "kpds_get_attribute failed for KPDS_VALUE_DATA_TYPE.");
         return(FALSE);
      }

     /*
      * Select the appropriate common data type for processing the data.
      * Want to minimize any corruption on the data integrity.
      * Recommended data types are:
      *      - KUBYTE = unsigned character 
      *      - KLONG = signed long integer
      *      - KULONG = unsigned long integer
      *      - KDOUBLE = double precision floating point
      *      - KDCOMPLEX = double precision complex
      */

      pdata_type = kdatatype_cast_process(data_type,data_type,
                   (KUBYTE | KLONG | KULONG | KDOUBLE | KDCOMPLEX) );

     /*
      * If data is byte representable then process it as unsigned char
      * (KUBYTE).  If data is integer representable then process it as
      * long int (KLONG).  If data is float representable then process
      * it as double float (KDOUBLE).  If data is COMPLEX representable
      * then process it as double complex (KDCOMPLEX).  In the destination
      * object the data type will be automatically casted to the appropriate
      * data type.
      */
      if ( (pdata_type == KUBYTE) || (pdata_type == KLONG) ||
           (pdata_type == KULONG) || (pdata_type == KDOUBLE) ||
           (pdata_type == KDCOMPLEX) ) {
         if (! kpds_set_attribute(src_ref, KPDS_VALUE_DATA_TYPE,
           pdata_type)) {
           kerror(lib, rtn,
                "kpds_set_attribute failed for KPDS_VALUE_DATA_TYPE");
           return(FALSE);
         }
         if (! kpds_set_attribute(dest_ref, KPDS_VALUE_DATA_TYPE,
           pdata_type)) {
           kerror(lib, rtn,
                "kpds_set_attribute failed for KPDS_VALUE_DATA_TYPE");
           return(FALSE);
         }
         kinfo(KVERBOSE,"Input   data type is: %d\n",data_type);
         kinfo(KVERBOSE,"Process data type is: %d\n",pdata_type);
      }

      for (loop_count = 0; loop_count < num_dim; loop_count++) {
          region_size[0] = region_size[1] = region_size[2] = 1;
          region_size[3] = region_size[4] = 1;
          if (loop_count == 1) {
             if ((src_ref=kpds_reference_object(dest_ref)) == KOBJECT_INVALID) {
                  kerror(lib, rtn, "Failed to reference source object.");
                  return(FALSE);
             }
          }
          switch (dim_list[loop_count])
          {
                 case KWIDTH:
                                region_size[0] = w;
                                break;
                 case KHEIGHT:
                                region_size[1] = h;
                                break;
                 case KDEPTH:
                                region_size[2] = d;
                                break;
                 case KTIME:
                                region_size[3] = t;
                                break;
                 case KELEMENTS:
                                region_size[4] = e;
                                break;
          }
          kpds_set_attributes(src_ref,KPDS_VALUE_REGION_SIZE,
                             region_size[0],region_size[1],
                             region_size[2],region_size[3],
                             region_size[4],KPDS_VALUE_POSITION,
                             0,0,0,0,0,NULL);
          kpds_set_attributes(dest_ref,KPDS_VALUE_REGION_SIZE,
                             region_size[0],region_size[1],
                             region_size[2],region_size[3],
                             region_size[4],KPDS_VALUE_POSITION,
                             0,0,0,0,0,NULL);
          data_length = region_size[0]*region_size[1]*region_size[2]*
                        region_size[3]*region_size[4];
          num_regions = (w*h*d*t*e)/data_length;
          for (count=0; count < num_regions; count++) {
            /*
             * Process KUBYTE data.
             */
             if ( (pdata_type == KUBYTE) ) {
                bdata = (unsigned char *)(kpds_get_data(src_ref,
                                          KPDS_VALUE_REGION, bdata));
                if (bdata == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for bdata.");
                   return(FALSE);
                }
                if (bval == NULL) {
                   bval = (unsigned char *)
                          kmalloc(data_length*sizeof(unsigned char));
                }
                if (bval == NULL) {
                   kerror(lib, rtn, "kmalloc failed for bval.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  bval[i] = bdata[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_VALUE_REGION, bval)) {
                   kerror(lib, rtn, "kpds_put_data failed for ival.");
                   return(FALSE);
                }
             }
            /*
             * Process KLONG and KULONG data.
             */
             if ( (pdata_type == KLONG)  || (pdata_type == KULONG) ) {
                idata = (long *)(kpds_get_data(src_ref,
                                 KPDS_VALUE_REGION, idata));
                if (idata == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for idata.");
                   return(FALSE);
                }
                if (ival == NULL) {
                   ival = (long *)kmalloc(data_length*sizeof(long));
                }
                if (ival == NULL) {
                   kerror(lib, rtn, "kmalloc failed for ival.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  ival[i] = idata[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_VALUE_REGION, ival)) {
                   kerror(lib, rtn, "kpds_put_data failed for ival.");
                   return(FALSE);
                }
             }
            /*
             * Process KDOUBLE data.
             */
             if ( (pdata_type == KDOUBLE) ) {
                ddata = (double *)(kpds_get_data(src_ref, KPDS_VALUE_REGION,
                                   ddata));
                if (ddata == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for ddata.");
                   return(FALSE);
                }
		if (dval == NULL) {
		   dval = (double *)kmalloc(data_length*sizeof(double));
                }
                if (dval == NULL) {
                   kerror(lib, rtn, "kmalloc failed for dval.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  dval[i] = ddata[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_VALUE_REGION, dval)) {
                   kerror(lib, rtn, "kpds_put_data failed for dval.");
                   return(FALSE);
                }
             }
            /*
             * Process KDCOMPLEX data.
             */
             if ( (pdata_type == KDCOMPLEX) ) {
                cmplx_data = (kdcomplex *)(kpds_get_data(src_ref,
                                           KPDS_VALUE_REGION, cmplx_data));
                if (cmplx_data == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for cmplx_data.");
                   return(FALSE);
                }
		if (cmplx_val == NULL) {
		   cmplx_val = (kdcomplex *)
                               kmalloc(data_length*sizeof(kdcomplex));
                }
                if (cmplx_val == NULL) {
                   kerror(lib, rtn, "kmalloc failed for cmplx_val.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  cmplx_val[i] = cmplx_data[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_VALUE_REGION, cmplx_val)) {
                   kerror(lib, rtn, "kpds_put_data failed for cmplx_val.");
                   return(FALSE);
                }
             }

          }
          if (bdata) kfree(bdata);
          if (bval) kfree(bval);
          if (idata) kfree(idata);
          if (ival) kfree(ival);
          if (ddata) kfree(ddata);
          if (dval) kfree(dval);
          if (cmplx_data) kfree(cmplx_data);
          if (cmplx_val) kfree(cmplx_val);
      }

   } /* end processing for VALUE data */

  /*
   * Process MASK data.
   */
   if ( mask_exists ) {
     /*
      * Now set the presentation attributes on the source reference object.
      * These will not be applied to the destination object.
      *
      * The source object may contain a collection of images, thus we get
      * the KPDS_MASK_SIZE attribute which provides the following information:
      *          - w = number of parameters in the width direction
      *          - h = number of parameters in the height direction 
      *          - d = number of parameters in the depth direction
      *          - t = number of parameters in the time direction 
      *          - e = number of parameters in the elements direction 
      *
      */
      if (! kpds_get_attribute(src_ref, KPDS_MASK_SIZE, &w, &h, &d, &t, &e)) {
         kerror(lib, rtn, "kpds_get_attribute failed for KPDS_MASK_SIZE.");
         return(FALSE);
      }

     /*
      * Obtain data type of source object to be processed.
      *     - data_type = source object data type
      */
      if (! kpds_get_attribute(src_ref, KPDS_MASK_DATA_TYPE, &data_type)) {
         kerror(lib, rtn, "kpds_get_attribute failed for KPDS_MASK_DATA_TYPE.");
         return(FALSE);
      }


     /*
      * Select the appropriate common data type for processing the data.
      * Want to minimize any corruption on the data integrity.
      * Recommended data types are:
      *      - KUBYTE = unsigned character 
      *      - KLONG = signed long integer
      *      - KULONG = unsigned long integer
      *      - KDOUBLE = double precision floating point
      *      - KDCOMPLEX = double precision complex
      */

      pdata_type = kdatatype_cast_process(data_type,data_type,
                   (KUBYTE | KLONG | KULONG | KDOUBLE | KDCOMPLEX) );

     /*
      * If data is byte representable then process it as unsigned char
      * (KUBYTE).  If data is integer representable then process it as
      * long int (KLONG).  If data is float representable then process
      * it as double float (KDOUBLE).  If data is COMPLEX representable
      * then process it as double complex (KDCOMPLEX).  In the destination
      * object the data type will be automatically casted to the appropriate
      * data type.
      */
      if ( (pdata_type == KUBYTE) || (pdata_type == KLONG) ||
           (pdata_type == KULONG) || (pdata_type == KDOUBLE) ||
           (pdata_type == KDCOMPLEX) ) {
         if (! kpds_set_attribute(src_ref, KPDS_MASK_DATA_TYPE,
           pdata_type)) {
           kerror(lib, rtn,
                "kpds_set_attribute failed for KPDS_MASK_DATA_TYPE");
           return(FALSE);
         }
         if (! kpds_set_attribute(dest_ref, KPDS_MASK_DATA_TYPE,
           pdata_type)) {
           kerror(lib, rtn,
                "kpds_set_attribute failed for KPDS_MASK_DATA_TYPE");
           return(FALSE);
         }
      }

      for (loop_count = 0; loop_count < num_dim; loop_count++) {
          region_size[0] = region_size[1] = region_size[2] = 1;
          region_size[3] = region_size[4] = 1;
          if (loop_count == 1) {
             if ((src_ref=kpds_reference_object(dest_ref)) == KOBJECT_INVALID) {
                  kerror(lib, rtn, "Failed to reference source object.");
                  return(FALSE);
             }
          }
          switch (dim_list[loop_count])
          {
                 case KWIDTH:
                                region_size[0] = w;
                                break;
                 case KHEIGHT:
                                region_size[1] = h;
                                break;
                 case KDEPTH:
                                region_size[2] = d;
                                break;
                 case KTIME:
                                region_size[3] = t;
                                break;
                 case KELEMENTS:
                                region_size[4] = e;
                                break;
          }
          kpds_set_attributes(src_ref,KPDS_MASK_REGION_SIZE,
                             region_size[0],region_size[1],
                             region_size[2],region_size[3],
                             region_size[4],KPDS_MASK_POSITION,
                             0,0,0,0,0,NULL);
          kpds_set_attributes(dest_ref,KPDS_MASK_REGION_SIZE,
                             region_size[0],region_size[1],
                             region_size[2],region_size[3],
                             region_size[4],KPDS_MASK_POSITION,
                             0,0,0,0,0,NULL);
          data_length = region_size[0]*region_size[1]*region_size[2]*
                        region_size[3]*region_size[4];
          num_regions = (w*h*d*t*e)/data_length;
          for (count=0; count < num_regions; count++) {
            /*
             * Process KUBYTE data.
             */
             if ( (pdata_type == KUBYTE) ) {
                bdata = (unsigned char *)(kpds_get_data(src_ref,
                                          KPDS_MASK_REGION, bdata));
                if (bdata == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for bdata.");
                   return(FALSE);
                }
                if (bval == NULL) {
                   bval = (unsigned char *)
                          kmalloc(data_length*sizeof(unsigned char));
                }
                if (bval == NULL) {
                   kerror(lib, rtn, "kmalloc failed for bval.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  bval[i] = bdata[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_MASK_REGION, bval)) {
                   kerror(lib, rtn, "kpds_put_data failed for ival.");
                   return(FALSE);
                }
             }
            /*
             * Process KLONG and KULONG data.
             */
             if ( (pdata_type == KLONG)  || (pdata_type == KULONG) ) {
                idata = (long *)(kpds_get_data(src_ref, KPDS_MASK_REGION,
                                 idata));
                if (idata == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for idata.");
                   return(FALSE);
                }
                if (ival == NULL) {
                   ival = (long *)kmalloc(data_length*sizeof(long));
                }
                if (ival == NULL) {
                   kerror(lib, rtn, "kmalloc failed for ival.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  ival[i] = idata[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_MASK_REGION, ival)) {
                   kerror(lib, rtn, "kpds_put_data failed for ival.");
                   return(FALSE);
                }
             }
            /*
             * Process KDOUBLE data.
             */
             if ( (pdata_type == KDOUBLE) ) {
                ddata = (double *)(kpds_get_data(src_ref, KPDS_MASK_REGION,
                                   ddata));
                if (ddata == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for ddata.");
                   return(FALSE);
                }
                if (dval == NULL) {
                   dval = (double *)kmalloc(data_length*sizeof(double));
                }
                if (dval == NULL) {
                   kerror(lib, rtn, "kmalloc failed for dval.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  dval[i] = ddata[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_MASK_REGION, dval)) {
                   kerror(lib, rtn, "kpds_put_data failed for dval.");
                   return(FALSE);
                }
             }
            /*
             * Process KDCOMPLEX data.
             */
             if ( (pdata_type == KDCOMPLEX) ) {
                cmplx_data = (kdcomplex *)(kpds_get_data(src_ref,
                                           KPDS_MASK_REGION, cmplx_data));
                if (cmplx_data == NULL) {
                   kerror(lib, rtn, "kpds_get_data failed for cmplx_data.");
                   return(FALSE);
                }
                if (cmplx_val == NULL) {
                   cmplx_val = (kdcomplex *)
                               kmalloc(data_length*sizeof(kdcomplex));
                }
                if (cmplx_val == NULL) {
                   kerror(lib, rtn, "kmalloc failed for cmplx_val.");
                   return(FALSE);
                }
                for (i=0; i<data_length; i++) {
                  cmplx_val[i] = cmplx_data[data_length-1-i];
                }
                if (!kpds_put_data(dest_ref, KPDS_MASK_REGION, cmplx_val)) {
                   kerror(lib, rtn, "kpds_put_data failed for cmplx_val.");
                   return(FALSE);
                }
             }

          }
          if (bdata) kfree(bdata);
          if (bval) kfree(bval);
          if (idata) kfree(idata);
          if (ival) kfree(ival);
          if (ddata) kfree(ddata);
          if (dval) kfree(dval);
          if (cmplx_data) kfree(cmplx_data);
          if (cmplx_val) kfree(cmplx_val);
      }

   } /* end processing for MASK data */
  
   (void) kpds_close_object(src_ref); 
   (void) kpds_close_object(dest_ref); 

   return (TRUE);
}
/* -library_code_end */
