 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for kclip
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkclip
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
void kdmp_clip_uchar      PROTO((unsigned char *, unsigned char *,
                           unsigned char *, unsigned char, unsigned char,
                           unsigned char, unsigned char, int, int, int, 
			   int, int));

void kdmp_clip_ulong      PROTO((unsigned long *, unsigned long *,
                           unsigned long *, unsigned long, unsigned long,
                           unsigned long, unsigned long, int, int, int, 
			   int, int));

void kdmp_clip_long       PROTO((long *, long *,
                           long *, long, long,
                           long, long, int, int, int, int, int));

void kdmp_clip_double     PROTO((double *, double *,
                           double *, double, double,
                           double, double, int, int, int, int, int));

void kdmp_clip_dcomplex   PROTO((kdcomplex *, kdcomplex *,
                           kdcomplex *, kdcomplex, kdcomplex,
                           kdcomplex, kdcomplex, int, int, int, int, int));

void kdmp_thresh_uchar    PROTO((unsigned char *, unsigned char *,
                           unsigned char *, unsigned char, unsigned char,
                           unsigned char, unsigned char, int, int, int));

void kdmp_thresh_ulong    PROTO((unsigned long *, unsigned long *,
                           unsigned long *, unsigned long, unsigned long,
                           unsigned long, unsigned long, int, int, int));

void kdmp_thresh_long     PROTO((long *, long *,
                           long *, long, long,
                           long, long, int, int, int));

void kdmp_thresh_double   PROTO((double *, double *,
                           double *, double, double,
                           double, double, int, int, int));

void kdmp_thresh_dcomplex PROTO((kdcomplex *, kdcomplex *,
                           kdcomplex *, kdcomplex, kdcomplex,
                           kdcomplex, kdcomplex, int, int, int));
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkclip - clip or threshold values in the data object
* 
*       Purpose: Library Routine for kclip
*		 This function clips or thresholds the values in the 
*		 source object, src_obj and stores the result in the
*		 destination object, dst_obj.
*
*                Upper and/or lower cutoff values can be specified as 
*                a combination of the input constants (lval, uval) and 
*                the objects (lc_obj, uc_obj).  l_flag and u_flag are 
*		 flags that define which cutoffs are to be used.  If 
*                no cutoffs are flagged, lkclip prints an error message 
*                and retuns FALSE.  If a valid cutoff object exists, it
*                will be used instead of the corresponding constant value.
*
*		 The operation flag (op) defines whether the data should 
*		 be clipped (op=0) or thresholded (op = 1).
*
*		 Clipping:
*                When clipping, if the use_false and/or use_true flags 
*		 are set, the fval and/or tval passed in will be assigned 
*                as the lower result and the upper result for resulting 
*		 clipped data.  Otherwise, if use_false is FALSE, any data 
*		 clipped to the lower cutoff is set to the lower clipping 
*		 value (lower result = lc or lc_obj) instead of fval.  
*		 Likewise, if use_tval is FALSE, data clipped to the upper 
*		 cutoff is set to the upper clipping value (upper result = 
*		 uc or uc_obj).  Clipping results are assigned according 
*		 to the following rules.
*
*                Clipping Assingnment Rules:
*                1.  When lower cutoff < upper cutoff
*                    If the data value is less than the lower cutoff, set the
*                       output value to "lower result".
*                    If the data value is greater than the upper cutoff, set
*                       the output value to "upper result".
*                    Otherwise, the data value remains the same.
*
*                2.  When upper cutoff < lower cutoff
*                    If the data value is less than the upper cutoff and
*                       it is greater than the lower cutoff, set output
*                       value to the "lower result".
*                    Otherwise, the data value remains the same.
*
*		 Thresholding:
*		 Thresholding results are assigned according to the 
*		 following rules.
*
*                Thresholding Assingnment Rules:
*                1.  When lower cutoff < upper cutoff
*                    If the data value is less than the lower cutoff, or
*                       it is greater than the upper cutoff, set the output
*                       value to fval.
*                    Otherwise, set the output value to tval.
*
*                2.  When upper cutoff < lower cutoff
*                    If the data value is less than the lower cutoff and
*                       it is greater than the upper cutoff, set output
*                       value to fval.
*                    Otherwise, set the output value to tval.
*
*		 Map Data:
*		 If all source objects contain map data and no value 
*		 data, the operation is done directly on the map data.
*		 If only a souce object is passed in, and it contains
*		 map data, the operation is performed directly on the
*		 map.  Otherwise, if map and value data exist, the data
*		 is mapped before the operation is performed.
*		 If the destination object does not have the necessary map 
*		 and/or value segments, they are created for it.
*
*		 Location & Time Data:
*		 If more than one source object is supplied (src_obj, 
*		 lc_obj, uc_obj), and any of them contains time or 
*		 location data, lkclip will fail at this time.
*
*		 Mask Data:
*		 Data is masked according to the masking mode attribute
*		 of the source data objects.  The resulting object mask 
*		 is the logical AND of any input masks that exist.
*		 If the destination object does not have a mask, one is
*		 created for it.
*
*         Input: src_obj - the source object to be processed
*                lc_obj  - input object containing lower cutoff values
*                uc_obj  - input object containing upper cutoff values
*                lval    - lower clip/threshold value
*                uval    -  upper clip/threshold value
*                fval    - value assigned to points that are outside of the 
*			   clip/threshold range (ie. values less than lower 
*			   cutoff and/or greater than upper cutoff)
*                tval    - value assigned to points that are within the clip/
*			   threshold range (ie. values greater than or equal
*			   than the lower cutoff and less than or equal to 
*			   the upper cutoff)
*                l_flag  - if set, use lower cutoff value (lval or lc_obj)
*                u_flag  - if set, use upper cutoff value (uval or uc_obj)
*                use_false - if set, use fval as lower output value foR clip
*                use_true - if set, use tval as upper output value for clip 
*                op      - operation: clip = 0, threshold = 1
*        Output: dst_obj - the destination object
*
*       Returns: TRUE (1) on success, FALSE (0) on failure
*  Restrictions: 
*    Written By: Donna Koechner, Ashish Malhotra
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkclip (
   kobject src_obj,
   kobject lc_obj,
   kobject uc_obj,
   double  lval,
   double  uval,
   double  fval,
   double  tval,
   int     l_flag,
   int     u_flag,
   int     use_false,
   int     use_true,
   int     op,
   kobject dst_obj)
/* -library_def_end */

/* -library_code */
{
	kaddr  *sdata=NULL, *lcdata=NULL, *ucdata=NULL;
        int    type, num;
        int    w1, h1, d1, t1, e1;
        int    rw1, rh1, rd1, rt1, re1;
        int    num_pts, num_rgns;
	int    grid;
        int    error = FALSE;
	int    sval_exists=0, smask_exists=0, smap_exists=0;
	int    dval_exists=0, dmask_exists=0, dmap_exists=0;
	int    lcval_exists=0, lcmask_exists=0, lcmap_exists=0;
	int    ucval_exists=0, ucmask_exists=0, ucmap_exists=0;
	int    modify_map, modify_value, mult_modify_map, mult_modify_value;
        char   *region=NULL, *opt_region=NULL, *datasize=NULL, *datatype=NULL;
        char   *padtype=NULL, *interptype=NULL, *position=NULL; 
	char   *region_size=NULL;

	char   *prog="lkclip", *lib="kdatamanip";

        if ( (!u_flag) && (!l_flag) )
        {
           kerror(lib, prog, "l_flag and/or u_flag must be set.");
           kexit(KEXIT_FAILURE);
        }

        if ((src_obj == KOBJECT_INVALID) || (dst_obj == KOBJECT_INVALID))
	{
           kerror(lib, prog, "Invalid source and/or destination object");
           return(FALSE);
	}

	/*  
	 *  Reference the input data objects so that lkclip does not
	 *  have any side effects on them.
	 */
	if ((src_obj = kpds_reference_object(src_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, prog, "Failed to reference source object");
           return(FALSE);
        }

        if (l_flag && (lc_obj != KOBJECT_INVALID) && 
	   ((lc_obj = kpds_reference_object(lc_obj)) == KOBJECT_INVALID))
        {
           kpds_close_object(src_obj);
           kerror(lib, prog, "Failed to reference lower cutoff values object.");
           return(FALSE);
        }

        if (u_flag && (uc_obj != KOBJECT_INVALID) && 
	   ((uc_obj = kpds_reference_object(uc_obj)) == KOBJECT_INVALID))
        {
           (void) kpds_close_object(src_obj);
           if (lc_obj != NULL)  kpds_close_object(lc_obj);
           kerror(lib, prog, "Failed to reference upper cutoff values object.");
           return(FALSE);
        }

/* start utility routine here */
	/* 
	 *  Get information about segments that exist on objects
	 */
	sval_exists  = kpds_query_value(src_obj);
	smask_exists = kpds_query_mask(src_obj);
	smap_exists  = kpds_query_map(src_obj);
	dval_exists  = kpds_query_value(dst_obj);  
	dmask_exists = kpds_query_mask(dst_obj);  
	dmap_exists  = kpds_query_map(dst_obj);

	/*
	 *  Cases where the cutoff values are not input as objects. 
	 *
	 *  Case 1: modify_map
	 *  If single cutoff values are supplied, and a map exists (with or 
	 *  without value data), always operate directly on the map.
	 *  
	 *  Case 2: modify_value
	 *  If single cutoff values are supplied, and no map exists 
	 *  operate directly on the value data.  Use mask if it exists.
	 */
	modify_map = modify_value = mult_modify_map = mult_modify_value = FALSE;

	if ((lc_obj == NULL) && (uc_obj == NULL))
	{
	   if(smap_exists)
	   {
	      modify_map = TRUE;
	      if (!dmap_exists)
	      {
	         if (!kpds_create_map(dst_obj))
                 {
                    kpds_close_object(src_obj);
                    kerror(lib, prog, "Failed to create destination map.");
                    return(FALSE);
                 }
	      }
	   }
	   else if (sval_exists)
	   {
	      modify_value = TRUE;
	      if (!dval_exists)
	      {
	         if (!kpds_create_value(dst_obj))
                 {
                    kpds_close_object(src_obj);
                    kerror(lib, prog, "Failed to create destination value.");
                    return(FALSE);
                 }
	      }
	      if (smask_exists && (!dmask_exists))
	      {
	         if (!kpds_create_mask(dst_obj))
                 {
                    kpds_close_object(src_obj);
                    kerror(lib, prog, "Failed to create destination mask.");
                    return(FALSE);
                 }
	      }
	   }
	}

	/*  
	 *  Cases where at least one set of cutoff values is input as object. 
	 *
	 *  Case 3: mult_modify_map
	 *  If any cutoff values are specified as an object, and just
	 *  map data exist in all objects, operate directly on maps.
	 *
	 *  Case 4.a: mult_modify_value
	 *  If any cutoff values are specified as an object, and both
	 *  map and value data exist (so, all objects must have value 
	 *  data), map the data and operate on resulting value data.  If 
	 *  location or time, cannot operate in this case.  Use masks
	 *  if they exist.
	 *
	 *  Case 4.b: mult_modify_value
	 *  If only value data exists, operate on it.  If cutoff value
	 *  are supplied via objects, cannot deal with location or time
	 *  yet.  Use masks if they exist.
	 */
	else
	{
	   if (lc_obj != NULL)
	   {
	      lcval_exists  = kpds_query_value(lc_obj); 
	      lcmask_exists = kpds_query_mask(lc_obj); 
	      lcmap_exists  = kpds_query_map(lc_obj);
	   }
	   if (uc_obj != NULL)
	   {
	      ucval_exists  = kpds_query_value(uc_obj); 
	      ucmask_exists = kpds_query_mask(uc_obj); 
	      ucmap_exists  = kpds_query_map(uc_obj);
	   }

	   /*
	    *  All input objects have map and no value data.
	    */
	   if ( ((!sval_exists)   && (smap_exists))   &&
	        ((lc_obj == NULL) || ((!lcval_exists) && (lcmap_exists))) &&
	        ((uc_obj == NULL) || ((!ucval_exists) && (ucmap_exists))) )
	   {
	      mult_modify_map = TRUE;
	      if (!dmap_exists)
	      {
	         if (!kpds_create_map(dst_obj))
                 {
                    kerror(lib, prog, "Failed to create destination map.");
		    error = TRUE;
                 }
	      }
	   }
	   /*
	    *  All input objects have value and no map data.
	    */
	   else if ( ((sval_exists)   && (!smap_exists))   &&
	           ((lc_obj == NULL) || ((lcval_exists) && (!lcmap_exists))) &&
	           ((uc_obj == NULL) || ((ucval_exists) && (!ucmap_exists))) )
	   {
	      mult_modify_value = TRUE;
	      if (!dval_exists)
	      {
	         if (!kpds_create_value(dst_obj))
                 {
                    kerror(lib, prog, "Failed to create destination value.");
		    error = TRUE;
                 }
	      }
	   }
	   /*
	    *  Input objects have combinations of value and map data, so
	    *  data needs to be mapped before continuing.
	    */
	   else
	   {
	      mult_modify_value = TRUE;
	      if (smap_exists)
	      {
		 if (!kpds_set_attribute(src_obj, KPDS_MAPPING_MODE, KMAPPED))
                 {
                    kerror(lib, prog, "Failed to map source data.");
		    error = TRUE;
                 }
		 smap_exists = FALSE;
	      }
	      if ((lc_obj != NULL) && lcmap_exists)
	      {
		 if (!kpds_set_attribute(lc_obj, KPDS_MAPPING_MODE, KMAPPED))
                 {
                    kerror(lib, prog, "Failed to map lower cutoff data.");
		    error = TRUE;
                 }
		 lcmap_exists = FALSE;
	      }
	      if ((uc_obj != NULL) && ucmap_exists)
	      {
		 if (!kpds_set_attribute(uc_obj, KPDS_MAPPING_MODE, KMAPPED))
                 {
                    kerror(lib, prog, "Failed to map upper cutoff data.");
		    error = TRUE;
                 }
		 ucmap_exists = FALSE;
	      }
	   }
	}
/* end utility routine here */

	if (mult_modify_value)
	{
	   /* 
	    *  If there is more than one source object, and any of
	    *  them have rectilinear or curvilinear location data or
	    *  time data, cannot operate on them yet. 
	    */
	   if (kpds_query_location(src_obj))
	   {
	      if (!kpds_get_attribute(src_obj, KPDS_LOCATION_GRID, &grid))
	      {
		 kerror(lib, prog, 
			"Unable to retrieve location grid attribute.");
		 return(FALSE);
	      }
	      if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				       grid == KCURVILINEAR))
	      {
		 kerror(lib, prog, 
			"Source object cannot contain explicit rectilinear "
			"or curvilinear location data.");
		 return(FALSE);
	      }
	   }
	   if ((lc_obj != NULL) && kpds_query_location(lc_obj))
	   {
	      if (!kpds_get_attribute(lc_obj, KPDS_LOCATION_GRID, &grid))
	      {
		 kerror(lib, prog, 
			"Unable to retrieve location grid attribute.");
		 return(FALSE);
	      }
	      if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				       grid == KCURVILINEAR))
	      {
		 kerror(lib, prog, 
			"Source object cannot contain explicit rectilinear "
			"or curvilinear location data.");
		 return(FALSE);
	      }
	   }
	   if ((uc_obj != NULL) && kpds_query_location(uc_obj))
	   {
	      if (!kpds_get_attribute(uc_obj, KPDS_LOCATION_GRID, &grid))
	      {
		 kerror(lib, prog, 
			"Unable to retrieve location grid attribute.");
		 return(FALSE);
	      }
	      if (grid != KUNIFORM && (grid == KRECTILINEAR || 
				       grid == KCURVILINEAR))
	      {
		 kerror(lib, prog, 
			"Source object cannot contain explicit rectilinear "
			"or curvilinear location data.");
		 return(FALSE);
	      }
	   }
	   if ( kpds_query_time(src_obj)  || 
		((lc_obj != NULL) && kpds_query_time(lc_obj))  || 
	        ((uc_obj != NULL) && kpds_query_time(uc_obj)))
	   {
	      kerror(lib, prog,
			"Input objects cannot have time data.");
	      error = TRUE;
	   }
	}

	if (modify_map)
	{
	   region      = KPDS_MAP_REGION;
           region_size = KPDS_MAP_REGION_SIZE;
           opt_region  = KPDS_MAP_OPTIMAL_REGION_SIZE;
           datasize    = KPDS_MAP_SIZE;
           datatype    = KPDS_MAP_DATA_TYPE;
           padtype     = KPDS_MAP_PAD_VALUE;
           interptype  = KPDS_MAP_INTERPOLATE;
           position    = KPDS_MAP_POSITION;
        }
        else if (modify_value || mult_modify_value || mult_modify_map)
        {
           region      = KPDS_VALUE_REGION;
           region_size = KPDS_VALUE_REGION_SIZE;
           opt_region  = KPDS_VALUE_OPTIMAL_REGION_SIZE;
           datasize    = KPDS_VALUE_SIZE;
           datatype    = KPDS_VALUE_DATA_TYPE;
           padtype     = KPDS_VALUE_PAD_VALUE;
           interptype  = KPDS_VALUE_INTERPOLATE;
           position    = KPDS_VALUE_POSITION;
	   if (dmap_exists)
	   {
	      if (!kpds_destroy_map(dst_obj))
	      {
	         kerror(lib, prog, "Unable to destroy destination map.");
	         error = TRUE;
	      }
	   }
	   /*
	    *  If any mask exists, set up destination object mask.  
	    */
	   if (smask_exists || lcmask_exists || ucmask_exists)
	   {
	      if (!dmask_exists) 
	      {
	         if (!kpds_create_mask(dst_obj))
	         {
	            kerror(lib, prog, "Unable to create destination mask.");
	            error = TRUE;
	         }
	         dmask_exists = TRUE;
	      }
	   }
        }
	else
	{
	   kerror(lib, prog, "Did not determine how to process data.");
	   error = TRUE;
	}

        /*
         *  Get source object attributes.  Set lc_obj, uc_obj, 
	 *  and dst_obj to be the same size as src_obj.
         */
        if (!kpds_get_attributes(src_obj, 
		datasize, &w1, &h1, &d1, &t1, &e1,
		opt_region, &rw1, &rh1, &rd1, &rt1, &re1, &num_rgns,
		datatype, &type, NULL))
        {
           kerror(lib, prog, "Unable to get source data attributes.");
           error = TRUE;
        }

        if ((lc_obj != NULL) &&
           (!kpds_set_attributes(lc_obj, 
           	interptype, KPAD,
           	padtype, 0.0, 0.0,
		datasize, w1, h1, d1, t1, e1,
           	region_size, rw1, rh1, rd1, rt1, re1,
           	position, 0, 0, 0, 0, 0, NULL)))
        {
           kerror(lib, prog, "Unable to set lower value object attributes.");
           error = TRUE;
        }

        if ((uc_obj != NULL) &&
           (!kpds_set_attributes(uc_obj, 
           	interptype, KPAD,
           	padtype, 0.0, 0.0,
		datasize, w1, h1, d1, t1, e1,
           	region_size, rw1, rh1, rd1, rt1, re1,
           	position, 0, 0, 0, 0, 0, NULL)))
        {
           kerror(lib, prog, "Unable to set upper value object attributes.");
           error = TRUE;
        }

        if (!kpds_set_attributes(dst_obj, 
		datasize, w1, h1, d1, t1, e1,
		datatype, type, NULL))
        {
           kerror(lib, prog, "Unable to set destination object attributes.");
           error = TRUE;
        }

	/* 
	 *  Reference the destination object.
	 */
	if ((dst_obj = kpds_reference_object(dst_obj)) == KOBJECT_INVALID)
        {
           kerror(lib, prog, "Failed to reference destination object");
           error = TRUE;
        }

	if (error)
	{
	   kpds_close_object(src_obj);
	   if (lc_obj != NULL) kpds_close_object(lc_obj);
	   if (uc_obj != NULL) kpds_close_object(uc_obj);
	   return(FALSE);
	}

	/*
         *  Decide the data type to be used in processing and set it for
         *  src_obj, lc_obj, uc_obj, and dst_obj.
         */
	type = kdatatype_cast_process(type, type, 
			(KUBYTE | KLONG | KULONG | KDOUBLE | KDCOMPLEX));

        if (!kpds_set_attributes(src_obj, datatype, type,
			region_size, rw1, rh1, rd1, rt1, re1, NULL))
	{
           kerror(lib, prog, "Failed to set source object attributes.");
           error = TRUE;
	}

        if ((lc_obj != NULL) && (!kpds_set_attribute(lc_obj, datatype, type)))
        {
           kerror(lib, prog, "Failed to set lower cutoff object data type.");
           error = TRUE;
        }
        
        if ((uc_obj != NULL) && (!kpds_set_attribute(uc_obj, datatype, type)))
        {
           kerror(lib, prog, "Failed to set upper cutoff object data type.");
           error = TRUE;
        }

        if (!kpds_set_attributes(dst_obj, datatype, type,
			region_size, rw1, rh1, rd1, rt1, re1, 
			position, 0, 0, 0, 0, 0, NULL))
        {
           kerror(lib, prog, "Failed to set destination object attributes.");
           error = TRUE;
        }

	if (error)
	{
	   kpds_close_object(src_obj);
	   kpds_close_object(dst_obj);
	   if (lc_obj != NULL) kpds_close_object(lc_obj);
	   if (uc_obj != NULL) kpds_close_object(uc_obj);
	   return(FALSE);
	}

	num_pts = rw1 * rh1 * rd1 * rt1 * re1;

        for (num = 0; num < num_rgns; num++)
        {
           if ((sdata = kpds_get_data(src_obj, region, sdata)) == NULL)
	   {
	      kerror(lib,prog,
		"Get data failed for source object for region %d.", num);
	      error = TRUE;
	      break;
	   }

           if (lc_obj != NULL)
           {
             if ((lcdata = kpds_get_data(lc_obj, region, lcdata)) == NULL)
	     {
	        kerror(lib,prog,
		  "Get data failed for lower cutoff object for region %d.",num);
	        error = TRUE;
	        break;
	     }
           }
           if (uc_obj != NULL)
           {
             if ((ucdata = kpds_get_data(uc_obj, region, ucdata)) == NULL)
	     {
	        kerror(lib,prog,
		  "Get data failed for upper cutoff object for region %d.",num);
	        error = TRUE;
	        break;
	     }
           }

           if (type == KUBYTE)
           {
	      if (op == 0)
	      {
              kdmp_clip_uchar((unsigned char *) sdata, 
	         (unsigned char *) lcdata, (unsigned char *) ucdata,
                 (unsigned char) lval, (unsigned char) uval, 
                 (unsigned char) fval, (unsigned char) tval, 
		 l_flag, u_flag, use_false, use_true, num_pts);
	      }
	      else
	      {
              kdmp_thresh_uchar((unsigned char *) sdata, 
	         (unsigned char *) lcdata, (unsigned char *) ucdata,
		 (unsigned char) lval, (unsigned char) uval, 
		 (unsigned char) fval, (unsigned char) tval, 
		 l_flag, u_flag, num_pts);
	      }
           }

           else if (type == KULONG)
           {
	      if (op == 0)
	      {
                 kdmp_clip_ulong((unsigned long *) sdata, 
		    (unsigned long *) lcdata, (unsigned long *) ucdata,
                    (unsigned long) lval, (unsigned long) uval, 
                    (unsigned long) fval, (unsigned long) tval, 
		    l_flag, u_flag, use_false, use_true, num_pts);
	      }
	      else
	      {
                 kdmp_thresh_ulong((unsigned long *) sdata, 
		    (unsigned long *) lcdata, (unsigned long *) ucdata,
                    (unsigned long) lval, (unsigned long) uval, 
                    (unsigned long) fval, (unsigned long) tval, 
		    l_flag, u_flag, num_pts);
	      }
           }

           else if (type == KLONG)
           {
	      if (op == 0)
	      {
                 kdmp_clip_long((long *) sdata, (long *) lcdata, 
		    (long *) ucdata, (long) lval, (long) uval, (long) fval, 
		    (long) tval, l_flag, u_flag, use_false, use_true, num_pts);
	      }
	      else
	      {
                 kdmp_thresh_long((long *) sdata, (long *) lcdata, 
		    (long *) ucdata, (long) lval, (long) uval, (long) fval, 
		    (long) tval, l_flag, u_flag, num_pts);
	      }
           }

           else if (type == KDOUBLE)
           {
              if (op == 0)
              {
                 kdmp_clip_double((double *) sdata, (double *) lcdata, 
		    (double *) ucdata, (double) lval, (double) uval, 
                    (double) fval, (double) tval, 
		    l_flag, u_flag, use_false, use_true, num_pts);
              }
              else
              {
                 kdmp_thresh_double((double *) sdata, (double *) lcdata, 
		    (double *) ucdata, (double) lval, (double) uval, 
                    (double) fval, (double) tval, l_flag, u_flag, num_pts);
              }
           }

           else if (type == KDCOMPLEX)
           {
              if (op == 0)
              {
                 kdmp_clip_dcomplex((kdcomplex *) sdata, 
		    (kdcomplex *) lcdata, (kdcomplex *) ucdata, 
		    (kdccomp(lval,lval)), (kdccomp(uval,uval)), 
		    (kdccomp(fval,fval)), (kdccomp(tval,tval)), 
		    l_flag, u_flag, use_false, use_true, num_pts);
              }
              else
              {
                 kdmp_thresh_dcomplex((kdcomplex *) sdata, 
		    (kdcomplex *) lcdata, (kdcomplex *) ucdata, 
		    (kdccomp(lval,lval)), (kdccomp(uval,uval)), 
		    (kdccomp(fval,fval)), (kdccomp(tval,tval)), 
		    l_flag, u_flag, num_pts);
              }
           }

           else
           {
             /* unrecognized processing type - print error and get out */
              error = TRUE;
           }

           if (! kpds_put_data(dst_obj, region, (kaddr) sdata) )
           {
              kerror(lib, prog,
                     "Failed to put %s data in output object.", region);
              error = TRUE;
           }
	   if (error)  break;
        } 

	/* 
	 *  Create correct output mask 
	 */
	if (!error && mult_modify_value)
	{
	   if (!kdmanip_mask_ops(dst_obj, KDMANIP_MASK_AND, 
	      3, src_obj, lc_obj, uc_obj))
	      error = TRUE;
	}

        kpds_close_object(src_obj);
        if (lc_obj) (void) kpds_close_object(lc_obj);
        if (uc_obj) (void) kpds_close_object(uc_obj);
	kpds_close_object(dst_obj);
        kfree(sdata);
        kfree(ucdata);
        kfree(lcdata);
        if (error)
           return(FALSE);
        else
           return(TRUE);
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_clip_uchar - clip unsigned char data
|
|       Purpose: This function clips the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 If the use_low_val and/or use_high_val flags are true, the
|		 low_val and/or high_val passed in will be assigned to the
|		 results according to the rules below.  If use_low_val is 
|		 false, low_val is set to the lower clipping value.  If 
|		 use_high_val is false, high_val is set to the upper 
|		 clipping value.
|		 
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, set the 
|		        output value to the low_val.  
|		     If the data value is greater than the upper cutoff, set 
|			the output value to the high_val.  
|		     Otherwise, the data value remains the same.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the upper cutoff and
|		        it is greater than the lower cutoff, set output 
|		        value to the low_val.  
|		     Otherwise, the data value remains the same.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                low_val - value assigned if data falls outside of cutoff range
|                high_val - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|                use_low_val - if set, use specified low_val
|                use_high_val - if set, use specified high_val
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of clip operation are stored in "data"
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_clip_uchar(
   unsigned char *data,
   unsigned char *lc_array,
   unsigned char *uc_array,
   unsigned char lc, 
   unsigned char uc,
   unsigned char low_val,
   unsigned char high_val,
   int l_flag, 
   int u_flag, 
   int use_low_val, 
   int use_high_val, 
   int num_pts)
{
        int j;
	unsigned char lower_bound=0, upper_bound=0;
	unsigned char low_replace, high_replace;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array==NULL) && (uc_array==NULL))
	{
	   low_replace = ((use_low_val) ? low_val : lc);
	   high_replace = ((use_high_val) ? high_val : uc);
	   if (l_flag && u_flag)
           {
	      if (uc < lc)
	      {
	         for (j=0; j < num_pts; j++)
	            data[j] = ((data[j]>uc && data[j]<lc) ? low_val : data[j]);
	      }  
	      else
	      {
	         for (j=0; j < num_pts; j++)
		 {
		    if      (data[j]<lc)   data[j] = low_replace;
		    else if (data[j]>uc)   data[j] = high_replace;
		 }
	      }
	   }
	   else if (l_flag && !u_flag)
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? low_replace  : data[j];
	   }
	   else
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? high_replace : data[j];
           }
	}
	else
	{
           if (l_flag && u_flag) 
           {
              for (j=0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
   
                 if (upper_bound < lower_bound)
                 {
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                   ? low_val : data[j];
                 }      
                 else
                 {
                    if      (data[j]<lower_bound)  data[j] = low_replace;
                    else if (data[j]>upper_bound)  data[j] = high_replace;
                 }
              }
	   }
           else if (l_flag && !u_flag)
	   {
              for (j=0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
                 data[j] = (data[j]<lower_bound) ? low_replace  : data[j];
	      }
	   }
              else
	   {
              for (j=0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
                 data[j] = (data[j]>upper_bound) ? high_replace : data[j];
	      }
           }
        }
}

/*-----------------------------------------------------------
|
|  Routine Name: kdmp_clip_ulong - clip unsigned long integer data
|
|       Purpose: This function clips the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 If the use_low_val and/or use_high_val flags are true, the
|		 low_val and/or high_val passed in will be assigned to the
|		 results according to the rules below.  If use_low_val is 
|		 false, low_val is set to the lower clipping value.  If 
|		 use_high_val is false, high_val is set to the upper 
|		 clipping value.
|		 
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, set the 
|		        output value to the low_val.  
|		     If the data value is greater than the upper cutoff, set 
|			the output value to the high_val.  
|		     Otherwise, the data value remains the same.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the upper cutoff and
|		        it is greater than the lower cutoff, set output 
|		        value to the low_val.  
|		     Otherwise, the data value remains the same.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                low_val - value assigned if data falls outside of cutoff range
|                high_val - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|                use_low_val - if set, use specified low_val
|                use_high_val - if set, use specified high_val
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of clip operation are stored in "data"
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_clip_ulong(
   unsigned long *data,
   unsigned long *lc_array,
   unsigned long *uc_array,
   unsigned long lc, 
   unsigned long uc,
   unsigned long low_val,
   unsigned long high_val,
   int l_flag, 
   int u_flag, 
   int use_low_val, 
   int use_high_val, 
   int num_pts)
{
        int j;
	unsigned long lower_bound=0, upper_bound=0;
	unsigned long low_replace, high_replace;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array==NULL) && (uc_array==NULL))
	{
	   low_replace = ((use_low_val) ? low_val : lc);
	   high_replace = ((use_high_val) ? high_val : uc);
	   if (l_flag && u_flag)
           {
	      if (uc < lc)
	      {
	         for (j=0; j < num_pts; j++)
	            data[j] = ((data[j]>uc && data[j]<lc) ? low_val : data[j]);
	      }  
	      else
	      {
	         for (j=0; j < num_pts; j++)
		 {
		    if      (data[j]<lc)   data[j] = low_replace;
		    else if (data[j]>uc)   data[j] = high_replace;
		 }
	      }
	   }
	   else if (l_flag && !u_flag)
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? low_replace  : data[j];
	   }
	   else
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? high_replace : data[j];
           }
	}
	else
	{
           if (l_flag && u_flag) 
           {
              for (j=0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
   
                 if (upper_bound < lower_bound)
                 {
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                   ? low_val : data[j];
                 }      
                 else
                 {
                    if      (data[j]<lower_bound)  data[j] = low_replace;
                    else if (data[j]>upper_bound)  data[j] = high_replace;
                 }
              }
	   }
           else if (l_flag && !u_flag)
	   {
              for (j=0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
                 data[j] = (data[j]<lower_bound) ? low_replace  : data[j];
	      }
	   }
              else
	   {
              for (j=0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
                 data[j] = (data[j]>upper_bound) ? high_replace : data[j];
	      }
           }
        }
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_clip_long - clip long integer data
|
|       Purpose: This function clips the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 If the use_low_val and/or use_high_val flags are true, the
|		 low_val and/or high_val passed in will be assigned to the
|		 results according to the rules below.  If use_low_val is 
|		 false, low_val is set to the lower clipping value.  If 
|		 use_high_val is false, high_val is set to the upper 
|		 clipping value.
|		 
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, set the 
|		        output value to the low_val.  
|		     If the data value is greater than the upper cutoff, set 
|			the output value to the high_val.  
|		     Otherwise, the data value remains the same.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the upper cutoff and
|		        it is greater than the lower cutoff, set output 
|		        value to the low_val.  
|		     Otherwise, the data value remains the same.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                low_val - value assigned if data falls outside of cutoff range
|                high_val - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|                use_low_val - if set, use specified low_val
|                use_high_val - if set, use specified high_val
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of clip operation are stored in "data"
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_clip_long(
   long *data,
   long *lc_array,
   long *uc_array,
   long lc, 
   long uc,
   long low_val,
   long high_val,
   int l_flag, 
   int u_flag, 
   int use_low_val, 
   int use_high_val, 
   int num_pts)
{
        int j;
	long lower_bound=0, upper_bound=0;
	long low_replace, high_replace;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array==NULL) && (uc_array==NULL))
	{
	   low_replace = ((use_low_val) ? low_val : lc);
	   high_replace = ((use_high_val) ? high_val : uc);
	   if (l_flag && u_flag)
           {
	      if (uc < lc)
	      {
	         for (j=0; j < num_pts; j++)
	            data[j] = ((data[j]>uc && data[j]<lc) ? low_val : data[j]);
	      }  
	      else
	      {
	         for (j=0; j < num_pts; j++)
		 {
		    if      (data[j]<lc)   data[j] = low_replace;
		    else if (data[j]>uc)   data[j] = high_replace;
		 }
	      }
	   }
	   else if (l_flag && !u_flag)
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? low_replace  : data[j];
	   }
	   else
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? high_replace : data[j];
           }
	}
	else
	{
           if (l_flag && u_flag) 
           {
              for (j=0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
   
                 if (upper_bound < lower_bound)
                 {
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                   ? low_val : data[j];
                 }      
                 else
                 {
                    if      (data[j]<lower_bound)  data[j] = low_replace;
                    else if (data[j]>upper_bound)  data[j] = high_replace;
                 }
              }
	   }
           else if (l_flag && !u_flag)
	   {
              for (j=0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
                 data[j] = (data[j]<lower_bound) ? low_replace  : data[j];
	      }
	   }
              else
	   {
              for (j=0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
                 data[j] = (data[j]>upper_bound) ? high_replace : data[j];
	      }
           }
        }
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_clip_double - clip double data
|
|       Purpose: This function clips the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 If the use_low_val and/or use_high_val flags are true, the
|		 low_val and/or high_val passed in will be assigned to the
|		 results according to the rules below.  If use_low_val is 
|		 false, low_val is set to the lower clipping value.  If 
|		 use_high_val is false, high_val is set to the upper 
|		 clipping value.
|		 
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, set the 
|		        output value to the low_val.  
|		     If the data value is greater than the upper cutoff, set 
|			the output value to the high_val.  
|		     Otherwise, the data value remains the same.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the upper cutoff and
|		        it is greater than the lower cutoff, set output 
|		        value to the low_val.  
|		     Otherwise, the data value remains the same.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                low_val - value assigned if data falls outside of cutoff range
|                high_val - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|                use_low_val - if set, use specified low_val
|                use_high_val - if set, use specified high_val
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of clip operation are stored in "data"
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_clip_double(
   double *data,
   double *lc_array,
   double *uc_array,
   double lc, 
   double uc,
   double low_val,
   double high_val,
   int l_flag, 
   int u_flag, 
   int use_low_val, 
   int use_high_val, 
   int num_pts)
{
        int j;
	double lower_bound=0, upper_bound=0;
	double low_replace, high_replace;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array==NULL) && (uc_array==NULL))
	{
	   low_replace = ((use_low_val) ? low_val : lc);
	   high_replace = ((use_high_val) ? high_val : uc);
	   if (l_flag && u_flag)
           {
	      if (uc < lc)
	      {
	         for (j=0; j < num_pts; j++)
	            data[j] = ((data[j]>uc && data[j]<lc) ? low_val : data[j]);
	      }  
	      else
	      {
	         for (j=0; j < num_pts; j++)
		 {
		    if      (data[j]<lc)   data[j] = low_replace;
		    else if (data[j]>uc)   data[j] = high_replace;
		 }
	      }
	   }
	   else if (l_flag && !u_flag)
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? low_replace  : data[j];
	   }
	   else
	   {
	      for (j=0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? high_replace : data[j];
           }
	}
	else
	{
           if (l_flag && u_flag) 
           {
              for (j=0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
   
                 if (upper_bound < lower_bound)
                 {
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                   ? low_val : data[j];
                 }      
                 else
                 {
                    if      (data[j]<lower_bound)  data[j] = low_replace;
                    else if (data[j]>upper_bound)  data[j] = high_replace;
                 }
              }
	   }
           else if (l_flag && !u_flag)
	   {
              for (j=0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
                 data[j] = (data[j]<lower_bound) ? low_replace  : data[j];
	      }
	   }
              else
	   {
              for (j=0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
                 data[j] = (data[j]>upper_bound) ? high_replace : data[j];
	      }
           }
        }
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_clip_dcomplex - clip double complex data
|
|       Purpose: This function clips the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 If the use_low_val and/or use_high_val flags are true, the
|		 low_val and/or high_val passed in will be assigned to the
|		 results according to the rules below.  If use_low_val is 
|		 false, low_val is set to the lower clipping value.  If 
|		 use_high_val is false, high_val is set to the upper 
|		 clipping value.
|		 
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, set the 
|		        output value to the low_val.  
|		     If the data value is greater than the upper cutoff, set 
|			the output value to the high_val.  
|		     Otherwise, the data value remains the same.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the upper cutoff and
|		        it is greater than the lower cutoff, set output 
|		        value to the low_val.  
|		     Otherwise, the data value remains the same.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                low_val - value assigned if data falls outside of cutoff range
|                high_val - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|                use_low_val - if set, use specified low_val
|                use_high_val - if set, use specified high_val
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of clip operation are stored in "data"
|       Returns: nothing
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_clip_dcomplex(
   kdcomplex *data,
   kdcomplex *lc_array,
   kdcomplex *uc_array,
   kdcomplex lc, 
   kdcomplex uc,
   kdcomplex low_val,
   kdcomplex high_val,
   int l_flag, 
   int u_flag, 
   int use_low_val, 
   int use_high_val, 
   int num_pts)
{
        int j;
	kdcomplex lower_bound, upper_bound;
	kdcomplex low_replace, high_replace;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array==NULL) && (uc_array==NULL))
	{
	   low_replace = ((use_low_val) ? low_val : lc);
	   high_replace = ((use_high_val) ? high_val : uc);
           if (l_flag && u_flag) 
           {
              for (j=0; j < num_pts; j++)
	      {
	         if (uc.r < lc.r)
		 {
	            data[j].r = ((data[j].r>uc.r && data[j].r<lc.r) 
				   ? low_val.r : data[j].r);
		 }
		 else
		 {
		    if (data[j].r<lc.r)        data[j].r = low_replace.r;
		    else if (data[j].r>uc.r)  data[j].r = high_replace.r;
		 }
	         if (uc.i < lc.i)
		 {
	            data[j].i = ((data[j].i>uc.i && data[j].i<lc.i) 
				   ? low_val.i : data[j].i);
		 }
		 else
		 {
		    if (data[j].i<lc.i)        data[j].i = low_replace.i;
		    else if (data[j].i>uc.i)  data[j].i = high_replace.i;
		 }
	      }
	   }
	   else if (l_flag) 
           {
              for (j=0; j < num_pts; j++)
	      {
		 data[j].r = (data[j].r<lc.r) ? low_replace.r  : data[j].r;
		 data[j].i = (data[j].i<lc.i) ? low_replace.i  : data[j].i;
	      }
	   }
	   else
	   {
              for (j=0; j < num_pts; j++)
	      {
		 data[j].r = (data[j].r>uc.r) ? high_replace.r : data[j].r;
		 data[j].i = (data[j].i>uc.i) ? high_replace.i : data[j].i;
	      }
           }
	}
	else
	{
           if (l_flag && u_flag) 
           {
              for (j=0; j < num_pts; j++)
	      {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
   
	         if (upper_bound.r < lower_bound.r)
		 {
                    data[j].r = (data[j].r>upper_bound.r && 
			data[j].r<lower_bound.r) ? low_val.r : data[j].r;
		 }
		 else
		 {
		    if (data[j].r<lower_bound.r)  data[j].r = low_replace.r;
	            else if (data[j].r>upper_bound.r) data[j].r=high_replace.r;
		 }
	         if (upper_bound.i < lower_bound.i)
		 {
                    data[j].i = (data[j].i>upper_bound.i && 
			data[j].i<lower_bound.i) ? low_val.i : data[j].i;
		 }
		 else
		 {
		    if (data[j].i<lower_bound.i)  data[j].i = low_replace.i;
	            else if (data[j].i>upper_bound.i) data[j].i=high_replace.i;
		 }
	      }
	   }
	   else if (l_flag)
	   {
              for (j=0; j < num_pts; j++)
	      {
	         /* Set lower bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
	         low_replace = ((use_low_val) ? low_val : lower_bound);

	         data[j].r = (data[j].r<lower_bound.r) 
					? low_replace.r  : data[j].r;
	         data[j].i = (data[j].i<lower_bound.i) 
					? low_replace.i  : data[j].i;
              }
	   }
	   else
	   {
              for (j=0; j < num_pts; j++)
	      {
	         /* Set upper bounds */
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	         high_replace = ((use_high_val) ? high_val : upper_bound);
	         data[j].r = (data[j].r>upper_bound.r) 
					? high_replace.r : data[j].r;
	         data[j].i = (data[j].i>upper_bound.i) 
					? high_replace.i : data[j].i;
              }
           }
        }
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_thresh_uchar - threshold unsigned char data
|
|       Purpose: This function thresholds the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, or
|		        it is greater than the upper cutoff, set the output 
|			value to fval.  
|		     Otherwise, set the output value to tval.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the lower cutoff and
|		        it is greater than the upper cutoff, set output 
|		        value to fval.  
|		     Otherwise, set the output value to tval.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                fval - value assigned if data falls outside of cutoff range
|                tval - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of threshold operation are stored in "data"
|       Returns: nada
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_thresh_uchar(
   unsigned char *data,
   unsigned char *lc_array,
   unsigned char *uc_array,
   unsigned char lc, 
   unsigned char uc,
   unsigned char fval,
   unsigned char tval,
   int l_flag, 
   int u_flag, 
   int num_pts)
{
        int j;
	unsigned char lower_bound=0, upper_bound=0;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array == NULL) && (uc_array == NULL))
	{
           if (l_flag && u_flag) 
           {
	      for (j = 0; j < num_pts; j++)
	      {
                 if (uc < lc)
	            data[j] = (data[j]>uc && data[j]<lc) ? fval : tval;
		 else
	            data[j] = (data[j]>uc || data[j]<lc) ? fval : tval;
	      }
	   }
	   else if (l_flag && ! u_flag)
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? fval : tval;
	   }
	   else
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? fval : tval;
	   }
	}
	else
	{
	   if (l_flag && u_flag)
           {
              for (j = 0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	      
                 if (upper_bound < lower_bound)
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                ? fval : tval;
                 else
                    data[j] = (data[j]>upper_bound || data[j]<lower_bound)
                                ? fval : tval;
              }
	   }
           else if (l_flag)
	   {
              for (j = 0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 data[j] = (data[j]<lower_bound) ? fval : tval;
	      }
	   }
           else
	   {
              for (j = 0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
                 data[j] = (data[j]>upper_bound) ? fval : tval;
	      }
           }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_thresh_ulong - threshold unsigned long integer data
|
|       Purpose: This function thresholds the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, or
|		        it is greater than the upper cutoff, set the output 
|			value to fval.  
|		     Otherwise, set the output value to tval.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the lower cutoff and
|		        it is greater than the upper cutoff, set output 
|		        value to fval.  
|		     Otherwise, set the output value to tval.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                fval - value assigned if data falls outside of cutoff range
|                tval - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of threshold operation are stored in "data"
|       Returns: nada
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_thresh_ulong(
   unsigned long *data,
   unsigned long *lc_array,
   unsigned long *uc_array,
   unsigned long lc, 
   unsigned long uc,
   unsigned long fval,
   unsigned long tval,
   int l_flag, 
   int u_flag, 
   int num_pts)
{
        int j;
	unsigned long lower_bound=0, upper_bound=0;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array == NULL) && (uc_array == NULL))
	{
           if (l_flag && u_flag) 
           {
	      for (j = 0; j < num_pts; j++)
	      {
                 if (uc < lc)
	            data[j] = (data[j]>uc && data[j]<lc) ? fval : tval;
		 else
	            data[j] = (data[j]>uc || data[j]<lc) ? fval : tval;
	      }
	   }
	   else if (l_flag && ! u_flag)
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? fval : tval;
	   }
	   else
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? fval : tval;
	   }
	}
	else
	{
	   if (l_flag && u_flag)
           {
              for (j = 0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	      
                 if (upper_bound < lower_bound)
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                ? fval : tval;
                 else
                    data[j] = (data[j]>upper_bound || data[j]<lower_bound)
                                ? fval : tval;
              }
	   }
           else if (l_flag)
	   {
              for (j = 0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 data[j] = (data[j]<lower_bound) ? fval : tval;
	      }
	   }
           else
	   {
              for (j = 0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
                 data[j] = (data[j]>upper_bound) ? fval : tval;
	      }
           }
	}
}

/*-----------------------------------------------------------
|
|  Routine Name: kdmp_thresh_long - threshold long integer data
|
|       Purpose: This function thresholds the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, or
|		        it is greater than the upper cutoff, set the output 
|			value to fval.  
|		     Otherwise, set the output value to tval.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the lower cutoff and
|		        it is greater than the upper cutoff, set output 
|		        value to fval.  
|		     Otherwise, set the output value to tval.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                fval - value assigned if data falls outside of cutoff range
|                tval - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of threshold operation are stored in "data"
|       Returns: nada
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_thresh_long(
   long *data,
   long *lc_array,
   long *uc_array,
   long lc, 
   long uc,
   long fval,
   long tval,
   int l_flag, 
   int u_flag, 
   int num_pts)
{
        int j;
	long lower_bound=0, upper_bound=0;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array == NULL) && (uc_array == NULL))
	{
           if (l_flag && u_flag) 
           {
	      for (j = 0; j < num_pts; j++)
	      {
                 if (uc < lc)
	            data[j] = (data[j]>uc && data[j]<lc) ? fval : tval;
		 else
	            data[j] = (data[j]>uc || data[j]<lc) ? fval : tval;
	      }
	   }
	   else if (l_flag && ! u_flag)
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? fval : tval;
	   }
	   else
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? fval : tval;
	   }
	}
	else
	{
	   if (l_flag && u_flag)
           {
              for (j = 0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	      
                 if (upper_bound < lower_bound)
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                ? fval : tval;
                 else
                    data[j] = (data[j]>upper_bound || data[j]<lower_bound)
                                ? fval : tval;
              }
	   }
           else if (l_flag)
	   {
              for (j = 0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 data[j] = (data[j]<lower_bound) ? fval : tval;
	      }
	   }
           else
	   {
              for (j = 0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
                 data[j] = (data[j]>upper_bound) ? fval : tval;
	      }
           }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_thresh_double - threshold double data
|
|       Purpose: This function thresholds the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, or
|		        it is greater than the upper cutoff, set the output 
|			value to fval.  
|		     Otherwise, set the output value to tval.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the lower cutoff and
|		        it is greater than the upper cutoff, set output 
|		        value to fval.  
|		     Otherwise, set the output value to tval.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                fval - value assigned if data falls outside of cutoff range
|                tval - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of threshold operation are stored in "data"
|       Returns: nada
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_thresh_double(
   double *data,
   double *lc_array,
   double *uc_array,
   double lc, 
   double uc,
   double fval,
   double tval,
   int l_flag, 
   int u_flag, 
   int num_pts)
{
        int j;
	double lower_bound=0, upper_bound=0;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array == NULL) && (uc_array == NULL))
	{
           if (l_flag && u_flag) 
           {
	      for (j = 0; j < num_pts; j++)
	      {
                 if (uc < lc)
	            data[j] = (data[j]>uc && data[j]<lc) ? fval : tval;
		 else
	            data[j] = (data[j]>uc || data[j]<lc) ? fval : tval;
	      }
	   }
	   else if (l_flag && ! u_flag)
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]<lc) ? fval : tval;
	   }
	   else
	   {
	      for (j = 0; j < num_pts; j++)
	         data[j] = (data[j]>uc) ? fval : tval;
	   }
	}
	else
	{
	   if (l_flag && u_flag)
           {
              for (j = 0; j < num_pts; j++)
              {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	      
                 if (upper_bound < lower_bound)
                    data[j] = (data[j]>upper_bound && data[j]<lower_bound)
                                ? fval : tval;
                 else
                    data[j] = (data[j]>upper_bound || data[j]<lower_bound)
                                ? fval : tval;
              }
	   }
           else if (l_flag)
	   {
              for (j = 0; j < num_pts; j++)
              {
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 data[j] = (data[j]<lower_bound) ? fval : tval;
	      }
	   }
           else
	   {
              for (j = 0; j < num_pts; j++)
              {
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
                 data[j] = (data[j]>upper_bound) ? fval : tval;
	      }
           }
	}
}


/*-----------------------------------------------------------
|
|  Routine Name: kdmp_thresh_dcomplex - threshold dcomplex data
|
|       Purpose: This function thresholds the values in the array called data.  
|                The upper and/or lower cutoff values can be specified as a 
|		 combination of the constants (lc, uc) and arrays (lc_array, 
|		 uc_array).  l_flag and u_flag are flags that define which 
|		 cutoffs are to be used.  If no cutoffs are flagged, no 
|		 operation is performed.  If a cutoff array is non-NULL, it 
|		 will be used instead of the corresponding constant value.  
|
|		 Assingnment Rules: 
|		 1.  lower cutoff < upper cutoff
|		     If the data value is less than the lower cutoff, or
|		        it is greater than the upper cutoff, set the output 
|			value to fval.  
|		     Otherwise, set the output value to tval.
|		 
|		 2.  upper cutoff < lower cutoff
|		     If the data value is less than the lower cutoff and
|		        it is greater than the upper cutoff, set output 
|		        value to fval.  
|		     Otherwise, set the output value to tval.
|
|		 If lc_array or uc_array are non-NULL, they must be 
|		 the same size as data (num_pts).
|
|		 The result is stored in the array called data.
|
|         Input: data - data array (num_pts long)
|                lc_array - array of lower cutoff values (num_pts long)
|                uc_array - array of upper cutoff values (num_pts long)
|                lc - single value defining lower cutoff
|                uc - single value defining upper cutoff
|                fval - value assigned if data falls outside of cutoff range
|                tval - value assigned if data falls within cutoff range
|                l_flag - if set, use lower cutoff
|                u_flag - if set, use upper cutoff
|		 num_pts - number of points in data, lc_array, and uc_array 
|
|        Output: data - results of threshold operation are stored in "data"
|       Returns: nada
|
|    Written By: Donna Koechner
|          Date: Tue Jun  7 1994
| Modifications:
|
------------------------------------------------------------*/

void kdmp_thresh_dcomplex(
   kdcomplex *data,
   kdcomplex *lc_array,
   kdcomplex *uc_array,
   kdcomplex lc, 
   kdcomplex uc,
   kdcomplex fval,
   kdcomplex tval,
   int l_flag, 
   int u_flag, 
   int num_pts)
{
        int j;
	kdcomplex lower_bound, upper_bound;

	if (!l_flag && !u_flag)
	   return;

	if ((lc_array == NULL) && (uc_array == NULL))
	{
	   if (l_flag && u_flag) 
           {
              for (j = 0; j < num_pts; j++)
	      {
                 if (uc.r < lc.r)
		 {
		    data[j].r = (data[j].r>uc.r && data[j].r<lc.r) 
				? fval.r : tval.r;
		 }
		 else
		 {
		    data[j].r = (data[j].r>uc.r || data[j].r<lc.r) 
				? fval.r : tval.r;
		 }
                 if (uc.i < lc.i)
		 {
		    data[j].i = (data[j].i>uc.i && data[j].i<lc.i) 
				? tval.i : fval.i;
		 }
		 else
		 {
		    data[j].i = (data[j].i>uc.i || data[j].i<lc.i) 
				? fval.i : tval.i;
		 }
	      }
	   }
	   else if (l_flag)  
	   {
              for (j = 0; j < num_pts; j++)
	      {
		 data[j].r = (data[j].r<lc.r) ? fval.r : tval.r;
		 data[j].i = (data[j].i<lc.i) ? fval.i : tval.i;
	      }
	   }
	   else
           {
              for (j = 0; j < num_pts; j++)
	      {
		 data[j].r = (data[j].r>uc.r) ? fval.r : tval.r;
		 data[j].i = (data[j].i>uc.i) ? fval.i : tval.i;
	      }
	   }
	}
	else
	{
	   if (l_flag && u_flag) 
           {
              for (j = 0; j < num_pts; j++)
	      {
	         /* Set lower and upper bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
	      
	         if (upper_bound.r < lower_bound.r)
		 {
	            data[j].r = (data[j].r>upper_bound.r && 
			      data[j].r<lower_bound.r) ? fval.r : tval.r;
		 }
		 else
		 {
	            data[j].r = (data[j].r>upper_bound.r || 
			      data[j].r<lower_bound.r) ? fval.r : tval.r;
		 }
	         if (upper_bound.i < lower_bound.i)
		 {
	            data[j].i = (data[j].i>upper_bound.i && 
			      data[j].i<lower_bound.i) ? fval.i : tval.i;
		 }
		 else
		 {
	            data[j].i = (data[j].i>upper_bound.i || 
			      data[j].i<lower_bound.i) ? fval.i : tval.i;
		 }
	      }
	   }
	   else if (l_flag)  
	   {
              for (j = 0; j < num_pts; j++)
	      {
	         /* Set lower bounds */
                 lower_bound = ((lc_array==NULL) ? lc : lc_array[j]);
		 data[j].r = (data[j].r<lower_bound.r) ? fval.r : tval.r;
		 data[j].i = (data[j].i<lower_bound.i) ? fval.i : tval.i;
	      }
           }
	   else
	   {
              for (j = 0; j < num_pts; j++)
	      {
	         /* Set upper bounds */
                 upper_bound = ((uc_array==NULL) ? uc : uc_array[j]);
		 data[j].r = (data[j].r>upper_bound.r) ? fval.r : tval.r;
		 data[j].i = (data[j].i>upper_bound.i) ? fval.i : tval.i;
	      }
	   }
	}
}
/* -library_code_end */
