 /*
  * Khoros: $Id$
  */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

 /*
  * $Log$
  */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>> 
   >>>> 	Library Routine for karith2
   >>>> 
   >>>>  Private: 
   >>>> 
   >>>>   Static: 
   >>>>   Public: 
   >>>> 	lkarith2
   >>>> 
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */


#include "internals.h"

/* -library_includes */
/*
 * Operation structure declaration
 */
struct _operation
{
        char    *name;
	int     routine;
	int     complex;
        double  real_pad_value;
        double  imag_pad_value;
};

/*
 * Function declaration list
 */
static struct _operation operations[] =
{
    { "add",      KADD, TRUE, 0.0, 0.0 },
    { "sub",      KSUB, TRUE, 0.0, 0.0 },
    { "subfrom",  KSUBFROM, TRUE, 0.0, 0.0 },
    { "mult",     KMULT, TRUE, 1.0, 0.0 },
    { "div",      KDIV, TRUE, 1.0, 0.0 },
    { "divinto",  KDIVINTO, TRUE, 1.0, 0.0 },
    { "absdiff",  KABSDIFF, FALSE, 0.0, 0.0 },
    { "mod",	  KMODULO, FALSE, 1.0, 0.0 },
    { "minimum",  KMINIMUM, FALSE, 0.0, 0.0 },
    { "maximum",  KMAXIMUM, FALSE, 0.0, 0.0 },
    { "atan2",    KATAN2, FALSE, 1.0, 0.0 },
    { "hypot",    KHYPOT, FALSE, 0.0, 0.0 },
    { "ldexp",	  KLDEXP, FALSE, 0.0, 0.0 },
    { "pow",      KPOW, TRUE, 1.0, 0.0 },
};
/* -library_includes_end */


/****************************************************************
* 
*  Routine Name: lkarith2 - perform pointwise dual operand arithmetic on data object(s)
* 
*       Purpose: The lkarith2 library routine performs an arithmetic 
*		 operation on two source data objects.  The routine takes 
*		 two input data objects (or a single data object and 
*                a complex constant), a function name, and a destination 
*		 data object in which to store the result of the function. 
*
*		 The supported functions are:
*                .TS 
*                center tab(:) ;
*                l l .
*		 add     :  dest = src1 + (src2 or val)
*		 sub     :  dest = src1 - (src2 or val)
*		 subfrom :  dest = (src2 or val) - src1
*		 mult    :  dest = src1 * (src2 or val)
*		 div     :  dest = src1 / (src2 or val)
*		 divinto :  dest = (src2 or val) / src1 
*		 pow     :  dest = src1 raised to power of (src2 or val)
*		 absdiff :  dest = | src1 - (src2 or val) |
*		 mod     :  dest = remainder (src1/(src2 or val))
*		 atan2   :  dest = atan(src1,(src2 or val)) = atan(src1/(src2 or val))
*		 hypot   :  dest = hypot(src1,src2)=sqrt(src1*src1+src2*src2)
*		 minimum :  dest = lower value between src1 and (src2 or val)
*		 maximum :  dest = higher value between src1 and (src2 or val)
*		 ldexp   :  dest = ldexp(src1,(src2 or val)) = src1 * 2**(src2 or val)
*		 .TE
*
*		 Where val is defined by the real and imag parameters passed 
*		 into lkarith2.
*
*		 Therefore if lkarith2() were called with two input data
*		 objects:
*
*		 !    lkarith2(src1, src2, 0.0, 0.0, NULL, "add", dest);
*
*		 then the following would be performed:
*
*		 !    dest = src1 + src2
*
*		 If lkarith2 were called with a single data object and
*		 a constant:
*
*		 !    lkarith2(src1, NULL, 10.0, 0.0, NULL, "add", dest);
*
*		 then the following would be performed:
*
*		 !    dest = src1 + 10.0
*
*		 If the use_subpos parameter is TRUE,
*		 the subobject position attribute of src2 is applied as 
*		 an offset before the final destination object size is 
*		 determined, and before the arithmetic operation is 
*		 performed.  The subobject position of src1 is not applied.
*		 If the use_subpos parameter is TRUE, both objects are
*		 aligned at (0,0,0,0,0).
*
*		 If either source has mask data, the destination will 
*		 have a mask that is the logical and of the source masks.
*		 Masks are padded with 1.
*
*		 If a single source object is supplied, and it has map
*		 data, then the operation is performed on the map.  If
*		 both source objects are supplied with combinations of 
*		 map and value data, then the objects are operated on 
*		 as described in the kapu_upgrade_objects() man page.
*
*		 If the first source object has location or time data,
*		 they will be transferred to the destination object.  
*		 If the object is resized, uniform location attributes 
*	         will not be modified.  If the location data is curvilinear 
*		 or rectilinear, padded may occur using the src1 object's 
*		 location/time pad attributes.
*
*         Input: src1 - the input object to be processed
*		 src2 - the second input object (if NULL then uses the
*			 constants "real" and "imag")
*		 real - real value to be instead of the second data object
*		 imag - imaginary value to be instead of the second data object
*		 function - the function to be applied
*		 use_subpos - if TRUE, offset src2 by its subobject position
*        Output: dest  - the output object
*
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*  Restrictions: 
*    Written By: Donna Koechner & Mark Young
*          Date: Apr 08, 1995
*      Verified: 
*  Side Effects: 
* Modifications: 
****************************************************************/
/* -library_def */
int lkarith2 (
   kobject src1,
   kobject src2,
   double  real,
   double  imag,
   char    *function,
   int     use_subpos,
   kobject dest)
/* -library_def_end */

/* -library_code */
{
        klist   *objlist=NULL;
	char    *lib="kdatamanip", *rtn="lkarith2";
	int     routine, indx, num; 
        int     type1, type2;
	int	w1, h1, d1, t1, e1;
	int	woff=0, hoff=0, doff=0, toff=0, eoff=0;
        int     num_pts, num_rgns;
	char	*opt_region, *rgn_size, *region, *datatype, *padvalue;
	char	*interptype, *position;
        int     masktype, maptype, valuetype;
	kaddr   data1 = NULL, data2 = NULL;
	int	transfer_loc=FALSE, transfer_time=FALSE, grid;
	int	process_map=FALSE, process_value=FALSE, process_mask=FALSE; 
	kdcomplex  complex;
 
        /*----------------------------------------------------------------
	| Combine the real and imaginary values passed in to create a 
	| kdcomplex value.
        ----------------------------------------------------------------*/
	complex = kdccomp(real, imag);

/*------------------------------------------------------------------------
|  INITIAL ERROR CHECKING
------------------------------------------------------------------------*/
        /*----------------------------------------------------------------
        | Check if the src and dest objects passed into the lroutine are 
        | valid data objects. With these checks pushed into the lroutine
        | the driver routine need not necessarily check the status on the 
        | kpds_open_input_object and kpds_open_output_object calls, though 
	| it is a good idea to do it.  These checks are provided here to 
	| make the lroutine complete in its implementation so that it can 
	| be called by anyone wishing to do a karith2 supported operation 
	| on kobjects.  src2 is not checked because when it is not valid, 
	| we will use the real and imaginary values passed in for operation. 
        ----------------------------------------------------------------*/
        if (src1 == KOBJECT_INVALID)
        {
	   errno = KINVALID_PARAMETER;
           kerror(lib, rtn, "Source object 1 is not a valid data object.");
           return(FALSE);
        }
        if (dest == KOBJECT_INVALID)
        {
	   errno = KINVALID_PARAMETER;
           kerror(lib, rtn, "Destination object is not a valid data object.");
           return (FALSE);
        }

        /*----------------------------------------------------------------
        | Check the function and data_name arguments to see if the 
        | operation is supported by lkarith2. Since the driver routine 
        | is not aware of what operations karith2 supports and what data 
        | can be operated on by it these checks have to be done here.
        ----------------------------------------------------------------*/
	for (num = 0, indx = -1; num < knumber(operations); num++)
	{
	   if (kstrcmp(function, operations[num].name) == 0)
	   {
	      indx = num;
	      break;
	   }
	}
	if (indx == -1)
	{
	   kerror(lib, rtn, "Sorry, but the function '%s' %s", function, 
		  "is not in the list of callable operations.");
	   return(FALSE);
	}
	else if (operations[indx].routine == ERROR)
	{
	   kerror(lib, rtn, "Sorry, not unable to run the operation '%s' %s", 
		  function, "on data objects at this time.");
	   return(FALSE);
	}
	else
	   routine = (int) operations[indx].routine;

        /*----------------------------------------------------------------
        | Check if sources have location and/or time. If source 1 does, 
	| transfer it to the destination.  Set up destination here, and
        | copy the data later on (after setting up all of the other
        | attributes).
        ----------------------------------------------------------------*/
        transfer_loc = kapu_multi_input_loc(&grid, 2, src1, src2);
        if (transfer_loc)
	{
	   if (!kpds_query_location(dest)) KCALL(kpds_create_location(dest));
           KCALL(kpds_copy_location_attr(src1, dest));
	}
        transfer_time = kapu_multi_input_time(2, src1, src2);
        if (transfer_time)
	{
	   if (!kpds_query_time(dest))  KCALL(kpds_create_time(dest));
           KCALL(kpds_copy_time_attr(src1, dest));
	}

        /*----------------------------------------------------------------
        | Reference source objects 1 & 2. Referencing a source object 
        | ensures that none of the physical and presentation attributes 
        | associated with the object are modified when processing is done 
        | on it. For example if the input object has its position set to 
        | (64,64,1,1,1) it will have that same position after lkarith2
        | returns.  The kapu_upgrade_objects() function may change some
	| of the object attributes, so the objects should be referenced 
	| before they are passed into it.
	|
        | An error message is returned if the reference operation on 
        | either of the source objects fails. After creating the reference 
        | object we need to make sure that we close it whenever we return 
        | from lkarith2.  To make sure that the objects get closed before 
	| the function returns, karith2 adds the referenced object to 
	| list.  The klist_free() function is called before returning,
	| and it will free all objects and memory that has been added to
	| the the list.
	|
	| Throughout the lkarith2 code, you will see KCALL wrappers around 
	| the kpds functions.  The KCALL() function automatically prints 
	| an error message, calls klist_free(), and returns FALSE if the 
	| data services call fails.  This insures error checking, while
	| keeping the code clean.
        ----------------------------------------------------------------*/
	KCALL((src1 = kpds_reference_object(src1)) != KOBJECT_INVALID);
	objlist = klist_add(objlist,src1,"KOBJECT");

        if (src2)
	{
	   KCALL((src2 = kpds_reference_object(src2))!= KOBJECT_INVALID);
	   objlist = klist_add(objlist,src1,"KOBJECT");
	}

        /*----------------------------------------------------------------
        | Check that objects can work together polymorphically & set them up.
        ----------------------------------------------------------------*/
        if (!kapu_upgrade_objects(&process_map, &process_value, &process_mask,
				  &maptype, &valuetype, &masktype, 
                        	  TRUE, TRUE, TRUE, TRUE, 2, src1, src2))
        {
           (void)klist_free(objlist, (kfunc_void)lkcall_free);
           return(FALSE);
        }

        /*----------------------------------------------------------------
        | Source objects must have value and/or map data
        ----------------------------------------------------------------*/
        if (!process_map && !process_value)
        {
           kerror(lib,rtn,"No Map or Value data exist for source objects");
           (void)klist_free(objlist, (kfunc_void)lkcall_free);
           return (FALSE);
        }

        /*----------------------------------------------------------------
        | Check and create segments in destination object if they do not
        | exist.  Otherwise, if data does not exist in source objects, but
        | does exist in the destination, destroy it in the destination.
        ----------------------------------------------------------------*/
        if (process_map)
        {
           if (!kpds_query_map(dest))  KCALL(kpds_create_map(dest));
           KCALL(kpds_copy_attribute(src1, dest, KPDS_MAP_SIZE));
           KCALL(kpds_set_attribute(dest, KPDS_MAP_DATA_TYPE, maptype));
        }
        else
           if (kpds_query_map(dest))  KCALL(kpds_destroy_map(dest));
 
        if (process_mask)
        {
           if (!kpds_query_mask(dest))  KCALL(kpds_create_mask(dest));
           KCALL(kpds_set_attribute(dest, KPDS_MASK_DATA_TYPE, masktype));
        }
        else
           if (kpds_query_mask(dest))  KCALL(kpds_destroy_mask(dest));
 
        if (process_value)
        {
           if (!kpds_query_value(dest))  KCALL(kpds_create_value(dest));
           KCALL(kpds_copy_attribute(src1, dest, KPDS_VALUE_SIZE));
           KCALL(kpds_set_attribute(dest, KPDS_VALUE_DATA_TYPE, valuetype));
        }
        else
           if (kpds_query_value(dest))  KCALL(kpds_destroy_value(dest));
 
        /*----------------------------------------------------------------
	| Whenever possible, lkarith2 will operate directly on the map 
	| data - so no mapping through value data will occur.  
        ----------------------------------------------------------------*/
	if (process_map && !process_value)
	{
	   region       = KPDS_MAP_REGION;
	   opt_region   = KPDS_MAP_OPTIMAL_REGION_SIZE;
	   rgn_size	= KPDS_MAP_REGION_SIZE;
	   datatype     = KPDS_MAP_DATA_TYPE;
	   position     = KPDS_MAP_POSITION;
	   padvalue     = KPDS_MAP_PAD_VALUE;
	   interptype   = KPDS_MAP_INTERPOLATE;
	}
	else
	{
	   region     = KPDS_VALUE_REGION;
	   opt_region = KPDS_VALUE_OPTIMAL_REGION_SIZE;
	   rgn_size   = KPDS_VALUE_REGION_SIZE;
	   datatype   = KPDS_VALUE_DATA_TYPE;
	   position   = KPDS_VALUE_POSITION;
	   padvalue   = KPDS_VALUE_PAD_VALUE;
	   interptype = KPDS_VALUE_INTERPOLATE;
/* some day, make use_subobject_pos an input parameter to lkarith2 
	   if (src1 && src2)
	      use_subobject_pos = TRUE;
	   else
	      use_subobject_pos = FALSE;
*/
	}

        /*----------------------------------------------------------------
        | Set the presentation related attributes on the src1 and src2.  
        | Interpolation is set so that if the sizes of the two source 
	| objects mismatch the smaller object is padded with the specified 
	| pad value which is set by setting the pad value attribute.
        ----------------------------------------------------------------*/
	KCALL(kpds_set_attribute(src1, position, 0, 0, 0, 0, 0));
	KCALL(kpds_set_attribute(src1, interptype, KPAD));
	KCALL(kpds_set_attribute(src1, padvalue, 
	      operations[num].real_pad_value, operations[num].imag_pad_value));

	if (src2)
        {
           KCALL(kpds_set_attribute(src2, position, 0, 0, 0, 0, 0));
           KCALL(kpds_set_attribute(src2, interptype, KPAD));
           KCALL(kpds_set_attribute(src2, padvalue, 
	      operations[num].real_pad_value, operations[num].imag_pad_value));

	   if (use_subpos && src2 && process_value)
	   {
              KCALL(kpds_get_attribute(src2, KPDS_SUBOBJECT_POSITION, 
			&woff, &hoff, &doff, &toff, &eoff));
              KCALL(kpds_set_attribute(src2, KPDS_VALUE_OFFSET, 
			-woff, -hoff, -doff, -toff, -eoff));
	      if (process_mask)
	      {
	         /*-- create mask if none exists //SK  4-7-95 --*/
                 if (!kpds_query_mask(src2))  KCALL(kpds_create_mask(src2));
	         KCALL(kpds_set_attribute(src2, KPDS_MASK_OFFSET,
                        -woff, -hoff, -doff, -toff, -eoff));
	      }
	   }
        }

        /*----------------------------------------------------------------
	|  Set source data types to one of the following: KUBYTE, 
	|  KLONG, KULONG, KDOUBLE, or KDCOMPLEX.  Certain operations
	|  are only supported in double.  They are KPOW, KATAN2, KHYPOT,
	|  and KLDEXP.  If the operation selected is one of these 
	|  pass it in as an argument to the cast process routine.
        ----------------------------------------------------------------*/
	KCALL(kpds_get_attribute(src1, datatype, &type1));
	if (routine == KPOW || routine == KATAN2 || 
	    routine == KHYPOT || routine == KLDEXP)
	   type2 = KDOUBLE;
	else 
	   type2 = type1;

	type1 = kdatatype_cast_process(type1, type2,
                        (KUBYTE | KLONG | KULONG | KDOUBLE | KDCOMPLEX));

        /*----------------------------------------------------------------
        | Currently COMPLEX datatype does not support operations like
        | absdiff,hypotenuse.. A warning is printed and the type set
	| to double.  Set this data type in the destination object before
	| it is referenced.
        ----------------------------------------------------------------*/
        if (type1 == KDCOMPLEX && operations[indx].complex == FALSE)
	{
	   kinfo(KSTANDARD,"lkarith2: Input data is complex, %s %s '%s'",
		 "but operation only supports double.  Casting data to", 
		 "double for arithmetic operation", function);
           type1 = KDOUBLE;
           KCALL(kpds_set_attribute(dest, datatype, type1));
	}

        /*----------------------------------------------------------------
	| If a value instead of a second input object is supplied for
	| the unsigned byte case, decide on the lowest datatype that 
	| can be used.  Set the attributes of the objects to match this
	| type.  This is done because casting the float value supplied
	| by the user might cause unexpected results.  (Since the datatype
	| attribute has already been set on the destination object, data
	| will still be written to the output object in the correct type.
        ----------------------------------------------------------------*/
	if (!src2)
	{
	   if (type1 < KDOUBLE)
	   {
              /*----------------------------------------------------------
	      | First check if the value passed in has any significant 
	      | digits to the right of the decimal point, if it does, set 
	      | the data type to double.
              ----------------------------------------------------------*/
	      if (kfraction(real) != 0.0)
		 type1 = KDOUBLE;

              /*----------------------------------------------------------
	      | So, there is nothing to the right of the decimal point.  
	      | Set the processing datatype to long, unless it is unsigned 
	      | long, in which case, it needs to be set to double.
              ----------------------------------------------------------*/
	      else if (type1 == KULONG)
		 type1 = KDOUBLE;
	      else
	         type1 = KLONG;
	   }
	}

        KCALL(kpds_set_attribute(src1, datatype, type1));
	if (src2)
	   KCALL(kpds_set_attribute(src2, datatype, type1));

        /*----------------------------------------------------------------
	| Create a reference on the destination object. We have
        | changed all attributes that we wanted to be reflected 
        | in the physical storage of the kobject. In order
        | to preserve the other attributes a reference copy
        | is made on the destination object. Also set the position
        | on the output destination reference object to (0,0,0,0,0) and its
        | datatype to the type set for the two src object references.
        | It is recommended that the destination object in any kroutine 
        | be referenced as soon as all changes in attributes that have
        | been explicitly requested by the parameters of the lkroutine
        | or that are needed for the proper functioning of the lkroutine
        | have been made.  
        ----------------------------------------------------------------*/
	KCALL((dest = kpds_reference_object(dest)) != KOBJECT_INVALID);
	objlist = klist_add(objlist,dest,"KOBJECT");

        KCALL(kpds_set_attribute(dest, datatype, type1));
        KCALL(kpds_set_attribute(dest, position, 0, 0, 0, 0, 0));

        /*----------------------------------------------------------------
        | Get the dimension and number of optimal regions from src1.
        | Since the sizes and index order on all 3 objects are 
        | the same we can obtain this information from either
        | one of src1,src2(if one exists) or dest.
        ----------------------------------------------------------------*/

        KCALL(kpds_get_attribute(src1, opt_region, 
			&w1,&h1,&d1,&t1,&e1, &num_rgns));
	num_pts = w1*h1*d1*t1*e1;
	KCALL(kpds_set_attribute(src1, rgn_size, w1,h1,d1,t1,e1));
	KCALL(kpds_set_attribute(dest, rgn_size, w1,h1,d1,t1,e1));
	if (src2)
	   KCALL(kpds_set_attribute(src2, rgn_size, w1,h1,d1,t1,e1));

	for (num = 0; num < num_rgns; num++)
	{
	   if ((data1 = kpds_get_data(src1, region, data1)) == NULL)
	   {
	      kerror(lib, rtn, "Failed to get %s from src1 object", region);
              (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
	   if (num == 0)
              objlist = klist_add(objlist,data1,"KMALLOC");

	   if (src2)
	   {
	      if ((data2 = kpds_get_data(src2, region, data2)) == NULL)
	      {
	         kerror(lib, rtn, "Failed to get %s from src1 object", region);
                 (void)klist_free(objlist, (kfunc_void)lkcall_free);
	         return(FALSE);
	      }
	      if (num == 0)
                 objlist = klist_add(objlist,data2,"KMALLOC");
	   }

	   if (type1 == KUBYTE)
	   {
	      kdata_arith2_ubyte(routine, num_pts, (unsigned char *) data1,
			        (unsigned char *) data2, NULL, NULL);
	   }
	   else if (type1 == KULONG)
	   {
	      kdata_arith2_ulong(routine, num_pts, (unsigned long *) data1,
			        (unsigned long *) data2, NULL, NULL);
	   }
	   else if (type1 == KLONG)
	   {
	      kdata_arith2_long(routine, num_pts, (long *) data1,
			       (long *) data2, (long) real, NULL, NULL);
	   }
	   else if (type1 == KDOUBLE)
	   {
	      kdata_arith2_double(routine, num_pts, (double *) data1,
			         (double *) data2, real, NULL, NULL);
	   }
	   else if (type1 == KDCOMPLEX)
	   {
	      kdata_arith2_dcomplex(routine, num_pts, (kdcomplex *) data1,
			           (kdcomplex *) data2, complex, NULL, NULL);
	   }

	   if (!kpds_put_data(dest, region, data1))
	   {
	      kerror(lib, rtn, "Failed to put %s in output object", region);
              (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
	}

        /*----------------------------------------------------------------
	| If either of the source objects had masks, AND them and put the
	| result into the destination mask.
        ----------------------------------------------------------------*/
	if (process_mask)
	{
           if (!kapu_mask_ops(dest, KDMANIP_MASK_AND, 2, src1, src2))
	   {
              (void)klist_free(objlist, (kfunc_void)lkcall_free);
	      return(FALSE);
	   }
	}

        /*----------------------------------------------------------------
	| If source 1 has location or time data, transfer it to destination
        ----------------------------------------------------------------*/
	if (transfer_loc)
           KCALL(kpds_copy_location_data(src1, dest, TRUE));
 
        if (transfer_time)
           KCALL(kpds_copy_time_data(src1, dest, TRUE));


        (void)klist_free(objlist, (kfunc_void)lkcall_free);
        return(TRUE);
}
/* -library_code_end */
