/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Khoros Utilities
   >>>>
   >>>>  Private:
   >>>>		    ktoolbox_binpath()
   >>>>		    ktoolbox_file_initialize()
   >>>>		    ktoolbox_file_list2array()
   >>>>		    ktoolbox_query_tbname()
   >>>>		    _kalias_initialize()
   >>>>             _kalias_replace()
   >>>>             _cleanup_string()
   >>>>             _expand_variable()
   >>>>             _expand_tilda()
   >>>>		    ktmpnam()
   >>>>   Public:
   >>>>             kalias_list2array()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


typedef struct _khoros_aliases
{
        char    *alias;
        char    *fullpath;

        struct _khoros_aliases *next;
} Khoros_aliases;

static  Khoros_aliases *khoros_alias_list = NULL;


#define DEFAULT_TOOLBOX_FILE "$BOOTSTRAP/repos/Toolboxes"

#ifndef SITE_INCLUDE_FILE
#define SITE_INCLUDE_FILE NULL
#endif

/*-----------------------------------------------------------
|
|  Routine Name: ktoolbox_binpath - 
|
|       Purpose: The toolbox bin path.
|
|         Input: tbname - the toolbox name
|       Returns: return the toolbox bin path or NULL
|
|    Written By: Mark Young & John Salas
|          Date: Jul 17, 1994
| Modifications:
|
------------------------------------------------------------*/

static char *ktoolbox_binpath(
   char *tbname)
{
	kfile *file;
	int   count, status;
        char  *karch, *kopsys, *kwidset, temp[KLENGTH], path[KLENGTH],
	      *binpath = NULL, **scan;
 

        if ((karch = kgetenv("KARCH")) != NULL)
        {
           kopsys  = kgetenv("KOPSYS");
           kwidset = kgetenv("KWIDSET");
           if ((kwidset == NULL && kopsys == NULL) ||
               (kwidset != NULL && kopsys == NULL))
           {
              ksprintf(temp, "Site.%s", karch);
           }
           else if (kopsys != NULL && kwidset == NULL)
              ksprintf(temp, "Site.%s-%s", karch, kopsys);
           else if (kopsys != NULL && kwidset != NULL)
              ksprintf(temp, "Site.%s-%s-%s", karch, kopsys, kwidset);
        }
	else if (SITE_INCLUDE_FILE != NULL)
	{
	   kstrcpy(temp, SITE_INCLUDE_FILE);
	   kstring_replace(temp, "<", NULL, temp);
	   kstring_replace(temp, ">", NULL, temp);
	}
	else
	   return(NULL);

	ksprintf(path, "$%s/config/%s", tbname, temp);
	if ((file = kfinput(path)) != NULL)
	{
	   scan = kparse_file_scan_delimit(file, "#.*define.*BinDir", "$",
			KIGNORE_CASE, "[ \t]", NULL, NULL, &count, &status);
	   if (count > 1) binpath = kstrdup(scan[1]);
	   karray_free(scan, count, NULL);
	   kfclose(file);
	}
	else
	{
	   ksprintf(path, "$BOOTSTRAP/config/%s", temp);
	   if ((file = kfinput(path)) != NULL)
	   {
	      scan = kparse_file_scan_delimit(file, "#.*define.*BinDir", "$",
			KIGNORE_CASE, "[ \t]", NULL, NULL, &count, &status);
	      if (count > 1) binpath = kstrdup(scan[1]);
	      karray_free(scan, count, NULL);
	      kfclose(file);
	   }
	}
	return(binpath);
}

/*-----------------------------------------------------------
|
|  Routine Name: ktoolbox_file_initialize - initialize the toolbox routines
|
|       Purpose: This subroutine initializes the global list of
|		 toolboxes used to represent the different toolboxes
|		 currently to be used by the Khoros programs.
|
|         Input:
|        Output:
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young  
|          Date: Sep 23, 1993
| Modifications:
|
------------------------------------------------------------*/

int ktoolbox_file_initialize(void)
{
	static int initialized = FALSE;

	int    num;
	kfile  *file;
	char   *filenames, *tmp, *binpath, filename[KLENGTH],
		temp[KLENGTH], tbname[KLENGTH], tbpath[KLENGTH],
		name[KLENGTH], def[KLENGTH], options[KLENGTH];

	if (initialized)
	   return(TRUE);
	else
	   initialized = TRUE;

	if ((filenames = (char *) kgetenv("KHOROS_TOOLBOX")) == NULL)
	{
	   kinfo(KSYSLIB, "ktoolbox_file_initialize", NULL, "Environmental \
variable KHOROS_TOOLBOX not set.  This is a required environment \
variable.  Setting it to %s\n", DEFAULT_TOOLBOX_FILE);
	   filenames = kstrdup(DEFAULT_TOOLBOX_FILE);
	}

	while (filenames != NULL)
	{
	   if ((tmp = kstrchr(filenames, ':')) != NULL)
	   {
	      kstring_ncopy(filenames, tmp-filenames, filename);
	      tmp++;
	   }
	   else
	      kstring_copy(filenames, filename);

	   /* KHOROS_TOOLBOX variable cannot be opened */
	   if ((file = kfinput(filename)) == NULL)
	   {
	      kinfo(KSYSLIB, "ktoolbox_file_initialize: Unable to open \
toolbox file '%s' for reading.", filename);
	      filenames = tmp;
	      continue;
	   }

	   while (kfgets(temp, KLENGTH, file) != NULL)
	   {
	      if ((num = ksscanf(temp,"%[^#:]:%[^ \t\n:]:%[^\n:]",
				tbname, tbpath, options)) != 3 &&
		 ((num = ksscanf(temp, "%[^#:]:%[^ \t\n:]", tbname,
				 tbpath)) != 2))
	      {
		 continue;
	      }
	      kstring_cleanup(tbname, tbname);
	      kstring_cleanup(tbpath, tbpath);

	      if (kgetenv(tbname) == NULL)
	      {
	         ksprintf(temp, "%s=%s", tbname, tbpath);
	         kputenv(temp);
	      }

	      ksprintf(temp, "%sBIN", tbname);
	      if (kgetenv(temp) == NULL)
	      {
		 /*
		  *  Check to see if this toolbox might be supprting multiple
		  *  toolbox bins.  We know this if there is an options part
		  *  of the toolbox definition (from the khoros toolbox file)
		  *  and "machdefs=TRUE" is present.
		  */
		 binpath = NULL;
	         if (num == 3 && ksscanf(options,"%[^=]=%[^:]", name, def) == 2)
	         {
		    kstring_cleanup(def, def);
		    kstring_lower(def, def);
		    if ((kstrcmp(def, "yes") == 0  ||
		         kstrcmp(def, "true") == 0 || 
		         kstrcmp(def, "on") == 0 || 
		         kstrcmp(def, "1") == 0) &&
		         (binpath = ktoolbox_binpath(tbname)) != NULL)
		    {
	               ksprintf(def, "%s=%s", temp, binpath);
		    }
	         }

	         if (binpath == NULL)
	            ksprintf(def, "%s=%s/bin", temp, tbpath);

		 kinfo(KDEBUG, "defining binpath: '%s'\n", def);
	         kputenv(def);
	      }
	   }
	   kfclose(file);
	   filenames = tmp;
	}
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: ktoolbox_file_list2array - create a list of toolbox names
|
|       Purpose: This routine takes the global toolbox list, and
|		 returns an array of toolbox names to the calling
|		 routine.  It also takes a count parameter that
|		 is set to the number of entries in the toolbox array.
|
|         Input: none
|
|        Output: num - The number of toolbox names in the array; 0 on error.
|       Returns: The list of toolbox names if successful, or
|			  NULL on error.
|
|    Written By: Mark Young
|          Date: Sep 23, 1993
| Modifications:
|
------------------------------------------------------------*/

char **ktoolbox_file_list2array(
   int *num)
{
	kfile   *file;
	int     size = 0;
	char    *filenames, filename[KLENGTH], tbname[KLENGTH], tbpath[KLENGTH],
		temp[KLENGTH], *tmp, *variable, **toolboxes = NULL;


	if ((filenames = (char *) kgetenv("KHOROS_TOOLBOX")) == NULL)
	{
	   kinfo(KSYSLIB, "ktoolbox_file_list2array", NULL, "Environmental \
variable KHOROS_TOOLBOX not set.  This is a required environment \
variable.  Setting it to %s\n", DEFAULT_TOOLBOX_FILE);
	   filenames = kstrdup(DEFAULT_TOOLBOX_FILE);
	}

	while (filenames != NULL)
	{
	   if ((tmp = kstrchr(filenames, ':')) != NULL)
	   {
	      kstring_ncopy(filenames, tmp-filenames, filename);
	      tmp++;
	   }
	   else
	      kstring_copy(filenames, filename);

	   /* KHOROS_TOOLBOX variable cannot be opened */
	   if ((file = kfinput(filename)) == NULL)
	   {
	      kinfo(KSYSLIB, "ktoolbox_file_list2array: Unable to open \
toolbox file '%s' for reading.", filename);
	      filenames = tmp;
	      continue;
	   }

	   while (kfgets(temp, KLENGTH, file) != NULL)
	   {
	      if (ksscanf(temp,"%[^#:]:%[^ \t\n:]", tbname, tbpath) != 2)
		 continue;

	      kstring_cleanup(tbname, tbname);
	      variable = kstrdup(tbname);
	      toolboxes = karray_add(toolboxes, (kaddr) variable, size++);
	   }
	   kfclose(file);
	   filenames = tmp;
	}
	if (num) *num = size;
	return(toolboxes);
}

/*-----------------------------------------------------------
|
|  Routine Name: ktoolbox_query_tbname - check to see if the toolbox exists
|				in the toolbox list.
|
|       Purpose: This routine checks to see if a toolbox exists in the
|		 toolbox list.
|         Input: tbname - the toolbox name to check for
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Mark Young
|          Date: Jul 29, 1994
| Modifications:
|
------------------------------------------------------------*/

int ktoolbox_query_tbname(
   char *tbname)
{
	char **toolboxes;
	int  i, num, found = FALSE;


	if ((toolboxes = ktoolbox_file_list2array(&num)) != NULL)
	{
	   for (i = 0; i < num; i++)
	   {
	      if (kstrcasecmp(tbname, toolboxes[i]) == 0)
	      {
		 found = TRUE;
	         break;
	      }
	   }
	   karray_free(toolboxes, num, NULL);
	}
	return(found);
}


/*-----------------------------------------------------------
|
|  Routine Name: file_initialize - initialize a alias file
|
|       Purpose: This routine parses a alias file, and fills
|		 out a alias entry with the information. This
|		 is the static routine that does the work for
|		 _kalias_initialize().
|
|         Input: filename - alias filenames to parse
|        Output: 
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Danielle Argiro
|          Date: Jul 03, 1992 14:05
| Modifications:
|
------------------------------------------------------------*/

static int file_initialize(
   char *filename)
{
	kfile *file; 
	char  buffer[KLENGTH];
	int   i, indx, done, linelength = KLENGTH;
	char  alias[KLENGTH], new_key[KLENGTH], comp_key[KLENGTH];
	char  path[KLENGTH], remainder[KLENGTH], fullpath[KLENGTH];
	Khoros_aliases *keywd_ptr, *new_keywd, *last_ptr;



	/* KHOROS_ALIASES variable cannot be opened */
	if (kaccess(filename, R_OK) == -1)
	{
	   kinfo(KSYSLIB,"alias_initialize: No Alias file provided in '%s'", 
		 filename);
	   return(FALSE);
	}
 	else if ((file = kfinput(filename)) == NULL)
	{
	   kinfo(KSYSLIB,"alias_initialize: Alias file provided in '%s', but \
unable to open the file", filename);
	   return(FALSE);
	}

	if (kfeof(file))
	{
	   kfclose(file);
	   return(FALSE);
	}

	while (kfgets(buffer, linelength, file) != NULL)
	{
	   /* skip over leading white space */
	   i = 0;  while ((buffer[i] == ' ') || (buffer[i]=='\t'))  i++;

	   /* skip over blank lines */
	   if ((buffer[0] != '\n')  && (buffer[0] != '#'))
	   {
	      /* get the alias */
	      ksscanf(buffer, "%s", alias);

	      /* now skip over the alias */
	      kstring_copy(&buffer[i + kstrlen(alias)], remainder);

	      /* skip over white space following alias */
	      indx = 0;  
	      while ((remainder[indx] == ' ') || (remainder[indx]=='\t'))
	         indx++;

	      /* what remains should be the path */
	      kstring_copy(&remainder[indx], path);

	      /* take off <cr> at end of path */
	      path[kstrlen(path)-1] = '\0';
		
	      /* path specified for alias cannot be expanded */
	      if (!kfullpath(path, NULL, fullpath))
	      {
		 kinfo(KSYSLIB,"kinitialize_aliases: Error in aliases \
file %s", filename);
		 kfclose(file);
		 return(FALSE);
	      }
	      new_keywd = (Khoros_aliases *)kcalloc(1,sizeof(Khoros_aliases));
	      new_keywd->alias = kstrdup(alias);
	      new_keywd->fullpath = kstrdup(fullpath);

	      if (khoros_alias_list == NULL)
	      {
		 khoros_alias_list = new_keywd;
		 khoros_alias_list->next = NULL;
	      }
	      else
	      {
		 last_ptr  = NULL;
		 keywd_ptr = khoros_alias_list;
		 done = FALSE;
		 while ((keywd_ptr != NULL) && (!done))
		 {
		    if (keywd_ptr->alias[0] == ':')
		       ksprintf(comp_key, "%s", &keywd_ptr->alias[1]);
		    else
		       ksprintf(comp_key, "%s", keywd_ptr->alias);

		    if (new_keywd->alias[0] == ':')
		       ksprintf(new_key, "%s", &new_keywd->alias[1]);
		    else
		       ksprintf(new_key, "%s", new_keywd->alias);
		      
		    if (kstrcmp(new_key, comp_key) > 0)
		       done = TRUE;
		    else
		    {
		       last_ptr = keywd_ptr;
		       keywd_ptr = keywd_ptr->next;
		    }
		 }

		 if (last_ptr != NULL)
		 {
		    new_keywd->next = last_ptr->next;
		    last_ptr->next = new_keywd;
		 }
		 else
		 {
		    new_keywd->next = khoros_alias_list;
		    khoros_alias_list = new_keywd;
		 }
	      }
	   }
	}
	kfclose(file);
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kalias_initialize - initialize the alias database
|
|       Purpose: This routine initializes the global alias database
|                that allows aliases to represent paths to input
|                data files.
|
|         Input: none
|
|        Output: none
|                RETURNS: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Danielle Argiro
|          Date: Jul 17, 1993
| Modifications:
|
------------------------------------------------------------*/

int _kalias_initialize(void)
{
	int  i, num_toolboxes;
	char **toolboxes, *temp, path[KLENGTH];


	if (khoros_alias_list != NULL)
	   return(TRUE);

	if ((temp = kgetenv("KHOROS_ALIASES")) != NULL)
	{
	   if (!file_initialize(temp))
	   {
	       errno = KNOENVIRONMENT_VAR;
	       kerror("kutils", "_kalias_initialize",
		      "Invalid alias file found for the KHOROS_ALIASES \
environment '%s'", temp);
	   }
	}

	if ((toolboxes = ktoolbox_file_list2array(&num_toolboxes)) != NULL)
	{
	   for (i = 0; i < num_toolboxes; i++)
	   {
	      ksprintf(path, "$%s/repos/Aliases", toolboxes[i]);
	      file_initialize(path);
	   }
	   karray_free(toolboxes, num_toolboxes, NULL);
	}
	else return(FALSE);

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kalias_replace - convert a alias to its real path
|
|       Purpose: This routine replaces the alias part of a path
|		 with its corresponding real path value.
|
|         Input: alias - the alias string to convert.
|
|        Output: none
|                RETURNS: a pointer to the filename on success, or the
|                         alias on fail
|
|    Written By: Danielle Argiro & Mark Young
|          Date: Jul 03, 1992 14:07
| Modifications: Converted from vreplace_alias in Khoros 1.0 (SJ)
|
------------------------------------------------------------*/

char *_kalias_replace(
   char *alias)
{
	Khoros_aliases *keywd_ptr;

	/*
	*  Initialize the khoros aliases
	*/
	_kalias_initialize();
	if (kstrchr(alias,':') == NULL)
	return(NULL);


	keywd_ptr = khoros_alias_list;
	while (keywd_ptr != NULL)
	{
	   if (kstrcmp(alias,keywd_ptr->alias) == 0)
	      return(keywd_ptr->fullpath);

	   keywd_ptr = keywd_ptr->next;
	}
	return(alias);
}


/*-----------------------------------------------------------
|
|  Routine Name: _cleanup_string - internal clean up string routine
|
|       Purpose: This function is used to strip the white space
|		 off the beginning and end of a string using
|		 isspace() to detect white space.
|		 Note: this routine changes the input string directly.
|
|         Input: string - the string to clean up
|
|        Output: none
|                RETURNS: the cleaned up string, or NULL on error
|
|    Written By: Mark Young
|          Date: Jul 03, 1992 14:06
| Modifications: Converted from _cleanup_string in Khoros 1.0 (SJ)
|
------------------------------------------------------------*/
char *_cleanup_string(
   register char *string)
{
	register char *temp;


	/*
	 *  Delete any white space from the head and tail of the
	 *  string.
	 */
	while (isspace(*string))
	   string++;

	if (*string == '\0')
	   return(NULL);

	temp = string + kstrlen(string);
	while (isspace(*--temp))
		;
	*(temp+1) = '\0';

	return(string);
}


/*-----------------------------------------------------------
|
|  Routine Name: _expand_variable - expand environment variables.
|
|       Purpose: This function is used to expand an environment
|		 variable in the supplied string.  The variable
|		 is the environment variable that we are going
|		 to expand in the string.  The variable may contain
|		 a $VARIABLE or just VARIABLE inside the string
|		 or the variable itself.  The other possibility
|		 the string.  This can be done by passing in
|		 NULL for the variable name, in which case any
|		 $VARAIBLE encountered within the string will
|		 be expanded.  (note:  only variables that begin
|		 with a '$' will be expanded since the variable
|		 was not explicitly given).
|
|         Input: string   - the string to be expanded
|		 variable - the variable to check for, NULL means
|			    convert all strings
|
|        Output: none
|                RETURNS: returns the expanded string, or NULL on error.
|
|    Written By: Mark Young
|          Date: Jul 03, 1992 14:06
| Modifications: converted from _expand_variable in Khoros 1.0
|
------------------------------------------------------------*/
char *_expand_variable(
   char *string,
   char *variable)
{
	int i, j, k, c, flength, vlength, length;
	char temp[KLENGTH], *var, *name;

	flength = kstrlen(string);
	vlength  = kstrlen(variable);

	if (flength < vlength)
	   return(string);

	i = k = 0;
	name = variable;
	while (i < flength)
	{
	   j = 0;
	   if (variable == NULL)
	   {
	      if (string[i] == '$')
	      {
		 i++; j = 0;
		 do
		 {
		    temp[j++] = string[i++];
		    c = (int) string[i];
		 } while (c != '$' && c != '\0' && c != '/' && i < flength);
		 name = temp; name[j] = '\0';
		 vlength = kstrlen(name);
	      }
	      else
	         vlength = j+1;  /* keeps us from doing an expand */
	   }
	   else
	   {
	      if (string[i] == '$')
	         i++;

	      while (variable[j] == string[i] && j < vlength)
	      {
		 i++; j++;
	      }
	   }

	   if (j >= vlength && (string[i] == '/' || string[i] == '\0'))
	   {
	      var = kgetenv(name);
	      if (var == NULL)
	      {
		 char upper[KLENGTH];

		 ktoolbox_file_initialize();
		 var = kgetenv(kstring_upper(name, upper));
	      }

	      if (var == NULL && variable != NULL)
	      {
	         errno = KNOENVIRONMENT_VAR;
		 kerror("kutils", "kfullpath", "Environment variable '%s' is \
undefined.", name);
		 return(NULL);
	      }
	      else if (var == NULL)
	      {
		 i++;
	      }
	      else
	      {
		 /*
		  *  Need to combine the variable with the beginning and
		  *  remainder of the string.  But we want to combine the
		  *  string so that we don't have '//' in the path.
		  */
		 if (k == 0)
		    temp[0] = '\0';
		 else
		 {
		    kstrncpy(temp, string, k);
		    if (string[k-1] == '/' && var[0] == '/')
		       temp[k-1] = '\0';
		    else
		       temp[k] = '\0';
		 }
		 kstrcat(temp, var);
		 if ((length = kstrlen(temp)) > 0)
		 {
		    if (string[i] == '/' && temp[length-1] == '/')
		       temp[length-1] = '\0';
		 }
		 kstrcat(temp, &string[i]);
		 kstrcpy(string, temp);
		 flength = kstrlen(string);
		 i = 0;
	      }
	   }
	   else
	      i++;

	   k = i;
	}
	return(string);
}

/*-----------------------------------------------------------
|
|  Routine Name: _expand_tilda - expand a tilda in a path
|
|       Purpose: This function is used to expand a username.
|		 If the string has a ~username or ~/ or ~ this
|		 routine will expand the string to the full
|		 path of that string
|
|         Input: string - the string to be expanded
|
|        Output: none
|                RETURNS: returns the expanded string, or NULL on error
|
|    Written By: Mark Young
|          Date: Jul 03, 1992 14:06
| Modifications: Converted from _expand_tilda in Khoros 1.0 (SJ)
|
------------------------------------------------------------*/
char *_expand_tilda(
   char *string)
{
	int    size;
	struct passwd *passwd_ent;
	char   user[KLENGTH], *tmp;


	if (string[0] == '~')
	{
	   if (kstrncmp(string, "~/", 2) == 0 || kstrcmp(string, "~") == 0)
	   {
	      tmp = &string[1];
	      passwd_ent = getpwuid(getuid());
	   }
	   else 
	   {
	      if ((tmp = kstrchr(string, '/')) != NULL)
	      {
		 size = tmp - &string[1];
		 kstrncpy(user, &string[1], size);
		 user[size] = '\0';
	      }
	      else
	         kstrcpy(user, &string[1]);

	      passwd_ent = getpwnam(user);
	   }

	   if (passwd_ent == NULL)
	   {
	      errno = KINVALID_USER;
	      kerror("kutils", "kfullpath", "Not a valid user (%s)", string);
	      return(NULL);
	   }

	   /*
	    *   Need to copy the password entry back into the string.  If
	    *   there was remainder string then we just copy the password
	    *   directory directly into the string array.  Otherwise we
	    *   copy it and the remainder into the user array and then back
	    *   into the string array.
	    */
	   if (tmp == NULL)
	      kstrcpy(string, passwd_ent->pw_dir);
	   else
	   {
	      (void) ksprintf(user, "%s%s", passwd_ent->pw_dir, tmp);
	      kstrcpy(string, user);
	   }
	}
	return(string);
}

/*-----------------------------------------------------------
|
|  Routine Name: ktmpnam - create a temporary name
|
|       Purpose: This module is a private routine that is used
|		 by the transport library to create a template
|		 filename which is used to help identify the
|		 the application routine.  It uses this template
|		 to create a unique pathname.  If no template
|		 is passed to the routine, TEMPLATE is used.
|
|         Input: template - template of the file to be created.
|        Output:
|       Returns: returns a string containing a unique temporary filename.
|
|    Written By: Mark Young
|          Date: Jul 03, 1992
| Modifications:
|
------------------------------------------------------------*/
char  *ktmpnam(
   char *template)
{
	char xvname[KLENGTH], *temp, *filename, *directory;

#if ! KTEMPNAM_DEF
	int i, num_dirs = 3;
	static char *alternate_dirs[] = { ".", "/usr/tmp", "/tmp" };
	char path[KLENGTH], *dir;
#endif

	if (template == NULL)
	   kstrcpy(xvname, TEMPLATE);
	else
	   kstrcpy(xvname, template);

	/*
	 *  Want to seperate the directory (if any) from the actual
	 *  template filename.
	 */
	if ((temp = kstrrchr(xvname,'/')) != NULL)
	{
	   if (xvname != temp)
	      directory = xvname;
	   else
	      directory = kstrdup("/");

	   *temp = '\0';
	   filename = ++temp;
	}
	else
	{
	   filename = xvname;
	   directory = ".";
	}

#if ! KTEMPNAM_DEF
	/*
	 *  Check to see if the user has overridden the directory with
	 *  TMPDIR.  If so that is the directory we wish to use.
	 */
	if ((dir = kgetenv("TMPDIR")) != NULL)
	   directory = dir;

	/* file named vroutine.xvXXXXXX */
	kstrcat(filename, ".xvXXXXXX");

	/*
	 *  Check to see if we can create the file.
	 */
	if (kaccess(directory, W_OK) == -1)
	{
	   for (i = 0; i < num_dirs; i++)
	   {
	      if (kaccess(alternate_dirs[i], W_OK) == 0)
	      {
	         ksprintf(path,"%s/%s", alternate_dirs[i], filename);
		 break;
	      }
	   } 

	   if (i == num_dirs)
	   {
	      kerror("kutils", "_ktemname", "Cannot create temporary file %s.",
		     template);
	      return(NULL);
	   }
	}
        else
        {
	   if (kstrcmp(directory,"/") != 0)
	      ksprintf(path,"%s/%s", directory, filename);
	   else
	      ksprintf(path,"/%s", filename);
	}
	tmpnam(path);
	return(kstrdup(path));
#else
        temp = (char *) tempnam(directory, filename);
        if (temp == NULL)
        {
           kerror("kutils", "ktmpnam", "Cannot create temporary file %s.",
		  template);
           return(NULL);
        }
        return(temp);
#endif
}


/************************************************************
*
*  Routine Name: kalias_list2array - returns a string array of aliases
*
*       Purpose: This routine converts the global alias list for files
*		 specified by each toolbox's $TOOLBOX/repos/Alias file,
*		 and converts the alias names to an array of alphabetically
*		 sorted strings.  This can be used for browsers to
*		 map aliases to real paths on disk.
*        Output: num_entries - If this paramater is a valid address, it will
*			       hold the number of aliases in the array.
*			       If the parameter is passed in as NULL, this
*			       routine will ignore it.
*       Returns: A malloc'ed string array of aliases upon success conversion
*		 of the alias list, otherwise NULL is returned.  Note
*		 you should use kfree to free the array, NOT the 
*		 karray_free, since the array contents are shared.
*    Written By: Mark Young
*          Date: Jan 19, 1993 12:15
*************************************************************/

char **kalias_list2array(
   int *num_entries)
{
	int num = 0;
	char **array = NULL;
	Khoros_aliases *list;


	/*
	 *  Initialize the khoros alias list
	 */
	_kalias_initialize();
 	list = khoros_alias_list;

	/*
	 *  Loop thru the alias list
	 */
	while (list != NULL)
	{
	   array = karray_add(array, list->alias, num); num++;
	   list = list->next;
	}
        array = karray_sort(array, num, FALSE);

	/*
	 *  Set the number of entries
	 */
	if (num_entries) *num_entries = num;
	return(array);
}
