/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            string routines
   >>>>
   >>>>  Private:
   >>>>             none
   >>>>   Public:
   >>>>             kstrstr()
   >>>>             kstrpbrk()
   >>>>             kstrtok()
   >>>>             kstrdup()
   >>>>             kstrspn()
   >>>>             kstrcspn()
   >>>>             kstrcasecmp()
   >>>>             kstrncasecmp()
   >>>>             kstrlen()
   >>>>             kstrcmp()
   >>>>             kstrncmp()
   >>>>             kstrchr()
   >>>>             kstrrchr()
   >>>>             kstrcpy()
   >>>>             kstrncpy()
   >>>>             kstrcat()
   >>>>             kstrncat()
   >>>>
   >>>>             kstring_lower()
   >>>>             kstring_upper()
   >>>>             kstring_capitalize()
   >>>>             kstring_replace()
   >>>>             kchar_replace()
   >>>>             kstring_cleanup()
   >>>>             kstring_format()
   >>>>             kstring_cat()
   >>>>             kstring_ncat()
   >>>>             kstring_3cat()
   >>>>             kstring_subcmp()
   >>>>             kstring_copy()
   >>>>             kstring_ncopy()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


#define NEEDS_STRDUP
#define NEEDS_STRCASECMP
#define NEEDS_STRNCASECMP

#define nullstring(str)	((str) ? str : "")

/************************************************************
*
*  Routine Name: kstrstr - find a substring within a string
*       Purpose: This routine is a replacement for the system
*		 library routine strstr.  This routine scans the
*		 input string for the first occurance
*		 of a substring specified by sub_str.
*         Input: istr - the string to search
*		 sub_str   - the sub string to look for.
*       Returns: If the sub_str is found in istr, a pointer indicating the
*		 address of the first character of the substring within the
*		 istr string is returned.  If the substring is not
*		 a part of the input string, a NULL is returned.
*
*		 This routine also protects against NULL on both the input and
*		 substring variables.  If either is NULL, the value of NULL is
*		 returned. 
*    Written By: Hideo Noro & Mark Young & Tom Sauer
*          Date: Jul 03, 1992 14:05
* Modifications: Converted from vstrstr in Khoros 1.0 (SJ)
*************************************************************/

char *kstrstr(
   char *istr,
   char *sub_str)
{
#ifdef NEEDS_STRSTR
	char *string;
	int  len;
#endif

	if (istr == NULL || sub_str == NULL)
	{
	   if (istr == NULL)
	      kinfo(KSYSLIB,"kstrstr: NULL input string encountered" );

	   if (sub_str == NULL)
	      kinfo(KSYSLIB,"kstrstr: NULL sub-string encountered" );

	   return(NULL);
	}

	/*
	 *  March thru the string using kstrncmp() to find the sub-string
	 */
#ifdef NEEDS_STRSTR

	len = kstrlen(sub_str);
	string = istr;
	while (*string != '\0')
	{
	   if (kstrncmp(string, sub_str, len) == 0)
	      return(string);
	   else
	      string++;
	}
	return(NULL);
#else
	return(strstr(istr, sub_str));
#endif
}

/************************************************************
*
*  Routine Name: kstrpbrk - find the first occurance of a character in a string
*			    out of a specified list of chararacters
*       Purpose: This routine is a replacement for the system
*		 library routine kstrpbrk.  This routine searches
*		 the input string, istr, for the first occurance of any
*		 character in a list of characters, specified by sub_str.
*         Input: istr    - the string to search
*		 sub_str - the string holding the characters to look for.
*       Returns: If one of the characters in sub_str is found in istr, a
*		 pointer indicating the address of the character it found
*		 that was a member of the list specified by sub_str.
*                If no characters in sub_str list appear in the input string,
*		 a NULL is returned.
*
*                This routine also protects against NULL on both the input and
*                substring variables.  If either is NULL, the value of NULL is
*                returned.
*    Written By: Mark Young & Steven Jorgensen
*          Date: Jul 03, 1992 14:05
* Modifications: Converted from vstrpbrk in Khoros 1.0 (SJ)
*************************************************************/

char *kstrpbrk(
   char *istr,
   char *sub_str)
{
#ifdef NEEDS_STRPBRK
	char    *string;
#endif

	if (istr == NULL || sub_str == NULL)
	{
	   if (istr == NULL)
	      kinfo(KSYSLIB,"kstrpbrk: NULL input string encountered" );

	   if (sub_str == NULL)
	      kinfo(KSYSLIB,"kstrpbrk: NULL token string encountered" );

	   return(NULL);
	}

	/*
	 *  March thru the string using kstrchr() to find the sub-string
	 */
#ifdef NEEDS_STRPBRK
	string = istr;
	while (*string != '\0')
	{
	   if (kstrchr(sub_str, *string))
	      return(string);

	   string++;
	}
	return(NULL);
#else
	return(strpbrk(istr, sub_str));
#endif
}

/************************************************************
*
*  Routine Name: kstrtok - find a token within a string
*       Purpose: This routine is a replacement for the system
*		 library routine strtok.  kstrtok searches
*		 the input string, istr, for a token separator, which is
*		 specified in the second string parameter, sub_str.
*		 It then returns the pointer to the input string
*		 after inserting a '\\0' at the token separator.  If
*		 it is called again with a NULL as the input string, the
*		 routine continues to parse the previous string passed in.
*         Input: istr - the string to search
*		 sub_str   - the sub string to look for.
*       Returns: A pointer to the first token in the istr.
*		 Otherwise, it returns NULL on error, or if the
*		 token is NULL.
*    Written By: Mark Young & Steven Jorgensen
*          Date: Jul 03, 1992 14:05
*  Side Effects: This routine adds '\\0's to the original string.  It also
*		 has no way to check if the istr has been kfree'd
*		 between calls to kstrtok.
* Modifications: Converted from vstrtok in Khoros 1.0 (SJ)
*************************************************************/

char *kstrtok(
   char *istr,
   char *sub_str)
{
	static char *string = NULL;

	if (istr == NULL && string == NULL)
	   return(NULL);
	else if (istr != NULL)
	   string = istr;

#ifdef NEEDS_STRTOK
	if (istr == NULL)
	   istr = string;
	if ((string = kstrpbrk(string, sub_str)) != NULL)
	{
	   *string++ = '\0';
	   while (*string != '\0' && kstrchr(sub_str, *string))
	      string++;
	}
	return(istr);
#else
	return(strtok(istr, sub_str));
#endif
}


/************************************************************
*
*  Routine Name: kstrdup - return a duplicate of the input string
*       Purpose: This routine is similiar to the system routine
*		 strdup.  This routine will uses kmalloc to create
*		 a memory buffer large enough to hold the string passed
*		 into this routine, and then proceeds to copy the
*		 input string to the buffer.  This buffer is then
*		 returned to the user. 
*         Input: string - the string to be duplicated
*       Returns: The duplicated of string.  This routine will return
*		 NULL on a malloc error, or if the input string is
*		 NULL.
*    Written By: Mark Young
*          Date: Nov 28, 1992 22:46
*************************************************************/

char *kstrdup(
   char *string)
{
#ifdef NEEDS_STRDUP
	return(!string ? NULL : kdupalloc(string, strlen(string)+1));
#else
	return((string == NULL) ? NULL : strdup(string));
#endif
}


/************************************************************
*
*  Routine Name: kstrspn - return the number of matched characters
*       Purpose: This routine is similiar to the system routine
*		 strspn.  It counts the number of characters at the start of the
*		 input string that consist entirely of characters from
*		 the charset string.  The count stops at the first
*		 character in the input string that is not in the charset
*		 string.
*         Input: string - the string to be checked
*		 charset - the character set to check against
*       Returns: The number of characters matched in the string,
*		 from the charset.   If either the string or charset
*		 strings are NULL, a value of 0 is returned.
*    Written By: Mark Young
*          Date: Nov 28, 1992 22:46
*************************************************************/

int kstrspn(
   char *string,
   char *charset)
{
	register char *str = string;

	/*
	 *  Sanity check...
	 */
	if (string == NULL || charset == NULL)
	   return(0);

#ifdef NEEDS_STRSPN
	while (*str != '\0')
	{
	   if (!kstrchr(charset, *str))
	      break;
	   else str++;
	}
	return(str - string);
#else
	return(strspn(string, charset));
#endif
}


/************************************************************
*
*  Routine Name: kstrcspn - return the number of characters not matched
*       Purpose: This routine is similiar to the system routine
*                strcspn.  It counts the number of characters at the start of
*		 the input string that consist entirely of characters not in
*                list of characters specified by the charset string.  The
*		 count stops at the first character in the input string that
*		 is in the charset string.
*         Input: string - the string to be checked
*		 charset - the character set to check against
*       Returns: The number of characters not matched in the string,
*		 from the charset, or 0 otherwise
*    Written By: Mark Young
*          Date: Nov 28, 1992 22:46
*************************************************************/

int kstrcspn(
   char *string,
   char *charset)
{
	register char *str = string;

	/*
	 *  Sanity check...
	 */
	if (string == NULL || charset == NULL)
	   return(0);

#ifdef NEEDS_STRCSPN
	while (*str != '\0')
	{
	   if (kstrchr(charset, *str))
	      break;
	   else str++;
	}
	return(str - string);
#else
	return(strcspn(string, charset));
#endif
}


/************************************************************
*
*  Routine Name: kstrcasecmp - do a case insensitive string compare
*       Purpose: This routine is a replacement for the system
*		 strcasecmp.  It does a character by character case
*		 insensitive comparison of both input strings until the
*		 current character for one string does not equal the current
*		 character of the other string or until the end of the input
*		 strings are reached.  It protects against NULL on the input
*		 strings by replacing NULL pointers with a reference to an
*		 empty string.
*         Input: istr1 - the first string to compare
*                istr2 - the second string to compare
*       RETURNS: If the two strings are identical, the value of 0 is returned.
*		 If the two strings differ, the ASCII value difference of
*		 the first character that differs in the two strings will be
*		 returned.  If the ASCII value of the differing character in
*		 istr1 is greater than the one in istr2, then the return
*		 value is positive, and implies that istr1 is greater than
*		 istr2.  If the difference between the two ASCII values is
*		 negative, the return value implies that istr2 is greater than
*		 istr1.
*    Written By: Mark Young and Steven Jorgensen
*          Date: Nov 28, 1992 22:33
*************************************************************/

int kstrcasecmp(
   char *istr1,
   char *istr2)
{
	istr1 = nullstring(istr1); 
	istr2 = nullstring(istr2); 

#ifdef NEEDS_STRCASECMP
	while (tolower(*istr1) == tolower(*istr2))
	{
	   if (*istr1++ == '\0') return(0);
	   istr2++;
	}
	return(tolower(*istr1) - tolower(*istr2));
#else
	return(strcasecmp(istr1, istr2));
#endif
}


/************************************************************
*
*  Routine Name: kstrncasecmp - do a case insensitive string compare on n
*				characters
*       Purpose: This routine is a replacement for the system
*                strcasecmp.  It does a character by character 
*		 case insensitive comparison of both input strings until
*		 the current character for one string does not equal the
*		 current character of the other string, until the end of the
*		 input strings are reached, or until num characters have been
*		 compared.  It protects against NULL on the input strings
*                by replacing NULL pointers with a reference to an empty
*                string.
*         Input: istr1 - the first string to compare
*                istr2 - the second string to compare
*		 num  - the number of bytes to compare
*       RETURNS: If the two strings are identical, the value of 0 is returned.
*		 If the two strings differ, the ASCII value difference of
*                the first character that differs in the two strings will be
*                returned.  If the ASCII value of the differing character in
*                istr1 is greater than the one in istr2, then the return
*                value is positive, and implies that istr1 is greater than 
*                istr2.  If the difference between the two ASCII values is
*                negative, the return value implies that istr2 is greater than
*                istr1.
*    Written By: Mark Young and Steven Jorgensen
*          Date: Nov 28, 1992 22:35
*************************************************************/

int kstrncasecmp(
   char *istr1,
   char *istr2,
   int  num)
{
	istr1 = nullstring(istr1); 
	istr2 = nullstring(istr2); 

#ifdef NEEDS_STRNCASECMP
	while (--num > 0 && tolower(*istr1) == tolower(*istr2))
	{
	   if (*istr1++ == '\0') return(0);
	   istr2++;
	}
	return(tolower(*istr1) - tolower(*istr2));
#else
	return(strncasecmp(istr1, istr2, num));
#endif
}


/************************************************************
*
*  Routine Name: kstrlen - return the length of a string
*       Purpose: This routine is similiar to the system routine
*		 strlen; it counts the number of characters in a
*		 string until it reaches a '\\0' character.
*		 This routine is better than many standard unix
*		 strlens, because it treats a NULL pointer as an
*		 empty string.
*         Input: string - the string to get the length of.
*       Returns: The number of characters in the string.  On
*		 a NULL input string, a value of 0 is returned.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:31
* Modifications: Rewritten from Khoros 1.0 (MY)
*************************************************************/

int kstrlen(
   char *string)
{
	register int i = 0;

	/*
	 *  Sanity check...
	 */
	if (!string)
	   return(0);

#ifdef NEEDS_STRLEN
	while (*string++ != '\0') i++;
	return(i);
#else
	return(strlen(string));
#endif
}

/************************************************************
*
*  Routine Name: kstrcmp - compare two strings
*       Purpose: This routine is a replacement for the system
*                strcmp.  It does a character by character
*                case sensitive comparison of both input strings until
*                the current character for one string does not equal the
*                current character of the other string, until the end of the
*                input strings are reached, or until num characters have been
*                compared.  It protects against NULL on the input strings
*                by replacing NULL pointers with a reference to an empty
*                string.
*         Input: istr1 - a character pointer to the first string
*		 istr2 - a character pointer to the second string
*       Returns: If the two strings are identical, the value of 0 is returned.
*                If the two strings differ, the ASCII value difference of
*                the first character that differs in the two strings will be
*                returned.  If the ASCII value of the differing character in
*                istr1 is greater than the one in istr2, then the return
*                value is positive, and implies that istr1 is greater than
*                istr2.  If the difference between the two ASCII values is
*                negative, the return value implies that istr2 is greater than
*                istr1.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:31
*************************************************************/

int kstrcmp(
   char *istr1,
   char *istr2)
{
	istr1 = nullstring(istr1); 
	istr2 = nullstring(istr2); 

#ifdef NEEDS_STRCMP
	while (*istr1 == *istr2)
	{
	   if (*istr1++ == '\0') return(0);
	   istr2++;
	}
	return(*istr1 - *istr2);
#else
	return(strcmp(istr1,istr2));
#endif
}

/************************************************************
*
*  Routine Name: kstrncmp - compare the first n characters of two strings
*       Purpose: This routine is a replacement for the system
*                strcasecmp.  It does a character by character
*                case sensitive comparison of both input strings until
*                the current character for one string does not equal the
*                current character of the other string, until the end of the
*                input strings are reached, or until num characters have been
*                compared.  It protects against NULL on the input strings
*                by replacing NULL pointers with a reference to an empty
*                string.
*         Input: istr1 - a character pointer to the first string
*		 istr2 - a character pointer to the second string
*		 num   - number of characters to compare
*       Returns: If the two strings are identical, the value of 0 is returned.
*                If the two strings differ, the ASCII value difference of
*                the first character that differs in the two strings will be
*                returned.  If the ASCII value of the differing character in
*                istr1 is greater than the one in istr2, then the return
*                value is positive, and implies that istr1 is greater than
*                istr2.  If the difference between the two ASCII values is
*                negative, the return value implies that istr2 is greater than
*                istr1.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:33
*************************************************************/

int kstrncmp(
   char *istr1,
   char *istr2,
   int  num)
{
	istr1 = nullstring(istr1); 
	istr2 = nullstring(istr2); 

#ifdef NEEDS_STRNCMP
	while (--num > 0 && *istr1 == *istr2)
	{
	   if (*istr1++ == '\0') return(0);
	   istr2++;
	}
	return(*istr1 - *istr2);
#else
	return(strncmp(istr1, istr2, num));
#endif
}

/************************************************************
*
*  Routine Name: kstrchr - find a character in a string
*       Purpose: This routine is similar to the system strchr call.
*		 kstrchr searches the input string for the first
*		 occurance of a character, ch.
*         Input: str - string to search
*		 ch  - character to look for
*       Returns: A pointer to the address in the input string where
*		 the 'ch' character is located.  NULL is returned
*		 if 'ch' is not in the input string, or if the input
*		 string is NULL
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:35
*************************************************************/

char *kstrchr(
   char *str,
   int  ch)
{
	/*
	 *  Sanity check...
	 */
	if (!str)
	   return(NULL);

#ifdef NEEDS_STRCHR
	while (*str != ch && *str != '\0')
	   str++;

	return(*str == ch ? str : NULL);
#else
	return(strchr(str, ch));
#endif
}

/************************************************************
*
*  Routine Name: kstrrchr - find a character in a string
*       Purpose: This function is similar to system call strrchr.
*		 kstsrrchr searches the input string from back
*		 to front for a character specified by 'ch' in the
*		 input string.
*         Input: str - string to search
*		 ch  - character to look for
*       Returns: A pointer to the address in the input string where
*		 the first occurance from the end of the string of the 'ch'
*		 character.  NULL is returned if 'ch' is not in the
*		 input string, or if a NULL input string is provided.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:36
*************************************************************/

char *kstrrchr(
   char *str,
   int  ch)
{
#ifdef NEEDS_STRRCHR
	register char *strend;
#endif

	/*
	 *  Sanity check...
	 */
	if (str == NULL)
	   return(NULL);

#ifdef NEEDS_STRRCHR
	strend = str + kstrlen(str);
	while (*strend != ch && strend != str)
	   strend--;

	return(*strend == ch ? strend : NULL);
#else
	return(strrchr(str, ch));
#endif
}


/************************************************************
*
*  Routine Name: kstrcpy - copy a string
*       Purpose: This function is similar to system call strcpy.
*		 kstrcpy is used used to copy each character in the input
*		 string to the output string.
*         Input: istr2 - string to copy from
*        Output: istr1 - string to copy into
*       Returns: A pointer to istr1 after istr2 is copied to istr1.  If
*		 istr2 is NULL, istr1 is returned without modification.
*		 if istr1 is NULL, then NULL is returned.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:41
*************************************************************/

char *kstrcpy(
   char *istr1,
   char *istr2)
{
	char *string = istr1;

	/*
	 *  Sanity check...
	 */
	if (istr1 == NULL)
	   return(NULL);
	else if (istr2 == NULL)
	   return(istr1);

#ifdef NEEDS_STRCPY
	/*EMPTY*/
	while ((*istr1++ = *istr2++) != '\0');

	return(string);
#else
	return(strcpy(istr1,istr2));
#endif
}

/************************************************************
*
*  Routine Name: kstrncpy - copy the first n characters in a string
*       Purpose: This function is similar to system call strncpy().
*		 kstrncopy copies up to num characters in the input string to
*		 the output string.
*         Input: istr2 - string to copy from
*		 num   - number of characters to copy
*        Output: istr1 - string to copy into
*       Returns: A pointer to istr1 after up to num istr2 is copied to istr1.
*		 If istr1 or istr2 is NULL, then NULL is returned.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:42
*************************************************************/

char *kstrncpy(
   char *istr1,
   char *istr2,
   int  num)
{
	char *string = istr1;

	/*
	 *  Sanity check...
	 */
	if (istr1 == NULL)
	   return(NULL);
#ifdef NEEDS_STRNCPY
	else if (istr2 != NULL)
	{
	   /*EMPTY*/
	   while (num-- > 0 && (*istr1++ = *istr2++) != '\0');
	}
	while (num-- > 0) *istr1++ = '\0';
	return(string);
#else
	return(!istr1 || !istr2 ? NULL : strncpy(istr1, istr2, num));
#endif
}

/************************************************************
*
*  Routine Name: kstrcat - concatenate two strings
*       Purpose: This function is similar to system call strcat.
*		 kstrcat appends all the characters from istr2 to the
*		 end of istr1.
*         Input: istr1 - first string concatenate
*		 istr2 - second string to concatenate
*        Output: istr1 - resulting combined output string
*       Returns: A pointer to istr1 after all the characters in istr2 are
*		 appended onto istr1.  It will return NULL if istr1 is NULL.
*		 If istr2 is NULL, istr1 is returned unchanged.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:44
*************************************************************/

char *kstrcat(
   char *istr1,
   char *istr2)
{
	char *string = istr1;

	/*
	 *  Sanity check...
	 */
	if (istr1 == NULL)
	   return(NULL);
	else if (istr2 == NULL)
	   return(istr1);

#ifdef NEEDS_STRCAT
	istr1 += kstrlen(istr1);

	/*EMPTY*/
	while ((*istr1++ = *istr2++) != '\0');
	return(string);
#else
	return(strcat(istr1, istr2));
#endif
}

/************************************************************
*
*  Routine Name: kstrncat - concatenate up to n characters on a string
*       Purpose: This function is similar to system call strncat.
*		 kstrncat appends upto num characters from istr2 to the
*		 end of istr1.
*         Input: istr1 - first string concatenate
*		 istr2 - second string to concatenate
*		 num   - number of characters in istr2 to concat onto istr1
*        Output: istr1 - resulting combined output string
*       Returns: A pointer to istr1 after num characters in istr2 are appended
*		 onto istr1.  It will return NULL, if istr1 is NULL.
*		 if istr2 is NULL, istr1 is returned unchanged.
*    Written By: Mark Young
*          Date: Nov 28, 1992 11:45
*************************************************************/

char *kstrncat(
   char *istr1,
   char *istr2,
   int  num)
{
	char *string = istr1;

	/*
	 *  Sanity check...
	 */
	if (istr1 == NULL)
	   return(NULL);
	else if (istr2 != NULL)
	   return(istr1);

#ifdef NEEDS_STRNCAT
	/*EMPTY*/
	while (num-- > 0 && (*istr1++ = *istr2++) != '\0');
	return(string);
#else
	return(strncat(istr1, istr2, num));
#endif
}


/************************************************************
*
*  Routine Name: kstring_lower - convert a string to lower case.
*       Purpose: This routine performs a character by character
*		 scan for uppercase characters.  When an uppercase
*		 character is found, it calls tolower to get the
*		 lowercase equivalent.
*         Input: istr - string to convert to lower case
*        Output: ostr - string that holds the converted string.
*			If it's NULL, it kmallocs the space necessary.
*       Returns: ostr if it is not NULL, a pointer to the resulting
*		 kmalloc'ed string if ostr is NULL.  NULL is
*		 returned on an alloc error or if istr is NULL.
*    Written By: Mark Young
*          Date: Jul 13, 1993
*  Side Effects: Mallocs the space for the output if ostr is NULL
* Modifications: Converted from vlower_string in Khoros 1.0 (SJ)
*************************************************************/

char *kstring_lower(
   char *istr,
   char *ostr)
{
	char *string;

	if (!istr)
	{
	   kinfo(KSYSLIB,"kstring_lower: NULL input string encountered" );
	   return(NULL);
	}

	/*
	 *  If the ostr is NULL then kstrdup the input string
	 */
	if (!ostr)
	   ostr = kstrdup(istr);
	else if (istr != ostr)
	   kstrcpy(ostr, istr);

	string = ostr;
	while (*string != '\0')
	{
	   if (isupper(*string))
	      *string = tolower(*string);

	   string++;
	}
	return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_upper - convert a string to upper case.
*       Purpose: This routine performs a character by character
*                scan for lowercase characters.  When an lowercase
*                character is found, it calls toupper to get the
*                uppercase equivalent.
*         Input: istr - string to convert to upper case
*        Output: ostr - string that holds the converted string.
*			If it's NULL, it kmallocs the space necessary.
*       Returns: ostr if it is not NULL, a pointer to the
*		 resulting kmalloc'ed string if it is NULL.  NULL on
*		 an error.
*    Written By: Mark Young
*          Date: Jul 13, 1993
*  Side Effects: Mallocs the space for the output if ostr is NULL
* Modifications: Converted from vupper_string in Khoros 1.0 (SJ)
*************************************************************/
char *kstring_upper(
   char *istr,
   char *ostr)
{
	char *string;

	if (!istr)
	{
	   kinfo(KSYSLIB,"kstring_upper: NULL input string encountered" );
	   return(NULL);
	}

	/*
	 *  If the ostr is NULL then kstrdup the input string
	 */
	if (!ostr)
	   ostr = kstrdup(istr);
	else if (istr != ostr)
	   kstrcpy(ostr, istr);

	string = ostr;
	while (*string != '\0')
	{
	   if (islower(*string))
	      *string = toupper(*string);

	   string++;
	}
	return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_capitalize - convert a string to capitalization
*       Purpose: This routine checks to see that the first character
*		 of each word is capitalized.  A word is defined to be
*		 something separated by whitespace (i.e. separated by
*		 one or more tabs or spaces).
*         Input: istr - string to convert to capital
*        Output: ostr - string that holds the converted string.
*			If it's NULL, it kmallocs the space necessary.
*       Returns: ostr if it is not NULL, a pointer to the
*		 resulting kmalloc'ed string if it is NULL.  NULL on
*		 an error.
*    Written By: Mark Young
*          Date: Feb 20, 1994
*  Side Effects: Mallocs the space for the output if ostr is NULL
*************************************************************/

char *kstring_capitalize(
   char *istr,
   char *ostr)
{
	char *string;

	if (!istr)
	{
	   kinfo(KSYSLIB,"kstring_capitalize: NULL input string encountered" );
	   return(NULL);
	}

	/*
	 *  If the ostr is NULL then kstrdup the input string
	 */
	if (!ostr)
	   ostr = kstrdup(istr);
	else if (istr != ostr)
	   kstrcpy(ostr, istr);

	string = ostr;
	while (*string != '\0')
	{
	   if (islower(*string))
	      *string++ = toupper(*string);
	   else
	      string++;

	  /*
	   * STEVE:  Whitespace isn't the only "new word" marker what about
	   * comma period, colon, etc?
	   */
	   while (*string != '\0' && !isspace(*string))
	      string++;

	   while (*string != '\0' && isspace(*string))
	      string++;
	}
	return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_replace - replace one substring with another.
*       Purpose: This function performs a global change of text on
*		 the input string.  It returns a string where every
*		 occurrence of the scan string is replaced with the
*		 replacement string.  If ostr is sent in as NULL,
*		 the result will be kmalloc'ed for you.  If ostr
*		 is provided as non-null, the result will be stored in it.
*		 Note that if the latter use is chosen, ostr must
*		 be previously allocated with a sufficient amount of storage
*		 space before this routine is called.
*
*		 For example, if we call:
*		 new_string = kstring_replace("Welcome to Khoros", "Khoros",
*					      "the Twilight Zone", NULL);
*		 the result will be:
*		 new_string = "Welcome to the Twilight Zone".
*         Input: istr  - string to be changed
*		 scan_str      - string to be replaced
*		 replace_str   - string to replace the scan_str
*        Output: ostr - string that holds the converted string.
*				If it's NULL, it kmallocs the space necessary.
*       Returns: ostr if it is not NULL, a pointer to the
*		 resulting kmalloc'ed string if it is NULL.  NULL on
*		 an error.
*    Written By: Mark Young
*          Date: Jul 03, 1992 14:04
*  Side Effects: Mallocs the space for the output if ostr is NULL
* Modifications: Converted from vreplace_string in Khoros 1.0 (SJ)
*************************************************************/

char *kstring_replace(
   char *istr,
   char *scan_str,
   char *replace_str,
   char *ostr)
{
    char    *string, *temp = NULL, *save = istr;
    int	ilen, rlen, size = 0, length, numchar;

    if (istr == NULL)
      {
	kinfo(KSYSLIB,"kstring_replace: NULL input string encountered" );
	return(NULL);
      }

    if (scan_str == NULL)
      {
	kinfo(KSYSLIB,"kstring_replace: NULL scan string encountered" );
	return(NULL);
      }

    ilen = kstrlen(scan_str);
    rlen = kstrlen(replace_str);
    length = kmax(rlen, KLENGTH);

	/*
	 *  If the ostr is NULL then we need to kallocate temporary
	 *  space for the string.  Another problem case is when the input
	 *  and output strings are the same but the replacement token is
	 *  larger than the input token.  In this case we use a temporary
	 *  array and then kstrcpy the result into the output array, hoping
	 *  that user was smart enough to kmalloc enough space for us.
	 */
    if ((save == ostr && rlen > ilen) || (ostr==NULL))
      {
	size = length;
	if ((temp = (char *) kmalloc((unsigned) sizeof(char)*size)) == NULL)
	  {
	    kinfo(KSYSLIB,"kstring_replace: Out of memory for temporary \
string");
	    return(NULL);
	  }
	string = temp;
      }
    else
      string = ostr;

    while (*istr != '\0')
      {
	if (temp != NULL)
	  {
	    numchar = string - temp;
	    if (numchar >= (size-rlen-1))
	      {
	        size += length;
	        if ((temp = (char *) krealloc(temp, size)) == NULL)
	          {
		    kinfo(KSYSLIB,"kstring_replace: Out of memory for \
temporary string");
		    return(NULL);
	          }
	       string = temp + numchar;
	      }
	  }

	if (kstrncmp(istr, scan_str, ilen) == 0)
	  {
	    kstrncpy(string, replace_str, rlen);
	    string += rlen;
	    istr += ilen;
	  }
	else
	  *string++ = *istr++;
      }
    *string = '\0';

	/*
	 *  Check to see if the output string should be kmalloc'ed or if
	 *  the input and output strings are the same but we used a temporary
	 *  since the input token was smaller than the replacement token.
	 */
    if (ostr == NULL)
      {
	ostr = kstrdup(temp);
	kfree(temp);
      }
    else if (save == ostr && rlen > ilen)
      {
	kstrcpy(ostr, temp);
	kfree(temp);
      }
    return(ostr);
  }

/************************************************************
*
*  Routine Name: kchar_replace - replace a character with another
*       Purpose: This function performs a global change of character on
*		 the input string.  It returns a string where every
*		 occurrence of the scan character is replaced with the
*		 replacement character.  If ostr is sent in as NULL,
*		 the result will be kmalloc'ed for you.  If ostr
*		 is provided as non-null, the result will be stored in
*		 ostr.  Note that if the latter use is chosen,
*		 ostr must be previously kallocated with a sufficient
*		 amount of storage space before this routine is called.
*
*		 For example, if we call:
*		 new_string = kchar_replace("Many fishes", 'f', 'w', NULL);
*		 the result will be:
*		 new_string = "Many wishes".
*         Input: istr  - the string to be changed
*		 scan_char     - the character to be replaced
*		 replace_char  - the character to replace the scan_char
*        Output: ostr - string that holds the converted string.
*				 If it's NULL, it kmallocs the space necessary.
*       Returns: ostr if it is not NULL, a pointer to the
*		 resulting kmalloc'ed string if it is NULL.  NULL on
*		 an error.
*    Written By: Mark Young
*          Date: Jul 03, 1992
*  Side Effects: Mallocs the space for the output if ostr is NULL
* Modifications: Converted from vreplace_char in Khoros 1.0 (SJ)
*************************************************************/

char *kchar_replace(
   char *istr,
   char scan_char,
   char replace_char,
   char *ostr)
{
	char *string;

	if (istr == NULL)
	{
	   kinfo(KSYSLIB, "kchar_replace: NULL input string encountered");
	   return(NULL);
	}

	/*
	 *  If the ostr is NULL then kstrdup the input string
	 */
	if (ostr == NULL)
	   ostr = kstrdup(istr);
	else if (istr != ostr)
	   kstrcpy(ostr, istr);

	string = ostr;
	while (string != NULL)
	{
	   if ((string = kstrchr(string, scan_char)) != NULL)
	   {
	      *string++ = replace_char;
	      if (*string == '\0') break;
	   }
	}
	return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_cleanup - remove white space from the ends of a string
*       Purpose: This routine checks the beginning and end of the string
*		 for white space (using isspace()) and removes these
*		 characters.  White space is defined to be a tab or space.
*         Input: istr - string to cleanup
*        Output: ostr - string that holds the converted string.
*			If it's NULL, it kmallocs the space necessary.
*       Returns: ostr if it is not NULL, a pointer to the
*		 resulting kmalloc'ed string if it is NULL.  NULL on
*		 an error.
*    Written By: Mark Young
*          Date: Jul 03, 1992
*  Side Effects: Mallocs the space for the output if ostr is NULL
* Modifications: Converted from vcleanup_string in Khoros 1.0 (SJ)
*************************************************************/

char *kstring_cleanup(
   char *istr,
   char *ostr)
{
	char *string, *temp;

	if (istr == NULL)
	{
	   if (ostr) ostr[0] = '\0';
	   return(NULL);
	}

	if (ostr == NULL)
	   string = kstrdup(istr);
	else if (istr != ostr)
	   string = kstrcpy(ostr, istr);
	else
	   string = ostr;

	if ((temp = _cleanup_string(string)) == NULL)
	{
	   if (ostr != NULL) ostr[0] = '\0';
	      else kfree(string);

	   return(NULL);
	}
	else if (temp != string)
	{
	   kstrcpy(string, temp);
	}
	return(string);
}

/************************************************************
*
*  Routine Name: kstring_format - Format a string
*       Purpose: This routine formats a string.
*         Input: istr - string to be formatted
*        Output: ostr - string that holds the formatted string.
*			If it's NULL, it kmallocs the space necessary.
*       Returns: ostr if it is not NULL, a pointer to the
*		 resulting kmalloc'ed string if it is NULL.  NULL on
*		 an error.
*  Restrictions: NOTE TO MARK this routine doesn't do anything, and
*		 it returns a bad pointer.. :) (stevej)
*    Written By: Mark Young
*          Date: Jul 03, 1992
*************************************************************/

char *kstring_format(
   char *istr,
   char *prepend,
   char *append,
   int  format_width,
   int  ignore_cr,
   char *ostr)
{
	char *string, temp[10*KLENGTH];


	if (!ostr)
	   string = temp;
	else
	   string = temp;

	return(string);
}

/*****************************************************************
*
*  Routine Name: kstring_cat - concatenate two strings
*       Purpose: This routine concatenates two strings together,
*		 into a third.  If the output parameter is not
*		 NULL, the concatenated string is put into that
*		 string.  Otherwise, the resulting string is kmalloc'ed.
*		 In either case, the resulting string is returned.
*		 Note: the second string is added to the end of the first.
*         Input: istr1 - first string to concatenate.
*		 istr2 - second string to concatenate.
*        Output: ostr  - string that holds the concatenated string.
*                        If it's NULL, it kmallocs the space necessary
*       Returns: ostr if it is non-NULL, the kmalloc'ed 
*		 string if ostr is NULL, or NULL on error.
*  Restrictions: This routine uses kmalloc to create a temporary string
*		 that holds the result. 
*    Written By: Steven Jorgensen & Nick Ruprecht
*          Date: Jul 03, 1992 14:05
*************************************************************/
char *kstring_cat(
   char *istr1,
   char *istr2,
   char *ostr)
{
	int n1 = kstrlen(istr1), n2 = kstrlen(istr2), nout = n1 + n2;

	if (nout <= 0)
	   return (NULL);
	else if (!ostr && (ostr = kmalloc(nout+1)) == NULL)
	{
	   kinfo(KSYSLIB,"kstring_cat: Cannot kmalloc enough memory for \
resulting string");
	}

	if (n2 >= 0)
	   kmemcpy(ostr + n1, istr2, n2);
	if (n1 >= 0)
	   kmemcpy(ostr, istr1, n1);

	ostr[nout] = '\0';
	return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_ncat - concatenate two partial strings
*       Purpose: This routine concatenates two partial strings together,
*		 into a third.  If the output parameter is not
*		 NULL, the concatenated string is put into that
*		 string.  Otherwise, the resulting string is kmalloc'ed.
*		 In either case, the resulting NULL-terminated string is 
*                returned.
*		 Note: the second partial string is added to the end of the
*		 partial string first.  Also, this routine puts a '\0'
*		 on the resulting string.
*         Input: istr1 - first string to concatenate.
*		 istr2 - second string to concatenate.
*		 num1  - number of characters from istr1 
*		         to put in final string
*		 num2  - number of characters from istr2 to put in `
*		         final string
*        Output: ostr - string that holds the concatenated string.
*                      If it's NULL, it kmallocs the space necessary
*       Returns: ostr if it is non-NULL, the kmalloc'ed 
*		 string if ostr is NULL, or NULL on error.
*
*  Restrictions: This routine uses kmalloc to create a temporary string
*		 that holds the result. 
*    Written By: Steven Jorgensen & Nick Ruprecht
*          Date: Jul 03, 1992 14:05
*
*************************************************************/

char *kstring_ncat(
   char *istr1,
   char *istr2,
   int  num1,
   int  num2,
   char *ostr)
{
	int n1 = kstrlen(istr1), n2 = kstrlen(istr2), nout;

	n1 = (num1 >= 0 && num1 < n1) ? num1 : n1;
	n2 = (num2 >= 0 && num2 < n2) ? num2 : n2;
	nout = n1 + n2;
	if (nout <= 0)
	   return (NULL);
	else if ((ostr==NULL) && ((ostr=kmalloc(nout+1)) == NULL))
	{
	   kinfo(KSYSLIB,"kstring_ncat: Out of memory for resulting string");
	   return(NULL);
	}

	if (n2)
	   kmemcpy(ostr + n1, istr2, n2);
	if (n1)
	   kmemcpy(ostr, istr1, n1);

	ostr[nout] = '\0';
	return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_3cat - concatenate three strings together
*       Purpose: This routine concatenates three strings together,
*		 into a new string.  If the output parameter is not
*		 NULL, the concatenated string is put into that
*		 string.  Otherwise, the resulting string is kmalloc'ed.
*		 In either case, the resulting NULL-terminated
*                string is returned.
*		 Note: the second string is added to the end of the first
*		 and the third string is added to the end of the first two.
*         Input: istr1 - first string to concatenate.
*		 istr2 - second string to concatenate.
*		 istr3 - third string to concatenate.
*        Output: ostr  - string that holds the concatenated string.
*                        If it's NULL, it kmallocs the space necessary
*       Returns: ostr if it is non-NULL, the kmalloc'ed
*		 string if ostr is NULL, or NULL on error.
*  Restrictions: This routine uses kmalloc to create a temporary string
*		 that holds the result. 
*    Written By: Steven Jorgensen & Nick Ruprecht
*          Date: Jul 03, 1992 14:05
*************************************************************/

char *kstring_3cat(
   char *istr1,
   char *istr2,
   char *istr3,
   char *ostr)
{
	int n1 = kstrlen(istr1), n2 = kstrlen(istr2), n3 = kstrlen(istr3),
	    nout = n1 + n2 + n3;

	if (nout <= 0)
	   return(NULL);
	else if (!ostr && (ostr = kmalloc(nout+1)) == NULL)
	{
	   kinfo(KSYSLIB,"kstring_3cat: Out of memory for resulting string");
	}

	if (n3 >= 0)
	   kmemcpy(ostr + n1 + n2, istr3, n3);
	if (n2 >= 0)
	   kmemcpy(ostr + n1, istr2, n2);
	if (n1 >= 0 && ostr != istr1)
	   kmemcpy(ostr, istr1, n1);

	ostr[nout] = '\0';
	return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_subcmp - compares two sub-strings
*       Purpose: This routine is similar to kstrncmp() where
*		 num is an implied value which is taken to be the size of
*		 the smaller string.  The input strings are compared,
*		 character by character until the end of the shorter string is
*		 encountered for either string.
*         Input: istr1 - first string to be compared.
*		 istr2 - second string to be compared.
*	Returns: If the two strings up to the end of the short string are
*		 identical, the value of 0 is returned.
*                If the two strings differ, the ASCII value difference of
*                the first character that differs in the two strings will be
*                returned.  If the ASCII value of the differing character in
*                istr1 is greater than the one in istr2, then the return
*                value is positive, and implies that istr1 is greater than
*                istr2.  If the difference between the two ASCII values is
*                negative, the return value implies that istr2 is greater than
*                istr1.
*    Written By: Mark Young
*          Date: Aug 26, 1994
*************************************************************/
int kstring_subcmp(
   char *istr1,
   char *istr2)
{
	istr1 = nullstring(istr1); 
	istr2 = nullstring(istr2); 

	while (*istr1 == *istr2 && *istr1 != '\0' && *istr2 != '\0')
	   istr1++, istr2++;

	if (*istr2 == '\0')
	   return(0);
	else
	   return(*istr1 - *istr2);
}

/************************************************************
*
*  Routine Name: kstring_copy - copy a string
*       Purpose: This routine copies a string into another string.  If the
*		 output parameter is not NULL, the concatenated string is put
*		 into that string.  Otherwise, the resulting string is
*		 kmalloc'ed.  In either case, the resulting string is returned.
*         Input: istr - the string to be copied
*        Output: ostr - string that holds the copied string.
*                       If it's NULL, it kmallocs the space necessary
*       Returns: out if it is non-NULL, the kmalloc'ed string if
*		 ostr is NULL, or NULL on error.
*  Restrictions: This routine uses kmalloc to create a temporary string
*		 that holds the result. 
*    Written By: Mark Young
*          Date: Sep 23, 1993
*************************************************************/

char *kstring_copy(
   char *istr,
   char *ostr)
{
        if (!istr)
           return(NULL);
	else if (ostr)
	   ostr[0] = '\0';
        else if ((ostr = kmalloc(kstrlen(istr) +1)) == NULL)
        {
           kinfo(KSYSLIB,"kstring_copy: Out of memory for output string");
           return(NULL);
        }
	kstrcpy(ostr, istr);
        return(ostr);
}

/************************************************************
*
*  Routine Name: kstring_ncopy - copy up to n characters of a string
*       Purpose: This routine concatenates two strings together,
*		 into a third.  If the output parameter is not
*		 NULL, the concatenated string is put into that
*		 string.  Otherwise, the resulting string is kmalloc'ed.
*		 In either case, the resulting string is returned.
*		 Note: unlike kstrncpy, this routine puts a '\0' on the
*		 end of the new string.
*         Input: istr - string to copy characters from.
*		 num  - the number of characters to copy from in.
*        Output: ostr - string that holds the concatenated string.
*                       If it's NULL, it kmallocs the space necessary
*       Returns: ostr if it is non-NULL, the kmalloc'ed 
*		 string if ostr is NULL, or NULL on error.
*
*  Restrictions: This routine uses kmalloc to create a temporary string
*		 that holds the result. 
*    Written By: Mark Young
*          Date: Sep 23, 1993
*************************************************************/

char *kstring_ncopy(
   char *istr,
   int  num,
   char *ostr)
{
	char *string;

	/*
	 *  Sanity check...
	 */
	if (!istr)
	   return(NULL);
	else if (!ostr)
	   string = ostr = kmalloc(num+1);
	else
	   string = ostr;

	/*EMPTY*/
	while (num-- > 0 && (*ostr++ = *istr++) != '\0')
	    ;
	*ostr = '\0';
	return(string);
}
