/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Transport Utilities
   >>>>
   >>>>  Private:
   >>>>			kfile_initialize
   >>>>			kfile_checkid
   >>>>			kfile_checkfile
   >>>>			kfile_add
   >>>>			kfile_delete
   >>>>			kfile_free
   >>>>			kfile_read
   >>>>			kfile_write
   >>>>			kfile_close
   >>>>			kfile_open
   >>>>			kfile_rename
   >>>>			kfile_dup
   >>>>			kfile_seek
   >>>>			kfile_tell
   >>>>			kfile_closeall
   >>>>			kfile_listall
   >>>>
   >>>>			ktransport_token
   >>>>			ktransport_location
   >>>>			ktransport_identifier
   >>>>			ktransport_routines
   >>>>			ktransport_machines
   >>>>
   >>>>			kcommand_to_args
   >>>>			build_command
   >>>>
   >>>>   Public:
   >>>>			kfile_reopen
   >>>>			kfile_getpermanence
   >>>>			kfile_getmachtype
   >>>>			kfile_setmachtype
   >>>>			kfile_remotehost
   >>>>			kfile_listhost
   >>>>			kfile_readdata
   >>>>			kfile_writedata
   >>>>			kfile_seddata
   >>>>			kfile_copydata
   >>>>			kfile_comparedata
   >>>>			kfile_filename
   >>>>			kfile_flags
   >>>>			kfile_type
   >>>>			kfile_state
   >>>>			kfile_setstate
   >>>>			kfile_clrstate
   >>>>			ktransport_list
   >>>>			ktransport_add
   >>>>			ktransport_delete
   >>>>			kflags_to_type
   >>>>			ktype_to_flags
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "transport.h"	


#define ALLOCSIZE  10000

static int    kfile_bufsiz;
static int    num_kfile_entries = 0;
static kfile  **kfile_entries = NULL;


/*
 *  If the user doesn't specify any convention what is the default transport.
 *  Most likely it should be FILE, but in some cases it may be different.
 */
TransportInformation *DefaultTransport = file_transport;

/*
 *  Initialize the transports to be used...
 */
static int    num_transport_entries = 0;
static TransportInformation **transport_entries = NULL;

/*
 *  The following are semiprivate routines and should *usually* not be called
 *  by the application programmer.  These routines are available for the
 *  more sophisticated application.  These routines are used as utilities
 *  by the public routines found in "transport.c".  The reason why they are
 *  semi-public is that there maybe an occasion when the user needs a
 *  "transport" routine like kopen(), but slightly different.  In which case
 *  may find these routines helpful.
 *
 *	routines:
 *			kfile_initialize
 *			kfile_checkid
 *			kfile_checkfile
 *			kfile_add
 *			kfile_delete
 *			kfile_free
 *			kfile_read
 *			kfile_write
 *			kfile_seek
 *			kfile_tell
 *			kfile_close
 *			kfile_open
 *			kfile_closeall
 *			kfile_listall
 */


/*-------------------------------------------------------------
|  
|  Routine Name: kfile_initialize -
|  
|       Purpose: This function is used to retrieve a kfile structure
|	         from the kfile structure list "kfile_entries".
|  
|         Input: none
|        Output: none
|   Called From: initialize the kfile_enties structure
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/
static int transport_initialized = FALSE;

static void kfile_initialize(void)
{
	TransportInformation *tmp;

	int    i;
	char   *bufsiz, *temp;

	/*
	 *  Initialize the kfile entry structure if it has not already been
	 *  done.
	 */
	if (!transport_initialized)
	{
	   transport_initialized = -1;
	   for (i = 0; i < NumTransports; i++)
	   {
	      if (transports[i] != NULL)
	      {
		 transport_entries = (TransportInformation **) karray_add(
			(char **) transport_entries, (kaddr) transports[i],
			num_transport_entries++);
	      }
	   }

	   /*
	    *  See if the default transport is good enough..
	    */
	   if ((tmp = ktransport_routines(kgetenv("KHOROS_TRANSPORT"))) != NULL)
              DefaultTransport = tmp;

	   kfile_bufsiz = KLENGTH;
	   if ((bufsiz = kgetenv("KHOROS_TRANSPORT_BUFSIZE")) != NULL)
	   {
	      i = strtol(bufsiz, &temp, 0);
	      if (bufsiz != temp && i >= 0) kfile_bufsiz = i;
	   }
	   knegotiate_transport(FALSE);
	   (void) kopen("file=-", KOPEN_RDONLY, 0); /* kstdin  */
	   (void) kopen("file=-", KOPEN_WRONLY, 0); /* kstdout */
	   (void) kopen("file=#", KOPEN_WRONLY, 0); /* kstderr */
	   knegotiate_transport(TRUE);
	   transport_initialized = TRUE;
	}
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_checkid -
|  
|       Purpose: This function is used to retrieve a kfile structure
|	         from the kfile structure list "kfile_entries".
|  
|         Input: id - the identifier to be checked for
|        Output: none
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications:
|  
-------------------------------------------------------------*/

kfile *kfile_checkid(
   int id)
{
	/*
	 *  Initialize the kfile entry structure if it has not already been
	 *  done.
	 */
	if (!transport_initialized)
	   kfile_initialize();

	if (id < 0 || id >= num_kfile_entries || !kfile_entries ||
	    !kfile_entries[id])
	{
	   errno = EBADF;
	   return(NULL);
	}
	return(kfile_entries[id]);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_checkfile
|  
|       Purpose: This function is used to retrieve a kfile structure
|	         from the kfile structure list.
|  
|         Input: file - the kfile structure to be checked.
|        Output: none
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

kfile *kfile_checkfile(
   kfile *file)
{
	if (!transport_initialized)
	   kfile_initialize();

	if (file)
	{
	   if (file == kstderr)
	      return(num_kfile_entries > 2 ? kfile_entries[2] : NULL);
	   else if (file == kstdout)
	      return(num_kfile_entries > 1 ? kfile_entries[1] : NULL);
	   else if (file == kstdin)
	      return(num_kfile_entries > 0 ? kfile_entries[0] : NULL);
	   else if (kfile_checkid(file->id) == file)
	      return(file);
	}
	errno = EBADF;
	return(NULL);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_add(entry)
|  
|       Purpose: This function is used to retrieve a kfile structure
|	         from 
|	         with the kfile structure.
|  
|         Input: file - the kfile structure to be freed.
|        Output: none
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

int kfile_add(
   kfile *file)
{
	int  id;

	/*
	 *  See if there are any free entries in the current table
	 *  before allocating any more space for new entries.
	 */
	if (!transport_initialized)
	   kfile_initialize();

	for (id = 0; id < num_kfile_entries; id++)
	{
	   if (kfile_entries[id] == NULL)
	   {
	      file->id = id;
	      kfile_entries[id] = file;
	      return(id);
	   }
	}
	file->id = num_kfile_entries;
	kfile_entries = (kfile **) karray_add((char **) kfile_entries, file,
				num_kfile_entries++);
	return(file->id);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_delete
|  
|       Purpose: This function is used to remove a kfile entry
|	         from the entry table.
|  
|         Input: file - the kfile structure to be freed
|        Output: returns 0 or -1 if an error occurs
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications:
|  
-------------------------------------------------------------*/

int kfile_delete(
   kfile *file)
{
	if (file->id >= num_kfile_entries || kfile_entries == NULL)
	   return(-1);
	else
	   kfile_entries[file->id] = NULL;

	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_free
|  
|       Purpose: This function is used to free the resources associated
|	         with the kfile structure.
|  
|         Input: file - the kfile structure to be freed.
|        Output:
|       Returns: 0 on success, -1 otherwise
|  
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

int kfile_free(
   kfile *file)
{
	/*
	 *  If object is NULL or the object routines are NULL then
	 *  return.
	 */
	if (file == NULL)
	   return(-1);

	if (file->path != NULL)
	   kfree(file->path);

	if (file->resources != NULL)
	   kfree(file->resources);

	if (kfile_ismybuf(file))
	   kfree(file->buf);

	kfree(file->type);

	/*
	 *  Delete us from the list
	 */
	if (file->next != NULL)
	{
	   file->next->prev = file->prev;
	   if (!file->next->prev && !file->next->next)
	      kfile_clrstate(file->next, KFILE_DUP);
	}

	if (file->prev != NULL)
	{
	   file->prev->next = file->next;
	   if (!file->prev->next && !file->prev->prev)
	      kfile_clrstate(file->prev, KFILE_DUP);
	}
	kfile_delete(file);
	kfree(file);
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_read()
|  
|       Purpose: This function is used to read using the async
|		 read/write buffer.
|  
|         Input:  file   - the kfile structure.
|                 data   - the pointer to store the data into.
|                 nbytes - the number of bytes to read.
|        Output: None
|       Returns: returns the number of bytes read from the kfile
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

int kfile_read(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
        register   int count, numread = 0;
	TransportInformation *routines = file->routines;


	/*
	 *  Sanity check to make sure that a read buffer exists
	 */
	if (nbytes <= 0 || kfile_iseof(file))
	   return(0);
	else if (kfile_isrdwr(file) && kfile_ispermanent(file))
	{
	   numread = routines->read(file, ptr, nbytes);
	   file->position += numread;
	   return(numread);
	}
	else if (!kfile_ispermanent(file) &&
	    	 (file->num_bytes - file->offset) < nbytes)
	{
	   /*
	    *  Increase the transport buffer by the amount we want to read.
	    */
	   count = kmax(nbytes - (file->num_bytes - file->offset),kfile_bufsiz);
	   file->bufsiz += count;
	   file->buf = krealloc(file->buf, file->bufsiz);

	   /*
	    *  Read in the data, so that the data will be in order transport
	    *  buffer.  This will result in the data being copied back into
	    *  the data "ptr".
	    */
	   count = routines->read(file, &file->buf[file->num_bytes], count);
	   if (count > 0)
	      file->num_bytes += count;
	   else
	      kfile_setstate(file, KFILE_EOF);
	}

	/*
	 *  Read from our transport buffer before reading directly from the
	 *  transport.
	 */
	if ((numread = kmin(file->num_bytes - file->offset, nbytes)) > 0)
	{
	   kmemcpy(ptr, file->buf + file->offset, numread);
	   file->offset += numread;
	   file->position += numread;
	   nbytes -= numread;
	}

	if (nbytes > 0)
	{
	   if (nbytes > file->bufsiz)
	   {
	      /*
	       *  Just do it...  No need to buffer the data if the amount of
	       *  data being read is larger than our buffer.  Simply read
	       *  the data then buffer the remaining bytes back into the
	       *  transport buffers, as if the data had been buffered.
	       */
	      if ((count = routines->read(file, &ptr[numread], nbytes)) > 0)
		 numread += count, file->position += count;
	      else
		 kfile_setstate(file, KFILE_EOF);

	      file->num_bytes = file->offset = kmin(numread, file->bufsiz);
	      if (kfile_ismybuf(file))
	      {
	         kmemcpy(file->buf, &ptr[numread-file->num_bytes],
				file->num_bytes);
	      }
	   }
	   else
	   {
	      if ((count = routines->read(file, file->buf, file->bufsiz)) > 0)
	      {
		 nbytes = kmin(count, nbytes);
		 file->offset    = nbytes;
		 file->num_bytes = count;
	         kmemcpy(&ptr[numread], file->buf, nbytes);
	         numread += nbytes;
	         file->position += nbytes;
	      }
	      else
	         kfile_setstate(file, KFILE_EOF);
	   }
	}
	return(numread);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_write()
|  
|       Purpose: This function is used to write using the async
|		 read/write buffer.
|
|         Input:  file   - the kfile structure.
|                 ptr    - the pointer to store the data into.
|                 nbytes - the number of bytes to write.
|        Output: None
|	Returns: returns the number of bytes written to the kfile
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

int kfile_write(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	register   int numwrite = 0;
	TransportInformation *routines = file->routines;


	/*
	 *  Sanity check to make sure that a write buffer exists
	 */
	if (nbytes <= 0)
	   return(0);
	else if (kfile_isrdwr(file) && kfile_ispermanent(file))
	{
	   if (!routines)
	      return(numwrite);
	   numwrite = routines->write(file, ptr, nbytes);
	   file->position += numwrite;
	   return(numwrite);
	}
	else if (!kfile_ispermanent(file) &&
	         (file->bufsiz - file->offset) < nbytes)
	{
	   file->bufsiz = nbytes + file->offset;
	   file->buf = krealloc(file->buf, file->bufsiz);
	}

	/*
	 *  Just do it...  No need to buffer the data if the amount of
	 *  data being written is larger than our buffer.  Simply write
	 *  the data in the buffer in case there has been any data buffered.
	 *  Then write the user's data.
	 */
	if ((file->bufsiz - file->offset) < nbytes)
	{
	   if (file->offset > 0)
	   {
	      (void) routines->write(file, file->buf, file->offset);
	      file->offset = file->num_bytes = 0;
	   }
	   numwrite = routines->write(file, ptr, nbytes);
	}
	else
	{
	   kmemcpy(file->buf + file->offset, ptr, nbytes);
	   file->offset += nbytes;
	   file->num_bytes = kmax(file->offset, file->num_bytes);
	   numwrite = nbytes;
	}
	file->position += numwrite;
	return(numwrite);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_close()
|  
|       Purpose: This function is used to close a transport and
|		 delete it from the kfile entry structure.
|
|         Input:  
|        Output: None
|	Returns: returns the kfile structure or NULL upon failure
|  
|    Written By:  Mark Young
|          Date:  Oct 13, 1992
| Modifications:
|  
-------------------------------------------------------------*/

int kfile_close(
   kfile *file)
{
	int	status;
	TransportInformation *routines;


	/*
	 *  Call the "close()" routine to close the transport
	 */
	if ((routines = file->routines) == NULL)
	   return(0);

	kinfo(KDEBUG, "kfile_close: Attempting to close file '%s'", file->path);

	/*
	 *  Flush the buffer before closing...
	 */
	(void) kfflush(file);
	status = routines->close(file);
	if (kfile_istemp(file)) kunlink(file->path);

	/*
	 *  After closing the transport check to see if we should
	 *  inform the negotiator of the fact that the transport for
	 *  this file has been closed.
	 */
	if (transport_initialized == TRUE)
	   (void) knegotiate_close(file->path,  file->flags);

	kfile_free(file);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_open - open the file
|  
|       Purpose: This function is a replacement for the system "open"
|	         call.  The only difference is that kopen() supports
|	         more than just files, it supports other data transports
|	         as well.  The path should be in the form of an identifier,
|	         followed by an "=" equal indicator, and then the transport
|	         token. ie) a Shared memory path would look like:
|  
|  			"shm=1032"
|  
|	         If a file was desired then either:
|  
|  			"/usr/tmp/vadd1235"
|  			"file=/usr/tmp/vadd1235"
|  
|         Input: path - is the string containing the path name to the
|		        desired file to be open.  The path name identifies
|		        the file to be opened, just like the regular UNIX
|		        open() command.
|  
|	        flags - how the file is to be opened.
|  
|  			KOPEN_RDONLY - Open file for only reading
|  			KOPEN_WRONLY - Open file for only writing
|  			KOPEN_RDWR   - Open file for both reading & writing
|  			KOPEN_NONBLOCK - Whether to block when reading and
|  				         writing to the file.
|  			KOPEN_APPEND - Append to the file (writing).
|  			KOPEN_TRUNC  - Truncate the file for writing.
|  			KOPEN_CREAT  - Create the file if it does not alread
|				       exist
|  			KOPEN_EXCL   - Error if file exists
|  
|	         mode - the permissions to be used when creating a new file.
|        Output: none
|       Returns: the kfile id on success, -1 otherwise
|  
|  Restrictions:
|    Written By: Mark Young  
|          Date: Jul 08, 1992
|      Verified:
|  Side Effects:
| Modifications: Converted from kopen() in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

kfile *kfile_open(
   char *path,
   int  flags,
   int  mode)
{
	kfile   *file;
	int	id, locking, operation;
	TransportInformation *routines;
	char	*identifier, *filename, *check, *tmp, name[KLENGTH],
		temp[KLENGTH], type[KLENGTH];


	/*
	 *  Initialize the kfile entry structure if it has not already been
	 *  done.
	 */
	if (!transport_initialized)
	   kfile_initialize();
	kinfo(KDEBUG, "kfile_open: Attempting to open file '%s'", path);

	/*
	 *  Before opening the transport check to see if we should
	 *  negotiate a new transport for this file, or if not to
	 *  actually negotiate to inform the "negotiator" of our
	 *  attentions to open this transport.
	 */
	if (transport_initialized == TRUE)
	   (void) knegotiate_open(path, flags);

	/*
	 *  Expand the path to be it's fullpath name
	 */
	if (!kfullpath(path, NULL, name))
	   return(NULL);

	/*
	 *  Check to see if this is a compressed file (either compress or
	 *  gzip)...
	 */
	if ((tmp = kstrrchr(name, '.')) != NULL)
	{
	   if (kstrcmp(tmp, ".Z") == 0)
	   {
	      if (flags & KOPEN_WRONLY)
	         ksprintf(temp,"compress -c > %s", name);
	      else
	         ksprintf(temp,"uncompress -c %s", name);

	      return(kpopen(temp, kflags_to_type(flags, type)));
	   }
	   else if (kstrcmp(tmp, ".gz") == 0 ||
		    kstrcmp(tmp, ".z") == 0)
	   {
	      if (flags & KOPEN_WRONLY)
	         ksprintf(temp,"gzip -c > %s", name);
	      else
	         ksprintf(temp,"gunzip -c %s", name);

	      return(kpopen(temp, kflags_to_type(flags, type)));
	   }
	}

	/*
	 *  According to the transport routines
	 */
	identifier = ktransport_identifier(name, temp);
	if ((routines = ktransport_routines(identifier)) == NULL)
	   return(NULL);

	/*
	 *  Get the filename component and fullpath it...
	 */
	if ((filename = ktransport_token(name, temp)) == NULL)
	   return(NULL);

	/*
	 *  Malloc the file structure.
	 */
	if ((file = (kfile *) kcalloc(1, sizeof(struct _kobject))) == NULL)
	{
	   kinfo(KSYSLIB, "kfile_open:  Not enough memory.  Unable to kmalloc \
(%d) bytes for the khoros file structure.", sizeof(kfile));
	   return(NULL);
	}

	file->state  = 0;
	file->mode   = mode;
	file->flags  = flags;
	file->type   = kflags_to_type(flags, NULL);
	file->routines = routines;
	file->machtype = kmach_type(NULL);
	file->path     = kstrdup(name);

	/*
	 *  Set the read/write state of the transport
	 */
	if (flags & KOPEN_WRONLY)
	   kfile_setstate(file, KFILE_WRITE);
	else if (flags & KOPEN_RDWR)
	   kfile_setstate(file, KFILE_RDWR);
	else
	   kfile_setstate(file, KFILE_READ);

	/*
	 *  See if the transport has data permanence
	 */
	if (routines->data_permanence || flags & KOPEN_STREAM)
	   kfile_setstate(file, KFILE_PERM);

	/*
	 *  Initialize the file buf to use a local malloc'ed copy
	 */
	kfile_setstate(file, KFILE_MYBUF);
	file->bufsiz = (kfile_isrdwr(file) ? 0 : kfile_bufsiz);
	file->buf = (char *) kmalloc(file->bufsiz);

	/*
	 *  add the file entry to the kfile table and then open the transport
	 */
	(void) kfile_add(file);
	if (routines->open(filename, flags, mode, file) == -1)
	{
	   check = filename;
	   while (*check != '\0')
	   {
	      if (!isalnum(*check) && *check != '/' && *check != '.' &&
		  *check != '_' && *check != '~' && *check != '#' &&
		  *check != '-' && *check != '+')
	      {
	         if (*check != '\\')
		 {
	            kfile_free(file);
	            return(NULL);
		 }
		 check++;
	      }
	      check++;
	   }
	   kdirname(filename, name);

	   if (!(flags & KOPEN_CREAT) || errno != ENOENT ||
	       !kmake_dir(name, 0777) ||
	       routines->open(filename, flags, mode, file) == -1)
	   {
	      kfile_free(file);
	      return(NULL);
	   }
	}
	kfile_setstate(file, KFILE_OPEN);
	if ((flags & KOPEN_LOCK) && routines->lock != NULL)
	{
	   if (kfile_iswrite(file) == TRUE)
	      locking = KLOCK_EX;
	   else
	      locking = KLOCK_SH;

	   if (flags & KOPEN_NONBLOCK)
	      locking |= KLOCK_NB;

	   kfile_setstate(file, KFILE_LOCK);
	   (void) routines->lock(file, locking);
	}
	return(file);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_rename()
|  
|       Purpose: This function is used to re-name a transport using
|		 a different 
|
|         Input: file - the kfile structure to be re-named
|		 rename - the rename to be used
|
|        Output: None
|	Returns: returns 0 or -1 if an error occurs
|    Written By: Mark Young
|          Date: Oct 13, 1992
| Modifications:
|  
-------------------------------------------------------------*/

int kfile_rename(
   kfile *file,
   char  *name)
{
	TransportInformation *routines;
	char	*filename, *path, *identifier, temp1[KLENGTH], temp2[KLENGTH];


	/*
	 *  According to the identifier what are the transport routines
	 */
	identifier = ktransport_identifier(name, temp1);
	if ((routines = ktransport_routines(identifier)) == NULL)
	   return(-1);

	/*
	 *  Get the filename component of the new path or if that is empty
	 *  then get old path filename
	 */
	if ((filename = ktransport_token(name, temp2)) == NULL		&&
	    (filename = ktransport_token(file->path, temp2)) == NULL)
	   return(FALSE);

	/*
	 *  Build the new identifier...
	 */
	if (identifier != NULL)
	   path = kstring_3cat(identifier, "=", filename, NULL);
	else
	   path = kstrdup(filename);

	/*
	 *  close the file if it is currently opened
	 */
	if (kfile_isopen(file))
	{
	   (void) file->routines->close(file);
	   errno = 0;
	}

	/*
	 *  Before opening the transport check to see if we should
	 *  negotiate a new transport for this file, or if not to
	 *  actually negotiate to inform the "negotiator" of our
	 *  attentions to open this transport.
	 */
	if (transport_initialized == TRUE)
	{
	   (void) knegotiate_close(file->path, file->flags);
	   (void) knegotiate_open(filename, file->flags);
	}

	/*
	 *  Free up the contents of the old file structure.  Namely the
	 *  resources and path
	 */
	kfree(file->path);
	kfree(file->resources);

	/*
	 *  open the file entry to the kfile table
	 */
	file->path     = path;
	file->routines = routines;
	if (routines->open(filename, file->flags, file->mode, file) == -1)
	{
	   kfile_free(file);
	   return(-1);
	}
	kfile_setstate(file, KFILE_OPEN);
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_dup()
|  
|       Purpose: This function is used to dup a transport using
|		 the specified id2 is the new descriptor entry.
|
|         Input:  id1 - the old descriptor
|		  id2 - the new descriptor
|
|
|        Output: None
|	Returns: returns new dup'ed descriptor on succes or -1 upon failure
|  
|    Written By:  Mark Young
|          Date:  Oct 13, 1992
| Modifications:
|  
-------------------------------------------------------------*/
/* ARGSUSED */
int kfile_dup(
   int id1,
   int id2)
{
	int   flags;
	kfile *file1, *file2;


	if ((file1 = kfile_checkid(id1)) == NULL)
	   return(-1);

	flags = ~KOPEN_TRUNC & file1->flags;
	if ((file2 = kfile_open(file1->path, flags, file1->mode)) == NULL)
	   return(-1);

	/*
	 *  Go ahead and link file2 into the file1's list of dup'ed objects,
	 *  as well designate the fact that both transports have dup'ed
	 *  objects associated with it.
	 */
	kfile_setstate(file1, KFILE_DUP);
	kfile_setstate(file2, KFILE_DUP);

	/*
	 *  Copy the machine type, since it is suppose to be a dup of the
	 *  original.
	 */
	file2->machtype = file1->machtype;

	if (file1->next != NULL)
	   file1->next->prev = file2;

	file2->prev = file1;
	file2->next = file1->next;
	file1->next = file2;
	return(kfileno(file2));
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_seek()
|  
|       Purpose: This function is used to seek on the transport,
|		 which also affects the async read/write buffer.
|
|         Input:  file   - the kfile structure.
|  		  offset - the offset in which to seek
|  		  whence - the control of how the offset will be applied
|
|        Output: None
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

int kfile_seek(
   kfile *file,
   int   offset,
   int   whence)
{
	char buffer[KLENGTH*10];
	TransportInformation *routines;
	int  begin, num, pos = -1, status = 0;


	/*
	 *  if whence is KSEEK_CUR then return the position
	 */
	if (whence == KSEEK_CUR && offset == 0L)
	   return(file->position);

	/*
	 *  Rewind to the initial dup'ed transport...
	 */
	while (file->prev != NULL)
	   file = file->prev;

	do
	{
	   /*
	    *  If we are seeking within our transport buffer, then all we
	    *  should have to do is update our "offset" and "position" fields
	    *  within the file structure.
	    */
	   if (whence == KSEEK_CUR || whence == KSEEK_SET)
	   {
	      /*
	       *  Calculate the absolute position of what we want our offset
	       *  to be.  We will use this to see if we are still within our
	       *  transport buffer.
	       */
	      if (whence == KSEEK_CUR)
	      {
		 offset += file->position;
		 whence = KSEEK_SET;
	      }
	      pos = offset;

	      /*
	       *  If the desired file position is less than our current then
	       *  make sure that EOF is not set.
	       */
	      if (kfile_iseof(file))
	         kfile_clrstate(file, KFILE_EOF);

	      /*
	       *  Calculate where the beginning of our transport buffer is,
	       *  relative our seek pointer.  We will then see if the desired
	       *  position "pos" is within current transport buffer.
	       */
	      begin = file->position - (int) file->offset;
	      if (pos >= begin && pos < begin+file->num_bytes)
	      {
		 /*
		  *  Reset our read/write offset pointer within the transport
		  *  buffer relative to our new position.  Also, update our
		  *  file position to the requested new position.
		  */
		 file->offset = pos - begin;
		 file->position = pos;

		 /*
		  *  Advance to our next dup'ed transport.
	          */
		 file = file->next;
		 continue;
	      }
	   }

	   /*
	    *  Call the "seek()" routine to seek to the desired position.
	    */
	   if (!kfile_ispermanent(file))
	   {
	      if (whence == KSEEK_END)
	      {
	         if (kfile_iswrite(file))
		 {
		    file->position = file->offset = file->num_bytes;
		 }
		 else
		 {
		    while (!kfile_iseof(file))
	               kfile_read(file, buffer, KLENGTH*10);
		 }
	      }
	      else if (whence == KSEEK_SET)
	      {
	         if (kfile_iswrite(file))
		 {
		    file->position = file->offset = file->num_bytes;
		    num = offset - file->position;
		    while (num > 0)
	               num -= kfile_write(file, buffer, kmin(num, KLENGTH*10));
		 }
		 else
		 {
		    num = offset - file->position;
		    while (num > 0 && !kfile_iseof(file))
	               num -= kfile_read(file, buffer, kmin(num, KLENGTH*10));
		 }
	      }
	      pos = file->position;
	   }
	   else if ((routines = file->routines) != NULL && routines->lseek)
	   {
	      /*
	       *  Reset the read/write async buf offset and size
	       */
	      (void) kfflush(file);
	      file->offset = 0;
	      if (kfile_ismybuf(file))
	         file->num_bytes = 0;

	      if ((pos = routines->lseek(file, offset, whence)) != -1)
	      {
	         file->position = pos;
	         if (!kfile_ismybuf(file))
		 {
	            file->offset = pos;
		    file->num_bytes = file->bufsiz - file->offset;
		 }
	      }
	      else status = -1;
	   }
	   else status = -1;

	   file = file->next;
	} while (file != NULL);

	return(status == -1 ? -1 : pos);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_tell()
|  
|       Purpose: This function is used to tell on the transport.
|
|         Input:  file   - the kfile structure.
|
|        Output: None
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

int kfile_tell(
   kfile *file)
{
	return(file->position);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_closeall()
|  
|       Purpose: This function is used to close all entries and
|	         remove the entries from the table.  A status
|	         of 0 or -1 will be returned indicating
|	         whether the all entries were successfully closed.
|	         If any entry could not be closed then a status
|	         of -1 is returned, otherwise if *all* entries
|	         were successfully closed then 0 is returned.
|  
|  
|         Input:
|        Output:
|	Returns: returns the status of the closing of the open()
|	         files.
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

int kfile_closeall(void)
{
	kfile	*file;
	int	i, status = TRUE;


	for (i = 0; i < num_kfile_entries; i++)
	{
	   /*
	    *  Retrieve the file entry from the table and then NULL the
	    *  entry out of the table.
	    */
	   if ((file = kfile_entries[i]) != NULL)
	      status |= kfclose(file);
	}

	/*
	 *  Clean it up...
	 */
	num_kfile_entries = 0;
	kfree(kfile_entries);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kfile_listall()
|  
|       Purpose: This function is used to print all entries to kstderr.
|	         It is mainly used as a debugging tool.
|  
|         Input: 
|        Output:
|	Returns: 
|    Written By: Mark Young
|          Date: Apr 01, 1994
| Modifications:
|  
-------------------------------------------------------------*/

void kfile_listall(void)
{
	int	i;
	char	*name;

	for (i = 0; i < num_kfile_entries; i++)
	{
	   /*
	    *  Retrieve the file entry from the table and then NULL the
	    *  entry out of the table.
	    */
	   if (kfile_entries[i] != NULL && (name = kfile_filename(i)) != NULL)
	      kfprintf(kstderr,"%d: %s\n", i, name);
	}
}

/*
 *  The following are private routines and should not be called by the
 *  application programmer.
 *
 *	routines:
 *			ktransport_routines
 *			ktransport_identifier
 *			ktransport_location
 *			ktransport_token
 *			ktransport_machines
 */

/*-------------------------------------------------------------
|  
|  Routine Name: ktransport_routines(identifier)
|  
|       Purpose: This function is used to find the appropriate transport
|	         information
|  
|  
|         Input: identifier - the transport identifier string
|  
|        Output: None
|	Returns: None
|  
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

TransportInformation *ktransport_routines(
   char *identifier)
{
	int	i;


	if (identifier == NULL)
	   return(DefaultTransport);

	for (i = 0; i < num_transport_entries; i++)
	{
	   if (transport_entries[i] != NULL)
	   {
	      if (kstrcmp(identifier, transport_entries[i]->identifier) == 0)
		 return(transport_entries[i]);
	   }
	}
	kinfo(KSYSLIB, "ktransport_routines: Unsupported transport..  Unable \
to find support for the following transport (%s).\n", identifier);
	return(NULL);
}

/*-------------------------------------------------------------
|
|  Routine Name: ktransport_identifier(name, identifier)
|
|       Purpose: This function is used to find the transport identifier
|
|
|         Input: name - the transport string
|
|        Output: none
|
|   Called From: various kfile utilities
|
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|
-------------------------------------------------------------*/

char *ktransport_identifier(
   char *name,
   char *identifier)
{
	int  num;
	char *temp, value[KLENGTH];


	if (name == NULL)
	{
	   if (identifier) identifier[0] = '\0';
	   return(NULL);
	}

	if ((temp = kstrrchr(name, '@')) != NULL)
	{
	   identifier = kstring_copy("@", identifier);
	}
	else if ((temp = kstrchr(name, '=')) != NULL)
	{
	   num = (temp-name);
	   kstrncpy(value, name, num);
	   value[num] = '\0';
	   identifier = (char *) kstring_cleanup(value, identifier);
	}
	else
	{
	   if (identifier) identifier[0] = '\0';
	   identifier = NULL;
	}
	return(identifier);
}

/*-------------------------------------------------------------
|  
|  Routine Name: ktransport_token(name, identifier)
|  
|       Purpose: This function is used to find the token part of
|	         the transport
|  
|  
|         Input: name - the transport string
|  
|        Output: none
|  
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

char *ktransport_token(
   char *name,
   char *token)
{
	char *temp, value[KLENGTH];


	if (name == NULL)
	{
	   token = NULL;
	}
	else
	{
	   if ((temp = kstrrchr(name, '@')) != NULL)
	   {
	      kstring_ncopy(name, temp-name, value);
	      token = (char *) kstring_cleanup(value, token);
	   }
	   else if ((temp = kstrchr(name, '=')) != NULL)
	   {
	      kstrcpy(value, temp+1);
	      token = (char *) kstring_cleanup(value, token);
	   }
	   else
	   {
	      token = (char *) kstring_cleanup(name, token);
	   }
	}

	if (token == NULL)
	{
	   errno = ENOENT;
	   kinfo(KSYSLIB, "ktransport_token: No filename component.  Unable \
to find any filename for the path (%s).\n", name);
	}
	return(token);
}

/*-------------------------------------------------------------
|  
|  Routine Name: ktransport_location(name, location)
|  
|       Purpose: This function is used to find the location or
|	         machine in which the user wants to re-direct to
|	         via some transport.
|  
|  
|         Input: string - the transport string
|  
|        Output: location - the location to execute the command
|	         string   - the string will be modified to hold the
|		  	  original string minus the machine in
|		  	  which we will executing.
|  
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

char *ktransport_location(
   char *string,
   char *location)
{
	char *temp, *term, value[5*KLENGTH];


	if (string == NULL)
	   return(NULL);

	/*
	 *  Copy the name into the location and then terminate the
	 *  string after finding the first available space or tab.
	 */
	kstrcpy(value, string);
	if ((term = kstrchr(value,' '))  != NULL ||
	    (term = kstrchr(value,'\t')) != NULL)
	   *term++ = '\0';

	if ((temp = kstrrchr(value,'@')) != NULL)
	   location = (char *) kstring_cleanup(temp+1, location);
	else
	   location = NULL;

	return(location);
}

/*-------------------------------------------------------------
|  
|  Routine Name: ktransport_machines(machine, identifier)
|  
|       Purpose: This function is used to find the type of transport
|	         identifier necessary in  or
|	         machine in which the user wants to re-direct to
|	         via some transport.
|  
|  
|         Input: machine - the machine which we wish to find
|  
|        Output: none
|  
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

char *ktransport_machines(
   char *machine,
   char *identifier)
{
	char *temp, value[KLENGTH], temp2[KLENGTH];


	if (machine == NULL)
	   return(NULL);

	kstrcpy(value, machine);
	if ((temp = ktransport_identifier(value, identifier)) != NULL)
	   return(temp);
	else
	{
	   if ((temp = (char *) kstring_cleanup(value, temp2)) == NULL)
	      return(NULL);
	   if ((temp = (char *) kstring_lower(temp, value)) == NULL)
	      return(NULL);

	   if (kstrcmp(temp, "localhost") == 0)
	      temp = NULL;
	   else if (kgethostname(NULL, temp2, KLENGTH) == TRUE)
	   {
	      if (kstrcmp(temp, temp2) == 0)
		 temp = NULL;
	      else
		 temp = ktransport_identifier("@", identifier);
	   }
	   else
	      temp = ktransport_identifier("@", identifier);
	}
	return(temp);
}

/*-------------------------------------------------------------
|  
|  Routine Name: kcommand_to_args(command, args)
|  
|       Purpose: This function is used to find the appropriate daemon
|	         information
|  
|  
|         Input: command - the command to parse
|  
|        Output: args - the args structure
|  
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

void kcommand_to_args(
   char *command,
   char *args[])
{
	int     i, num = 0;
	char	*tmp, *location, temp[KLENGTH], temp2[KLENGTH], echar;


	/*
	 *  Parse the command string into seperate arguments tokens...
	 */
	command = kstrdup(command);
	if (kstrchr(command,'|') || kstrchr(command,';') ||
	    kstrchr(command,'>') || kstrchr(command,'<') ||
	    kstrchr(command,'~'))
	{
	   /*
	    *  Get the machine in which to execute this system.  If the
	    *  location is NULL then we are expected to execute it locally.
	    */
	   if ((location = ktransport_location(command, temp2)) != NULL)
	   {
	      (void) ksprintf(temp, "/bin/csh@%s", location);
	      args[num++] = temp;
	   }
	   else
	      args[num++] = "/bin/csh";

	   args[num++] = "-cf";
	   args[num++] = command;
	}
	else
	{
	   while ((int) *command != '\0' && num < MAX_ARGS)
	   {
	      while (isspace(*command) && (int) *command != '\0')
	         command++;

	      if (*command != '\0')
	      {
	         if ((int) *command == '\'' || (int) *command == '"')
	         {
		    echar = *command++;
		    args[num++] = command;
		    while ((echar != (int)*command) && ((int) *command != '\0'))
		    {
		       if ((int) *command == '\\' && (int) *(command+1) != '\0')
		          command += 2;
		       else
		          command++;
		    }

		    if (*command != '\0')
		       *command = ' ';
	         }
		 else
		    args[num++] = command;

	         while (!isspace(*command) && (int) *command != '\0')
	            command++;

	         if (*command != '\0')
	         {
	            *command = '\0';
	            command++;
	         }
	      }
	   }
	}
	args[num] = NULL;

	for (i = 0; i < num; i++)
	{
	   if (*args[i] == '$')
	   {
	      if ((tmp = kgetenv(&args[i][1])) != NULL)
		 args[i] = tmp;
	   }
	}
}

/*-------------------------------------------------------------
|  
|  Routine Name: build_command(args, command)
|  
|       Purpose: This function is used to build an appropriate command
|	         from an array of arguments.
|  
|  
|         Input: args - the arguments to parse
|  
|        Output: command - the command to build
|  
|   Called From: various kfile utilities
|  
|    Written By:  Mark Young
|          Date:  Jul 08, 1992
| Modifications: Converted from in Khoros 1.0 (MY)
|  
-------------------------------------------------------------*/

void build_command(
   char *args[],
   char *command)
{
	command[0] = '\0';
	while (*args != NULL)
	{
	   if (kstrchr(*args, ' ') || kstrchr(*args, '\t'))
	   {
	      kstrcat(command,"\'");
	      kstrcat(command, *args);
	      kstrcat(command,"\'");
	   }
	   else
	      kstrcat(command, *args);

	   args++;
	   if (*args != NULL) kstrcat(command," ");
	}
}

/*
 *  The following are public routines which are called by the application
 *  programmer.  These routines are available for the more sophisticated
 *  application.
 *
 *	routines:
 *			kfile_getpermanence
 *			kfile_getmachtype
 *			kfile_setmachtype
 *			kfile_remotehost
 *			kfile_hostlist
 *			ktransport_list
 */

/************************************************************
*
*  Routine Name: kfile_reopen - re-open a stream khoros transport
*       Purpose: This function is similar to the khoros kfreopen()
*		 call.  The only difference is that kfile_reopen()
*		 re-open the a khoros stream transport as well close
*		 and re-open all of transports that have been kdup'ed
*		 from the existing transport.
*
*		 kfile_reopen() will open a specified file on for an existing
*		 stream.  The existings stream is closed before the new
*		 filename is opened.  This function is typically used to
*		 open a specified file as one of the predefined streams;
*		 such as standard input, standard output, or standard error.
*
*         Input: path - is the string containing the path name to the
*			desired file to be open.  The path name identifies
*		 	the file to be opened, just like the regular UNIX
*		 	freopen() command.
*
*		 type - how the file is to be re-opened.
*	  	 !	"r"  - Open file for only reading
*	  	 !	"w"  - Open file for only writing
*  		 !	"a"  - append file for only writing
*  	 	 !	"A"  - append file for writing; non overwrite
*	  	 !	"r+" - Open file for both reading & writing
*	  	 !	"w+" - Open file for both reading & writing
*	  	 !	"a+" - append file for only writing
*	  	 !	"A+" - append file for writing; non overwrite
*
*		 file - the existing khoros stream transport to be reopened
*			for the specified filename.
*       Returns: returns the kfile structure or NULL upon failure
*    Written By: Mark Young
*          Date: Apr 28, 1993
*************************************************************/

kfile *kfile_reopen(
   char *path,
   char *type,
   kfile *file)
{
	char  *otype;
	int   oflags;
	kfile *temp = file;


	/*
	 *  Go ahead and rewind to the initial dup'ed object
	 */
	while (temp->prev != NULL)
	   temp = temp->prev;

	/*
	 *  Save the old flags and reopen the file with the new type
	 */
	do
	{
	   otype  = temp->type;
	   oflags = temp->flags;
	   temp->type  = type;
	   temp->flags = ktype_to_flags(type);

	   if (kfile_rename(temp, path) == -1)
	   {
	      temp->flags = oflags;
	      temp->type  = otype;
	      continue;
	   }
	   temp->type = kstrdup(temp->type);
	   kfree(otype);

	   temp = temp->next;
	} while (temp != NULL);

	return(file);
}


/**************************************************************
*  
*  Routine Name: kfile_getpermanence - returns whether a file has data
*				       permanence or not
*       Purpose: This function is used to return whether a particular
*	         file or transport identifier is a permanent data
*	         transport (like shared memory) or a connection oriented
*	         protocol like (stream or sockets).
*         Input: path  - path of identifier to inquire about
*	Returns: returns TRUE or FALSE depending whether the transport
*	         is known to support data permanence or -1 upon error
*    Written By:  Mark Young
*          Date:  Jul 08, 1992
* Modifications: Converted from in Khoros 1.0 (MY)
**************************************************************/

int kfile_getpermanence(
   char *path)
{
	TransportInformation *routines;
	char	*identifier, *filename, temp[KLENGTH];

	/*
         *  Get the filename component.  Remember the fact that if the
	 *  filename component is either a "-" or "#" this is used to
	 *  designate stdin, stdout, kstderr which definitely does not
	 *  have data permanence.
	 */
        if ((filename = ktransport_token(path, temp)) != NULL)
	{
	   if (kstrcmp(filename, "-") == 0 || kstrcmp(filename, "#") == 0)
	      return(FALSE);
	}

	/*
	 *  According to the transport routines what is the 
	 *  transport data permanence.
	 */
	identifier = ktransport_identifier(path, temp);
	routines = ktransport_routines(identifier);

	if (routines == NULL)
	   return(-1);
	else
	   return(routines->data_permanence);
}


/**************************************************************
*  
*  Routine Name: kfile_getmachtype - gets the machine architecture type
*				     for a file transport id
*       Purpose: This function returns the machine architecture type
*	         for a particular file id.  The machine type is used
*	         in conjunction with the kread_XXXX & kwrite_XXXX routines
*	         which provide data conversion for data independence.
*         Input: id - the file id to be retrieve the machtype, which was opened
*		 earlier with kopen().
*       Returns: the machine type on success, -1 otherwise
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications: Converted from in Khoros 1.0 (MY)
**************************************************************/

int kfile_getmachtype(
   int id)
{
	kfile	*file;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	/*
	 *  Return the machine type associated 
	 */
	return(file->machtype);
}


/**************************************************************
*  
*  Routine Name: kfile_setmachtype - sets the machine architecture type
*				     for a file transport id
*       Purpose: This function sets the machine architecture type
*	         for a particular file id.  The machine type is used
*	         in conjunction with the kread_XXXX & kwrite_XXXX routines
*	         which provide data conversion for data independence.
*         Input: id - the file id on which to set the machtype, which was opened
*		      earlier with kopen().
*		 machtype - the machine architecture type to set the file id to.
*       Returns: 0 on success, -1 otherwise
*    Written By: Mark Young  
*          Date: Jul 08, 1992
* Modifications: Converted from in Khoros 1.0 (MY)
**************************************************************/

int kfile_setmachtype(
   int id,
   int machtype)
{
	kfile	*file;


	/*
	 *  Get the file entry associated with the id
	 */
	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	/*
	 *  Set the machine architecture type
	 */
	file->machtype = machtype;
	return(0);
}


/**************************************************************
*  
*  Routine Name: kfile_remotehost - returns whether a host is remote
*       Purpose: This function determines whether or not the machine name
*		 passed in is the hostname or an alias of the current
*		 machine, or if it is the name of another machine
*		 on the network.
*         Input: machine - the machine name we wish to find out about
*       Returns: TRUE (1) if the machine name is a remote name,
*		 FALSE (0) otherwise
*    Written By:  Mark Young
*          Date:  Jul 08, 1992
* Modifications: Converted from in Khoros 1.0 (MY)
**************************************************************/

int kfile_remotehost(
   char *machine)
{
	char *temp, value[KLENGTH], temp2[KLENGTH];


	if (machine == NULL)
	   return(FALSE);

	kstrcpy(value, machine);
	if ((temp = (char *) kstring_cleanup(value, temp2)) == NULL)
	   return(FALSE);
	if ((temp = (char *) kstring_lower(temp, value)) == NULL)
	   return(FALSE);

	if (kstrcmp(temp, "localhost") == 0)
	   return(FALSE);
	else if (kgethostname(NULL, temp2, KLENGTH) == TRUE)
	{
	   if (kstrcmp(temp, temp2) == 0)
	      return(TRUE);
	   else
	      return(FALSE);
	}
	return(TRUE);
}


/**************************************************************
*  
*  Routine Name: kfile_listhost - get the list of machines that processes
*				  can be distributed to
*       Purpose: This function is used to return the list of initial
*	         machines to try using.  They are gotten via the user
*	         specified file and also by broadcasting to see if any
*	         khoros daemons are available (Not Available as of yet).
*        Output: num_entries - the number of entries returned
*	Returns: a pointer to an array of strings with the list of machine
*		 names.  It will return NULL on error.
*    Written By:  Mark Young
*          Date:  Jul 08, 1992
* Modifications: Converted from in Khoros 1.0 (MY)
**************************************************************/

char **kfile_listhost(
   int *num_entries)
{
	static  char **list;
	static  int num, initialized = FALSE;
	char    *host_file;


	if (initialized == FALSE)
	{
	   initialized = TRUE;
	   if ((host_file = kgetenv("KHOROS_HOSTS")) != NULL)
	   {
	      list = karray_filelist(host_file, "$USER", TRUE, &num);
	   }
	   else
	   {
	      list = NULL;
	      num = 0;
	   }
	}
	*num_entries = num;
	return(list);
}


/************************************************************
*
*  Routine Name: kfile_readdata - read the contents of a khoros transport
*				  into a data array
*       Purpose: This routine reads the contents of a file, using the khoros
*		 transport mechanisms, into an array.  If the data array is
*		 NULL then a data array of suitable size is allocated and
*		 passed back.  An optional "num_read" can be used to find
*		 out the actual number of bytes read.  If "num_read" is
*		 NULL then the parameter is ignored.
*         Input: id  - the khoros transport descriptor
*		 ptr - data to be used if not NULL
*        Output: num_read - if not NULL the number of bytes read is returned
*       Returns: a pointer to the contents of the file or NULL upon failure
*    Written By: Mark Young  
*          Date: Jan 16, 1993
*  Side Effects: the data returned is allocated and needs to freed using
*		 kfree().
*************************************************************/

kaddr kfile_readdata(
   int   id,
   kaddr ptr,
   int   *num_read)
{
	int num = 0, num_bytes, size = 0;
	char *data = (char *) ptr;


	do
	{
	    if (((num + ALLOCSIZE) > size) && !ptr)
	    {
	       size += ALLOCSIZE;
	       data = (char *) krealloc((kaddr) data, (unsigned) size+1);
	    }

	    if ((num_bytes = kread(id, &data[num], ALLOCSIZE)) > 0)
	       num += num_bytes;

	} while (num_bytes > 0);

	if (!ptr)
	   data = (char *) krealloc((kaddr) data, (unsigned) num+1);
	if (num_read)
	   *num_read = num;

	/*
	 *  Terminate the data in case it is really an ascii string
	 */
	data[num] = '\0';
	return(data);
}


/************************************************************
*
*  Routine Name: kfile_writedata - write the contents of the data to a
*				   khoros transport
*       Purpose: This routine writes the contents of a data array to the
*		 specified khoros transport mechanism.  An optional "num_write"
*		 can be used to specify the actual number of bytes to be
*		 written.  If "num_write" is 0 then the number of bytes
*		 to be written is computed from the data by using kstrlen().
*         Input: id	   - the khoros transport descriptor
*		 data      - the data array (or string) to be written
*		 num_write - if not 0 the number of bytes to be written.
*       Returns: The number of bytes written or -1 upon failure
*    Written By: Mark Young  
*          Date: Jan 16, 1993
*************************************************************/

int kfile_writedata(
   int   id,
   kaddr data,
   int   num_write)
{
	if (num_write == 0)
	   num_write = kstrlen(data);

	return(kwrite(id, data, num_write));
}

/************************************************************
*
*  Routine Name: kfile_seddata - string edit from one transport to another
*       Purpose: This routine takes two transport id's, and copies the data
*		 from the source file to the destination file.  It also
*		 uses a variable argument list of strings pairs, where
*		 each pair indication a search pattern and a replacement
*		 pattern.  This list will be used to replace strings in
*		 pattern strings in the input transport with the corresponding
*		 replacement pattern.  Note, the variable argument list MUST
*		 be terminated with a NULL in one of the search pattern slots
*
*                For example, the following is a correct termination:
*
*               ! kfile_seddata(id1, id2, FALSE, "t1", "t2", NULL);
*
*               And this is an incorrect termination:
*
*               ! kfile_seddata(id1, id2, FALSE, "t1", NULL);
*
*               The reason the NULL must be on the search pattern, is
*               so that NULL can be sent in as the replacement string.
*               The replacement mechanism is kstring_replace, and
*               its documentation explains restrictions on search strings
*               and replace strings.
*
*		The regex parameter is used to indicate whether the search
*		and replacement parameters are specified in regular expression
*		form, or are just simply string replacement patterns.  For
*		more information about regular expressions please see the
*		khoros string parsing section of the Khoros Programmers Manual.
*
*		kfile_seddata does not support the "mode" operation as used
*		by ksedfile().  The mode operation is used to how the data
*		should be updated, but it relies on the ability to read
*		as well as write to the destination id (id2).
*
*         Input: id1 - source transport id to copy data from
*                id2 - destination transport id to copy data to
*		 regex   - whether the search and replacement patterns are
*			   regular expressions or not
*                kvalist - list of search patterns and replacement
*                          patterns to be used to modify the data as
*                          it is being copied.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young and Steve Jorgensen
*          Date: Feb 20, 1994
*   Declaration: int kfile_seddata(
*		 !   int id1,
*		 !   int id2,
*		 !   int regex,
*		 !   kvalist)
*************************************************************/
/* ARGSUSED */
int kfile_seddata(
   int id1,
   int id2,
   int regex,
   kvalist)
{
	kva_list valist;
	char *data, *temp, *istr, *ostr;


	kva_start(valist, regex);
	if ((data = (char *) kfile_readdata(id1, NULL, NULL)) == NULL)
	   return(FALSE);

	while ((istr = kva_arg(valist, char *)) != NULL)
	{
	   ostr = kva_arg(valist, char *);
	   if (regex == FALSE)
	      temp = kstring_replace(data, istr, ostr, NULL);
	   else
	      temp = kregex_replace(data, istr, ostr, NULL);

	   kfree(data); data = temp;
	}

	if (kfile_writedata(id2, data, kstrlen(data)) == -1)
	{
	   kfree(data);
	   return(FALSE);
	}
	kfree(data);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kfile_copydata - copy the contents of a transport id to
*				  another transport id
*       Purpose: This routine copies the contents of a input transport
*		 descriptor to an output transport descriptor.  These
*		 descriptors are specified by the Khoros transport
*		 mechanism.  An optional "num_copy" can be used to
*		 specify the actual number of bytes to be copied.  If
*		 "num_copy" is 0 then all the data from input is written
*		 to the output.
*         Input: id1	   - the input khoros transport descriptor
*		 id2	   - the output khoros transport descriptor
*		 num_copy  - if not 0 the number of bytes to be copied.
*       Returns: the number of bytes copied or -1 upon failure
*    Written By: Mark Young  
*          Date: Jan 20, 1993
*  Side Effects: the data written to the output descriptor will not be flushed
*		 until a kfflush() or kclose() is performed.
*************************************************************/

int kfile_copydata(
   int   id1,
   int   id2,
   int   num_copy)
{
	char  data[ALLOCSIZE];
	int   num = 0, num_bytes;


	do
	{
	    if (num_copy != 0)
	       num_bytes = kmin(num_copy, ALLOCSIZE);
	    else
	       num_bytes = ALLOCSIZE;

	    /*
	     *  Read from id1 at most num_bytes
	     */
	    if ((num_bytes = kread(id1, data, num_bytes)) == -1)
	       return(-1);

	    /*
	     *  Write to id2 num_bytes...
	     */
	    if (kwrite(id2, data, num_bytes) != num_bytes)
	       return(-1);

	    if (num_bytes > 0)
	    {
	       num += num_bytes;
	       if (num_copy != 0) num_copy -= num_bytes;
	    }
	} while (num_bytes > 0);

	return(num);
}


/************************************************************
*
*  Routine Name: kfile_comparedata - compare the contents of a transport id to
*				     another transport id
*       Purpose: This routine compare the contents of a input transport
*		 descriptor to an output transport descriptor.  These
*		 descriptors are specified by the Khoros transport
*		 mechanism.  An optional "num_compare" can be used to
*		 specify the actual number of bytes to be compared.  If
*		 "num_compare" is 0 then all the data from the two transports
*		 are compared.
*         Input: id1 - the first khoros transport descriptor
*		 id2 - the second khoros transport descriptor
*		 num - if not 0 the number of bytes to be compared.
*        Output: num_compared - the number of bytes actually compared.
*       Returns: returns an integer less than, equal to, or greater than 0,
*		 according as id1 is lexicographically less than, equal to,
*		 or greater than id2.
*    Written By: Mark Young  
*          Date: Jan 20, 1993
*  Side Effects: data is read from id1 and id2, which means if the transports
*		 do not support data permanence seeking maynot work.
*************************************************************/

int kfile_comparedata(
   int   id1,
   int   id2,
   int   num,
   int   *num_compared)
{
	char  data1[ALLOCSIZE], data2[ALLOCSIZE];
	int   i, num_processed = 0, num1, num2, num_bytes;


	do
	{
	    if (num != 0)
	       num_bytes = kmin(num - num_processed, ALLOCSIZE);
	    else
	       num_bytes = ALLOCSIZE;

	    /*
	     *  Read from id1 at most num_bytes
	     */
	    if ((num1 = kread(id1, data1, num_bytes)) == -1)
	    {
	       if (num_compared) *num_compared = num_processed;
	       return(-1);
	    }

	    /*
	     *  Write to id2 num_bytes...
	     */
	    if ((num2 = kread(id2, data2, num_bytes)) == -1)
	    {
	       if (num_compared) *num_compared = num_processed;
	       return(-1);
	    }

	    num_bytes = kmin(num1, num2);
	    for (i = 0; i < num_bytes; i++, num_processed++)
	    {
	       if (data1[i] != data2[i])
	       {
		  if (num_compared) *num_compared = num_processed;
		  return((int) (data1[i] - data2[i]));
	       }
	    }

	    if (num1 != num2)
	    {
	       if (num_compared) *num_compared = num_processed;
	       return(-1);
	    }
	} while (num_bytes > 0);

	if (num_compared) *num_compared = num_processed;
	return(0);
}


/************************************************************
*
*  Routine Name: kfile_filename - return the filename associated with
*				  khoros transport id
*       Purpose: This routine returns the filename associated with
*		 a khoros transport descriptor id.  This is the filename
*		 used in originally creating the transport.
*         Input: id - the transport id from which the filename is returned
*       Returns: The filename associated with the id or NULL upon failure
*    Written By: Mark Young  
*          Date: Jan 20, 1993
*  Side Effects: the filename returned is the internal copy of the
*		 kfile transport.  This means that you should not
*		 modify or free the string.
*************************************************************/

char *kfile_filename(
   int id)
{
	kfile *file;

	if ((file = kfile_checkid(id)) == NULL)
	   return(NULL);

	return(file->path);
}


/************************************************************
*
*  Routine Name: kfile_flags - return the flags associated with transport id
*       Purpose: This routine returns the open flags associated with
*		 a khoros transport descriptor id.  This is the flags
*		 used in originally creating/opening of the transport.
*         Input: id - the transport id from which the flags is returned
*       Returns: The flags associated with the id or -1 upon failure
*    Written By: Mark Young  
*          Date: Apr 27, 1993
*************************************************************/

int kfile_flags(
   int id)
{
	kfile *file;

	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	return(file->flags);
}

/************************************************************
*
*  Routine Name: kfile_setstate - adds a flag to the open transport id state
*       Purpose: This routine is used to set a flag within the transport id
*		 state field.
*         Input: file - the transport id from which to set the flag for
*		 flag - the flag to be set (such as KFILE_TEMP)
*       Returns: returns the updated flags associated with the id or -1 upon
*		 failure
*    Written By: Mark Young  
*          Date: Jul 21, 1994
*************************************************************/

int kfile_setstate(
   kfile *file,
   int flag)
{

/*	STEVE: Why is this commented out?
	if ((file = kfile_checkfile(file)) == NULL)
	   return(-1);
 */
	file->state |= flag;
	return(file->state);
}

/************************************************************
*
*  Routine Name: kfile_clrstate - remove a flag from an open transport id state
*       Purpose: This routine is used to clear a flag within the transport id
*		 state field.
*         Input: file - the transport id from which to clear the flag for
*		 flag - the flag to be clear (such as KFILE_TEMP)
*       Returns: The updated flags associated with the id or -1 upon failure
*    Written By: Mark Young  
*          Date: Jul 21, 1994
*************************************************************/

int kfile_clrstate(
   kfile *file,
   int flag)
{
/*
	if ((file = kfile_checkfile(file)) == NULL)
	   return(-1);
 */
	file->state &= ~flag;
	return(file->state);
}

/************************************************************
*
*  Routine Name: kfile_getstate - return the current internal stream transport
*			       state
*       Purpose: This routine returns the internal state of a khoros stream
*		 transport.  The state is mask of the following defines:
*
*		 ! 	KFILE_READ    -  whether transport is readable
*		 ! 	KFILE_WRITE   -  whether transport is writeable
*		 ! 	KFILE_RDWR    -  whether transport is readable &
*					 writeable
*		 ! 	KFILE_MYBUF   -  whether transport uses user defined
*					 stream buffer
*		 ! 	KFILE_EOF     -  whether transport is at end of file
*		 ! 	KFILE_ERR     -  whether transport is in an error
*		 ! 	KFILE_PERM    -  whether transport has data permanence
*		 ! 	KFILE_OPEN    -  whether transport is currently opened
*		 ! 	KFILE_DUP     -  whether transport is currently dupped
*         Input: file - the kfile transport to be returned
*       Returns: The type associated with the kfile transport on success,
*		 or NULL upon failure
*    Written By: Mark Young  
*          Date: Apr 27, 1993
*  Side Effects: the type field returned is the internal copy of the
*		 kfile transport.  This means that you should not
*		 modify or free the string.
*************************************************************/

int kfile_getstate(
   kfile *file)
{
	if ((file = kfile_checkfile(file)) == NULL)
	   return(FALSE);

	return(file->state);
}

/************************************************************
*
*  Routine Name: kfile_type - return the type flag field used when opening
*			      the transport with kfopen()
*
*       Purpose: This routine returns the type field associated with
*		 a khoros transport.  This is the type fields used
*		 originally when opening the transport with kfopen().
*
*         Input: file - the kfile transport returned earlier by kfopen()
*       Returns: The type associated with the kfile transport on success,
*		 or NULL upon failure
*    Written By: Mark Young  
*          Date: Apr 27, 1993
*  Side Effects: the type field returned is the internal copy of the
*		 kfile transport.  This means that you should not
*		 modify or free the string.
*************************************************************/

char *kfile_type(
   kfile *file)
{
	if ((file = kfile_checkfile(file)) == NULL)
	   return(NULL);

	return(file->type);
}


/************************************************************
*
*  Routine Name: kfile_mode - return the mode associated with transport id
*
*       Purpose: This routine returns the mode associated with
*                a khoros transport descriptor id.  This is the mode
*                used in originally creating/opening of the transport.
*
*         Input: id - the transport id from which the flags is returned
*       Returns: The mode associated with the id or -1 upon failure
*    Written By: Mark Young
*          Date: Nov 15, 1993
*************************************************************/

int kfile_mode(
   int id)
{
	kfile *file;

	if ((file = kfile_checkid(id)) == NULL)
	   return(-1);

	return(file->mode);
}

/**************************************************************
*  
*  Routine Name: ktransport_list - get the list of supported transports
*       Purpose: This function creates an array of available transports.
*		 Boolean inputs allow the calling routine to selectively mask
*		 in or out local, stream, and permanent transports.
*         Input: local       - a boolean indicating that local transports
*		  	       should be listed
*		 stream      - a boolean indicating that stream transports
*			       should be included in the list
*                permanence  - a boolean indicating that data permanent
*		  	       transports should be included in the list
*        Output: num_entries - the number of entries returned
*	Returns: A pointer to an array of the available transports, or NULL
*		 on error.
*    Written By:  Mark Young
*          Date:  Jul 08, 1992
* Modifications: Converted from in Khoros 1.0 (MY)
*   Declaration: char **ktransport_list(
*		 !    int  local,
*		 !    int  permanence,
*		 !    int  stream,
*		 !    int *num_entries)
**************************************************************/
/* ARGSUSED */
char **ktransport_list(
   int local,
   int permanence,
   int stream,
   int *num_entries)
{
	int	i, j;
	char	**list = NULL, temp[KLENGTH];


	for (i = 0, j = 0; i < num_transport_entries; i++)
	{
	   if ((transport_entries[i]->data_permanence && permanence == TRUE) ||
	       (!transport_entries[i]->data_permanence && stream == TRUE))
	    {
	       if (transport_entries[i]->label == NULL)
		  ksprintf(temp, "%4s", transport_entries[i]->identifier);
	       else
		  ksprintf(temp, "%4s (%s)", transport_entries[i]->identifier,
			transport_entries[i]->label);

	       list = karray_add(list, kstrdup(temp), j++);
	    }
	}
	*num_entries = j;
	return(list);
}

/**************************************************************
*  
*  Routine Name: ktransport_add - add a new transport to the list of transports
*       Purpose: This function is used to add a new transport to the compiled
*		 list of transports.  The transport is dynamically added to
*		 during runtime, rather than during compile time.
*         Input: transport - the transport to be added
*	Returns: TRUE if the transport was successfully added, FALSE otherwise
*    Written By:  Mark Young
*          Date:  Jan 14, 1994
**************************************************************/

int ktransport_add(
   TransportInformation *transport)
{
	if (transport && karray_locate((char **) transport_entries,
		(kaddr) transport, num_transport_entries) == -1)
	{
	   transport_entries = (TransportInformation **) karray_add(
			(char **) transport_entries, (kaddr) transport,
			num_transport_entries++);
	}
	else
	   return(FALSE);

	return(TRUE);
}

/**************************************************************
*  
*  Routine Name: ktransport_delete - delete a transport from the list of
*				     all transports
*       Purpose: This function is used to delete a new transport from the
*		 compiled list of transports.  The transport is dynamically
*		 deleted from the list, rather than during compile time.
*         Input: transport - the transport to be deleted
*	Returns: TRUE if the transport was successfully deleted, FALSE otherwise
*    Written By:  Mark Young
*          Date:  Jan 14, 1994
**************************************************************/

int ktransport_delete(
   TransportInformation *transport)
{
	if (transport && karray_locate((char **) transport_entries,
		(kaddr) transport, num_transport_entries) != -1)
	{
	   transport_entries = (TransportInformation **) karray_delete(
			(char **) transport_entries, (kaddr) transport,
			num_transport_entries--);
	}
	else
	   return(FALSE);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: kflags_to_type - convert kopen() flags to kfopen() type
*				    parameter
*       Purpose: The kflags_to_type() routine is used to convert the
*		 flags used in the kopen() call into their corresponding
*		 equivalent in the type field of the kfopen() call.
*		 The "flags" parameter is converted and stored into the
*		 "type" parameter.  If the "type" parameter is NULL then
*		 space is malloc'ed and returned.  The following is an
*		 example of how to use kflags_to_type():
*
*		 !   type = kflags_to_type(KOPEN_WRONLY | KOPEN_CREAT | KOPEN_TRUNC, NULL);
*
*		 the input flags are converted and result returned to
*		 type will be "w".
*
*                Here is the table as it translates:
*
*                ! a+  = KOPEN_RDWR   | KOPEN_APPEND | KOPEN_CREAT
*                ! w+  = KOPEN_RDWR   | KOPEN_CREAT  | KOPEN_TRUNC
*                ! r+  = KOPEN_RDWR
*                ! a   = KOPEN_WRONLY | KOPEN_APPEND | KOPEN_CREAT
*                ! w   = KOPEN_WRONLY | KOPEN_CREAT  | KOPEN_TRUNC
*                ! r   = KOPEN_RDONLY
*         Input: flags - kopen flags
*        Output: type  - a string to old the kfopen type field.  If NULL then
*			 space is malloc'ed
*       Returns: type is returned if it is not NULL.  Otherwise the malloc'ed
*		 string is returned.  NULL is returned on an error.
*    Written By: Mark Young
*          Date: Apr 27, 1993
*************************************************************/

char *kflags_to_type(
   int  flags,
   char *type)
{
	char	temp[KLENGTH];


	/*
	 *
	 *   Here it is.  This is what is all about, well with a few liberties
	 *   thrown in...
	 *   ----------------------------------------------------------------
	 *   a+  = KOPEN_RDWR KOPEN_APPEND KOPEN_CREAT
	 *   w+  = KOPEN_RDWR KOPEN_CREAT KOPEN_TRUNC 
	 *   r+  = KOPEN_RDWR 
	 *   a   = KOPEN_WRONLY KOPEN_APPEND KOPEN_CREAT
	 *   w   = KOPEN_WRONLY KOPEN_CREAT KOPEN_TRUNC
	 *   r   = KOPEN_RDONLY
	 * 
	 */
	if (flags & KOPEN_RDWR)
	{
	   if (flags & KOPEN_APPEND)
	      kstrcpy(temp, "a+");
	   else if (flags & KOPEN_TRUNC || flags & KOPEN_CREAT)
	      kstrcpy(temp, "w+");
	   else
	      kstrcpy(temp, "r+");
	}
	else if (flags & KOPEN_APPEND)
	   kstrcpy(temp, "a");
	else if (flags & KOPEN_WRONLY)
	   kstrcpy(temp, "w");
	else
	   kstrcpy(temp, "r");


	if (type == NULL)
	   type = kstrdup(temp);
	else
	   kstrcpy(type, temp);

	return(type);
}


/************************************************************
*
*  Routine Name: ktype_to_flags - convert kfopen() type to kopen() flags
*
*       Purpose: The ktype_to_flags() routine is used to convert the
*		 type field used in the kfopen() call into their corresponding
*		 equivalent in the flags field of the kopen() call.
*		 The "type" parameter is converted and returned in
*		 kopen flag form.  The following is an example of how
*		 to use ktype_to_flags():
*
*			flags = ktype_to_flags("w");
*
*		 the input flags are converted and result returned to
*		 type will be KOPEN_WRONLY|KOPEN_CREAT|KOPEN_TRUNC.
*
*                Here is the table as it translates:
*
*                ! a+  = KOPEN_RDWR   | KOPEN_APPEND | KOPEN_CREAT
*                ! w+  = KOPEN_RDWR   | KOPEN_CREAT  | KOPEN_TRUNC
*                ! r+  = KOPEN_RDWR
*                ! a   = KOPEN_WRONLY | KOPEN_APPEND | KOPEN_CREAT
*                ! w   = KOPEN_WRONLY | KOPEN_CREAT  | KOPEN_TRUNC
*                ! r   = KOPEN_RDONLY
*         Input: type - the kfopen string field.
*       Returns: the kopen flags on success.  If type is NULL or an invalid
*		 type, a -1 will be returned.
*    Written By: Mark Young
*          Date: Apr 27, 1993
*************************************************************/

int ktype_to_flags(
   char *type)
{
	int	flags = 0;

	if (!type)
	   return(0);

	switch (type[0])
	{
	   case 'r':
	   case 'R':
		flags = (((int) type[1] == '+') ? KOPEN_RDWR : KOPEN_RDONLY);

		if ((int) type[0] == 'R')
		   flags |= KOPEN_NONBLOCK;

		/*
	 	 *  Extended 
	 	 */
		break;

	   case 'w':
	   case 'W':
	   case 'a':
	   case 'A':
		flags = ((type[1] == '+') ? KOPEN_RDWR : KOPEN_WRONLY) | KOPEN_CREAT;
		if ((int) type[0] == 'W' || (int) type[0] == 'A')
		   flags |= KOPEN_NONBLOCK;

		if ((int) type[0] == 'a' || (int) type[0] == 'A')
		   flags |= KOPEN_APPEND;
		else
		   flags |= KOPEN_TRUNC;
	 	break;

	    default:
		return(0);
	}
	return(flags);
}
