/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            File Utilities
   >>>>
   >>>>  Private:
   >>>>             None
   >>>>   Public:
   >>>>             keditfile()
   >>>>             kprintfile()
   >>>>             kreadfile()
   >>>>             kwritefile()
   >>>>             ksedfile()
   >>>>             kcopyfile()
   >>>>             kcomparefile()
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


#define DEFAULT_EDITOR    "vi"
#define ALLOCSIZE 10000


/************************************************************
*
*  Routine Name: keditfile - start up an edit program to edit an input file
*
*       Purpose: This routine executes a system call to the system editor.
*		 It uses the environment variable KHOROS_EDITOR to determine
*		 which system editor to use.  If KEDITOR is not set then
*		 EDITOR will be checked.  If neither of these environment
*		 variables are set then the routine will use 'vi' as the
*		 default.
*
*		 Also if the DISPLAY environment variable is set then the
*		 editor will be executed within an xterm.  The xterm and
*		 editor will be started in background and keditfile will
*		 return immediately.  If not then the editor will be executed
*		 locally and will not return until the process finishes
*		 executing.
*
*		 Also, options can be specified to keditfile.  The following
*		 are a list of options:
*
*		 !      KEDITOR_GEOMETRY   - the geometry of the editor
*		 !      KEDITOR_TITLE      - the editor title
*		 !      KEDITOR_FOREGROUND - the editor foreground color
*		 !      KEDITOR_BACKGROUND - the editor background color
*		 !      KEDITOR_ICON       - the editor icon file
*		 !      KEDITOR_PID        - the pid of the editor
*		 !      KEDITOR_CHDIR      - change to file's directory
*         Input: filename - the filename to be edited
*		 spawn    - to spawn the process into background or not
*		 kvalist  - the NULL terminated list of editfile options
*    Written By: Mark Young, Steve Kubica, Neil Bowers, and Danielle Argiro
*          Date: Apr 13, 1994
* Declaration: int keditfile(
*	       !   char *filename,
*	       !   int  spawn,
*	       !   kvalist)
*************************************************************/

typedef struct {
	char *name;
	char *spec;
	char *defvalue;
	char *value;
} EditOptions;

EditOptions edit_options[] = {
  { KEDITOR_GEOMETRY,   "%g",   "80x48", NULL },
  { KEDITOR_TITLE,	"%t",   NULL,    NULL },
  { KEDITOR_FOREGROUND, "%fg",  "black", NULL },
  { KEDITOR_BACKGROUND, "%bg",  "white", NULL },
  { KEDITOR_ICON,	"%icon", NULL,   NULL }
};

int keditfile(
   char *filename,
   int  spawn,
   kvalist)
{
	kva_list list;
	int	 i, tmp_pid = -1, *pid = NULL;
	int      do_chdir = 0;
	char     temp[KLENGTH], buffer[KLENGTH], *editor, *name;
	char     file_dir[KLENGTH];
	char     file_base[KLENGTH];


	kva_start(list, spawn);

	/*
	 *  Check to see if the user has specified a preferred editor.  If not
	 *  then default to using the default DEFAULT_EDITOR.
	 */
	if (kgetenv("DISPLAY") == NULL			 ||
	    ((editor = kgetenv("KHOROS_EDITOR")) == NULL &&
	     (editor = kgetenv("KEDITOR")) == NULL))
	{
	   if ((editor = kgetenv("EDITOR")) == NULL &&
	       (editor = kgetenv("VISUAL")) == NULL)
	      editor = DEFAULT_EDITOR;

	   if (kgetenv("DISPLAY"))
	      ksprintf(buffer, "xterm -geometry %%g -e %s %%f", editor);
	   else
	      ksprintf(buffer, "%s %%f", editor);
	}
	else
	   kstrcpy(buffer, editor);

	for (i = 0; i < knumber(edit_options); i++)
	   edit_options[i].value = edit_options[i].defvalue;

	while ((name = kva_arg(list, kstring)) != NULL)
	{
	   for (i = 0; i < knumber(edit_options); i++)
	   {
	      if (kstrcmp(name, edit_options[i].name) == 0)
	      {
		 edit_options[i].value = kva_arg(list, kstring);
		 break;
	      }
	   }

	   /*
	    *  Check to see if we were able to find "name" in the options
	    *  table.  If not then print a warning, and then
	    */
	   if (kstrcmp(KEDITOR_PID, name) == 0)
	   {
	      pid = kva_arg(list, int *);
	   }
	   else if (kstrcmp(KEDITOR_CHDIR, name) == 0)
	   {
	      do_chdir = kva_arg(list, int);
	   }
	   else if (i == knumber(edit_options))
	   {
	      errno = KINVALID_PARAMETER;
	      kwarn("kutils", "keditfile",
		    "Unknown option '%s' passed to edit file '%s'",
		    name, filename);
	      (void) kva_arg(list, kstring);
	      return(FALSE);
	   }
	}

	for (i = 0; i < knumber(edit_options); i++)
	   kstring_replace(buffer, edit_options[i].spec, edit_options[i].value,
			   buffer);

	/* do we want to move into the file's directory? */
	if (do_chdir)
	{
	   if (kdirname(filename, file_dir) == NULL)
	   {
	      kerror("kutils", "keditfile",
		     "Unable to determine directory for %s.\n", filename);
	      return FALSE;
	   }
	   if (kbasename(filename, file_base) == NULL)
	   {
	      kerror("kutils", "keditfile",
		     "Unable to determine basename for %s.\n", filename);
	      return FALSE;
	   }
	   ksprintf(temp, "cd %s ; %s", file_dir, buffer);
	   kstrcpy(buffer, temp);
	}

	kfullpath(filename,NULL,temp);
	kstring_replace(temp, " ", "\\\\ ", temp);
	kstring_replace(buffer, "%f", (do_chdir ? file_base : temp), buffer);

	if (spawn)
	   tmp_pid = kspawn(buffer);
	else
	   (void) ksystem(buffer);

	if (pid) *pid = tmp_pid;
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kprintfile - print a file to a printer
*       Purpose: This routine executes a system call to the system printer.
*		 It uses the environment variable PRINTER to determine
*		 which printer to use.  This can be overriden using the
*		 KPRINTER_NAME option below.  If neither of these are
*		 specified then the routine will use 'default' printer
*		 by calling print command with no print option.
*
*		 Also, options can be specified to kprintfile.  The following
*		 are a list of options:
*
*			KPRINTER_NAME   - the printer name to print the file to
*
*         Input: filename - the filename to be printed
*	Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young
*          Date: Apr 21, 1994
*   Declaration: int kprintfile(
*		 !    char *filename,
*		 !    kvalist)
*************************************************************/

typedef struct {
	char *name;
	char *spec;
	char *defvalue;
	char *value;
} PrintOptions;

PrintOptions *print_options = NULL;

int kprintfile(
   char *filename,
   kvalist)
{
	int	 i;
	kva_list list;
	char     temp[KLENGTH], buffer[KLENGTH], *name;


	kva_start(list, filename);

	for (i = 0; i < knumber(print_options); i++)
	   print_options[i].value = print_options[i].defvalue;

	while ((name = kva_arg(list, kstring)) != NULL)
	{
	   for (i = 0; i < knumber(print_options); i++)
	   {
	      if (kstrcmp(name, print_options[i].name) == 0)
	      {
		 print_options[i].value = kva_arg(list, kstring);
		 break;
	      }
	   }

	   if (i == knumber(print_options))
	   {
	      errno = KINVALID_PARAMETER;
	      kwarn("kutils", "kprintfile", "Unknown option '%s' passed to \
print file '%s'", name, filename);
	      (void) kva_arg(list, kstring);
	      return(FALSE);
	   }
	}

	for (i = 0; i < knumber(print_options); i++)
	   kstring_replace(buffer, print_options[i].spec,
			   print_options[i].value, buffer);

	kfullpath(filename,NULL,temp);
	kstring_replace(buffer, "%f", temp, buffer);
	(void) ksystem(buffer);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kreadfile - read the contents of a file into a data array
*       Purpose: This routine opens the specified and using the khoros
*		 transport mechanisms reads the contents of the file
*		 into an array.  An optional "num_read" can be used to
*		 find out the actual number of bytes read.  If "num_read"
*		 is NULL then the parameter is ignored.
*         Input: filename - the filename which contains the data to be read
*        Output: num_read - if not NULL the number of bytes read is returned
*       Returns: returns a pointer to the contents of the file or NULL upon
*		 failure
*    Written By: Mark Young  
*          Date: Jan 15, 1993
*  Side Effects: the data returned is allocated and needs to freed using
*		 kfree().
*************************************************************/
kaddr kreadfile(
   char *filename,
   int  *num_read)
{
	int   flags, fid;
	kaddr data = NULL;


	/*
	 *  Better initialize the number read...
	 */
	if (num_read)
	   *num_read = 0;

	flags = KOPEN_RDONLY | KOPEN_LOCK;
	if ((fid = kopen(filename, flags, 0666)) == -1)
	   return(NULL);

	/*
	 *  Call the kfile_readdata() to do the actual reading of data
	 */
	data = kfile_readdata(fid, data, num_read);
	(void) kclose(fid);
	return(data);
}

/************************************************************
*
*  Routine Name: kwritefile - write the contents of the data to a file
*       Purpose: This routine opens the specified and using the khoros
*		 transport mechanisms writes the contents of the data
*		 array to the specified file.  An optional "num_write" can
*		 be used to specify the actual number of bytes to be written.
*		 If "num_write" is 0 then the number of bytes to be written
*		 is computed from the data by using kstrlen().
*         Input: filename  - the filename which the data will be written to
*		 data      - the data array (or string) to be written
*		 num_write - if not 0 the number of bytes to be written.
*       Returns: returns the number of actual bytes written or -1 upon
*		 failure.
*    Written By: Mark Young  
*          Date: Jan 15, 1993
*************************************************************/

int kwritefile(
   char  *filename,
   kaddr data,
   int   num_write)
{
	int   fid, flags;


	flags = KOPEN_WRONLY | KOPEN_CREAT | KOPEN_TRUNC | KOPEN_LOCK;
	if ((fid = kopen(filename, flags, 0666)) == -1)
	   return(-1);

	/*
	 *  Call the kfile_writedata() to do the actual writing of data
	 */
	num_write = kfile_writedata(fid, data, num_write);
	(void) kclose(fid);
	return(num_write);
}

/************************************************************
*
*  Routine Name: ksedfile - khoros string edit a file
*
*       Purpose: This routine takes two filenames, and copies the data
*		 from the source file to the destination file.  It also
*		 uses a variable argument list of pairs of strings indication
*		 a search pattern and a replacement pattern.  This list will
*		 be used to replace strings in template files with their
*		 "correct" values.  Note, the variable argument list MUST
*		 be terminated with a NULL in one of the search pattern slots
*
*                For example, the following is a correct termination:
*
*               ! ksedfile(ifilename, ofilename, FALSE, KFILE_UPDATE, NULL, "t1", "t2", NULL);
*
*               And this is an incorrect termination:
*
*               ! ksedfile(ifilename, ofilename, FALSE, KFILE_UPDATE, NULL, "t1", NULL);
*
*               The reason the NULL must be on the search pattern, is
*               so that NULL can be sent in as the replacement string.
*               The replacement mechanism is kstring_replace, and
*               its documentation explains restrictions on search strings
*               and replace strings.
*
*		The regex parameter is used to indicate whether the search
*		and replacement parameters are specified in regular expression
*		form, or are just simply string replacement patterns.  For
*		more information about regular expressions please see the
*		khoros string parsing section of the Khoros Programmers Manual.
*
*		The mode parameter dictates how ksedfile should perform
*		the string editing.  The first mode is to query what would
*		happen if string edition would take place.  Whether any
*		difference exists in which to update the destination.  The
*		second is to update the destination, only if a difference
*		exists.  The final mode is to overwrite the destination
*		irregardless if a difference exists or not.  The mode
*		parameter can be one of:
*
*		! KFILE_QUERY  - returns whether an update would take place
*		! KFILE_UPDATE - updates the destination only if different
*		! KFILE_OVERWRITE - updates the destination irregardless if
*				    different
*
*		if KFILE_QUERY is specified then the "status" parameter
*		will retain whether or not the output file would be updated.
*		If KFILE_UPDATE is used then the "status" parameter is will
*		retain whether to output file was actually changed or not.
*		If KFILE_OVERWRITE is specified then the "status" parameter
*		will always be TRUE, unless an error is encountered and
*		ksedfile() returns FALSE.
*
*         Input: ifilename - source filename to copy data from
*                ofilename - destination filename to copy data to
*		 regex     - whether the search and replacement patterns are
*			     regular expressions or not
*		 mode      - the mode in which to update the destination file
*                kvalist   - list of search patterns and replacement
*                            patterns to be used to modify the data as
*                            it is being copied.
*        Output: status    - the status of whether the destination file was
*			     or could be changed, set according to the specified
*			     mode.
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Mark Young and Steve Jorgensen
*          Date: Aug 30, 1993
*************************************************************/

int ksedfile(
   char *ifilename,
   char *ofilename,
   int  regex,
   int  mode,
   int  *status,
   kvalist)
{
	kva_list valist;
	int      different;
	char     *idata, *odata, *temp, *istr, *ostr;


	kva_start(valist, status);
	if (status) *status = FALSE;
	if ((idata = (char *) kreadfile(ifilename, NULL)) == NULL)
	   return(FALSE);

	while ((istr = kva_arg(valist, char *)) != NULL)
	{
	   ostr = kva_arg(valist, char *);
	   if (regex == FALSE)
	      temp = kstring_replace(idata, istr, ostr, NULL);
	   else
	      temp = kregex_replace(idata, istr, ostr, NULL);

	   kfree(idata); idata = temp;
	}

	if (mode == KFILE_QUERY || mode == KFILE_UPDATE)
	{
	   odata = (char *) kreadfile(ofilename, NULL);
	   different = kstrcmp(idata, odata) != 0;
	}
	else
	{
	   odata = NULL; different = TRUE;
	}

	if (different == TRUE)
	{
	   if (mode != KFILE_QUERY &&
	       kwritefile(ofilename, idata, kstrlen(idata)) == -1)
	   {
	      kfree(idata); kfree(odata);
	      return(FALSE);
	   }
	   if (status) *status = TRUE;
	}
	kfree(idata); kfree(odata);
	return(TRUE);
}

/************************************************************
*
*  Routine Name: kcopyfile - copy the contents of one filename to another
*       Purpose: This routine copies the contents of a input filename
*		 to an output filename.  The contents of the files are
*		 copied using the Khoros transport mechanism.
*         Input: ifilename  - the input file to be copied
*		 ofilename  - the output file to be copied
*       Returns: the number of actual bytes written or -1 upon failure.
*    Written By: Mark Young
*          Date: Jan 15, 1993
*************************************************************/

int kcopyfile(
   char *ifilename,
   char *ofilename)
{
	int   flags, ifid, ofid, num_copied;


	flags = KOPEN_RDONLY | KOPEN_LOCK;
	if ((ifid = kopen(ifilename, flags, 0666)) == -1)
	   return(-1);

	flags = KOPEN_WRONLY | KOPEN_CREAT | KOPEN_TRUNC | KOPEN_LOCK;
	if ((ofid = kopen(ofilename, flags, 0666)) == -1)
	   return(-1);

	/*
	 *  Call the kfile_copydata() to do the actual copying of data
	 */
	num_copied = kfile_copydata(ifid, ofid, 0);
	(void) kclose(ifid);
	(void) kclose(ofid);
	return(num_copied);
}

/************************************************************
*
*  Routine Name: kcomparefile - compare the contents of one filename to another
*       Purpose: This routine compares the contents of the first filename
*		 to that of second filename.  The contents of the files are
*		 compared using memcmp().
*
*         Input: filename1  - the first file to be copied
*		 filename2  - the output file to be copied
*		 num        - the number of bytes to be compared, if 0
*			      then all are compared
*        Output: num_compared - if not NULL then returns the number of bytes
*				actually compared.
*       Returns: An integer less than, equal to, or greater than 0,
*		 according as  is lexicographically less than, equal to,
*		 or greater than id2.
*    Written By: Mark Young
*          Date: Feb 19, 1993
*************************************************************/

int kcomparefile(
   char *filename1,
   char *filename2,
   int  num,
   int  *num_compared)
{
	int   fid1, fid2, flags, status;


	/*
	 *  Better initialize the num compared to 0, just in case...
	 */
	if (num_compared)
	   *num_compared = 0;

	flags = KOPEN_RDONLY | KOPEN_LOCK;
	if ((fid1 = kopen(filename1, flags, 0666)) == -1)
	   return(-1);

	flags = KOPEN_RDONLY | KOPEN_LOCK;
	if ((fid2 = kopen(filename2, flags, 0666)) == -1)
	   return(-1);

	/*
	 *  Call the kfile_comparedata() to do the actual comparison of data
	 */
	status = kfile_comparedata(fid1, fid2, num, num_compared);
	(void) kclose(fid1);
	(void) kclose(fid2);
	return(status);
}
