/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Environment Utilities
   >>>>
   >>>>  Private:
   >>>>             None
   >>>>   Public:
   >>>>             kgetenv()
   >>>>             kputenv()
   >>>>             kremenv()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


/************************************************************
*
*  Routine Name: kgetenv - get an environment variable from the environ list.
*       Purpose: This module is used to get an environment variable
*		 from the environ list.  For example, if the user has
*                their TMPDIR environment variable set to /usr/var/tmp, 
*                and this routine is called with (name = "TMPDIR"), then
*                "/usr/var/tmp" will be returned.
*         Input: name - the name of the environament variable to look for
*        Output: none
*       Returns: the environment string upon success, NULL on failure
*  Restrictions: The string returned is simply an address into the environ
*		 list, so it should \fBNOT\fP be freed, or changed in any way
*		 by the calling routine.
*    Written By: Mark Young  
*          Date: Sep 27, 1992 11:55
*************************************************************/

char *kgetenv(
   char *name)
{
	int  num = 0, len = kstrlen(name);
	char *string, **envp = kprog_get_envp();


	while (envp[num] != NULL)
	{
	   string = envp[num];
	   if (kstrncmp(name, string, len) == 0 && (int) string[len] == '=')
	   {
	      string = &string[len+1];
	      return(string);
	   }
	   num++;
	}
	return(NULL);
}

/************************************************************
*
*  Routine Name: kputenv - put an environment variable into the environ list.
*       Purpose: This module is used to put an environment variable
*		 in the environ list.  If "PUTENV" is defined then
*		 we call putenv() otherwise we do our own putenv.  The
*		 only difference is that kputenv() will malloc space
*		 for the incoming environment variable.
*         Input: name - string to be added into putenv
*       Returns: (0) on success, (-1) otherwise
*    Written By: Mark Young  
*          Date: Jul 03, 1992 13:58
*  Side Effects: the input "name" is copied into the environment list
* Modifications: Converted from vputenv in Khoros 1.0 (SJ)
*************************************************************/

int kputenv(
   char *name)
{
	int    len, num;
	static int init = FALSE;
	char   *string, **temp = NULL, **envp = kprog_get_envp();


	/*
	 *  If we are changing the environ for the first time then we need
	 *  copy the current environ into an array of strings kallocated by
	 *  kmalloc.
	 */
	if (init == FALSE && envp != NULL)
	{
	   init = TRUE;

	   num = 0;
	   while (envp[num] != NULL)
	   {
	      if ((temp = karray_add(temp, kstrdup(envp[num]), num)) == NULL)
		 return(-1);
	      num++;
	   }
	   envp = karray_add(temp, NULL, num);
	   kprog_set_envp(envp);
	}

	name = kstrdup(name);
	if ((string = kstrchr(name, '=')) != NULL)
	   len = string - name;
	else
	   len = kstrlen(name);

	/*
	 *  Search to see if this is an existing environment variable.  If
	 *  so then replace the existing one and return.  Otherwise add it
	 *  to the environment variable list.
	 */
	num = 0;
	while (envp[num] != NULL)
	{
	   string = envp[num];
	   if (kstrncmp(name, string, len) == 0 && (int) string[len] == '=')
	   {
	      if (init) kfree(string);
	      envp[num] = name;
	      return(0);
	   }
	   num++;
	}
	envp[num++] = name;
	envp = (char **) karray_add(envp, NULL, num);
	kprog_set_envp(envp);
	return(0);
}


/************************************************************
*
*  Routine Name: kremenv - remove an environment variable from the environment
*       Purpose: This module is used to remove an environment variable
*		 from the environ list.  Since there does not seem
*		 to be a standard routine for this on many machines, we
*		 wrote our own.
*         Input: name - a string containing the environment variable to remove.
*       Returns: (0) on success, (-1) otherwise
*  Restrictions: This routine will not work if the environ list is
*		 not stored as an array of strings.
*    Written By: Mark Young
*          Date: Jul 03, 1992 14:04
*  Side Effects: Does not kfree string associated with environment variable
*		 deleted, since we cannot know how it was created.
* Modifications: Converted from vremenv in Khoros 1.0 (SJ)
*************************************************************/

int kremenv(
   char *name)
{
	int    len, num = 0;
	char   *string, **temp, **envp = kprog_get_envp();


	if (name == NULL)
	   return(-1);

	len = kstrlen(name);
	while (envp[num] != NULL)
	{
	   string = envp[num];
	   if (kstrncmp(name, string, len) == 0 && (int) string[len] == '=')
	   {
	      while (envp[num] != NULL) num++;
	      temp = (char **) karray_delete(envp, string, num);
	      kprog_set_envp(temp);
	      kfree(string);
	      return(0);
	   }
	   num++;
	}
	return(-1);
}
