/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Socket Transport Drivers
   >>>>
   >>>>  Private:
   >>>>			socket_tempnam
   >>>>			socket_open
   >>>>			socket_close
   >>>>			socket_read
   >>>>			socket_write
   >>>>			socket_access
   >>>>			socket_unlink
   >>>>			socket_lock
   >>>>			socket_descriptors
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include <machine/kmachine.h>
#if KOPSYS_LOCAL == KOPSYS_SOLARIS || KOPSYS_LOCAL == KOPSYS_UNICOS
#undef _POSIX_SOURCE
#endif

#include "internals.h"


#if !defined(KSOCKET_DEF)
TransportInformation socket_transport[] = {NULL};
#else

/* Berkeley socket (unix/local) transport */
#include <sys/socket.h>
#include <sys/un.h>

static int socket_tempnam     PROTO((char *, char *, char *));
static int socket_open	      PROTO((char *, int, int, kfile *));
static int socket_close	      PROTO((kfile *));
static int socket_read	      PROTO((kfile *, char *, int));
static int socket_write	      PROTO((kfile *, char *, int));
static int socket_access      PROTO((char *, char *, int));
static int socket_unlink      PROTO((char *, char *));
static int socket_lock	      PROTO((kfile *, int));
static int socket_descriptors PROTO((kfile *, int *, int *));

TransportInformation socket_transport[] =
{
    {
	"Standard Socket (AF_UNIX)",
	"socket",
	FALSE,
	socket_tempnam,
	socket_open,
	socket_close,
	socket_read,
	socket_write,
	NULL,
	socket_access,
	socket_unlink,
	socket_lock,
	socket_descriptors,
    }
};

/*
 *  Internal resource structures for "Standard Socket (AF_UNIX)" transport
 */
typedef struct
{
	int	socket;
} ResourceStruct;



/*-------------------------------------------------------------------*
|  
|		  Private Socket Routines
|  
--------------------------------------------------------------------*/

/*-------------------------------------------------------------
|  
|  Routine Name: socket_connect
|  
|       Purpose: This function opens a "socket" connection.  It is an
|	         internal driver to open a socket, which is called by
|	         the the internal socket routines such as socket_open().
|  
|		  	"socket=XXXXXXXXXX"
|  
|         Input:  file - the kfile structure describing the transport
|		  path - the socket path
|        Output:  returns -1 or the socket id
|   Called From:  internal routine called from kopen()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_connect(
   kfile *file,
   char  *path)
{
	char	name[KLENGTH];
	struct	sockaddr_un server;
	int     status, sock, len, nsock, reuse;


	/*
	 *  Create either a local UNIX or TCP connection based socket
	 */
	if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
	{
	   kinfo(KSYSLIB,"socket_connect:  Creation failure for socket '%s'",
			path);
	   return(-1);
	}

	reuse = TRUE;
	status = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &reuse,
			sizeof(reuse));
	if (status < 0)
	{
	   kinfo(KSYSLIB,"socket_connect:  Cannot set option (REUSEADDR) for \
socket '%s'", path);
	}

	/*
	 *  Get the path name to connet to, the socket name component of
	 *  the path.
	 */
	if (ksscanf(path,"socket=%s", name) != 1)
	   kstrcpy(name, path);

	/*
	 *  Assign our server's address and then connect to the returned
	 *  socket "server_sock".
	 */
	kmemset((char *) &server, 0, sizeof(server));
	server.sun_family = AF_UNIX;
	kstrcpy(server.sun_path, name);

#if KOPSYS_LOCAL == KOPSYS_UNICOS
	len = kstrlen(server.sun_path) + 4;
#else
	len = kstrlen(server.sun_path) + sizeof(server.sun_family);
#endif

	if (kfile_iswrite(file))
	{
	   if (bind(sock, (struct sockaddr *) &server, len) == -1)
	   {
	      close(sock);
	      kinfo(KSYSLIB,"socket_connect:  Bind failure for socket '%s'",
			path);
	      return(-1);
	   }
	   listen(sock, 5);

	   if ((nsock = accept(sock, (struct sockaddr *) &server, &len)) == -1)
	   {
	      close(sock);
	      kinfo(KSYSLIB,"socket_connect:  Accept failure for socket '%s'",
			path);
	      return(-1);
	   }
	   close(sock);
	   sock = nsock;
	}
	else
	{
	   if (connect(sock, (struct sockaddr *) &server, len) == -1)
	   {
	      close(sock);
	      kinfo(KSYSLIB,"socket_connect:  Connect failure for socket '%s'",
			path);
	      return(-1);
	   }
	}
	return(sock);
}


/*-------------------------------------------------------------------*
|  
|		  Socket Routines
|  
--------------------------------------------------------------------*/

/*-------------------------------------------------------------
|  
|  Routine Name: socket_tempnam
|  
|       Purpose: This function initializes a tempnam for a "socket"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"socket=XXXXXXXXXX"
|  
|         Input:  identifier - the transport identifier
|		  template   - the filename template to be modeled
|        Output:  result     - the newly created transport name
|   Called From:  internal routine called from ktempnam()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_tempnam(
   char *identifier,
   char *template,
   char *result)
{
	int	sock, len;
	struct	sockaddr_un server;
	char	*name, filename[KLENGTH];


	/*
	 *  Make a tempnam from the template
	 */
	(void) kstrcpy(filename, template);
	if ((name = ktmpnam(filename)) == NULL)
	   return(-1);

	/*
	 *  Create either a local UNIX or TCP connection based socket
	 */
	if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
	{
	   kinfo(KSYSLIB, "socket_tempnam: Unable to create temporary socket \
'%s'", name);
	   return(-1);
	}

	/*
	 *  Assign our server's address and then connect to the returned
	 *  socket "server_sock".
	 */
	kmemset((char *) &server, 0, sizeof(server));
	server.sun_family = AF_UNIX;
	kstrcpy(server.sun_path, name);

#if KOPSYS_LOCAL == KOPSYS_UNICOS
	len = kstrlen(server.sun_path) + 4;
#else
	len = kstrlen(server.sun_path) + sizeof(server.sun_family);
#endif

	if (bind(sock, (struct sockaddr *) &server, len) == -1)
	{
	   close(sock);
	   kinfo(KSYSLIB, "socket_tempnam:  Bind failure for socket '%s'",
			name);
	   return(-1);
	}
	unlink(name);
	close(sock);

	/*
	 *  Copy the result into the result array
	 */
	if (identifier == NULL)
	   (void) kstrcpy(result, name);
	else
	   (void) sprintf(result,"%s=%s", identifier, name);
	kfree(name);
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_open
|  
|       Purpose: This function opens a "socket" connection.  It is the
|	         internal driver to open a file, which is called by
|	         the user when they call kopen().  kopen() calls when
|	         it gets an open request with a filename of the following
|	         syntax:
|  
|		  	"filename"
|		  	"/path/filename"
|		  	"./path/filename"
|		  	"file=XXXXXXXXXX"
|		  	     ...etc....
|  
|         Input: 
|        Output:  returns 0 or -1 if an error occurs
|   Called From:  internal routine called from kopen()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_open(
   char  *path,
   int   flags,
   int   mode,
   kfile *file)
{
	ResourceStruct *resources;
	int   sock;


	/*
	 *  Connect to the socket
	 */
	if ((sock = socket_connect(file, path)) < 0)
	   return(-1);

	/*
	 *  Create the socket structure.
	 */
	if ((resources = (ResourceStruct *) kcalloc(1,
			sizeof(ResourceStruct))) == NULL)
	{
           kinfo(KSYSLIB, "socket_open: Not enough memory to kcalloc (%d) \
bytes for the khoros socket structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}
	resources->socket = sock;
	file->resources = (kaddr) resources;
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_close
|  
|       Purpose: This function closes a "socket".  It is the internal driver
|	         to close a data socket, which is called by the user
|	         when they call kclose().  kclose() calls the "socket" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the initial kopen().
|  
|         Input:  file - the kfile structure.
|        Output:  returns whether we were able to close the socket
|   Called From:  internal routine called from kclose()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_close(
   kfile *file)
{
	int	     sock, status;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  First, since this is a UNIX socket we must delete the socket
	 *  if it was opened as the server.  Otherwise we won't be able
	 *  to bind to the file again.
	 */
	if (kfile_iswrite(file))
	   kunlink(file->path);

	/*
	 *  Simply call close() since "kfile" is just a regular file
	 *  descriptor.
	 */
	sock = resources->socket;
	status = close(sock);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_read
|  
|       Purpose: This function reads a "socket".  It is the internal driver
|	         to read data from a socket, which is called by the user
|	         when they call kread().  kread() calls the "socket" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to store the data into.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes read from the file
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_read(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int count;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call read() since "kfile" is just a regular file
	 *  descriptor.
	 */
	count = read(resources->socket, ptr, nbytes);
	return(count);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_write
|  
|       Purpose: This function writes to a "socket".  It is the internal driver
|	         to write data from the supplied data array to a socket, which
|	         is called by the user when they call kwrite().  kwrite() calls
|	         the "socket" internal drivers by the nature of the transport,
|	         which is dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to write the data from.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_write(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int count;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call write() since "kfile" is just a regular file
	 *  descriptor.
	 */
	count = write(resources->socket, ptr, nbytes);
	return(count);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_access
|  
|       Purpose: This function is used to do see if a file is accessable.
|	         This is called when the user calls kaccess().  kaccess()
|	         calls the "socket" internal drivers by the nature of the
|	         transport, which is dictated by the ktransport_identifier().
|  
|         Input:  path     - the initial path.
|		  filename - the filename that they wish to access
|		  mode     - the access mode information
|        Output:  returns the result
|   Called From:  internal routine called from kaccess()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_access(
   char *path,
   char *filename,
   int  mode)
{
	int	   status;

	/*
	 *  Simply call access() since "filename" is just a regular filename
	 */
	status = access(filename, mode);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_unlink
|  
|       Purpose: This function initializes a tempnam for a "socket"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"socket=XXXXXXXXXX"
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kunlink()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_unlink(
   char *identifier,
   char *filename)
{
	int	status;


	status = unlink(filename);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_lock
|  
|       Purpose: This function locks a "socket" transport.  Given
|	         a request with a filename of the following syntax:
|  
|		  	"socket=xxxxx"
|  
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kflock()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_lock(
   kfile *file,
   int   operation)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;

	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: socket_descriptors
|  
|       Purpose: This function shouldn't exist, but there is occasion
|	         when the user wants to know the actual file descriptor
|	         associated with the file descriptor.
|  
|         Input: file structure
|        Output:  inum - the input file descriptor
|   Called From:  internal routine called from kdescriptros()
|    Written By:  Mark Young
|          Date:  Jul 08, 1992 16:06
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int socket_descriptors(
   kfile *file,
   int   *inum,
   int   *onum)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	if (inum != NULL) *inum = resources->socket;
	if (onum != NULL) *onum = resources->socket;
	return(0);
}

#endif  /* KSOCKET_DEF */
/* don't add after the endif */
