/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>         UNIX File Transport Routines
   >>>>
   >>>>  Private:
   >>>>			file_tempnam
   >>>>			file_open
   >>>>			file_close
   >>>>			file_read
   >>>>			file_write
   >>>>			file_lseek
   >>>>			file_access
   >>>>			file_unlink
   >>>>			file_lock
   >>>>			file_descriptors
   >>>>
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"	


#if !defined(KFILE_DEF)
#define KFILE_DEF 1
#endif

#if !defined(KFILE_DEF)
TransportInformation file_transport[] = {NULL};
#else

static int file_tempnam	    PROTO((char *, char *, char *));
static int file_open	    PROTO((char *, int, int, kfile *));
static int file_close	    PROTO((kfile *));
static int file_read	    PROTO((kfile *, char *, int));
static int file_write	    PROTO((kfile *, char *, int));
static int file_lseek	    PROTO((kfile *, int, int));
static int file_access	    PROTO((char *, char *, int));
static int file_unlink	    PROTO((char *, char *));
static int file_lock	    PROTO((kfile *, int));
static int file_descriptors PROTO((kfile *, int *, int *));

TransportInformation file_transport[] =
{
    {
	"Standard Unix Files",
	"file",
	TRUE,
	file_tempnam,
	file_open,
	file_close,
	file_read,
	file_write,
	file_lseek,
	file_access,
	file_unlink,
	file_lock,
	file_descriptors,
    }
};

/*
 *  Internal resource structures for "standard file" transport
 */
typedef struct
{
	int fid;
	int used_open;
} ResourceStruct;


/*-------------------------------------------------------------
|  
|  Routine Name: file_tempnam
|  
|       Purpose: This function initializes a tempnam for a "file"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"filename"
|		  	"/path/filename"
|		  	"./path/filename"
|		  	"file=XXXXXXXXXX"
|		  	     ...etc....
|
|         Input:  identifier - the transport identifier
|		  template   - the filename template to be modeled
|        Output:  result     - the newly created transport name
|		  returns -1 or 0 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from ktempnam()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_tempnam(
   char *identifier,
   char *template,
   char *result)
{
	char	*temp, filename[KLENGTH];


	/*
	 *  If the template is "-" or "#" this means that it is kstdin,
	 *  stdout, or kstderr, which means don't make a tempnam simply
	 *  copy this as the output file
	 */
	if (kstrcmp(template, "-") == 0 || kstrcmp(template, "#") == 0)
	{
	   kstrcpy(result, template);
	   return(0);
	}

	/*
	 *  Make a tempnam from the template
	 */
	(void) kstrcpy(filename, template);
	if ((temp = ktmpnam(filename)) == NULL)
	   return(-1);

	/*
	 *  Create the unique file so that no one steals away from us...
	 */
	if (ktouch(temp, 0666) == -1)
	{
	   kinfo(KSYSLIB, "file_tempnam: Unable to create temporary file '%s'",
			temp);
	   return(-1);
	}

	if (identifier == NULL)
	   (void) kstrcpy(result, temp);
	else
	   (void) sprintf(result,"%s=%s", identifier, temp);

	kfree(temp);
	return(0);
}

/*-------------------------------------------------------------
|  
|  Routine Name: file_open
|  
|       Purpose: This function opens a "file" connection.  It is the
|	         internal driver to open a file, which is called by
|	         the user when they call kopen().  kopen() calls when
|	         it gets an open request with a filename of the following
|	         syntax:
|  
|		  	"filename"
|		  	"/path/filename"
|		  	"./path/filename"
|		  	"file=XXXXXXXXXX"
|		  	     ...etc....
|  
|         Input: 
|        Output:  returns 0 or -1 if an error occurs
|   Called From:  internal routine called from kopen()
|  
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_open(
   char  *path,
   int   flags,
   int   mode,
   kfile *file)
{
	ResourceStruct *resources;
	int fid, fid_access, used_open = FALSE;


	if (kstrcmp(path, "-") == 0)
	{
	   fid_access = flags & KOPEN_ACCMODE;
	   if (fid_access == KOPEN_RDONLY)
	      fid = fileno(stdin);
	   else if (fid_access == KOPEN_WRONLY)
	      fid = fileno(stdout);
	   else if (fid_access == KOPEN_RDWR)
	   {
	      fid = fileno(stdout);
	      kfile_clrstate(file, KFILE_RDWR);
	      kfile_setstate(file, KFILE_WRITE);
	   }
	   else
	   {
	      kinfo(KSYSLIB, "file_open: Invalid open access for file '-'. \
The '-' can only be used for exclusively reading or writing.");
		 return(-1);
	   }
	   kfile_clrstate(file, KFILE_PERM);
	}
	else if (kstrcmp(path, "#") == 0)
	{
	   fid = fileno(stderr);
	   kfile_clrstate(file, KFILE_PERM);
	}
	else if ((fid = open(path, flags, mode)) < 0)
	{
	   kinfo(KSYSLIB, "file_open:  Unable to open file '%s'", path);
	   return(-1);
	}
	else used_open = TRUE;

	if ((resources = (ResourceStruct *) kcalloc(1,
				sizeof(ResourceStruct))) == NULL)
	{
	   kinfo(KSYSLIB, "file_open: Not enough memory to kcalloc (%d) bytes \
for the khoros file structure.\n", sizeof(ResourceStruct));
	   return(-1);
	}
	resources->fid = fid;
	resources->used_open = used_open;
	file->resources = (kaddr) resources;

#if 0
if (kfile_checkfile(kstderr))
kfprintf(kstderr,"Opening '%s' for '%s'\n", path,
	flags & KOPEN_WRONLY | flags & KOPEN_RDWR ? "writing" : "reading");
#endif

	return(0);
}


/*-------------------------------------------------------------
|  
|  Routine Name: file_close
|  
|       Purpose: This function closes a "file".  It is the internal driver
|	         to close a data file, which is called by the user
|	         when they call kclose().  kclose() calls the "file" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the initial kopen().
|  
|         Input:  file - the kfile structure.
|        Output:  returns whether we were able to close the file
|   Called From:  internal routine called from kclose()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_close(
   kfile *file)
{
	int	   status = 0;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call close() since "kfile" is just a regular file
	 *  descriptor.
	 */
	if (resources->used_open == TRUE)
	   status = close(resources->fid);

#if 0
if (kfile_checkfile(kstderr))
kfprintf(kstderr,"Closing '%s' for '%s'\n", file->path,
	kfile_iswrite(file) ? "writing" : "reading");
#endif
	return(status);
}


/*-------------------------------------------------------------
|  
|  Routine Name: file_read
|  
|       Purpose: This function reads a "file".  It is the internal driver
|	         to read data from a file, which is called by the user
|	         when they call kread().  kread() calls the "file" internal
|	         drivers by the nature of the transport, which is dictated
|	         by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr  - the pointer to store the data into.
|		  nbytes - the number of bytes to read.
|        Output:  returns the number of bytes read from the file
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_read(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int	   count, numread = 0, state = file->state;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	if (state & KFILE_LOCK && !(state & (KFILE_WRITE | KFILE_RDWR)))
	   file_lock(file, KLOCK_SH);

	/*
	 *  Simply call read() since "kfile" is just a regular file
	 *  descriptor.
	 */
#if 0
        do
        {
           count = nbytes - numread;
           count = read(resources->fid, (char *) (ptr + numread), count);
           numread += count;
        }
        while (numread < nbytes && count > 0);
#else
        numread = read(resources->fid, (char *) (ptr + numread), nbytes);
#endif

	if (state & KFILE_LOCK && !(state & (KFILE_WRITE | KFILE_RDWR)))
	   file_lock(file, KLOCK_UN);

	return(numread);
}


/*-------------------------------------------------------------
|  
|  Routine Name: file_write
|  
|       Purpose: This function writes to a "file".  It is the internal driver
|	         to write data from the supplied data array to a file, which
|	         is called by the user when they call kwrite().  kwrite() calls
|	         the "file" internal drivers by the nature of the transport,
|	         which is dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  ptr    - the pointer to write the data from.
|		  nbytes - the number of bytes to write.
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kwrite()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_write(
   kfile *file,
   char  *ptr,
   int   nbytes)
{
	int	   count = nbytes;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call write() since "kfile" is just a regular file
	 *  descriptor.
	 */
	do
	{
	   if ((count = write(resources->fid, ptr, nbytes)) == -1)
	      return(-1);

	   nbytes -= count;
	} while (nbytes > 0 && count > 0);
	return(count);
}


/*-------------------------------------------------------------
|  
|  Routine Name: file_lseek
|  
|       Purpose: This function is used to do a "lseek".  It is the internal
|	         driver to rewind to a specific point so that the data can be
|	         skipped or re-read.  This is called when the user calls
|	         klseek().  klseek() calls the "file" internal drivers by the
|	         nature of the transport, which is dictated by the kopen().
|  
|         Input:  file   - the kfile structure.
|		  offset - the offset in which to seek
|		  whence - the control of how the offset will be applied
|        Output:  returns the number of bytes written to the kfile
|   Called From:  internal routine called from kread()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_lseek(
   kfile *file,
   int   offset,
   int   whence)
{
	int	   pos;
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	/*
	 *  Simply call lseek() since "kfile" is just a regular file
	 *  descriptor.
	 */
	pos = lseek(resources->fid, offset, whence);
	return(pos);
}


/*-------------------------------------------------------------
|  
|  Routine Name: file_access
|  
|       Purpose: This function is used to do see if a file is accessable.
|	         This is called when the user calls kaccess().  kaccess()
|	         calls the "file" internal drivers by the nature of the
|	         transport, which is dictated by the ktransport_identifier().
|  
|         Input:  path     - the initial path.
|		  filename - the filename that they wish to access
|		  mode     - the access mode information
|        Output:  returns the result
|   Called From:  internal routine called from kaccess()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_access(
   char *path,
   char *filename,
   int  mode)
{
	int	   status;

	/*
	 *  Simply call access() since "filename" is just a regular filename
	 */
	if (kstrcmp(filename, "-") == 0)
	   return(0);

	status = access(filename, mode);
	return(status);
}


/*-------------------------------------------------------------
|  
|  Routine Name: file_unlink
|  
|       Purpose: This function initializes a tempnam for a "file"
|	         type transport.  Given a request with a filename of
|	         the following syntax:
|  
|		  	"filename"
|		  	"/path/filename"
|		  	"./path/filename"
|		  	"file=XXXXXXXXXX"
|		  	     ...etc....
|  
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kunlink()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_unlink(
   char *identifier,
   char *filename)
{
	int	status;

	status = unlink(filename);
	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: file_lock
|  
|       Purpose: This function locks a "file" transport.  Given
|	         a request with a filename of the following syntax:
|  
|		  	"filename"
|		  	"/path/filename"
|		  	"./path/filename"
|		  	"file=XXXXXXXXXX"
|		  	     ...etc....
|  
|         Input: 
|        Output:  returns 0 or -1 depending whether we
|		  sucessfully created the template
|   Called From:  internal routine called from kflock()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_lock(
   kfile *file,
   int   operation)
{
	struct  flock lock;
	int	status, cmd, type = -1;
char *op;
	ResourceStruct *resources = (ResourceStruct *) file->resources;

	/*
	 *  Set the lock to be the whole file...
	 */
	lock.l_len    = 0;
	lock.l_start  = 0;
	lock.l_whence = KSEEK_SET;

	/*
	 *  Translate the lock operation in fcntl() lock type.
	 */
	if (operation & KLOCK_SH)
	{
	   lock.l_type = F_RDLCK;
op = "shared";
	   type = 1;
	}
	else if (operation & KLOCK_EX)
	{
	   lock.l_type = F_WRLCK;
op = "exclusive";
	   type = 2;
	}
	else if (operation & KLOCK_UN)
	{
	   lock.l_type = F_UNLCK;
op = "unlocked";
	   type = 8;
	}
	else
	   return(-1);

	/*
	 *  Blocking is translated into the type of set lock command for
	 *  fcntl.  The F_SETLK means set file lock, but don't block, and
	 *  F_SETLKW means set file lock, but go ahead and wait.
	 */
	if (operation & KLOCK_NB)
	   cmd = F_SETLK;
	else
	   cmd = F_SETLKW;

	if (kgetenv("KHOROS_TRANSPORT_USELOCKF"))
	   status = fcntl(resources->fid, cmd, &lock) == -1 ? -1 : 0;
	else
	   status = flock(resources->fid, type);

#if 0
if (kfile_checkfile(kstderr))
kfprintf(kstderr,"Locking '%s' for '%s'\n", file->path, op);
#endif

	return(status);
}

/*-------------------------------------------------------------
|  
|  Routine Name: file_descriptors
|  
|       Purpose: This function shouldn't exist, but there is occasion
|	         when the user wants to know the actual file descriptor
|	         associated with the file descriptor.
|  
|         Input: file structure
|        Output:  inum - the input file descriptor
|   Called From:  internal routine called from kdescriptor()
|    Written By:  Mark Young
|  
-------------------------------------------------------------*/
/*ARGSUSED*/
static int file_descriptors(
   kfile *file,
   int   *inum,
   int   *onum)
{
	ResourceStruct *resources = (ResourceStruct *) file->resources;


	if (inum != NULL) *inum = resources->fid;
	if (onum != NULL) *onum = resources->fid;
	return(0);
}

#endif  /* KFILE_DEF */
/* don't add after the endif */
