/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Manipulation Utilities
   >>>>
   >>>>  Private:
   >>>>		    kdata_size()
   >>>>   Static:
   >>>>   Public:
   >>>>		    kdatatype_cast_process()
   >>>>		    kdatatype_to_define()
   >>>>		    kdefine_to_datatype()
   >>>>		    kdatatype_cast_output()
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"

#define _is_complex(t) ((t & KCOMPLEX) || (t & KDCOMPLEX))
#define _is_float(t)   ((t & KFLOAT) || (t & KDOUBLE))
#define _is_signed(t)  ((t & KLONG) || (t & KINT) || (t & KSHORT) || (t & KBYTE))
#define _is_unsigned(t) ((t & KULONG) || (t & KUINT) || (t & KUSHORT) || (t & KUBYTE) || (t & KBIT))
#define _is_kcomplex(t) ((t & KCOMPLEX))
#define _is_kdcomplex(t) ((t & KDCOMPLEX))
#define _is_kdouble(t) ((t & KDOUBLE))
#define _is_kfloat(t)  ((t & KFLOAT))
#define _is_kbit(t)  ((t & KBIT))
#define _is_long(t) ((t & KLONG) || (t & KULONG))
#define _is_int(t)  ((t & KINT) || (t & KUINT))
#define _is_short(t)  ((t & KSHORT) || (t & KUSHORT))
#define _is_byte(t)  ((t & KBYTE) || (t & KUBYTE))
#define _is_string(t)  ((t & KSTRING))
#define _is_struct(t)  ((t & KSTRUCT))

/*-----------------------------------------------------------
|
|  Routine Name: kdata_size()
|
|       Purpose: This function returns the size of an
|                element of data of a certain data type
|                on the current machine.  For example,
|                this routine will return sizeof(float)
|                if kdata_size(KFLOAT) is called.
|
|		 Invoking this call:
|
|		     kmach_sizeof(kmach_type(NULL),datatype)
|
|		 will result in exactly the same information.
|
|         Input: datatype - the data type for which the
|
|                            size is being requested.
|	 Output:
|       Returns: The size of the data type specified by
|                the datatype input argument.  If the
|                datatype is not known, this function
|                returns 0.
|
|  Restrictions: For the bit case, which really should return
|                1/8, this function returns 1.  If you can't
|                figure out how long a bit is, you shouldn't
|                be calling this routine anyway!!
|    Written By: Jeremy Worley
|          Date: Aug 24, 1992 14:05
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

int 
kdata_size(int datatype)
{
   switch (datatype) {
      case KBIT:return (1);
   case KBYTE:
      return (sizeof(char));
   case KUBYTE:
      return (sizeof(unsigned char));
   case KSHORT:
      return (sizeof(short));
   case KUSHORT:
      return (sizeof(unsigned short));
   case KLONG:
      return (sizeof(long));
   case KULONG:
      return (sizeof(unsigned long));
   case KINT:
      return (sizeof(int));
   case KUINT:
      return (sizeof(unsigned int));
   case KFLOAT:
      return (sizeof(float));
   case KDOUBLE:
      return (sizeof(double));
   case KCOMPLEX:
      return (sizeof(kcomplex));
   case KDCOMPLEX:
      return (sizeof(kdcomplex));
   case KSTRING:
      return (sizeof(char *));
   case KSTRUCT:
      return (sizeof(kaddr)); /* -- not sure what else this could be! //SK -- */
   default:
      return (0);
   }
}


/*************************************************************
*
*  Routine Name: kdatatype_cast_process - cast type for processing
*
*       Purpose: The routine is used to recommend an appropriate
*                common data type to be used as when processing data.
*                What this routine does is examine the input data
*                types for precision, sign, range, etc. and recommends
*                one of four data types that can be used with minimal
*                risk of destroying data integrity if the variable
*		 return_values is KANYTYPE.  The values are KLONG, KULONG,
*		 KDOUBLE, or KDCOMPLEX.  If the variable return_values
*		 is not KANYTYPE this routine will return only one of the
*		 data types specified.  To specify more than one posible
*		 return value you must OR the data types together:
*
*        !     proc_type  = kdatatype_cast_process(type1, type2,
*        !                                         KUBYTE | KDOUBLE);
*
*		 Any number of data types can be specified.  This routine
*		 will always try to return the data type that will not destroy
*		 data integrity.  However, it will never return a data type
*		 not specified in return_values.  It is up to the programmer
*		 to ensure that the data type specified by the return value 
*		 either preserves the data integrity or handles the loss
*		 in data integrity properly.
*		 
*		 For example, you would lose data integrity if you called 
*		 this routine in the following manner :
*
*        !     proc_type  = kdatatype_cast_process(KUBYTE, KSHORT,
*        !                                         KUBYTE | KUSHORT);
*
*		 Return_values = KUBYTE | KUSHORT, the returned value
*		 will be KUSHORT, but since KSHORT means that negative numbers
*		 are posible the data processing the data as KUSHORT will not
*		 not reflect this properly.  The call should OR KINT or KLONG
*		 data type so that negative numbers are processed as negative
*		 numbers.  However, as stated above it is up to the programmer
*		 to specify the return_values so that data integrity is
*		 preserved or handles this lose in there processing routine
*		 correctly.
*
*		 If either of the types is KSTRING or KSTRUCT than
*		 KNONE is returned.
*
*         Input: type1 - first data type to be used in the determination.
*                type2 - second data type to be used in the determination.
*		 return_values - the OR'ed return values that are acceptable
*
*        Output: recommended_type -
*
*       Returns: The recommended data type.
*
*    Written By: Jeremy Worley & John M. Salas & Mark Young & Donna Koechner
*          Date: Oct 09, 1992 11:39
* Modifications:
*
*************************************************************/

int
kdatatype_cast_process(int type1, int type2, int return_values)
{
   int max_type, value, orig_value;
   int def_value = KDOUBLE;
      

   if (_is_string(type1) || _is_string(type2) ||
       _is_struct(type1) || _is_struct(type2))
      return(KNONE);

   if ( return_values == KANYTYPE )
   {
      if (_is_complex(type1) || _is_complex(type2))
         return (KDCOMPLEX);
      if (_is_float(type1) || _is_float(type2))
         return (KDOUBLE);
      if (_is_signed(type1) || _is_signed(type2))
         return (KLONG);

      return (KULONG);
   }

   max_type = type2;
   if ( type1 > type2 )
   {
      max_type = type1;
   }

   orig_value = max_type;
   if ( max_type <= return_values )
   {
      do
      {
	 if ((value = (max_type & return_values)) != 0)
	 {
	    if ( !(_is_unsigned(orig_value) && _is_signed(value) &&
		   kdata_size(orig_value) >= kdata_size(value)) ||
		  _is_float(value) || _is_complex(value) )
	       return (value);

	    def_value = value;
	 }
	 max_type <<= 1;
      } while ( max_type <= return_values );

      return (def_value);
   }

   do
   {
      if ((value = (max_type & return_values)) != 0)
	 return (value);

      max_type >>= 1;
   } while ( max_type > 0 );
   return(KNONE);
}

/*************************************************************
*
*  Routine Name: kdatatype_to_define - takes the string version of
*				       the data type and returns
*				       the #define value.
*
*       Purpose: This routine returns the integer
*		 value of the string data type passed in.
*		 Here is a table of string data types and abbreviations
*		 of string data types that can be used and the return values.
*
*	 !Data Type		Abbreviation	Return Value
*	 ! ---------------------------------------------------
*	 !     bit		     bi		    KBIT
*	 !     byte		     by		    KBYTE
*	 !     unsigned byte	     un by	    KUBYTE
*	 !     ubyte		     uby	    KUBYTE
*	 !     short		     sh		    KSHORT
*	 !     unsigned short	     un sh	    KUSHORT
*	 !     ushort		     ush	    KUSHORT
*	 !     integer		     in		    KINT
*	 !     unsigned integer	     un in	    KUINT
*	 !     uint		     ui		    KUINT
*	 !     long		     lon	    KLONG
*	 !     unsigned long	     un lon	    KULONG
*	 !     ulong		     ul		    KULONG
*	 !     float		     fl		    KFLOAT
*	 !     double		     do		    KDOUBLE
*	 !     complex		     co		    KCOMPLEX
*	 !     double complex	     do co	    KDCOMPLEX
*	 !     dcomplex		     dc		    KDCOMPLEX
*
*	   Note: KSTRING and KSTRUCT are not supported.
*
*         Input: stype - string giving name of data type
*
*        Output:
*
*       Returns: The integer value of the data type or 0 if not a datatype
*
*    Written By: Donna Koechner & John M. Salas
*          Date: Feb 28, 1993
* Modifications:
*
*************************************************************/

int
kdatatype_to_define(char *stype)
{
  int type;
  char *lower_stype;

  lower_stype = NULL;

  /* convert the string to all lowers, and use kstrstr to look for
     minimum unique letter combinations in the string.
        bit                             bi
        byte                            by
        unsigned byte                   un by
        ubyte                           uby

        short                           sh
        unsigned short                  un sh
        ushort                          ush

        integer                         in
        unsigned integer                un in
        uint                            ui

        long                            lon
        unsigned long                   un lon
        ulong                           ul

        float                           fl
        double                          do
        complex                         co
        double complex                  do co
        dcomplex                        dc       */

  lower_stype = kstring_lower(stype, lower_stype);

   /* string for BIT type */
   if (kstrstr(lower_stype, "bi") != NULL) {
        type = KBIT;
   }

   /* string for BYTE types */
   else if (kstrstr(lower_stype, "uby") != NULL) {
      type = KUBYTE;
   }
   else if (kstrstr(lower_stype, "by") != NULL) {
      if (kstrstr(lower_stype, "un") != NULL)
         type = KUBYTE;
      else
         type = KBYTE;
   }

   /* string for SHORT types */
   else if (kstrstr(lower_stype, "ush") != NULL) {
      type = KUSHORT;
   }
   else if (kstrstr(lower_stype, "sh") != NULL) {
      if (kstrstr(lower_stype, "un") != NULL)
         type = KUSHORT;
      else
         type = KSHORT;
   }

   /* string for INTEGER types */
   else if (kstrstr(lower_stype, "ui") != NULL) {
      type = KUINT;
   }
   else if (kstrstr(lower_stype, "in") != NULL) {
      if (kstrstr(lower_stype, "un") != NULL)
         type = KUINT;
      else
         type = KINT;
   }

   /* string for LONG types */
   else if (kstrstr(lower_stype, "ul") != NULL) {
      type = KULONG;
   }
   else if (kstrstr(lower_stype, "lon") != NULL) {
      if (kstrstr(lower_stype, "un") != NULL)
         type = KULONG;
      else
         type = KLONG;
   }

   /* string for FLOAT type */
   else if (kstrstr(lower_stype, "fl") != NULL) {
      type = KFLOAT;
   }

   /* string for COMPLEX and DOUBLE types */
   else if (kstrstr(lower_stype, "do") != NULL) {
      if (kstrstr(lower_stype, "co") != NULL)
         type = KDCOMPLEX;
      else
         type = KDOUBLE;
   }
   else if (kstrstr(lower_stype, "co") != NULL) {
      if (kstrstr(lower_stype, "dc") != NULL)
         type = KDCOMPLEX;
      else
         type = KCOMPLEX;
   }
   else {
      kinfo(KSTANDARD, "cannot interpret data type string: %s, %s",
            stype, lower_stype);
      return(FALSE);
   }

   return(type);

}

/*************************************************************
*
*  Routine Name: kdefine_to_datatype - takes the #define data type
*				       value and returns the string
*				       value
*
*       Purpose: This routine returns a string value corresponding
*		 to the #define data type passed in.
*		 Here is a table of types
*		 that can be used and the return values.
*
*        !   Data Type           Return Value
*        ! ---------------------------------------------------
*        !    KBIT            bit
*        !    KBYTE           byte
*        !    KUBYTE          unsigned byte
*        !    KSHORT          short
*        !    KUSHORT         unsigned short
*        !    KINT            integer
*        !    KUINT           unsigned integer
*        !    KLONG           long
*        !    KULONG          unsigned long
*        !    KFLOAT          float
*        !    KDOUBLE         double
*        !    KCOMPLEX        complex
*        !    KDCOMPLEX       double complex
*        !    KSTRING         string
*        !    KSTRUCT         struct
*
*         Input: type - #define of data type
*
*        Output:
*
*       Returns: string that represents data type
*
*    Written By: John M. Salas
*          Date: Mar 31, 1994
* Modifications:
*
*************************************************************/

char *
kdefine_to_datatype(int type)
{
   switch (type)
   {
      case KBIT:
	 return("Bit");
      case KBYTE:
	 return("Byte");
      case KUBYTE:
	 return("Unsigned Byte");
      case KLOGICAL:
         return("Logical");
      case KSHORT:
	 return("Short");
      case KUSHORT:
	 return("Unsigned Short");
      case KINT:
	 return("Integer");
      case KUINT:
	 return("Unsigned Iinteger");
      case KLONG:
	 return("Long");
      case KULONG:
	 return("Unsigned Long");
      case KFLOAT:
	 return("Float");
      case KDOUBLE:
	 return("Double");
      case KCOMPLEX:
	 return("Complex");
      case KDCOMPLEX:
	 return("Double Complex");
      case KSTRING:
	 return("String");
      case KSTRUCT:
	 return("Struct");
      default:
         return("Unknown");
   }
}

/************************************************************
*
*  Routine Name: kdatatype_cast_output - recommend an appropriate common
*					 data type for processing
*
*       Purpose: The routine is used to recommend an
*                appropriate common data type to be used
*                as when outputing data that is the result
*                of calculations involving data of the
*                types indicated in the input arguments.
*                What this routine does is examine the input
*                data types for precision, sign, range, etc.
*                and recommends one of the legal data types
*                that can be used with minimal risk of
*                destroying data integrity.
*
*
*         Input: type1 - first data type to be used in
*                        the determination.
*                type2 - second data type to be used in
*                        the determination.
*
*        Output:
*       Returns: The recommended data type.
*
*    Written By: Jeremy Worley
*          Date: Oct 09, 1992 11:56
* Modifications:
*
************************************************************/

int 
kdatatype_cast_output(int type1, int type2)
{
   int is_signed;

   if (_is_kdcomplex(type1) || _is_kdcomplex(type2))
      return (KDCOMPLEX);

   if (_is_kcomplex(type1) || _is_kcomplex(type2))
      return (KCOMPLEX);

   if (_is_kdouble(type1) || _is_kdouble(type2))
      return (KDOUBLE);

   if (_is_kfloat(type1) || _is_kfloat(type2))
      return (KFLOAT);

   is_signed = (_is_signed(type1) || _is_signed(type2));

   if (_is_long(type1) || _is_long(type2)) {
      if (is_signed)
         return (KLONG);
      else
         return (KULONG);
   }
   if (_is_int(type1) || _is_int(type2)) {
      if (is_signed)
         return (KINT);
      else
         return (KUINT);
   }
   if (_is_short(type1) || _is_short(type2)) {
      if (is_signed)
         return (KSHORT);
      else
         return (KUSHORT);
   }
   if (_is_byte(type1) || _is_byte(type2)) {
      if (is_signed)
         return (KBYTE);
      else
         return (KUBYTE);
   }
   return (KBIT);
}

