/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Manipulation Utilities
   >>>>
   >>>>  Private:
   >>>>		    kdata_scale()
   >>>>   Static:
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: kdata_scale
|
|       Purpose: This routine accepts as input a vector of
|		 any data type and scales and shifts each
|		 value in the data set by specified scaling
|		 and shifting constants.
|
|         Input: idata  - the input data sequence.
|		 num    - number of input and output data
|	   		  values.
|		 dtype  - data type of input and output
|			  data.
|		 rscale - real component of the scale factor.
|		 iscale - imaginary component of the scale
|			  factor.
|		 rshift - real component of the shift value.
|		 ishift - imaginary component of the shift
|			  value.
|
|        Output: odata  - output data sequence.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions:
|    Written By: Jeremy Worley
|          Date: Oct 05, 1992 11:59
|      Verified:
|  Side Effects:
| Modifications:
|
|-----------------------------------------------------------*/

int 
kdata_scale(
   kaddr	idata,
   int		num,
   int		dtype,
   double	rscale,
   double	iscale,
   double	rshift,
   double	ishift,
   kaddr	odata)
{
   int i;

   switch (dtype) {
   case KBIT:

      /*
       * the easy case...shifting and scaling doesn't make sense, so just
       * return
       */
      return (TRUE);

   case KBYTE:
      {
	 char *itmp = (char *) idata;
	 char *otmp = (char *) odata;

         for (i = 0; i < num; i++)
	    otmp[i] = (char) ((double) itmp[i] * rscale + rshift);
         break;
      }
   case KUBYTE:
      {
	 unsigned char *itmp = (unsigned char *) idata;
	 unsigned char *otmp = (unsigned char *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (unsigned char) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KSHORT:
      {
	 short *itmp = (short *) idata;
	 short *otmp = (short *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (short) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KUSHORT:
      {
	 unsigned short *itmp = (unsigned short *) idata;
	 unsigned short *otmp = (unsigned short *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (unsigned short) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KLONG:
      {
	 long *itmp = (long *) idata;
	 long *otmp = (long *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (long) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KULONG:
      {
	 unsigned long *itmp = (unsigned long *) idata;
	 unsigned long *otmp = (unsigned long *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (unsigned long) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KINT:
      {
	 int *itmp = (int *) idata;
	 int *otmp = (int *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (int) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KUINT:
      {
	 unsigned int *itmp = (unsigned int *) idata;
	 unsigned int *otmp = (unsigned int *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (unsigned int) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KFLOAT:
      {
	 float *itmp = (float *) idata;
	 float *otmp = (float *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = (float) ((double) itmp[i] * rscale + rshift);
      }
      break;
   case KDOUBLE:
      {
	 double *itmp = (double *) idata;
	 double *otmp = (double *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = itmp[i] * rscale + rshift;
      }
      break;
   case KCOMPLEX:
      {
	 kcomplex *itmp = (kcomplex *) idata;
	 kcomplex *otmp = (kcomplex *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = kcadd(kcmult(itmp[i],
		  kccomp((float) rscale, (float) iscale)),
	       kccomp((float) rshift, (float) ishift));
      }
      break;
   case KDCOMPLEX:
      {
	 kdcomplex *itmp = (kdcomplex *) idata;
	 kdcomplex *otmp = (kdcomplex *) odata;

	 for (i = 0; i < num; i++)
	    otmp[i] = kdcadd(kdcmult(itmp[i], kdccomp(rscale, iscale)),
	       kdccomp(rshift, ishift));
      }
      break;
   default:
      errno = KINVALID_DATATYPE;
      kerror("kmath", "kdata_scale()",
	 "The output data type is not recognized (%d).",dtype);
      return (FALSE);
   }

   return (TRUE);
}
