/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>              Data Manipulation Utilities
   >>>>
   >>>>  Private:
   >>>>             kcast_data()
   >>>>             _kcast_data()
   >>>>
   >>>>   Static:
   >>>>             _cast_from_bit()
   >>>>             _cast_from_bit_lsb()
   >>>>             _cast_from_kcomplex()
   >>>>             _cast_from_kdcomplex()
   >>>>   Public:
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "kdata_cast.h"

/*
 * this array comes from Khoros 1.0's vrev.h file.  It is an
 * array that can be indexed into to get bit reversed values.
 * Written by: Tom Sauer and Scoot Wilson.
 */
static unsigned char rev[] = {
        0, 128, 64, 192, 32, 160, 96, 224, 
        16, 144, 80, 208, 48, 176, 112, 240,
        8, 136, 72, 200, 40, 168, 104, 232,
        24, 152, 88, 216, 56, 184, 120, 248,
        4, 132, 68, 196, 36, 164, 100, 228,
        20, 148, 84, 212, 52, 180, 116, 244,
        12, 140, 76, 204, 44, 172, 108, 236,
        28, 156, 92, 220, 60, 188, 124, 252,
        2, 130, 66, 194, 34, 162, 98, 226,
        18, 146, 82, 210, 50, 178, 114, 242,
        10, 138, 74, 202, 42, 170, 106, 234,
        26, 154, 90, 218, 58, 186, 122, 250,
        6, 134, 70, 198, 38, 166, 102, 230,
        22, 150, 86, 214, 54, 182, 118, 246,
        14, 142, 78, 206, 46, 174, 110, 238,
        30, 158, 94, 222, 62, 190, 126, 254,
        1, 129, 65, 193, 33, 161, 97, 225, 
        17, 145, 81, 209, 49, 177, 113, 241,
        9, 137, 73, 201, 41, 169, 105, 233,
        25, 153, 89, 217, 57, 185, 121, 249,
        5, 133, 69, 197, 37, 165, 101, 229,
        21, 149, 85, 213, 53, 181, 117, 245,
        13, 141, 77, 205, 45, 173, 109, 237,
        29, 157, 93, 221, 61, 189, 125, 253,
        3, 131, 67, 195, 35, 163, 99, 227, 19,
        147, 83, 211, 51, 179, 115, 243, 11, 
        139, 75, 203, 43, 171, 107, 235, 27, 
        155, 91, 219, 59, 187, 123, 251, 7,
        135, 71, 199, 39, 167, 103, 231, 23,
        151, 87, 215, 55, 183, 119, 247, 15,
        143, 79, 207, 47, 175, 111, 239, 31,
        159, 95, 223, 63, 191, 127, 255 };

/*-----------------------------------------------------------
|
|  Routine Name: (static) _cast_from_bit
|
|       Purpose: This routine casts data from a bit
|                representation into any of the other known
|                data types.
|
|		 Input and output strides are provided to
|		 allow for arbitrary indexing through the
|		 input and output vectors.  Both of these
|		 strides are of type float because this
|		 allows a primitive sub- or super-sampling
|		 capability.
|
|         Input: in         - bit data
|                num_elem   - number of signed bytes in
|                             input and output vectors.
|                odatatype - desired output data type.
|
|        Output: out        - vector of data after it
|                             has been cast to the new
|                             data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 26, 1992 18:02
| Modifications:
|
------------------------------------------------------------*/

static int 
_cast_from_bit(
   char *in, 
   int num_elem, 
   int odatatype, 
   int convert, 
   char *out)
{
   int i;
   register unsigned char mask, *iptr;

   iptr = (unsigned char *) in;

   switch (odatatype) {
   case KBIT:
      {
	 unsigned char *optr = (unsigned char *) out;
	 unsigned char maskout;

	 for (i = 0; i < (num_elem >> 3) + ((num_elem % 8) != 0); i++)
	    optr[i] = 0;
	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    maskout = 0x80 >> (i % 8);
	    optr[(int) (i >> 3)] |=
	       ((iptr[(int) (i >> 3)] & mask) ? maskout : 0x0);
	 }
      }
      break;
   case KBYTE:
      {
	 signed char *optr = (signed char *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0);
	 }
      }
      break;
   case KUBYTE:
      {
	 unsigned char *optr = (unsigned char *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       (iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0;
	 }
      }
      break;
   case KSHORT:
      {
	 short *optr = (short *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0);
	 }
      }
      break;
   case KUSHORT:
      {
	 unsigned short *optr = (unsigned short *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0);
	 }
      }
      break;
   case KLONG:
      {
	 long *optr = (long *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0);
	 }
      }
      break;
   case KULONG:
      {
	 unsigned long *optr = (unsigned long *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0);
	 }
      }
      break;
   case KINT:
      {
	 int *optr = (int *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0);
	 }
      }
      break;
   case KUINT:
      {
	 unsigned int *optr = (unsigned int *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 0x1 : 0x0);
	 }
      }
      break;
   case KFLOAT:
      {
	 float *optr = (float *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 1.0 : 0.0);
	 }
      }
      break;
   case KDOUBLE:
      {
	 double *optr = (double *) out;

	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> ((int) (i) % 8);
	    *optr++ =
	       ((iptr[(int) (i >> 3)] & mask) ? 1.0 : 0.0);
	 }
      }
      break;
   case KCOMPLEX:
      {
	 kcomplex *optr = (kcomplex *) out;
	 switch (convert) {
	   case KREAL:
	   case KMAGNITUDE:
	      for (i = 0; i < num_elem; i++) {
	          mask = 0x80 >> ((int) (i) % 8);
	          *optr++ =
	             ((iptr[(int) (i >> 3)] & mask) ?
	             kccomp((float) 1.0, (float) 0.0) :
	             kccomp((float) 0.0, (float) 0.0));
	      }
	      break;
	   case KIMAGINARY:
	      for (i = 0; i < num_elem; i++) {
	          mask = 0x80 >> ((int) (i) % 8);
	          *optr++ =
	             ((iptr[(int) (i >> 3)] & mask) ?
	             kccomp((float) 0.0, (float) 1.0) :
	             kccomp((float) 0.0, (float) 0.0));
	      }
	      break;
	   case KPHASE:
	      for (i = 0; i < num_elem; i++) {
	          mask = 0x80 >> ((int) (i) % 8);
	          *optr++ =
	             ((iptr[(int) (i >> 3)] & mask) ?
	             kccomp((float) 0.999847, (float) 0.017452) : /*1 rad ;^)*/
	             kccomp((float) 0.0, (float) 0.0));
	      }
	      break;
	 }
      }
      break;
   case KDCOMPLEX:
      {
	 kdcomplex *optr = (kdcomplex *) out;
	 switch (convert) {
	   case KREAL:
	   case KMAGNITUDE:
	      for (i = 0; i < num_elem; i++) {
	         mask = 0x80 >> ((int) (i) % 8);
	         *optr++ =
	            ((iptr[(int) (i >> 3)] & mask) ?
	            kdccomp((double) 1.0, (double) 0.0) :
	            kdccomp((double) 0.0, (double) 0.0));
	      }
	      break;
	   case KIMAGINARY:
	      for (i = 0; i < num_elem; i++) {
	         mask = 0x80 >> ((int) (i) % 8);
	         *optr++ =
	            ((iptr[(int) (i >> 3)] & mask) ?
	            kdccomp((double) 1.0, (double) 0.0) :
	            kdccomp((double) 0.0, (double) 0.0));
	      }
	      break;
	   case KPHASE:
	      for (i = 0; i < num_elem; i++) {
	          mask = 0x80 >> ((int) (i) % 8);
	          *optr++ =
	             ((iptr[(int) (i >> 3)] & mask) ?
	             kdccomp((double) 0.999847, (double) 0.017452) : 
	             kdccomp((double) 0.0, (double) 0.0));
	      }
	      break;
	 }
      }
      break;
   default:
      errno = KINVALID_DATATYPE;
      kerror("kmath", "kdata_cast()",
	 "The output data type is not recognized (%d).",odatatype);
      return (FALSE);
   }
   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: (static) _cast_from_bit_lsb
|
|       Purpose: This routine casts data from a LSB bit
|                representation into any of the other known
|                data types.
|
|		 Input and output strides are provided to
|		 allow for arbitrary indexing through the
|		 input and output vectors.  Both of these
|		 strides are of type float because this
|		 allows a primitive sub- or super-sampling
|		 capability.
|
|         Input: in         - lsb bit data
|                num_elem   - number of signed bytes in
|                             input and output vectors.
|                odatatype - desired output data type.
|
|        Output: out        - vector of data after it
|                             has been cast to the new
|                             data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Jun 24, 1994 09:14
| Modifications:
|
------------------------------------------------------------*/

static int 
_cast_from_bit_lsb(
   char *in, 
   int num_elem, 
   int odatatype, 
   int convert, 
   char *out)
{
   int num = (num_elem + 7) >> 3;
   unsigned char *tmp = kmalloc(num);
   unsigned char *tmp1 = (unsigned char *)in;
   int status;
   int i;
   
   for (i = 0; i < num; i++)
      tmp[i] = rev[tmp1[i]];

   status = _cast_from_bit((char *)tmp, num_elem, odatatype, convert, out);

   kfree(tmp);

   return (status);
}

/*-----------------------------------------------------------
|
|  Routine Name: _cast_from_kcomplex
|
|       Purpose: This routine casts data from a kcomplex
|                representation into any of the other known
|                data types.
|
|		 Input and output strides are provided to
|		 allow for arbitrary indexing through the
|		 input and output vectors.  Both of these
|		 strides are of type double because this
|		 allows a primitive sub- or super-sampling
|		 capability.
|
|         Input: in         - kcomplex data
|                num_elem   - number of signed bytes in
|                             input and output vectors.
|                odatatype - desired output data type.
|
|        Output: out        - vector of data after it
|                             has been cast to the new
|                             data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 27, 1992 14:05
| Modifications:
|
------------------------------------------------------------*/

static int 
_cast_from_kcomplex(
   char *in, 
   int num_elem, 
   int odatatype, 
   int convert, 
   char *out)
{
   register int i;
   register unsigned char mask;
   register kcomplex *iptr = (kcomplex *) in;

   switch (odatatype) {
   case KBIT:
      {
	 unsigned char *optr = (unsigned char *) out;

	 mask = 0x80;
	 for (i = 0; i < (num_elem >> 3) + ((num_elem % 8) != 0); i++)
	    optr[i] = 0;
	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> (i % 8);
	    optr[(int) (i >> 3)] |=
	       (kcmag(*iptr++) == 0.0) ? 0x0 : mask;
	 }
      }
      break;
   case KBYTE:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, signed char)
      break;
   case KUBYTE:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, unsigned char)
      break;
   case KSHORT:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, short)
      break;
   case KUSHORT:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, unsigned short)
      break;
   case KLONG:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, long)
      break;
   case KULONG:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, unsigned long)
      break;
   case KINT:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, int)
      break;
   case KUINT:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, unsigned int)
      break;
   case KFLOAT:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, float)
      break;
   case KDOUBLE:
      _do_cast_from_complex(out, iptr, num_elem, i, convert, double)
      break;
   case KCOMPLEX:
      {
	 kcomplex *optr = (kcomplex *) out;

	 for (i = 0; i < num_elem; i++)
	     *optr++ = *iptr++;
      }
      break;
   case KDCOMPLEX:
      {
	 kdcomplex *optr = (kdcomplex *) out;

	 for (i = 0; i < num_elem; i++)
	     *optr++ =kcomp2dcomp(*iptr++);
      }
      break;
   default:
      errno = KINVALID_DATATYPE;
      kerror("kmath", "kdata_cast()",
	 "The output data type is not recognized (%d).", odatatype);
      return (FALSE);
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _cast_from_kdcomplex
|
|       Purpose: This routine casts data from a double kcomplex
|                representation into any of the other known
|                data types.
|
|		 Input and output strides are provided to
|		 allow for arbitrary indexing through the
|		 input and output vectors.  Both of these
|		 strides are of type float because this
|		 allows a primitive sub- or super-sampling
|		 capability.
|
|         Input: in         - double kcomplex data
|                num_elem   - number of signed bytes in
|                             input and output vectors.
|                odatatype - desired output data type.
|
|        Output: out        - vector of data after it
|                             has been cast to the new
|                             data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Jeremy Worley
|          Date: Aug 27, 1992 14:43
| Modifications:
|
------------------------------------------------------------*/
static int 
_cast_from_kdcomplex(
   char *in, 
   int num_elem, 
   int odatatype, 
   int convert,
   char *out)
{
   register int i;
   register unsigned char mask;
   register kdcomplex *iptr = (kdcomplex *) in;

   switch (odatatype) {
   case KBIT:
      {
	 unsigned char *optr = (unsigned char *) out;

	 mask = 0x80;
	 for (i = 0; i < (num_elem >> 3) + ((num_elem % 8) != 0); i++)
	    optr[i] = 0;
	 for (i = 0; i < num_elem; i++) {
	    mask = 0x80 >> (i % 8);
	    optr[(int) (i >> 3)] |=
	       (kdcmag(*iptr++) == 0.0) ? 0x0 : mask;
	 }
      }
      break;
   case KBYTE: 
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, signed char)
     break;
   case KUBYTE:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, unsigned char)
      break;
   case KSHORT:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, short)
      break;
   case KUSHORT:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, unsigned short)
      break;
   case KLONG:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, long)
      break;
   case KULONG:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, unsigned long)
      break;
   case KINT:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, int)
      break;
   case KUINT:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, unsigned int)
      break;
   case KFLOAT:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, float)
      break;
   case KDOUBLE:
      _do_cast_from_dcomplex(out, iptr, num_elem, i, convert, double)
      break;
   case KCOMPLEX:
      {
	 kcomplex *optr = (kcomplex *) out;

	 for (i = 0; i < num_elem; i++)
	     *optr++ = kdcomp2comp(*iptr++);
      }
      break;
   case KDCOMPLEX:
      {
	 kdcomplex *optr = (kdcomplex *) out;

	 for (i = 0; i < num_elem; i++)
	     *optr++ = *iptr++;
      }
      break;
   default:
      errno = KINVALID_DATATYPE;
      kerror("kmath", "kdata_cast()",
	 "The output data type is not recognized (%d).",odatatype);
      return (FALSE);
   }

   return (TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: _kdata_cast
|       Purpose: A private routine for casting data.  This one
|                takes an extra argument.  It would be better
|                to do something totally different with the
|                casting stuff (i.e. no floating point strides), 
|                but that'll come with time.
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jun 04, 1994 15:13
| Modifications:
|
------------------------------------------------------------*/

int 
_kdata_cast(
   kaddr	idata,
   int		idatatype,
   int		odatatype,
   int		num_elems,
   int		convert,
   kaddr	odata)
{
   int i;
   
   /*
    * Check to see if either data vector is NULL.  If so, then this routine
    * should not have been called.
    */
   if (idata == NULL || odata == NULL) {
      errno = KNULL_PARAMETER;
      kerror("kmath", "kdata_cast()",
	 "The input and output vectors must not be NULL.");
      return (FALSE);
   }

   /*
    * convert the data from one data type to the other
    */
   switch (idatatype) {
   case KBIT:
      return (_cast_from_bit(idata, num_elems, odatatype, convert, odata));
   case KBIT_LSB:
      return (_cast_from_bit_lsb(idata, num_elems, odatatype, convert, odata));
   case KBYTE:
      _do_type_cast(odata, idata, num_elems, i, odatatype, signed char, convert)
   case KUBYTE:
      _do_type_cast(odata, idata, num_elems, i, odatatype, unsigned char, 
		    convert)
   case KSHORT:
      _do_type_cast(odata, idata, num_elems, i, odatatype, short, convert)
   case KUSHORT:
      _do_type_cast(odata, idata, num_elems, i, odatatype, unsigned short, 
		    convert)
   case KLONG:
      _do_type_cast(odata, idata, num_elems, i, odatatype, long, convert)
   case KULONG:
      _do_type_cast(odata, idata, num_elems, i, odatatype, unsigned long, 
		    convert)
   case KINT:
      _do_type_cast(odata, idata, num_elems, i, odatatype, int, convert)
   case KUINT:
      _do_type_cast(odata, idata, num_elems, i, odatatype, unsigned int, 
		    convert)
   case KFLOAT:
      _do_type_cast(odata, idata, num_elems, i, odatatype, float, convert)
   case KDOUBLE:
      _do_type_cast(odata, idata, num_elems, i, odatatype, double, convert)
   case KCOMPLEX:
      return (_cast_from_kcomplex(idata, num_elems, odatatype, convert,odata));
   case KDCOMPLEX:
      return (_cast_from_kdcomplex(idata, num_elems, odatatype,convert,odata));
   default:
      errno = KINVALID_DATATYPE;
      kerror("kmath", "kdata_cast()",
	 "The input data type is not recognized (%d).",idatatype);
   }				/* end of switch from hell */
   return (FALSE);
}

/*----------------------------------------------------------*
|
|  Routine Name: kdata_cast
|
|       Purpose: This function is used to rapidly cast
|                an array of one data type into another
|                data type.  There are separate input
|                and output data types.  This function
|                calls one of thirteen functions to
|                perform the operation once the data
|                types have been chosen.
|
|                Input and output strides are provided to
|                allow for arbitrary indexing through the
|                input and output vectors.  Both of these
|                strides are of type double because this
|                allows a primitive sub- or super-sampling
|                capability.
|
|
|         Input: idata      - input data that is to
|                             be cast to the target data
|                             type.
|                idatatype - data type of the input
|                             vector.
|                odatatype - data type of the output
|                             vector.  This is the target
|                             data type for the convertsion.
|                num_elems  - number of elements in both
|                             the input and output arrays.
|                convert    - complex conversion mode.  can
|                             be one of KREAL, KIMAGINARY, KMAGNITUDE,
|                             or KPHASE.
|                istride    - a double that describes the
|                             stride or skip factor to be
|                             used when indexing through
|                             the input.
|                ostride    - a double that describes the
|                             stride or skip factor to be
|                             used when indexing through
|                             the output.
|
|        Output: odata      - vector that will contain
|                             the input data cast to the
|                             odatatype data type.
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|  Restrictions:
|    Written By: Jeremy Worley
|          Date: Aug 26, 1992 17:28
|      Verified:
|  Side Effects:
| Modifications:
|
------------------------------------------------------------*/

int 
kdata_cast(
   kaddr        idata,
   int          idatatype,
   int          odatatype,
   int          num_elems,
   int          convert,
   double       istride,
   double       ostride,
   kaddr        odata)
{
   return(_kdata_cast(idata, idatatype, odatatype, num_elems, convert, odata));
}
