/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>            Data Type Conversion Utilities
   >>>>
   >>>>  Private:
   >>>>             
   >>>>   Static:
   >>>>             
   >>>>   Public:
   >>>>             kdccomp()
   >>>>             kdcreal()
   >>>>             kdcimag()
   >>>>             kdcr2p()
   >>>>             kdcp2r()
   >>>>             
   >>>>             kccomp()
   >>>>             kcreal()
   >>>>             kcimag()
   >>>>             kcr2p()
   >>>>             kcp2r()
   >>>>
   >>>>		    kcomp2dcomp()
   >>>>		    kdcomp2comp()
   >>>>             
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/************************************************************
*
*  Routine Name: kdccomp() - construct a double precision 
*                            complex number from two real numbers.
*
*       Purpose: kdccomp() constructs a double precision 
*                complex number from two real numbers.  The 
*                first argument becomes the real component 
*                and the second argument becomes the 
*                imaginary component of the resulting double 
*                precision complex number.
*               
*         Input: a - the real component of the kdcomplex 
*                    number
*                b - the imaginary component of the kdcomplex 
*                    number
*
*        Output: 
*                
*       Returns: A kdcomplex number composed of the two
*                inputs where the first input is the real
*                component and the second input is the 
*                imaginary component.
*
*  Restrictions: This function returns an aggregate.  If 
*                your compiler cannot deal with this, you
*                are in trouble.
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 10:07 
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kdcomplex kdccomp(
   double a,
   double b)
{
   kdcomplex c;

   c.r = a;
   c.i = b;
   return(c);
}


/************************************************************
*
*  Routine Name: kdcreal() - return the real component of a
*                            double precision complex number.
*
*       Purpose: kdcreal() returns the real component of the
*                double precision complex input argument.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The real component of the kdcomplex input 
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 11:07
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

double kdcreal(
   kdcomplex a)
{
   return(a.r);
}

/************************************************************
*
*  Routine Name: kdcimag() - return the imaginary component
*                            of a double precision complex number.
*
*       Purpose: kdcimag() returns the value of the imaginary
*                component of the double precision complex 
*                input argument.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: The value of the imaginary component of the 
*                input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992  10:20
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

double kdcimag(
   kdcomplex a)
{
   return(a.i);
}

/************************************************************
*
*  Routine Name: kdcr2p() - convert double complex from rectangular
*			    coordinates to polar coordinates
*
*       Purpose: kdcr2p() convert a double complex from rectangular
*		 coordinates to polar coordinates.  The radius is
*		 stored as the real, and the angle as the imaginary.
*
*         Input: r - kdcomplex number
*
*        Output: 
*                
*       Returns: The polar coordinate double complex value
*
*  Restrictions: 
*
*    Written By: Donna Koechner and Mark Young
*
*          Date: Mar 10, 1993 22:07
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kdcomplex kdcr2p(
   kdcomplex r)
{
   kdcomplex p;

   p.r = ksqrt(r.r * r.r + r.i * r.i);
   p.i = katan2(r.i,r.r);
   return(p);
}

/************************************************************
*
*  Routine Name: kdcp2r() - convert double complex from polar
*			    coordinates to rectangular coordinates
*
*       Purpose: kdcr2p() convert a double complex from polar
*		 coordinates to rectangular coordinates.  Stored
*		 as the (real,imaginary) pair.  Where the polar
*		 coordinate input is the (radius,angle) pair.  This
*                pair is stored in the same structure used for
*                dcomplex data, except that what was formerly the
*                real part is now the magnitude, and what was the
*                imaginary component will now be the phase in radians.
*
*         Input: p - kdcomplex number
*
*        Output: 
*                
*       Returns: The rectangular coordinate double complex value
*
*  Restrictions: 
*
*    Written By: Donna Koechner and Mark Young
*
*          Date: Mar 10, 1993 22:07
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kdcomplex kdcp2r(
   kdcomplex p)
{
   kdcomplex r;

   r.r = p.r * cos(p.i);
   r.i = p.r * sin(p.i);
   return(r);
}


/************************************************************
*
*  Routine Name: kccomp() - construct a complex number from
*                           two real numbers.
*
*       Purpose: kccomp() constructs a complex number from
*                two real numbers.  The first argument 
*                becomes the real component and the second
*                argument becomes the imaginary component
*                of the resulting complex number.
*               
*         Input: a - the real component of the complex number
*                b - the imaginary component of the complex 
*                    number
*
*        Output: 
*                
*       Returns: A complex number composed of the two
*                inputs where the first input is the real
*                component and the second input is the 
*                imaginary component.
*
*  Restrictions: This function returns an aggregate.  If 
*                your compiler cannot deal with this, you
*                are in trouble.
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 10:07 
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kcomplex kccomp(
   double a,
   double b)
{
   kcomplex c;

   c.r = a;
   c.i = b;
   return(c);
}


/************************************************************
*
*  Routine Name: kcreal() - return the real component of a
*                           complex number.
*
*       Purpose: kcreal() returns the real component of the
*                complex input argument.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The real component of the complex input 
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992 11:07
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

float kcreal(
   kcomplex a)
{
   return(a.r);
}
   
/************************************************************
*
*  Routine Name: kcimag() - return the imaginary component
*                          of a complex number.
*
*       Purpose: kcimag() returns the value of the imaginary
*                component of the complex input argument.
*
*         Input: a - complex number
*
*        Output: 
*                
*       Returns: The value of the imaginary component of the 
*                input argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Jul  2, 1992  10:20
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

float kcimag(
   kcomplex a)
{
   return(a.i);
}

/************************************************************
*
*  Routine Name: kcr2p() - convert complex from rectangular
*			   coordinates to polar coordinates
*
*       Purpose: kdcr2p() convert a complex from rectangular
*		 coordinates to polar coordinates.  The radius is
*		 stored as the real, and the angle as the imaginary.
*
*         Input: r - rectangular coordinate (kcomplex)
*
*        Output: 
*                
*       Returns: The polar coordinate complex value
*
*  Restrictions: 
*
*    Written By: Donna Koechner and Mark Young
*
*          Date: Mar 10, 1993 22:07
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kcomplex kcr2p(
   kcomplex r)
{
   kcomplex p;

   p.r = sqrt((double) (r.r * r.r + r.i * r.i));
   p.i = atan((double) (r.i / r.r));
   return(p);
}

/************************************************************
*
*  Routine Name: kcp2r() - convert complex from polar
*			   coordinates to rectangular coordinates
*
*       Purpose: kdcr2p() convert a complex from polar
*		 coordinates to rectangular coordinates.  Stored
*		 as the (real,imaginary) pair in a kcomplex data type.  
*		 The polar coordinate input is also encoded in a kcomplex
*		 datatype, but the interpretation is a (radius,angle) pair.
*
*         Input: p - polar coordinate (kcomplex)
*
*        Output: 
*                
*       Returns: The rectangular coordinate complex value
*
*  Restrictions: 
*
*    Written By: Donna Koechner and Mark Young
*
*          Date: Mar 10, 1993 22:07
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kcomplex kcp2r(
   kcomplex p)
{
   kcomplex r;

   r.r = p.r * cos((double) p.i);
   r.i = p.r * sin((double) p.i);
   return(r);
}

/************************************************************
*
*  Routine Name: kcomp2dcomp - convert a kcomplex number to a
*                kdcomplex number.
*
*       Purpose: kcomp2dcomp converts a single precision
*                complex number in a typedef'd "kcomplex"
*                structure into a double precision
*                complex number in a typedef'd "kdcomplex"
*                structure.
*
*         Input: a - kcomplex number
*
*        Output: 
*                
*       Returns: the kdcomplex conversion of the input
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Aug 31, 1992 14:55
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kdcomplex kcomp2dcomp(
   kcomplex a)
{
   kdcomplex b;

   b.r = (double)a.r;
   b.i = (double)a.i;
   return(b);
}

/************************************************************
*
*  Routine Name: kdcomp2comp - convert a kdcomplex number to a
*                             kcomplex number.
*
*       Purpose: kdcomp2comp converts a double precision
*                complex number in a typedef'd "kdcomplex"
*                structure into a single precision
*                complex number in a typedef'd "kcomplex"
*                structure.
*
*         Input: a - kdcomplex number
*
*        Output: 
*                
*       Returns: the complex conversion of the input
*                argument.
*
*  Restrictions: 
*
*    Written By: Jeremy Worley  
*
*          Date: Aug 31, 1992 14:56
*
*      Verified:
*
*  Side Effects: 
*
* Modifications:
*
*************************************************************/

kcomplex kdcomp2comp(
   kdcomplex a)
{
   kcomplex b;

   b.r = (float)a.r;
   b.i = (float)a.i;
   return(b);
}
