/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>
   >>>>             Khoros'ized Gnu DBM System
   >>>>
   >>>>   Static:
   >>>>			get_next_key
   >>>>			write_header
   >>>>			read_header
   >>>>			pop_avail_block
   >>>>			get_elem
   >>>>			get_block
   >>>>			put_av_elem
   >>>>			push_avail_block
   >>>>			adjust_bucket_avail
   >>>>			find_entry
   >>>>			read_entry
   >>>>			init_cache
   >>>>			new_bucket
   >>>>			get_bucket
   >>>>			split_bucket
   >>>>			write_bucket
   >>>>			findkey
   >>>>			end_update
   >>>>			fatal
   >>>>			hash_dbm
   >>>>			alloc_dbm
   >>>>			free_dbm
   >>>>             
   >>>>   Public:
   >>>>			kdbm_open
   >>>>			kdbm_fdopen
   >>>>			kdbm_close
   >>>>			kdbm_fetch
   >>>>			kdbm_store
   >>>>			kdbm_delete
   >>>>			kdbm_firstkey
   >>>>			kdbm_nextkey
   >>>>			kdbm_checkkey
   >>>>			kdbm_read
   >>>>			kdbm_write
   >>>>			kdbm_lseek
   >>>>			kdbm_getmachtype
   >>>>
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"
#include "dbm.h"


/*  This file is part of KDBM, the GNU data base manager, by Philip A. Nelson.
    Copyright (C) 1990, 1991, 1993

    KDBM is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    KDBM is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with KDBM; see the file COPYING.  If not, write to
    the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

    You may contact the author by:
       e-mail:  phil@cs.wwu.edu
      us-mail:  Philip A. Nelson
                Computer Science Department
                Western Washington University
                Bellingham, WA 98226
       
*************************************************************************/

/* The global variables used for the "original" interface. */
static kdbm  *file = NULL;

/* Memory for return data for the "original" interface. */
static char *fetch_val = NULL;	/* Used by fetch. */

static int   hash_dbm  PROTO((kdatum));
static int   alloc_dbm PROTO((kdbm *, int));
static void  free_dbm  PROTO((kdbm *, long, long));
static void  end_update PROTO((kdbm *));

/*
 * defines for the machtype conversion utilities
 */
#define KGDBM_ARCH_DECSTATION 1
#define KGDBM_ARCH_NS32000    2
#define KGDBM_ARCH_SUN        8
#define KGDBM_ARCH_VAX	      9
#define KGDBM_ARCH_CRAY	      16
#define KGDBM_ARCH_I386	      20
#define KGDBM_ARCH_ALPHA      23	

#if (KMACH_LOCAL == KMACH_ALPHA) || (KMACH_LOCAL == KMACH_CRAY)
#define STATBLKSIZE 2048
#else
#define STATBLKSIZE 1024
#endif

/*-----------------------------------------------------------
|
|  Routine Name: _convert_from_gdb_mach
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 16, 1994 12:13
| Modifications:
|
------------------------------------------------------------*/

static unsigned char
_convert_from_gdb_mach(unsigned char gdbmach)
{
   /*
    *  This is black magic...kill a chicken, rub some ashes on your
    *  privates and dive right in! 
    */
   switch(gdbmach)
   {
      case KGDBM_ARCH_DECSTATION:
      case KGDBM_ARCH_NS32000:
      case KGDBM_ARCH_I386:
	 return (KORDER_LITTLE_ENDIAN | KFORMAT_IEEE);
      case KGDBM_ARCH_ALPHA:
	 return (KORDER_LITTLE_ENDIAN | KFORMAT_IEEE_ALPHA);
      case KGDBM_ARCH_VAX:
	 return (KORDER_LITTLE_ENDIAN | KFORMAT_VAX);
      case KGDBM_ARCH_CRAY:
	 return (KORDER_BIG_ENDIAN | KFORMAT_CRAY);
      default:
	 return (KORDER_BIG_ENDIAN | KFORMAT_IEEE);
   }
}

/*-----------------------------------------------------------
|
|  Routine Name: _convert_to_gdb_mach
|       Purpose:
|         Input:
|        Output:
|       Returns:
|    Written By: Jeremy Worley
|          Date: Jul 16, 1994 12:29
| Modifications:
|
------------------------------------------------------------*/

static unsigned char
_convert_to_gdb_mach(unsigned char mach)
{
   switch(mach)
   {
      case KORDER_BIG_ENDIAN | KFORMAT_IEEE:
	 return(KGDBM_ARCH_SUN);
      case KORDER_LITTLE_ENDIAN | KFORMAT_IEEE_ALPHA:
	 return(KGDBM_ARCH_ALPHA);
      case KORDER_LITTLE_ENDIAN | KFORMAT_IEEE:
	 return(KGDBM_ARCH_DECSTATION);
      case KORDER_LITTLE_ENDIAN | KFORMAT_VAX:
	 return(KGDBM_ARCH_VAX);
      case KORDER_BIG_ENDIAN | KFORMAT_CRAY:
	 return(KGDBM_ARCH_CRAY);
      default:
	 return(KGDBM_ARCH_SUN);
   }
}
   
/*-----------------------------------------------------------
|
|  Routine Name: fatal
|
|       Purpose: If a fatal error is detected, come here and exit. VAL
|		 tells which fatal error occured.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/
/*ARGSUSED*/
static int fatal(
     kdbm *dbf,
     char *val)
{
	kfprintf(kstderr,"kdbm fatal: %s.\n", val);
	return(FALSE);
}


/*-----------------------------------------------------------
|
|  Routine Name: get_entry
|
|       Purpose: get the data found in bucket entry ELEM_LOC in file
|		 and cache the value into the database file.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static data_cache_elem *get_entry(
   kdbm *dbf,
   int elem_loc)
{
	int num_bytes;
	data_cache_elem *data_ca;

	/* Sanity check for the insane */
	if (elem_loc < 0)
	   return(NULL);

	/* Is it already in the cache? */
	if (dbf->cache_entry->ca_data.elem_loc == elem_loc)
	   return(&dbf->cache_entry->ca_data);


	/* Set up the cache with the sizes and pointers. */
	end_update(dbf);
	data_ca = &dbf->cache_entry->ca_data;
	kfree(data_ca->key);
	kfree(data_ca->data);
	data_ca->seek_pos  = 0;
	data_ca->elem_loc  = elem_loc;
	data_ca->hash      = dbf->bucket->h_table[elem_loc].hash_value;
	data_ca->key_size  = dbf->bucket->h_table[elem_loc].key_size;
	data_ca->data_size = dbf->bucket->h_table[elem_loc].data_size;
	data_ca->file_pos  = dbf->bucket->h_table[elem_loc].data_pointer;

	/* Seek into the cache. */
	if (klseek(dbf->desc, data_ca->file_pos, KSEEK_SET) != data_ca->file_pos)
	{
	   fatal(dbf, "klseek error");
	   return(NULL);
	}

	num_bytes = data_ca->key_size;
	if ((data_ca->key = (char *) kmalloc(num_bytes)) == NULL)
	{
	   fatal(dbf, "malloc error");
	   return(NULL);
	}

	if (kread(dbf->desc, data_ca->key, num_bytes) != num_bytes)
	{
	   fatal(dbf, "kread error");
	   return(NULL);
	}
	return(data_ca);
}

/*-----------------------------------------------------------
|
|  Routine Name: read_entry
|
|       Purpose: Read the data found in bucket entry ELEM_LOC in file
|		 DBF and return a pointer to it.  Also, cache the read
|		 value.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static data_cache_elem *read_entry(
     kdbm *dbf,
     int elem_loc)
{
	int  num_bytes, file_pos;
	data_cache_elem *data_ca;


	if ((data_ca = get_entry(dbf, elem_loc)) == NULL || data_ca->data)
	   return(data_ca);

	num_bytes = data_ca->data_size;
	if ((data_ca->data = (char *) kmalloc(num_bytes)) == NULL)
	{
	   fatal(dbf, "malloc error");
	   return(NULL);
	}

	/* Seek into the cache. */
	file_pos = data_ca->file_pos + data_ca->key_size;
	if (klseek(dbf->desc, file_pos, KSEEK_SET) != file_pos)
	{
	   fatal(dbf, "klseek error");
	   return(NULL);
	}

	if (kread(dbf->desc, data_ca->data, num_bytes) != num_bytes)
	{
	   fatal(dbf, "kread error");
	   return(NULL);
	}
	return(data_ca);
}

/*-----------------------------------------------------------
|
|  Routine Name: init_cache
|
|       Purpose: initialize the bucket_cache
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int init_cache(
    kdbm *dbf,
    int size)
{
  register int index;

  if (dbf->bucket_cache == NULL) {
    dbf->bucket_cache = (cache_elem *) kmalloc(sizeof(cache_elem) * size);
    if(dbf->bucket_cache == NULL) {
      return(-1);
    }
    dbf->cache_size = size;

    for(index = 0; index < size; index++) {
      (dbf->bucket_cache[index]).ca_bucket
            = (hash_bucket *) kcalloc(1, STATBLKSIZE);
      if ((dbf->bucket_cache[index]).ca_bucket == NULL) {
	return(-1);
      }
      (dbf->bucket_cache[index]).ca_adr = 0;
      (dbf->bucket_cache[index]).ca_changed = FALSE;
      (dbf->bucket_cache[index]).ca_data.hash     = -1;
      (dbf->bucket_cache[index]).ca_data.elem_loc = -1;
      (dbf->bucket_cache[index]).ca_data.data = NULL;
      (dbf->bucket_cache[index]).ca_data.key = NULL;
    }
    dbf->bucket = dbf->bucket_cache[0].ca_bucket;
    dbf->cache_entry = &dbf->bucket_cache[0];
  }
  return(0);
}

/*-----------------------------------------------------------
|
|  Routine Name: write_bucket
|
|       Purpose: The only place where a bucket is written.  CA_ENTRY
|		 is the cache entry containing the bucket to be
|		 written.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void write_bucket(
     kdbm *dbf,
     cache_elem *ca_entry)
{
  int  i, size;		/* The return value for kwrite. */
  int  file_pos;	/* The return value for klseek. */
  
  file_pos = klseek(dbf->desc, ca_entry->ca_adr, KSEEK_SET);
  if (file_pos != ca_entry->ca_adr)
    fatal(dbf, "klseek error");

  if (kwrite_int(dbf->desc, (int  *) ca_entry->ca_bucket, 3) != 3)
    fatal(dbf, "kwrite error");

  if (kwrite_long(dbf->desc, (long *) &ca_entry->ca_bucket->bucket_avail,
		  BUCKET_AVAIL * 2) != (BUCKET_AVAIL * 2))
    fatal(dbf, "kwrite error");

  for (i = 0; i < dbf->header->bucket_elems; i++)
  {
        if (kwrite_int(dbf->desc,
		       &ca_entry->ca_bucket->h_table[i].hash_value, 3) != 3)
	  fatal(dbf, "kwrite error");

        if (kwrite_long(dbf->desc,
		        &ca_entry->ca_bucket->h_table[i].data_pointer, 1) != 1)
	  fatal(dbf, "kwrite error");

	if (kwrite(dbf->desc, &ca_entry->ca_bucket->h_table[i].key_start[0],
			SMALL) != SMALL)
	{
	   fatal(dbf, "kwrite error");
	}
  }
  size = dbf->header->bucket_size - 3*kmach_sizeof(dbf->header->machtype,KINT) -
	 (BUCKET_AVAIL * 2) * kmach_sizeof(dbf->header->machtype,KLONG) -
	 dbf->header->bucket_elems*
	 (3*kmach_sizeof(dbf->header->machtype,KINT) +
	  kmach_sizeof(dbf->header->machtype,KLONG) +
	  kmach_sizeof(dbf->header->machtype,KBYTE) * SMALL);

  if (size && klseek(dbf->desc, size, KSEEK_CUR) == -1)
     fatal(dbf, "klseek error");

  ca_entry->ca_changed = FALSE;
  ca_entry->ca_data.hash     = -1;
  ca_entry->ca_data.elem_loc = -1;
}

/*-----------------------------------------------------------
|
|  Routine Name: get_bucket
|
|       Purpose: Find a bucket for DBF that is pointed to by the
|		 bucket directory from location DIR_INDEX.  The bucket
|		 cache is first checked to see if it is already in
|		 memory.  If not, a bucket may be tossed to read the
|		 new bucket.  In any case, the requested bucket is
|		 make the "current" bucket and dbf->bucket points to
|		 the correct bucket.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void get_bucket(
     kdbm *dbf,
     int   dir_index)
{
  long   bucket_adr;	/* The address of the correct hash bucket.  */
  int  	i, file_pos;	/* The return address for klseek. */
  register int index;	/* Loop index. */


  /* Initial set up. */
  dbf->bucket_dir = dir_index;
  bucket_adr = dbf->dir [dir_index];
  
  if (dbf->bucket_cache == NULL) {
    if(init_cache(dbf, DEFAULT_CACHESIZE) == -1)
      fatal(dbf, "couldn't init cache");
  }

  /* Is that one is not already current, we must find it. */
  if (dbf->cache_entry->ca_adr != bucket_adr)
    {
      /* Look in the cache. */
      for (index = 0; index < dbf->cache_size; index++) {
	if (dbf->bucket_cache[index].ca_adr == bucket_adr) {
	    dbf->bucket = dbf->bucket_cache[index].ca_bucket;
	    dbf->cache_entry = &dbf->bucket_cache[index];
	    return;
	}
      }

      /* It is not in the cache, read it from the disk. */
      dbf->last_read = (dbf->last_read + 1) % dbf->cache_size;
      if (dbf->bucket_cache[dbf->last_read].ca_changed)
	write_bucket(dbf, &dbf->bucket_cache[dbf->last_read]);
      dbf->bucket_cache[dbf->last_read].ca_adr = bucket_adr;
      dbf->bucket = dbf->bucket_cache[dbf->last_read].ca_bucket;
      dbf->cache_entry = &dbf->bucket_cache[dbf->last_read];
      dbf->cache_entry->ca_data.elem_loc = -1;
      dbf->cache_entry->ca_changed = FALSE;

      /* Read the bucket. */
      file_pos = klseek(dbf->desc, bucket_adr, KSEEK_SET);
      if (file_pos != bucket_adr)
	fatal(dbf, "klseek error");

      if (kread_int(dbf->desc, (int  *) dbf->bucket, 3) != 3)
	fatal(dbf, "kread error");

      if (kread_long(dbf->desc, (long *) dbf->bucket->bucket_avail,
		     BUCKET_AVAIL * 2 ) != (BUCKET_AVAIL * 2 ))
	fatal(dbf, "kread error");

      for (i = 0; i < dbf->header->bucket_elems; i++)
      {
        if (kread_int(dbf->desc, &dbf->bucket->h_table[i].hash_value, 3) != 3)
	  fatal(dbf, "kread error");

        if (kread_long(dbf->desc,
		       &dbf->bucket->h_table[i].data_pointer, 1) != 1)
	  fatal(dbf, "kread error");

	if (kread(dbf->desc, &dbf->bucket->h_table[i].key_start[0],
			SMALL) != SMALL)
	{
	   fatal(dbf, "kread error");
	}
      }
    }
}

/*-----------------------------------------------------------
|
|  Routine Name: get_next_key
|
|       Purpose: Find and read the next entry in the hash structure
|		 for DBF starting at ELEM_LOC of the current bucket
|		 and using RETURN_VAL as the place to put the data
|		 that is found.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int get_next_key(
     kdbm *dbf,
     int elem_loc,
     kdatum *return_val)
{
	int   found;			/* Have we found the next key. */
        data_cache_elem *data_ca;

	/* Find the next key. */
	found = FALSE;
	while (!found)
	{
	   /* Advance to the next location in the bucket. */
	   elem_loc++;
	   if (elem_loc == dbf->header->bucket_elems)
	   {
	      /* We have finished the current bucket, get the next bucket.  */
	      elem_loc = 0;

	      /* Find the next bucket.  It is possible several entries in
		 the bucket directory point to the same bucket. */
	      while (dbf->bucket_dir < dbf->mach_dir_size/sizeof(long) &&
		     dbf->cache_entry->ca_adr == dbf->dir[dbf->bucket_dir])
	      {
		 dbf->bucket_dir++;
	      }

	      /* Check to see if there was a next bucket. */
	      if (dbf->bucket_dir < dbf->mach_dir_size/sizeof(long))
		 get_bucket(dbf, dbf->bucket_dir);	      
	      else
		 /* No next key, just return. */
		 return(FALSE);
	   }
	   found = dbf->bucket->h_table[elem_loc].hash_value != -1;
	}
  
	/* Found the next key, read it into return_val. */
	data_ca = get_entry(dbf, elem_loc);
	if ( !data_ca )
	{
	   return(FALSE);
	}
	return_val->dptr  = data_ca->key;
	return_val->dsize = data_ca->key_size;
	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: write_header
|
|       Purpose: This procedure writes the header back to the file 
|		 described by DBF.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int write_header(
     kdbm *dbf)
{
	int   num, size;      	/* Return value for kwrite.	      */
	kaddr data;		/* Temporary data pointer for writing */
	int   file_pos; 	/* Return value for klseek.	      */
	unsigned char machtype;
	
	file_pos = klseek(dbf->desc, 0L, KSEEK_SET);
	if (file_pos != 0) fatal(dbf, "klseek error");
  
	machtype = (unsigned char)_convert_to_gdb_mach(dbf->header->machtype);
	
	/* Write out the machine type */
	if (kwrite_ubyte(dbf->desc, &machtype, 1) != 1)
        {
           errno = KDBM_FILE_WRITE_ERROR;
           return(FALSE);
        }

	data = (kaddr) &dbf->header->header_magic;
	if (kwrite_int(dbf->desc, data, 6) != 6)
	{
	   errno = KDBM_FILE_WRITE_ERROR;
	   return(FALSE);
	}

	data = (kaddr) &dbf->header->next_block;
	if (kwrite_long(dbf->desc, data, 2) != 2)
	{
	   errno = KDBM_FILE_WRITE_ERROR;
	   return(FALSE);
	}

	data = (kaddr) &dbf->header->avail;
	if (kwrite_int(dbf->desc, data, 2) != 2)
	{
	   errno = KDBM_FILE_WRITE_ERROR;
	   return(FALSE);
	}

	data = (kaddr) &dbf->header->avail.next_block;
	if (kwrite_long(dbf->desc, data, 1) != 1)
	{
	   errno = KDBM_FILE_WRITE_ERROR;
	   return(FALSE);
	}

	data = (kaddr) &dbf->header->avail.av_table;
	if (kwrite_long(dbf->desc, data, 2) != 2)
	{
	   errno = KDBM_FILE_WRITE_ERROR;
	   return(FALSE);
	}

	/* write out rest of av_table */
	klseek(dbf->desc, dbf->size_header, KSEEK_SET);
	num = (dbf->header->avail.size-1) * 2;
	if (kwrite_long(dbf->desc, (long *) &dbf->header->avail.av_table[1],
		num) != num)
	{
	   errno = KDBM_FILE_WRITE_ERROR;
	   return(FALSE);
	}

	size = dbf->header->block_size - dbf->size_header -
	       (kmach_sizeof(dbf->header->machtype,KLONG) * num);
	if (size && klseek(dbf->desc, size, KSEEK_CUR) == -1)
	   fatal(dbf, "klseek error");

	return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: read_header
|
|       Purpose: This procedure reads the header from the file 
|		 described by DBF.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int read_header(
     kdbm *dbf)
{
	/* This is an old database.  Read in the information from the file
	 header and initialize the hash directory. */

	kaddr data;
	kdbm_header temp;  /* For the first part of it. */
	int   size, file_pos;
	float factor;

	if (kread_ubyte(dbf->desc, &temp.machtype, 1) != 1)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return(FALSE);
	}
	temp.machtype = _convert_from_gdb_mach(temp.machtype);
	kfile_setmachtype(dbf->desc, temp.machtype);

	data = (kaddr) &temp.header_magic;
	if (kread_int(dbf->desc, data, 6) != 6)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return(FALSE);
	}

	/* Is the magic number good? */
	if (temp.header_magic != KDBM_MAGIC_NUMBER)
	{
	   errno = KDBM_BAD_MAGIC_NUMBER;
	   if (temp.header_magic == KDBM_MAGIC_NUMBER_OLD)
	   kwarn("kutils", "read_header", "This dbm file '%s' is in a "
		 "pre-Khoros 2.0 form, please run the 'dbcnvrt' program on it.",
		 dbf->name);
	   return(FALSE);
	}

	data = (kaddr) &temp.next_block;
	if (kread_long(dbf->desc, data, 2) != 2)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return(FALSE);
	}

	data = (kaddr) &temp.avail;
	if (kread_int(dbf->desc, data, 2) != 2)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return(FALSE);
	}

	data = (kaddr) &temp.avail.next_block;
	if (kread_long(dbf->desc, data, 1) != 1)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return(FALSE);
	}

	data = (kaddr) &temp.avail.av_table;
	if (kread_long(dbf->desc, data, 2) != 2)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return(FALSE);
	}

	factor = ((float) sizeof(long))/
		  ((float) kmach_sizeof(temp.machtype,KLONG));
	dbf->size_header      = 8 * kmach_sizeof(temp.machtype,KINT) +
				5 * kmach_sizeof(temp.machtype,KLONG) +
				kmach_sizeof(temp.machtype,KBYTE);
	dbf->size_avail_elem  = 2 * kmach_sizeof(temp.machtype,KLONG);
	dbf->size_avail_block = 2 * kmach_sizeof(temp.machtype,KINT) +
				kmach_sizeof(temp.machtype,KLONG) +
				dbf->size_avail_elem;

	/* It is a good database, read the entire header. */
	dbf->header = (kdbm_header *) kmalloc((unsigned int)
					       (factor * temp.block_size));
	if (dbf->header == NULL)
	  return(FALSE);

	kbcopy(&temp, dbf->header, sizeof(kdbm_header));

	/* read in rest of av_table */
	klseek(dbf->desc, dbf->size_header, KSEEK_SET);
	size = (dbf->header->avail.size-1) * 2;
	if (kread_long(dbf->desc, (long *) &dbf->header->avail.av_table[1],
			size) != size)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return(FALSE);
	}

	dbf->mach_dir_size = (int) dbf->header->dir_size * factor;
	dbf->mach_dir      = (long) dbf->header->dir * factor;

	/* Allocate space for the hash table directory.  */
	if ((dbf->dir = (long *) kmalloc(dbf->mach_dir_size)) == NULL)
	   return(FALSE);

	/* Read the hash table directory. */
	file_pos = klseek(dbf->desc, dbf->header->dir, KSEEK_SET);
	if (file_pos != dbf->header->dir)
	{
	   errno = KDBM_FILE_SEEK_ERROR;
	   return FALSE;
	}

	size = dbf->header->dir_size/kmach_sizeof(dbf->header->machtype,KLONG);
	if (kread_long(dbf->desc, dbf->dir, size) != size)
	{
	   errno = KDBM_FILE_READ_ERROR;
	   return FALSE;
	}
	return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: put_av_elem
|
|       Purpose: This routine inserts a single NEW_EL into the
|		 AV_TABLE block in sorted order. This routine does no
|		 I/O.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int put_av_elem(
     avail_elem new_el,
     avail_elem av_table[],
     int *av_count)
{
  int index;			/* For searching through the avail block. */
  int index1;

  /* Is it too small to deal with? */
  if (new_el.av_size <= IGNORE_SIZE)
    return FALSE; 

  /* Search for place to put element.  List is sorted by size. */
  index = 0;
  while (index < *av_count && av_table[index].av_size < new_el.av_size)
    {
      index++;
    }

  /* Move all others up one. */
  index1 = *av_count-1;
  while (index1 >= index)
    {
      av_table[index1+1] = av_table[index1];
      index1--;
    }
  
  /* Add the new element. */
  av_table[index] = new_el;

  /* Increment the number of elements. */
  *av_count += 1;  

  return TRUE;
}

/*-----------------------------------------------------------
|
|  Routine Name: pop_avail_block
|
|       Purpose: Gets the avail block at the top of the stack and
|		 loads it into the active avail block.  It does a
|		 "free" for itself!
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void pop_avail_block(
     kdbm *dbf)
{
  int  size;			/* For use with the kread system call. */
  int  file_pos;		/* For use with the klseek system call. */
  kaddr data;
  avail_elem temp;

  /* Set up variables. */
  temp.av_adr = dbf->header->avail.next_block;
  temp.av_size = ((dbf->header->avail.size-1) * sizeof(avail_elem))
		  + sizeof(avail_block);

  /* Read the block. */
  file_pos = klseek(dbf->desc, temp.av_adr, KSEEK_SET);
  if (file_pos != temp.av_adr)
     fatal(dbf, "klseek error");

  data = (kaddr) &dbf->header->avail;
  if (kread_int(dbf->desc, data, 2) != 2)
     fatal(dbf, "kread error");

  data = (kaddr) &dbf->header->avail.next_block;
  if (kread_long(dbf->desc, data, 1) != 1)
     fatal(dbf, "kread error");

  data = (kaddr) &dbf->header->avail.av_table;
  if (kread_long(dbf->desc, data, 2) != 2)
     fatal(dbf, "kread error");

  size = (dbf->header->avail.size-1) * 2;
  if (kread_long(dbf->desc, (long *) &dbf->header->avail.av_table[1],
		 size) != size)
    fatal(dbf, "kread error");

/* JMAS
  size = temp.av_size/kmach_sizeof(dbf->header->machtype,KINT);
  if (kread_int(dbf->desc, (int *) &dbf->header->avail, size) != size)
     fatal(dbf, "kread error");
*/

  /* We changed the header. */
  dbf->header_changed = TRUE;

  /* Free the previous avail block. */
  put_av_elem(temp, dbf->header->avail.av_table,
		     &dbf->header->avail.count);
}

/*-----------------------------------------------------------
|
|  Routine Name: get_elem
|
|       Purpose: Get_elem returns an element in the AV_TABLE block
|		 which is larger than SIZE.  AV_COUNT is the number of
|		 elements in the AV_TABLE.  If an item is found, it
|		 extracts it from the AV_TABLE and moves the other
|		 elements up to fill the space. If no block is found
|		 larger than SIZE, get_elem returns a size of zero.
|		 This routine does no I/O.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static avail_elem get_elem(
     int size,
     avail_elem av_table[],
     int *av_count)
{
  int index;			/* For searching through the avail block. */
  avail_elem val;		/* The default return value. */

  /* Initialize default return value. */
  val.av_adr = 0;
  val.av_size = 0;

  /* Search for element.  List is sorted by size. */
  index = 0;
  while (index < *av_count && av_table[index].av_size < size)
    {
      index++;
    }

  /* Did we find one of the right size? */
  if (index >= *av_count)
    return val;

  /* Ok, save that element and move all others up one. */
  val = av_table[index];
  *av_count -= 1;
  while (index < *av_count)
    {
      av_table[index] = av_table[index+1];
      index++;
    }

  return val;
}


/*-----------------------------------------------------------
|
|  Routine Name: get_block
|
|       Purpose: Get_block "allocates" new file space and the end of
|		 the file.  This is done in integral block sizes.
|		 (This helps insure that data smaller than one block
|		 size is in a single block.)  Enough blocks are
|		 allocated to make sure the number of bytes allocated
|		 in the blocks is larger than SIZE.  DBF contains the
|		 file header that needs updating.  This routine does
|		 no I/O.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static avail_elem get_block(
     int size,
     kdbm *dbf)
{
  avail_elem val;

  /* Need at least one block. */
  val.av_adr  = dbf->header->next_block;
  val.av_size = dbf->header->block_size;

  /* Get enough blocks to fit the need. */
  while (val.av_size < size)
    val.av_size += dbf->header->block_size;

  /* Update the header and return. */
  dbf->header->next_block += val.av_size;

  /* We changed the header. */
  dbf->header_changed = TRUE;

  return val;
  
}

/*-----------------------------------------------------------
|
|  Routine Name: push_avail_block
|
|       Purpose: Splits the header avail block and pushes half onto
|		 the avail stack.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void push_avail_block(
     kdbm *dbf)
{
  int		size;
  int		av_size;
  long		av_adr;
  int		index;
  int		file_pos;
  kaddr		data;
  avail_block	*temp;
  avail_elem	new_loc;
 

  /* Caclulate the size of the split block. */
  av_size = ((dbf->header->avail.size-1) * sizeof(avail_elem))
            + sizeof(avail_block);

  /* Get address in file for new av_size bytes. */
  new_loc = get_elem(av_size, dbf->header->avail.av_table,
		     &dbf->header->avail.count);

  if (new_loc.av_size == 0)
    new_loc = get_block(av_size, dbf);
  av_adr = new_loc.av_adr;

  /* Split the header block. */
  temp = (avail_block *) kcalloc(1, av_size);
  /* Set the size to be correct AFTER the pop_avail_block. */
  temp->size = dbf->header->avail.size;
  temp->count = 0;
  temp->next_block = dbf->header->avail.next_block;
  dbf->header->avail.next_block = av_adr;
  for (index = 1; index < dbf->header->avail.count; index++)
    if ( (index & 0x1) == 1)	/* Index is odd. */
      temp->av_table[temp->count++] = dbf->header->avail.av_table[index];
    else
      dbf->header->avail.av_table[index>>1]
	= dbf->header->avail.av_table[index];

  /* Update the header avail count to previous size divided by 2. */
  dbf->header->avail.count >>= 1;

  /* Free the unneeded space. */
  new_loc.av_adr += av_size;
  new_loc.av_size -= av_size;
  free_dbm(dbf, new_loc.av_adr, new_loc.av_size);

  /* Update the disk. */
  file_pos = klseek(dbf->desc, av_adr, KSEEK_SET);
  if (file_pos != av_adr)
     fatal(dbf, "klseek error");

  data = (kaddr) &temp;
  if (kwrite_int(dbf->desc, data, 2) != 2)
     fatal(dbf, "kwrite error");

  data = (kaddr) &temp->next_block;
  if (kwrite_long(dbf->desc, data, 1) != 1)
     fatal(dbf, "kwrite error");

  data = (kaddr) &temp->av_table;
  if (kwrite_long(dbf->desc, data, 2) != 2)
     fatal(dbf, "kwrite error");

  size = (temp->size-1) * 2;
  if (kwrite_long(dbf->desc, (long *) &temp->av_table[1], size) != size)
    fatal(dbf, "kwrite error");

/* JMAS
  size = av_size/kmach_sizeof(dbf->header->machtype,KINT);
  if (kwrite_int(dbf->desc, (int *) temp, size) != size)
     fatal(dbf, "kwrite error");
*/
  kfree(temp);
}

/*-----------------------------------------------------------
| 
|  Routine Name: adjust_bucket_avail
|
|       Purpose: When the header already needs writing, we can make
|		 sure the current bucket has its avail block as close
|		 to 1/2 full as possible.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void adjust_bucket_avail(
     kdbm *dbf)
{
  int third = BUCKET_AVAIL / 3;
  avail_elem av_el;

  /* Can we add more entries to the bucket? */
  if (dbf->bucket->av_count < third)
    {
      if (dbf->header->avail.count > 0)
	{
	  dbf->header->avail.count -= 1;
	  av_el = dbf->header->avail.av_table[dbf->header->avail.count];
	  put_av_elem(av_el, dbf->bucket->bucket_avail,
			     &dbf->bucket->av_count);
	  dbf->bucket_changed = TRUE;
	}
      return;
    }

  /* Is there too much in the bucket? */
  while (dbf->bucket->av_count > BUCKET_AVAIL-third
	 && dbf->header->avail.count < dbf->header->avail.size)
    {
      av_el = get_elem(0, dbf->bucket->bucket_avail, &dbf->bucket->av_count);
      put_av_elem(av_el, dbf->header->avail.av_table,
			 &dbf->header->avail.count);
      dbf->bucket_changed = TRUE;
    }
}

/*-----------------------------------------------------------
|
|  Routine Name: new_bucket
|
|       Purpose: Initializing a new hash buckets sets all bucket
|		 entries to -1 hash value.
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void new_bucket(
     kdbm *dbf,
     hash_bucket *bucket,
     int bits)
{
  int index;
  
  /* Initialize the avail block. */
  bucket->av_count = 0;

  /* Set the information fields first. */
  bucket->bucket_bits = bits;
  bucket->count = 0;
  
  /* Initialize all bucket elements. */
  for (index = 0; index < dbf->header->bucket_elems; index++)
    bucket->h_table[index].hash_value = -1;
}

/*-----------------------------------------------------------
|
|  Routine Name: split_bucket
|
|       Purpose: Split the current bucket.  This includes moving all
|		 items in the bucket to a new bucket.  This doesn't
|		 require any disk reads because all hash values are
|		 stored in the buckets.  Splitting the current bucket
|		 may require doubling the size of the hash directory.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void split_bucket(
     kdbm *dbf,
     int   next_insert)
{
  hash_bucket   *bucket[2]; 	/* Pointers to the new buckets. */

  int           new_bits;	/* The number of bits for the new buckets. */
  int	        cache_0;	/* Location in the cache for the buckets. */
  int	        cache_1;
  long          adr_0;		/* File address of the new bucket 0. */
  long          adr_1;		/* File address of the new bucket 1. */
  avail_elem    old_bucket;	/* Avail Struct for the old bucket. */

  long          dir_start0;	/* Used in updating the directory. */
  long          dir_start1;
  long          dir_end;

  long          *new_dir;	/* Pointer to the new directory. */
  long          dir_adr;	/* Address of the new directory. */
  int           dir_size;	/* Size of the new directory. */
  long          old_adr[31]; 	/* Address of the old directories. */
  long          old_size[31]; 	/* Size of the old directories. */
  int	        old_count;	/* Number of old directories. */

  int            index;		/* Used in array indexing. */
  int            index1;	/* Used in array indexing. */
  int            elem_loc;	/* Location in new bucket to put element. */
  bucket_element *old_el;	/* Pointer into the old bucket. */
  int	         select;	/* Used to index bucket during movement. */
  float		 factor;


  /* No directories are yet old. */
  old_count = 0;

  if (dbf->bucket_cache == NULL) {
    if(init_cache(dbf, DEFAULT_CACHESIZE) == -1)
      fatal(dbf, "couldn't init cache");
  }

  while (dbf->bucket->count == dbf->header->bucket_elems)
    {
      /* Initialize the "new" buckets in the cache. */
      do
	{
	  dbf->last_read = (dbf->last_read + 1) % dbf->cache_size;
	  cache_0 = dbf->last_read;
	}      
      while (dbf->bucket_cache[cache_0].ca_bucket == dbf->bucket);
      bucket[0] = dbf->bucket_cache[cache_0].ca_bucket;
      if (dbf->bucket_cache[cache_0].ca_changed)
	write_bucket(dbf, &dbf->bucket_cache[cache_0]);
      do
	{
	  dbf->last_read = (dbf->last_read + 1) % dbf->cache_size;
	  cache_1 = dbf->last_read;
	}      
      while (dbf->bucket_cache[cache_1].ca_bucket == dbf->bucket);
      bucket[1] = dbf->bucket_cache[cache_1].ca_bucket;
      if (dbf->bucket_cache[cache_1].ca_changed)
	write_bucket(dbf, &dbf->bucket_cache[cache_1]);
      new_bits = dbf->bucket->bucket_bits+1;
      new_bucket(dbf, bucket[0], new_bits);
      new_bucket(dbf, bucket[1], new_bits);
      adr_0 = alloc_dbm(dbf, dbf->header->bucket_size); 
      dbf->bucket_cache[cache_0].ca_adr = adr_0;
      adr_1 = alloc_dbm(dbf, dbf->header->bucket_size);
      dbf->bucket_cache[cache_1].ca_adr = adr_1;

      
      
      /* Double the directory size if necessary. */
      if (dbf->header->dir_bits == dbf->bucket->bucket_bits)
	{
	  dir_size = dbf->mach_dir_size * 2;
	  dir_adr  = alloc_dbm(dbf, dir_size);

	  new_dir  = (long *) kcalloc(1, dir_size);
	  if (new_dir == NULL) fatal(dbf, "malloc error");
	  for (index = 0;
	  	index < dbf->mach_dir_size/sizeof(long); index++)
	    {
	      new_dir[2*index]   = dbf->dir[index];
	      new_dir[2*index+1] = dbf->dir[index];
	    }
	  
	  /* Update header. */
	  old_adr[old_count] = dbf->mach_dir;
	  dbf->mach_dir = dir_adr;
	  old_size[old_count] = dbf->mach_dir_size;
	  dbf->mach_dir_size = dir_size;
	  dbf->header->dir_bits = new_bits;
	  old_count++;
	  
	  /* Now update dbf.  */
	  dbf->header_changed = TRUE;
	  dbf->bucket_dir *= 2;
	  kfree(dbf->dir);
	  dbf->dir = new_dir;
	}

      factor = ((float) kmach_sizeof(dbf->header->machtype,KLONG))/
		((float) sizeof(long));
      dbf->header->dir_size = (int) dbf->mach_dir_size * factor;
      dbf->header->dir      = (long) dbf->mach_dir * factor;


      /* Copy all elements in dbf->bucket into the new buckets. */
      for (index = 0; index < dbf->header->bucket_elems; index++)
	{
	  old_el = & (dbf->bucket->h_table[index]);
	  select = (old_el->hash_value >> (31-new_bits)) & 1;
	  elem_loc = old_el->hash_value % dbf->header->bucket_elems;
	  while (bucket[select]->h_table[elem_loc].hash_value != -1)
	    elem_loc = (elem_loc + 1) % dbf->header->bucket_elems;
	  bucket[select]->h_table[elem_loc] = *old_el;
	  bucket[select]->count += 1;
	}
      
      /* Allocate avail space for the bucket[1]. */
      bucket[1]->bucket_avail[0].av_adr
	= alloc_dbm(dbf, dbf->header->block_size);
      bucket[1]->bucket_avail[0].av_size = dbf->header->block_size;
      bucket[1]->av_count = 1;
      
      /* Copy the avail elements in dbf->bucket to bucket[0]. */
      bucket[0]->av_count = dbf->bucket->av_count;
      index = 0;
      index1 = 0;
      if (bucket[0]->av_count == BUCKET_AVAIL)
	{
	  /* The avail is full, move the first one to bucket[1]. */
	  put_av_elem(dbf->bucket->bucket_avail[0],
			     bucket[1]->bucket_avail,
			     &bucket[1]->av_count);
	  index = 1;
	  bucket[0]->av_count --;
	}
      for (; index < dbf->bucket->av_count; index++)
	{
	  bucket[0]->bucket_avail[index1++] = dbf->bucket->bucket_avail[index];
	}
      
      /* Update the directory.  We have new file addresses for both buckets. */
      dir_start1 = (dbf->bucket_dir >> (dbf->header->dir_bits - new_bits)) | 1;
      dir_end = (dir_start1 + 1) << (dbf->header->dir_bits - new_bits);
      dir_start1 = dir_start1 << (dbf->header->dir_bits - new_bits);
      dir_start0 = dir_start1 - (dir_end - dir_start1);
      for (index = dir_start0; index < dir_start1; index++)
	dbf->dir[index] = adr_0;
      for (index = dir_start1; index < dir_end; index++)
	dbf->dir[index] = adr_1;
      
      
      /* Set changed flags. */
      dbf->bucket_cache[cache_0].ca_changed = TRUE;
      dbf->bucket_cache[cache_1].ca_changed = TRUE;
      dbf->bucket_changed = TRUE;
      dbf->directory_changed = TRUE;
      dbf->second_changed = TRUE;
      
      /* Update the cache! */
      dbf->bucket_dir = next_insert >> (31-dbf->header->dir_bits);
      
      /* Invalidate old cache entry. */
      old_bucket.av_adr  = dbf->cache_entry->ca_adr;
      old_bucket.av_size = dbf->header->bucket_size;
      dbf->cache_entry->ca_adr = 0;
      dbf->cache_entry->ca_changed = FALSE;
      
      /* Set dbf->bucket to the proper bucket. */
      if (dbf->dir[dbf->bucket_dir] == adr_0)
	{
	  dbf->bucket = bucket[0];
	  dbf->cache_entry = &dbf->bucket_cache[cache_0];
	  put_av_elem(old_bucket,
			     bucket[1]->bucket_avail,
			     &bucket[1]->av_count);
	}
      else
	{
	  dbf->bucket = bucket[1];
	  dbf->cache_entry = &dbf->bucket_cache[cache_1];
	  put_av_elem(old_bucket,
			     bucket[0]->bucket_avail,
			     &bucket[0]->av_count);
	}
      
    }

  /* Get rid of old directories. */
  for (index = 0; index < old_count; index++)
    free_dbm(dbf, old_adr[index], old_size[index]);
}


/*-----------------------------------------------------------
|
|  Routine Name: findkey - The routine that finds a key entry 
|			in the file.
|
|       Purpose: Find the KEY in the file and get ready to read the
|		 associated data.  The return value is the location in
|		 the current hash bucket of the KEY's entry.  If it is
|		 found, a pointer to the data and the key are returned
|		 in DPTR.  If it is not found, the value -1 is
|		 returned.  Since find key computes the hash value of
|		 key, that value
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int findkey(
     kdbm   *dbf,
     kdatum key,
     char    **dptr,
     int     *hash)		/* The new hash value. */
{
	int    bucket_hash;	/* The hash value from the bucket. */

	int    elem_loc;	/* The location in the bucket. */
	int    home_loc;	/* The home location in the bucket. */
	int    key_size;	/* Size of the key on the file.  */
	int    value;		/* Size of the key on the file.  */
        data_cache_elem *data_ca;

	/* Compute hash value and load proper bucket.  */
	value = hash_dbm(key);
	if (hash) *hash = value;
	get_bucket(dbf, value >> (31-dbf->header->dir_bits));

	/* Is the element the last one found for this bucket? */
	if (dbf->cache_entry->ca_data.elem_loc != -1        &&
	    value == dbf->cache_entry->ca_data.hash         &&
	    dbf->cache_entry->ca_data.key_size == key.dsize &&
	    kmemcmp(dbf->cache_entry->ca_data.key, key.dptr, key.dsize) == 0)
	{
	   /* This is it. Return the cache pointer. */
	   elem_loc = dbf->cache_entry->ca_data.elem_loc;
	   if (dptr)
	   {
	      data_ca = read_entry(dbf, elem_loc);
	      *dptr = data_ca->data;
	   }
	   return(elem_loc);
	}
	    
	/* It is not the cached value, search for element in the bucket. */
	elem_loc = value % dbf->header->bucket_elems;
	home_loc = elem_loc;
	bucket_hash = dbf->bucket->h_table[elem_loc].hash_value;
	while (bucket_hash != -1)
	{
	   key_size = dbf->bucket->h_table[elem_loc].key_size;
	   if (bucket_hash != value || key_size != key.dsize /*||
	       kmemcmp(dbf->bucket->h_table[elem_loc].key_start, key.dptr,
			(SMALL < key_size ? SMALL : key_size)) != 0 JMAS*/) 
	   {
	      /* Current elem_loc is not the item, go to next item. */
	      elem_loc = (elem_loc + 1) % dbf->header->bucket_elems;
	      if (elem_loc == home_loc)
		 return(-1);

	      bucket_hash = dbf->bucket->h_table[elem_loc].hash_value;
	   }
	   else
	   {
	      /* This may be the one we want.  The only way to tell is to
	         read it. */
	      if ((data_ca = get_entry(dbf, elem_loc)) != NULL &&
		  kmemcmp(data_ca->key, key.dptr, key_size) == 0)
	      {
	         /* This is the item. */
		 if (dptr)
		 {
		    data_ca = read_entry(dbf, elem_loc);
		    *dptr = data_ca->data;
		 }
		 return(elem_loc);
	      }
	      else
	      {
	         /* Not the item, try the next one.  Return if not found. */
		 elem_loc = (elem_loc + 1) % dbf->header->bucket_elems;
		 if (elem_loc == home_loc) return -1;
		 bucket_hash = dbf->bucket->h_table[elem_loc].hash_value;
	      }
	   }
	}

	/* If we get here, we never found the key. */
	return(-1);
}


/*-----------------------------------------------------------
|
|  Routine Name: end_update
|
|       Purpose: After all changes have been made in memory, we now
|		 write them all to disk.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void end_update(
     kdbm *dbf)
{
  int  size;		/* Return value for kwrite. */
  int  file_pos;	/* Return value for klseek. */
  
  
  /* Write the current bucket. */
  if (dbf->bucket_changed && (dbf->cache_entry != NULL)) {
      write_bucket(dbf, dbf->cache_entry);
      dbf->bucket_changed = FALSE;
  }

  /* Write the other changed buckets if there are any. */
  if (dbf->second_changed) {
    if(dbf->bucket_cache != NULL) {
      register int index;

      for (index = 0; index < dbf->cache_size; index++) {
	if (dbf->bucket_cache[index].ca_changed) {
	    write_bucket(dbf, &dbf->bucket_cache[index]);
	}
      }
    }
    dbf->second_changed = FALSE;
  }
  
  /* Write the directory. */
  if (dbf->directory_changed)
    {
      file_pos = klseek(dbf->desc, dbf->header->dir, KSEEK_SET);
      if (file_pos != dbf->header->dir) fatal(dbf, "klseek error");

      size = dbf->header->dir_size/kmach_sizeof(dbf->header->machtype,KLONG);
      if (kwrite_long(dbf->desc, dbf->dir, size) != size)
	fatal(dbf, "kwrite error");
      dbf->directory_changed = FALSE;
    }

  /* Final write of the header. */
  if (dbf->header_changed)
    {
      write_header(dbf);
      dbf->header_changed = FALSE;
    }
}

/*-----------------------------------------------------------
|
|  Routine Name: hash_dbm - The kdbm hash function.
|
|       Purpose: This hash function computes a 31 bit value.  The
|		 value is used to index the hash directory using the
|		 top n bits.  It is also used in a hash bucket to find
|		 the home position of the element by taking the value
|		 modulo the bucket hash table size.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int hash_dbm(
     kdatum key)
{
  unsigned int value;	/* Used to compute the hash value.  */
  int   index;		/* Used to cycle through random values. */


  /* Set the initial value from key. */
  value = 0x238F13AF * key.dsize;
  for (index = 0; index < key.dsize; index++)
    value = (value + (key.dptr[index] << (index*5 % 24))) & 0x7FFFFFFF;

  value = (1103515243 * value + 12345) & 0x7FFFFFFF;  

  /* Return the value. */
  return((int) value);
}

/*-----------------------------------------------------------
|
|  Routine Name: alloc_dbm
|
| 	Purpose: Allocate space in the file DBF for a block NUM_BYTES
|		 in length.  Return the file address of the start of
|		 the block.
|
|		 Each hash bucket has a fixed size avail table.  We
|		 first check this avail table to satisfy the request
|		 for space.  In most cases we can and this causes
|		 changes to be only in the current hash bucket.
|		 Allocation is done on a first fit basis from the
|		 entries.  If a request can not be satisfied from the
|		 current hash bucket, then it is satisfied from the
|		 file header avail block.  If nothing is there that
|		 has enough space, another block at the end of the
|		 file is allocated and the unused portion is returned
|		 to the avail block.  This routine "guarantees" that
|		 an allocation does not cross a block boundary unless
|		 the size is larger than a single block.  The avail
|		 structure is changed by this routine if a change is
|		 needed.  If an error occurs, the value of 0 will be
|		 returned.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static int alloc_dbm(
     kdbm *dbf,
     int num_bytes)
{
  int        file_adr;		/* The address of the block. */
  avail_elem av_el;		/* For temporary use. */

  /* The current bucket is the first place to look for space. */
  av_el = get_elem(num_bytes, dbf->bucket->bucket_avail,
		    &dbf->bucket->av_count);

  /* If we did not find some space, we have more work to do. */
  if (av_el.av_size == 0)
    {
      if ((dbf->header->avail.count == 0)
	  && (dbf->header->avail.next_block != 0))
	pop_avail_block(dbf);
      
      av_el = get_elem (num_bytes, dbf->header->avail.av_table,
		  &dbf->header->avail.count);

      if (av_el.av_size == 0)
	av_el = get_block (num_bytes, dbf);

      dbf->header_changed = TRUE;
    }

  /* We now have the place from which we will allocate the new space. */
  file_adr = av_el.av_adr;

  /* Put the unused space back in the avail block. */
  av_el.av_adr += num_bytes;
  av_el.av_size -= num_bytes;
  free_dbm(dbf, av_el.av_adr, av_el.av_size);

  /* Return the address. */
  return file_adr;
  
}

/*-----------------------------------------------------------
|
|  Routine Name: free_dbm
|
|       Purpose: Free space of size NUM_BYTES in the file DBF at file
|		 address FILE_ADR.  Make it avaliable for reuse
|		 through alloc.  This routine changes the avail
|		 structure.  The value TRUE is returned if there were
|		 errors.  If no errors occured, the value FALSE is
|		 returned.
|
|         Input: 
|
|        Output: 
|
|       Returns: TRUE (1) on success, FALSE (0) otherwise
|
|    Written By: Philip A. Nelson
|          Date: Nov 08, 1993
| Modifications:
|
------------------------------------------------------------*/

static void free_dbm(
     kdbm *dbf,
     long file_adr,
     long num_bytes)
{
  avail_elem temp;

  /* Is it too small to worry about? */
  if (num_bytes <= IGNORE_SIZE)
    return;

  /* Initialize the avail element. */
  temp.av_size = num_bytes;
  temp.av_adr = file_adr;

  /* Is the freed space large or small? */
  if (num_bytes >= dbf->header->block_size)
    {
      if (dbf->header->avail.count == dbf->header->avail.size)
	{
	  push_avail_block(dbf);
	}
      put_av_elem(temp, dbf->header->avail.av_table,
			 &dbf->header->avail.count);
      dbf->header_changed = TRUE;
    }
  else
    {
      /* Try to put into the current bucket. */
      if (dbf->bucket->av_count < BUCKET_AVAIL)
	put_av_elem(temp, dbf->bucket->bucket_avail,
			   &dbf->bucket->av_count);
      else
	{
	  if (dbf->header->avail.count == dbf->header->avail.size)
	    {
	      push_avail_block(dbf);
	    }
	  put_av_elem(temp, dbf->header->avail.av_table,
			     &dbf->header->avail.count);
	  dbf->header_changed = TRUE;
	}
    }

  if (dbf->header_changed)
    adjust_bucket_avail(dbf);

  /* All work is done. */
  return;
}


/************************************************************
*
*  Routine Name: kdbm_fdopen - Open the dbm file and initialize data 
*			     structures for use.
*       Purpose: Initialize dbm system.  kfile is a pointer to the file
*		 name.  If the file has a size of zero bytes, a file
*		 initialization procedure is performed, setting up the
*		 initial structure in the file.
*
*		 If "flags" is set to KOPEN_RDONLY the user wants to just
*		 read the database and any call to dbm_store or dbm_delete
*		 will fail. Many readers can access the database at the
*		 same time.  If "flags" is set to KOPEN_WRONLY, the user wants
*		 both read and write access to the database and requires
*		 exclusive access.  If "flags" is KOPEN_WRONLY|KOPEN_CREAT, the
*		 user wants both read and write access to the database and
*		 if the database does not exist, create a new one.  If
*		 "flags" is KOPEN_WRONLY|KOPEN_CREAT|KOPEN_TRUNC, the user want
*		 a new database created, regardless of whether one existed, and
*		 wants read and write access to the new database.  Any error
*		 detected will cause a return value of null and an
*		 approprate value will be in errno.  If no errors
*		 occur, a pointer to the "kdbm file descriptor" will be
*		 returned.
*         Input: filename - file name of database to open
*		 flags    - open flags used during the kopen system call.
*		 mode     - access modes to be used when creating a new database
*       Returns: a pointer to a open database structure
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

kdbm *kdbm_fdopen(
     int  fid,
     int  flags,
     int  mode)
{
	kdbm *dbf;		/* The record to return.	*/
	int   len;		/* Length of the file.          */
	int   num_bytes;	/* Used in reading and writing. */
	int   index;		/* Used as a loop index. */
	int   i, size;


	/* Allocate new info structure. */
	if ((dbf = (kdbm *) kcalloc(1, sizeof(kdbm))) == NULL)
	   return(NULL);

	/* Save name of file. */
	dbf->name = kfile_filename(fid);

	/* Set the default return value for not finding a first entry. */
	dbf->keyptr.dsize = 0;
	dbf->keyptr.dptr  = NULL;

	/* Open the file. */
	if (flags & KOPEN_WRONLY)
	   flags = (flags & ~KOPEN_WRONLY) | KOPEN_RDWR;

	dbf->desc = fid;
	flags = kfile_flags(fid);
	if (flags & KOPEN_WRONLY)
	{
	   kfree(dbf);
	   return(NULL);
	}

	/* Get the status of the file. */
	len  = klseek(dbf->desc, 0, KSEEK_END);
	(void) klseek(dbf->desc, 0, KSEEK_SET);

	/* Lock the file in the approprate way. */
	if ((flags & KOPEN_RDWR) == 0)
	{
	   if (len == 0)
	   {
	      kclose(dbf->desc);
	      kfree(dbf);
	      errno = KDBM_EMPTY_DATABASE;
	      return(NULL);
	   }
	}

	/* Record the kind of user. */
	dbf->rdonly = ((flags & KOPEN_RDWR) == 0);

	/* Decide if this is a new file or an old file. */
	if (len == 0)
	{
	   /* Get space for the file header. */
	   if ((dbf->header = (kdbm_header *) kcalloc(1, STATBLKSIZE)) == NULL)
	   {
	      dbf->desc = -1;
	      kdbm_close(dbf);
	      return(NULL);
	   }

	   /* Set the magic number and the block_size. */
	   dbf->header->header_magic = KDBM_MAGIC_NUMBER;
	   dbf->header->block_size   = STATBLKSIZE;
	   dbf->header->machtype     = kmach_type(NULL);
	   dbf->size_header	     = 8 * sizeof(int) + 5 * sizeof(long) +
				       sizeof(char);
	   dbf->size_avail_elem      = 2 * sizeof(long);
	   dbf->size_avail_block     = 2 * sizeof(int) + sizeof(long) +
				       dbf->size_avail_elem;



	   /* Create the initial hash table directory.  */
	   dbf->header->dir_size = 8 * sizeof(long);
	   dbf->header->dir_bits = 3;
	   while (dbf->header->dir_size < dbf->header->block_size)
	   {
	      dbf->header->dir_size <<= 1;
	      dbf->header->dir_bits += 1;
	   }

	   /* Check for correct block_size. */
	   if (dbf->header->dir_size != dbf->header->block_size)
	   {
	      dbf->desc = -1;
	      kdbm_close(dbf);
	      errno = KDBM_BLOCK_SIZE_ERROR;
	      return(NULL);
	   }

	   /* Allocate the space for the directory. */
	   if ((dbf->dir = (long *) kcalloc(1, dbf->header->dir_size)) == NULL)
	   {
	      dbf->desc = -1;
	      kdbm_close(dbf);
	      return(NULL);
	   }
	   dbf->header->dir = dbf->header->block_size;
           dbf->mach_dir_size	     = dbf->header->dir_size;
	   dbf->mach_dir             = dbf->header->dir;

	   /* Create the first and only hash bucket. */
	   dbf->header->bucket_elems = 48;
	   dbf->header->bucket_size  = dbf->header->block_size;

/* used to be alloca */
/* get me a bucket! ;^) */
	   if ((dbf->bucket = (hash_bucket *)  kcalloc(1, STATBLKSIZE)) == NULL)
	   {
	      dbf->desc = -1;
	      kdbm_close(dbf);
	      return(NULL);
	   }
	   new_bucket(dbf, dbf->bucket, 0);
	   dbf->bucket->av_count = 1;
	   dbf->bucket->bucket_avail[0].av_adr  = 3*dbf->header->block_size;
	   dbf->bucket->bucket_avail[0].av_size = dbf->header->block_size;

	   /* Set table entries to point to hash buckets. */
	   for (index = 0; index < dbf->header->dir_size/sizeof(long); index++)
	      dbf->dir[index] = 2*dbf->header->block_size;

	   /* Initialize the active avail block. */
	   dbf->header->avail.size = 122;
	   dbf->header->avail.count = 0;
	   dbf->header->avail.next_block = 0;
	   dbf->header->next_block  = 4*dbf->header->block_size;

	   /* Write initial configuration to the file. */
	   /* Block 0 is the file header and active avail block. */
	   if (!write_header(dbf))
	   {
	      dbf->desc = -1;
	      kdbm_close(dbf);
	      errno = KDBM_FILE_WRITE_ERROR;
	      return NULL;
	   }

	   /* Block 1 is the initial bucket directory. */
	   num_bytes = kwrite_long(dbf->desc, dbf->dir,
				   dbf->header->dir_size/sizeof(long));
	   if (num_bytes != dbf->header->dir_size/sizeof(long))
	   {
	      dbf->desc = -1;
	      kdbm_close(dbf);
	      errno = KDBM_FILE_WRITE_ERROR;
	      return(NULL);
	   }

	   /* Block 2 is the only bucket. */
           if (kwrite_int(dbf->desc, (int  *) dbf->bucket, 3) != 3)
             fatal(dbf, "kwrite error");

           if (kwrite_long(dbf->desc, (long *) &dbf->bucket->bucket_avail,
		  BUCKET_AVAIL * 2) != (BUCKET_AVAIL * 2))
             fatal(dbf, "kwrite error");

           for (i = 0; i < dbf->header->bucket_elems; i++)
           {
              if (kwrite_int(dbf->desc,
			     &dbf->bucket->h_table[i].hash_value, 3) != 3)
	        fatal(dbf, "kwrite error");

              if (kwrite_long(dbf->desc,
			      &dbf->bucket->h_table[i].data_pointer, 1) != 1)
	        fatal(dbf, "kwrite error");

              if (kwrite(dbf->desc, &dbf->bucket->h_table[i].key_start[0],
			SMALL) != SMALL)
	      {
	           fatal(dbf, "kwrite error");
	      }
	  }

	  /*
	   * do dbf->header->bucket_elems-1 since sizeof(hash_bucket) includes
	   * one h_table
	   */
	  size = dbf->header->bucket_size - sizeof(hash_bucket) -
		 (dbf->header->bucket_elems-1)*(3*sizeof(int) +
		 sizeof(long) + sizeof(char) * SMALL);

	  if (size && klseek(dbf->desc, size, KSEEK_CUR) == -1)
	     fatal(dbf, "klseek error");
	}
	else
	{
	   if (read_header(dbf) == FALSE)
	   {
	      dbf->desc = -1;
	      kdbm_close(dbf);
	      return(NULL);
	   }
	}

	/* Finish initializing dbf. */
	dbf->last_read = -1;
	dbf->bucket = NULL;
	dbf->bucket_dir = 0;
	dbf->cache_entry = NULL;
	dbf->header_changed = FALSE;
	dbf->directory_changed = FALSE;
	dbf->bucket_changed = FALSE;
	dbf->second_changed = FALSE;
	
	/* Everything is fine, return the pointer to the file
	   information structure.  */
	return(dbf);
}

/************************************************************
*
*  Routine Name: kdbm_open - Open the dbm file and initialize data 
*			     structures for use.
*       Purpose: Initialize dbm system.  FILE is a pointer to the file
*		 name.  If the file has a size of zero bytes, a file
*		 initialization procedure is performed, setting up the
*		 initial structure in the file.
*
*		 If "flags" is set to KOPEN_RDONLY the user wants to just
*		 read the database and any call to dbm_store or dbm_delete
*		 will fail. Many readers can access the database at the
*		 same time.  If "flags" is set to KOPEN_WRONLY, the user wants
*		 both read and write access to the database and requires
*		 exclusive access.  If "flags" is KOPEN_WRONLY|KOPEN_CREAT, the
*		 user wants both read and write access to the database and
*		 if the database does not exist, create a new one.  If
*		 "flags" is KOPEN_WRONLY|KOPEN_CREAT|KOPEN_TRUNC, the user want
*		 a new database created, regardless of whether one existed, and
*		 wants read and write access to the new database.  Any error
*		 detected will cause a return value of null and an
*		 approprate value will be in errno.  If no errors
*		 occur, a pointer to the "kdbm file descriptor" will be
*		 returned.
*         Input: filename - file name of database to open
*		 flags    - open flags used during the kopen system call.
*		 mode     - access modes to be used when creating a new database
*       Returns: a pointer to a open database structure
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

kdbm *kdbm_open(
     char *filename,
     int  flags,
     int  mode)
{
	int   fid;		/* The open khoros file descriptor */
	kdbm *dbf;		/* The newly opened database file  */

	/* Open the file. */
	if (flags & KOPEN_WRONLY)
	   flags = (flags & ~KOPEN_WRONLY) | KOPEN_RDWR;

	if ((fid = kopen(filename, flags, mode)) < 0)
	   return(NULL);

	if ((dbf = kdbm_fdopen(fid, flags, mode)) == NULL)
	{
	   kclose(fid);
	   return(NULL);
	}
	return(dbf);
}

/************************************************************
*
*  Routine Name: kdbm_close - close a previously opened kdbm file
*       Purpose: Close the dbm file and free all memory associated with the
*                file DBF.  Before freeing members of DBF, check and make 
*		 sure that they were allocated.
*         Input: dbf - open database pointer to be closed
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

void kdbm_close(
     kdbm *dbf)
{
	register int index;	/* For freeing the bucket cache. */


	if (!dbf)
	   return;

	/* Close the file and free all malloced memory. */
	end_update(dbf);
	if (dbf->desc != -1)
	   kclose(dbf->desc);

	if (dbf->dir != NULL) kfree(dbf->dir);

	if (dbf->bucket_cache != NULL)
	{
	   for (index = 0; index < dbf->cache_size; index++)
	   {
	      kfree(dbf->bucket_cache[index].ca_bucket);
	      kfree(dbf->bucket_cache[index].ca_data.data);
	      kfree(dbf->bucket_cache[index].ca_data.key);
	   }
	   kfree(dbf->bucket_cache);
	}
	if (dbf->header != NULL) kfree(dbf->header);
	kfree(dbf);
}

/************************************************************
*
*  Routine Name: kdbm_fetch - Find a key and return the associated data.
*       Purpose: Look up a given KEY and return the information 
*                associated with that KEY.  The pointer in the 
*		 structure that is  returned is a pointer to dynamically
*		 allocated memory block.
*         Input: dbf - open database pointer
*		 key - database key to search for
*       Returns: The data associated with the specified key.  If the key
*		 is not a part of the database, the returned data.dsize 
*		 and data.dptr will be 0 and NULL respectively
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

kdatum kdbm_fetch(
     kdbm *dbf,
     kdatum key)
{
	kdatum  return_val;		/* The return value. */
	int    elem_loc;		/* The location in the bucket. */
	char   *find_data;		/* Returned from find_key. */

	/* Set the default return value. */
	return_val.dsize = 0;
	return_val.dptr  = NULL;

	if (!dbf)
	   return(return_val);

	/* Find the key and return a pointer to the data. */
	elem_loc = findkey(dbf, key, &find_data, NULL);

	/* Copy the data if the key was found.  */
	if (elem_loc >= 0)
	{
	   /* This is the item.  Return the associated data. */
	   return_val.dptr = find_data;
	   return_val.dsize = dbf->bucket->h_table[elem_loc].data_size;
	}

	/* Check for an error and return. */
	if (return_val.dptr == NULL) errno = KDBM_ITEM_NOT_FOUND;
	return(return_val);
}

/************************************************************
*
*  Routine Name: kdbm_store - Add a new key/data pair to the database.
*       Purpose: Add a new element to the database.  CONTENT is keyed
*		 by KEY.  The file on disk is updated to reflect the
*		 structure of the new database before returning from
*		 this procedure.  The FLAGS define the action to take
*		 when the KEY is already in the database.  The value
*		 KDBM_REPLACE asks that the old data be replaced by
*		 the new CONTENT.  The value KDBM_INSERT asks that an
*		 error be returned and no action taken.  A return
*		 value of 0 means no errors.  A return value of -1
*		 means that the item was not stored in the data base
*		 because the caller was not an official writer. A
*		 return value of 0 means that the item was not stored
*		 because the argument FLAGS was KDBM_INSERT and the
*		 KEY was already in the database.
*         Input: dbf     - open database pointer
*		 key     - key to store the information under
*		 content - data to store in the database
*		 flags   - data overwrite options
*       Returns: 0 on success, -1 otherwise
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

int kdbm_store(
     kdbm *dbf,
     kdatum key,
     kdatum content,
     int flags)
{
	int  hash;		/* The new hash value. */
	int  elem_loc;		/* The location in hash bucket. */
	int  file_adr;		/* The address of new space in the file.  */
	int  file_pos;		/* The position after a klseek. */
	int  num_bytes;		/* Used for error detection. */
	long free_adr;		/* For keeping track of a freed section. */
	long free_size;

	int  new_size;		/* Used in allocating space. */


	/* First check to make sure this guy is a writer. */
	if (!dbf || dbf->rdonly == TRUE)
	{
	   errno = KDBM_READER_CANT_STORE;
	   return(-1);
	}

	/* Check for illegal data values.  A NULL dptr field is illegal because
	   NULL dptr returned by a lookup procedure indicates an error. */
	if ((key.dptr == NULL) || (content.dptr == NULL))
	{
	   errno = KDBM_ILLEGAL_DATA;
	   return(-1);
	}


	/* Look for the key in the file.  A side effect loads the correct
	   bucket and calculates the hash value. */
	if ((elem_loc = findkey(dbf, key, NULL, &hash)) != -1)
	{
	   if (flags == KDBM_REPLACE)
	   {
	      /* Just replace the data. */
	      free_adr  = dbf->bucket->h_table[elem_loc].data_pointer;
	      free_size = dbf->bucket->h_table[elem_loc].key_size +
	                  dbf->bucket->h_table[elem_loc].data_size;
	      free_dbm(dbf, free_adr, free_size);
	   }
	   else
	   {
	      errno = KDBM_CANNOT_REPLACE;
	      return(1);
	   }
	}


	/* Get the file address for the new space.  (Current bucket's free
	   space is first place to look.) */
	new_size = key.dsize+content.dsize;
	file_adr = alloc_dbm(dbf, new_size);

	/* If this is a new entry in the bucket, we need to do special
	   things. */
	if (elem_loc == -1)
	{
	   if (dbf->bucket->count == dbf->header->bucket_elems)
	   {
	      /* Split the current bucket. */
	      split_bucket(dbf, hash);
	   }
	    
	   /* Find space to insert into bucket and set elem_loc to that
	      place. */
	   elem_loc = hash % dbf->header->bucket_elems;
	   while (dbf->bucket->h_table[elem_loc].hash_value != -1)
	       elem_loc = (elem_loc + 1) % dbf->header->bucket_elems;

	   /* We now have another element in the bucket.  Add
	      the new information.*/
	   dbf->bucket->count += 1;
	   dbf->bucket->h_table[elem_loc].hash_value = hash;
	   kbcopy(key.dptr, dbf->bucket->h_table[elem_loc].key_start,
	       (SMALL < key.dsize ? SMALL : key.dsize));
	}

	/* Update current bucket data pointer and sizes. */
	dbf->bucket->h_table[elem_loc].data_pointer = file_adr;
	dbf->bucket->h_table[elem_loc].key_size = key.dsize;
	dbf->bucket->h_table[elem_loc].data_size = content.dsize;

	/* Write the data to the file. */
	file_pos = klseek(dbf->desc, file_adr, KSEEK_SET);
	if (file_pos != file_adr) fatal(dbf, "klseek error");
	num_bytes = kwrite(dbf->desc, key.dptr, key.dsize);
	if (num_bytes != key.dsize) fatal(dbf, "kwrite error");
	num_bytes = kwrite(dbf->desc, content.dptr, content.dsize);
	if (num_bytes != content.dsize) fatal(dbf, "kwrite error");

	/* Current bucket has changed. */
	dbf->cache_entry->ca_changed = TRUE;
	dbf->bucket_changed = TRUE;

	/* Write everything that is needed to the disk. */
	end_update(dbf);
	return(0);
}

/************************************************************
*  Routine Name: kdbm_delete - Remove the key and its associated data
*			from the database.
*       Purpose: Remove the KEYed item and the KEY from the database
*		 DBF.  The file on disk is updated to reflect the
*		 structure of the new database before returning from
*		 this procedure.
*         Input: dbf - open database pointer
*       Returns: 0 on success, -1 otherwise
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/
int kdbm_delete(
     kdbm *dbf,
     kdatum key)
{
	bucket_element elem;	/* The element to be deleted. */
	int	elem_loc;	/* The location in the current hash bucket. */
	int	last_loc;	/* Last location emptied by the delete.  */
	int	home;		/* Home position of an item. */
	long	free_adr;	/* Temporary stroage for address and size. */
	long	free_size;

	/* First check to make sure this guy is a writer. */
	if (!dbf || dbf->rdonly == TRUE)
	{
	   errno = KDBM_READER_CANT_DELETE;
	   return(-1);
	}

	/* Find the item. */
	if ((elem_loc = findkey(dbf, key, NULL, NULL)) == -1)
	{
	   errno = KDBM_ITEM_NOT_FOUND;
	   return(-1);
	}

	/* Save the element.  */
	elem = dbf->bucket->h_table[elem_loc];

	/* Delete the element.  */
	dbf->bucket->h_table[elem_loc].hash_value = -1;
	dbf->bucket->count -= 1;

	/* Move other elements to guarantee that they can be found. */
	last_loc = elem_loc;
	elem_loc = (elem_loc + 1) % dbf->header->bucket_elems;
	while (elem_loc != last_loc &&
	       dbf->bucket->h_table[elem_loc].hash_value != -1)
	{
	   home = dbf->bucket->h_table[elem_loc].hash_value %
		  dbf->header->bucket_elems;
	   if ((last_loc < elem_loc && (home <= last_loc || home > elem_loc)) ||
	       (last_loc > elem_loc && home <= last_loc && home > elem_loc))
	   {
	      dbf->bucket->h_table[last_loc] = dbf->bucket->h_table[elem_loc];
	      dbf->bucket->h_table[elem_loc].hash_value = -1;
	      last_loc = elem_loc;
	   }
	   elem_loc = (elem_loc + 1) % dbf->header->bucket_elems;
	}

	/* Free the file space. */
	free_adr = elem.data_pointer;
	free_size = elem.key_size + elem.data_size;
	free_dbm(dbf, free_adr, free_size);

	/* Set the flags. */
	dbf->bucket_changed = TRUE;

	/* Clear out the data cache for the current bucket. */
	kfree(dbf->cache_entry->ca_data.key);
	kfree(dbf->cache_entry->ca_data.data);
	dbf->cache_entry->ca_data.hash     = -1;
	dbf->cache_entry->ca_data.key_size =  0;
	dbf->cache_entry->ca_data.elem_loc = -1;

	/* Do the writes. */
	end_update(dbf);
	return(0);  
}


/************************************************************
*
*  Routine Name: kdbm_firstkey - get the first key in the database
*       Purpose: Find the first key in the database, and return it
*		 to the user.
*         Input: dbf - the database pointer
*       Returns: The first key of the database.  If dbf is NULL, or
*		 there are no keys in the database, the returned
*		 key.dsize and key.dptr will be 0 and NULL respectively.
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

kdatum kdbm_firstkey(
     kdbm *dbf)
{
	kdatum nullitem;

	if (!dbf)
	{
	   nullitem.dsize = 0;
	   nullitem.dptr  = NULL;
	   return(nullitem);
	}

	/* Get the first bucket.  */
	get_bucket(dbf, 0);

	/* Set the default return value for not finding a first entry. */
	dbf->keyptr.dsize = 0;
	dbf->keyptr.dptr  = NULL;

	/* Look for first entry. */
	(void) get_next_key(dbf, -1, &dbf->keyptr);
	return(dbf->keyptr);
}

/************************************************************
*
*  Routine Name: kdbm_nextkey - get the next key in the database
*       Purpose: After an initial call to kdbm_firstkey, this routine
*		 can be used to traverse the list of keys in a database.
*         Input: dbf - the database pointer
*       Returns: The next key in the database.  If dbf is NULL, or
*		 there are no more keys in the database, it
*		 will return key.dsize and key.dptr as 0 and NULL respectively.
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

kdatum kdbm_nextkey(
     kdbm *dbf)
{
	int    elem_loc;		/* The location in the bucket. */
	kdatum nullitem;

	if (!dbf)
	{
	   nullitem.dsize = 0;
	   nullitem.dptr  = NULL;
	   return(nullitem);
	}

	/* Do we have a valid key? */
	if (!dbf->keyptr.dptr && dbf->keyptr.dsize > 0)
	   return(dbf->keyptr);

	/* Find the key.  */
	if ((elem_loc = findkey(dbf, dbf->keyptr, NULL, NULL)) == -1)
	{
	   /* Set the default return value for no next entry. */
	   dbf->keyptr.dptr  = NULL;
	   dbf->keyptr.dsize = 0;
	   return(dbf->keyptr);
	}

	/* Find the next key. */  
	if (!get_next_key(dbf, elem_loc, &dbf->keyptr))
	{
	   dbf->keyptr.dptr  = NULL;
	   dbf->keyptr.dsize = 0;
	}
	return(dbf->keyptr);
}

/************************************************************
*
*  Routine Name: kdbm_checkkey - check to see if a key exists in the database
*       Purpose: Check to see if a key exists.  The key passed in
*		 is checked against the list of database keys.
*         Input: dbf - the database file
*		 key - the key to check for
*       Returns: TRUE (1) on success, FALSE (0) otherwise
*    Written By: Philip A. Nelson
*          Date: Nov 08, 1993
*************************************************************/

int kdbm_checkkey(
     kdbm *dbf,
     kdatum key)
{
	int    elem_loc;


	if (!dbf || !key.dptr)
	   return(FALSE);

	if ((elem_loc = findkey(dbf, key, NULL, NULL)) == -1)
	   return(FALSE);

	return(TRUE);
}

/************************************************************
*
*  Routine Name: kdbm_read - Find a key and reads the associated data.
*
*       Purpose: Look up a given KEY and return the information 
*                associated with that KEY.  The pointer in the 
*		 structure that is  returned is a pointer to dynamically
*		 allocated memory block.  This is similar to the
*		 kdbm_fetch routine, except that kdbm_read supports
*		 incremental reads.
*         Input: dbf - the database file
*                key - the key in which to read from
*                data - the data array to read into
*		 num  - number of data points to read
*		 type - the type of data points to read from
*       Returns: the number of data points actually read, or -1 upon error
*    Written By: Mark Young
*          Date: Dec 4, 1993
*************************************************************/

int kdbm_read(
    kdbm   *dbf,
    kdatum key,
    kaddr   data,
    int     num,
    int     type)
{
	data_cache_elem *data_ca;
	int    file_pos, elem_loc;


	if (!dbf || !key.dptr ||
	    (elem_loc = findkey(dbf, key, NULL, NULL)) == -1 ||
	    (data_ca = get_entry(dbf, elem_loc)) == NULL)
	{
	    return(-1);
	}

	/* Seek into the cache. */
	if (data_ca->seek_pos >= data_ca->data_size)
	   return(0);

	file_pos = data_ca->file_pos + data_ca->key_size + data_ca->seek_pos;
	if (klseek(dbf->desc, file_pos, KSEEK_SET) != file_pos)
	{
	   fatal(dbf, "klseek error");
	   return(-1);
	}

	num = kmin(num, data_ca->data_size - data_ca->seek_pos);
	if ((num = kread_generic(dbf->desc, data, num, type)) == -1)
	   data_ca->seek_pos = data_ca->data_size;
	else
	{
	   if (type != KSTRING)
	      data_ca->seek_pos += kmach_sizeof(dbf->header->machtype,type)*num;
	   else
	      data_ca->seek_pos += klseek(dbf->desc, 0, KSEEK_CUR) - file_pos;
	}
	return(num);
}

/************************************************************
*
*  Routine Name: kdbm_write - Simple database write routine
*
*       Purpose: Add a new element to the database.  CONTENT is keyed
*                by KEY.  The file on disk is updated to reflect the
*                structure of the new database before returning from
*                this procedure.  This routine is similar to kdbm_store
*		 except that it allows incremental writes to this key
*		 in the database.
*         Input: dbf  - open database file pointer
*                key  - data base key to write to
*		 data - data to write to key
*		 num  - number of data items to write
*		 type - type of data to write out
*       Returns: the number of data points actually written, or -1 upon error
*    Written By: Mark Young
*          Date: Sep 29, 1994
*************************************************************/
int kdbm_write(
    kdbm   *dbf,
    kdatum key,
    kaddr   data,
    int     num,
    int     type)
{
	avail_elem av_el;
	data_cache_elem *data_ca;
	int    file_pos, elem_loc, num_bytes, hash;


	if (!dbf || !key.dptr || num < 0)
	    return(-1);

	num_bytes = kmach_sizeof(dbf->header->machtype, type) * num;

	/* Look for the key in the file.  A side effect loads the correct
	    bucket and calculates the hash value. */
	if ((elem_loc = findkey(dbf, key, NULL, &hash)) == -1)
	{
           if (dbf->bucket->count == dbf->header->bucket_elems)
           {
              /* Split the current bucket. */
              split_bucket(dbf, hash);
           }
 
           /* Find space to insert into bucket and set elem_loc to that
              place. */
           elem_loc = hash % dbf->header->bucket_elems;
           while (dbf->bucket->h_table[elem_loc].hash_value != -1)
               elem_loc = (elem_loc + 1) % dbf->header->bucket_elems;
 
           /* We now have another element in the bucket.  Add
              the new information.*/

           dbf->bucket->count++;
           dbf->bucket->h_table[elem_loc].hash_value = hash;
           kbcopy(key.dptr, dbf->bucket->h_table[elem_loc].key_start,
                (SMALL < key.dsize ? SMALL : key.dsize));

	   /* Get another full block from end of file. */
	   av_el = get_block(key.dsize+num_bytes, dbf);
	   file_pos = av_el.av_adr;
	   /*num_bytes = av_el.av_size - key.dsize;*/

           /* Update current bucket data pointer and sizes. */
           dbf->bucket->h_table[elem_loc].data_pointer = file_pos;
           dbf->bucket->h_table[elem_loc].key_size = key.dsize;
           dbf->bucket->h_table[elem_loc].data_size = num_bytes;

	   /* Write the data to the file. */
	   if (klseek(dbf->desc, file_pos, KSEEK_SET) != file_pos)
	      fatal(dbf, "klseek error");
	   if (kwrite(dbf->desc, key.dptr, key.dsize) != key.dsize)
	      fatal(dbf, "kwrite error");

	   /* Current bucket has changed. */
	   dbf->cache_entry->ca_changed = TRUE;
	   dbf->bucket_changed = TRUE;

	   /* Write everything that is needed to the disk. */
	   end_update(dbf);
	}

	if ((data_ca = get_entry(dbf, elem_loc)) == NULL)
	{
	   fatal(dbf, "get data cache error");
	   return(-1);
	}

	file_pos = data_ca->file_pos + data_ca->key_size + data_ca->seek_pos;
	if (klseek(dbf->desc, file_pos, KSEEK_SET) != file_pos)
	{
	   fatal(dbf, "klseek error");
	   return(-1);
	}

	if ((num = kwrite_generic(dbf->desc, data, num, type)) == -1)
	   data_ca->seek_pos = data_ca->data_size;
	else
	{
	   if (type != KSTRING)
	      data_ca->seek_pos += kmach_sizeof(dbf->header->machtype,type)*num;
	   else
	      data_ca->seek_pos += klseek(dbf->desc, 0, KSEEK_CUR) - file_pos;
	}


	/* figure out how big are we currently */
	if (data_ca->seek_pos > data_ca->data_size)
	{
	   /* We changed the bucket. */
	   dbf->bucket_changed = TRUE;

	   dbf->cache_entry->ca_changed = TRUE;
	   dbf->second_changed = TRUE;

	   /* Write everything that is needed to the disk. */
	   data_ca->data_size =
	   dbf->bucket->h_table[elem_loc].data_size = data_ca->seek_pos;

	   /* We changed the header. */
	   dbf->header_changed = TRUE;

	   /* Update the header and return. */
	   dbf->header->next_block = data_ca->file_pos + data_ca->key_size +
				     data_ca->data_size;
	}
	return(num);
}

/************************************************************
*
*  Routine Name: kdbm_lseek - move read/write pointer of the key pointer
*       Purpose: This function is used to seek to the position of the
*		 database relative to the given key.
*         Input: dbf - the database file
*                key - the key in which to seek for
*		 offset - the offset in which to seek
*		 whence - the control of how the offset will be applied
*       Returns: returns -1 or the new seeked position
*    Written By: Mark Young
*          Date: Dec 4, 1993
*************************************************************/

int kdbm_lseek(
    kdbm   *dbf,
    kdatum key,
    int     offset,
    int     whence)
{
	int    elem_loc;
	data_cache_elem *data_ca;


	if (!dbf || !key.dptr)
	   return(-1);

	if ((elem_loc = findkey(dbf, key, NULL, NULL)) == -1)
	{
	   if (dbf->rdonly || kdbm_write(dbf, key, NULL, 0, KUBYTE) == -1)
	      return(-1);
	   elem_loc = findkey(dbf, key, NULL, NULL);
	}

	if ((data_ca = get_entry(dbf, elem_loc)) == NULL)
	   return(-1);

	/* Seek into the cache. */
	if (whence == KSEEK_SET)
	   data_ca->seek_pos = offset;
	else if (whence == KSEEK_CUR)
	   data_ca->seek_pos += offset;
	else if (whence == KSEEK_END)
	   data_ca->seek_pos = data_ca->data_size + offset;
	else
	{
	   errno = EINVAL;
	   return(-1);
	}
	return(data_ca->seek_pos);
}

/************************************************************
*
*  Routine Name: kdbm_getmachtype - gets the machine type for the database
*       Purpose: This function is used to return the current machine type for
*		 the database.
*         Input: dbf - the database file
*       Returns: returns the machine type or -1 on error
*    Written By: Mark Young
*          Date: Dec 16, 1993
*************************************************************/

int kdbm_getmachtype(
    kdbm   *dbf)
{
	if (!dbf)
	   return(-1);

	return(dbf->header->machtype);
}

/************************************************************
*
*  Routine Name: kdbm_check - check where file descripter is a valid kdbm file
*       Purpose: This function is used check if the file descriptor points 
*                to a valid kdbm file.
*         Input: fid - file descriptor
*       Returns: TRUE if it is a kdbm file and FALSE if it is not.
*    Written By: Mark Young & Jeremy Worley
*          Date: Mar 22, 1994 10:24
*************************************************************/

int kdbm_check(
   int fid)
{
   int magic_number;
   unsigned char o_machtype, machtype;

   if (kread_ubyte(fid, &machtype, 1) != 1)
      return(FALSE);

   o_machtype = kfile_getmachtype(fid);
   kfile_setmachtype(fid, _convert_from_gdb_mach(machtype));

   /* Read the temp file header. */
   if (kread_int(fid, &magic_number, 1) != 1) {
      kfile_setmachtype(fid, o_machtype);
      return(FALSE);
   }

   /* Is the magic number good? */
   if (magic_number != KDBM_MAGIC_NUMBER) {
      kfile_setmachtype(fid, o_machtype);
      return(FALSE);
   }
   return(TRUE);
}
