/*
 * Khoros: $Id$
 */

#if !defined(__lint) && !defined(__CODECENTER__)
static char rcsid[] = "Khoros: $Id$";
#endif

/*
 * $Log$
 */

/*
 * Copyright (C) 1993, 1994, 1995, Khoral Research, Inc., ("KRI").
 * All rights reserved.  See $BOOTSTRAP/repos/license/License or run klicense.
 */


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>                UIS Utility Routines                   <<<<
   >>>>  Private:                                             <<<<
   >>>>                kvf_get_line_type()                    <<<<
   >>>>                kvf_ascii_typeflag()                   <<<<
   >>>>                kvf_get_comments()                     <<<<
   >>>>                kvf_allocation_error_mesg()            <<<<
   >>>>                kvf_parsing_error_mesg()               <<<<
   >>>>                kvf_initialize()                       <<<<
   >>>>                kvf_get_routine_syntax()               <<<<
   >>>>                kvf_gen_parse_kformstruct()            <<<<
   >>>>                kvf_gen_deparse_kformstruct()          <<<<
   >>>>                kvf_append_toggle_values_to_buffer()   <<<<
   >>>>                kvf_calculate_y_position()             <<<<
   >>>>                kvf_find_specified_selection()         <<<<
   >>>>                                                       <<<<
   >>>>                kvf_set_context()                      <<<<
   >>>>                kvf_set_expr_handler()                 <<<<
   >>>>                kvf_get_expr_handler()                 <<<<
   >>>>                kvf_call_expr_handler()                <<<<
   >>>>                                                       <<<<
   >>>>                kvf_set_int_from_literal()             <<<<
   >>>>                kvf_set_float_from_literal()           <<<<
   >>>>                kvf_set_double_from_literal()          <<<<
   >>>>                kvf_set_file_from_literal()            <<<<
   >>>>   Static:                                             <<<<
   >>>>   Public:                                             <<<<
   >>>>                kvf_gui_item_exists()        	      <<<<
   >>>>                kvf_append_selections()                <<<<
   >>>>                kvf_check_for_single_pane()            <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

#include "internals.h"


/*-----------------------------------------------------------
|
|  Routine Name: kvf_get_line_type
|
|       Purpose: Returns a #define'd integer (see line_info.h)
|		 that indicates the type of UIS line 
|
|         Input: line - UIS line 
|        Output: Returns the int rep. of the UIS linetype on
|                success, -1 on failure
|          Date: Jul 13, 1992
|    Written By: Danielle Argiro 
| Modifications: Converted from Khoros 1.0 (DA)
|
-------------------------------------------------------------*/

int kvf_get_line_type(
   char *line)
{
	char flag;
	char temp[KLENGTH];
	int  status;

	if (line == NULL)
	{
	     errno = KCALL;
	     kerror("kforms", "kvf_get_line_type",
             	    "NULL UIS line recieved.\n");
             return(-1);
	}

	else if (kstrcmp(line, "") == 0) return(KUIS_COMMENT);

	else if (line[0] == '-') flag = line[1];

	else if ((ksscanf(line, "%c", &flag))!=1) 
  	{
	    ksprintf(temp, "Unrecognisable UIS line '-%s'", line);
	    errno = KUIS_SYNTAX;
	    kerror("kforms", "kvf_get_line_type", temp);
	    return(-1);
	}

	if ((flag != '#') && (flag != '@') && (flag != '\n') && (flag != '\0'))
	{
	   if (line[0] != '-')
	   {
	 	line[kstrlen(line)-1] = '\0';
	        ksprintf(temp, "Unrecognisable UIS line '%s'", line);
	        errno = KUIS_SYNTAX;
	        kerror("kforms", "kvf_get_line_type", temp);
	        return(-1);
	   }
	}

	switch (flag) {

		case 'F': 	status = KUIS_STARTFORM; 	break;
		case 'S': 	status = KUIS_STARTMASTER; 	break;
		case 'D': 	status = KUIS_STARTSUBMENU; 	break;
		case 'M': 	status = KUIS_STARTSUBFORM; 	break;
		case 'P': 	status = KUIS_STARTPANE; 	break;
		case 'G': 	status = KUIS_STARTGUIDE; 	break;
		case 'E': 	status = KUIS_END; 		break;
		case 'd': 	status = KUIS_SUBFORMBUTTON; 	break;
		case 'u': 	status = KUIS_PSEUDOSUBFORM; 	break;
		case 'n':	status = KUIS_MASTERACTION; 	break;
		case 'g': 	status = KUIS_GUIDEBUTTON; 	break;
		case 'Q': 	status = KUIS_QUIT; 		break;
		case 'm': 	status = KUIS_SUBFORMACTION; 	break;
		case 'I': 	status = KUIS_INPUTFILE; 	break;
		case 'O':	status = KUIS_OUTPUTFILE; 	break;
		case 'e': 	status = KUIS_STDIN; 		break;
		case 'o':	status = KUIS_STDOUT; 		break;
		case 'i': 	status = KUIS_INTEGER; 		break;
		case 'f': 	status = KUIS_FLOAT; 		break;
		case 'h': 	status = KUIS_DOUBLE; 		break;
		case 's': 	status = KUIS_STRING; 		break;
		case 'y': 	status = KUIS_STRINGLIST; 	break;
		case 'l': 	status = KUIS_LOGICAL; 		break;
		case 't': 	status = KUIS_FLAG; 		break;
		case 'T': 	status = KUIS_TOGGLE; 		break;
		case 'R': 	status = KUIS_ROUTINE; 		break;
		case 'H': 	status = KUIS_HELP; 		break;
		case 'b': 	status = KUIS_BLANK; 		break;
		case 'a': 	status = KUIS_PANEACTION; 	break;
		case 'C': 	status = KUIS_MUTEXCL; 		break;
		case 'B': 	status = KUIS_MUTINCL; 		break;
		case 'K': 	status = KUIS_GROUP; 		break;
		case 'p': 	status = KUIS_INCLUDEPANE; 	break;
		case 'k': 	status = KUIS_INCLUDESUBFORM;	break;
		case 'w': 	status = KUIS_WORKSPACE; 	break;
		case 'c': 	status = KUIS_CYCLE; 		break;
		case 'x': 	status = KUIS_LIST; 		break;
		case 'z': 	status = KUIS_DISPLAYLIST; 	break;
		case '#': 	
		case '@': 	
                case '\n':
	        case '\0':      status = (KUIS_COMMENT); 	        break;

	        default:
	              ksprintf(temp, "Unknown UIS line type '-%c' in line '%s'",
			       flag, line);
		      errno = KUIS_SYNTAX;
	              kerror("kforms", "kvf_get_line_type", temp);
		      status = -1;
		      break;

	}  /* end switch */

	return(status); 		

} /* end kvf_get_line_type */



/*------------------------------------------------------------
|
|  Routine Name: kvf_ascii_typeflag
|
|       Purpose: Given a UIS type_flag #define, returns the
|                ascii string representation the #define
|
|         Input: typeflag - the UIS typeflag
|        Output:
|       Returns: The ascii string representation of the #define
|                Note: don't modify or free the string!
|          Date: Sep 17, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/

char *kvf_ascii_typeflag(
   int typeflag)
{
    char *str;

    switch (typeflag)
    {
        case KUIS_STARTFORM:      str = "StartForm";      break;
        case KUIS_STARTMASTER:    str = "StartMaster";    break;
        case KUIS_STARTSUBMENU:   str = "StartSubmenu";   break;
        case KUIS_STARTSUBFORM:   str = "StartSubform";   break;
        case KUIS_STARTGUIDE:     str = "StartGuide";     break;
        case KUIS_STARTPANE:      str = "StartPane";      break;
        case KUIS_END:            str = "End";            break;
        case KUIS_MASTERACTION:   str = "MasterAction";   break;
        case KUIS_SUBFORMACTION:  str = "SubformAction";  break;
        case KUIS_SUBFORMBUTTON:  str = "SubformButton";  break;
        case KUIS_GUIDEBUTTON:    str = "GuideButton";    break;
        case KUIS_INCLUDESUBFORM: str = "IncludeSubform"; break;
        case KUIS_INCLUDEPANE:    str = "IncludePane";    break;
        case KUIS_INPUTFILE:      str = "InputFile";      break;
        case KUIS_OUTPUTFILE:     str = "OutputFile";     break;
        case KUIS_INTEGER:        str = "Integer";        break;
        case KUIS_FLOAT:          str = "Float";          break;
        case KUIS_DOUBLE:         str = "Double";         break;
        case KUIS_STRING:         str = "String";         break;
        case KUIS_LOGICAL:        str = "Logical";        break;
        case KUIS_CYCLE:          str = "Cycle";          break;
        case KUIS_LIST:           str = "List";           break;
        case KUIS_DISPLAYLIST:    str = "DisplayList";    break;
        case KUIS_TOGGLE:         str = "Toggle";         break;
        case KUIS_MUTEXCL:        str = "MutExcl";        break;
        case KUIS_MUTINCL:        str = "MutIncl";        break;
        case KUIS_GROUP:          str = "Group";          break;
        case KUIS_ROUTINE:        str = "Routine";        break;
        case KUIS_HELP:           str = "Help";           break;
        case KUIS_QUIT:           str = "Quit";           break;
        case KUIS_BLANK:          str = "Blank";          break;
        case KUIS_PANEACTION:     str = "PaneAction";     break;
        case KUIS_WORKSPACE:      str = "Workspace";      break;
        case KUIS_FLAG:           str = "Flag";           break;
        case KUIS_STRINGLIST:     str = "Stringlist";     break;
        case KUIS_STDIN:          str = "Stdin";          break;
        case KUIS_STDOUT:         str = "Stdout";         break;

	default: 		  str = NULL;             break;
    }
    return(str);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_get_comments
|       Purpose: Collects comment lines and carriage returns out
|                of internal UIS, and returns them as a block
|                of characters for storage in the form tree
|         Input: database  - internal UIS
|                kvf_index - index into UIS
|        Output: Returns pointer to block of characters or NULL if there
|                is no comment or carriage returns.
|          Date: Apr 10, 1992
|    Written By: Danielle Argiro
| Modifications:
|
---------------------------------------------------------------*/
char *kvf_get_comments(
   char **database,
   int  *kvf_index)
{
        int  flag;
        char buffer[20*KLENGTH], temp[KLENGTH];
        char *comment;

        kmemset(buffer, 0, 20*KLENGTH);
        flag = kvf_get_line_type(database[*kvf_index]);
        if (flag != KUIS_COMMENT) return(NULL);

        ksprintf(buffer, "%s\n", database[(*kvf_index)++]);
        flag = kvf_get_line_type(database[*kvf_index]);
        while (flag == KUIS_COMMENT)
        {
           ksprintf(temp, "%s\n", database[(*kvf_index)++]);
           kstrcat(buffer, temp);
           flag = kvf_get_line_type(database[*kvf_index]);
        }
        comment = kstrdup(buffer);
        return(comment);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_allocation_error_mesg
|
|       Purpose: Prints appropriate error message on failure to malloc
|
|         Input: none
|        Output: none
|          Date: May 5, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/

void kvf_allocation_error_mesg(void)
{
	kerror("kforms", "xvf_create_form", 
               "Cannot allocate room for internal form tree node; \
aborting form creation");
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_parsing_error_mesg
|
|       Purpose: print appropriate error message on failure to parse UIS line
|
|         Input: index           - index into the UIS of current line
|                filename_lookup - indexes into 'filenames' array
|                linenum_lookup  - holds actual line number of file for UIS line
|                filenames       - array of filenames comprising UIS
|
|        Output: none
|          Date: May 5, 1992
|    Written By: Danielle Argiro 
| Modifications:
|
---------------------------------------------------------------*/
void kvf_parsing_error_mesg(
   int  index,
   int  *filename_lookup,
   int  *linenum_lookup,
   char **filenames)
{
	errno = KUIS_SYNTAX;
	kerror("kforms", "kvf_create_form",
	       "Incorrect syntax on UIS line %d of %s",
	       linenum_lookup[index], filenames[filename_lookup[index]]);
}

/*-------------------------------------------------------------
|
|  Routine Name: kvf_initialize
|
|       Purpose: Does initializations in preparation for the use of
|                the kforms library by the Khoros application.
|
|         Input: None
|        Output: None
|       Returns: None
|
|    Written By: Danielle Argiro
|          Date: March 31, 1993
| Modifications:
|
-------------------------------------------------------------*/
void kvf_initialize(void)
{
        static int initialized = FALSE;

        if (initialized == TRUE)
           return;
        else initialized = TRUE;

        kformstruct_hdr   = NULL;
        kformstruct_tail  = NULL;
	kvf_context       = KVF_CLUI;

	kvf_init_attributes();
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_get_routine_syntax
|
|       Purpose: Finds the routine button on a form (assuming
|                a single pane on a single subform) and returns
|                the text for the routine on the -R line, eg,
|                "karith -add" or "preview".
|
|         Input: form - pointer to the form tree
|        Output: none
|       Returns: The text from the -R line on success, NULL on failure
|          Date: March 17, 1994
|    Written By: Danielle Argiro
| Modifications:
|
------------------------------------------------------------*/

char *kvf_get_routine_syntax(
    kform *form)
{
	kselection *selection;
	Line_Info  line_info;
	char       *routine_syntax;
	kguide     *guide;

	if (form->subform == NULL)
	{
	    kerror("kforms", "kvf_get_routine_syntax", 
		   "Not supporting forms with multiple subforms");
	    return(NULL);
	}

	guide = kvf_search_sel_guide(form->subform);
	
	selection = guide->pane->sel_list;

	while ((selection != NULL) && (selection->type != KUIS_ROUTINE))
	    selection = selection->next;

	if (selection == NULL)
	    return(NULL);

	kvf_clear_line_info(&line_info);
	kvf_parse_routine_line(selection->line, &line_info);
	routine_syntax = kstrdup(line_info.routine);
	kvf_free_line_info_strings(&line_info);
	return(routine_syntax);
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_gen_parse_kformstruct
|
|       Purpose: Given a generic kform_struct, fills in the 
|                values of the Line_Info structure according to the
|		 UIS line associated with the form, subform,
|		 guide, pane, or selection.
|         Input: kformstruct - generic kform_struct
|        Output: The UIS line associated with kform_struct
|          Date: March 16, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
int kvf_gen_parse_kformstruct(
   kform_struct *kformstruct,
   Line_Info    *line_info)
{
	int  status;
	char *line = NULL;

	kvf_clear_line_info(line_info);

	if (kformstruct->type == KSELECTION)
	   line = kformstruct->Selptr->line;

	else if (kformstruct->type == KMASTER)   
	   line = kformstruct->Controlptr->back_form->control_line;

	else if ((kformstruct->type == KGUIDEPANE) ||
		 (kformstruct->type == KPANE))
	   line = kformstruct->Controlptr->control_line;

	else if (kformstruct->type == KGUIDE)
	   line = kformstruct->Guideptr->line;

	else if (kformstruct->type == KSUBFORM)
	       line = kformstruct->Subformptr->control_line;

	else if (kformstruct->type == KFORM) 
	   line = kformstruct->Formptr->control_line;

	if (line == NULL)
	{
	    errno = KINTERNAL;
	    kerror("kforms", "kvf_gen_parse_kformstruct",
		   "Can't find appropriate UIS line for kformstruct");
	    return(FALSE);
	}

	status = kvf_gen_parse(line, line_info);
	return(status);
}

/*------------------------------------------------------------
|
|  Routine Name: kvf_gen_deparse_kformstruct
|
|       Purpose: Given a generic kform_struct, deparses the
|		 UIS line associated with the form, subform,
|		 guide, pane, or selection,  according to the
|                values of the Line_Info structure given
|         Input: kformstruct - generic kform_struct
|        Output: The UIS line associated with kform_struct
|          Date: March 16, 1992
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
void kvf_gen_deparse_kformstruct(
   kform_struct *kformstruct,
   Line_Info  *line_info)
{
	if (kformstruct->type == KSELECTION)
	   kvf_gen_deparse(line_info, &kformstruct->Selptr->line);

	else if (kformstruct->type == KMASTER)   
	   kvf_gen_deparse(line_info, 
			&kformstruct->Controlptr->back_form->control_line);

	else if ((kformstruct->type == KPANE)    ||
		 (kformstruct->type == KGUIDEPANE))
	   kvf_gen_deparse(line_info, &kformstruct->Controlptr->control_line);

	else if (kformstruct->type == KGUIDE)
	   kvf_gen_deparse(line_info, &kformstruct->Guideptr->line);

	else if (kformstruct->type == KSUBFORM)
	   kvf_gen_deparse(line_info, &kformstruct->Subformptr->control_line);
	 
	else if (kformstruct->type == KFORM)
	   kvf_gen_deparse(line_info, 
			&kformstruct->Formptr->control_line);

	kvf_free_line_info_strings(line_info);
}


/*------------------------------------------------------------
|
|  Routine Name: kvf_append_toggle_values_to_buffer
|
|       Purpose: Given a toggle selection and a text buffer
|                (which may already have something in it)
|                appends the legal values for the toggle into the buffer.
|
|         Input: selection - the toggle selection
|        Output: buffer    - text buffer with legal toggle values appended
|       Returns: none
|          Date: April 6, 1994
|    Written By: Danielle Argiro
| Modifications:
|
-------------------------------------------------------------*/
void kvf_append_toggle_values_to_buffer(
    kselection *selection, 
    char       *buffer)
{
        char **contents;
        char **descriptions;
        char temp[KLENGTH];
        int  i, toggle_size;

        kvf_get_attributes(selection->back_kformstruct,
                           KVF_TOGGLE_CONTENTS,      &contents,
                           KVF_TOGGLE_CONTENTS_DESC, &descriptions,
                           KVF_TOGGLE_SIZE,          &toggle_size,
                           NULL);

        for (i = 0; i < toggle_size; i++)
        {
            if (i == toggle_size -1)
            {
		if (descriptions[i] != NULL)
                    sprintf(temp, " or %s (%s)", contents[i], descriptions[i]);
		else sprintf(temp, " or %s (no description)", contents[i]);
                kstring_cat(buffer, temp, buffer);
            }
            else
            {
		if (descriptions[i] != NULL)
                    sprintf(temp, " %s (%s),", contents[i], descriptions[i]);
		else sprintf(temp, " %s (no description),", contents[i]);
                kstring_cat(buffer, temp, buffer);
            }
        }
}


/************************************************************
*
*  Routine Name: kvf_gui_item_exists - determine whether at least one instance
*                                      of a particular GUI item exists in 
*                                      a selection list of a form tree
*
*       Purpose: Determines if there is a gui item of the type specified
*                in the selection list.
*
*         Input: sel_list - header of the selection list
*                flag     - typeflag indicating selection type we're looking for
*        Output:
*       Returns: TRUE if a GUI item of the type specified is in the list,
*                FALSE otherwise
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: May 12, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_gui_item_exists(
   kselection *sel_list,
   int        flag)
{
	kselection *selection;

	selection = sel_list;
        while (selection != NULL)
        {
             if (selection->type == flag)
                return(TRUE);
             selection = selection->next;
        }
        return(FALSE);
}


/************************************************************
*
*  Routine Name: kvf_gui_named_item_exists - determine whether at least one 
*                                            instance of a particular GUI item 
*                                            with the specified variable name
*                                            exists in a selection list of a 
*                                            form tree
*
*       Purpose: Determines if there is a gui item of the type specified
*                with the variable name specified in the selection list.
*
*         Input: sel_list  - header of the selection list
*                flag      - typeflag indicating selection type 
*                var_token - token representing variable name in question;
*                            specify -1 if variable name doesn't matter
*        Output:
*       Returns: TRUE if a GUI item of the type specified with the 
*                given variable name is in the list, FALSE otherwise
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: May 1, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

kselection *kvf_gui_named_item_exists(
   kselection *sel_list,
   int        flag,
   int        var_token)
{
        kselection *selection;

        selection = sel_list;
        while (selection != NULL)
        {
	     if (var_token != -1)
	     {
                 if ((selection->type == flag) && 
	             (selection->var_token == var_token))
                    return(selection);
	     }
	     else
	     {
		if (selection->type == flag) return(selection);
	     }
             selection = selection->next;
        }
        return(selection);
}


/************************************************************
*
*  Routine Name: kvf_append_selections - append selections from one form tree 
*                                        onto the selection list of another
*
*       Purpose: Given the paths to two UIS files specifying single panes
*                on single subforms, reads in the two forms, appends the
*                selection list of the second onto the selection list
*                of the first, and returns the form tree representing the
*                result.  Used by kgen_initialize() to merge the pane defining 
*                CLUI arguments of a program with the pane defining
*                the standardized command line arguments for that
*                program type.
*
*         Input: form1     - the old form
*                panepath2 - path to *.pane file of 2nd software object
*                            (in use by kgen_initialize, this is the pane
*                             defining the standardized arguments).
*        Output:
*       Returns: TRUE if the new selections were successfully appended,
*                FALSE otherwise
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: March 16, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/


int kvf_append_selections(
    kform *form1,
    char *panepath2)
{
	int        var_token;
	kselection *selection;
	kform      *form2;

	/*
	 *  read in the *.pane file defining CLUI of software object, errorcheck
	 */
	if (form1 == NULL) return(FALSE);

	/*
	 *  read in the *.pane file defining standard arguments, errorcheck
	 */
        form2 = kvf_create_form(panepath2, NONE, NULL, NULL);
        if (form2 == NULL) return(FALSE);

	if (!(kvf_check_for_single_pane(form1, "kvf_append_selections")))
	    return(FALSE);

	if (!(kvf_check_for_single_pane(form2, "kvf_append_selections")))
	    return(FALSE);

	/*
	 * special case: [-x] and [y] are standardized arguments for
	 * xvroutines, but they also appear in the *.pane files of some
	 * xvroutines like animate and putdata, to allow the user to set the 
	 * location of the display which will be popping up from w/in cantata.
	 *
	 * however, wrt usage, prompting, code generation, and so on, we
	 * really don't want two [-x]'s and two [-y]'s, so get rid of the ones 
	 * in the *.pane file (if any), and just use the ones on the 
	 * standardized pane.
	 */
	var_token = kstring_to_token("x");
        if ((kvf_gui_named_item_exists(form1->subform->guide->pane->sel_list, 
				       KUIS_INTEGER, var_token) != NULL) &&
            (kvf_gui_named_item_exists(form2->subform->guide->pane->sel_list, 
				       KUIS_INTEGER, var_token) != NULL))
        {
                selection = kvf_find_specified_selection(
				form1->subform->guide->pane->sel_list,
                                KUIS_INTEGER, var_token);
                kvf_delete_selection(selection);
        }
	var_token = kstring_to_token("y");
        if ((kvf_gui_named_item_exists(form1->subform->guide->pane->sel_list, 
                                       KUIS_INTEGER, var_token) != NULL) &&
            (kvf_gui_named_item_exists(form2->subform->guide->pane->sel_list, 
                                       KUIS_INTEGER, var_token) != NULL))
        {
                selection = kvf_find_specified_selection(
                                form1->subform->guide->pane->sel_list,
                                KUIS_INTEGER, var_token);
                kvf_delete_selection(selection);
        }

	/*
	 *  append selection list of standard args form onto the
         *  end of the selection list of the regular *.pane form
	 */
	selection = form1->subform->guide->pane->sel_list;
	if (selection == NULL)
	{
	    form1->subform->guide->pane->sel_list = 
		form2->subform->guide->pane->sel_list;
	}
	else
	{
	    while (selection->next != NULL)
	        selection = selection->next;

	    selection->next = form2->subform->guide->pane->sel_list;
	}
	selection = form2->subform->guide->pane->sel_list;
	while (selection != NULL)
	{
	    selection->back_form    = form1;
	    selection->back_subform = form1->subform;
	    selection->back_guide   = form1->subform->guide;
	    selection->back_control = form1->subform->guide->pane;
	    selection = selection->next;
	}

	/*
	 *  free everything from the form2, but can't call
         *  kvf_destroy_form because we don't want to free the selection
         *  list.  luckily the free'ing procedure is pretty simple here.
	 */
	kfree(form2->subform->guide->pane->control_line);
        kfree(form2->subform->guide->pane->control_comment);
	kvf_unlink_formstruct(form2->subform->guide->pane->back_kformstruct);
	kfree(form2->subform->guide->pane->back_kformstruct);
	kfree(form2->subform->guide->pane);
	kfree(form2->subform->guide->line);
        kfree(form2->subform->guide->control_line);
        kfree(form2->subform->guide->comment);
	kvf_unlink_formstruct(form2->subform->guide->back_kformstruct);
	kfree(form2->subform->guide->back_kformstruct);
	kfree(form2->subform->guide);
        kfree(form2->subform->line);
        kfree(form2->subform->comment);
        kfree(form2->subform->control_line);
        kfree(form2->subform->control_comment);
	kvf_unlink_formstruct(form2->subform->back_kformstruct);
	kfree(form2->subform->back_kformstruct);
	kvf_delete_entry((char *) form2->subform);
        kfree(form2->subform);
	kfree(form2->control_line);
        kfree(form2->control_comment);
	kfree(form2->back_kformstruct);
	kvf_delete_entry((char *) form2);
	kfree(form2->uis_location);
        kfree(form2);

	return(TRUE);
}


/************************************************************
*
*  Routine Name: kvf_check_for_single_pane - verify single pane 
*                                            on single subform
*
*       Purpose: Checks to see that the form tree passed in has 
*                a single pane on a single subform, for use by 
*                routines such as those in the code generators,
*                that are limited to cases like that.
*
*         Input: form    - pointer to the form tree
*                routine - name of calling routine (for error messages); pass 
*                          NULL if no error messages are to be printed.
*        Output: TRUE if the form has a single pane on a single subform,
*                FALSE otherwise
*  Restrictions:
*    Written By: Danielle Argiro
*          Date: March 16, 1994
*      Verified:
*  Side Effects:
* Modifications:
*
*************************************************************/

int kvf_check_for_single_pane(
    kform *form,
    char  *routine)
{
        if (form->master != NULL)
        {
	    if (routine != NULL)
                kerror("kforms", routine, "Can't handle a master form");
            return(FALSE);
        }
        if (form->subform == NULL)
        {
	    if (routine != NULL)
                kerror("kforms", routine, "form->subform is NULL");
            return(FALSE);
        }
        if (form->subform->guidepane != NULL)
        {
	    if (routine != NULL)
                kerror("kforms", routine,
                       "Can't handle a subform with a guidepane");
            return(FALSE);
        }
        if (form->subform->guide == NULL)
        {
	    if (routine != NULL)
                kerror("kforms", routine,
                       "form->subform->guide is NULL");
            return(FALSE);
        }
        if (form->subform->guide->pane == NULL)
        {
	    if (routine != NULL)
                kerror("kforms", routine,
                       "form->subform->guide->pane is NULL");
            return(FALSE);
        }
      return(TRUE);
}
/*-----------------------------------------------------------
|
|  Routine Name: kvf_calculate_y_position
|
|       Purpose: Calculates the Y position for a new selection -
|                simply puts it at the bottom of the form, using
|                all the current geometries of the UIS lines of
|                selections in the selection list (as opposed to
|                xvf_calculate_y_position(), which uses all the
|                widgets on the backplane.
|
|         Input: control - pane, guidepane, or master
|        Output: None
|       Returns: returns the floating point Y position
|    Written By: Danielle Argiro
|          Date: Tues Dec  8, 1992
| Modifications:
|
------------------------------------------------------------*/
float kvf_calculate_y_position(
   kcontrol *control)
{
	kselection *selection;
	float      geom, y, height, max_y = 0;
	

	selection = control->sel_list;

	while (selection != NULL)
	{
	     if (selection->type == KUIS_BLANK)
	     {
		 height = 1.0;
	         kvf_get_attribute(selection->back_kformstruct, KVF_YPOS, &y);
	     }
	     else
	     {
	         kvf_get_attributes(selection->back_kformstruct,
			        KVF_Y,      &y,
			        KVF_HEIGHT, &height,
			        NULL);
	     }
	     geom = y + height;
	     if (geom > max_y) max_y = geom;
	     selection = selection->next;
	}

        return(max_y);
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_find_specified_selection
|
|       Purpose: Finds a selection in a selection list
|                based on typeflag and variable name
|
|         Input: sel_list  - header of selection list to search  
|                typeflag  - type of selection we're looking for
|                var_token - tokenized version of variable if we care
|                            what the variable is, -1 otherwise
|        Output: 
|       Returns: The selection if it finds it, NULL otherwise.
|    Written By: Danielle Argiro
|          Date: Tues Dec  8, 1992
| Modifications:
|
------------------------------------------------------------*/

kselection *kvf_find_specified_selection(
    kselection *sel_list,
    int        typeflag,
    int        var_token)
{
        kselection *selection = NULL;

        selection = sel_list;
        /*
         * header of selection list the type we're looking for
         */
        if (sel_list->type == typeflag)
        {
             /* don't care about the variable name */
             if (var_token == -1)
                return(sel_list);

             /* do care about the variable name; only return on match */
             else if (var_token == sel_list->var_token)
                return(sel_list);
        }

        /*
         * have to look for selection of correct type,
         * but don't care about variable
         */
        if (var_token == -1)
        {
            while (selection != NULL) 
	    {
		if (selection->type == typeflag)
		    return(selection);
                selection = selection->next;
	    }
        }

        /*
         * have to look for selection of correct type with the
         * correct variable name as specified by var_token
         */
        else
        {
            while (selection != NULL) 
	    {
	        if ((selection->type == typeflag) &&
                    (selection->var_token == var_token))
		    return(selection);
                selection = selection->next;
	    }
        }
        return(NULL);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_set_context
|
|       Purpose: Sets the context for CLUI argument / GUI selection
|                error checking to KVF_CLUI (for command line)
|                or KVF_GUI (for graphical user interface).
|                All it does is set the global kvf_context variable,
|                which is used by kvf_check_double() bounds-checking routine to
|                choose an appropriate error message to print for the user.
|
|         Input: context   - KVF_CLUI or KVF_GUI
|        Output:
|       Returns: The selection if it finds it, NULL otherwise.
|    Written By: Danielle Argiro
|          Date: Aug 9, 1994
| Modifications:
|
------------------------------------------------------------*/

void kvf_set_context(
    int context)
{
	if ((context != KVF_CLUI) &&
	    (context != KVF_GUI))
	{
	    kerror("kforms", "kvf_set_context",
		   "Context must be either KVF_CLUI or KVF_GUI");
	    return;
	}
	kvf_context = context;
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_set_expr_handler
|
|       Purpose: Installs an expression handler provided by the caller.  
|
| 		 This routine was written to allow  cantata to install
|                a special expression handler which will evaluate
|                expressions provided by the user in text parameter
|                boxes of ints, floats, doubles, strings, and files.
|
|                By allowing cantata to install an expression handler
|                that uses the kexpr library to evaluate the expression,
|                we avoid making kforms and xvforms dependant on the
|                kexpr library.
|
|         Input: expr_handler - the expression handler to be installed
|        Output:
|       Returns: The previously installed expression handler.
|    Written By: Danielle Argiro & Mark Young
|          Date: Aug 11, 1994
| Modifications:
|
------------------------------------------------------------*/
static kfunc_int kvf_expr_handler = NULL;  /* global expression handler */

kfunc_int kvf_set_expr_handler(
    kfunc_int expr_handler)
{
	kfunc_int tmp_handler = kvf_expr_handler;

	kvf_expr_handler = expr_handler;

	return(tmp_handler);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_get_expr_handler
|
|       Purpose: Inquires if an expression handler has been 
|                provided by the caller.
|
|         Input:
|        Output:
|       Returns: The installed expression handler or NULL.
|    Written By: Danielle Argiro & Mark Young
|          Date: Aug 11, 1994
| Modifications:
|
------------------------------------------------------------*/

kfunc_int kvf_get_expr_handler(void)
{
        return(kvf_expr_handler);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_call_expr_handler
|
|       Purpose: Calls the expression handler provided by the caller.  
|
|                This routine was written to allow  cantata to install
|                a special expression handler which will evaluate
|                expressions provided by the user in text parameter
|                boxes of ints, floats, doubles, strings, and files.
|
|                By allowing cantata to install an expression handler
|                that uses the kexpr library to evaluate the expression,
|                we avoid making kforms and xvforms dependant on the
|                kexpr library.
|
|         Input: expr_handler - the expression handler to be installed
|        Output:
|       Returns: The previously installed expression handler.
|    Written By: Danielle Argiro & Mark Young
|          Date: Aug 11, 1994
| Modifications:
|
------------------------------------------------------------*/
int kvf_call_expr_handler(
    kform *form,
    char  *string,
    int   type,
    kaddr return_value,
    char  *error_string)
{
	kaddr      client_data = NULL;
	kform_list *entry;

	if (kvf_expr_handler == NULL)
	    return(TRUE);
	
	entry = kvf_get_entry(form);
	if (entry != NULL)
	   client_data =  entry->client_data;
        return((*kvf_expr_handler)(form, client_data, string, type, 
				   return_value, error_string));
}


/*-----------------------------------------------------------
|
|  Routine Name: kvf_set_int_from_literal
|
|       Purpose: Sets the value of an integer selection
|                from the value of its literal
|
|         Input: selection - the integer selection
|                literal   - the literal string value
|        Output:
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro 
|          Date: Oct 20, 1994
| Modifications:
|
------------------------------------------------------------*/

int kvf_set_int_from_literal(
    kselection *selection,
    char       *literal)
{
        char leftover[KLENGTH];
        int  tmp_int, status;

	if (selection->type != KUIS_INTEGER)
	{
	    kerror("kforms", "kvf_set_int_from_literal",
		   "selection of type other than integer passed in");
	    return(FALSE);
	}

	/*
         *  it is an expression
         */
	status = ksscanf(literal,"%d%s", &tmp_int, leftover); 
        if (status != 2 || leftover[0] != '\0')
        {
            /*
             *  call installed expression handler.
             */
            if (kvf_call_expr_handler(selection->back_form, literal, 
					KINT, (kaddr) &tmp_int, leftover))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_INT_VAL, tmp_int)))
                    return(FALSE);
            }
	}

        /*
         * it is a straightforward integer
         */
        else
        {
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_INT_VAL, tmp_int)))
                return(FALSE);
        }
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_set_float_from_literal
|
|       Purpose: Sets the value of a float selection
|                from the value of its literal
|
|         Input: selection - the float selection
|                literal   - the literal string value
|        Output:
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 20, 1994
| Modifications:
|
------------------------------------------------------------*/

int kvf_set_float_from_literal(
    kselection *selection,
    char       *literal)
{
        char  leftover[KLENGTH];
        float tmp_float;

        if (selection->type != KUIS_FLOAT)
        {
            kerror("kforms", "kvf_set_float_from_literal",
                   "selection of type other than float passed in");
            return(FALSE);
        }

        /*
         *  it is an expression
         */
        if (ksscanf(literal,"%f%s", &tmp_float, leftover) != 2 ||
                    leftover[0] != '\0')
        {
            /*
             *  call installed expression handler.
             */
            if (kvf_call_expr_handler(selection->back_form, literal,
                                        KFLOAT, (kaddr) &tmp_float, leftover))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_FLOAT_VAL, tmp_float)))
                    return(FALSE);
            }
        }

        /*
         * it is a straightforward float
         */
        else
        {
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_FLOAT_VAL, tmp_float)))
                return(FALSE);
        }
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_set_double_from_literal
|
|       Purpose: Sets the value of a double selection
|                from the value of its literal
|
|         Input: selection - the double selection
|                literal   - the literal string value
|        Output:
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 20, 1994
| Modifications:
|
------------------------------------------------------------*/

int kvf_set_double_from_literal(
    kselection *selection,
    char       *literal)
{
        char   leftover[KLENGTH];
        double tmp_double;

        if (selection->type != KUIS_DOUBLE)
        {
            kerror("kforms", "kvf_set_double_from_literal",
                   "selection of type other than double passed in");
            return(FALSE);
        }

        /*
         *  it is an expression
         */
        if (ksscanf(literal,"%lg%s", &tmp_double, leftover) != 2 ||
                    leftover[0] != '\0')
        {
            /*
             *  call installed expression handler.
             */
            if (kvf_call_expr_handler(selection->back_form, literal,
                                        KDOUBLE, (kaddr) &tmp_double, leftover))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_DOUBLE_VAL, tmp_double)))
                    return(FALSE);
            }
        }

        /*
         * it is a straightforward double
         */
        else
        {
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_DOUBLE_VAL, tmp_double)))
                return(FALSE);
        }
        return(TRUE);
}

/*-----------------------------------------------------------
|
|  Routine Name: kvf_set_file_from_literal
|
|       Purpose: Sets the value of a file
|                from the value of its literal
|
|         Input: selection - the inputfile or outputfile selection
|                literal   - the literal string value
|        Output:
|       Returns: TRUE on success, FALSE on failure
|    Written By: Danielle Argiro
|          Date: Oct 20, 1994
| Modifications:
|
------------------------------------------------------------*/

int kvf_set_file_from_literal(
    kselection *selection,
    char       *literal)
{
        char   leftover[KLENGTH]; 
	char   tmp_file[KLENGTH];

        if ((selection->type != KUIS_INPUTFILE)    &&
	    (selection->type != KUIS_OUTPUTFILE))
        {
            kerror("kforms", "kvf_set_file_from_literal",
                   "selection of type other than inputfile or outputfile passed in");
            return(FALSE);
        }


	/*
         * it's ok if the filename is NULL
         */
        tmp_file[0] = '\0';
        (void) kstring_cleanup(literal, literal);

        /*
         *  it is an expression
         */
        if (ksscanf(literal, "%s%s", tmp_file, leftover) != 1 ||
                    leftover[0] != '\0')
        {
            /*
             *  call installed expression handler.
             */
            if (kvf_call_expr_handler(selection->back_form, literal,
                                      KSTRING, (kaddr) &tmp_file, leftover))
            {
                if (!(kvf_set_attribute(selection->back_kformstruct,
                                        KVF_FILE_NAME, tmp_file)))
                    return(FALSE);
            }
        }

        /*
         * it is a straightforward filename
         */
        else
        {
            if (!(kvf_set_attribute(selection->back_kformstruct,
                                    KVF_FILE_NAME, tmp_file)))
                return(FALSE);
        }
        return(TRUE);
}

